/*
 * stream.c - Source for TpStreamEngineAudioStream
 * Copyright (C) 2006-2008 Collabora Ltd.
 * Copyright (C) 2006-2008 Nokia Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "audiostream.h"

#include <string.h>

#include <gst/farsight/fs-conference-iface.h>

#include "tp-stream-engine.h"

#include "tp-stream-engine-signals-marshal.h"
#include "util.h"

G_DEFINE_TYPE (TpStreamEngineAudioStream, tp_stream_engine_audio_stream,
    G_TYPE_OBJECT);

#define DEBUG(self, format, ...)                \
  g_debug ("stream %u (audio) %s: " format,     \
      tf_stream_get_id ((self)->priv->stream),  \
      G_STRFUNC,                                \
      ##__VA_ARGS__)

#define WARNING(self, format, ...)              \
  g_warning ("stream %u (audio) %s: " format,   \
      tf_stream_get_id ((self)->priv->stream),  \
      G_STRFUNC,                                \
      ##__VA_ARGS__)


struct _TpStreamEngineAudioStreamPrivate
{
  TfStream *stream;

  GstElement *srcbin;

  GstElement *bin;

  gulong src_pad_added_handler_id;
  gulong request_resource_handler_id;
  gulong free_resource_handler_id;

  GError *construction_error;

  GMutex *mutex;
  /* Everything below this line is protected by the mutex */
  guint error_idle_id;
  GList *sinkbins;

  gboolean sending;
  gboolean playing;
};


/* properties */
enum
{
  PROP_0,
  PROP_STREAM,
  PROP_BIN,
  PROP_SRC
};

/* signal enum */
enum
{
  SINK_ADDED_SIGNAL,
  LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = {0};


static GstElement *
tp_stream_engine_audio_stream_make_src_bin (TpStreamEngineAudioStream *self,
    GError **error);


static void tp_stream_engine_audio_stream_set_property  (GObject *object,
    guint property_id,
    const GValue *value,
    GParamSpec *pspec);

static void tp_stream_engine_audio_stream_get_property  (GObject *object,
    guint property_id,
    GValue *value,
    GParamSpec *pspec);

static void src_pad_added_cb (TfStream *stream, GstPad *pad,
    FsCodec *codec, gpointer user_data);

static gboolean request_resource (TfStream *stream,
    TpMediaStreamDirection dir,
    TpStreamEngineAudioStream *self);
static void free_resource (TfStream *stream,
    TpMediaStreamDirection dir,
    TpStreamEngineAudioStream *self);



static void
tp_stream_engine_audio_stream_init (TpStreamEngineAudioStream *self)
{
  TpStreamEngineAudioStreamPrivate *priv = G_TYPE_INSTANCE_GET_PRIVATE (self,
      TP_STREAM_ENGINE_TYPE_AUDIO_STREAM, TpStreamEngineAudioStreamPrivate);

  self->priv = priv;

  self->priv->playing = TRUE;
  self->priv->mutex = g_mutex_new ();
}


static GObject *
tp_stream_engine_audio_stream_constructor (GType type,
    guint n_props,
    GObjectConstructParam *props)
{
  GObject *obj;
  TpStreamEngineAudioStream *self = NULL;
  GstPad *src_pad = NULL;
  GstPad *sink_pad = NULL;
  GstElement *srcbin = NULL;

  obj = G_OBJECT_CLASS (tp_stream_engine_audio_stream_parent_class)->constructor (type, n_props, props);

  self = (TpStreamEngineAudioStream *) obj;

  srcbin = tp_stream_engine_audio_stream_make_src_bin (self,
      &self->priv->construction_error);
  if (!srcbin)
    goto out;

  if (!gst_bin_add (GST_BIN (self->priv->bin), srcbin))
    {
      gst_object_unref (srcbin);
      self->priv->construction_error = g_error_new (TP_STREAM_ENGINE_ERROR,
          TP_STREAM_ENGINE_ERROR_CONSTRUCTION, "Could not add src to bin");
      goto out;
    }
  self->priv->srcbin = srcbin;


  g_object_get (self->priv->stream, "sink-pad", &sink_pad, NULL);

  if (!sink_pad)
    {
      self->priv->construction_error = g_error_new (TP_STREAM_ENGINE_ERROR,
          TP_STREAM_ENGINE_ERROR_CONSTRUCTION, "Could not get stream sink pad");
      goto out;
    }

  src_pad = gst_element_get_static_pad (self->priv->srcbin, "src");

  if (!src_pad)
    {
      self->priv->construction_error = g_error_new (TP_STREAM_ENGINE_ERROR,
          TP_STREAM_ENGINE_ERROR_CONSTRUCTION,
          "Could not get src pad from src");
      gst_object_unref (sink_pad);
      goto out;
    }

  if (GST_PAD_LINK_FAILED (gst_pad_link (src_pad, sink_pad)))
    {
      self->priv->construction_error = g_error_new (TP_STREAM_ENGINE_ERROR,
          TP_STREAM_ENGINE_ERROR_CONSTRUCTION, "Could not link src to stream");
      gst_object_unref (src_pad);
      gst_object_unref (sink_pad);
      goto out;
    }

  gst_object_unref (src_pad);
  gst_object_unref (sink_pad);

  gst_element_set_locked_state (self->priv->srcbin, TRUE);
  gst_element_set_state (self->priv->srcbin, GST_STATE_READY);

  self->priv->request_resource_handler_id = g_signal_connect (
      self->priv->stream, "request-resource", G_CALLBACK (request_resource),
      self);
  self->priv->free_resource_handler_id = g_signal_connect (
      self->priv->stream, "free-resource", G_CALLBACK (free_resource),
      self);

  self->priv->src_pad_added_handler_id = g_signal_connect_object (
      self->priv->stream, "src-pad-added", G_CALLBACK (src_pad_added_cb), self,
      0);

 out:
  return obj;
}

static void
free_sinkbin (gpointer data, gpointer user_data)
{
  TpStreamEngineAudioStream *self = TP_STREAM_ENGINE_AUDIO_STREAM (user_data);
  GstElement *bin = data;

  gst_element_set_locked_state (bin, TRUE);
  gst_element_set_state (bin, GST_STATE_NULL);

  gst_bin_remove (GST_BIN (self->priv->bin), bin);
}

static void
tp_stream_engine_audio_stream_dispose (GObject *object)
{
  TpStreamEngineAudioStream *self = TP_STREAM_ENGINE_AUDIO_STREAM (object);

  g_mutex_lock (self->priv->mutex);
  if (self->priv->error_idle_id)
    {
      g_source_remove (self->priv->error_idle_id);
      self->priv->error_idle_id = 0;
    }
  g_mutex_unlock (self->priv->mutex);

  if (self->priv->src_pad_added_handler_id)
    {
      g_signal_handler_disconnect (self->priv->stream,
          self->priv->src_pad_added_handler_id);
      self->priv->src_pad_added_handler_id = 0;
    }

  if (self->priv->request_resource_handler_id)
    {
      g_signal_handler_disconnect (self->priv->stream,
          self->priv->request_resource_handler_id);
      self->priv->request_resource_handler_id = 0;
    }

  if (self->priv->free_resource_handler_id)
    {
      g_signal_handler_disconnect (self->priv->stream,
          self->priv->free_resource_handler_id);
      self->priv->free_resource_handler_id = 0;
    }

  g_mutex_lock (self->priv->mutex);
  g_list_foreach (self->priv->sinkbins, free_sinkbin, self);
  g_list_free (self->priv->sinkbins);
  self->priv->sinkbins = NULL;
  g_mutex_unlock (self->priv->mutex);

  if (self->priv->srcbin)
    {
      gst_element_set_state (self->priv->srcbin, GST_STATE_NULL);
      gst_bin_remove (GST_BIN (self->priv->bin), self->priv->srcbin);
      self->priv->srcbin = NULL;
    }

  if (self->priv->bin)
    {
      gst_object_unref (self->priv->bin);
      self->priv->bin = NULL;
    }

  if (self->priv->stream)
    {
      g_object_unref (self->priv->stream);
      self->priv->stream = NULL;
    }

  if (G_OBJECT_CLASS (tp_stream_engine_audio_stream_parent_class)->dispose)
    G_OBJECT_CLASS (tp_stream_engine_audio_stream_parent_class)->dispose (object);
}

static void
tp_stream_engine_audio_stream_finalize (GObject *object)
{
  TpStreamEngineAudioStream *self = TP_STREAM_ENGINE_AUDIO_STREAM (object);

  g_mutex_free (self->priv->mutex);

  if (G_OBJECT_CLASS (tp_stream_engine_audio_stream_parent_class)->finalize)
    G_OBJECT_CLASS (tp_stream_engine_audio_stream_parent_class)->finalize (object);
}

static void
tp_stream_engine_audio_stream_class_init (TpStreamEngineAudioStreamClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  g_type_class_add_private (klass, sizeof (TpStreamEngineAudioStreamPrivate));
  object_class->dispose = tp_stream_engine_audio_stream_dispose;
  object_class->finalize = tp_stream_engine_audio_stream_finalize;
  object_class->constructor = tp_stream_engine_audio_stream_constructor;
  object_class->set_property = tp_stream_engine_audio_stream_set_property;
  object_class->get_property = tp_stream_engine_audio_stream_get_property;

  g_object_class_install_property (object_class, PROP_STREAM,
      g_param_spec_object ("stream",
          "Tp StreamEngine Stream",
          "The Telepathy Stream Engine Stream",
          TF_TYPE_STREAM,
          G_PARAM_CONSTRUCT_ONLY | G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS));

  g_object_class_install_property (object_class, PROP_BIN,
      g_param_spec_object ("bin",
          "The Bin to add stuff to",
          "The Bin to add the elements to",
          GST_TYPE_BIN,
          G_PARAM_CONSTRUCT_ONLY | G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS));

  g_object_class_install_property (object_class, PROP_SRC,
      g_param_spec_object ("src",
          "The audio src",
          "The audio src element",
          GST_TYPE_ELEMENT,
          G_PARAM_READABLE | G_PARAM_STATIC_STRINGS));

  signals[SINK_ADDED_SIGNAL] = g_signal_new ("sink-added",
      G_OBJECT_CLASS_TYPE (klass),
      G_SIGNAL_RUN_LAST,
      0,
      NULL, NULL,
      g_cclosure_marshal_VOID__OBJECT,
      G_TYPE_NONE, 1, G_TYPE_OBJECT);
}

static void
tp_stream_engine_audio_stream_set_property  (GObject *object,
    guint property_id,
    const GValue *value,
    GParamSpec *pspec)
{
  TpStreamEngineAudioStream *self = TP_STREAM_ENGINE_AUDIO_STREAM (object);

    switch (property_id)
    {
    case PROP_STREAM:
      self->priv->stream = g_value_dup_object (value);
      break;
    case PROP_BIN:
      self->priv->bin = g_value_dup_object (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
      break;
    }
}

static void
tp_stream_engine_audio_stream_get_property  (GObject *object,
    guint property_id,
    GValue *value,
    GParamSpec *pspec)
{
  TpStreamEngineAudioStream *self = TP_STREAM_ENGINE_AUDIO_STREAM (object);

    switch (property_id)
    {
    case PROP_STREAM:
      g_value_set_object (value, self->priv->stream);
      break;
    case PROP_BIN:
      g_value_set_object (value, self->priv->bin);
      break;
    case PROP_SRC:
      g_value_set_object (value, self->priv->srcbin);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
      break;
    }
}

static GstElement *
tp_stream_engine_audio_stream_make_src_bin (TpStreamEngineAudioStream *self,
    GError **error)
{
  const gchar *audiosrc = "pulsesrc ! "
      "capsfilter caps=\"audio/x-raw-int,rate=(int)8000,channels=(int)1\"";

  if (g_getenv ("FS_AUDIOSRC"))
    audiosrc = g_getenv ("FS_AUDIOSRC");
  else if (g_getenv ("FS_AUDIO_SRC"))
    audiosrc = g_getenv ("FS_AUDIO_SRC");

  return gst_parse_bin_from_description (audiosrc, TRUE, error);
}


TpStreamEngineAudioStream *
tp_stream_engine_audio_stream_new (TfStream *stream,
    GstBin *bin,
    GError **error)
{
  TpStreamEngineAudioStream *self = NULL;

  g_return_val_if_fail (TF_IS_STREAM (stream) &&
      GST_IS_BIN (bin), NULL);

  self = g_object_new (TP_STREAM_ENGINE_TYPE_AUDIO_STREAM,
      "stream", stream,
      "bin", bin,
      NULL);

  if (self->priv->construction_error)
    {
      g_propagate_error (error, self->priv->construction_error);
      g_object_unref (self);
      return NULL;
    }

  return self;
}

static gboolean
src_pad_added_idle_error (gpointer user_data)
{
  TpStreamEngineAudioStream *self = TP_STREAM_ENGINE_AUDIO_STREAM (user_data);

  tf_stream_error (self->priv->stream, TP_MEDIA_STREAM_ERROR_MEDIA_ERROR,
      "Error setting up audio reception");

  g_mutex_lock (self->priv->mutex);
  self->priv->error_idle_id = 0;
  g_mutex_unlock (self->priv->mutex);

  return FALSE;
}

/* This creates the following pipeline
 *
 * farsight-pad -> { pulsesrc }
 */

static void
src_pad_added_cb (TfStream *stream, GstPad *pad, FsCodec *codec,
    gpointer user_data)
{
  TpStreamEngineAudioStream *self = TP_STREAM_ENGINE_AUDIO_STREAM (user_data);
  GstPad *sinkpad = NULL;
  gchar *padname = gst_pad_get_name (pad);
  GstElement *sink = NULL;
  gint session_id, ssrc, pt;

  DEBUG (self, "New pad added: %s", padname);

  if (sscanf (padname, "src_%d_%d_%d", &session_id, &ssrc, &pt) != 3)
    {
      WARNING (self, "Pad %s, is not a valid farsight src pad", padname);
      g_free (padname);
      goto error_finish;
    }

  g_free (padname);

  if (g_getenv ("FS_AUDIOSINK"))
    sink = gst_parse_bin_from_description (g_getenv ("FS_AUDIOSINK"), TRUE,
        NULL);
  if (!sink && g_getenv ("FS_AUDIO_SINK"))
    sink = gst_parse_bin_from_description (g_getenv ("FS_AUDIO_SINK"), TRUE,
        NULL);
  if (!sink)
    sink = gst_element_factory_make ("pulsesink", NULL);

  if (!gst_bin_add (GST_BIN (self->priv->bin), sink))
    {
      WARNING (self, "could not add sinkbin to the pipeline");
      goto error_created;
    }

  if (gst_element_set_state (sink, GST_STATE_PLAYING) ==
      GST_STATE_CHANGE_FAILURE)
    {
      WARNING (self, "Could not start audio sink filter bin");
      goto error_added;
    }

  sinkpad = gst_element_get_static_pad (sink, "sink");

  if (GST_PAD_LINK_FAILED (gst_pad_link (pad, sinkpad)))
    {
      WARNING (self, "Could not link farsight pad to sink");
      gst_object_unref (sinkpad);
      goto error_added;
    }

  gst_object_unref (sinkpad);

  g_signal_emit (self, signals[SINK_ADDED_SIGNAL], 0, sink);

  g_mutex_lock (self->priv->mutex);
  self->priv->sinkbins = g_list_append (self->priv->sinkbins, sink);
  g_mutex_unlock (self->priv->mutex);

  return;

 error_finish:
  g_mutex_lock (self->priv->mutex);
  if (!self->priv->error_idle_id)
    self->priv->error_idle_id =
        g_idle_add (src_pad_added_idle_error, self);
  g_mutex_unlock (self->priv->mutex);
  return;

 error_created:
  gst_object_unref (sink);
  goto error_finish;

 error_added:
  gst_bin_remove (GST_BIN (self->priv->bin), sink);
  goto error_finish;

}

static gboolean
request_resource (TfStream *stream,
    TpMediaStreamDirection dir,
    TpStreamEngineAudioStream *self)
{
  GstStateChangeReturn ret;

  if (!(dir & TP_MEDIA_STREAM_DIRECTION_SEND))
    return TRUE;

  if (!self->priv->playing)
    {
      self->priv->sending = TRUE;
      return TRUE;
    }

  ret = gst_element_set_state (self->priv->srcbin, GST_STATE_PLAYING);

  if (ret == GST_STATE_CHANGE_FAILURE)
    return FALSE;
  else
    {
      self->priv->sending = TRUE;
      return TRUE;
    }
}

static void
free_resource (TfStream *stream,
    TpMediaStreamDirection dir,
    TpStreamEngineAudioStream *self)

{
  if (!(dir & TP_MEDIA_STREAM_DIRECTION_SEND))
    return;

  self->priv->sending = FALSE;

  if (!self->priv->playing)
    return;

  gst_element_set_state (self->priv->srcbin, GST_STATE_READY);

}

gboolean
tp_stream_engine_audio_stream_set_playing (TpStreamEngineAudioStream *self,
    gboolean playing)
{
  GstStateChangeReturn ret;

  if (playing)
    {
      if (self->priv->sending)
        ret = gst_element_set_state (self->priv->srcbin, GST_STATE_PLAYING);
      else
        ret = gst_element_set_state (self->priv->srcbin, GST_STATE_READY);

      if (ret == GST_STATE_CHANGE_FAILURE)
        tf_stream_error (self->priv->stream, TP_MEDIA_STREAM_ERROR_MEDIA_ERROR,
            "Error re-starting the audio srouce");
    }
  else
    {
      ret = gst_element_set_state (self->priv->srcbin, GST_STATE_NULL);
      g_assert (ret != GST_STATE_CHANGE_FAILURE);
    }

  self->priv->playing = playing;

  return (ret != GST_STATE_CHANGE_FAILURE);
}
