/**
 * @file rss_item.c RSS/RDF item parsing 
 *
 * Copyright (C) 2003, 2004 Lars Lindner <lars.lindner@gmx.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <string.h>

#include <libxml/xmlmemory.h>
#include <libxml/parser.h>
#include <osso-log.h>

#include "rss_item.h"
#include "metadata.h"
#include "common.h"

#define RDF_NS	BAD_CAST"http://www.w3.org/1999/02/22-rdf-syntax-ns#"

extern GHashTable *RssToMetadataMapping;

/* uses the same namespace handler as rss_channel */
extern GHashTable *rss_nstable;
extern GHashTable *ns_rss_ns_uri_table;

/************************************************************************/
/*                        PUBLIC FUNCTIONS                              */
/************************************************************************/

/* method to parse standard tags for each item element */
itemPtr parseRSSItem(feedPtr fp, xmlNodePtr cur)
{
    gchar *tmp = NULL, *tmp2 = NULL, *tmp3 = NULL;
    NsHandler *nsh = NULL;
    parseItemTagFunc pf;
    itemPtr ip = NULL;

    g_assert(NULL != cur);

    ip = item_new();
    ip->tmpdata =
        g_hash_table_new_full(g_str_hash, g_str_equal, NULL, g_free);

    /* try to get an item about id */
    if (NULL != (tmp = (gchar *)xmlGetProp(cur, BAD_CAST "about"))) {
        item_set_id(ip, tmp);
        item_set_source(ip, tmp);
        g_free(tmp);
    }

    cur = cur->xmlChildrenNode;
    while (cur != NULL) {
        if (cur->type != XML_ELEMENT_NODE || cur->name == NULL) {
            cur = cur->next;
            continue;
        }

        /* check namespace of this tag */
        if (NULL != cur->ns) {
            if (((cur->ns->href != NULL) && NULL != (nsh = (NsHandler *)
                                                     g_hash_table_lookup
                                                     (ns_rss_ns_uri_table,
                                                      (gpointer) cur->ns->
                                                      href)))
                || ((cur->ns->prefix != NULL)
                    && NULL != (nsh = (NsHandler *)
                                g_hash_table_lookup(rss_nstable,
                                                    (gpointer) cur->ns->
                                                    prefix)))) {
                pf = nsh->parseItemTag;
                if (NULL != pf)
                    (*pf) (ip, cur);
                cur = cur->next;
                continue;
            }
        }

        /* explicitly no following else!!! */
        /* check for metadata tags */
        if ((tmp2 =
             g_hash_table_lookup(RssToMetadataMapping, cur->name)) != NULL) {
            tmp3 =
                utf8_fix(xmlNodeListGetString
                         (cur->doc, cur->xmlChildrenNode, TRUE));
            if (tmp3 != NULL) {
                ip->metadata = metadata_list_append(ip->metadata, tmp2, tmp3);
                g_free(tmp3);
            }
        }
        /* check for specific tags */
        else if (!xmlStrcmp(cur->name, BAD_CAST "pubDate")) {
            tmp =
                utf8_fix(xmlNodeListGetString
                         (cur->doc, cur->xmlChildrenNode, 1));
            if (NULL != tmp) {
                item_set_time(ip, parseRFC822Date(tmp));
                g_free(tmp);
            }
        } else if (!xmlStrcmp(cur->name, BAD_CAST "enclosure")) {
            /* RSS 0.93 allows multiple enclosures */
            //      ULOG_DEBUG("\nFound an enclosure\n");
            tmp = utf8_fix(xmlGetProp(cur, BAD_CAST "url"));
            if (NULL != tmp) {
                /*ip->metadata =
                 * metadata_list_append(ip->metadata, "enclosure", tmp); */

                if ((strstr(tmp, "://") == NULL) &&
                    (fp->htmlUrl != NULL) &&
                    (fp->htmlUrl[0] != '|') &&
                    (strstr(fp->htmlUrl, "://") != NULL)) {
                    /* add base URL if necessary and possible */
                    tmp2 = g_strdup_printf("%s/%s", fp->htmlUrl, tmp);
                    g_free(tmp);
                    tmp = tmp2;
                }
                struct enclosure_attribute *encl_attrs;
                gchar *type;
                encl_attrs = g_new(struct enclosure_attribute, 1);
                encl_attrs->ul = g_strdup(tmp);
                type = utf8_fix(xmlGetNsProp(cur, BAD_CAST "type", NULL));
                //          ULOG_DEBUG("\nFound type:%s\n",type);
                encl_attrs->type = g_strdup(type);
                encl_attrs->length =
                    g_ascii_strtod(utf8_fix
                                   (xmlGetNsProp
                                    (cur, BAD_CAST "length", NULL)), NULL);
                //          ULOG_DEBUG("\nFound length: %d\n",encl_attrs->length);
                ip->metadata =
                    metadata_list_append_enclosure_attr(ip->metadata,
                                                        "enclosure",
                                                        encl_attrs);
                //          ULOG_DEBUG("\nAdded data\n");
                g_free(tmp);
                g_free(type);

            }
        } else if (!xmlStrcmp(cur->name, BAD_CAST "guid")) {
            if (NULL == item_get_id(ip)) {
                tmp = (gchar *)xmlNodeListGetString(cur->doc, cur->xmlChildrenNode, 1);
                if (NULL != tmp) {
                    item_set_id(ip, tmp);
                    tmp2 = (gchar *)xmlGetProp(cur, (const xmlChar *)"isPermaLink");
                    if (item_get_source(ip) == NULL
                        && (tmp2 == NULL
                            || !xmlStrcmp((const xmlChar *)tmp2, BAD_CAST "true")))
                        item_set_source(ip, tmp);   /* Per the RSS 2.0 spec. */
                    if (tmp2 != NULL)
                        xmlFree(tmp2);
                    xmlFree(tmp);
                }
            }
        } else if (!xmlStrcmp(cur->name, BAD_CAST "title")) {
            //        tmp =
            //          unhtmlize(utf8_fix
            //            (xmlNodeListGetString
            //             (cur->doc, cur->xmlChildrenNode, TRUE)));
            tmp =
                utf8_fix(xmlNodeListGetString
                         (cur->doc, cur->xmlChildrenNode, TRUE));
            if (NULL != tmp) {
                item_set_title(ip, tmp);
                g_free(tmp);
            }
        } else if (!xmlStrcmp(cur->name, BAD_CAST "link")) {
            tmp =
                unhtmlize(utf8_fix
                          (xmlNodeListGetString
                           (cur->doc, cur->xmlChildrenNode, TRUE)));
            if (NULL != tmp) {
                item_set_source(ip, tmp);
                g_free(tmp);
            }
        } else if (!xmlStrcmp(cur->name, BAD_CAST "description")) {
            /*TOASK: why converToHTML ? Shouldn't it already be html tags?? */

            tmp =
                convertToHTML(utf8_fix
                              (xmlNodeListGetString
                               (cur->doc, cur->xmlChildrenNode, TRUE)));
            if (NULL != tmp) {
                /* don't overwrite content:encoded descriptions... */
                if (NULL == item_get_description(ip)) {
                    gchar *p = prepare_description(tmp);
                    item_set_description(ip, p);

                    g_free(p);
                }
                g_free(tmp);
            }
        } else if (!xmlStrcmp(cur->name, BAD_CAST "source")) {
            tmp = utf8_fix(xmlGetProp(cur, BAD_CAST "url"));
            if (NULL != tmp) {
                item_set_real_source_url(ip, tmp);
                g_free(tmp);
            }
            tmp =
                unhtmlize(utf8_fix
                          (xmlNodeListGetString
                           (cur->doc, cur->xmlChildrenNode, 1)));
            if (NULL != tmp) {
                item_set_real_source_title(ip, tmp);
                g_free(tmp);
            }
        }

        cur = cur->next;
    }

    item_set_read_status(ip, FALSE);

    g_hash_table_destroy(ip->tmpdata);
    ip->tmpdata = NULL;

    return ip;
}
