/**
 * @file update.c feed update request processing
 *
 * Copyright (C) 2003, 2004 Lars Lindner <lars.lindner@gmx.net>
 * Copyright (C) 2004 Nathan J. Conrad <t98502@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <unistd.h>
#include <stdio.h>
#include <sys/wait.h>
#include <osso-log.h>
#include <strings.h>
#include <gtkhtml/gtkhtml.h>
#include <gtkhtml/gtkhtml-stream.h>
#include <fcntl.h>
#include <libgnomevfs/gnome-vfs.h>
#include <libgnomevfs/gnome-vfs-utils.h>
#include "net/download.h"
#include "callbacks.h"

#include "debug_new.h"

#define DOWNLOAD_TIMEOUT 120


/* must never be less than 2, because first thread works exclusivly on high
 * prio requests */
#define DEFAULT_UPDATE_THREAD_CONCURRENCY	3



/* communication queues for requesting updates and sending the results */
GQueue *requests_high_prio = NULL;
GQueue *requests_normal_prio = NULL;
GQueue *requests_image = NULL;
GQueue *results = NULL;
GHashTable *url_hash = NULL;
GHashTable *download_hash = NULL;

extern AppData *app_data;

/* Variable for signaling the threads they can run */
static gboolean canRun = TRUE;
static gpointer dummyPointer = (gpointer) 1;


static volatile gboolean image_mode = FALSE;

static gint timeout_id = 0;

/* Counters */
static gint rss_current_download;
static gint rss_max_download;
static gint images_current_download;
static gint images_max_download;
static gint timeout_count;


static gboolean downloading_in_progress = FALSE;
static guint download_thread_main_id = 0;

static guint update_try_download_id = 0;

/* prototypes */
static gboolean download_thread_main(gpointer data);
gboolean set_image_mode_simple(gboolean is_img);
gboolean call_update_progressbar(gpointer data);
gboolean call_ui_mainwindow_finish(gpointer data);
gboolean pop_request(struct request **request);
static gboolean download_process_results(gpointer data);
static gboolean call_refresh_finished(gpointer data);
static void update_progress_finalize(gboolean);
/************************************************************************/
/* UTILITY FUNCTIONS */
/************************************************************************/

void update_req_ended(void)
{
    downloading_in_progress = FALSE;
    if (!download_thread_main_id)
        download_thread_main_id =
            g_idle_add(download_thread_main, (gpointer) TRUE);
}

/* read the counter . It needs to be atomic */
gint get_counter(int type)
{
    switch (type) {
        case C_IMG_CURRENT:
            return g_atomic_int_get(&images_max_download) - g_queue_get_length(requests_image); //g_atomic_int_get(&images_current_download);
        case C_IMG_MAX:
            return g_atomic_int_get(&images_max_download);
        case C_RSS_CURRENT:
            return g_atomic_int_get(&rss_current_download);
        case C_RSS_MAX:
            return g_atomic_int_get(&rss_max_download);

        default:
            ULOG_DEBUG
                ("oooooooooooooooooooooERRORooooooooooooooooooooooooooooo");
            return 0;
    }
    return 0;
}

static void inc_counter(int type)
{

    switch (type) {
        case C_IMG_CURRENT:
            g_atomic_int_inc(&images_current_download);
            break;
        case C_IMG_MAX:
            g_atomic_int_inc(&images_max_download);
            break;
        case C_RSS_CURRENT:
            g_atomic_int_inc(&rss_current_download);
            break;
        case C_RSS_MAX:
            g_atomic_int_inc(&rss_max_download);
            break;

        default:
            ULOG_DEBUG
                ("oooooooooooooooooooooERRORooooooooooooooooooooooooooooo");
    }

}

static void reset_counter(int type)
{
    ULOG_DEBUG("**** Reset counter: %d = %d ****", type, get_counter(type));

    switch (type) {
        case C_IMG_CURRENT:
            while (!g_atomic_int_compare_and_exchange
                   (&images_current_download, images_current_download, 0)) ;
            break;
        case C_IMG_MAX:
            while (!g_atomic_int_compare_and_exchange
                   (&images_max_download, images_max_download, 0)) ;
            break;
        case C_RSS_CURRENT:
            while (!g_atomic_int_compare_and_exchange
                   (&rss_current_download, rss_current_download, 0)) ;
            break;
        case C_RSS_MAX:
            while (!g_atomic_int_compare_and_exchange
                   (&rss_max_download, rss_max_download, 0)) ;
            break;

        default:
            ULOG_DEBUG
                ("oooooooooooooooooooooERRORooooooooooooooooooooooooooooo");
    }
}

/************************************************************************/
/* PRIVATE FUNCTIONS */
/************************************************************************/

static gboolean dl_timeout_cb(gpointer data)
{
    download_cancel_all(FALSE);
    return FALSE;
}

static void setup_dl_timeout()
{
    if (timeout_id)
        g_source_remove(timeout_id);
    timeout_count++;
    timeout_id = g_timeout_add(DOWNLOAD_TIMEOUT * 1000, dl_timeout_cb, NULL);
}

static void cancel_dl_timeout()
{
    if (timeout_id)
        g_source_remove(timeout_id);
    timeout_id = 0;
}

static void remove_dl_timeout()
{
    if (--timeout_count <= 0) {
        timeout_count = 0;
        if (timeout_id)
            g_source_remove(timeout_id);
        timeout_id = 0;
    }
}

gboolean pop_request(struct request ** request)
{
    if ((*request = g_queue_pop_head(requests_normal_prio)) == NULL) {  //the queues are empty. lets swith to image mode
        if (!is_image_mode() && !is_image_queue_empty()) {
            set_image_mode(TRUE, TRUE);

        }
        if (is_image_mode()) {
            call_update_progressbar(NULL);
        }
        *request = g_queue_pop_head(requests_image);
    }

    return *request != NULL;
}

void update_download_finished(struct request *nrequest)
{
    /**the remaining part of download_process**/
    if (nrequest && (nrequest->httpstatus >= 400)) {
        g_free(nrequest->data);
        nrequest->data = NULL;
        nrequest->size = 0;
    }


    if (nrequest && (nrequest->jump)) {       //not continuing, the download_process was not called from the download_thread_main
        feed_parse_new_download(nrequest);
        return;
    }

    /** The remaining part of download_thread_main */
    remove_dl_timeout();
    if (!canRun) {
        ULOG_DEBUG("download_thread_main: aborted!");
        download_request_free(&nrequest);
        return;
    }


    /* return the request so the GUI thread can merge the feeds 
     * and display the results... */



    if (app_data &&
        app_data->app_ui_data &&
        app_data->app_ui_data->search_mode == SFM_REFRESH) {
        g_queue_push_tail(results, (gpointer) nrequest);
        if (!app_data->app_ui_data->download_dequeue_source) {
            app_data->app_ui_data->download_dequeue_source =
                g_idle_add(download_process_results, NULL);
        }
    } else if (nrequest != NULL) {
        download_request_free(&nrequest);
    }

}

#define dv(fmt, val) g_debug(#val " = " fmt, val)

static gboolean download_thread_main(gpointer data)
{
    struct request *request = NULL;
    gboolean from_idle = (gboolean) data;

    if (from_idle)
      download_thread_main_id = 0;
    
    dv("%d", downloading_in_progress);
    if (downloading_in_progress||download_is_in_progress()) {
        return FALSE;
    }
    /* do update processing */
    ULOG_DEBUG("download_thread_main: waiting for request... %p",
               g_thread_self());
    if ((request = g_queue_pop_head(requests_high_prio)) == NULL) {
        if (!pop_request(&request)) {
            download_thread_main_id = 0;
            update_progress_finalize(TRUE);
            return FALSE;
        }
    }

    if (request && request != dummyPointer) {
        gboolean use_hash = is_image_mode()
            && request->type_image;
        if (use_hash)
            g_hash_table_remove(url_hash, request->source);

        if (request->callback == NULL) {
            download_request_free(&request);
        } else {
            if (use_hash)
                g_hash_table_insert(download_hash, request->source, request);

            setup_dl_timeout();
            if (download_process(request)) {    //running a process... stucked somewhere???
                update_req_ended();
            }
        }


    }

    
    return FALSE;

}

static gint g_idle_id = 0;
static gint g_idle_pb_id = 0;

gboolean call_ui_mainwindow_finish(gpointer data)
{
    g_idle_id = 0;

    ULOG_DEBUG("Calling ui_mainwindow_finish from idle ....");

    if (app_data && app_data->app_ui_data &&
        !app_data->app_ui_data->feed_displayed &&
        !app_data->app_ui_data->nospace)
        ui_mainwindow_finish();

    return FALSE;
}

gboolean call_update_progressbar(gpointer data)
{
    g_idle_pb_id = 0;
    ULOG_DEBUG("Update progressbar");
    update_progress_bar();
    ULOG_DEBUG("Update progressbar done");
    return FALSE;
}

static gboolean call_refresh_finished(gpointer data)
{
    cancel_dl_timeout();
    on_refresh_finished(app_data->app_ui_data->start_on_background, FALSE);
    set_image_mode(FALSE, FALSE);
    return FALSE;
}

static gboolean update_try_download_on_idle(gpointer data)
{
    struct request *nreq = (struct request *) data;
    if (download_process(nreq)) {
        return TRUE;
    }
    update_try_download_id = 0;
    return FALSE;
}

void update_try_download(struct request *nreq)
{
    if (download_process(nreq)) {
        if (!update_try_download_id)
            update_try_download_id =
                g_idle_add(update_try_download_on_idle, nreq);
    }
}

static gboolean download_process_results(gpointer data)
{
    struct request *request = NULL;
    gboolean result = canRun;

    gboolean img = FALSE;
    app_data->app_ui_data->download_dequeue_source = 0;
    if (data) {
        request = (struct request *) data;
        if (request && request->callback) {
            img = request->type_image;
        }
        goto continue_new_reqest;
    }

    if ((request = g_queue_pop_head(results)) && canRun) {
        ULOG_DEBUG("download_process_results: %s", request->source);
        ULOG_DEBUG("%%%%%%%%%% Begin Process Request %%%%%%%%%%\n");
        if (request && request->callback) {
            if ((img = request->type_image)) {
                g_hash_table_remove(download_hash, request->source);
                if (request->data) {
                    file_cache_add(app_data->app_ui_data->img_cache,
                                   request->source, NULL, request->data,
                                   request->size);
                    ULOG_DEBUG("Caching image: %s", request->source);
                }
                (request->callback) (request);
            } else {
                (request->callback) (request);
            }

        }
        if (request && request->jump) {
            return FALSE;
        }

      continue_new_reqest:


        if (request != NULL) {
            download_request_free(&request);
        }

        if (img) {
            inc_counter(C_IMG_CURRENT);
        } else {
            inc_counter(C_RSS_CURRENT);
        }

        call_update_progressbar(NULL);

        //need to check this function
        if (!img && !g_idle_id)
            g_idle_id = g_idle_add(call_ui_mainwindow_finish, NULL);

        if (app_data->app_ui_data->search_mode == SFM_INTERRUPTED) {
            g_assert(app_data != NULL);
            g_assert(app_data->app_ui_data != NULL);

            download_cancel_all(TRUE);
        } else {
            gboolean refresh_finished = FALSE;


            gint remaining =
                get_counter(C_RSS_MAX) - get_counter(C_RSS_CURRENT);

            if (!is_image_mode()) {
                if (remaining <= 0) {
                    reset_counter(C_RSS_MAX);
                    reset_counter(C_RSS_CURRENT);
                    //autoswitch to download images
                    if (get_counter(C_IMG_MAX) /*max_img_download */ )
                        set_image_mode(TRUE, FALSE);
                    else
                        refresh_finished = TRUE;
                }
            } else
                if (get_counter(C_IMG_MAX) - get_counter(C_IMG_CURRENT) <= 0)
            {
                reset_counter(C_IMG_CURRENT);
                reset_counter(C_IMG_MAX);
                if (remaining > 0)
                    set_image_mode(FALSE, FALSE);
                else
                    refresh_finished = TRUE;
            }
            result = refresh_finished;

            if (refresh_finished) {
                ULOG_DEBUG("Planning to call refresh finished");
                g_idle_add(call_refresh_finished, NULL);
            }
        }
    }

    update_req_ended();
    return FALSE;
}

/************************************************************************/
/* PUBLIC FUNCTIONS */
/************************************************************************/

void clear_image_user_data1(gpointer key, gpointer value, gpointer user_data)
{
    struct request *req = (struct request *) (((GList *) (value))->data);
    if (req->user_data) {
        // TODO maybe close the stream??
        req->user_data = NULL;
    }
}

void clear_image_user_data2(gpointer key, gpointer value, gpointer user_data)
{
    struct request *req = (struct request *) value;
    if (req->user_data) {
        // TODO maybe close the stream??
        req->user_data = NULL;
    }
}

void download_clear_image_user_data()
{
    g_hash_table_foreach(url_hash, clear_image_user_data1, NULL);
    g_hash_table_foreach(download_hash, clear_image_user_data2, NULL);
}

gboolean download_process(struct request *request)
{
    g_assert(request != NULL);

    if (downloading_in_progress)
        return TRUE;

    downloading_in_progress = TRUE;

    request->data = NULL;

    if ((strncasecmp(request->source, "http://", 7) == 0) ||
               (strncasecmp(request->source, "https://", 8) == 0) ||
               (strncasecmp(request->source, "file://", 7) == 0) ||
               (strncasecmp(request->source, "smb://", 6) == 0) ||
               (strncasecmp(request->source, "obex://", 7) == 0)) {

        /*tvh: IMPORTANT here we need to process local files as well 
         * Added more compareing to process http:// or https:// only 
         * should not invoke netio stuff when processing file:///....
         */
        ULOG_DEBUG("Source: %s", request->source);
        return download_begin(request);
    } else {
        g_debug("wrong request source");
        app_data->app_ui_data->errors_in_refreshing = TRUE;
        
        inc_counter(C_RSS_CURRENT);
        call_update_progressbar(NULL);

        if (request->feed) {
            feed_set_available(request->feed, FALSE);
            ui_feed_update(request->feed);
        }
        
        download_request_free(&request);

        update_req_ended();
        
        return FALSE;
    }

    return FALSE;
}

gpointer download_request_new()
{
    struct request *request;

    debug_enter("update_request_new");

    request = g_new0(struct request, 1);

    debug_exit("update_request_new");

    return (gpointer) request;
}

void download_request_free(struct request **request)
{
    DMSG("Entering %s\n", __FUNCTION__);
    if (NULL != *request) {
        DMSG("Freeing a request\n");
        download_inform_for_free(*request);
        if (NULL != (gchar *) (*request)->source) {
            g_free((*request)->source);
            (*request)->source = NULL;
        }
        if ((*request)->filtercmd) {
            g_free((*request)->filtercmd);
            (*request)->filtercmd = NULL;
        }
        if ((*request)->etag) {
            g_free((*request)->etag);
            (*request)->etag = NULL;
        }
        if ((*request)->data) {
            g_free((*request)->data);
            (*request)->data = NULL;
        }
        if ((*request)->feed) {
            (*request)->feed->request = NULL;
        }
        /*FIXME: What about the other fields? */

        /* IMPORTANT:Might be dangersous here to free some of this
         * because we will destroy the feeds, the folder or even the ui
         * if freeing them...Needs coming back
         */
        if ((*request)->user_data)
            (*request)->user_data = NULL;

        //TODO: need this?
        g_free(*request);
        *request = NULL;
    }
    DMSG("Leaving %s\n", __FUNCTION__);
}


void download_init(void)
{
    /* counters */
    timeout_count = rss_current_download = rss_max_download =
        images_current_download = images_max_download = 0;


    requests_high_prio = g_queue_new();
    results = g_queue_new();

    requests_normal_prio = g_queue_new();
    requests_image = g_queue_new();

    download_hash = g_hash_table_new(g_str_hash, g_str_equal);
    url_hash = g_hash_table_new(g_str_hash, g_str_equal);
}

void download_done(void)
{
    gint i;

    ULOG_DEBUG("Download_done");
    canRun = FALSE;

    //to be sure the first thread will get the signal
    for (i = 0; i <= DEFAULT_UPDATE_THREAD_CONCURRENCY; i++)
        g_queue_push_tail(requests_high_prio, dummyPointer);
    g_queue_push_tail(results, dummyPointer);



    g_queue_free(requests_high_prio);
    requests_high_prio = NULL;
    g_queue_free(results);
    results = NULL;

    g_queue_free(requests_normal_prio);
    requests_normal_prio = NULL;
    g_queue_free(requests_image);
    requests_image = NULL;

    g_hash_table_destroy(download_hash);
    download_hash = NULL;
    g_hash_table_destroy(url_hash);
    url_hash = NULL;

}

/* Push a download request to the wait queue depending from is priority.
 * Should not be used for images! */
void download_queue(struct request *new_request)
{
    g_assert(NULL != new_request);
    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);

    if (!is_image_mode())
        switch_progressbar(PROGRESSBAR_REFRESH_MODE);

    inc_counter(C_RSS_MAX);
    /*NB 121919*/
//  app_data->app_ui_data->pending_requests_nr++;

    ULOG_DEBUG("===== New request: %s\n", new_request->source);
    if (new_request->priority == 1) {
        ULOG_DEBUG
            ("\n\n\n\n\n\nHigh priority request!!!!!!!!!!!!!\n\n\n\n\n\n");
        g_queue_push_tail(requests_high_prio, new_request);
    } else {
        g_queue_push_tail(requests_normal_prio, new_request);
    }
    g_timeout_add(500,download_thread_main,NULL);
    //download_thread_main(NULL);
}

void image_downloaded_result(struct request *requestdata)
{
    GtkHTMLStream *stream = requestdata->user_data;
    ULOG_DEBUG("gtkhtml: Downloaded image: %s", requestdata->source);
    if (requestdata->data) {
        if (stream) {
            if (requestdata->load_image) {
                gchar *tmp = app_data->app_ui_data->load_url;
                app_data->app_ui_data->load_url = NULL;
                g_free(tmp);
            }
            write_image((gpointer) stream, requestdata->data,
                        requestdata->size);
        }
    } else if (stream) {
        if (requestdata->load_image /*&& 
                                     * requestdata->returncode == NET_ERR_HTTP_404 */ ) {
            hildon_banner_show_information(GTK_WIDGET
                                           (app_data->app_ui_data->main_view),
                                           NULL,
                                           _("rss_ni_unable_load_image"));
            gchar *tmp = app_data->app_ui_data->load_url;
            app_data->app_ui_data->load_url = NULL;
            g_free(tmp);
        }
        gtk_html_stream_write(stream, "123", -1);
        gtk_html_stream_close(stream, GTK_HTML_STREAM_OK);
    }

    return;
}

gint queue_cmp(gconstpointer a, gconstpointer b, gpointer user_data)
{
    if (((struct request *) b)->load_image)
        return 1;

    return (((struct request *) a)->user_data ? -1 : 0);
}

void print_queue(GQueue * q)
{
    ULOG_DEBUG("Printing queue: ");
    GList *l = q->head;
    while (l) {
        ULOG_DEBUG("  %s, %s", ((struct request *) l->data)->source,
                   ((struct request *) l->data)->user_data ? "TRUE" : "-");
        l = l->next;
    }
}

GList *g_queue_push_sorted(GQueue * q, gpointer data)
{
    g_queue_insert_sorted(q, data, queue_cmp, NULL);
    GList *result = q->head;
    while (result->data != data)
        result = result->next;
    return result;
}

/* rewritten glib functions, to insert a link sorted into a queue */
GList *g_list_insert_before_link(GList * list, GList * sibling, GList * node)
{
    if (!list)
        return node;
    else {
        node->prev = sibling->prev;
        node->next = sibling;
        sibling->prev = node;
        if (node->prev) {
            node->prev->next = node;
            return list;
        } else
            return node;
    }
}

void g_queue_push_link_sorted(GQueue * queue, GList * link)
{
    GList *list;
    g_return_if_fail(queue != NULL);

    list = queue->head;
    while (list && queue_cmp(list->data, link->data, NULL) < 0)
        list = list->next;

    if (list) {
        queue->head = g_list_insert_before_link(queue->head, list, link);
        queue->length++;
    } else
        g_queue_push_tail_link(queue, link);
}

/* Push a download image request to the wait queue. */
void download_queue_image(const gchar * url, gpointer data,
                          gboolean load_image)
{
    g_assert(url != NULL);
    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);


    //check if it is not already in the cache ... in some rare cases it will because
    //there ...
    size_t size;
    gpointer cdata = file_cache_find(app_data->app_ui_data->img_cache,
                                     url, &size, FALSE);
    if (cdata) {
        //ULOG_DEBUG("#####Found in cache %s: ", url);

        if (data) {
            write_image(data, cdata, size);
        }

        g_free(cdata);
    } else {
        GList *l = NULL;
        struct request *r = NULL;
        if ((l = (GList *) (g_hash_table_lookup(url_hash, url)))) {
            //if the request is already in the download queue move it to the front
            //only if data is not null
            r = (struct request *) l->data;
            r->user_data = data;
            if (data) {
                g_queue_unlink(requests_image, l);
                g_queue_push_link_sorted(requests_image, l);
            }
        } else
            if ((r =
                 (struct request *) g_hash_table_lookup(download_hash,
                                                        url))) {
            r->user_data = data;
        } else {
            //not found in the queue, so add it ...

            inc_counter(C_IMG_MAX);

            r = download_request_new();
            r->source = g_strdup(url);
            r->callback = image_downloaded_result;
            r->type_image = TRUE;
            r->user_data = data;
            r->load_image = load_image;

            if (data) {
                ULOG_DEBUG("Inserting %s: ", r->source);
                g_hash_table_insert(url_hash, r->source,
                                    g_queue_push_sorted(requests_image, r));
            } else {
                g_queue_push_tail(requests_image, r);
                g_hash_table_insert(url_hash, r->source,
                                    g_queue_peek_tail_link(requests_image));
            }

            if (is_image_mode())
	    {
                //download_thread_main(NULL);
	    	g_timeout_add(500,download_thread_main,NULL);
	    }
        }
    }

}

void download_set_online(gboolean mode)
{

    if (app_data && app_data->app_ui_data &&
        app_data->app_ui_data->search_mode != SFM_REFRESH) {

        if (get_counter(C_RSS_MAX) > 0)
            set_image_mode(FALSE, TRUE);
        else if (get_counter(C_IMG_MAX) > 0)
            set_image_mode(TRUE, TRUE);
    }

}

gboolean download_is_online(void)
{
    return TRUE;
}


void download_empty_queue(GQueue * queue)
{
    gpointer data = NULL;
    struct request *req = NULL;
    g_assert(queue != NULL);

    while ((data = g_queue_pop_head(queue))) {
        req = (struct request *) data;
        /* image request will be freed by gtkhtml model */

        if (req->type_image == TRUE && req->user_data)
            req->callback(req);
        download_request_free(&req);
    }
}

gboolean remove_all(gpointer key, gpointer value, gpointer user_data)
{
    return TRUE;
}

extern gchar *prev_find_text;
static void update_progress_finalize(gboolean show_banner)
{
    prev_find_text = NULL;

    gtkhtml_load_images(FALSE);


    g_hash_table_foreach_remove(url_hash, remove_all, NULL);
    g_hash_table_foreach_remove(download_hash, remove_all, (gpointer) 1);

    download_empty_queue(requests_high_prio);
    download_empty_queue(results);

    download_empty_queue(requests_normal_prio);
    download_empty_queue(requests_image);


    reset_counter(C_RSS_CURRENT);
    reset_counter(C_RSS_MAX);
    reset_counter(C_IMG_CURRENT);
    reset_counter(C_IMG_MAX);

    set_image_mode_simple(FALSE);

    if (app_data->app_ui_data->feed_directory_loading) {
        app_data->app_ui_data->feed_directory_loading = FALSE;

        on_refresh_finished(show_banner, TRUE);
    } else {
        on_refresh_finished(app_data->app_ui_data->start_on_background
                            && show_banner, FALSE);
    }
}

void download_cancel_all(gboolean show_banner)
{
    static gboolean canceling = FALSE;

    cancel_dl_timeout();

    if (!canceling) {
        struct request *req_to_free=download_get_current_request();
        canceling = TRUE;

        app_data->app_ui_data->search_mode = SFM_INTERRUPTED;

        ULOG_DEBUG("________download_cancel_all_________");
        if (req_to_free)
        {
            /* this will cancel the current request too */
            download_request_free(&req_to_free);
        }
        update_progress_finalize(show_banner);
        update_req_ended();

        ULOG_DEBUG(">>>>>>>>>>download_cancel_all done<<<<<<<<<<<<<");
        canceling = FALSE;
    }
}

gboolean is_image_mode()
{
    gboolean result;
    result = image_mode;
    return result;
}

gboolean set_image_mode_simple(gboolean is_img)
{
    gboolean not_eq = (image_mode != is_img);
    image_mode = is_img;
    return not_eq;
}

void set_image_mode(gboolean is_img, gboolean mutex_locked)
{
    ULOG_DEBUG(">>>>>>requesting changing mode");
    if (set_image_mode_simple(is_img) || mutex_locked) {
        //We are switching to download images gathered in the queue
        if (is_img) {
            if (get_counter(C_IMG_MAX)) {
                if (!download_thread_main_id)
                    download_thread_main_id =
                        g_idle_add(download_thread_main, (gpointer) TRUE);
                switch_progressbar(PROGRESSBAR_IMAGE_MODE);
            }
        } else {
            if (get_counter(C_RSS_MAX))
                switch_progressbar(PROGRESSBAR_REFRESH_MODE);

            update_progress_bar();
        }

    }

    if (app_data->app_ui_data->nospace == TRUE) {
        ui_show_save_nodevicemem_dialog();
        app_data->app_ui_data->nospace = FALSE;
    }

}

gboolean is_image_queue_empty()
{
    ULOG_DEBUG("is_image_queue_empty: feed(%d, %d), img(%d, %d)",
               get_counter(C_RSS_CURRENT),
               get_counter(C_RSS_MAX),
               get_counter(C_IMG_CURRENT), get_counter(C_IMG_MAX));
    return g_queue_is_empty(requests_image);
}

gboolean results_empty()
{
    return g_queue_is_empty(results);
}
