/**
 * @file ui_feedlist.c GUI feed list handling
 *
 * Copyright (C) 2004 Lars Lindner <lars.lindner@gmx.net>
 * Copyright (C) 2004 Nathan J. Conrad <t98502@users.sourceforge.net>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 */

/* for strsep */
#define _BSD_SOURCE

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <errno.h>
#include <unistd.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <osso-log.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <gdk/gdkevents.h>
#include <hildon/hildon-banner.h>
#include <conic.h>

#define DBUS_API_SUBJECT_TO_CHANGE
#include <osso-rss-feed-reader/settings.h>
#include <osso-rss-feed-reader/dbus.h>
#include "support.h"
#include "interface.h"
#include "callbacks.h"
#include "feed_cache.h"
#include "folder.h"
#include "conf.h"
#include "ui_feedlist.h"
#include "ui_mainwindow.h"
#include "ui_feed.h"
#include "update.h"
#include "htmlview.h"
#include "favicon.h"
#include "debug.h"
#include "ui_notification.h"
#include "dbus.h"
#include "ui_mainwindow.h"

#include "debug_new.h"

#define AUTOMATIC_DELETION_TIMEOUT 24 * 60

typedef struct {
    gpointer ptr;
    gboolean found;
    GtkTreeIter iter;
    GtkTreePath *path;
} SearchData;

extern GtkWidget *mainwindow;
extern GHashTable *feedHandler;
extern AppData *app_data;
extern guint rss_max_feed;
extern gulong rss_unread;
extern time_t rss_updated;

GtkTreeModel *feedmodel = NULL;
GtkTreeStore *feedstore = NULL;

/* flag to enable/disable the GtkTreeModel filter */
gboolean filter_feeds_without_unread_headlines = FALSE;

static GtkTreeModel *filter = NULL;
static folderPtr root_folder_ptr = NULL;

gboolean ui_feedlist_auto_update(void *data);
void rss_feed_tree_model_init(GtkTreeModelIface * iface, gpointer user_data);
gboolean ui_feedlist_cursor_changed_cb(GtkWidget * widget,
                                       GdkEvent * event, gpointer user_data);
void ui_feedlist_selection_changed_cb(GtkTreeSelection * treeselection,
                                      gpointer user_data);
gboolean ui_feedlist_event_cb(GtkWidget * widget, GdkEvent * event,
                              gpointer user_data);
gboolean ui_feedlist_button_press_cb(GtkWidget * widget, GdkEvent * event,
                                     gpointer user_data);
void ui_feedlist_set_model(GtkTreeView * feedview,
                           GtkTreeStore * feedstore, gboolean filtered);

/************************************************************************/
/* PRIVATE FUNCTIONS */
/************************************************************************/

/**
 * Helper function to search a tree model by a node.
 * To be used with gtk_tree_model_foreach.
 *
 * @param model GtkTreeModel (not NULL)
 * @param path current path (not NULL)
 * @param iter current iterator (not NULL)
 * @param data pointer to a SearchData object (not NULL)
 * @return TRUE if the searched item was found, FALSE otherwise
 */
static
    gboolean
search_func(GtkTreeModel * model,
            GtkTreePath * path, GtkTreeIter * iter, gpointer data)
{
    SearchData *data1 = NULL;
    gpointer ptr = NULL;

    g_assert(model != NULL && path != NULL && iter != NULL);
    data1 = (SearchData *) data;
    g_assert(data1 != NULL);
    gtk_tree_model_get(model, iter, FS_PTR, &ptr, -1);
    if (data1->ptr == ptr) {
        g_assert(data1->path == NULL);
        data1->found = TRUE;
        data1->iter = *iter;
        data1->path = gtk_tree_path_copy(path);
        return TRUE;
    }
    return FALSE;
}


/**
 * Helper function to count the feeds in a folder hierarchy.
 *
 * @param node current node
 * @param data pointer to an integer (gint), which is incremented
 *   (not NULL)
 */
static
void count_feeds_func(nodePtr node, gpointer data)
{
    gint *count_ptr = NULL;

    /* We don't check the type of the node here. The caller of
     * ui_feedlist_do_for_all_full has to use an appropriate filter. */
    count_ptr = (gint *) data;
    g_assert(count_ptr != NULL);
    (*count_ptr)++;
}

/** Updates the feedlist
  *
  * @param iter current location in the feed tree
  */
static void ui_feedlist_update_(GtkTreeIter * iter)
{
    GtkTreeIter childiter;
    gboolean valid = FALSE;
    nodePtr ptr = NULL;

    if (iter != NULL) {
        gtk_tree_model_get(GTK_TREE_MODEL(feedstore), iter, FS_PTR, &ptr, -1);
        valid =
            gtk_tree_model_iter_children(GTK_TREE_MODEL(feedstore),
                                         &childiter, iter);
    } else {
        valid =
            gtk_tree_model_get_iter_first(GTK_TREE_MODEL(feedstore),
                                          &childiter);
    }

    if (ptr != NULL)
        ((ui_data *) (ptr->ui_data))->row = *iter;

    while (valid) {
        ui_feedlist_update_(&childiter);
        valid =
            gtk_tree_model_iter_next(GTK_TREE_MODEL(feedstore), &childiter);
    }

    if (ptr != NULL) {
        if (FST_FOLDER == ptr->type)
            ui_folder_update((folderPtr) ptr);
        else
            ui_feed_update((feedPtr) ptr);
    }
}

/** Handles feedlist key presses
  *
  * @param widget not used
  * @param event not used
  * @param data not used
  * @return always FALSE
  */
static gboolean
ui_feedlist_key_press_cb(GtkWidget * widget,
                         GdkEventKey * event, gpointer data)
{
    if (event->keyval == GDK_BackSpace) {
        if (app_data->app_ui_data->search_mode == SFM_REFRESH) {
 /*           hildon_banner_show_information(GTK_WIDGET
                                           (app_data->app_ui_data->main_view),
                                           NULL,
                                           _("rss_ib_unable_unsubscribe")); */
            return FALSE;
        } else if (app_data->app_ui_data->search_mode == SFM_SEARCH) {
/*            hildon_banner_show_information(GTK_WIDGET
                                           (app_data->app_ui_data->main_view),
                                           NULL,
                                           _
                                           ("rss_ib_unable_unsubscribe_searching")); */
            return FALSE;
        } else {
            nodePtr selected = ui_feedlist_get_selected();

            if (selected == NULL)
                return FALSE;

            if (selected->type == FST_FEED) {
                ui_feedlist_delete(selected);
            } else if (selected->type == FST_FOLDER) {
                ui_feedlist_confirm_and_delete_folder((folderPtr) selected);
            }
            return TRUE;
        }
    }
    return FALSE;
}

/** Function for deciding whether unread headlines should be shown or not
  *
  * @param model the tree model to use
  * @param iter current location in the tree
  * @param data not used
  */
static gboolean
filter_visible_function(GtkTreeModel * model,
                        GtkTreeIter * iter, gpointer data)
{
    gint count = 0;

    if (!filter_feeds_without_unread_headlines)
        return TRUE;

    gtk_tree_model_get(model, iter, FS_UNREAD, &count, -1);

    if (0 != count)
        return TRUE;
    else
        return FALSE;
}

G_DEFINE_TYPE_WITH_CODE(RssFeedTree, rss_feed_tree, G_TYPE_OBJECT,
                        G_IMPLEMENT_INTERFACE(GTK_TYPE_TREE_MODEL,
                                              rss_feed_tree_model_init)
                        G_IMPLEMENT_INTERFACE(GTK_TYPE_TREE_DRAG_DEST,
                                              rss_feed_tree_drag_init));

static void rss_feed_tree_init(RssFeedTree * self)
{
}

static void rss_feed_tree_class_init(RssFeedTreeClass * klass)
{
}

/************************************************************************/
/* PUBLIC FUNCTIONS */
/************************************************************************/

gboolean ui_feedlist_check_if_folder_empty(folderPtr folder)
{
    GtkTreeIter *iter = NULL;
    GtkTreeIter child_iter;
    nodePtr child_node = NULL;
    gint child_count = 0;

    g_assert(folder != NULL);

    if (folder_get_nonremovable(folder)) {
        return FALSE;
    } else {
        iter = &((ui_data *) (folder->ui_data))->row;
        g_assert(iter != NULL);
        child_count = gtk_tree_model_iter_n_children
            (GTK_TREE_MODEL(feedstore), iter);
        if(gtk_tree_model_iter_children(GTK_TREE_MODEL(feedstore),
                                        &child_iter, iter)) {
            gtk_tree_model_get(GTK_TREE_MODEL(feedstore), &child_iter,
                               FS_PTR, &child_node, -1);
        }
        /* An empty folder should have the "No feeds" element. */
        return !(child_count >= 2 || (child_count == 1 && child_node != NULL)); 
    }
}

static gboolean ui_feedlist_delete_proper(nodePtr ptr)
{
    g_assert(ptr != NULL);
    gboolean changes = FALSE;

    if (ptr == (nodePtr) root_folder_ptr) {
        root_folder_ptr = NULL;
    }
    if ((FST_FEED == ptr->type) || (FST_VFOLDER == ptr->type)) {
        ptr->type = ptr->type + 32; /* marks type for deleting so that export_OPML_feedlist wont include it in new datafile */

        if (conf_feedlist_save() == TRUE) {
            ptr->type = ptr->type - 32; /* changing type back to its original value */
            changes = TRUE;
            ui_notification_remove_feed((feedPtr) ptr); /* removes an existing
                                                         * notification for this
                                                         * feed */
            ui_folder_remove_node(ptr);
            feed_free((feedPtr) ptr, FALSE);
        } else {
            ptr->type = ptr->type - 32;
            hildon_banner_show_information(GTK_WIDGET
                                           (app_data->app_ui_data->main_view),
                                           NULL, dgettext("ke-recv",
                                                          "cerm_device_memory_full"));
        }
    } else {
        ui_feedlist_do_for_all(ptr,
                               ACTION_FILTER_CHILDREN | ACTION_FILTER_ANY,
                               ui_feedlist_delete_proper);
        folder_free((folderPtr) ptr);
    }

    if (changes == TRUE) {
        ui_feedlist_update();   /* because vfolder unread counts may have
                                 * changed */
        on_refresh_finished(TRUE, FALSE);
    }
    return changes;
}

void ui_feedlist_delete_(nodePtr ptr)
{
    g_assert(ptr != NULL);

    if (ptr == (nodePtr) root_folder_ptr) {
        root_folder_ptr = NULL;
    }
    if ((FST_FEED == ptr->type) || (FST_VFOLDER == ptr->type)) {
        ui_notification_remove_feed((feedPtr) ptr); /* removes an existing
                                                     * notification for this
                                                     * feed */
        ui_folder_remove_node(ptr);
        feed_free((feedPtr) ptr, TRUE);
    } else {
        ui_feedlist_do_for_all(ptr,
                               ACTION_FILTER_CHILDREN | ACTION_FILTER_ANY,
                               ui_feedlist_delete_);
        folder_free((folderPtr) ptr);
    }
    ui_feedlist_update();       /* because vfolder unread counts may have
                                 * changed */
}

gboolean ui_feedlist_is_ancestor(nodePtr ptr, GtkTreeIter * folder)
{
    GtkTreeIter *iter = NULL;
    GtkTreeModel *fm = NULL;
    GtkTreePath *node_path = NULL;
    GtkTreePath *folder_path = NULL;
    gboolean is_ancestor = FALSE;

    iter = &((ui_data *) (ptr->ui_data))->row;

    if (iter == NULL)
        return FALSE;

    fm = GTK_TREE_MODEL(feedstore);

    node_path = gtk_tree_model_get_path(fm, iter);
    folder_path = gtk_tree_model_get_path(fm, folder);

    is_ancestor = gtk_tree_path_is_ancestor(folder_path, node_path);

    gtk_tree_path_free(node_path);
    gtk_tree_path_free(folder_path);

    return is_ancestor;
}

GtkTreePath *ui_feedlist_get_previous_after_delete_folder(GtkTreeView *
                                                          treeview)
{
    GtkTreeSelection *select = NULL;
    GtkTreeModel *model = NULL;
    GtkTreeIter iter;
    model = gtk_tree_view_get_model(treeview);

    select = gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview));
    if (gtk_tree_selection_get_selected(select, &model, &iter)) {
        GtkTreePath *path_to_select = NULL;
        path_to_select =
            gtk_tree_model_get_path(GTK_TREE_MODEL(model), &iter);
        if (path_to_select) {
            if (gtk_tree_path_prev(path_to_select)) {
            } else {            // no previous item selecting simply up
                if (gtk_tree_path_up(path_to_select)) {
                } else {        // no up??? hmm... selecting root
                    path_to_select = gtk_tree_path_new_first();
                }
            }
        }
        return path_to_select;
    }
    return NULL;

}

folderPtr ui_feedlist_get_parent(nodePtr ptr)
{
    GtkTreeIter *iter = &((ui_data *) (ptr->ui_data))->row;
    GtkTreeIter parent;
    folderPtr parentPtr = NULL;

    if (gtk_tree_model_iter_parent(GTK_TREE_MODEL(feedstore), &parent, iter)) {
        gtk_tree_model_get(GTK_TREE_MODEL(feedstore), &parent, FS_PTR,
                           &parentPtr, -1);
        return parentPtr;
    }

    return NULL;
}

nodePtr ui_feedlist_get_selected(void)
{
    GtkTreeSelection *select = NULL;
    GtkTreeModel *model = NULL;
    GtkTreeIter iter;
    nodePtr ptr = NULL;

    select = gtk_tree_view_get_selection(GTK_TREE_VIEW(feedlist));
    if (gtk_tree_selection_get_selected(select, &model, &iter)) {
        gtk_tree_model_get(model, &iter, FS_PTR, &ptr, -1);
        return ptr;
    }

    return NULL;
}

gboolean ui_feedlist_display_news(gchar * feed, guint32 nr)
{
    g_assert(NULL != feed);

    nodePtr ptr = ui_feedlist_find_feed(NULL, feed);
    if (NULL == ptr) {
        return FALSE;
    }
    ui_feedlist_select(ptr);
    if (displayed_node != ptr) {
        ui_feedlist_load_selected(ptr);

        app_data->app_ui_data->scroll_to_feed = ptr;
        app_data->app_ui_data->scroll_to_item = nr;
    } else
        gtkhtml_scroll_to_item(ptr, nr);
    return TRUE;
}

nodePtr ui_feedlist_find_feed(GtkTreeIter * from, gchar * feed)
{
    GtkTreeIter iter;
    gboolean valid = FALSE;
    nodePtr node = NULL;

    g_assert(NULL != feedstore);

    if (from != NULL) {
        gtk_tree_model_get(GTK_TREE_MODEL(feedstore), from, FS_PTR, &node,
                           -1);
        valid =
            gtk_tree_model_iter_children(GTK_TREE_MODEL(feedstore), &iter,
                                         from);
    } else {
        valid =
            gtk_tree_model_get_iter_first(GTK_TREE_MODEL(feedstore), &iter);
    }

    while (valid) {
        gtk_tree_model_get(GTK_TREE_MODEL(feedstore), &iter,
                           FS_PTR, &node, -1);
        if (NULL != node) {
            if (FST_FEED == feed_get_type((feedPtr) node)) {
                if (strcmp(((feedPtr) node)->id, feed) == 0) {
                    return node;
                }
            }
            if (FST_FOLDER == feed_get_type((feedPtr) node)) {
                node = ui_feedlist_find_feed(&iter, feed);
                if (NULL != node) {
                    return node;
                }
            }
        }

        valid = gtk_tree_model_iter_next(GTK_TREE_MODEL(feedstore), &iter);
    }

    return NULL;
}

#define FIRST_DATE 1155000000

void
ui_feedlist_find_mainwindow_data(GtkTreeIter * from, gint * nr_feeds,
                                 gint * nr_unread,
                                 gint * nr_unread_feeds,
                                 gint * nr_marked, glong * last_poll_time)
{
    GtkTreeIter iter;
    gboolean valid = FALSE;
    nodePtr node = NULL;
    gint nr_unread_s = 0;

    g_assert(NULL != feedstore);
    if (from != NULL) {
        gtk_tree_model_get(GTK_TREE_MODEL(feedstore), from, FS_PTR, &node,
                           -1);
        valid =
            gtk_tree_model_iter_children(GTK_TREE_MODEL(feedstore), &iter,
                                         from);
    } else {
        valid =
            gtk_tree_model_get_iter_first(GTK_TREE_MODEL(feedstore), &iter);
    }

    while (valid) {
        nr_unread_s = 0;
        gtk_tree_model_get(GTK_TREE_MODEL(feedstore), &iter,
                           FS_PTR, &node, -1);
        if (NULL != node) {
            if (FST_FEED == feed_get_type((feedPtr) node)) {
                (*nr_feeds)++;

                /* Find the latest refresh time */
                if (((feedPtr) node)->lastPoll.tv_sec > *last_poll_time)
                    *last_poll_time = ((feedPtr) node)->lastPoll.tv_sec;

                /* Find the number of unread posts */
                if (((feedPtr) node)->unreadCount > 0) {
                    (*nr_unread) += ((feedPtr) node)->unreadCount;
                    (*nr_unread_feeds)++;
                }

                /* Find the number of items saved for later reference */
                (*nr_marked) += ((feedPtr) node)->markCount;

            } else if (FST_FOLDER == feed_get_type((feedPtr) node)) {
                ui_feedlist_find_mainwindow_data(&iter, nr_feeds,
                                                 nr_unread,
                                                 nr_unread_feeds,
                                                 nr_marked, last_poll_time);
            }
        }
        valid = gtk_tree_model_iter_next(GTK_TREE_MODEL(feedstore), &iter);
    }
}

void ui_feedlist_delete_read_items(GtkTreeIter * from)
{
    GtkTreeIter iter;
    gboolean valid = FALSE;
    nodePtr node = NULL;
    feedPtr fp = NULL;

    g_assert(NULL != feedstore);

    if (from != NULL) {
        gtk_tree_model_get(GTK_TREE_MODEL(feedstore), from, FS_PTR, &node,
                           -1);
        valid =
            gtk_tree_model_iter_children(GTK_TREE_MODEL(feedstore), &iter,
                                         from);
    } else {
        valid =
            gtk_tree_model_get_iter_first(GTK_TREE_MODEL(feedstore), &iter);
    }

    while (valid) {
        gtk_tree_model_get(GTK_TREE_MODEL(feedstore), &iter,
                           FS_PTR, &node, -1);
        if (NULL != node) {
            fp = (feedPtr) node;

            if (FST_FEED == feed_get_type(fp)) {
                if (feed_get_unread_counter(fp) == 0 &&
                    feed_get_feed_read(fp) != 0) {
                    GTimeVal time;

                    g_get_current_time(&time);

                    if (time.tv_sec - feed_get_feed_read(fp) >
                        AUTOMATIC_DELETION_TIMEOUT * 60) {
                        feed_load(fp);
                        feed_remove_unmarked(fp, FALSE);
                        feed_unload(fp);
                    }
                }
            } else if (FST_FOLDER == feed_get_type(fp)) {
                ui_feedlist_delete_read_items(&iter);
            }
        }
        valid = gtk_tree_model_iter_next(GTK_TREE_MODEL(feedstore), &iter);
    }
}

nodePtr ui_feedlist_get_selected_from_tree(GtkTreeView * treeview)
{
    GtkTreeSelection *select = NULL;
    GtkTreeModel *model = NULL;
    GtkTreeIter iter;
    nodePtr ptr = NULL;

    select = gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview));
    if (gtk_tree_selection_get_selected(select, &model, &iter)) {
        gtk_tree_model_get(model, &iter, FS_PTR, &ptr, -1);
        return ptr;
    }

    return NULL;
}

folderPtr ui_feedlist_get_target_folder_ptr(nodePtr ptr)
{
    GtkTreeIter iter;
    GtkTreeIter parent;
    GtkTreePath *path = NULL;
    GtkTreeSelection *select = NULL;
    GtkTreeModel *model = NULL;
    gboolean retval = FALSE;


    if (NULL == ptr) {
        select = gtk_tree_view_get_selection(GTK_TREE_VIEW(feedlist));
        if (!gtk_tree_selection_get_selected(select, &model, &iter)) {
            return ui_feedlist_get_root_folder();
        }

        if (model == NULL)
            return NULL;

        path = gtk_tree_model_get_path(model, &iter);

        if (gtk_tree_path_get_depth(path) == 1) {
            gtk_tree_path_free(path);
            return ui_feedlist_get_root_folder();
        }

        retval = gtk_tree_model_iter_parent(model, &parent, &iter);
	 g_assert(retval);
        gtk_tree_model_get(model, &parent, FS_PTR, &ptr, -1);

        gtk_tree_path_free(path);

        if (ptr == NULL)
            return NULL;

        if (FST_FOLDER == ptr->type)
            return (folderPtr) ptr;
        else
            return NULL;
    }

    ui_feedlist_convert_store_iter_to_visible_iter
        (&iter, &((ui_data *) (ptr->ui_data))->row);

    if (FST_FOLDER == ptr->type)
        return (folderPtr) ptr;
    else
        return ui_feedlist_get_parent(ptr);
}

folderPtr ui_feedlist_get_target_folder()
{
    return ui_feedlist_get_target_folder_ptr(ui_feedlist_get_selected());
}

void ui_feedlist_update_iter(GtkTreeIter * iter)
{
    rss_max_feed = 0;

    ui_feedlist_update_(iter);

    if (filter_feeds_without_unread_headlines)
        gtk_tree_model_filter_refilter(GTK_TREE_MODEL_FILTER(filter));

    gtk_widget_queue_draw(feedlist);
}

void rss_feed_tree_model_init(GtkTreeModelIface * iface, gpointer user_data)
{
    GtkWidget *widget =
        GTK_WIDGET(gtk_tree_model_filter_new
                   (GTK_TREE_MODEL(gtk_tree_store_new(1, G_TYPE_INT)),
                    NULL));
    GtkTreeModelIface *originaliface =
        g_type_interface_peek(((GTypeInstance *) widget)->g_class,
                              GTK_TYPE_TREE_MODEL);
    iface->get_flags = originaliface->get_flags;
    iface->get_n_columns = originaliface->get_n_columns;
    iface->get_column_type = originaliface->get_column_type;
    iface->get_iter = originaliface->get_iter;
    iface->get_path = originaliface->get_path;
    iface->get_value = originaliface->get_value;
    iface->iter_next = originaliface->iter_next;
    iface->iter_children = originaliface->iter_children;
    iface->iter_has_child = originaliface->iter_has_child;
    iface->iter_n_children = originaliface->iter_n_children;
    iface->iter_nth_child = originaliface->iter_nth_child;
    iface->iter_parent = originaliface->iter_parent;
    iface->ref_node = originaliface->ref_node;
    iface->unref_node = originaliface->unref_node;
}

/* Sets either the unread feeds filter model or the standard GTK tree model.
 * This is necessary because only the standard model supports drag and drop. */
void
ui_feedlist_set_model(GtkTreeView * feedview,
                      GtkTreeStore * feedstore, gboolean filtered)
{
    GtkTreeModel *model_without_root = NULL;
    GtkTreeModel *model = NULL;
    GtkTreePath *path = NULL;
    gboolean search_result = FALSE;

    search_result = ui_feedlist_search_model_by_ptr
        (GTK_TREE_MODEL(feedstore), (nodePtr) root_folder_ptr, &path, NULL);

    if (search_result && path != NULL) {
        model_without_root = gtk_tree_model_filter_new
            (GTK_TREE_MODEL(feedstore), path);
    } else {
        model_without_root = GTK_TREE_MODEL(feedstore);
    }
    if (path != NULL) {
        gtk_tree_path_free(path);
        path = NULL;
    }
    if (filtered) {
        filter = gtk_tree_model_filter_new(model_without_root, NULL);

        gtk_tree_model_filter_set_visible_func(GTK_TREE_MODEL_FILTER
                                               (filter),
                                               filter_visible_function,
                                               NULL, NULL);
        model = GTK_TREE_MODEL(filter);
    } else {
        model = model_without_root;
    }

    feedmodel = model;

    gtk_tree_view_set_model(GTK_TREE_VIEW(feedview), model);
}

gboolean
ui_feedlist_button_press_cb(GtkWidget * widget, GdkEvent * event,
                            gpointer user_data)
{
    GdkEventButton *eb = (GdkEventButton *) event;
    GtkTreeView *treeview = NULL;
    GtkTreeModel *model = NULL;
    GtkTreePath *path = NULL;
    GtkTreeIter iter;
    nodePtr at_cursor = NULL;
    nodePtr selected = NULL;

    treeview = GTK_TREE_VIEW(widget);
    g_assert(treeview != NULL);

    if ((event->type == GDK_BUTTON_PRESS) && (eb->button == 1)) {

        if (gtk_tree_view_get_path_at_pos(treeview,
                                          eb->x,
                                          eb->y, &path, NULL, NULL, NULL)) {
            model = gtk_tree_view_get_model(treeview);
            g_assert(model != NULL);
            gtk_tree_model_get_iter(model, &iter, path);
            gtk_tree_model_get(model, &iter, FS_PTR, &at_cursor, -1);
            gtk_tree_path_free(path);
            path = NULL;
            selected = (nodePtr) ui_feedlist_get_selected();

            if (at_cursor != NULL &&
                at_cursor == selected && at_cursor->type == FST_FEED) {
                ui_feedlist_load_selected(selected);
            }
        }
    }
    return FALSE;
}

gboolean
ui_feedlist_event_cb(GtkWidget * widget, GdkEvent * event, gpointer user_data)
{
    GdkEventButton *eb = (GdkEventButton *) event;
    GtkTreeView *treeview = NULL;
    GtkTreeModel *model = NULL;
    GtkTreePath *path = NULL;
    GtkTreeIter iter;
    nodePtr at_cursor = NULL;
    static gboolean expand = FALSE;

    treeview = GTK_TREE_VIEW(widget);
    g_assert(treeview != NULL);

    if ((event->type == GDK_BUTTON_PRESS) && (eb->button == 1)) {
        if (gtk_tree_view_get_path_at_pos(treeview,
                                          eb->x, eb->y,
                                          &path, NULL, NULL, NULL)) {
            model = gtk_tree_view_get_model(treeview);
            g_assert(model != NULL);
            gtk_tree_model_get_iter(model, &iter, path);
            gtk_tree_model_get(model, &iter, FS_PTR, &at_cursor, -1);
        }

        gboolean result = FALSE;

        expand = FALSE;

        if (!at_cursor && path) {
            // need to "eat" the event if the node under the cursor is a null 
            // one,
            // because of the tap-and-hold event

            gtk_tree_selection_select_path(gtk_tree_view_get_selection
                                           (treeview), path);

            result = TRUE;
        }
        gtk_tree_path_free(path);
        return result;
    }

    return FALSE;
}

void
ui_feedlist_selection_changed_cb(GtkTreeSelection * treeselection,
                                 gpointer user_data)
{
    AppUIData *app_ui_data;
    user_data = NULL;
    g_assert(app_data != NULL);

    app_ui_data = app_data->app_ui_data;

    nodePtr ptr = (nodePtr) ui_feedlist_get_selected();

    if(app_data->app_ui_data->search_mode == SFM_SEARCH)
    return;
	    
    if (NULL == ptr) {
        /* "no feeds" label clicked */
        ui_mainwindow_set_unsubscribe_sensitive(app_data, FALSE);
        ui_mainwindow_set_feed_deatils_sensitive(app_data, FALSE);
    } else {
        if (ptr->type == 1 ) {
            ui_mainwindow_show_statistics();
        }
    }
    app_ui_data->scrollbarposition =
        gtk_adjustment_get_value(gtk_scrolled_window_get_vadjustment
                                 (GTK_SCROLLED_WINDOW
                                  (app_ui_data->scrolledwindow)));
}

gboolean
ui_feedlist_cursor_changed_cb(GtkWidget * widget,
                              GdkEvent * event, gpointer user_data)
{
    user_data = NULL;
    AppUIData *app_ui_data;

    g_assert(app_data != NULL);

    app_ui_data = app_data->app_ui_data;
    nodePtr ptr = (nodePtr) ui_feedlist_get_selected();
    gboolean normalmode = app_ui_data->search_mode != SFM_SEARCH &&
        app_ui_data->search_mode != SFM_REFRESH;
    gboolean nonfactory = normalmode
        && !ui_feedlist_check_factory_delete(FALSE);
    gboolean canadd = normalmode && !ui_feedlist_check_factory_add(FALSE);

    gtk_widget_set_sensitive(GTK_WIDGET
                             (app_ui_data->new_feed_button), canadd);
    if (NULL != ptr && ptr->type == FST_FEED) {
        ui_mainwindow_set_feed_deatils_sensitive(app_data,
                                                 (app_ui_data->
                                                  progressbar_mode !=
                                                  PROGRESSBAR_SEARCH_MODE));
        ui_mainwindow_set_unsubscribe_sensitive(app_data, nonfactory
                                                && normalmode);
    } else {
        ui_mainwindow_set_feed_deatils_sensitive(app_data, FALSE);
        ui_mainwindow_set_unsubscribe_sensitive(app_data, FALSE);
    }
//    set_feed_properties_insens_messages(app_ui_data);
    unsubscribe_insens_messages(app_ui_data);
//    new_insens_messages(app_ui_data);
    return FALSE;
}

gboolean ui_feedlist_call_cursor_changed_on_idle(gpointer data)
{
    AppData *app_data = (AppData *) data;
    g_assert(app_data);
    ui_feedlist_cursor_changed_cb(NULL, NULL, app_data->app_ui_data);
    app_data->app_ui_data->cursor_changed_on_idle = 0;
    return FALSE;
}

void ui_feedlist_enable_def_buttons()
{
    ui_feedlist_cursor_changed_cb(NULL, NULL, app_data->app_ui_data);
}

/* sets up the entry list store and connects it to the entry list view in the 
 * main window */
void ui_feedlist_init(GtkWidget * feedview)
{
    GtkCellRenderer *textRenderer = NULL;
    GtkCellRenderer *iconRenderer = NULL;
    GtkTreeViewColumn *column = NULL;
    GtkTreeSelection *select = NULL;

    g_assert(mainwindow != NULL);
    g_assert(feedview != NULL);

    /* Set up store */
    feedstore = gtk_tree_store_new(FS_LEN,
                                   G_TYPE_STRING,
                                   GDK_TYPE_PIXBUF,
                                   G_TYPE_POINTER, G_TYPE_INT);

    /* Add root node */
    root_folder_ptr = create_root_folder(_("rss_ia_news_feed_folders"));
    ui_feedlist_add_new_folder_to_root((nodePtr) root_folder_ptr);

    ui_feedlist_update();

    ui_feedlist_set_model(GTK_TREE_VIEW(feedview), feedstore, FALSE);

    /* tvh: 2006May24: unref the model??? */
    g_object_unref(G_OBJECT(feedstore));
    /* we only render the state and title */
    iconRenderer = gtk_cell_renderer_pixbuf_new();
    textRenderer = gtk_cell_renderer_text_new();

    /* Make the folder able to ellipzise (trunkcate with ...) the text */
    g_object_set(G_OBJECT(textRenderer), "ellipsize", PANGO_ELLIPSIZE_NONE,
                 NULL);

    gtk_cell_renderer_set_fixed_size(textRenderer, -1 , -1);

    column = gtk_tree_view_column_new();

    gtk_tree_view_column_pack_start(column, iconRenderer, FALSE);
    gtk_tree_view_column_pack_start(column, textRenderer, TRUE);

    gtk_tree_view_column_add_attribute(column, iconRenderer, "pixbuf",
                                       FS_ICON);
    gtk_tree_view_column_add_attribute(column, textRenderer, "markup",
                                       FS_LABEL);

    gtk_tree_view_column_set_resizable(column, TRUE);
    gtk_tree_view_append_column(GTK_TREE_VIEW(feedview), column);
    /* And connect signals */
    /* row_activated signal doesn't get emitted under Hildon theme for some
     * reason, we need to catch it at much lower level.
     * g_signal_connect(G_OBJECT(feedview), "row-activated",
     * G_CALLBACK(ui_feedlist_row_activated_cb), NULL); */
    g_signal_connect(G_OBJECT(feedview), "event",
                     G_CALLBACK(ui_feedlist_event_cb), NULL);
    g_signal_connect(G_OBJECT(feedview), "button_press_event",
                     G_CALLBACK(ui_feedlist_button_press_cb), NULL);
    g_signal_connect(G_OBJECT(feedview), "row-expanded",
                     G_CALLBACK(ui_folder_collapse_expand_cb), NULL);
    g_signal_connect(G_OBJECT(feedview), "row-collapsed",
                     G_CALLBACK(ui_folder_collapse_expand_cb), NULL);
    g_signal_connect(G_OBJECT(feedview), "key-press-event",
                     G_CALLBACK(ui_feedlist_key_press_cb), NULL);
    g_signal_connect(G_OBJECT(feedview), "cursor-changed",
                     G_CALLBACK(ui_feedlist_cursor_changed_cb), NULL);

    /* Setup the selection handler for the main view */
    select = gtk_tree_view_get_selection(GTK_TREE_VIEW(feedview));
    gtk_tree_selection_set_mode(select, GTK_SELECTION_SINGLE);
    g_signal_connect(G_OBJECT(select), "changed",
                     G_CALLBACK(ui_feedlist_selection_changed_cb), NULL);
    ui_dnd_init();
}

void ui_feedlist_select(nodePtr np)
{
    GtkTreeIter iter;
    GtkTreeView *treeview;
    GtkTreeSelection *selection = NULL;
    GtkTreePath *path = NULL;
    gint count = 0;

    /* some comfort: select the created iter */
    treeview = GTK_TREE_VIEW(feedlist);

    selection = gtk_tree_view_get_selection(treeview);

    if (NULL != np) {
        if (filter_feeds_without_unread_headlines) {
            /* check if the node has unread items, if not it is not in the
             * filtered model and cannot be selected */
            gtk_tree_model_get(GTK_TREE_MODEL(feedstore),
                               &((ui_data *) (np->ui_data))->row,
                               FS_UNREAD, &count, -1);
            if (0 == count)
                return;
        }

        ui_feedlist_convert_store_iter_to_visible_iter
            (&iter, &((ui_data *) (np->ui_data))->row);
        path = gtk_tree_model_get_path(feedmodel, &iter);

        gtk_tree_view_scroll_to_cell(treeview, path, NULL, FALSE, 0.0, 0.0);
        gtk_tree_view_expand_to_path(treeview, path);
        /* commenting out the next line and added a new */
        /* because next line only added a "highlight" to the added */
        /* row, it didnt actually select it */
        /* gtk_tree_selection_select_path(selection, path); */
        gtk_tree_view_set_cursor(treeview, path, NULL, FALSE);
        gtk_tree_path_free(path);
    } else {
        gtk_tree_selection_unselect_all(selection);
    }
}

/* function for finding next unread item */
feedPtr ui_feedlist_find_unread_feed(nodePtr folder)
{
    feedPtr fp = NULL;
    nodePtr ptr = NULL;
    GtkTreeModel *model = NULL;
    GtkTreeIter iter, iter2, *parent = NULL;
    gboolean valid = FALSE;
    gint count = 0;

    if (folder != NULL) {
        ui_feedlist_convert_store_iter_to_visible_iter
            (&iter2, &((ui_data *) (folder->ui_data))->row);
        parent = &iter2;
    }

    model = gtk_tree_view_get_model(GTK_TREE_VIEW(feedlist));
    valid = gtk_tree_model_iter_children(model, &iter, parent);
    while (valid) {
        gtk_tree_model_get(model, &iter, FS_PTR, &ptr, FS_UNREAD, &count, -1);
        if (count > 0) {
            if ((FST_FEED == ptr->type) || (FST_VFOLDER == ptr->type)) {
                return (feedPtr) ptr;
            } else if (FST_FOLDER == ptr->type) {
                if (NULL != (fp = ui_feedlist_find_unread_feed(ptr)))
                    return fp;
            }                   /* Directories are never checked */
        }
        valid = gtk_tree_model_iter_next(model, &iter);
    }
    return NULL;
}

/*------------------------------------------------------------------------------*/
/* delete entry callbacks */
/*------------------------------------------------------------------------------*/

void ui_feedlist_delete_folder(folderPtr folder)
{
    ui_feedlist_delete_((nodePtr) folder);
    rss_dbus_signal_applet(OSSO_RSS_FEED_READER_REFRESH_FINISHED_SIGNAL);

    if (!app_data->app_ui_data->feed_displayed)
        ui_mainwindow_finish();
}


gboolean ui_feedlist_delete(nodePtr ptr)
{
    GtkTreeSelection *selection = NULL;
    GtkTreeModel *model = NULL;
    GtkTreeIter selected_iter;
    GtkTreePath *path = NULL;

    g_assert(ptr != NULL);
    g_assert(ptr->ui_data != NULL);

    if (ptr->type == FST_FEED) {    /* Unsubscribe will only delete feeds, not
                                     * folders. */
        feedPtr fp = (feedPtr) ptr;
        if (feed_get_nonremovable(fp)) {
            hildon_banner_show_information
                (GTK_WIDGET(app_data->app_ui_data->main_view), NULL,
                 _("rss_ib_unsubscribing_feed_not_allowed"));
        } else {
            g_assert(app_data != NULL);
            ConfirmationResponse resp = ui_show_feed_delete_confirmation
                (app_data->app_ui_data, fp->title);

            if (resp == CONFRESP_OK) {
                if (ptr == displayed_node)
                    gtkhtml_stop();

                if (ui_feedlist_delete_proper((nodePtr) fp)) {
                    selection =
                        gtk_tree_view_get_selection(GTK_TREE_VIEW(feedlist));
                    if (gtk_tree_selection_get_selected
                        (selection, &model, &selected_iter)) {
                        /* We are going to select the next or the previous feed
                         * in the tree, so that the cursor doesnt jump to the top 
                         * of the tree after deleting a feed */
                        if (gtk_tree_model_iter_next(model, &selected_iter)) {
                            gtk_tree_selection_select_iter(selection,
                                                           &selected_iter);
                        } else {
                            if (gtk_tree_selection_get_selected
                                (selection, &model, &selected_iter)) {
                                path =
                                    gtk_tree_model_get_path(model,
                                                            &selected_iter);
                                if (gtk_tree_path_prev(path)) {
                                    gtk_tree_selection_select_path(selection,
                                                                   path);
                                } /* Checking the depth here is just paranoia */
                                else if (gtk_tree_path_get_depth(path) > 1
                                         && gtk_tree_path_up(path)) {
                                    gtk_tree_selection_select_path(selection,
                                                                   path);
                                }
                                gtk_tree_path_free(path);
                            }
                        }
                    }
                }
                rss_dbus_signal_applet(OSSO_RSS_FEED_READER_REFRESH_FINISHED_SIGNAL);

                app_data->app_ui_data->feed_displayed = FALSE;
                ui_mainwindow_finish();
                return TRUE;
            }
        }
    }
    return FALSE;
}

/*------------------------------------------------------------------------------*/
/* property dialog callbacks */
/*------------------------------------------------------------------------------*/

void
on_popup_prop_selected(gpointer callback_data, guint callback_action,
                       GtkWidget * widget)
{
    feedPtr fp = (feedPtr) callback_data;

    g_assert(NULL != fp);
    if (NULL != fp) {
        if (FST_FEED == feed_get_type(fp)) {
            g_assert(app_data != NULL);
            g_assert(app_data->app_ui_data != NULL);
            ui_feed_propdialog_new(GTK_WINDOW
                                   (app_data->app_ui_data->main_view), fp);
            return;
        }
    }

/*    hildon_banner_show_information(GTK_WIDGET
                                   (app_data->app_ui_data->main_view), NULL,
                                   _("rss_ib_no_feed_selected")); */
}

/*------------------------------------------------------------------------------*/
/* new entry dialog callbacks */
/*------------------------------------------------------------------------------*/

void ui_feedlist_add(folderPtr parent, nodePtr node, gint position)
{
    GtkTreeIter *iter = NULL, *parentIter = NULL;

    g_assert(node->ui_data == NULL);

    /* if parent is NULL we have the root folder and don't create a new row! */
    node->ui_data = (gpointer) g_new0(struct ui_data, 1);
    iter = &(((ui_data *) (node->ui_data))->row);

    if (parent != NULL) {
        g_assert(parent->ui_data != NULL);
        parentIter = &(((ui_data *) (parent->ui_data))->row);
    }
    // NOTE: for now we should have always position = -1
    if (position < 0)
        gtk_tree_store_append(feedstore, iter, parentIter);
    else
        gtk_tree_store_insert(feedstore, iter, parentIter, position);

    gtk_tree_store_set(feedstore, iter, FS_PTR, node, -1);

    if (lifereaStarted) {
        ui_folder_check_if_empty();
        ui_feedlist_update();
    }
}

void ui_feedlist_remove(nodePtr ptr)
{
    g_assert(ptr != NULL);

    if ((FST_FEED == ptr->type) || (FST_VFOLDER == ptr->type)) {
        ui_notification_remove_feed((feedPtr) ptr); /* removes an existing
                                                     * notification for this
                                                     * feed */
        ui_folder_remove_node(ptr);
    }
    ui_feedlist_update();       /* because vfolder unread counts may have
     * changed */// does this apply here
}

void ui_feedlist_add_new_folder_to_root(nodePtr node)
{
    GtkTreeIter *iter = NULL;

    g_assert(node->ui_data == NULL);

    /* if parent is NULL we have the root folder and don't create a new row! */
    node->ui_data = (gpointer) g_new0(struct ui_data, 1);
    iter = &(((ui_data *) (node->ui_data))->row);

    gtk_tree_store_append(feedstore, iter, NULL);

    gtk_tree_store_set(feedstore, iter, FS_PTR, node, -1);
}

void
ui_feedlist_new_subscription(gchar * source,
                             const gchar * filter, gint flags,
                             folderPtr selected_parent)
{
    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);

    app_data->app_ui_data->iap_action = OSSO_IAP_NEW_SUBSCRIPTION;
    app_data->app_ui_data->new_subscription_flags = flags;
    app_data->app_ui_data->source = source;
    app_data->app_ui_data->filter = filter;

    request_iap_connect();
}

void
ui_feedlist_new_subscription_for_fd(const gchar * source,
                                    const gchar * filter, gint flags,
                                    folderPtr selected_parent,
                                    gboolean refresh,
                                    gboolean feedlist_update)
{
    feedPtr fp = NULL;
    gchar *tmp = NULL;
    int pos = -1;
    folderPtr parent = NULL;

    g_assert(app_data != NULL);

    fp = feed_new();
    tmp = conf_new_id();
    feed_set_id(fp, tmp);
    g_free(tmp);

    feed_set_source(fp, source);
    feed_set_title(fp, source);
    feed_set_filter(fp, filter);
    feed_set_added(fp, time(NULL));

    parent = (selected_parent != NULL)
        ? selected_parent : ui_feedlist_get_root_folder();

    ui_feedlist_add(parent, (nodePtr) fp, pos);

    if (feedlist_update) {
        ui_feedlist_update();
        ui_feedlist_select((nodePtr) fp);
    }

    if (refresh) {
#ifdef WITHOUT_CONNECTIVITY
        feed_schedule_update(fp,
                             flags | FEED_REQ_PRIORITY_HIGH |
                             FEED_REQ_DOWNLOAD_FAVICON |
                             FEED_REQ_AUTH_DIALOG);
#else
        if (app_data->app_ui_data->network_iap) {
            app_data->app_ui_data->connect_at_the_end = FALSE;
            feed_schedule_update(fp,
                                 flags | FEED_REQ_PRIORITY_HIGH |
                                 FEED_REQ_DOWNLOAD_FAVICON |
                                 FEED_REQ_AUTH_DIALOG);
        } else {
            app_data->app_ui_data->connect_at_the_end = TRUE;
            app_data->app_ui_data->iap_action = OSSO_IAP_REFRESH_LIST;
            app_data->app_ui_data->list_for_update =
                g_slist_append(app_data->app_ui_data->list_for_update, fp);
            app_data->app_ui_data->flags_for_list =
                flags | FEED_REQ_PRIORITY_HIGH | FEED_REQ_DOWNLOAD_FAVICON |
                FEED_REQ_AUTH_DIALOG;
        }

#endif
    }
    conf_feedlist_save_config();
}

void on_newbtn_clicked_cb(GtkButton * button, gpointer user_data)
{
    GtkWidget *newdialog;
    AppUIData *app_ui_data;

    g_assert(app_data != NULL);

    app_ui_data = app_data->app_ui_data;

    if (SFM_SEARCH == app_ui_data->search_mode) {
/*        hildon_banner_show_information(GTK_WIDGET
                                       (app_data->app_ui_data->main_view),
                                       NULL,
                                       _("rss_ib_unable_add_feed_searching")); */
        return;
    }

    /* TODO: downloading images also uses SFM_REFRESH mode */
    if (SFM_REFRESH == app_ui_data->search_mode) {
/*        hildon_banner_show_information(GTK_WIDGET
                                       (app_data->app_ui_data->main_view),
                                       NULL, _("rss_ib_unable_add_feed")); */
        return;
    }

    if (ui_feedlist_check_factory_add(TRUE))
        return;

    g_assert(app_ui_data != NULL);

    newdialog =
        ui_feed_newdialog_new(GTK_WINDOW(app_ui_data->main_view), NULL);

    if (newdialog != NULL)
        gtk_widget_show(newdialog);
}

/* Used in handling "mime_open" dbus signal and state restoring
 */
void on_newbtn_clicked(AppData * appdata, const gchar * fd_source)
{
    GtkWidget *newdialog;
    AppUIData *app_ui_data;

    g_assert(app_data != NULL);

    app_ui_data = app_data->app_ui_data;

    g_assert(app_ui_data != NULL);

    newdialog =
        ui_feed_newdialog_new(GTK_WINDOW(app_ui_data->main_view), fd_source);

    if (newdialog != NULL)
        gtk_widget_show(newdialog);
    return;
}

/* recursively calls func for every feed in the feed list */
void
ui_feedlist_do_for_all_full(nodePtr ptr, gint filter,
                            gpointer func, gint params, gpointer user_data)
{
    GtkTreeIter childiter;
    gboolean valid = FALSE, apply = FALSE, descend = FALSE;
    nodePtr child = NULL;
    if (NULL == ptr) {
        valid =
            gtk_tree_model_get_iter_root(GTK_TREE_MODEL(feedstore),
                                         &childiter);
    } else {
        g_assert(ptr->ui_data);
        valid =
            gtk_tree_model_iter_children(GTK_TREE_MODEL(feedstore),
                                         &childiter,
                                         &((ui_data *) ptr->ui_data)->row);
    }

    while (valid) {
        gtk_tree_model_get(GTK_TREE_MODEL(feedstore), &childiter, FS_PTR,
                           &child, -1);
        /* Must update counter here because the current node may be deleted! */
        valid =
            gtk_tree_model_iter_next(GTK_TREE_MODEL(feedstore), &childiter);
        /* If child == NULL, this is an empty node. */
        if (child != NULL) {
            gboolean directory = (FST_FEED == child->type) &&
                (((feedPtr) child)->fhp != NULL)
                && ((feedPtr) child)->fhp->directory;
            gboolean is_root =
                (FST_FOLDER == child->type) && ((folderPtr) child)->root;
            apply = (filter & ACTION_FILTER_CHILDREN) ||
                ((filter & ACTION_FILTER_FEED)
                 && (FST_FEED == child->type) &&
                 !directory) || ((filter & ACTION_FILTER_FEED)
                                 && (FST_VFOLDER == child->type) &&
                                 !directory)
                || ((filter & ACTION_FILTER_DIRECTORY)
                    && (FST_FEED == child->type) &&
                    directory) || ((filter & ACTION_FILTER_FOLDER)
                                   && (FST_FOLDER == child->type)
                                   && !is_root) ||
                ((filter & ACTION_FILTER_ROOT) && is_root);
            descend = !(filter & ACTION_FILTER_CHILDREN);
            if (TRUE == apply) {
                if (params == 0)
                    ((nodeActionFunc) func) (child);
                else
                    ((nodeActionDataFunc) func) (child, user_data);
            }

            /* if the iter has children and we are descending, iterate over
             * the children. */
            if (descend)
                ui_feedlist_do_for_all_full(child, filter, func, params,
                                            user_data);
        }
    }
}

/* recursively calls func for every feed in the feed list */
gboolean
ui_feedlist_do_for_all_full_with_int(nodePtr ptr, gint filter,
                                     gpointer func, gint params,
                                     gpointer user_data)
{
    GtkTreeIter childiter;
    gboolean valid = FALSE, apply = FALSE, descend = FALSE;
    nodePtr child = NULL;
    gboolean cont = FALSE;
    gboolean result = TRUE;

    if (NULL == ptr) {
        valid =
            gtk_tree_model_get_iter_root(GTK_TREE_MODEL(feedstore),
                                         &childiter);
    } else {
        g_assert(ptr->ui_data);
        valid =
            gtk_tree_model_iter_children(GTK_TREE_MODEL(feedstore),
                                         &childiter,
                                         &((ui_data *) ptr->ui_data)->row);
    }

    while (valid) {
        gtk_tree_model_get(GTK_TREE_MODEL(feedstore), &childiter, FS_PTR,
                           &child, -1);
        /* Must update counter here because the current node may be deleted! */
        valid =
            gtk_tree_model_iter_next(GTK_TREE_MODEL(feedstore), &childiter);
        /* If child == NULL, this is an empty node. */
        if (child != NULL) {
            gboolean directory = (FST_FEED == child->type)
                && (((feedPtr) child)->fhp != NULL)
                && ((feedPtr) child)->fhp->directory;
            gboolean is_root = (FST_FOLDER == child->type)
                && ((folderPtr) child)->root;
            apply = (filter & ACTION_FILTER_CHILDREN)
                || ((filter & ACTION_FILTER_FEED)
                    && (FST_FEED == child->type) && !directory)
                || ((filter & ACTION_FILTER_FEED)
                    && (FST_VFOLDER == child->type) && !directory)
                || ((filter & ACTION_FILTER_DIRECTORY)
                    && (FST_FEED == child->type) &&
                    directory) || ((filter & ACTION_FILTER_FOLDER)
                                   && (FST_FOLDER == child->type)
                                   && !is_root) ||
                ((filter & ACTION_FILTER_ROOT) && is_root);
            descend = !(filter & ACTION_FILTER_CHILDREN);
            if (TRUE == apply) {
                if (params == 0) {
                    cont = ((nodeActionFuncWithResult) func) (child);
                } else {
                    cont = ((nodeActionDataFuncWithResult)
                            func) (child, user_data);
                }
            } else {
                cont = TRUE;
            }

            if (!cont) {
                result = FALSE;
                break;
            }

            /* if the iter has children and we are descending, iterate over
             * the children. */
            if (descend) {
                result = ui_feedlist_do_for_all_full_with_int
                    (child, filter, func, params, user_data);
                if (!result) {
                    break;
                }
            }

        }
    }
    return result;
}

/* return TRUE if we are not going to refresh (either no IAP or disabled in settings)
 * rss reader will then quit
 * return FALSE otherwise */
gboolean ui_feedlist_auto_update(void *data)
{
    if (!getBooleanConfValue(RSS_SETTINGS_AUTOMATIC_UPDATES))
        return TRUE;

    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);

    if (app_data->app_ui_data->search_mode == SFM_NORMAL) {
	 /*NB#106690*/   
        //if (app_data->app_ui_data->network_iap)
        {
            app_data->app_ui_data->iap_action = OSSO_IAP_REFRESH_NO_DIALOG;
            request_iap_connect();
	    return FALSE;
        }
    }

    return TRUE;
}

/** Load the selected feed on the ui feed list on the left
  * This function will load discard the search view if search 
  * is currently going on

*/
void ui_feedlist_load_selected(nodePtr ptr)
{
    /* If we're searching, we shouldn't let users click a feed as this will
     * require 1. Display the normal search entry with sensitive add_feed
     * refresh....buttons. This is not such a good idea
     * 
     */
    if (app_data->app_ui_data->search_mode == SFM_SEARCH)
        return;

    if (ptr == NULL)
        return;

    // feed already displayed
    if (displayed_node == ptr) {
        return;
    } else {
        gtkhtml_stop();

        gtkhtml_unblock_tap_and_hold_signal();

        displayed_node = ptr;

        ui_mainwindow_set_search(FALSE);

        if ((FST_FEED == ptr->type) || (FST_VFOLDER == ptr->type)) {
            g_assert(app_data != NULL);
            g_assert(app_data->app_ui_data != NULL);

	    /*NB#95155*/
	    gchar *file_name_updated = g_strconcat(_("rss_ti_newsreader"),
			    " - ",
			    feed_get_title((feedPtr) ptr),
			    NULL);
	    
            /* Set title */
            gtk_window_set_title(GTK_WINDOW(app_data->app_ui_data->main_view),
                                 /*feed_get_title((feedPtr) ptr)*/file_name_updated);

            if (!feed_get_available((feedPtr) ptr)) {
                app_data->app_ui_data->iap_action = OSSO_IAP_REFRESH_FEED;
                app_data->app_ui_data->new_subscription = (feedPtr) ptr;
                app_data->app_ui_data->new_subscription_flags =
                    FEED_REQ_RESET_TITLE | FEED_REQ_RESET_UPDATE_INT |
                    FEED_REQ_AUTO_DISCOVER;
                request_iap_connect();

                feed_load((feedPtr) ptr);
                feed_set_feed_read((feedPtr) ptr, -1);
                feed_unload((feedPtr) ptr);
            } else if (feed_get_unread_counter((feedPtr) ptr) > 0) {
                feed_load((feedPtr) ptr);
                feed_set_feed_read((feedPtr) ptr, -1);
                feed_unload((feedPtr) ptr);
            }
            /* workaround to ensure the feedlist is focussed when we click it
             * (Mozilla might prevent this, ui_itemlist_display() depends on
             * this */
            gtk_widget_grab_focus(feedlist);

            /* Set up the item list */
            ui_itemlist_load((nodePtr) ptr);
            feed_mark_all_items_read((feedPtr /* nodePtr */ )ptr);

            ui_feedlist_update();

            if (app_data->app_ui_data->scroll_to_feed == displayed_node &&
                app_data->app_ui_data->scroll_to_item) {
                gtkhtml_scroll_to_item(app_data->app_ui_data->scroll_to_feed,
                                       app_data->app_ui_data->scroll_to_item);
                app_data->app_ui_data->scroll_to_feed = NULL;
                app_data->app_ui_data->scroll_to_item = 0;
            }
        }
    }
}

folderPtr ui_feedlist_get_root_folder(void)
{
    return root_folder_ptr;
}

void ui_feedlist_reset_root_folder(void)
{
    /* If root_folder_ptr is not NULL it has to be freed, too. */
    root_folder_ptr = NULL;
}


gboolean
ui_feedlist_search_model_by_ptr(GtkTreeModel * model,
                                nodePtr node,
                                GtkTreePath ** path, GtkTreeIter * iter)
{
    SearchData data;

    data.ptr = node;
    data.found = FALSE;
    data.path = NULL;
    gtk_tree_model_foreach(model, search_func, &data);
    if (path != NULL) {
        *path = data.path;
    }
    if (iter != NULL && data.found) {
        *iter = data.iter;
    }
    return data.found;
}


void
ui_feedlist_convert_store_iter_to_visible_iter(GtkTreeIter *
                                               iter_target,
                                               GtkTreeIter * iter_source)
{
    GtkTreeModel *child_model = NULL;
    GtkTreeIter tmp_iter;

    if (filter_feeds_without_unread_headlines) {
        child_model = gtk_tree_model_filter_get_model
            (GTK_TREE_MODEL_FILTER(feedmodel));
        gtk_tree_model_filter_convert_child_iter_to_iter
            (GTK_TREE_MODEL_FILTER(child_model), &tmp_iter, iter_source);
        gtk_tree_model_filter_convert_child_iter_to_iter
            (GTK_TREE_MODEL_FILTER(feedmodel), iter_target, &tmp_iter);
    } else {
        gtk_tree_model_filter_convert_child_iter_to_iter
            (GTK_TREE_MODEL_FILTER(feedmodel), iter_target, iter_source);
    }
}


void
ui_feedlist_convert_visible_iter_to_store_iter(GtkTreeIter *
                                               iter_target,
                                               GtkTreeIter * iter_source)
{
    GtkTreeModel *child_model = NULL;
    GtkTreeIter tmp_iter;

    if (filter_feeds_without_unread_headlines) {
        child_model = gtk_tree_model_filter_get_model
            (GTK_TREE_MODEL_FILTER(feedmodel));
        gtk_tree_model_filter_convert_iter_to_child_iter
            (GTK_TREE_MODEL_FILTER(feedmodel), &tmp_iter, iter_source);
        gtk_tree_model_filter_convert_iter_to_child_iter
            (GTK_TREE_MODEL_FILTER(child_model), iter_target, &tmp_iter);
    } else {
        gtk_tree_model_filter_convert_iter_to_child_iter
            (GTK_TREE_MODEL_FILTER(feedmodel), iter_target, iter_source);
    }
}


gint ui_feedlist_count_feeds(folderPtr folder)
{
    gint count = 0;
    g_assert(folder != NULL);

    ui_feedlist_do_for_all_full((nodePtr) folder,
                                ACTION_FILTER_FEED,
                                count_feeds_func, 1, &count);
    return count;
}


gboolean ui_feedlist_confirm_and_delete_folder(folderPtr folder)
{
    GtkTreeIter *iter = NULL;
    GtkTreeIter child_iter;
    nodePtr child_node = NULL;
    gint child_count = 0;
    gint n_feeds = 0;
    gboolean do_delete = FALSE;
    ConfirmationResponse resp = CONFRESP_ERROR;
    gboolean result = FALSE;

    g_assert(folder != NULL);

    if (folder_get_nonremovable(folder)) {
        hildon_banner_show_information(GTK_WIDGET
                                       (app_data->app_ui_data->main_view),
                                       NULL,
                                       dgettext
                                       (HILDON_COMMON_STRINGS_L10N_PACKAGE,
                                        "sfil_ib_unable_to_delete_selected_folder"));
    } else {
        iter = &((ui_data *) (folder->ui_data))->row;
        g_assert(iter != NULL);
        child_count = gtk_tree_model_iter_n_children
            (GTK_TREE_MODEL(feedstore), iter);
        if (gtk_tree_model_iter_children(GTK_TREE_MODEL(feedstore),
                                         &child_iter, iter)) {
            gtk_tree_model_get(GTK_TREE_MODEL(feedstore), &child_iter,
                               FS_PTR, &child_node, -1);
        }
        /* An empty folder should have the "No feeds" element. */
        if (child_count >= 2 || (child_count == 1 && child_node != NULL)) {
            g_assert(app_data != NULL);
            n_feeds = ui_feedlist_count_feeds(folder);
            resp = ui_show_folder_delete_confirmation
                (app_data->app_ui_data, n_feeds);
            do_delete = (resp == CONFRESP_OK);
        } else {
            do_delete = TRUE;
        }
        if (do_delete) {
            ui_feedlist_delete_folder(folder);
            result = TRUE;
        }
    }
    return result;
}

gboolean ui_feedlist_check_factory_add(gboolean show_info)
{
    folderPtr tfld = ui_feedlist_get_target_folder();
    if (folder_get_nonremovable(tfld) &&
        (ui_feedlist_get_root_folder() != tfld)) {
        if (show_info)
            hildon_banner_show_information(GTK_WIDGET
                                           (app_data->app_ui_data->main_view),
                                           NULL,
                                           _
                                           ("rss_ib_unable_to_move_items_here"));
        return TRUE;
    }

    return FALSE;
}

gboolean ui_feedlist_check_factory_delete(gboolean show_info)
{
    nodePtr ptr = (nodePtr) ui_feedlist_get_selected();
    gboolean result = FALSE;
    if (ptr) {
        if (ptr->type == FST_FOLDER) {
            result = folder_get_nonremovable((folderPtr) ptr);
            if (result && show_info)
                hildon_banner_show_information(GTK_WIDGET
                                               (app_data->app_ui_data->
                                                main_view), NULL,
                                               dgettext
                                               (HILDON_COMMON_STRINGS_L10N_PACKAGE,
                                                "sfil_ib_unable_to_delete_selected_folder"));
        } else if (ptr->type == FST_FEED) {
            result = ((feedPtr) ptr)->nonremovable;
            if (result && show_info)
                hildon_banner_show_information(GTK_WIDGET
                                               (app_data->app_ui_data->
                                                main_view), NULL,
                                               _
                                               ("rss_ib_unsubscribing_feed_not_allowed"));
        }
    }

    return result;
}
