/**
   @file applet.c

   Copyright (c) 2004-2009 Nokia. All rights reserved.

   Heavily modified after change the gtktreeview

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307 USA
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <libintl.h>
#include <locale.h>
#include <glib/gi18n-lib.h>
#include <tablet-browser-interface.h>
#include <libosso.h>
#include <hildon/hildon.h>
#include <string.h>
#include <gdk/gdk.h>

#include <sys/select.h>

/* Settings dialog and dbus message constants */
#include <osso-rss-feed-reader/dbus.h>
#include <osso-rss-feed-reader/settings.h>

#include "applet.h"
#include "applet-private.h"
#include "debug.h"
#include "applet_cache.h"
#include "rss-feed-applet-view.h"
#include "rss-feed-applet-button.h"

/* No. of elements to scroll on each button press */
#define SCROLL_STEPS 5
#define MAX_LOADED_HEADLINES 50

#define RSS_APPLET_WIDTH 316
#define RSS_APPLET_HEIGHT 376
#define RSS_APPLET_HEAD_HEIGHT 312
#define RSS_APPLET_MIN_WIDTH 300
#define RSS_APPLET_MIN_HEIGHT 160
#define MAX_HEADLINES 5 /* Reduced when the scrolling was removed */
#define FAVICON_WIDTH 32
#define HEADLINE_HEIGHT 60 /* Modified to include 5 feeds */
#define ICON_SIZE 48
#define HEADLINE_MIN_WIDTH (RSS_APPLET_WIDTH - FAVICON_WIDTH - 4*HILDON_MARGIN_DEFAULT)
#define HEADLINE_WIDTH (RSS_APPLET_WIDTH - FAVICON_WIDTH - 2*HILDON_MARGIN_DEFAULT)
#define TOOLBAR_ICON_SPACING (RSS_APPLET_WIDTH-(ICON_SIZE*3))/4

#define ANIMATION_TO_DOWN 2
#define ANIMATION_TO_DOWN_TIMEOUT 70
#define ANIMATION_TO_DOWN_WAIT 3000
#define ANIMATION_TO_DOWN_WAIT_ON_END 5000
#define ANIMATION_TO_DOWN_WAIT_ON_USER_ACTION 6000

#define LONG_BUTTON_PRESS_TIMEOUT_INTERVAL 500
#define CONT_SCROLL_TIMEOUT_INTERVAL 200

#define APPLET_DBUS_SIGNAL_LISTENER_FMT "type='signal',sender='%s',"    \
    "interface='%s'"

#define RSS_ICON_ITEM "general_rss"

#define MIN_INTERVAL 1

#define THEME_DIR "/etc/hildon/theme/images/"

#define HEAD_BG THEME_DIR "RSSAppletHead.png"
#define REFRESH_BUTTON  THEME_DIR "RSSAppletCenterButton.png"
#define REFRESH_BUTTON_PRESSED  THEME_DIR "RSSAppletCenterButtonPressed.png"
#define UP_BUTTON  THEME_DIR "RSSAppletLeftButton.png"
#define UP_BUTTON_PRESSED  THEME_DIR "RSSAppletLeftButtonPressed.png"
#define DOWN_BUTTON  THEME_DIR "RSSAppletRightButton.png"
#define DOWN_BUTTON_PRESSED  THEME_DIR "RSSAppletRightButtonPressed.png"

enum {
    ICON_COLUMN,
    URL_COLUMN,
    RENDER_COLUMN,
    TITLE_COLUMN,
    FEED_COLUMN,
    ITEMNR_COLUMN,
    N_COLUMNS,
};

#define WAITTIME_BEFORE_SHOWN_NEW_OUTOFMEM_DIALOG 30*60

#define RSS_HOME_APPLET_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE (obj, RSS_TYPE_HOME_APPLET, HomeRssAppletPrivate))

HD_DEFINE_PLUGIN_MODULE (HomeRssApplet, rss_home_applet, HD_TYPE_HOME_PLUGIN_ITEM);

const gchar* g_module_check_init(GModule *module);
const gchar* g_module_check_init(GModule *module)
{
   g_module_make_resident(module);
   return NULL;
}

/* GtkWidget redefinition */
static void     rss_home_applet_realize      (GtkWidget *widget);
static gboolean rss_home_applet_expose_event (GtkWidget      *widget,
                                              GdkEventExpose *event);

Iphb_cb_data iphb_applet_data; // Rama added

/* GConf - public */
static void _applet_gconf_init(HomeRssApplet * applet);
static void _applet_gconf_deinit(HomeRssApplet * applet);

/* DBus - public */
static DBusHandlerResult _applet_handle_signal_cb(DBusConnection * connection,
                                                  DBusMessage * signal,
                                                  gpointer user_data);
static gboolean _applet_dbus_display(HomeRssApplet *applet,
                                     gchar * feed,
                                     gchar * url,
                                     guint32 item);

/* UI / Callbacks - public */
static void _applet_refresh_completed(HomeRssApplet * applet);
static int _applet_create_temporary_file(gchar * file);
static void _applet_set_headlines(HomeRssApplet * applet);

#ifdef APPLET_TEST
static void on_menuitem_settings(GtkWidget * widget, gpointer user_data);
#endif
static void ui_show_save_nomem_dialog(HomeRssApplet * applet);
static void ui_show_save_nodevicemem_dialog(HomeRssApplet * applet);

static void create_common_view (HomeRssApplet *applet);

/* Animation - public */
static gboolean Scroll_Down_TW(gpointer data);
static gboolean Start_TW_Scroll_again(gpointer data);
static void stop_animation(HomeRssApplet *applet);
static void restart_animation(HomeRssApplet *applet, gdouble timeout);

static void applet_check_animation(HomeRssApplet * applet);

static void _applet_display_handler(osso_display_state_t state, gpointer data);

static gboolean applet_poll_iphb_fd(gpointer data);

static void _size_allocate_cb (GtkWidget * widget,
                               GtkAllocation *allocation,
                               gpointer user_data);

/* Transparency issues */
static void     rss_home_applet_change_colormap     (GtkWidget *widget);

/* Theme issues */
static void     load_theme_pixbufs                  (HomeRssApplet *applet);
static void     load_theme_icons                    (HomeRssApplet *applet);
static gboolean theme_pixbufs_loaded                (HomeRssApplet *applet);
static void     draw_themed_background              (HomeRssApplet *applet,
                                                     cairo_t *cr,
                                                     GdkRectangle *area);
static gboolean style_set_cb                        (GtkWidget *widget,
                                                     GtkStyle *previous_style,
                                                     gpointer user_data);
static void     free_theme_pixbufs                  (HomeRssApplet *applet);
static void     feed_view_set_pressed_color         (RssFeedAppletView *feed,
                                                     GdkColor *color);
static void     update_refresh_button_image         (HomeRssApplet *applet);

/* Apple feed view related*/
static void     rss_home_applet_hide_feed_list      (HomeRssApplet *applet);
static void     feed_view_clicked_cb                (GtkWidget *feed_view,
                                                     gpointer data);

/* Button related */
static void     refresh_button_clicked_cb           (GtkWidget *feed_button,
                                                     gpointer   data);
static void     up_button_clicked_cb                (GtkWidget *self,
                                                     gpointer   user_data);
static void     down_button_clicked_cb              (GtkWidget *self,
                                                     gpointer   user_data);

/* Misc */
static gint _setup_alarmd(gpointer data);

/************************************************************************/
/* PUBLIC FUNCTIONS */
/************************************************************************/
/* Applet interface functions under here */

static void applettop_changed(GObject * self,
                              GParamSpec * property_param,
                              gpointer user_data)
{
    HomeRssApplet *applet = RSS_HOME_APPLET (self);
    gboolean visible;

    g_object_get (self, "is-on-current-desktop", &visible, NULL);

    if(visible)
        {
	    applet->priv->backgrounded = FALSE;
            restart_animation(applet, ANIMATION_TO_DOWN_TIMEOUT);
        }
    else
        {
	    applet->priv->backgrounded = TRUE;
	    stop_animation(applet);
        }
}

static void
rss_home_applet_class_finalize (HomeRssAppletClass *klass)
{
    ULOG_DEBUG(__FUNCTION__);
}


void send_dbus_message_alarm_deletion(HomeRssApplet *applet)
{
    DBusMessage *message = NULL;

    g_assert(NULL != applet);
    g_assert(NULL != applet->priv->connection);

    message =
        dbus_message_new_method_call
        (RSS_SERVICE,
         RSS_OBJECT_PATH,
         RSS_INTERFACE, OSSO_RSS_FEED_READER_ALARMDEL_APPLETSHUTDOWN_SIGNAL);

    if (message == NULL)
        {
            ULOG_ERR("Creating DBus refresh call failed");
            return;
        }
    dbus_message_set_auto_start(message, TRUE);

    if (!_applet_create_temporary_file("alarmdelcheck"))
        {
            ULOG_ERR("Could not create temporary file \"alarmdelcheck\"");
            if (!applet->priv->backgrounded)
                {
                    ui_show_save_nodevicemem_dialog(applet);
                }
            dbus_message_unref(message);
            return;
        }

    if (!(dbus_connection_send(applet->priv->connection, message, NULL)))
        {
            ULOG_ERR("Sending DBus refresh message failed");
            dbus_message_unref(message);
            return;
        }

    if (message != NULL)
        {
            dbus_message_unref(message);
        }
}

static void
rss_home_applet_finalize (GObject *object)
{
    ULOG_DEBUG(__FUNCTION__);

    HomeRssApplet *applet = RSS_HOME_APPLET (object);
    HomeRssAppletPrivate *priv = applet->priv;
    DBusError err;

    dbus_error_init(&err);

	/* Rama - Free up IPHB resources */
#if 1
	close_iphb_handle(iphb_applet_data);
	remove_iphb_callback(iphb_applet_data.source_id);
	iphb_applet_data.source_id = 0;
	g_free(iphb_applet_data.user_data);
	iphb_applet_data.user_data = NULL;
#endif
	/* Till Here */

    send_dbus_message_alarm_deletion(applet);

    if (priv->scroll_down_tw_timeout)
      {
        g_source_remove(priv->scroll_down_tw_timeout);
        priv->scroll_down_tw_timeout = 0;
      }
    if (priv->scroll_down_tw_starter_timeout)
      {
        g_source_remove(priv->scroll_down_tw_starter_timeout);
        priv->scroll_down_tw_starter_timeout = 0;
      }
    if (priv->long_buttonpress_timeout)
      {
        g_source_remove(priv->long_buttonpress_timeout);
        priv->long_buttonpress_timeout = 0;
      }
    if (priv->cont_scroll_timeout)
      {
        g_source_remove(priv->cont_scroll_timeout);
        priv->cont_scroll_timeout = 0;
      }

    if (priv->timeout > 0)
      {
        ULOG_INFO("Removing old RSS Refresh timeout");
        g_source_remove(priv->timeout);
        priv->timeout = 0;
      }

#if 0	/* Rama - commented */
    if (priv->htimeout != 0)
      gtk_timeout_remove(priv->htimeout);
#endif

    _applet_gconf_deinit(applet);

    g_object_unref(priv->empty_label);

    gchar *rule = NULL;
    rule = g_strdup_printf(APPLET_DBUS_SIGNAL_LISTENER_FMT,
                           RSS_SERVICE,
                           RSS_INTERFACE);

    dbus_bus_remove_match(priv->connection, rule, &err);
    if (dbus_error_is_set(&err))
      {
        ULOG_ERR("Failed to set up DBus rule %s, error: %s", rule,
                 err.message);
        dbus_error_free(&err);
        g_free(rule);
        osso_hw_unset_event_cb(priv->osso,NULL);
        osso_deinitialize(priv->osso);
        return;
      }
    else
      {
        dbus_connection_remove_filter(priv->connection,
                                      (DBusHandleMessageFunction)
                                      _applet_handle_signal_cb, applet);
      }
    dbus_error_free(&err);
    g_free(rule);
    osso_hw_unset_event_cb(priv->osso,NULL);
    osso_deinitialize(priv->osso);

    if (priv->state_data != NULL)
      g_free(priv->state_data);

    /* theme */
    free_theme_pixbufs (applet);

    G_OBJECT_CLASS(rss_home_applet_parent_class)->finalize(object);
}

static void
rss_home_applet_class_init (HomeRssAppletClass *klass)
{
    GObjectClass *gobject_class = G_OBJECT_CLASS (klass);
    GtkWidgetClass *widget_class = GTK_WIDGET_CLASS (klass);

    /* gobject */
    gobject_class->finalize = rss_home_applet_finalize;

    /* gtkwidget */
    widget_class->expose_event = rss_home_applet_expose_event;
    widget_class->realize = rss_home_applet_realize;

    g_type_class_add_private (klass, sizeof (HomeRssAppletPrivate));
}

static void
rss_home_applet_init (HomeRssApplet *applet)
{
    HomeRssAppletPrivate *priv = RSS_HOME_APPLET_GET_PRIVATE (applet);
    gboolean ret = OSSO_OK;

    ULOG_DEBUG(__FUNCTION__);

    applet->priv = RSS_HOME_APPLET_GET_PRIVATE (applet);

    g_type_init();
    bindtextdomain(GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR);

    priv->avg_char_width = 10;
    priv->backgrounded = FALSE;
    priv->feed_list = NULL;
    priv->save_nomem_dialog_visible = FALSE;
    priv->save_nodevicemem_dialog_visible = FALSE;
    priv->loaded_headlines = 0;
    priv->first_headline_index = 0;

    /* Transparency issues */
    rss_home_applet_change_colormap (GTK_WIDGET (applet));
    create_common_view(applet);

    priv->htimeout = g_timeout_add(1000, _setup_alarmd, applet);

    hildon_helper_set_logical_color(priv->empty_label, GTK_RC_FG,
                                    GTK_STATE_NORMAL,
                                    "DefaultTextColor");
    hildon_helper_set_logical_color(priv->empty_label, GTK_RC_BG,
                                    GTK_STATE_NORMAL,
                                    "RssAppletBgColor");

    hildon_helper_set_logical_font (priv->rss_news, "HomeSystemFont");

    /* Theme */
    g_signal_connect(G_OBJECT(applet), "style-set",
                     G_CALLBACK(style_set_cb), (gpointer) applet);

    ret = osso_hw_set_display_event_cb(priv->osso, _applet_display_handler , applet);
    if( ret != OSSO_OK ) {
        g_warning( "Couldn't set hw event handler (%d)", ret );
    }

    g_signal_connect(applet, "notify::is-on-current-desktop",
                     G_CALLBACK(applettop_changed), NULL);

    /* First filling of the headlines */
    _applet_set_headlines(applet);

    iphb_applet_data.user_data = g_new0(HomeRssApplet, 1);
    memcpy(iphb_applet_data.user_data, (gpointer)(applet), sizeof(HomeRssApplet));
    return;
}

static gint
_setup_alarmd(gpointer data)
{
    ULOG_DEBUG(__FUNCTION__);

    if(iphb_applet_data.source_id == 0) {
	    iphb_applet_data.iphb_callback = (GSourceFunc)(applet_poll_iphb_fd);
	    iphb_applet_data.source_id = 0;
	    //iphb_applet_data.user_data = (gpointer)data;   

	    ULOG_ERR("ENTER alarm setup timeout\n");
	    setup_alarm_event_now(TRUE, &iphb_applet_data);
	    iphb_applet_data.source_id = setup_iphb_callback(&iphb_applet_data);
    }

    ULOG_ERR("LEAVE alarm setup timeout'n");
    return(FALSE);

}

#ifdef APPLET_TEST
void
hildon_home_applet_lib_background(void *applet_data)
{
    ULOG_DEBUG(__FUNCTION__);

    if (applet != NULL)
        {
            priv->backgrounded = TRUE;
            stop_animation();
            ULOG_DEBUG("animation stopped");
        }
    else
        ULOG_DEBUG("applet NULL, cannot update applet info");

    return;
}

void
hildon_home_applet_lib_foreground(void *applet_data)
{
    ULOG_DEBUG(__FUNCTION__);


    if (applet != NULL)
        {
            priv->backgrounded = FALSE;
            restart_animation(ANIMATION_TO_DOWN_TIMEOUT);
        }
    else
        ULOG_DEBUG("applet NULL, cannot update applet info");

    Unselect_all();

    return;
}

GtkWidget *hildon_home_applet_lib_settings(void *applet_data,
                                           GtkWindow *parent)
{
    ULOG_DEBUG(__FUNCTION__);

    GtkWidget *menuitem =
        gtk_menu_item_new_with_label(_("rss_ap_news_reader"));

    g_signal_connect(G_OBJECT(menuitem), "activate",
                     G_CALLBACK(on_menuitem_settings),
                     NULL);

    return menuitem;
}
#endif


/***************************************************************************/
/* PRIVATE FUNCTIONS */
/***************************************************************************/

/***************************************************************************/
/* Gconf handling */
/***************************************************************************/

/** Read data from gconf
 *
 * @param client gconf client
 * @param cnxn_id not used
 * @param entry not used
 * @param user_data the applet structure
 */
static void
_applet_gconf_read_cb(GConfClient * client, guint cnxn_id,
                      GConfEntry * entry, gpointer user_data)
{
    ULOG_DEBUG(__FUNCTION__);

    HomeRssApplet *applet = NULL;
    HomeRssAppletPrivate *priv = NULL;

    g_assert(client != NULL);
    g_assert(user_data != NULL);

    applet = RSS_HOME_APPLET (user_data);
    priv = applet->priv;

    priv->update =
        gconf_client_get_bool(client, RSS_SETTINGS_AUTOMATIC_UPDATES, NULL);
    priv->interval =
        gconf_client_get_int(client, RSS_SETTINGS_AUTOMATIC_UPDATES_INTERVAL,
                             NULL);
    priv->animation = FALSE; /* FIXME: scrolling support disabled, not
                                all code removed anyway */
/*         gconf_client_get_bool(client, RSS_SETTINGS_AUTOMATIC_SCROLLING, NULL); */
    priv->open_feeds_to =
        gconf_client_get_int(client, RSS_SETTINGS_OPEN_FEEDS_TO, NULL);

    if (priv->interval < MIN_INTERVAL)
        priv->interval = MIN_INTERVAL;

    if (priv->animation)
        {
            restart_animation(applet, ANIMATION_TO_DOWN_WAIT);
        }
    else
        {
            if (priv->scroll_down_tw_timeout)
                {
                    g_source_remove(priv->scroll_down_tw_timeout);
                    priv->scroll_down_tw_timeout = 0;
                }
            if (priv->scroll_down_tw_starter_timeout)
                {
                    g_source_remove(priv->scroll_down_tw_starter_timeout);
                    priv->scroll_down_tw_starter_timeout = 0;
                }
        }
}


/** Initialize gconf for reading the update related data
 *
 * @param applet the applet structure
 */
static void
_applet_gconf_init(HomeRssApplet *applet)
{
    ULOG_DEBUG(__FUNCTION__);

    GError *error = NULL;
    HomeRssAppletPrivate *priv = applet->priv;
    GConfClient *client = gconf_client_get_default();

    g_assert(NULL != client);
    g_assert(NULL != priv);

    priv->client = client;
    priv->timeout = 0;

    gconf_client_add_dir(client, RSS_SETTINGS, GCONF_CLIENT_PRELOAD_NONE,
                         &error);
    if (NULL != error)
        {
            ULOG_ERR
                ("Could not add %s directory to GConf clients watch list,"
                 "error: %s", RSS_SETTINGS, error->message);
            g_free(error);
        }

    if (NULL == error)
        {
            /* Should we save notification numbers and unregister later? */
            priv->gconf_notify_automatic_updates =
                gconf_client_notify_add(client, RSS_SETTINGS_AUTOMATIC_UPDATES,
                                        _applet_gconf_read_cb, applet, NULL,
                                        &error);
            if (NULL != error)
                {
                    ULOG_ERR("Could not set notify for GConf key %s, error: %s",
                             RSS_SETTINGS_AUTOMATIC_UPDATES, error->message);
                    g_free(error);
                }

            priv->gconf_notify_updates_interval =
                gconf_client_notify_add(client,
                                        RSS_SETTINGS_AUTOMATIC_UPDATES_INTERVAL,
                                        _applet_gconf_read_cb, applet, NULL,
                                        &error);
            if (NULL != error)
                {
                    ULOG_ERR("Could not set notify for GConf key %s, error: %s",
                             RSS_SETTINGS_AUTOMATIC_UPDATES_INTERVAL, error->message);
                    g_free(error);
                }
#if 0
            priv->gconf_notify_automatic_scrolling =
                gconf_client_notify_add(client, RSS_SETTINGS_AUTOMATIC_SCROLLING,
                                        _applet_gconf_read_cb, applet, NULL,
                                        &error);
            if (NULL != error)
                {
                    ULOG_ERR("Could not set notify for GConf key %s, error: %s",
                             RSS_SETTINGS_AUTOMATIC_SCROLLING, error->message);
                    g_free(error);
                }
#endif	    

            priv->gconf_notify_open_feeds_to =
                gconf_client_notify_add(client, RSS_SETTINGS_OPEN_FEEDS_TO,
                                        _applet_gconf_read_cb, applet, NULL,
                                        &error);
            if (NULL != error)
                {
                    ULOG_ERR("Could not set notify for GConf key %s, error: %s",
                             RSS_SETTINGS_OPEN_FEEDS_TO, error->message);
                    g_free(error);
                }
        }

    if (NULL != error)
        g_free(error);

    /* Try to install timeout even if we got errors setting up change
     * listeners */
    _applet_gconf_read_cb(client, 0, NULL, applet);
}

/** Deinitialize gconf
 *
 * @param applet the applet structure
 */
static void
_applet_gconf_deinit(HomeRssApplet *applet)
{
    ULOG_DEBUG(__FUNCTION__);

    GError *error = NULL;
    HomeRssAppletPrivate *priv = applet->priv;

    g_assert(priv != NULL);

    if (priv->gconf_notify_automatic_updates > 0)
        {
            gconf_client_notify_remove(priv->client, 
                                       priv->gconf_notify_automatic_updates);
        }
    gconf_client_remove_dir(priv->client, RSS_SETTINGS, &error);
    if (priv->gconf_notify_updates_interval > 0)
        {
            gconf_client_notify_remove(priv->client, 
                                       priv->gconf_notify_updates_interval);
        }
#if 0
    if (priv->gconf_notify_automatic_scrolling > 0)
        {
            gconf_client_notify_remove(priv->client,
                                       priv->gconf_notify_automatic_scrolling);
        }
#endif    
    if (priv->gconf_notify_open_feeds_to > 0)
        {
            gconf_client_notify_remove(priv->client,
                                       priv->gconf_notify_open_feeds_to);
        }
    g_free(error);
    g_object_unref(priv->client);
}


/***************************************************************************/
/* DBus handling */
/***************************************************************************/

/* Depending on the state of hw, do something */
static void
hw_event_handler(osso_hw_state_t * state, gpointer data)
{
    ULOG_DEBUG(__FUNCTION__);

    (void) data;

    g_assert(state != NULL);
    g_assert(RSS_IS_HOME_APPLET (data));

    HomeRssApplet *applet = RSS_HOME_APPLET (data);
    HomeRssAppletPrivate *priv = applet->priv;

    ULOG_DEBUG("Hardware event received: %d %d %d %d", state->save_unsaved_data_ind,
               state->memory_low_ind, state->save_unsaved_data_ind,
               state->system_inactivity_ind);

    if (state->save_unsaved_data_ind)
        {
        }
    if (state->memory_low_ind)
        {
            priv->startrss = TRUE;
        }
    else
        {

            priv->startrss = FALSE;

        }
    if (state->save_unsaved_data_ind)
        {
        }
}

gint event_deletion(gpointer data)
{
#if 1
	close_iphb_handle(iphb_applet_data);
	remove_iphb_callback(iphb_applet_data.source_id);
	iphb_applet_data.source_id = 0;
	//iphb_applet_data.user_data = NULL;
#endif	
	return FALSE;
}

	static void
_applet_display_handler(osso_display_state_t state, gpointer data)
{
    g_assert (RSS_IS_HOME_APPLET (data));

    HomeRssAppletPrivate *priv = (RSS_HOME_APPLET (data))->priv;

    if (state == OSSO_DISPLAY_OFF)
        {
            priv->animation = FALSE;
            stop_animation(RSS_HOME_APPLET (data));
	    /* This part of code will disable IPHB if display goes OFF.
	     * eventually network will not refresh if device is sleeping.
	     * Hence disabling the code for now. - Rama */
	    //g_timeout_add(200, event_deletion, NULL);
        }
    else if( state == OSSO_DISPLAY_ON )
        {
            priv->animation = FALSE; /* FIXME: scrolling support
                                        disabled, not all code removed
                                        anyway */
/*                 gconf_client_get_bool(priv->client, */
/*                                       RSS_SETTINGS_AUTOMATIC_SCROLLING, NULL); */

            restart_animation(RSS_HOME_APPLET (data), ANIMATION_TO_DOWN_TIMEOUT);
	    /* This part of code will disable IPHB if display goes OFF.
	     * eventually network will not refresh if device is sleeping.
	     * Hence disabling the code for now. - Rama */
	    //priv->htimeout = g_timeout_add(200, _setup_alarmd, data);	    
        }
}

/** The D-BUS messages are handled here
 *
 * @param connection the D-BUS connection
 * @param signal the incoming D-BUS message
 * @param user_data the applet structure
 * @return returns DBUS_HANDLER_RESULT_HANDLED on success and
 *         DBUS_HANDLER_RESULT_NOT_YET_HANDLED on failure
 */
static DBusHandlerResult
_applet_handle_signal_cb(DBusConnection *
                         connection, DBusMessage * signal, gpointer user_data)
{
    ULOG_DEBUG(__FUNCTION__);

    DBusHandlerResult msg_handled = DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
    HomeRssApplet *applet = (HomeRssApplet *) user_data;

    ULOG_DEBUG("DBus message: %s\n%s", dbus_message_get_member(signal),
               dbus_message_get_interface(signal));

    g_assert(NULL != connection);
    g_assert(NULL != signal);

    if (applet == NULL)
        {
            /* Why would this happen? */
            ULOG_DEBUG("Applet data is NULL");
            return msg_handled;
        }

    if (!dbus_message_is_method_call
        (signal, "com.nokia.statusbar", "statusbar_insensitive"))
        {
            if (dbus_message_is_signal(signal, RSS_INTERFACE,
                                       OSSO_RSS_FEED_READER_REFRESH_FINISHED_SIGNAL))
                {
                    ULOG_DEBUG("signal: refresh_finished");

                    _applet_refresh_completed(applet);
                    msg_handled = DBUS_HANDLER_RESULT_HANDLED;
                    applet->priv->refreshing = FALSE;
                }
            else if (dbus_message_is_signal(signal, RSS_INTERFACE,
                                            OSSO_RSS_FEED_READER_REFRESH_INTERRUPTED_SIGNAL))
                {
                    ULOG_DEBUG("signal: refresh_interrupted");
                    msg_handled = DBUS_HANDLER_RESULT_HANDLED;
                    _applet_refresh_completed(applet);
                    /*if (applet->refreshing)
                      {
                      applet->refreshing = FALSE;
                      redraw_refresh_button();
                      }*/
                }
            else if (dbus_message_is_signal(signal, RSS_INTERFACE,
                                            OSSO_RSS_FEED_READER_REFRESH_PREVENTED_SIGNAL))
                {
                    ULOG_DEBUG("signal: refresh_prevented");
                    /*             g_assert(NULL != applet->rss_image); */
                    _applet_refresh_completed(applet);
                    msg_handled = DBUS_HANDLER_RESULT_HANDLED;
                    /*if (applet->refreshing)
                      {
                      applet->refreshing = FALSE;
                      redraw_refresh_button();
                      }*/
                }
            else if (dbus_message_is_signal(signal, RSS_INTERFACE,
                                            OSSO_RSS_FEED_READER_FOLDER_READ_SIGNAL))
                {
                    ULOG_DEBUG("signal: folder_read");
                    msg_handled = DBUS_HANDLER_RESULT_HANDLED;
                }
            else if (dbus_message_is_signal(signal, RSS_INTERFACE,
                                            OSSO_RSS_FEED_READER_CONNECTING_STARTED))
                {
                    ULOG_DEBUG("signal: connecting_started");
                    applet->priv->refreshing = TRUE;
                    update_refresh_button_image (applet);
                    msg_handled = DBUS_HANDLER_RESULT_HANDLED;
                }
            else if (dbus_message_is_signal(signal, RSS_INTERFACE,
                                            OSSO_RSS_FEED_READER_REFRESH_STARTED))
                {
                    ULOG_DEBUG("refresh started");
                    applet->priv->refreshing = TRUE;
                    update_refresh_button_image (applet);
                    msg_handled = DBUS_HANDLER_RESULT_HANDLED;
                }

            else if (dbus_message_is_signal(signal, RSS_INTERFACE,
                                            OSSO_RSS_FEED_READER_ALARMDELETE_RSSSHUTDOWN_SIGNAL))
                {
                    ULOG_DEBUG("alarm deletion");
	    			g_timeout_add(200, event_deletion, NULL);
	    			applet->priv->htimeout = g_timeout_add(200, _setup_alarmd, user_data);	// Rama - commented
                    msg_handled = DBUS_HANDLER_RESULT_HANDLED;
                }
        }

    ULOG_DEBUG("end of _applet_handle_signal_cb");
    return msg_handled;
}

/** Gets a D-BUS connection
 * 
 * @param applet the applet structure
 * @param signal_handler the handler for the D-BUS connection
 * @return TRUE on success, FALSE on failure
 */
static gboolean
_applet_dbus_connect(HomeRssApplet * applet,
                     DBusHandleMessageFunction signal_handler)
{
    ULOG_DEBUG(__FUNCTION__);

    DBusError err;
    g_assert(applet);

    HomeRssAppletPrivate *priv = applet->priv;

    priv->osso = NULL;
    priv->connection = NULL;
    dbus_error_init(&err);

    if (applet != NULL && priv->connection == NULL)
        {
            priv->osso =
                osso_initialize("osso_rss_feed_reader_applet", PACKAGE_VERSION,
                                FALSE, NULL);
            g_assert(priv->osso != NULL);

            priv->connection =
                (DBusConnection *) osso_get_dbus_connection(priv->osso);

#ifdef APPLET_TEST
            //priv->connection = hd_status_plugin_item_get_dbus_connection (HD_HOME_PLUGIN_ITEM (Pointerapplet),
            //                                                          DBUS_BUS_SESSION, &err);
#endif

            g_assert(priv->connection != NULL);
            if (dbus_error_is_set (&err))
                {
                    g_warning ("Failed to open connection to Session Bus: %s", err.message);
                    dbus_error_free (&err);
                }


            /* Set handling changes in HW states. Note: not tested */
            osso_return_t ret = osso_hw_set_event_cb(priv->osso,
                                                     NULL, hw_event_handler,
                                                     applet);

            if (ret != OSSO_OK)
                {
                    ULOG_CRIT("Could not set callback for HW monitoring");
                }


            if (priv->connection == NULL || dbus_error_is_set(&err))
                {
                    dbus_error_free(&err);
                    return FALSE;
                }

            gchar *rule = NULL;

            rule = g_strdup_printf(APPLET_DBUS_SIGNAL_LISTENER_FMT,
                                   RSS_SERVICE, 
                                   RSS_INTERFACE);

            dbus_bus_add_match(priv->connection, rule, &err);

            if (dbus_error_is_set(&err))
                {
                    ULOG_ERR("Failed to set up DBus rule %s, error: %s", rule,
                             err.message);
                    dbus_error_free(&err);
                    g_free(rule);
                    return FALSE;
                }
            else
                {
                    dbus_connection_add_filter(priv->connection,
                                               signal_handler, applet, NULL);
                }
            g_free(rule);

            /* Finally, register the connection handling to main loop */
            dbus_connection_setup_with_g_main(priv->connection, NULL);
        }

    return TRUE;
}

/** The applet is refreshed by calling this function
 *
 * @param applet the applet structure
 * @param timed if the refresh should be timed or not
 * @return TRUE on success, FALSE on failure
 */
static gboolean
_applet_dbus_refresh(HomeRssApplet * applet, gboolean timed)
{
    ULOG_DEBUG(__FUNCTION__);

    gboolean timing = timed;

    if (applet != NULL && applet->priv->connection != NULL)
        {

            if (applet->priv->startrss == TRUE)
                {

                    ui_show_save_nomem_dialog(applet);

                    return TRUE;

                }

            DBusMessage *message = NULL;
            message =
                dbus_message_new_method_call
                (RSS_SERVICE,
                 RSS_OBJECT_PATH,
                 RSS_INTERFACE, OSSO_RSS_FEED_READER_REFRESH_SIGNAL);
            if (message == NULL)
                {
                    ULOG_ERR("Creating DBus refresh call failed");
                    return FALSE;
                }
            dbus_message_set_auto_start(message, TRUE);
            if (!dbus_message_append_args(message,
                                          DBUS_TYPE_BOOLEAN, &timing,
                                          DBUS_TYPE_INVALID))
                {
                    ULOG_ERR("Could not append args to message");
                }
            if (!_applet_create_temporary_file("background"))
                {
                    ULOG_ERR("Could not create temporary file \"background\"");
                    if (!applet->priv->backgrounded)
                        {
                            ui_show_save_nodevicemem_dialog(applet);
                        }
                    //call the refresh completed function, to restore the
                    //refresh image
                    _applet_refresh_completed(applet);
                    dbus_message_unref(message);
                    return TRUE;
                }
            if (!(dbus_connection_send(applet->priv->connection, message, NULL)))
                {
                    ULOG_ERR("Sending DBus refresh message failed");
                    dbus_message_unref(message);
                    return FALSE;
                }
            if (message != NULL)
                {
                    dbus_message_unref(message);
                }
        }
    else
        {
            ULOG_ERR
                ("Refresh message failed because applet is"
                 "not properly initialized");
        }
    return TRUE;
}

static gboolean applet_poll_iphb_fd(gpointer data)
{
	HomeRssApplet * applet = (HomeRssApplet *)(iphb_applet_data.user_data);
        GConfClient* gconf = gconf_client_get_default();
        int iphb_fd = gconf_client_get_int(gconf, RSS_SETTINGS_IPHB_HANDLE_ID, NULL);	
	fd_set fds;
	struct timeval time_wait;
        time_t cur_time = 0;	

	memset(&time_wait, 0, sizeof(struct timeval));

	time_wait.tv_sec = 60;
	time_wait.tv_usec = 0;

	if(0 >= iphb_fd) {
		return TRUE;
	}

	if(1/*CHECK IF DISPLAY IS ON */) {
		FD_ZERO(&fds);
		FD_SET((unsigned int)iphb_fd, &fds);
		if(select(iphb_fd+1,&fds,NULL,NULL,&time_wait)) {
			if( FD_ISSET(iphb_fd,&fds) ) {
				/* DO REFRESH */
                                cur_time = time(NULL);
                                gconf_client_set_int(gconf, RSS_SETTINGS_IPHB_LAST_REFRESH_TIME, cur_time, NULL);
				
				setup_alarm_event_now(TRUE, &iphb_applet_data);
				iphb_applet_data.source_id = setup_iphb_callback(&iphb_applet_data);
				
				if (_applet_dbus_refresh(applet, FALSE) == FALSE)
				{
					ULOG_ERR("User requested refresh DBus message failed");
				}
				return FALSE;    /* Continue the call callback */
			}
			else {
				return FALSE;     /* Continue the call callback */
			}
		}
	}
	return FALSE;
}

/** Calls the main application to display a selected feed
 *
 * @param applet the applet structure
 * @param feed the feed to display
 * @param item the item to display
 * @return TRUE on success, FALSE on failure
 */
static gboolean
_applet_dbus_display(HomeRssApplet * applet,
                     gchar * feed,
                     gchar * url,
                     guint32 item)
{
    ULOG_DEBUG(__FUNCTION__);


    g_assert(feed != NULL);

    if (applet != NULL && applet->priv->connection != NULL)
        {
            if (OPEN_TO_RSS == applet->priv->open_feeds_to)
                {
                    if (applet->priv->startrss == TRUE)
                        {
                            ui_show_save_nomem_dialog(applet);

                            return TRUE;
                        }

                    if (OSSO_OK != osso_rpc_async_run(applet->priv->osso,
                                                      RSS_SERVICE,
                                                      RSS_OBJECT_PATH,
                                                      RSS_INTERFACE,
                                                      OSSO_RSS_FEED_READER_DISPLAY_SIGNAL,
                                                      NULL, NULL,
                                                      DBUS_TYPE_STRING, feed,
                                                      DBUS_TYPE_UINT32, item,
                                                      DBUS_TYPE_INVALID, NULL))
                        {
                            ULOG_DEBUG("Could not send the display message");
                        }
                }
            else                    // opening feed in browser
                {
                    if (OSSO_OK != osso_rpc_run_with_defaults(applet->priv->osso,
                                                              "osso_browser",
                                                              OSSO_BROWSER_LOAD_URL_REQ,
                                                              NULL, DBUS_TYPE_STRING,
                                                              url, DBUS_TYPE_INVALID))
                        {
                            ULOG_DEBUG("Could not send the display message");
                        }
                }
        }
    else
        {
            ULOG_ERR
                ("Display message failed because applet"
                 "is not properly initialized");
        }

    return TRUE;
}


/***************************************************************************/
/* UI / Callbacks handling */
/***************************************************************************/

/***************************************************************************/
/* UI / Callbacks handling -- helper functions */
/***************************************************************************/
/* This method is used to update the line_height and
   avg_font_width. This last is used on the break point compute*/

static void
update_font_width(HomeRssApplet * applet)
{
    ULOG_DEBUG(__FUNCTION__);

    g_assert (RSS_IS_HOME_APPLET (applet));

    PangoContext *context;
    PangoFontMetrics *metrics;
    PangoFontDescription *font_desc;
    HomeRssAppletPrivate *priv = applet->priv;
    GtkWidget *feed_view = NULL;

    feed_view = GTK_WIDGET (priv->feed_list->data);

    /* In previous versions a merge was made between the font desc of
       the treeview and the widget itself, it was simplified when the
       treeview was removed */
    font_desc =
        pango_font_description_copy(feed_view->style->font_desc);

    context = gtk_widget_get_pango_context(feed_view);

    metrics = pango_context_get_metrics(context,
                                        font_desc,
                                        pango_context_get_language(context));
    priv->line_height = (pango_font_metrics_get_ascent(metrics) +
                         pango_font_metrics_get_descent(metrics)) /
        PANGO_SCALE + 2;
    pango_font_metrics_unref(metrics);

    ULOG_DEBUG("Character height: %d", priv->line_height);
    pango_font_description_set_size(font_desc,
                                    pango_font_description_get_size(font_desc)
                                    / (float) 1.2);

    metrics = pango_context_get_metrics(context,
                                        font_desc,
                                        pango_context_get_language(context));
    priv->avg_char_width =
        pango_font_metrics_get_approximate_char_width(metrics) / PANGO_SCALE +
        2;
    ULOG_DEBUG("Average character width: %d", priv->avg_char_width);
    pango_font_metrics_unref(metrics);
    pango_font_description_free(font_desc);
    /*     pango_font_description_free(font2); */
}

/*
 * Calculates which byte the text is needed to break in order to make the wrapping
 * NOTE: this code is a good candidate to be moded to RssFeedAppletView
 */
static gint
calc_breaking_position(HomeRssApplet *applet,
                       RssFeedAppletView *feed_view,
                       gchar *text)
{
    PangoContext *context;
    PangoLayout *layout;
    gint offs;
    gint trailing;
    gint width;
    gchar *brk;
    HomeRssAppletPrivate *priv = NULL;

    ULOG_DEBUG(__FUNCTION__);
    g_assert (RSS_IS_HOME_APPLET (applet));
    g_assert (RSS_FEED_APPLET_VIEW (feed_view));

    priv = applet->priv;
    width = HEADLINE_WIDTH;

    context = gtk_widget_get_pango_context(feed_view->title);
    layout = pango_layout_new(context);
    pango_layout_set_markup(layout, text, -1);
    ULOG_DEBUG("pango x: %d", width - priv->avg_char_width);
    pango_layout_xy_to_index(layout,
                             (width - 2*priv->avg_char_width) * PANGO_SCALE,
                             0, &offs, &trailing);
    g_object_unref(layout);
    ULOG_DEBUG("xy index %d trailing %d", offs, trailing);
    brk = g_utf8_offset_to_pointer(text + offs, trailing);
    ULOG_DEBUG("byte offset: %d", brk - text);
    return brk - text;
}

/* puts a newline at a space before offset (counted in bytes) or 
 * at offset it there's no space before that */
static gchar *
put_newline(gchar *title, gint offset)
{
    gint i;

    ULOG_DEBUG(__FUNCTION__);

    if (NULL == title)
        {
            return NULL;
        }
    
    if (offset >= strlen(title))
        {
            return title;
        }

    i = offset;
    while (i > 0 && title[i] != ' ')
        --i;
    
    if (i > 0)
        {
            title[i] = '\n';
            return title;
        }

    /* there was no space inside. we wrap the text simply */
    gchar *part1 = title;
    gchar *part2 = g_strdup(title + offset);
    part1[offset] = '\n';
    part1[offset+1] = 0;
    title = g_strjoin(NULL, part1, part2, NULL);
    g_free(part1);
    g_free(part2);

    return title;
}

/** Creates an empty file
 *
 * @param file the filename to create
 */
static int
_applet_create_temporary_file(gchar * file)
{
    ULOG_DEBUG(__FUNCTION__);

    gchar *filename = NULL;
    FILE *tmpf;
    filename = g_strdup_printf("%s" G_DIR_SEPARATOR_S
                               ".osso_rss_feed_reader"
                               G_DIR_SEPARATOR_S "%s",
                               g_get_home_dir(), file);
    test_if_cache_directory_exists();
    tmpf = fopen(filename, "wb");
    if (tmpf == NULL)
        {
            g_free(filename);
            return 0;
        }
    fclose(tmpf);
    g_free(filename);
    return 1;
}

/***************************************************************************/
/* UI / Callbacks handling -- Callbacks */
/***************************************************************************/

static gboolean
dialog_key_press(GtkWidget * widget, GdkEventKey * event, gpointer data)
{
    ULOG_DEBUG(__FUNCTION__);

    (void) data;

    if (event->state & (GDK_CONTROL_MASK |
                        GDK_SHIFT_MASK |
                        GDK_MOD1_MASK |
                        GDK_MOD3_MASK |
                        GDK_MOD4_MASK |
                        GDK_MOD5_MASK))
        {
            return FALSE;
        }

    switch (event->keyval)
        {
        case GDK_Escape:
            return FALSE;
            break;
        default:
            break;
        }

    return FALSE;
}

#ifdef APPLET_TEST
static void
on_menuitem_settings(GtkWidget * widget, gpointer user_data)
{
    ULOG_DEBUG(__FUNCTION__);

    execute_rss_settings(priv->osso, NULL, TRUE, TRUE, &iphb_applet_data);
}
#endif

/***************************************************************************/
/* UI / Callbacks handling -- UI creation */
/***************************************************************************/

static void
create_headline_display(HomeRssApplet * applet)
{
    gint i = 0;
    GtkWidget *new_feed = NULL;

    ULOG_DEBUG(__FUNCTION__);

    applet->priv->rss_news = gtk_vbox_new (FALSE, 0);
    gtk_box_set_homogeneous (GTK_BOX (applet->priv->rss_news), FALSE);
    gtk_widget_set_app_paintable (applet->priv->rss_news, TRUE);
    g_signal_connect (G_OBJECT (applet->priv->rss_news), "size-allocate",
                      G_CALLBACK (_size_allocate_cb), applet);
    gtk_widget_show (applet->priv->rss_news);

    gtk_container_add (GTK_CONTAINER (applet->priv->global_vbox),
                       applet->priv->rss_news);

    gtk_widget_set_size_request(applet->priv->applet_alignment,
                                RSS_APPLET_WIDTH,
                                RSS_APPLET_HEIGHT);

    gtk_widget_set_size_request(applet->priv->rss_news,
                                RSS_APPLET_WIDTH,
                                RSS_APPLET_HEAD_HEIGHT);

    ULOG_DEBUG ("HEADLINE SIZES = (%i, %i, %i)", HEADLINE_MIN_WIDTH, HEADLINE_WIDTH, HEADLINE_HEIGHT);

    for (i = 0; i < MAX_HEADLINES; i++)
      {
        new_feed = rss_feed_applet_view_new ();
        gtk_widget_set_size_request (new_feed, -1, HEADLINE_HEIGHT);
        applet->priv->feed_list = g_list_append (applet->priv->feed_list, new_feed);
        gtk_box_pack_start (GTK_BOX (applet->priv->rss_news), new_feed, FALSE, FALSE, 0);
        g_signal_connect (G_OBJECT (new_feed), "clicked",
                          G_CALLBACK (feed_view_clicked_cb), applet);
      }
}

void
create_buttons(HomeRssApplet * applet)
{
    ULOG_DEBUG(__FUNCTION__);

    /* Creating box contatining buttons */
    applet->priv->button_hbox = gtk_hbox_new (TRUE, 0);
    g_object_set (applet->priv->button_hbox, "border-width", 0, NULL);
    gtk_box_pack_start(GTK_BOX(applet->priv->global_vbox),
                       applet->priv->button_hbox, FALSE, FALSE,
                       0);
    gtk_widget_show(applet->priv->button_hbox);

    /* Creating buttons */
    applet->priv->move_up_button = rss_feed_applet_button_new ();
    applet->priv->move_down_button = rss_feed_applet_button_new ();
    applet->priv->refresh_button = rss_feed_applet_button_new ();

    rss_feed_applet_button_set_icon_align (RSS_FEED_APPLET_BUTTON (applet->priv->move_up_button),
                                           0.5, 0.0);
    rss_feed_applet_button_set_icon_align (RSS_FEED_APPLET_BUTTON (applet->priv->move_down_button),
                                           0.5, 0.0);
    rss_feed_applet_button_set_icon_align (RSS_FEED_APPLET_BUTTON (applet->priv->refresh_button),
                                           0.5, 0.0);
    rss_feed_applet_button_set_icon_padding (RSS_FEED_APPLET_BUTTON (applet->priv->move_up_button),
                                             4, 0);
    rss_feed_applet_button_set_icon_padding (RSS_FEED_APPLET_BUTTON (applet->priv->move_down_button),
                                             -4, 0);
    rss_feed_applet_button_set_icon_padding (RSS_FEED_APPLET_BUTTON (applet->priv->refresh_button),
                                             0, 0);
    /* Note: The specific pixbufs for the button are not being loaded
       here, as they depend on the theme. They are set on the
       style_set_cb callback */

    gtk_box_pack_start(GTK_BOX(applet->priv->button_hbox),
                       applet->priv->move_up_button, FALSE, FALSE,
                       0);
    gtk_box_pack_start(GTK_BOX(applet->priv->button_hbox),
                       applet->priv->refresh_button, FALSE, FALSE,
                       0);
    gtk_box_pack_start(GTK_BOX(applet->priv->button_hbox),
                       applet->priv->move_down_button, FALSE, FALSE,
                       0);
    gtk_widget_show_all (applet->priv->button_hbox);

    g_signal_connect(G_OBJECT(applet->priv->refresh_button), "clicked",
                     G_CALLBACK(refresh_button_clicked_cb), applet);
    g_signal_connect(G_OBJECT(applet->priv->move_up_button), "clicked",
                     G_CALLBACK(up_button_clicked_cb), applet);
    g_signal_connect(G_OBJECT(applet->priv->move_down_button), "clicked",
                     G_CALLBACK(down_button_clicked_cb), applet);
}

static void
create_common_view(HomeRssApplet *applet)
{
    ULOG_DEBUG(__FUNCTION__);

    GtkWidget *applet_alignment = NULL;
    GtkWidget *empty_label = NULL;

    /* applet alignment */
    applet_alignment = gtk_widget_new (GTK_TYPE_ALIGNMENT,
                                       "bottom-padding",
                                       HILDON_MARGIN_DEFAULT,
                                       "top-padding",
                                       HILDON_MARGIN_DEFAULT,
                                       "left-padding",
                                       HILDON_MARGIN_DEFAULT,
                                       "right-padding",
                                       HILDON_MARGIN_DEFAULT,
                                       NULL);

    gtk_alignment_set (GTK_ALIGNMENT (applet_alignment), 0.5, 0.5, 1.0, 1.0);
    gtk_widget_set_app_paintable (applet_alignment, TRUE);
    applet->priv->applet_alignment = applet_alignment;
    gtk_widget_show (applet_alignment);

    /* global vbox */
    applet->priv->global_vbox = gtk_vbox_new(FALSE, 0);
    g_object_set(applet->priv->global_vbox,
                 "homogeneous", FALSE, NULL);
    gtk_widget_set_app_paintable(applet->priv->global_vbox, TRUE);

    gtk_container_add(GTK_CONTAINER(applet_alignment),
                      applet->priv->global_vbox);
    gtk_widget_show (applet->priv->global_vbox);


    /* Scrolled window: scrolling support removed, scrolled window not used */
    create_headline_display (applet);
    create_buttons (applet);

    /* Creation empty label (displayed when no headlines are present) */
    empty_label = gtk_label_new(_("rss_ia_no_posts"));
    applet->priv->empty_label = empty_label;
    g_object_ref(empty_label);

    gtk_box_pack_start(GTK_BOX(applet->priv->rss_news), empty_label, TRUE, TRUE, 0);

    _applet_gconf_init(applet);
    {
        /* Check the gconf value at startup in case the applet was
         * started during the refresh operation.
         */
        gint refresh_state =
            gconf_client_get_int(applet->priv->client, REFRESH_STATE, NULL);
        applet->priv->refreshing = (refresh_state == 1);
        update_refresh_button_image (applet);
    }


    _applet_dbus_connect(applet, (DBusHandleMessageFunction) _applet_handle_signal_cb);

    update_font_width (applet);
    applet_check_animation (applet);

    gtk_container_add (GTK_CONTAINER (applet), applet->priv->applet_alignment);
}

static void
ui_show_save_nomem_dialog(HomeRssApplet * applet)
{
    ULOG_DEBUG(__FUNCTION__);

    if (applet->priv->save_nomem_dialog_visible==TRUE) {
        return;
    } else {
        GtkWidget *note = NULL;
        GtkWindow *parent = NULL;

        parent = GTK_WINDOW (gtk_widget_get_ancestor (applet->priv->applet_alignment, GTK_TYPE_WINDOW));

        note = hildon_note_new_information (parent,
                                            dgettext("hildon-common-strings",
                                                     "sfil_ni_not_enough_memory"));

        hildon_note_set_button_text(HILDON_NOTE (note), dgettext("hildon-common-strings",
                                                                 "sfil_ni_not_enough_memory_ok"));
        g_signal_connect(G_OBJECT(note), "key_press_event", G_CALLBACK(dialog_key_press), NULL);
        applet->priv->save_nomem_dialog_visible = TRUE;
        gtk_dialog_run(GTK_DIALOG(note));
        gtk_widget_destroy(GTK_WIDGET(note));
        applet->priv->save_nomem_dialog_visible = FALSE;
    }

}

static void
ui_show_save_nodevicemem_dialog(HomeRssApplet * applet)
{
    ULOG_DEBUG(__FUNCTION__);

    if (applet->priv->save_nodevicemem_dialog_visible==TRUE) {
        return;
    } else {
        GtkWidget *note = NULL;
        GtkWindow *parent = NULL;

        parent = GTK_WINDOW (gtk_widget_get_ancestor (applet->priv->applet_alignment, GTK_TYPE_WINDOW));

        note = hildon_note_new_information (parent, dgettext("ke-recv", "cerm_device_memory_full"));

        hildon_note_set_button_text(HILDON_NOTE (note), dgettext("hildon-common-strings",
                                                                 "sfil_ni_not_enough_memory_ok"));
        g_signal_connect(G_OBJECT(note), "key_press_event",
                         G_CALLBACK(dialog_key_press), NULL);
        applet->priv->save_nodevicemem_dialog_visible = TRUE;
        gtk_dialog_run(GTK_DIALOG(note));
        gtk_widget_destroy(GTK_WIDGET(note));
        applet->priv->save_nodevicemem_dialog_visible = FALSE;
    }
}

/***************************************************************************/
/* UI / Callbacks handling -- Headline list handling */
/***************************************************************************/
/** Shows or hides the empty label or the news.
 * @param show if the empty label should be shown or not
 */
static void
_applet_show_empty_label(HomeRssApplet * applet, gboolean show)
{
    ULOG_DEBUG(__FUNCTION__);

    if (show)
        {
            gtk_widget_show(applet->priv->empty_label);
        }
    else
        {
            gtk_widget_hide(applet->priv->empty_label);
        }
}

/** Set the list of feed headlines for the applet
 *
 * @param applet the applet structure
 */
static void
_applet_set_headlines(HomeRssApplet * applet)
{
    ULOG_DEBUG(__FUNCTION__);

    GList *items = NULL;
    GList *list_iter = NULL;
    GdkPixbuf *pixbuf = NULL;
    GdkPixbuf *default_pixbuf = NULL;
    GtkIconTheme *icon_theme = gtk_icon_theme_get_default();
    HeadLine *headline = NULL;
    gchar *item_title = NULL;
    GtkWidget *current_feed = NULL;
    HomeRssAppletPrivate *priv = applet->priv;

    g_assert (RSS_IS_HOME_APPLET (applet));

    priv->num_of_headlines = 0;

    items = feed_load_all(&(priv->updated));
    priv->loaded_headlines = g_list_length (items);
    /* Load only the first MAX_LOADED_HEADLINES items */
    if (priv->loaded_headlines > MAX_LOADED_HEADLINES)
       {
           items = g_list_reverse (items);
           while (priv->loaded_headlines > MAX_LOADED_HEADLINES)
               {
                   freeHeadLine ((HeadLine *) items->data);
                   items = g_list_delete_link (items, items);
                   priv->loaded_headlines--;
               }
           items = g_list_reverse (items);
       }

    default_pixbuf = gtk_icon_theme_load_icon(icon_theme, RSS_ICON_ITEM,
                                              HILDON_ICON_PIXEL_SIZE_SMALL, 0,
                                              NULL);
    rss_home_applet_hide_feed_list (applet);
    list_iter = g_list_nth (items, priv->first_headline_index);

    while ((NULL != list_iter)&&(priv->num_of_headlines < MAX_HEADLINES))
        {
            headline = (HeadLine *) list_iter->data;
            if (headline)
                {
                    current_feed = g_list_nth_data (priv->feed_list, priv->num_of_headlines);

                    /* Processing headline information */
                    if (headline->title)
                        {
                            item_title = g_markup_escape_text(headline->title, -1);
                            item_title = g_strstrip(item_title);
                            item_title = g_strdelimit(item_title, "\r\n\t", ' ');
                            gint brk = calc_breaking_position(applet,
                                                              RSS_FEED_APPLET_VIEW (current_feed),
                                                              item_title);
                            ULOG_DEBUG("The string: %s", item_title);
                            item_title = put_newline(item_title, brk);
                            ULOG_DEBUG("wrapped as:\n>%s<", item_title);
                        }
                    else
                        {
                            item_title = g_strdup(" ");
                        }

                    if (headline->icon == NULL)
                        pixbuf = default_pixbuf;
                    else
                        pixbuf = headline->icon;

                    ULOG_DEBUG("adding title %d %s", priv->num_of_headlines, item_title);

                    rss_feed_applet_view_set_image_from_pixbuf (RSS_FEED_APPLET_VIEW (current_feed),
                                                                pixbuf, pixbuf);
                    rss_feed_applet_view_set_title_markup (RSS_FEED_APPLET_VIEW (current_feed), item_title);
                    rss_feed_applet_view_set_url (RSS_FEED_APPLET_VIEW (current_feed), headline->url);
                    rss_feed_applet_view_set_feed (RSS_FEED_APPLET_VIEW (current_feed), headline->feed);
                    rss_feed_applet_view_set_nr (RSS_FEED_APPLET_VIEW (current_feed), headline->nr);
                    gtk_widget_show (current_feed);

                    g_free(item_title);
                    /* We don't need to free the pixbuf icon, although
                       _view_set_image adds one, it is made on the
                       freeHeadLine (see below)*/
                    priv->num_of_headlines = priv->num_of_headlines + 1;
                }
            list_iter = g_list_next(list_iter);
        }

    /* All the items where leaking memory. This propably fixes those leaks  */

    if (items != NULL ) list_iter = g_list_first(items);
    while (NULL != list_iter)
        {
            headline = (HeadLine *) list_iter->data;
            if (headline)
                {
                    freeHeadLine(headline);
                }
            list_iter = g_list_next(list_iter);
        }

    g_list_free(items);

    if (NULL != default_pixbuf) g_object_unref(default_pixbuf);

    /* Showing empty label or list */
    _applet_show_empty_label(applet,
                             0 == priv->num_of_headlines);
}

/** This is called after the refresh is completed in the main application
 *
 * @param applet the applet structure
 */
static void
_applet_refresh_completed(HomeRssApplet * applet)
{
    ULOG_DEBUG(__FUNCTION__);

    g_assert(applet != NULL);
    /*     g_assert(applet->rss_image != NULL); */

    if (applet->priv->refreshing)
        {
            applet->priv->refreshing = FALSE;
            update_refresh_button_image (applet);
        }

    _applet_set_headlines(applet);

    applet_check_animation(applet);
}

/***************************************************************************/
/* Animation handling */
/***************************************************************************/


static gboolean
Start_TW_Scroll_again(gpointer data)
{
    ULOG_DEBUG(__FUNCTION__);

    Scroll_Down_TW(NULL);

    g_return_val_if_fail (RSS_IS_HOME_APPLET (data), FALSE);

    HomeRssApplet *applet = RSS_HOME_APPLET (data);

    if (!applet->priv->scroll_down_tw_timeout && applet->priv->animation)
        {
            applet->priv->scroll_down_tw_timeout =
                g_timeout_add(ANIMATION_TO_DOWN_TIMEOUT, Scroll_Down_TW, NULL);
        }
    applet->priv->scroll_down_tw_starter_timeout = 0;
    return FALSE;
}

static gboolean
Scroll_Down_TW(gpointer data)
{
    ULOG_DEBUG(__FUNCTION__);

    /* FIXME: re-implement if required */

    return TRUE;
}

static void
stop_animation(HomeRssApplet *applet)
{
    ULOG_DEBUG(__FUNCTION__);

    if (applet->priv->scroll_down_tw_timeout)
        {
            g_source_remove(applet->priv->scroll_down_tw_timeout);
            applet->priv->scroll_down_tw_timeout = 0;
        }
    if (applet->priv->scroll_down_tw_starter_timeout)
        {
            g_source_remove(applet->priv->scroll_down_tw_starter_timeout);
            applet->priv->scroll_down_tw_starter_timeout = 0;
        }
}

static void
restart_animation (HomeRssApplet *applet, gdouble timeout)
{
    ULOG_DEBUG(__FUNCTION__);

    ULOG_DEBUG("restart_animation called, applet backgrounded: %d "
               "timeout: %d, animation: %d", 
               applet->priv->backgrounded,
               applet->priv->scroll_down_tw_starter_timeout,
               applet->priv->animation);

    if (!applet->priv->scroll_down_tw_starter_timeout
        && applet->priv->animation && !applet->priv->backgrounded /*&& GTK_WIDGET_REALIZED(priv->rss_news)*/)
        {
            applet->priv->scroll_down_tw_starter_timeout =
                g_timeout_add(timeout,
                              Start_TW_Scroll_again, applet);
        }
}

static void
applet_check_animation(HomeRssApplet * applet)
{
    ULOG_DEBUG(__FUNCTION__);

    if (applet->priv->num_of_headlines >= applet->priv->displayed_headlines)
        {
            ULOG_DEBUG("scrollbar visible");

            /* Animation needs to be restarted but only if it's switched on in
             * setup */
            applet->priv->animation = FALSE; /* FIXME: scrolling
                                                support disabled, not
                                                all code removed
                                                anyway */
/*                 gconf_client_get_bool(applet->priv->client, */
/*                                       RSS_SETTINGS_AUTOMATIC_SCROLLING, NULL); */
        }
    else
        {
            ULOG_DEBUG("invisible");

            applet->priv->animation = FALSE;
        }

    if (applet->priv->animation)
        {
            ULOG_DEBUG("animation on -- restarting");
            restart_animation(applet, ANIMATION_TO_DOWN_WAIT);
        }
    else
        {
            ULOG_DEBUG("animation on -- stopping");
            stop_animation(applet);
        }
}

/**************************************************************************
 * Continous scrolling
 **************************************************************************/
static void
rss_home_applet_realize (GtkWidget *widget)
{
    GTK_WIDGET_CLASS (rss_home_applet_parent_class)->realize (widget);
}

static gboolean
rss_home_applet_expose_event (GtkWidget      *widget,
                              GdkEventExpose *event)
{
    HomeRssApplet *applet = RSS_HOME_APPLET (widget);
    cairo_t *cr = NULL;

    cr = gdk_cairo_create(widget->window);

    cairo_set_operator (cr, CAIRO_OPERATOR_SOURCE);
    gdk_cairo_region (cr, event->region);
    cairo_fill (cr);

    if (theme_pixbufs_loaded (applet))
        draw_themed_background (applet, cr, &event->area);

    cairo_destroy (cr);

/*     paint_transparent_widget (widget, event, 0.0, 0.0, 0.4, 0.7); */

    return GTK_WIDGET_CLASS (rss_home_applet_parent_class)->expose_event (widget,
                                                                          event);
}

static void
rss_home_applet_change_colormap (GtkWidget *widget)
{
    GdkScreen *screen = NULL;
    GdkColormap *rgba = NULL;

    screen = gtk_widget_get_screen (widget);
    rgba = gdk_screen_get_rgba_colormap (screen);
    gtk_widget_set_colormap (widget, rgba);
    gtk_widget_set_app_paintable(widget, TRUE);
}

static void
draw_themed_background(HomeRssApplet *applet,
                       cairo_t *cr,
                       GdkRectangle *area)
{
    gdk_cairo_rectangle(cr, area);
    cairo_clip(cr);
    cairo_set_operator(cr, CAIRO_OPERATOR_SOURCE);
    gdk_cairo_set_source_pixbuf (cr, applet->priv->background, 0.0, 0.0);
    cairo_paint(cr);
}

static gboolean
theme_pixbufs_loaded (HomeRssApplet *applet)
{
    return applet->priv->background != NULL;
}

static void rss_home_applet_hide_feed_list (HomeRssApplet *applet)
{
  g_list_foreach (applet->priv->feed_list, (GFunc) gtk_widget_hide, NULL);
}

/**
 * Feed view clicked
 */
static void
feed_view_clicked_cb (GtkWidget *self,
                      gpointer user_data)
{
    RssFeedAppletView *feed_view = NULL;
    gchar *feed = NULL;
    gchar *url = NULL;
    gulong nr = 0;

    g_return_if_fail (RSS_IS_FEED_APPLET_VIEW (self));
    feed_view = RSS_FEED_APPLET_VIEW (self);

    feed = (gchar*) rss_feed_applet_view_get_feed (feed_view);
    url = (gchar*) rss_feed_applet_view_get_url (feed_view);
    nr = rss_feed_applet_view_get_nr (feed_view);

    if (_applet_dbus_display(user_data, feed, url, nr) == FALSE)
        {
            ULOG_ERR("Sending DBus message to display news item failed");
        }
}

/* As we require wrapping and ellipsize, we need the final values, when the
 * elements have allocated his values
 */
static void
_size_allocate_cb (GtkWidget * widget,
                   GtkAllocation *allocation,
                   gpointer user_data)
{
    ULOG_DEBUG(__FUNCTION__);

    HomeRssApplet *applet = NULL;
    HomeRssAppletPrivate *priv = NULL;
    gint width = HEADLINE_WIDTH;

    g_return_if_fail (RSS_IS_HOME_APPLET (user_data));
    applet = RSS_HOME_APPLET (user_data);
    priv = applet->priv;

    ULOG_DEBUG ("prev=(%i,%i), allocation=(%i, %i), width=(%i)",
                priv->prev_width, priv->prev_height,
                allocation->width, allocation->height,
                width);

    if ((priv->prev_height != allocation->height && priv->prev_width)
        || (priv->prev_width != allocation->width && width))
        {
            gint new_sw_height =
                /* allocation->height - BUTTON_HEIGHT - 2.5*HILDON_MARGIN_DOUBLE; */
                allocation->height;

            update_font_width(applet);

            priv->prev_width = allocation->width;
            priv->prev_height = allocation->height;

            priv->displayed_headlines = new_sw_height / HEADLINE_HEIGHT;
            ULOG_DEBUG ("Displayed headlines = %i", priv->displayed_headlines);

            /* We call again _applet_set_headlines as size data is used there to
               properly set the title */
            _applet_set_headlines(applet);

            applet_check_animation(applet);
        }
}

static GdkPixbuf *
load_pixbuf_from_file (const gchar *name)
{
    GError *error = NULL;
    GdkPixbuf *pixbuf;    

    pixbuf = gdk_pixbuf_new_from_file (name, &error);    
    if (error != NULL)
        {
            g_warning ("Unable to load theme file: %s", error->message);		
            g_error_free (error);
            error = NULL;
        }
    return pixbuf;
}

/* Theme issues */
static void
load_theme_pixbufs (HomeRssApplet *applet)
{
    GdkPixbuf *refresh_img, *refresh_pressed_img;
    GdkPixbuf *up_img, *up_pressed_img;
    GdkPixbuf *down_img, *down_pressed_img;

    /* background */
    applet->priv->background = load_pixbuf_from_file (HEAD_BG);

    /* refresh button */
    refresh_img = load_pixbuf_from_file (REFRESH_BUTTON);
    refresh_pressed_img = load_pixbuf_from_file (REFRESH_BUTTON_PRESSED);

    /* Up button */
    up_img = load_pixbuf_from_file (UP_BUTTON);
    up_pressed_img = load_pixbuf_from_file (UP_BUTTON_PRESSED);

    /* Down button */
    down_img = load_pixbuf_from_file (DOWN_BUTTON);
    down_pressed_img = load_pixbuf_from_file (DOWN_BUTTON_PRESSED);

    /* Add images to buttons */
    rss_feed_applet_button_set_image_from_pixbuf (RSS_FEED_APPLET_BUTTON (applet->priv->refresh_button),
                                                  refresh_img, refresh_pressed_img);
    rss_feed_applet_button_set_image_from_pixbuf (RSS_FEED_APPLET_BUTTON (applet->priv->move_up_button),
                                                  up_img, up_pressed_img);
    rss_feed_applet_button_set_image_from_pixbuf (RSS_FEED_APPLET_BUTTON (applet->priv->move_down_button),
                                                  down_img, down_pressed_img);

    /* The RSS applet button already holds references to these images */
    g_object_unref (refresh_img);
    g_object_unref (refresh_pressed_img);
    g_object_unref (up_img);
    g_object_unref (up_pressed_img);    
    g_object_unref (down_img);
    g_object_unref (down_pressed_img);    
}

static GdkPixbuf *
load_theme_icon_by_name (const gchar *name)
{
    GdkPixbuf *icon;
    GError *error = NULL;
    GtkIconTheme *theme = gtk_icon_theme_get_default ();

    icon = gtk_icon_theme_load_icon (theme, name, 48, 0, &error);    

    if (error != NULL)
      {
          g_warning ("Unable to load icon: %s", error->message);	      
          g_error_free (error);
          error = NULL;
      }

    return icon;
}

static void
load_theme_icons (HomeRssApplet *applet)
{
    GdkPixbuf *up_img, *down_img;

    /* Load images for the refresh button */
    applet->priv->refresh_img = load_theme_icon_by_name ("general_refresh");
    applet->priv->refreshing_img = load_theme_icon_by_name ("general_stop");

    /* Load images for the up and down buttons */
    up_img = load_theme_icon_by_name ("rss_reader_move_up");
    down_img = load_theme_icon_by_name ("rss_reader_move_down");

    /* Set all images */
    update_refresh_button_image (applet);
    rss_feed_applet_button_set_icon_from_pixbuf (RSS_FEED_APPLET_BUTTON (applet->priv->move_up_button),
                                                 up_img, up_img);
    rss_feed_applet_button_set_icon_from_pixbuf (RSS_FEED_APPLET_BUTTON (applet->priv->move_down_button),
                                                 down_img, down_img);

    /* Free unneeded images */
    g_object_unref (up_img);
    g_object_unref (down_img);
}

static void
update_refresh_button_image (HomeRssApplet *applet)
{
    ULOG_DEBUG(__FUNCTION__);

    if (applet->priv->refreshing)
    {
        rss_feed_applet_button_set_icon_from_pixbuf (RSS_FEED_APPLET_BUTTON (applet->priv->refresh_button),
                                                     applet->priv->refreshing_img,
                                                     applet->priv->refreshing_img);
    }
    else
    {
        rss_feed_applet_button_set_icon_from_pixbuf (RSS_FEED_APPLET_BUTTON (applet->priv->refresh_button),
                                                     applet->priv->refresh_img,
                                                     applet->priv->refresh_img);
    }
}

static void
free_theme_pixbufs (HomeRssApplet *applet)
{
    if (applet->priv->background)
        {
            g_object_unref (applet->priv->background);
            applet->priv->background = NULL;
        }
    /* Button img pixbufs are not being freed here, as we don't keep a
       extra reference on the applet. */

    if (applet->priv->refresh_img)
        {
            g_object_unref (applet->priv->refresh_img);
            applet->priv->refresh_img = NULL;
        }
    if (applet->priv->refreshing_img)
        {
            g_object_unref (applet->priv->refreshing_img);
            applet->priv->refreshing_img = NULL;
        }
}

/**/
static gboolean
style_set_cb (GtkWidget *widget,
              GtkStyle *previous_style,
              gpointer user_data)
{
    HomeRssApplet *applet = RSS_HOME_APPLET (user_data);
    GdkColor color;

    /* pixbufs */
    free_theme_pixbufs (applet);
    load_theme_pixbufs (applet);
    load_theme_icons (applet);

    /* hightlight color */
    if (gtk_style_lookup_color (widget->style, "SelectionColor", &color) == TRUE)
        {
            g_list_foreach (applet->priv->feed_list, (GFunc) feed_view_set_pressed_color, &color);
        }
    else
        {
            /* We don't launch any warning because the set-style could
               be called several times when the theme change, and
               sometimes it has no a proper "SelectionColor" */
        }


    return FALSE;
}

static void
feed_view_set_pressed_color (RssFeedAppletView *feed,
                             GdkColor *color)
{
    gfloat r;
    gfloat g;
    gfloat b;
    gfloat a;

    rss_feed_applet_view_get_pressed_color (feed, NULL, NULL, NULL, &a);

    r = (float) color->red / G_MAXUINT16;
    g = (float) color->green / G_MAXUINT16;
    b = (float) color->blue / G_MAXUINT16;
    
    rss_feed_applet_view_set_pressed_color (feed, r, g, b, a);
}

static void
refresh_button_clicked_cb (GtkWidget *self,
                           gpointer user_data)
{
    HomeRssApplet *applet = NULL;

    g_return_if_fail (RSS_IS_FEED_APPLET_BUTTON (self));
    g_return_if_fail (RSS_IS_HOME_APPLET (user_data));

    applet = RSS_HOME_APPLET (user_data);

    applet->priv->refreshing = !applet->priv->refreshing;

    update_refresh_button_image (applet);

    if (_applet_dbus_refresh (applet, FALSE) == FALSE)
        {
            ULOG_ERR ("User requested refresh DBus message failed");
        }
}

static void
up_button_clicked_cb (GtkWidget *self,
                      gpointer user_data)
{
    HomeRssApplet *applet = RSS_HOME_APPLET (user_data);
    HomeRssAppletPrivate *priv = applet->priv;
    if (priv->first_headline_index == 0)
        {
            priv->first_headline_index = MAX (0, priv->loaded_headlines - MAX_HEADLINES);
            if (priv->first_headline_index % SCROLL_STEPS != 0)
                {
                    priv->first_headline_index += SCROLL_STEPS - (priv->first_headline_index % SCROLL_STEPS);
                }
        }
    else
        {
            priv->first_headline_index = MAX (0, priv->first_headline_index - SCROLL_STEPS);
        }
    _applet_set_headlines (applet);
}

static void
down_button_clicked_cb (GtkWidget *self,
                        gpointer user_data)
{
    HomeRssApplet *applet = RSS_HOME_APPLET (user_data);
    HomeRssAppletPrivate *priv = applet->priv;
    if (priv->first_headline_index + MAX_HEADLINES >= priv->loaded_headlines)
        {
            priv->first_headline_index = 0;
        }
    else
        {
            priv->first_headline_index += SCROLL_STEPS;
        }
    _applet_set_headlines (applet);
}
