#include "pluginSupport.h"

// Contents of this file should be compiled only if we are enabling plugin
// support.
#ifdef PLUGIN_SUPPORT_ENABLED

#include <nsIDOMWindow.h>
#include "HelperFunctions.h"
#include "nsWindowListener.h"

#include "nsIDocument.h"
#include "nsIDOMDocument.h"
#include "nsIPresShell.h"
#include "nsIViewManager.h"

namespace { // anonymous namespace for local utlity functions

/**
 * Sends the given XEvent to the specified window. The event will have the
 * XEvent::send_event flag set to true.
 * @param aXEvent Pointer to the event structure
 * @param aWindow Target window's XID
 * @see XSendEvent()
 */
static void SendEvent(XEvent& aXEvent, Window aWindow)
{
  XEvent new_event = aXEvent;
  new_event.xany.display = GDK_DISPLAY();
  new_event.xany.window = aWindow;
  (void) XSendEvent(GDK_DISPLAY(), aWindow, False, NoEventMask, &new_event);
  XFlush(GDK_DISPLAY());
}


/**
 * Returns the widget for which the window of the specified XEvent has been
 * allocated to.
 * @param aXEvent XEvent to check
 * @return Allocating widget, or NULL of widget was not found.
 */
static GtkWidget* GetTargetWidget(XEvent& aXEvent)
{
  GtkWidget* widget = NULL;
  GdkWindow* window = gdk_window_lookup(aXEvent.xany.window);
  if (window) {
    gdk_window_get_user_data(window, (gpointer*) &widget);
  }
  return widget;
}


/**
 * @param aWidget Pointer to a GtkWidget
 * @return PR_TRUE if the specified widget is a part of a plugin.
 */
static PRBool IsPluginWidget(GtkWidget* aWidget)
{
  // this assumes that a plugin will create its own window
  if (aWidget) {
    // check !IS_MOZ_CONTAINER(aWidget) without linking against libgtkmozembed
    return strcmp(G_OBJECT_TYPE_NAME(aWidget), "MozContainer") != 0;
  }
  return PR_FALSE;
}


/**
 * Checks if the given XEvent is going to a plugin, and if so, also reports
 * the parent window of the plugin's window tree. Used to recognize panning
 * mode events that would be absorbed by a plugin.
 * @param aXEvent Received XEvent
 * @param aPluginParentWindow Output parameter: Will be set to the target
 *                            plugin's parent window if returning PR_TRUE.
 * @return PR_TRUE if the event would be going to a plugin.
 */
static PRBool CheckForPluginEvent(XEvent& aXEvent, Window& aPluginParentWindow)
{
  // first find out if the event targets is a plugin window
  GtkWidget* widget = GetTargetWidget(aXEvent);
  PRBool goingToPlugin = IsPluginWidget(widget);

  // Now find the plugin's parent window if one exists
  aPluginParentWindow = None;
  if (goingToPlugin) {
    GtkWidget* parent = NULL;
    while ((parent = gtk_widget_get_parent(widget)) != NULL && !GTK_IS_PLUG(widget)) {
      widget = parent;
    }
    if (parent) { // parent is a GtkSocket
      aPluginParentWindow = GDK_WINDOW_XWINDOW(parent->window);
    }
  }
  return goingToPlugin;
}


/**
 * Checks if the given XEvent would overlap with a plugin, and if so, also
 * reports the target window of the plugin where the event should go. Used to
 * detect whether main window dragging event should be given to plugin on
 * hover mode.
 * @param aXEvent Received XEvent
 * @param aTargetWindow Output parameter: If returning PR_TRUE, this will be
 *                      set to the XID of the topmost window of the plugin
 *                      where the event should go.
 * @return PR_TRUE if the event should go to a plugin.
 */
static PRBool CheckForPluginOverlap(XEvent& aXEvent, Window& aTargetWindow)
{
  if (!aXEvent.xmotion.subwindow)
    return PR_FALSE;
  
  // redirect the event straight to the subwindow
  aTargetWindow = aXEvent.xmotion.subwindow;

  // simply find the first leaf child and redirect to that.
  // TODO: better heuristics to find plugin's focus window
  Window root, parent, *children;
  unsigned int nchildren;
  while (XQueryTree(GDK_DISPLAY(), aTargetWindow, &root, &parent, &children, &nchildren) != 0) {
    Window next = None;
    if (nchildren > 0) {
      next = children[0];
    }
    XFree((char*) children);
    if (next)
      aTargetWindow = next;
    else
      break;
  }
  return PR_TRUE;
}


/**
 * @param aWindow XID of an existing window
 * @return XID of the parent window of aWindow.
 */
static Window GetWindowParent(Window aWindow)
{
  Window root_win, parent, *children;
  unsigned int nchildren;
  if (XQueryTree(GDK_DISPLAY(), aWindow, &root_win, &parent, &children, &nchildren) != 0) {
    XFree((char*) children);
    if (aWindow != root_win && parent != aWindow) {
      // aWindow was not the topmost window
      return parent;
    }
  }
  return None;
}


/**
 * @param aRoot XID of an existing window
 * @param aChild XID of another existing window
 * @return PR_TRUE if aRoot has a child window with XID aChild
 */
static PRBool BelongsToWindowTree(Window aRoot, Window aChild)
{
  if (!aRoot)
    return PR_FALSE;
  
  for (; aChild; aChild = GetWindowParent(aChild)) {
    //printf("BelongsToWindowTree aChild=%08x aRoot=%08x\n", aChild, aRoot);
    if (aChild == aRoot) {
      return PR_TRUE;
    }
  }
  return PR_FALSE;
}


/**
 * Checks if the XEvent belongs to this nsIDOMWindow instance
 * @param aXEvent Received XEvent
 * @param aDomWin Pointer to an nsIDOMWindow instance
 * @return PR_TRUE if XEvent is directed to the native window that has been
 *                 allocated for aDOMWin.
 */
static PRBool BelongsToWindow(XEvent& aXEvent, nsIDOMWindow* aDOMWin)
{
  // find the document of the DOMWindow
  nsCOMPtr<nsIDocument> doc;
  nsCOMPtr<nsIDOMDocument> domDoc;
  aDOMWin->GetDocument(getter_AddRefs(domDoc));
  NS_ENSURE_TRUE(domDoc, PR_FALSE);
  doc = do_QueryInterface(domDoc);
  NS_ENSURE_TRUE(doc, PR_FALSE);

  // get shell from the document
  nsIPresShell *shell = doc->GetPrimaryShell();
  NS_ENSURE_TRUE(shell, PR_FALSE);

  // get view manager from shell
  nsCOMPtr<nsIViewManager> viewManager;
  viewManager = shell->GetViewManager();
  NS_ENSURE_TRUE(viewManager, PR_FALSE);

  // get nsIWidget from view manager
  nsCOMPtr<nsIWidget> widget;
  viewManager->GetWidget(getter_AddRefs(widget));
  NS_ENSURE_TRUE(widget, PR_FALSE);

  // get native window from nsIWidget
  gpointer win = widget->GetNativeData(NS_NATIVE_WINDOW);
  if (GDK_IS_WINDOW(win)) {
    // check if our native window which is a parent of aXEvent
    return BelongsToWindowTree(GDK_WINDOW_XWINDOW(GDK_WINDOW(win)),
                               aXEvent.xany.window);
  }
  return PR_FALSE;
}


/**
 * Implementation of a GdkFilterFunc that will receive all (or most of the)
 * XEvents before the respective event subscribers can handle them.
 * @param aXEvent The native event to filter
 * @param aEvent The GDK event to which the X event will be translated.
 *               Not used in this implementation.
 * @param aData User data set when the filter was installed. will be set to a
 *              std::map<nsIDOMWindow*, nsWindowListener*> that contains all
 *              open DOM Windows and their respective window event listeners.
 * @return Filtering result
 * @see GdkFilterReturn
 * @see nsWidgetUtils::mlistenerContainer
 */
static GdkFilterReturn EventFilter(GdkXEvent *aXEvent,
                                   GdkEvent */*aEvent*/,
                                   gpointer aData)
{
  GdkFilterReturn result = GDK_FILTER_CONTINUE;
  XEvent* xev = (XEvent*) aXEvent;
  std::map<nsIDOMWindow*, nsWindowListener*>* listenerContainer =
      (std::map<nsIDOMWindow*, nsWindowListener*>*) aData;
  //printf("nsWindowListener::EventFilter XEvent type=%d xid=%08x\n", xev->type, xev->xany.window);

  // if XEvent::send_event is true, this event is probably re-sent by us,
  // so ignore it this time
  if (!xev || xev->xany.send_event || ((xev->type != ButtonPress) &&
                                       (xev->type != ButtonRelease) &&
                                       (xev->type != MotionNotify))) {
    // we do not handle other events than thee ones above
    return result;
  }

  // check if plugin support shuould be enabled currently
  PRBool enablePluginSupport = PR_FALSE;
  HelperFunctions::GetPref(PREF_TYPE_BOOL, "webaddon.widgetutils.pluginSupport",
                           &enablePluginSupport);
  if (enablePluginSupport) {
    // If plugin support is enabled, distribute events to all event handlers
    std::map<nsIDOMWindow*, nsWindowListener*>::iterator it;
    for (it = listenerContainer->begin();
         (result == GDK_FILTER_CONTINUE) && it != listenerContainer->end();
         ++it) {
      if (BelongsToWindow(*xev, it->first) && it->second) {
        PluginSupport& ps = it->second->GetPluginSupport();
        result = ps.HandleEvent(*xev, *(it->second));
      }
    }
  }
  return result;
}

}; // anonymous namespace for local utlity functions


// MEMBER FUNCTIONS ----------------------------------------------------------

PluginSupport::PluginSupport():
    mMouseDown(PR_FALSE),
    mButtonPressEventCached(PR_FALSE)
{
  // do nothing
};


void PluginSupport::Initialize(
    std::map<nsIDOMWindow*, nsWindowListener*>& aListenerContainer)
{
  // install the event filter
  gdk_window_add_filter(NULL, EventFilter, &aListenerContainer);
}


void PluginSupport::Uninitialize(
    std::map<nsIDOMWindow*, nsWindowListener*>& aListenerContainer)
{
  // remove the installed event filter (or do nothing if not installed)
  gdk_window_remove_filter(NULL, EventFilter, &aListenerContainer);
}


GdkFilterReturn PluginSupport::HandleEvent(XEvent& aXEvent, nsWindowListener& aWindowListener)
{
  GdkFilterReturn ret = GDK_FILTER_CONTINUE;
  
  if (aWindowListener.TouchScreenMode() == MODE_PANNING) {
      // Redirect the following XEvents according to current interaction mode
      // if necessary to allow panning over plugins
    Window pluginParentWindow = None;
    PRBool goingToPlugin = CheckForPluginEvent(aXEvent, pluginParentWindow);
    if (aXEvent.type == ButtonPress) {
      mMouseDown = PR_TRUE;
      if (goingToPlugin) {
          // cache this event and redirect the current event to plugin parent
        mCachedButtonPressEvent = aXEvent;
        mButtonPressEventCached = PR_TRUE;
        aXEvent.xany.window = pluginParentWindow;
      }
    }
    else if (aXEvent.type == ButtonRelease) {
      mMouseDown = PR_FALSE;
      if (aWindowListener.IsPanning()) {
        if (goingToPlugin) {
            // copy the button release to plugin parent to stop panning
          SendEvent(aXEvent, pluginParentWindow);
        }
      }
      else if (mButtonPressEventCached) {
          // send cached ButtonPress first, then re-schedule this ButtonRelease
          // to arrive after cached ButtonPress
        SendEvent(mCachedButtonPressEvent, mCachedButtonPressEvent.xany.window);
        SendEvent(aXEvent, mCachedButtonPressEvent.xany.window);
        mButtonPressEventCached = PR_FALSE;
          // redirect original ButtonRelease to parent
        aXEvent.xany.window = pluginParentWindow;
      }
    }
    else if ((aXEvent.type == MotionNotify) && mMouseDown) {
      if (goingToPlugin) {
          // Do not send motion notifications to plugin in panning mode
        aXEvent.xany.window = pluginParentWindow;
      }
    }
  }
  else if (aWindowListener.TouchScreenMode() == MODE_HOVER) {
      // Lift events higher if events intersect with plugin window
    if (aXEvent.type == MotionNotify) {
        //printf("nsWindowListener::EventFilter subwindow=%08x\n", xev->xmotion.subwindow);
      Window targetWindow = None;
      PRBool goingToPlugin = CheckForPluginOverlap(aXEvent, targetWindow);
      if (goingToPlugin) {
          // Redirect the event to the plugin that it overlaps
        aXEvent.xmotion.window = targetWindow;
        aXEvent.xmotion.subwindow = None;
      }
    }
  }
  return ret;
}

#endif // PLUGIN_SUPPORT_ENABLED
