/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* vim:set ts=2 sw=2 sts=2 tw=80 et cindent: */
/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is Mozilla's Element Optimizeing extension.
 *
 * The Initial Developer of the Original Code is the Mozilla Foundation.
 * Portions created by the Initial Developer are Copyright (C) 2006
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *   Oleg Romashin <romaxa@gmail.com> (original author)
 *   Brad Lassey <blassey@mozilla.com>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

#include "nsICategoryManager.h"
#include "nsStringGlue.h"
#include "nsWeakReference.h"
#include "nsIDOMMouseMotionListener.h"
#include "nsIDOMMouseListener.h"
#include "nsIDOMMouseEvent.h"
#include "nsIViewManager.h"
#include "nsITimer.h"
#include "nsITouchInteractListener.h"
#include "nsISelectionController.h"
#include "nsIScrollableView.h"
#include "speedmanager.h"
#include "nsWindowListener.h"
#include "HelperFunctions.h"
#include "nsXPCOM.h"
#include "nsIServiceManager.h"
#include "panningMode.h"
#include "hoverMode.h"
#include "inputMode.h"
#include "monoMode.h"
#include "nsIURI.h"
#include "modeSwitchWin.h"
#include "nsMemory.h"
#include "nsServiceManagerUtils.h"
#include "nsIDOMEventTarget.h"
#include "nsIDOMNode.h"
#include "nsIDOMDocument.h"
#include "nsIDOM3Document.h"
#include "nsIDOMWindow.h"
#include "nsIDOMWindowInternal.h"
#include "nsIDOMHTMLDocument.h"
#include "nsIDocument.h"
#include "nsIPresShell.h"
#include "nsIObserverService.h"

nsWindowListener::nsWindowListener() :
  mBlockSite(PR_FALSE),
  mMouseDown(PR_FALSE),
  mMonoEnable(PR_TRUE),
  mTouchScreenMode(MODE_UNSUPPORTED),
  mPanningMode(nsnull),
  mHoverMode(nsnull),
  mInputMode(nsnull),
  mMonoMode(nsnull),
  mModeSwitchWin(nsnull),
  mSBars(nsnull),
  mLastDOMWindow(nsnull),
  mBlockZoomEnabled(PR_TRUE)
{
  Init();
}

NS_IMETHODIMP
nsWindowListener::Init()
{
  mMonoEnable = PR_TRUE;
  HelperFunctions::GetPref(PREF_TYPE_BOOL, "webaddon.widgetutils.monomode", &mMonoEnable);
  char *blockSites = nsnull;
  HelperFunctions::GetPref(PREF_TYPE_STRING, "webaddon.widgetutils.block", &blockSites);
  if (blockSites) {
    mBlockDomains.AssignLiteral(blockSites);
    nsMemory::Free(blockSites);
  }

  if(mMonoEnable) {
    SetTouchScreenMode(MODE_MONO);
    mMonoMode = new MonoMode();
    NS_ENSURE_TRUE(mMonoMode, NS_ERROR_OUT_OF_MEMORY);
  } else {
    mSBars = new iScrollBars(this);
    SetTouchScreenMode(MODE_PANNING);
    mPanningMode = new PanningMode(*this, mSBars);
    NS_ENSURE_TRUE(mPanningMode, NS_ERROR_OUT_OF_MEMORY);
    mHoverMode = new HoverMode();
    NS_ENSURE_TRUE(mHoverMode, NS_ERROR_OUT_OF_MEMORY);
    mInputMode = new InputMode();
    NS_ENSURE_TRUE(mInputMode, NS_ERROR_OUT_OF_MEMORY);
    mModeSwitchWin = new ModeSwitchWin(*this);
    NS_ENSURE_TRUE(mModeSwitchWin, NS_ERROR_OUT_OF_MEMORY);
  }
#ifdef PLUGIN_SUPPORT_ENABLED
  mPluginSupport = new PluginSupport;
  NS_ENSURE_TRUE(mPluginSupport, NS_ERROR_OUT_OF_MEMORY);
#endif
  return NS_OK;
}

nsWindowListener::~nsWindowListener()
{
  delete mPanningMode;
  delete mHoverMode;
  delete mInputMode;
  delete mMonoMode;
  delete mModeSwitchWin;
#ifdef PLUGIN_SUPPORT_ENABLED
  delete mPluginSupport;
  mPluginSupport = nsnull;
#endif
  mPanningMode = nsnull;
  mHoverMode = nsnull;
  mInputMode = nsnull;
  mMonoMode = nsnull;
  mModeSwitchWin = nsnull;
  return;
}

NS_IMETHODIMP
nsWindowListener::MouseMove(nsIDOMEvent *aDOMEvent)
{
  NS_ENSURE_TRUE(aDOMEvent, NS_OK);
  NS_ENSURE_TRUE(mMouseDown, NS_OK);
  nsresult rv = NS_ERROR_FAILURE;
  DISPATCH_MOUSE_EVENT(rv);

  return NS_OK;
}

NS_IMETHODIMP
nsWindowListener::DragMove(nsIDOMEvent *aDOMEvent)
{
  return NS_OK;
}

NS_IMETHODIMP
nsWindowListener::HandleEvent(nsIDOMEvent *aDOMEvent)
{
  return NS_OK;
}

NS_IMETHODIMP
nsWindowListener::MouseDown(nsIDOMEvent *aDOMEvent)
{
  NS_ENSURE_TRUE(aDOMEvent, NS_OK);
  HelperFunctions::GetDOMWindowFromEvent(aDOMEvent, &mLastDOMWindow);
  if(mModeSwitchWin)
    mModeSwitchWin->UpdateFromEvent(aDOMEvent);
  if (HelperFunctions::IsWrongEventSequence(aDOMEvent)) {
    aDOMEvent->StopPropagation();
    aDOMEvent->PreventDefault();
    return NS_OK;
  }
  mMouseDown = PR_TRUE;
  CheckBlockSite(aDOMEvent);
  if(!mMonoEnable) {
    ModeChangeStart(aDOMEvent);
  }
  nsresult rv = NS_ERROR_FAILURE;
  DISPATCH_MOUSE_EVENT(rv);

  if(!mMonoEnable) {
    ModeChangeEnd(aDOMEvent);
  }

  return NS_OK;
}

NS_IMETHODIMP
nsWindowListener::MouseUp(nsIDOMEvent *aDOMEvent)
{
  NS_ENSURE_TRUE(aDOMEvent, NS_OK);
  HelperFunctions::IsWrongEventSequence(aDOMEvent); // only do update
  mMouseDown = PR_FALSE;

  if(!mMonoEnable) {
    ModeChangeStart(aDOMEvent);
  }
  nsresult rv = NS_ERROR_FAILURE;
  DISPATCH_MOUSE_EVENT(rv);

  if(!mMonoEnable) {
    ModeChangeEnd(aDOMEvent);
  }

  return NS_OK;
}

NS_IMETHODIMP
nsWindowListener::MouseClick(nsIDOMEvent *aDOMEvent)
{
  NS_ENSURE_TRUE(aDOMEvent, NS_OK);
  nsresult rv = NS_ERROR_FAILURE;
  DISPATCH_MOUSE_EVENT(rv);
  return NS_OK;
}

NS_IMETHODIMP
nsWindowListener::MouseDblClick(nsIDOMEvent *aDOMEvent)
{
  return NS_OK;
}

NS_IMETHODIMP
nsWindowListener::MouseOver(nsIDOMEvent *aDOMEvent)
{
  NS_ENSURE_TRUE(aDOMEvent, NS_OK);
  nsresult rv = NS_ERROR_FAILURE;
  DISPATCH_MOUSE_EVENT(rv);
  return NS_OK;
}

NS_IMETHODIMP
nsWindowListener::MouseOut(nsIDOMEvent *aDOMEvent)
{
  NS_ENSURE_TRUE(aDOMEvent, NS_OK);
  nsresult rv = NS_ERROR_FAILURE;
  DISPATCH_MOUSE_EVENT(rv);
  return NS_OK;
}

NS_IMETHODIMP
nsWindowListener::Focus(nsIDOMEvent *aDOMEvent)
{
  NS_ENSURE_TRUE(aDOMEvent, NS_OK);
  NS_ENSURE_TRUE(mModeSwitchWin, NS_OK);
  mModeSwitchWin->HostWinFocus(aDOMEvent);
  return NS_OK;
}

NS_IMETHODIMP
nsWindowListener::Blur(nsIDOMEvent *aDOMEvent)
{
  NS_ENSURE_TRUE(aDOMEvent, NS_OK);
  NS_ENSURE_TRUE(mModeSwitchWin, NS_OK);
  mModeSwitchWin->HostWinBlur(aDOMEvent);
  return NS_OK;
}

/*
NS_IMETHODIMP
nsWindowListener::DragEnter(nsIDOMEvent* aDOMEvent)
{
  //printf("%s\n", __FUNCTION__);
  NS_ENSURE_TRUE(aDOMEvent, NS_OK);
  nsresult rv;
  nsString eventType;
  rv = aDOMEvent->GetType(eventType);
  NS_ENSURE_SUCCESS(rv, NS_OK);
  //printf("%s\n", NS_ConvertUTF16toUTF8(eventType).get());
  return NS_OK;
}

NS_IMETHODIMP
nsWindowListener::DragOver(nsIDOMEvent* aDOMEvent)
{
  //printf("%s\n", __FUNCTION__);
  NS_ENSURE_TRUE(aDOMEvent, NS_OK);
  nsresult rv;
  nsString eventType;
  rv = aDOMEvent->GetType(eventType);
  NS_ENSURE_SUCCESS(rv, NS_OK);
  //printf("%s\n", NS_ConvertUTF16toUTF8(eventType).get());
  return NS_OK;
}

NS_IMETHODIMP
nsWindowListener::DragExit(nsIDOMEvent* aDOMEvent)
{
  //printf("%s\n", __FUNCTION__);
  NS_ENSURE_TRUE(aDOMEvent, NS_OK);
  nsresult rv;
  nsString eventType;
  rv = aDOMEvent->GetType(eventType);
  NS_ENSURE_SUCCESS(rv, NS_OK);
  //printf("%s\n", NS_ConvertUTF16toUTF8(eventType).get());
  return NS_OK;
}

NS_IMETHODIMP
nsWindowListener::DragDrop(nsIDOMEvent* aDOMEvent)
{
  //printf("%s\n", __FUNCTION__);
  NS_ENSURE_TRUE(aDOMEvent, NS_OK);
  nsresult rv;
  nsString eventType;
  rv = aDOMEvent->GetType(eventType);
  NS_ENSURE_SUCCESS(rv, NS_OK);
  //printf("%s\n", NS_ConvertUTF16toUTF8(eventType).get());
  return NS_OK;
}

NS_IMETHODIMP
nsWindowListener::DragGesture(nsIDOMEvent* aDOMEvent)
{
  //printf("%s\n", __FUNCTION__);
  NS_ENSURE_TRUE(aDOMEvent, NS_OK);
  nsresult rv;
  nsString eventType;
  rv = aDOMEvent->GetType(eventType);
  NS_ENSURE_SUCCESS(rv, NS_OK);
  //printf("%s\n", NS_ConvertUTF16toUTF8(eventType).get());
  return NS_OK;
}

NS_IMETHODIMP
nsWindowListener::DragEnd(nsIDOMEvent* aDOMEvent)
{
  //printf("%s\n", __FUNCTION__);
  NS_ENSURE_TRUE(aDOMEvent, NS_OK);
  nsresult rv;
  nsString eventType;
  rv = aDOMEvent->GetType(eventType);
  NS_ENSURE_SUCCESS(rv, NS_OK);
  //printf("%s\n", NS_ConvertUTF16toUTF8(eventType).get());
  return NS_OK;
}

NS_IMETHODIMP
nsWindowListener::Drag(nsIDOMEvent* aDOMEvent)
{
  //printf("%s\n", __FUNCTION__);
  NS_ENSURE_TRUE(aDOMEvent, NS_OK);
  nsresult rv;
  nsString eventType;
  rv = aDOMEvent->GetType(eventType);
  NS_ENSURE_SUCCESS(rv, NS_OK);
  //printf("%s\n", NS_ConvertUTF16toUTF8(eventType).get());
  return NS_OK;
}
*/

NS_IMETHODIMP
nsWindowListener::ModeChangeStart(nsIDOMEvent *aDOMEvent)
{
  NS_ENSURE_TRUE(aDOMEvent, NS_ERROR_FAILURE);

  PRInt32 mousetype = HelperFunctions::GetMouseEventType(aDOMEvent);
  NS_ENSURE_TRUE(mousetype != EVENT_TYPE_UNSUPPORTED, NS_ERROR_FAILURE);
  if(mPanningMode && mPanningMode->IsDispatchEvent())
    return NS_OK;

  if (mousetype == MOUSE_DOWN) {
    if(HelperFunctions::InActiveRegion(aDOMEvent)) {
      SetTouchScreenMode(MODE_HOVER);
      if(mModeSwitchWin)
        mModeSwitchWin->ShowModeSwitchWin(MODE_HOVER, aDOMEvent);
    }
    else if (mTouchScreenMode == MODE_PANNING) {
      if(mModeSwitchWin)
        mModeSwitchWin->HideModeSwitchWin();
    }
  } // mousedown
  else if (mousetype == MOUSE_UP) {

  } // mouseup
  else if (mousetype == MOUSE_MOVE) {

  } // mousemove

  // update mouse cursor visibility flag to event's widget
  SetCursorVisible(mTouchScreenMode == MODE_HOVER ||
                   mTouchScreenMode == MODE_DINPUT);
  HelperFunctions::UpdateCursorVisibility(aDOMEvent, &mCursorVisible);

  return NS_OK;
}

NS_IMETHODIMP
nsWindowListener::ModeChangeEnd(nsIDOMEvent *aDOMEvent)
{
  NS_ENSURE_TRUE(aDOMEvent, NS_ERROR_FAILURE);

  PRInt32 mousetype = HelperFunctions::GetMouseEventType(aDOMEvent);
  NS_ENSURE_TRUE(mousetype!=EVENT_TYPE_UNSUPPORTED, NS_ERROR_FAILURE);
  if(mPanningMode && mPanningMode->IsDispatchEvent())
    return NS_OK;

  if (mousetype == MOUSE_DOWN) {

  } // mousedown
  else if (mousetype == MOUSE_UP) {
    if (mTouchScreenMode == MODE_HOVER) {
      SetTouchScreenMode(MODE_PANNING);
      if(mModeSwitchWin) {
        mModeSwitchWin->ShowModeSwitchWin(MODE_PANNING, aDOMEvent);
        mModeSwitchWin->HideModeSwitchWin(PR_TRUE);
      }
    }
  } // mouseup
  else if (mousetype == MOUSE_MOVE) {

  } // mousemove

  // update mouse cursor visibility flag to event's widget
  SetCursorVisible(mTouchScreenMode == MODE_HOVER ||
                   mTouchScreenMode == MODE_DINPUT);
  HelperFunctions::UpdateCursorVisibility(aDOMEvent, &mCursorVisible);

  return NS_OK;
}

PRBool
nsWindowListener::CheckBlockSite(nsIDOMEvent *aDOMEvent)
{
  NS_ENSURE_TRUE(aDOMEvent, PR_FALSE);

  mBlockSite = PR_FALSE;

  nsCOMPtr<nsIDOMEventTarget> eventTarget;
  aDOMEvent->GetTarget(getter_AddRefs(eventTarget));
  NS_ENSURE_TRUE(eventTarget , PR_FALSE);
  nsCOMPtr<nsIDOMNode> eventNode = do_QueryInterface(eventTarget);
  NS_ENSURE_TRUE(eventNode , PR_FALSE);
  nsCOMPtr<nsIDOMDocument> domDoc;
  eventNode->GetOwnerDocument(getter_AddRefs(domDoc));
  NS_ENSURE_TRUE(domDoc , PR_FALSE);
  nsCOMPtr<nsIDOMHTMLDocument> htmlDoc = do_QueryInterface(domDoc);
  nsString domain;
  if (htmlDoc)
    htmlDoc->GetDomain(domain);
  if (domain.IsEmpty())
    return mBlockSite;

  PRInt32 pos = mBlockDomains.Find(domain, PR_TRUE);
  mBlockSite = pos == -1 ? PR_FALSE : PR_TRUE;

  return mBlockSite;
}

nsresult
nsWindowListener::SetBlockZoomEnabled(PRBool aEnabled)
{
  NS_ENSURE_TRUE(mLastDOMWindow, NS_ERROR_FAILURE);
  nsCOMPtr<nsIObserverService> obsSvc = do_GetService(NS_OBSERVERSERVICE_CONTRACTID);
  NS_ENSURE_STATE(obsSvc);
  nsresult rv = obsSvc->NotifyObservers(
    (nsISupports*) mLastDOMWindow,
    aEnabled ? "block-zoom-enable-mouse-event-handling"
             : "block-zoom-disable-mouse-event-handling",
    nsnull);
  NS_ENSURE_SUCCESS(rv, rv);
  mBlockZoomEnabled = aEnabled;
  return NS_OK;
}

nsresult
nsWindowListener::SetTouchScreenMode(PRInt32 aNewMode)
{
  mTouchScreenMode = aNewMode;

  // enable block zooming only in panning mode, disable otherwise
  if (mTouchScreenMode == MODE_PANNING && !mBlockZoomEnabled) {
    SetBlockZoomEnabled(PR_TRUE);
  }
  else if (mTouchScreenMode != MODE_PANNING && mBlockZoomEnabled) {
    SetBlockZoomEnabled(PR_FALSE);
  }
  return NS_OK;
}

PRInt32
nsWindowListener::TouchScreenMode() const
{
  return mTouchScreenMode;
}

nsresult
nsWindowListener::SetCursorVisible(PRBool aVisible)
{
  mCursorVisible = aVisible;
  return NS_OK;
}

NS_IMETHODIMP
nsWindowListener::OnStateChange(nsIWebProgress *aWebProgress, nsIRequest *aRequest,
                                PRUint32 aStateFlags, nsresult aStatus)
{
  NS_ENSURE_TRUE(mSBars, NS_OK);
  mSBars->OnStateChange(aWebProgress, aStateFlags);
  return NS_OK;
}

NS_IMETHODIMP
nsWindowListener::OnProgressChange(nsIWebProgress *aWebProgress, nsIRequest *aRequest,
                                   PRInt32 aCurSelfProgress, PRInt32 aMaxSelfProgress,
                                   PRInt32 aCurTotalProgress, PRInt32 aMaxTotalProgress)
{
  return NS_OK;
}

NS_IMETHODIMP
nsWindowListener::OnLocationChange(nsIWebProgress *aWebProgress, nsIRequest *aRequest,
                                   nsIURI *aLocation)
{
  // url changed, back to panning mode
  if (!mMonoEnable) {
    SetTouchScreenMode(MODE_PANNING);
    if(mModeSwitchWin)
      mModeSwitchWin->HideModeSwitchWin();
  }
  return NS_OK;
}

NS_IMETHODIMP
nsWindowListener::OnStatusChange(nsIWebProgress *aWebProgress, nsIRequest *aRequest,
                                 nsresult aStatus, const PRUnichar *aMessage)
{
  return NS_OK;
}

NS_IMETHODIMP
nsWindowListener::OnSecurityChange(nsIWebProgress *aWebProgress, nsIRequest *aRequest,
                                   PRUint32 aState)
{
  return NS_OK;
}

NS_IMPL_ISUPPORTS5(nsWindowListener, nsIDOMMouseMotionListener, nsIDOMMouseListener,
                   nsIDOMFocusListener, nsIWebProgressListener,
                   nsISupportsWeakReference)

#ifdef PLUGIN_SUPPORT_ENABLED

PluginSupport&
nsWindowListener::GetPluginSupport()
{
    return *mPluginSupport;
}

PRBool
nsWindowListener::IsPanning() const
{
    return mPanningMode && (mPanningMode->IsPanning() > 0);
}

#endif // PLUGIN_SUPPORT_ENABLED
