#include "nsIDOMMouseEvent.h"
#include "nsITimer.h"
#include "nsITouchInteractListener.h"
#include "speedmanager.h"
#include "HelperFunctions.h"
#include "nsXPCOM.h"
#include "panningMode.h"
#include "nsIViewManager.h"
#include "nsComponentManagerUtils.h"
#include "nsIDOMEventTarget.h"
#include "nsIDOMHTMLInputElement.h"
#include "nsIDOMHTMLTextAreaElement.h"
#include "nsIDOMHTMLAnchorElement.h"
#include "nsIDOMHTMLSelectElement.h"
#include "nsStringGlue.h"
#include "nsIDOMNSEvent.h"
#include "nsIWeakReference.h"
#include "nsIWeakReferenceUtils.h"
#include "nsGUIEvent.h"
#include "nsIMarkupDocumentViewer.h"
#include "nsWindowListener.h"

#define PAN_DECAY_STEP 50.0

enum PANNING_MODE
{
  HORIZONTAL_PANNING,
  VERTICAL_PANNING,
  ANYDIRECTION_PANNING,
  LAST_PANNING_MODE
};

volatile PRInt32 *PanningMode::sPanning = new PRInt32;

NS_IMPL_ISUPPORTS1(PanningMode, IBaseMode)

PanningMode::PanningMode(nsWindowListener& aWindowListener, iScrollBars *aSBars) :
  mWindowListener(aWindowListener),
  mPanningTimer(nsnull),
  mEndPanTimer(nsnull),
  mViewManager(nsnull),
  mSpeedX(nsnull),
  mSpeedY(nsnull),
  mKineticPanStep(0),
  mMouseDown(PR_FALSE),
  mFocusElement(nsnull),
  mPanningMode(LAST_PANNING_MODE),
  mHasDispatchEvent(PR_FALSE),
  mSBars(aSBars),
  mMouseDownZoomLevel(1.0)
{
  Init();
}

nsresult
PanningMode::Init()
{
  INIT_MOUSE_DISPATCH();
  if(sPanning)
    *sPanning = 0;
  nsresult rv;
  mPanningTimer = do_CreateInstance(NS_TIMER_CONTRACTID, &rv);
  NS_ENSURE_SUCCESS(rv, rv);
  mEndPanTimer = do_CreateInstance(NS_TIMER_CONTRACTID, &rv);
  NS_ENSURE_SUCCESS(rv, rv);

  memset((void*)&mFirstMousedownEvent, 0, sizeof(mFirstMousedownEvent));
  memset((void*)&mLastMouseEvent, 0, sizeof(mLastMouseEvent));
  memset((void*)&mCurrentMouseEvent, 0, sizeof(mCurrentMouseEvent));
  mMaxPressure = -1;

  mSpeedX = new SpeedManager;
  NS_ENSURE_TRUE(mSpeedX, NS_ERROR_OUT_OF_MEMORY);
  mSpeedY = new SpeedManager;
  NS_ENSURE_TRUE(mSpeedY, NS_ERROR_OUT_OF_MEMORY);

  return NS_OK;
}

PanningMode::~PanningMode()
{
  HelperFunctions::CancelShowCxtMenu();
  if (mPanningTimer)
    mPanningTimer->Cancel();
  if (mEndPanTimer)
    mEndPanTimer->Cancel();
  delete mSpeedX;
  delete mSpeedY;
  mSpeedX = nsnull;
  mSpeedY = nsnull;

  return;
}

PRBool
PanningMode::SetMouseDown(PRBool aMouseDown)
{
  mMouseDown = aMouseDown;
  return mMouseDown;
}

PRBool
PanningMode::CanShowCxtMenu()
{
  NS_ENSURE_TRUE(sPanning, PR_TRUE);
  return (*sPanning==0);
}

nsIDOMEvent*
PanningMode::GetDOMEvent()
{
  return mDOMEvent;
}

nsresult
PanningMode::OnPanning(nsIDOMEvent *aDOMEvent)
{
  NS_ENSURE_TRUE(aDOMEvent, NS_ERROR_FAILURE);
  mDOMEvent = do_QueryInterface(aDOMEvent);
  return CALL_MOUSE_DISPATCH(aDOMEvent);
}

PRInt32
PanningMode::IsPanning()
{
  NS_ENSURE_TRUE(sPanning, 0);
  return *sPanning;
}

nsresult
PanningMode::EndPan(PRBool aDelay)
{
  if (!aDelay) {
    if (mPanningTimer)
      mPanningTimer->Cancel();
    if (mEndPanTimer)
      mEndPanTimer->Cancel();
    if(sPanning)
      *sPanning = 0;
    if (mSpeedX)
      mSpeedX->ClearSpeed();
    if (mSpeedY)
      mSpeedY->ClearSpeed();
    mKineticPanStep = 0;
    mPanningMode = LAST_PANNING_MODE;

    mWindowListener.SetBlockZoomEnabled(PR_TRUE); // ignore return value
    if (mSBars)
      mSBars->MouseUp(nsnull);
  } else {
    if (mEndPanTimer)
      mEndPanTimer->InitWithFuncCallback(PanningMode::EndPanDelayCallBack,
                                         (void*)this,
                                         500,
                                         nsITimer::TYPE_ONE_SHOT);
  }

  return NS_OK;
}

nsresult
PanningMode::MouseDown(nsIDOMEvent *aDOMEvent)
{
  NS_ENSURE_TRUE(aDOMEvent, NS_ERROR_FAILURE);
  NS_ENSURE_FALSE(HelperFunctions::IsXULNode(aDOMEvent), NS_OK);
  
  PRBool absorbMouseDown = PR_FALSE;
  HelperFunctions::GetPref(
    PREF_TYPE_BOOL, "webaddon.widgetutils.absorbMouseDown", &absorbMouseDown);
  
  if (mHasDispatchEvent) {
    if (absorbMouseDown &&
        (HelperFunctions::IsHTMLSelectElement(aDOMEvent) ||
         HelperFunctions::IsHTMLOptionElement(aDOMEvent) ||
         HelperFunctions::IsHTMLInputElement(aDOMEvent) ||
        HelperFunctions::IsHTMLTextAreaElement(aDOMEvent))) {
      aDOMEvent->StopPropagation();
      aDOMEvent->PreventDefault();
    }
    return NS_OK;
  }
  nsCOMPtr<nsIViewManager> tmpViewManager;
  HelperFunctions::GetViewManagerFromEvent(aDOMEvent, getter_AddRefs(tmpViewManager));
  if (tmpViewManager) // check mouse move out of screen
    NS_IF_ADDREF(mViewManager = tmpViewManager);
  NS_ENSURE_TRUE(mViewManager, NS_ERROR_FAILURE);

  nsCOMPtr<nsIDOMWindow> DOMWindow;
  HelperFunctions::GetDOMWindowFromEvent(aDOMEvent,getter_AddRefs(DOMWindow));
  nsCOMPtr<nsIMarkupDocumentViewer> markupViewer;
  nsresult rv = HelperFunctions::GetMarkupViewerByWindow(DOMWindow, getter_AddRefs(markupViewer));
  if (markupViewer)
    markupViewer->GetFullZoom(&mMouseDownZoomLevel);

  if (!HelperFunctions::IsHTMLOptionElement(aDOMEvent))
    mMouseDown = PR_TRUE;
  EndPan();
  NS_ENSURE_FALSE(HelperFunctions::IsXULNode(aDOMEvent), NS_OK);

  MouseEventArg tmpMousedownEvent;
  memcpy((void*)&tmpMousedownEvent, (void*)&mFirstMousedownEvent, sizeof(MouseEventArg));
  HelperFunctions::UpdateMouseEventArg(aDOMEvent, &mFirstMousedownEvent);
  HelperFunctions::UpdateMouseEventArg(aDOMEvent, &mLastMouseEvent);
  HelperFunctions::RemoveSelection(aDOMEvent);
  HelperFunctions::CloseSelection(aDOMEvent);
  ChangeFocus(aDOMEvent);
  if (!HelperFunctions::IsHTMLSelectElement(aDOMEvent) &&
      !HelperFunctions::IsHTMLOptionElement(aDOMEvent) &&
      !HelperFunctions::IsHTMLInputElement(aDOMEvent) &&
      !HelperFunctions::IsHTMLTextAreaElement(aDOMEvent)) {
    if (absorbMouseDown) {
      aDOMEvent->StopPropagation();
      aDOMEvent->PreventDefault();
    }
    PRInt32 delta = mFirstMousedownEvent.timeStamp - tmpMousedownEvent.timeStamp;
    if(PR_ABS(delta) > 250)
      HelperFunctions::StartShowCxtMenu(this);
  }

  mMaxPressure = 0.1;
  if (mSBars)
    mSBars->MouseDown(aDOMEvent);

  GET_STATIC_INTERACTLISTENER(sListener);
  if (!sListener) return NS_OK;

  nsCOMPtr<nsIWidget> widget;
  HelperFunctions::GetWidgetFromEvent(aDOMEvent, getter_AddRefs(widget));
  return sListener->OnMouseDownPan(aDOMEvent, widget);
}

nsresult
PanningMode::MouseMove(nsIDOMEvent *aDOMEvent)
{
  NS_ENSURE_TRUE(aDOMEvent, NS_ERROR_FAILURE);
  NS_ENSURE_TRUE(mMouseDown, NS_OK);
  NS_ENSURE_FALSE(HelperFunctions::IsXULNode(aDOMEvent), NS_OK);

  PRInt32 delta = 0;
  PRInt32 dx = 0;
  PRInt32 dy = 0;

  HelperFunctions::UpdateMouseEventArg(aDOMEvent, &mCurrentMouseEvent);
  if (mCurrentMouseEvent.pressure >= 0.05 && mCurrentMouseEvent.pressure <= 1)
    mMaxPressure = PR_MAX(mCurrentMouseEvent.pressure, mMaxPressure);

  // To reduce jitters, return if the mousemove was a small delta from the first event
  PRBool isJittering = HelperFunctions::IsMouseJittering(&mCurrentMouseEvent,
                                                         &mFirstMousedownEvent, NULL, NULL, NULL);
  NS_ENSURE_FALSE(isJittering, NS_OK);

  // calculate cursor velocity based on the last event
  isJittering = HelperFunctions::IsMouseJittering(&mCurrentMouseEvent,
                                                  &mLastMouseEvent,
                                                  &delta,
                                                  &dx,
                                                  &dy);
  if (mSpeedX && mSpeedY) {
    mSpeedX->AddSpeed(delta, dx);
    mSpeedY->AddSpeed(delta, dy);
  }

  HelperFunctions::UpdateMouseEventArg(aDOMEvent, &mLastMouseEvent);
  nsCOMPtr<nsIViewManager> tmpViewManager;
  HelperFunctions::GetViewManagerFromEvent(aDOMEvent, getter_AddRefs(tmpViewManager));
  if(tmpViewManager) // check mouse move out of screen
    NS_IF_ADDREF(mViewManager = tmpViewManager);
  NS_ENSURE_TRUE(mViewManager, NS_ERROR_FAILURE);
  DoPan(aDOMEvent, mViewManager, dx, dy);

  if (sPanning && *sPanning > 0) {
    aDOMEvent->StopPropagation();
    aDOMEvent->PreventDefault();
  }

  GET_STATIC_INTERACTLISTENER(sListener);
  if (!sListener) return NS_OK;

  nsCOMPtr<nsIWidget> widget;
  HelperFunctions::GetWidgetFromEvent(aDOMEvent, getter_AddRefs(widget));
  return sListener->OnMouseMovePan(aDOMEvent, widget);
}

nsresult
PanningMode::MouseUp(nsIDOMEvent *aDOMEvent)
{
  NS_ENSURE_TRUE(aDOMEvent, NS_ERROR_FAILURE);
  if(mHasDispatchEvent) {
    aDOMEvent->StopPropagation();
    aDOMEvent->PreventDefault();
    mHasDispatchEvent = PR_FALSE;
    return NS_OK;
  }
  mMouseDown = PR_FALSE;
  if (mMaxPressure > 0.25 && IsPanning() == 1)
    *sPanning = 0;
  mMaxPressure = 0;
  NS_ENSURE_FALSE(HelperFunctions::IsXULNode(aDOMEvent), NS_OK);

  nsCOMPtr<nsIDOMMouseEvent> currentEvent = do_QueryInterface(aDOMEvent);
  if (!currentEvent || mFirstMousedownEvent.timeStamp == 0)
    return NS_OK;

  GET_STATIC_INTERACTLISTENER(sListener);
  if (sListener) {
    nsCOMPtr<nsIWidget> widget;
    HelperFunctions::GetWidgetFromEvent(aDOMEvent, getter_AddRefs(widget));
    sListener->OnMouseUpPan(aDOMEvent, widget);
  }

  HelperFunctions::CancelShowCxtMenu();
  NS_ENSURE_TRUE(sPanning, NS_OK);
  if (*sPanning == 0)
    return NS_OK;

  HelperFunctions::UpdateMouseEventArg(aDOMEvent, &mCurrentMouseEvent);
  PRInt32 distanceX = mFirstMousedownEvent.screenX - mCurrentMouseEvent.screenX;
  PRInt32 distanceY = mFirstMousedownEvent.screenY - mCurrentMouseEvent.screenY;

  if (PR_ABS(distanceX) < 10 && PR_ABS(distanceY) < 10) {
    EndPan(PR_TRUE);//delay
    return NS_OK;
  }

  if (*sPanning > 0) {
    //aDOMEvent->StopPropagation();
    //aDOMEvent->PreventDefault();
    nsresult rv;
    if (mPanningTimer) {
      rv = mPanningTimer->InitWithFuncCallback(PanningMode::DoKineticPan,
                                               (void*)this,
                                               0,
                                               nsITimer::TYPE_ONE_SHOT);
      if (NS_SUCCEEDED(rv))
        return NS_OK;
    }
    EndPan(PR_TRUE);//delay
  }
  return NS_OK;
}

nsresult
PanningMode::MouseClick(nsIDOMEvent *aDOMEvent)
{
  NS_ENSURE_TRUE(aDOMEvent, NS_OK);
  if (sPanning && *sPanning > 0) {
    aDOMEvent->StopPropagation();
    aDOMEvent->PreventDefault();
  } else if (mPanningTimer &&
          !HelperFunctions::InActiveRegion(aDOMEvent) &&
          !HelperFunctions::IsHTMLSelectElement(aDOMEvent)) {
    PanningMode::DispatchMouseEvent(mPanningTimer, this);
    //mPanningTimer->InitWithFuncCallback(PanningMode::DispatchMouseEvent,
    //                                    (void*) this,
    //                                    0,
    //                                    nsITimer::TYPE_ONE_SHOT);
    
    // Synchronous dispatching done, reset flag
    mHasDispatchEvent = PR_FALSE; 
  }
  return NS_OK;
}

PRInt32
PanningMode::DetectPanningMode(nsIViewManager* aViewManager, PRInt32 aDx, PRInt32 aDy)
{
  NS_ENSURE_TRUE(aViewManager, mPanningMode = ANYDIRECTION_PANNING);
  if (mPanningMode == LAST_PANNING_MODE) {
    if (PR_ABS(aDy) * 2 < PR_ABS(aDx) && HelperFunctions::CanScrollHorizontal(aViewManager)) {
      mPanningMode = HORIZONTAL_PANNING;
    } else if (PR_ABS(aDx) * 2 < PR_ABS(aDy) && HelperFunctions::CanScrollVertical(aViewManager)) {
      mPanningMode = VERTICAL_PANNING;
    } else {
      mPanningMode = ANYDIRECTION_PANNING;
    }
  }
  return mPanningMode;
}

nsresult
PanningMode::DoPan(nsIDOMEvent *aDOMEvent,
                   nsIViewManager* aViewManager,
                   PRInt32 aDx,
                   PRInt32 aDy)
{
  // !!! aDOMEvent can be nsnull
  NS_ENSURE_TRUE(aViewManager, NS_ERROR_FAILURE);
  if(aDOMEvent) {
    PRInt32 distanceX = mFirstMousedownEvent.screenX - mCurrentMouseEvent.screenX;
    PRInt32 distanceY = mFirstMousedownEvent.screenY - mCurrentMouseEvent.screenY;
    if(distanceX < 100 || distanceY < 100)
      DetectPanningMode(aViewManager, aDx, aDy);
  }

  if(mPanningMode == HORIZONTAL_PANNING)
    aDy = 0;
  if(mPanningMode == VERTICAL_PANNING)
    aDx = 0;

  if (mSBars)
    mSBars->MouseMove(aDOMEvent);
  HelperFunctions::ScrollWindow(aDOMEvent, aViewManager, -aDx, -aDy);
  if(sPanning) {
    (*sPanning)++;
    if (*sPanning == 1) {
      // first panning event, try to disable block zoom mouse event handling
      mWindowListener.SetBlockZoomEnabled(PR_FALSE); // ignore return value
    }
  }

  return NS_OK;
}

SpeedManager*
PanningMode::GetSpeedMngX()
{
  return mSpeedX;
}

SpeedManager*
PanningMode::GetSpeedMngY()
{
  return mSpeedY;
}

PRInt32
PanningMode::GetKineticPanStep()
{
  // ++ for kinetic slow down
  return mKineticPanStep++;
}

//static
void
PanningMode::DoKineticPan(nsITimer *timer, void *closure)
{
  NS_ENSURE_TRUE(timer, );
  NS_ENSURE_TRUE(closure, );
  PanningMode *self = (PanningMode*)closure;
  NS_ENSURE_TRUE(self, );

  const double kDecayFactor = 0.95;
  const PRInt32 kCutOff = 2;
  const PRInt32 kSpeedLimit = 100;

  SpeedManager *spdmngX = self->GetSpeedMngX();
  SpeedManager *spdmngY = self->GetSpeedMngY();
  PRInt32 step = self->GetKineticPanStep();
  if(!spdmngX || !spdmngY || !self->mViewManager) {
    self->EndPan(PR_TRUE);
    return;
  }
  PRInt32 speedX = spdmngX->GetSpeed();
  PRInt32 speedY = spdmngY->GetSpeed();

  if (PR_ABS(speedX) > kSpeedLimit)
    speedX = speedX > 0 ? kSpeedLimit : -kSpeedLimit;

  if (PR_ABS(speedY) > kSpeedLimit)
    speedY = speedY > 0 ? kSpeedLimit : -kSpeedLimit;

  self->DoPan(nsnull, self->mViewManager, speedX, speedY);

  // slow down.
  speedX = PRInt32(speedX*(kDecayFactor - step/PAN_DECAY_STEP));
  speedY = PRInt32(speedY*(kDecayFactor - step/PAN_DECAY_STEP));

  spdmngX->SetSpeed(speedX);
  spdmngY->SetSpeed(speedY);

  // see if we should continue
  if (PR_ABS(speedX) > kCutOff || PR_ABS(speedY) > kCutOff)
    timer->InitWithFuncCallback(DoKineticPan,
                                (void*)self,
                                step*10,
                                nsITimer::TYPE_ONE_SHOT);
  else
    self->EndPan(PR_TRUE);//delay endpan
}

//static
void
PanningMode::EndPanDelayCallBack(nsITimer *timer, void *closure)
{
  NS_ENSURE_TRUE(timer, );
  NS_ENSURE_TRUE(closure, );
  PanningMode *self = (PanningMode*)closure;
  NS_ENSURE_TRUE(self, );
  self->EndPan();
}

//static
void
PanningMode::DispatchMouseEvent(nsITimer *timer, void *closure)
{
  NS_ENSURE_TRUE(timer, );
  NS_ENSURE_TRUE(closure, );
  PanningMode *self = (PanningMode*)closure;
  NS_ENSURE_TRUE(self, );

  nsIView* rootView = nsnull;
  NS_ENSURE_TRUE(self->mViewManager, );
  self->mViewManager->GetRootView(rootView);
  NS_ENSURE_TRUE(rootView, );

  // start dispatching events
  self->mHasDispatchEvent = PR_TRUE;

  nsEventStatus status;
  nsMouseEvent event(PR_TRUE, NS_MOUSE_MOVE, rootView->GetWidget(), nsMouseEvent::eReal);
  event.refPoint.x = self->mFirstMousedownEvent.clientX * self->GetZoomLevel();
  event.refPoint.y = self->mFirstMousedownEvent.clientY * self->GetZoomLevel();
  event.time = PR_IntervalNow();
  self->mViewManager->DispatchEvent(&event, &status);
  event.message = NS_MOUSE_BUTTON_DOWN;
  event.time = PR_IntervalNow();
  self->mViewManager->DispatchEvent(&event, &status);
  event.message = NS_MOUSE_BUTTON_UP;
  event.time = PR_IntervalNow();
  self->mViewManager->DispatchEvent(&event, &status);
}

PRBool
PanningMode::IsDispatchEvent()
{
  return mHasDispatchEvent;
}

nsresult
PanningMode::ChangeFocus(nsIDOMEvent *aDOMEvent)
{
  NS_ENSURE_TRUE(aDOMEvent, NS_ERROR_FAILURE);
//  if (mFocusElement) {
//    nsCOMPtr<nsIDOMHTMLAnchorElement> anchor = do_QueryInterface(mFocusElement);
//    if(anchor)
//      anchor->Blur();
//    nsCOMPtr<nsIDOMHTMLInputElement> input = do_QueryInterface(mFocusElement);
//    if(input)
//      input->Blur();
//    nsCOMPtr<nsIDOMHTMLSelectElement> select = do_QueryInterface(mFocusElement);
//    if(select)
//      select->Blur();
//    nsCOMPtr<nsIDOMHTMLTextAreaElement> textarea = do_QueryInterface(mFocusElement);
//    if(textarea)
//      textarea->Blur();
//  }

  nsCOMPtr<nsIDOMEventTarget> eventTarget;
  aDOMEvent->GetTarget(getter_AddRefs(eventTarget));
  NS_ENSURE_TRUE(eventTarget, NS_ERROR_FAILURE);
  nsCOMPtr<nsIDOMNode> eventNode = do_QueryInterface(eventTarget);
  NS_ENSURE_TRUE(eventNode, NS_ERROR_FAILURE);
  nsCOMPtr<nsIDOMNode> parentNode;
  nsString uTag;
  while(eventNode) {
    eventNode->GetLocalName(uTag);
    //printf("%s\n", NS_ConvertUTF16toUTF8(uTag).get());
    if (uTag.LowerCaseEqualsLiteral("a")) { // AnchorElement
      nsCOMPtr<nsIDOMHTMLAnchorElement> anchor = do_QueryInterface(eventNode);
      if(anchor)
        anchor->Focus();
      break;
    } else if (uTag.LowerCaseEqualsLiteral("input")) { // InputElement
      nsCOMPtr<nsIDOMHTMLInputElement> input = do_QueryInterface(eventNode);
      if(input)
        input->Focus();
      break;
    } else if (uTag.LowerCaseEqualsLiteral("select")) { // SelectElement
      nsCOMPtr<nsIDOMHTMLSelectElement> select = do_QueryInterface(eventNode);
      if(select)
        select->Focus();
      break;
    } else if (uTag.LowerCaseEqualsLiteral("textarea")) { // TextAreaElement
      nsCOMPtr<nsIDOMHTMLTextAreaElement> textarea = do_QueryInterface(eventNode);
      if(textarea)
        textarea->Focus();
      break;
    }
    eventNode->GetParentNode(getter_AddRefs(parentNode));
    eventNode = parentNode;
  }
  mFocusElement = eventNode;

  return NS_OK;
}
