/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* vim:set ts=2 sw=2 sts=2 tw=80 et cindent: */
/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is Mozilla's Element Optimizeing extension.
 *
 * The Initial Developer of the Original Code is the Mozilla Foundation.
 * Portions created by the Initial Developer are Copyright (C) 2006
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *   Oleg Romashin <romaxa@gmail.com> (original author)
 *   Brad Lassey <blassey@mozilla.com>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

#include "nsICategoryManager.h"
#include "nsIDOMWindow.h"
#include "nsIGenericFactory.h"
#include "nsIObserver.h"
#include "nsWeakReference.h"
#include "nsIObserverService.h"
#include "nsIDOMEventTarget.h"
#include <map>
#include "nsWindowListener.h"
#include "nsServiceManagerUtils.h"
#include "nsPIDOMWindow.h"
#include "nsStringGlue.h"
#include "nsMemory.h"
#include "HelperFunctions.h"
#include "nsIWebBrowser.h"
#include "nsIWindowWatcher.h"
#include "nsIWebBrowserChrome.h"
#include "pluginSupport.h"

class nsWidgetUtils : public nsIObserver, public nsSupportsWeakReference
{
public:
  nsWidgetUtils();
  virtual ~nsWidgetUtils();

  NS_DECL_ISUPPORTS
  NS_DECL_NSIOBSERVER

private:
  nsresult Init(void);
  nsresult GetChromeEventHandler(nsIDOMWindow *aDOMWin,
                                 nsIDOMEventTarget **aChromeTarget);
  nsresult AttachWindowListeners(nsIDOMWindow *aDOMWin);
  nsresult RemoveWindowListeners(nsIDOMWindow *aDOMWin);

private:
  std::map<nsIDOMWindow*, nsWindowListener*> mlistenerContainer;
};

nsWidgetUtils::nsWidgetUtils()
{
  Init();
}

NS_IMETHODIMP
nsWidgetUtils::Init()
{
#ifdef PLUGIN_SUPPORT_ENABLED
  // Install the event filter as early as possible to catch all the events
  PluginSupport::Initialize(mlistenerContainer);
#endif

  nsCOMPtr<nsIObserverService> obsSvc =
      do_GetService("@mozilla.org/observer-service;1");
  NS_ENSURE_STATE(obsSvc);

  nsresult rv = obsSvc->AddObserver(this, "domwindowopened", PR_FALSE);
  NS_ENSURE_SUCCESS(rv, rv);
  rv = obsSvc->AddObserver(this, "domwindowclosed", PR_FALSE);
  NS_ENSURE_SUCCESS(rv, rv);
}

nsresult
nsWidgetUtils::GetChromeEventHandler(nsIDOMWindow *aDOMWin,
                                     nsIDOMEventTarget **aChromeTarget)
{
  NS_ENSURE_TRUE(aDOMWin, NS_ERROR_FAILURE);
  NS_ENSURE_TRUE(aChromeTarget, NS_ERROR_FAILURE);
  nsCOMPtr<nsPIDOMWindow> privateDOMWindow(do_QueryInterface(aDOMWin));
  NS_ENSURE_TRUE(privateDOMWindow, NS_ERROR_FAILURE);
  nsPIDOMEventTarget *chromeEventHandler = nsnull;
  chromeEventHandler = privateDOMWindow->GetChromeEventHandler();
  NS_ENSURE_TRUE(chromeEventHandler, NS_ERROR_FAILURE);
  nsCOMPtr<nsIDOMEventTarget> target(do_QueryInterface(chromeEventHandler));
  NS_ENSURE_TRUE(target, NS_ERROR_FAILURE);
  NS_IF_ADDREF(*aChromeTarget = target);
  return NS_OK;
}

nsresult
nsWidgetUtils::RemoveWindowListeners(nsIDOMWindow *aDOMWin)
{
  NS_ENSURE_TRUE(aDOMWin, NS_ERROR_FAILURE);
  nsCOMPtr<nsIDOMEventTarget> eventHandler;
  GetChromeEventHandler(aDOMWin, getter_AddRefs(eventHandler));
  NS_ENSURE_TRUE(eventHandler, NS_ERROR_FAILURE);

  nsWindowListener *listener = mlistenerContainer[aDOMWin];
  NS_ENSURE_TRUE(listener, NS_ERROR_FAILURE);

  eventHandler->RemoveEventListener(NS_LITERAL_STRING("mousemove"),
                                    static_cast<nsIDOMMouseMotionListener*>(listener),
                                    PR_TRUE);

  eventHandler->RemoveEventListener(NS_LITERAL_STRING("mouseup"),
                                    static_cast<nsIDOMMouseListener*>(listener),
                                    PR_TRUE);

  eventHandler->RemoveEventListener(NS_LITERAL_STRING("focus"),
                                    static_cast<nsIDOMFocusListener*> (listener),
                                    PR_TRUE);

  eventHandler->RemoveEventListener(NS_LITERAL_STRING("blur"),
                                    static_cast<nsIDOMFocusListener*>(listener),
                                    PR_TRUE);

  nsCOMPtr<nsPIDOMEventTarget> piTarget(do_QueryInterface(eventHandler));
  piTarget->RemoveEventListenerByIID(static_cast<nsIDOMDragListener*>(listener),
                                     NS_GET_IID(nsIDOMDragListener));

  nsCOMPtr<nsIWindowWatcher> wwatch = do_GetService("@mozilla.org/embedcomp/window-watcher;1");
  NS_ENSURE_TRUE(wwatch, NS_ERROR_FAILURE);
  nsCOMPtr<nsIWebBrowserChrome> chrome;
  wwatch->GetChromeForWindow(aDOMWin, getter_AddRefs(chrome));
  NS_ENSURE_TRUE(chrome, NS_ERROR_FAILURE);
  nsCOMPtr<nsIWebBrowser> webBrowser;
  chrome->GetWebBrowser(getter_AddRefs(webBrowser));
  NS_ENSURE_TRUE(webBrowser, NS_ERROR_FAILURE);
  nsCOMPtr<nsISupportsWeakReference> supportsWeak;
  supportsWeak = do_QueryInterface(static_cast<nsIWebProgressListener*> (listener));
  NS_ENSURE_TRUE(supportsWeak, NS_ERROR_FAILURE);
  nsCOMPtr<nsIWeakReference> weakRef;
  supportsWeak->GetWeakReference(getter_AddRefs(weakRef));
  NS_ENSURE_TRUE(weakRef, NS_ERROR_FAILURE);
  webBrowser->RemoveWebBrowserListener(weakRef, NS_GET_IID(nsIWebProgressListener));

  mlistenerContainer.erase(aDOMWin);
  //!!!libxul will delete it, we shouldn't do it.
  //delete listener;
  return NS_OK;
}

nsresult
nsWidgetUtils::AttachWindowListeners(nsIDOMWindow *aDOMWin)
{
  NS_ENSURE_TRUE(aDOMWin, NS_ERROR_FAILURE);
  nsCOMPtr<nsIDOMEventTarget> eventHandler;
  GetChromeEventHandler(aDOMWin, getter_AddRefs(eventHandler));
  NS_ENSURE_TRUE(eventHandler, NS_ERROR_FAILURE);

  nsWindowListener *listener = new nsWindowListener();
  NS_ENSURE_TRUE(listener, NS_ERROR_OUT_OF_MEMORY);

  mlistenerContainer[aDOMWin] = listener;
  eventHandler->AddEventListener(NS_LITERAL_STRING("mousemove"),
                                 static_cast<nsIDOMMouseMotionListener*>(listener),
                                 PR_TRUE);

  eventHandler->AddEventListener(NS_LITERAL_STRING("mouseup"),
                                 static_cast<nsIDOMMouseListener*>(listener),
                                 PR_TRUE);

  eventHandler->AddEventListener(NS_LITERAL_STRING("focus"),
                                 static_cast<nsIDOMFocusListener*>(listener),
                                 PR_TRUE);

  eventHandler->AddEventListener(NS_LITERAL_STRING("blur"),
                                 static_cast<nsIDOMFocusListener*>(listener),
                                 PR_TRUE);

  nsCOMPtr<nsPIDOMEventTarget> piTarget(do_QueryInterface(eventHandler));
  piTarget->AddEventListenerByIID(static_cast<nsIDOMDragListener*>(listener),
                                  NS_GET_IID(nsIDOMDragListener));

  nsCOMPtr<nsIWindowWatcher> wwatch = do_GetService("@mozilla.org/embedcomp/window-watcher;1");
  NS_ENSURE_TRUE(wwatch, NS_ERROR_FAILURE);
  nsCOMPtr<nsIWebBrowserChrome> chrome;
  wwatch->GetChromeForWindow(aDOMWin, getter_AddRefs(chrome));
  NS_ENSURE_TRUE(chrome, NS_ERROR_FAILURE);
  nsCOMPtr<nsIWebBrowser> webBrowser;
  chrome->GetWebBrowser(getter_AddRefs(webBrowser));
  NS_ENSURE_TRUE(webBrowser, NS_ERROR_FAILURE);
  nsCOMPtr<nsISupportsWeakReference> supportsWeak;
  supportsWeak = do_QueryInterface(static_cast<nsIWebProgressListener*>(listener));
  NS_ENSURE_TRUE(supportsWeak, NS_ERROR_FAILURE);
  nsCOMPtr<nsIWeakReference> weakRef;
  supportsWeak->GetWeakReference(getter_AddRefs(weakRef));
  NS_ENSURE_TRUE(weakRef, NS_ERROR_FAILURE);
  webBrowser->AddWebBrowserListener(weakRef, NS_GET_IID(nsIWebProgressListener));

  return NS_OK;
}

nsWidgetUtils::~nsWidgetUtils()
{
#ifdef PLUGIN_SUPPORT_ENABLED
  PluginSupport::Uninitialize(mlistenerContainer);
#endif
}

NS_IMPL_ISUPPORTS2(nsWidgetUtils, nsIObserver, nsISupportsWeakReference)

NS_IMETHODIMP
nsWidgetUtils::Observe(nsISupports *aSubject, const char *aTopic, const PRUnichar *aData)
{
  NS_ENSURE_TRUE(aSubject, NS_OK);
  NS_ENSURE_TRUE(aTopic, NS_OK);
  PRBool tempBool = PR_FALSE;
  nsresult rv = HelperFunctions::GetPref(PREF_TYPE_BOOL, "webaddon.widgetutils.enabled",
                                        &tempBool);
  NS_ENSURE_SUCCESS(rv, NS_OK);
  NS_ENSURE_TRUE(tempBool, NS_OK);

  if (!strcmp(aTopic, "domwindowopened")) {
    nsCOMPtr<nsIDOMWindow> chromeWindow = do_QueryInterface(aSubject);
    if (chromeWindow)
      AttachWindowListeners(chromeWindow);
    return NS_OK;
  }

  if (!strcmp(aTopic, "domwindowclosed")) {
    nsCOMPtr<nsIDOMWindow> chromeWindow = do_QueryInterface(aSubject);
    if (chromeWindow)
      RemoveWindowListeners(chromeWindow);
    return NS_OK;
  }
  return NS_OK;
}

//------------------------------------------------------------------------------
//  XPCOM REGISTRATION BELOW
//------------------------------------------------------------------------------

#define WidgetUtils_CID \
	{  0x5e24d411, 0xfd6d, 0xfd6d, \
		{0xa8, 0x3a, 0x3a, 0x51, 0x0a, 0x95, 0x9a, 0x7b} }

#define WidgetUtils_ContractID "@browser.garage.maemo.org/microb/widgetutils;1"

static NS_METHOD WidgetUtilsRegistration(nsIComponentManager *aCompMgr,
                                         nsIFile *aPath,
                                         const char *registryLocation,
                                         const char *componentType,
                                         const nsModuleComponentInfo *info)
{
  NS_ENSURE_TRUE(aCompMgr, NS_ERROR_FAILURE);
  nsresult rv = NS_ERROR_FAILURE;
  nsCOMPtr<nsIServiceManager> servman =
      do_QueryInterface((nsISupports*)aCompMgr, &rv);
  NS_ENSURE_SUCCESS(rv, rv);

  NS_ENSURE_TRUE(servman, NS_ERROR_FAILURE);
  nsCOMPtr<nsICategoryManager> catman;
  servman->GetServiceByContractID(NS_CATEGORYMANAGER_CONTRACTID,
                                  NS_GET_IID(nsICategoryManager),
                                  getter_AddRefs(catman));
  NS_ENSURE_TRUE(catman, NS_ERROR_FAILURE);

  char *previous = nsnull;
  rv = catman->AddCategoryEntry("app-startup", "WidgetUtils",
                                WidgetUtils_ContractID, PR_TRUE,
                                PR_TRUE, &previous);
  if (previous)
    nsMemory::Free(previous);

  return rv;
}

static NS_METHOD WidgetUtilsUnregistration(nsIComponentManager *aCompMgr,
                                           nsIFile *aPath,
                                           const char *registryLocation,
                                           const nsModuleComponentInfo *info)
{
  NS_ENSURE_TRUE(aCompMgr, NS_ERROR_FAILURE);
  nsresult rv = NS_ERROR_FAILURE;
  nsCOMPtr<nsIServiceManager> servman =
      do_QueryInterface((nsISupports*)aCompMgr, &rv);
  NS_ENSURE_SUCCESS(rv, rv);

  NS_ENSURE_TRUE(servman, NS_ERROR_FAILURE);
  nsCOMPtr<nsICategoryManager> catman;
  servman->GetServiceByContractID(NS_CATEGORYMANAGER_CONTRACTID,
                                  NS_GET_IID(nsICategoryManager),
                                  getter_AddRefs(catman));
  NS_ENSURE_TRUE(catman, NS_ERROR_FAILURE);

  return catman->DeleteCategoryEntry("app-startup", "WidgetUtils", PR_TRUE);
}

NS_GENERIC_FACTORY_CONSTRUCTOR(nsWidgetUtils)

static const nsModuleComponentInfo components[] =
{
  { "nsWidgetUtilsService",
    WidgetUtils_CID,
    WidgetUtils_ContractID,
    nsWidgetUtilsConstructor,
    WidgetUtilsRegistration,
    WidgetUtilsUnregistration
  }
};

NS_IMPL_NSGETMODULE(nsWidgetUtilsModule, components)
