/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is mozilla.org code.
 *
 * The Initial Developer of the Original Code is
 * Christopher Blizzard.
 * Portions created by the Initial Developer are Copyright (C) 2001
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *   Christopher Blizzard <blizzard@mozilla.org>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

#include <nsIDocShell.h>
#include <nsIWebProgress.h>
#include "nsIWidget.h"
#include "nsCRT.h"
#include "nsNetUtil.h"
#include "nsIWebBrowserStream.h"
#include "nsIWebBrowserFocus.h"
#include "nsIDirectoryService.h"
#include "nsAppDirectoryServiceDefs.h"

// for do_GetInterface
#include <nsIInterfaceRequestor.h>
// for do_CreateInstance
#include <nsIComponentManager.h>

// for initializing our window watcher service
#include <nsIWindowWatcher.h>

#include <nsILocalFile.h>

#include "nsXULAppAPI.h"

// all of the crap that we need for event listeners
// and when chrome windows finish loading
#include <nsIDOMWindow.h>
#include <nsPIDOMWindow.h>
#include <nsIDOMWindowInternal.h>

// For seting scrollbar visibilty
#include <nsIDOMBarProp.h>

// for the focus hacking we need to do
#include <nsIFocusController.h>

// app component registration
#include <nsIGenericFactory.h>
#include <nsIComponentRegistrar.h>

// all of our local includes
#include "EmbedPrivate.h"
#include "EmbedWindow.h"
#include "EmbedProgress.h"
#include "EmbedContentListener.h"
#include "EmbedEventListener.h"
#include "EmbedWindowCreator.h"

#ifdef MOZ_ACCESSIBILITY_ATK
#include "nsIAccessibilityService.h"
#include "nsIAccessible.h"
#include "nsIDOMDocument.h"
#endif

#include "EmbedGtkTools.h"

#include "nsEmbedCID.h"
#include "nsIFormControl.h"
#include "nsICacheService.h"
#include "nsIDOMNSHTMLTextAreaElement.h"
#include "nsIDOMHTMLInputElement.h"
#include "nsIDOMNSHTMLInputElement.h"
#include "nsIDOMHTMLTextAreaElement.h"
#include "nsIEditingSession.h"
#include "nsIEditor.h"
#include "nsIPlaintextEditor.h"
#include "nsIHTMLEditor.h"
#include "nsIDocument.h"
#include "nsIDOMDocumentView.h"
#include "nsIDOMAbstractView.h"
#include "nsIDOMNSDocument.h"
#include "nsICacheSession.h"
#include "nsIHistoryEntry.h"
#include "nsIDocCharset.h"
#include "nsIMarkupDocumentViewer.h"
#include "nsIDOMWindowCollection.h"
#include "nsIContentViewer.h"
#include "nsIWebBrowserFind.h"
#include "nsICommandManager.h"
#include "nsIDOMNSEditableElement.h"
#include "nsISelectionController.h"
#include "nsWidgetsCID.h"

#include "nsILoginManager.h"
#include "nsILoginInfo.h"
#include "nsIDOMEventTarget.h"
#include "nsIDOMNSHTMLElement.h"
#include "imgIRequest.h"

#include <sys/types.h>
#include <unistd.h>
#include <signal.h>

#if 1
#define OSSOLOG_COMPILE 1
#include <osso-log.h>
#else
#define ULOG_INFO(FMT, ARG...) printf(\
     "EXTR_LOG:%s:%s:%d: " FMT "\n", __FILE__, __FUNCTION__, __LINE__, ## ARG)
#endif

static NS_DEFINE_CID(kAppShellCID,               NS_APPSHELL_CID);
static NS_DEFINE_CID(kCacheServiceCID,           NS_CACHESERVICE_CID);
static nsICacheService* sCacheService;

static EmbedCommon* sEmbedCommon = nsnull;

/* static */
EmbedCommon*
EmbedCommon::GetInstance()
{
  if (!sEmbedCommon)
  {
    sEmbedCommon = new EmbedCommon();
    if (!sEmbedCommon)
      return nsnull;
    if (NS_FAILED(sEmbedCommon->Init()))
    {
      return nsnull;
    }
  }
  return sEmbedCommon;
}

/* static */
void
EmbedCommon::DeleteInstance()
{
  if (sEmbedCommon)
  {
    delete sEmbedCommon;
    sEmbedCommon = nsnull;
  }
}

nsresult
EmbedCommon::Init(void)
{
    mCommon = NULL;
    return NS_OK;
}

PRUint32     EmbedPrivate::sWidgetCount = 0;
PRBool       EmbedPrivate::sXpcomStarted = PR_FALSE;

char        *EmbedPrivate::sPath        = nsnull;
char        *EmbedPrivate::sCompPath    = nsnull;
nsVoidArray *EmbedPrivate::sWindowList  = nsnull;
nsILocalFile *EmbedPrivate::sProfileDir  = nsnull;
nsISupports  *EmbedPrivate::sProfileLock = nsnull;

nsIDirectoryServiceProvider *EmbedPrivate::sAppFileLocProvider = nsnull;

GtkMozEmbed*
EmbedCommon::GetAnyLiveWidget()
{
  if (!EmbedPrivate::sWidgetCount || !EmbedPrivate::sWindowList)
    return nsnull;

  // Get the number of browser windows.
  PRInt32 count = EmbedPrivate::sWindowList->Count();
  // This function doesn't get called very often at all (only when
  // creating a new window) so it's OK to walk the list of open
  // windows.
  //FIXME need to choose right window
  GtkMozEmbed *ret = nsnull;
  for (int i = 0; i < count; i++) {
    EmbedPrivate *tmpPrivate = static_cast<EmbedPrivate *>
                                          (EmbedPrivate::sWindowList->ElementAt(i));
    ret = tmpPrivate->mOwningWidget;
  }
  return ret;
}

nsresult
EmbedCommon::SuspendNative(PRBool aOn) {
static PRBool suspended = PR_FALSE;
   if (suspended == aOn)
     return NS_OK;
   nsCOMPtr<nsIAppShell> appShell = do_GetService(kAppShellCID);
   NS_ENSURE_TRUE(appShell, NS_ERROR_FAILURE);
   suspended = aOn;
   return aOn?appShell->SuspendNative():appShell->ResumeNative();
}

class GTKEmbedDirectoryProvider : public nsIDirectoryServiceProvider2
{
public:
  NS_DECL_ISUPPORTS_INHERITED
  NS_DECL_NSIDIRECTORYSERVICEPROVIDER
  NS_DECL_NSIDIRECTORYSERVICEPROVIDER2
};

static const GTKEmbedDirectoryProvider kDirectoryProvider;

NS_IMPL_QUERY_INTERFACE2(GTKEmbedDirectoryProvider,
                         nsIDirectoryServiceProvider,
                         nsIDirectoryServiceProvider2)

NS_IMETHODIMP_(nsrefcnt)
GTKEmbedDirectoryProvider::AddRef()
{
  return 1;
}

NS_IMETHODIMP_(nsrefcnt)
GTKEmbedDirectoryProvider::Release()
{
  return 1;
}

NS_IMETHODIMP
GTKEmbedDirectoryProvider::GetFile(const char *aKey, PRBool *aPersist,
                                   nsIFile* *aResult)
{
  if (EmbedPrivate::sAppFileLocProvider) {
    nsresult rv = EmbedPrivate::sAppFileLocProvider->GetFile(aKey, aPersist,
                                                             aResult);
    if (NS_SUCCEEDED(rv))
      return rv;
  }

  if (EmbedPrivate::sProfileDir &&
      (!strcmp(aKey, NS_APP_USER_PROFILE_50_DIR)
       || !strcmp(aKey, NS_APP_PROFILE_DIR_STARTUP)
       || !strcmp(aKey, NS_APP_USER_PROFILE_LOCAL_50_DIR)
      )
  ) {
    *aPersist = PR_TRUE;
    return EmbedPrivate::sProfileDir->Clone(aResult);
  }

  *aResult = nsnull;
  return NS_ERROR_FAILURE;
}

NS_IMETHODIMP
GTKEmbedDirectoryProvider::GetFiles(const char *aKey,
                                    nsISimpleEnumerator* *aResult)
{
  nsCOMPtr<nsIDirectoryServiceProvider2>
    dp2(do_QueryInterface(EmbedPrivate::sAppFileLocProvider));

  if (!dp2)
    return NS_ERROR_FAILURE;

  return dp2->GetFiles(aKey, aResult);
}

static const nsModuleComponentInfo defaultAppComps[] = {
// content handler component info
};

const nsModuleComponentInfo *EmbedPrivate::sAppComps = defaultAppComps;
int   EmbedPrivate::sNumAppComps = sizeof(defaultAppComps) / sizeof(nsModuleComponentInfo);

const nsStaticModuleInfo *EmbedPrivate::sStaticComps = nsnull;
int   EmbedPrivate::sNumStaticComps = 0;

EmbedPrivate::EmbedPrivate(void)
{
  mOwningWidget     = nsnull;
  mWindow           = nsnull;
  mProgress         = nsnull;
  mContentListener  = nsnull;
  mEventListener    = nsnull;
  mChromeMask       = nsIWebBrowserChrome::CHROME_ALL;
  mIsChrome         = PR_FALSE;
  mChromeLoaded     = PR_FALSE;
  mLoadFinished     = PR_TRUE;
  mListenersAttached = PR_FALSE;
  mIsDestroyed      = PR_FALSE;
  mOpenBlock        = PR_FALSE;
  mIsUploading      = PR_FALSE;
  mInitialized      = PR_FALSE;

  PushStartup();
  if (!sWindowList) {
    sWindowList = new nsVoidArray();
  }
  sWindowList->AppendElement(this);
  // increment the number of widgets
  sWidgetCount++;
}

EmbedPrivate::~EmbedPrivate()
{
  sWindowList->RemoveElement(this);
  sWidgetCount--;
  if (mProgress)
    mProgress->Shutdown();
  if (mEventListener)
    mEventListener->Shutdown();
  mOwningWidget = nsnull;
  mInitialized = PR_FALSE;
  if (sWidgetCount) return;
  ULOG_INFO("Destroy last window:%p", this);
  gboolean bval = FALSE;
  if (gtk_moz_embed_common_get_pref (G_TYPE_BOOLEAN, (gchar*)"gtkmozembed.no_destroy_on_last_window", &bval) && bval)
    return;

  int pid = getpid();
  ULOG_INFO("Before Kill 9 :%p", this);
  kill (pid, SIGKILL);
}

nsresult
EmbedPrivate::Init(GtkMozEmbed *aOwningWidget)
{
  // are we being re-initialized?
  if (mOwningWidget)
    return NS_OK;

  // hang on with a reference to the owning widget
  mOwningWidget = aOwningWidget;

  // Create our embed window, and create an owning reference to it and
  // initialize it.  It is assumed that this window will be destroyed
  // when we go out of scope.
  mWindow = new EmbedWindow();
  mWindowGuard = static_cast<nsIWebBrowserChrome *>(mWindow);
  mWindow->Init(this);

  // Create our progress listener object, make an owning reference,
  // and initialize it.  It is assumed that this progress listener
  // will be destroyed when we go out of scope.
  mProgress = new EmbedProgress();
  mProgressGuard = static_cast<nsIWebProgressListener *>
                                (mProgress);
  mProgress->Init(this);

  // Create our content listener object, initialize it and attach it.
  // It is assumed that this will be destroyed when we go out of
  // scope.
  mContentListener = new EmbedContentListener();
  mContentListenerGuard = static_cast<nsISupports*>(static_cast<nsIURIContentListener*>(mContentListener));
  mContentListener->Init(this);

  // Create our key listener object and initialize it.  It is assumed
  // that this will be destroyed before we go out of scope.
  mEventListener = new EmbedEventListener();
  mEventListenerGuard =
    static_cast<nsISupports *>(static_cast<nsIDOMKeyListener *>(mEventListener));
  mEventListener->Init(this);

  // has the window creator service been set up?
  static int initialized = PR_FALSE;
  // Set up our window creator ( only once )
  if (!initialized) {
    // We set this flag here instead of on success.  If it failed we
    // don't want to keep trying and leaking window creator objects.
    initialized = PR_TRUE;

    // create our local object
    EmbedWindowCreator *creator = new EmbedWindowCreator(&mOpenBlock);
    nsCOMPtr<nsIWindowCreator> windowCreator;
    windowCreator = static_cast<nsIWindowCreator *>(creator);

    // Attach it via the watcher service
    nsCOMPtr<nsIWindowWatcher> watcher = do_GetService(NS_WINDOWWATCHER_CONTRACTID);
    if (watcher)
      watcher->SetWindowCreator(windowCreator);
  }

  // Ensure that loginmgr is running
  nsCOMPtr<nsILoginManager> loginmgr(
    do_GetService("@mozilla.org/login-manager;1"));

  return NS_OK;
}

nsresult
EmbedPrivate::Realize(PRBool *aAlreadyRealized, PRBool notRealized)
{
  if (!mWindow)
    return NS_ERROR_FAILURE;

  *aAlreadyRealized = PR_FALSE;


    mInitialized = PR_TRUE;

  // Get the nsIWebBrowser object for our embedded window.
  nsCOMPtr<nsIWebBrowser> webBrowser;
  mWindow->GetWebBrowser(getter_AddRefs(webBrowser));

  // get a handle on the navigation object
  mNavigation = do_QueryInterface(webBrowser);

  // Create our session history object and tell the navigation object
  // to use it.  We need to do this before we create the web browser
  // window.
  mSessionHistory = do_CreateInstance(NS_SHISTORY_CONTRACTID);
  if (mSessionHistory)
  mNavigation->SetSessionHistory(mSessionHistory);

  // create the window
  nsresult rv = mWindow->CreateWindow();
  if (NS_FAILED(rv))
    return rv;

  // bind the progress listener to the browser object
  nsCOMPtr<nsISupportsWeakReference> supportsWeak;
  supportsWeak = do_QueryInterface(mProgressGuard);
  nsCOMPtr<nsIWeakReference> weakRef;
  supportsWeak->GetWeakReference(getter_AddRefs(weakRef));
  webBrowser->AddWebBrowserListener(weakRef,
				    NS_GET_IID(nsIWebProgressListener));

  // set ourselves as the parent uri content listener
  nsCOMPtr<nsIURIContentListener> uriListener;
  uriListener = do_QueryInterface(mContentListenerGuard);
  webBrowser->SetParentURIContentListener(uriListener);

  // Apply the current chrome mask
  ApplyChromeMask();

  mInitialized = PR_TRUE;

  return NS_OK;
}

void
EmbedPrivate::Unrealize(void)
{
  mInitialized = PR_FALSE;
}

void
EmbedPrivate::Show(void)
{
  if (!mWindow)
    return;

  // Get the nsIWebBrowser object for our embedded window.
  nsCOMPtr<nsIWebBrowser> webBrowser;
  mWindow->GetWebBrowser(getter_AddRefs(webBrowser));

  // and set the visibility on the thing
  nsCOMPtr<nsIBaseWindow> baseWindow = do_QueryInterface(webBrowser);

  // XXX hack around problem. probably widget/gtk2 window initialization.
  NS_ENSURE_TRUE(baseWindow, );
  baseWindow->SetVisibility(PR_FALSE);
  mWindow->mHidden = 0;
  baseWindow->SetVisibility(PR_TRUE);
}

void
EmbedPrivate::Hide(void)
{
  if (!mWindow)
    return;

  // Get the nsIWebBrowser object for our embedded window.
  nsCOMPtr<nsIWebBrowser> webBrowser;
  mWindow->GetWebBrowser(getter_AddRefs(webBrowser));

  // and set the visibility on the thing
  nsCOMPtr<nsIBaseWindow> baseWindow = do_QueryInterface(webBrowser);
  baseWindow->SetVisibility(PR_FALSE);
}

void
EmbedPrivate::Resize(PRUint32 aWidth, PRUint32 aHeight)
{
  if (mIsDestroyed || !mWindow)
    return;

  mWindow->SetDimensions(nsIEmbeddingSiteWindow::DIM_FLAGS_POSITION |
			 nsIEmbeddingSiteWindow::DIM_FLAGS_SIZE_INNER,
			 0, 0, aWidth, aHeight);
}

void
EmbedPrivate::Destroy(void)
{
  // This flag might have been set from
  // EmbedWindow::DestroyBrowserWindow() as well if someone used a
  // window.close() or something or some other script action to close
  // the window.  No harm setting it again.

  mIsDestroyed = PR_TRUE;

  // Get the nsIWebBrowser object for our embedded window.
  nsCOMPtr<nsIWebBrowser> webBrowser;
  mWindow->GetWebBrowser(getter_AddRefs(webBrowser));

  // Release our progress listener
  nsCOMPtr<nsISupportsWeakReference> supportsWeak;
  supportsWeak = do_QueryInterface(mProgressGuard);
  nsCOMPtr<nsIWeakReference> weakRef;
  supportsWeak->GetWeakReference(getter_AddRefs(weakRef));
  webBrowser->RemoveWebBrowserListener(weakRef,
				       NS_GET_IID(nsIWebProgressListener));
  weakRef = nsnull;
  supportsWeak = nsnull;

  // Release our content listener
  webBrowser->SetParentURIContentListener(nsnull);
  mContentListenerGuard = nsnull;
  mContentListener = nsnull;

  // Now that we have removed the listener, release our progress
  // object
  mProgressGuard = nsnull;
  mProgress = nsnull;

  // detach our event listeners and release the event receiver
  DetachListeners();
  if (mEventTarget)
    mEventTarget = nsnull;

  mEventListenerGuard = nsnull;
  mEventListener = nsnull;

  // destroy our child window
  mWindow->ReleaseChildren();

  // release navigation
  mNavigation = nsnull;

  // release session history
  mSessionHistory = nsnull;

  mOwningWidget = nsnull;
}

void
EmbedPrivate::SetURI(const char *aURI)
{
  mURI.Assign(aURI);
}

void
EmbedPrivate::LoadCurrentURI(void)
{
  if (mURI.Length() && !mIsDestroyed && mWindow) {
    nsCOMPtr<nsPIDOMWindow> piWin;
    GetPIDOMWindow(getter_AddRefs(piWin));

    nsAutoPopupStatePusher popupStatePusher(piWin, openAllowed);

    mWindow->mHidden = 1;
    NS_ENSURE_TRUE(mNavigation, );
    mNavigation->LoadURI(NS_ConvertUTF8toUTF16(mURI).get(), // URI string
                         nsIWebNavigation::LOAD_FLAGS_NONE | // Load flags
                         nsIWebNavigation::LOAD_FLAGS_ALLOW_THIRD_PARTY_FIXUP,  // Allow keyword.URL.. functionality
                         nsnull,                            // Referring URI
                         nsnull,                            // Post data
                         nsnull);                           // extra headers
  }
}

void
EmbedPrivate::Reload(PRUint32 reloadFlags)
{
  /* Use the session history if it is available, this
   * allows framesets to reload correctly */
  nsCOMPtr<nsIWebNavigation> wn;
  PRInt32 count = 0;

  mSessionHistory->GetCount(&count);
  if (mSessionHistory && count) {
    wn = do_QueryInterface(mSessionHistory);
  }
  if (!wn)
    wn = mNavigation;

  if (wn)
    wn->Reload(reloadFlags);
}


void
EmbedPrivate::ApplyChromeMask()
{
   if (mWindow) {
      nsCOMPtr<nsIWebBrowser> webBrowser;
      mWindow->GetWebBrowser(getter_AddRefs(webBrowser));

      nsCOMPtr<nsIDOMWindow> domWindow;
      webBrowser->GetContentDOMWindow(getter_AddRefs(domWindow));
      if (domWindow) {

         nsCOMPtr<nsIDOMBarProp> scrollbars;
         domWindow->GetScrollbars(getter_AddRefs(scrollbars));
         if (scrollbars) {

            scrollbars->SetVisible
               (mChromeMask & nsIWebBrowserChrome::CHROME_SCROLLBARS ?
                PR_TRUE : PR_FALSE);
         }
      }
   }
}


void
EmbedPrivate::SetChromeMask(PRUint32 aChromeMask)
{
  mChromeMask = aChromeMask;
  mIsChrome = (mChromeMask & nsIWebBrowserChrome::CHROME_OPENAS_CHROME);

  ApplyChromeMask();
}


/* static */
void
EmbedPrivate::PushStartup(void)
{
  // if this is the first widget, fire up xpcom
  if (sWidgetCount > 0 || sXpcomStarted)
		return;

  nsresult rv;
	nsCOMPtr<nsILocalFile> binDir;
	nsCOMPtr<nsILocalFile> compDir;
	if (EmbedPrivate::sCompPath) {
		rv = NS_NewNativeLocalFile(nsDependentCString(EmbedPrivate::sCompPath), 1, getter_AddRefs(binDir));
		rv = NS_NewNativeLocalFile(nsDependentCString(EmbedPrivate::sCompPath), 1, getter_AddRefs(compDir));
		if (NS_FAILED(rv))
			return;
		PRBool exists;
		rv = compDir->AppendNative(nsDependentCString("components"));
		compDir->Exists(&exists);
		if (!exists)
			rv = compDir->Create(nsIFile::DIRECTORY_TYPE, 0700);
		if (NS_FAILED(rv))
			return;
	} else
		NS_ASSERTION(EmbedPrivate::sCompPath, "Warning: Failed to init Component Path.\n");

	const char *grePath = sPath;
	NS_ASSERTION(grePath, "Warning: Failed to init grePath.\n");

  if (!grePath)
    grePath = getenv("MOZILLA_FIVE_HOME");

  if (!grePath)
    return;

  nsCOMPtr<nsILocalFile> greDir;
  rv = NS_NewNativeLocalFile(nsDependentCString(grePath), PR_TRUE,
                             getter_AddRefs(greDir));
  if (NS_FAILED(rv))
    return;

  if (sProfileDir && !sProfileLock) {
    rv = XRE_LockProfileDirectory(sProfileDir,
                                  &sProfileLock);
    if (NS_FAILED(rv)) return;
  }

  rv = XRE_InitEmbedding(greDir, binDir,
                         const_cast<GTKEmbedDirectoryProvider*>(&kDirectoryProvider),
                         sStaticComps, sNumStaticComps);
  if (NS_FAILED(rv))
    return;

  if (sProfileDir)
		XRE_NotifyProfile();

	rv = RegisterAppComponents();
	NS_ASSERTION(NS_SUCCEEDED(rv), "Warning: Failed to register app components.\n");
  sXpcomStarted = PR_TRUE;
}

/* static */
void
EmbedPrivate::PopStartup(void)
{
  if (sWidgetCount != 0 || !sXpcomStarted)
    return;

  EmbedCommon::DeleteInstance();

  NS_IF_RELEASE(sCacheService);

  // we no longer need a reference to the DirectoryServiceProvider
  if (sAppFileLocProvider) {
    NS_RELEASE(sAppFileLocProvider);
    sAppFileLocProvider = nsnull;
  }

  // shut down XPCOM/Embedding
  ULOG_INFO("Before Term Embedding");
  sXpcomStarted = PR_FALSE;
  XRE_TermEmbedding();
  ULOG_INFO("After Term Embedding");

  NS_IF_RELEASE(sProfileLock);
  NS_IF_RELEASE(sProfileDir);
}

/* static */
void EmbedPrivate::SetPath(const char *aPath)
{
  if (sPath)
    free(sPath);
  if (aPath) {
    sPath = strdup(aPath);
    setenv("MOZILLA_FIVE_HOME", sPath, 0);
  }
  else
    sPath = nsnull;
}

/* static */
void
EmbedPrivate::SetCompPath(const char *aPath)
{
  if (sCompPath)
    free(sCompPath);
  if (aPath)
    sCompPath = strdup(aPath);
  else
    sCompPath = nsnull;
}

/* static */
void
EmbedPrivate::SetAppComponents(const nsModuleComponentInfo* aComps,
                               int aNumComponents)
{
  sAppComps = aComps;
  sNumAppComps = aNumComponents;
}

void
EmbedPrivate::SetStaticComponents(const nsStaticModuleInfo* aComps,
                                  int aNumComponents)
{
  sStaticComps = aComps;
  sNumStaticComps = aNumComponents;
}

/* static */
void
EmbedPrivate::SetProfilePath(const char *aDir, const char *aName)
{
  if (sProfileDir) {
    if (sWidgetCount) {
      NS_ERROR("Cannot change profile directory during run.");
      return;
    }

    NS_RELEASE(sProfileDir);
    NS_RELEASE(sProfileLock);
  }

  nsresult rv =
    NS_NewNativeLocalFile(nsDependentCString(aDir), PR_TRUE, &sProfileDir);

  if (NS_SUCCEEDED(rv) && aName)
    rv = sProfileDir->AppendNative(nsDependentCString(aName));

  if (NS_SUCCEEDED(rv)) {
    PRBool exists = PR_FALSE;
    rv = sProfileDir->Exists(&exists);
    if (!exists)
      rv = sProfileDir->Create(nsIFile::DIRECTORY_TYPE, 0700);
    rv = XRE_LockProfileDirectory(sProfileDir, &sProfileLock);
  }

  if (NS_SUCCEEDED(rv)) {
    if (sWidgetCount)
      XRE_NotifyProfile();

    return;
  }

  NS_WARNING("Failed to lock profile.");

  // Failed
  NS_IF_RELEASE(sProfileDir);
  NS_IF_RELEASE(sProfileLock);
}

void
EmbedPrivate::SetDirectoryServiceProvider(nsIDirectoryServiceProvider * appFileLocProvider)
{
  if (sAppFileLocProvider)
    NS_RELEASE(sAppFileLocProvider);

  if (appFileLocProvider) {
    sAppFileLocProvider = appFileLocProvider;
    NS_ADDREF(sAppFileLocProvider);
  }
}

nsresult
EmbedPrivate::OpenStream(const char *aBaseURI, const char *aContentType)
{
  nsCOMPtr<nsIWebBrowser> webBrowser;
  if (!mWindow) return NS_ERROR_FAILURE;
  mWindow->GetWebBrowser(getter_AddRefs(webBrowser));

  nsCOMPtr<nsIWebBrowserStream> wbStream = do_QueryInterface(webBrowser);
  if (!wbStream) return NS_ERROR_FAILURE;

  nsCOMPtr<nsIURI> uri;
  nsresult rv = NS_NewURI(getter_AddRefs(uri), aBaseURI);
  if (NS_FAILED(rv))
    return rv;

  rv = wbStream->OpenStream(uri, nsDependentCString(aContentType));
  return rv;
}

nsresult
EmbedPrivate::AppendToStream(const PRUint8 *aData, PRUint32 aLen)
{
  // Attach listeners to this document since in some cases we don't
  // get updates for content added this way.
  ContentStateChange();

  nsCOMPtr<nsIWebBrowser> webBrowser;
  if (!mWindow) return NS_ERROR_FAILURE;
  mWindow->GetWebBrowser(getter_AddRefs(webBrowser));

  nsCOMPtr<nsIWebBrowserStream> wbStream = do_QueryInterface(webBrowser);
  if (!wbStream) return NS_ERROR_FAILURE;

  return wbStream->AppendToStream(aData, aLen);
}

nsresult
EmbedPrivate::CloseStream(void)
{
  nsCOMPtr<nsIWebBrowser> webBrowser;
  if (!mWindow) return NS_ERROR_FAILURE;
  mWindow->GetWebBrowser(getter_AddRefs(webBrowser));

  nsCOMPtr<nsIWebBrowserStream> wbStream = do_QueryInterface(webBrowser);
  if (!wbStream) return NS_ERROR_FAILURE;

  return wbStream->CloseStream();
}

/* static */
EmbedPrivate *
EmbedPrivate::FindPrivateForBrowser(nsIWebBrowserChrome *aBrowser)
{
  if (!sWindowList)
    return nsnull;

  // Get the number of browser windows.
  PRInt32 count = sWindowList->Count();
  // This function doesn't get called very often at all ( only when
  // creating a new window ) so it's OK to walk the list of open
  // windows.
  for (int i = 0; i < count; i++) {
    EmbedPrivate *tmpPrivate = static_cast<EmbedPrivate *>(
					      sWindowList->ElementAt(i));
    // get the browser object for that window
    nsIWebBrowserChrome *chrome = static_cast<nsIWebBrowserChrome *>(
						 tmpPrivate->mWindow);
    if (chrome == aBrowser)
      return tmpPrivate;
  }

  return nsnull;
}

void
EmbedPrivate::ContentStateChange(void)
{

  // we don't attach listeners to chrome
  if (mListenersAttached && !mIsChrome)
    return;

  GetListener();

  if (!mEventTarget)
    return;

  AttachListeners();

  mProgress->SetLoginAttached(PR_FALSE);
}

PRUint32
EmbedPrivate::GetLoginsForCurrentHost(nsILoginInfo*** logins)
{
    PRUint32 count = 0;
    
    nsCOMPtr<nsILoginManager> loginmgr(
        do_GetService("@mozilla.org/login-manager;1"));
    
    if (!loginmgr) {
        return count;
    }
    
    nsresult rv;
    
    nsString empty;
    rv = loginmgr->FindLogins(&count, NS_ConvertUTF8toUTF16(mPrePath),
        empty, empty, logins);
    NS_ENSURE_SUCCESS(rv, count);
    
    return count;
}

void
EmbedPrivate::ContentFinishedLoading(void)
{
  if (mIsChrome) {
    // We're done loading.
    mChromeLoaded = PR_TRUE;

    NS_ENSURE_TRUE(mWindow, );
    // get the web browser
    nsCOMPtr<nsIWebBrowser> webBrowser;
    mWindow->GetWebBrowser(getter_AddRefs(webBrowser));

    // get the content DOM window for that web browser
    nsCOMPtr<nsIDOMWindow> domWindow;
    webBrowser->GetContentDOMWindow(getter_AddRefs(domWindow));
    if (!domWindow) {
      NS_WARNING("no dom window in content finished loading\n");
      return;
    }

    // resize the content
    PRUint32 chromeFlags;
    mWindow->GetChromeFlags(&chromeFlags);
    if (!(chromeFlags & GTK_MOZ_EMBED_FLAG_WINDOWRESIZEON))
      domWindow->SizeToContent();

    // and since we're done loading show the window, assuming that the
    // visibility flag has been set.
    PRBool visibility;
    mWindow->GetVisibility(&visibility);
    if (visibility)
      mWindow->SetVisibility(PR_TRUE);
  }

  GList* users_list = nsnull;
  PRUint32 count = 0;
  gint retval = -1;

  nsILoginInfo** logins;
  count = GetLoginsForCurrentHost(&logins);

  for (PRUint32 i = 0; i < count; i++) {
    nsILoginInfo* login = logins[i];
    nsString userName;
    if(login) {
      login->GetUsername(userName);
      if (!userName.IsEmpty())
        users_list = g_list_append(users_list, ToNewUTF8String(userName));
    }
  }

  if (users_list && mProgress->IsLoginAttached()) {
    g_signal_emit(GTK_OBJECT(mOwningWidget->common),
                  moz_embed_common_signals[COMMON_SELECT_LOGIN], 0, mOwningWidget,
                  users_list,
                  &retval);
    if (retval != -1) {
      const gchar* selectedUser = (const gchar*)g_list_nth_data(users_list, retval);
      nsString loginPass;
      nsString userName;
      nsILoginInfo* login;
      for (PRUint32 i = 0; i < count; i++) {
        login = logins[i];
        login->GetUsername(userName);
        if (userName.Equals(NS_ConvertUTF8toUTF16(selectedUser))) {
          login->GetPassword(loginPass);
          break;
        }
      }
      mProgress->InsertLogin(selectedUser, NS_ConvertUTF16toUTF8(loginPass).get());
    }
    mProgress->SetLoginAttached(PR_FALSE);
    g_list_free(users_list);
  }

  NS_FREE_XPCOM_ISUPPORTS_POINTER_ARRAY(count, logins);
}

void
EmbedPrivate::ChildFocusIn(void)
{
  if (mIsDestroyed)
    return;

  nsresult rv;
  nsCOMPtr<nsIWebBrowser> webBrowser;
  rv = mWindow->GetWebBrowser(getter_AddRefs(webBrowser));
  if (NS_FAILED(rv))
    return;

  nsCOMPtr<nsIWebBrowserFocus> webBrowserFocus(do_QueryInterface(webBrowser));
  if (!webBrowserFocus)
    return;
  
  mWindow->EnableFocusEmit(PR_FALSE);
  webBrowserFocus->Activate();
  mWindow->EnableFocusEmit(PR_TRUE);
}

void
EmbedPrivate::ChildFocusOut(void)
{
  if (mIsDestroyed)
    return;

  nsresult rv;
  nsCOMPtr<nsIWebBrowser> webBrowser;
  rv = mWindow->GetWebBrowser(getter_AddRefs(webBrowser));
  if (NS_FAILED(rv))
	  return;

  nsCOMPtr<nsIWebBrowserFocus> webBrowserFocus(do_QueryInterface(webBrowser));
  if (!webBrowserFocus)
	  return;
  
  webBrowserFocus->Deactivate();
}

// Get the event listener for the chrome event handler.

void
EmbedPrivate::GetListener(void)
{
  if (mEventTarget)
    return;

  nsCOMPtr<nsPIDOMWindow> piWin;
  GetPIDOMWindow(getter_AddRefs(piWin));

  if (!piWin)
    return;

  mEventTarget = do_QueryInterface(piWin->GetChromeEventHandler());
}

// attach key and mouse event listeners

void
EmbedPrivate::AttachListeners(void)
{
  if (!mEventTarget || mListenersAttached)
    return;

  nsIDOMEventListener *eventListener =
    static_cast<nsIDOMEventListener *>(static_cast<nsIDOMKeyListener *>(mEventListener));

  // add the key listener
  nsresult rv;
  rv = mEventTarget->AddEventListenerByIID(eventListener,
					     NS_GET_IID(nsIDOMKeyListener));
  if (NS_FAILED(rv)) {
    NS_WARNING("Failed to add key listener\n");
    return;
  }

  rv = mEventTarget->AddEventListenerByIID(eventListener,
					     NS_GET_IID(nsIDOMMouseListener));
  if (NS_FAILED(rv)) {
    NS_WARNING("Failed to add mouse listener\n");
    return;
  }

  rv = mEventTarget->AddEventListenerByIID(eventListener,
                                             NS_GET_IID(nsIDOMUIListener));
  if (NS_FAILED(rv)) {
    NS_WARNING("Failed to add UI listener\n");
    return;
  }

  rv = mEventTarget->AddEventListenerByIID(eventListener,
                                             NS_GET_IID(nsIDOMContextMenuListener));
  if (NS_FAILED(rv)) {
    NS_WARNING("Failed to add context menu listener\n");
    return;
  }
  
  nsCOMPtr<nsIDOMEventTarget> target(do_QueryInterface(mEventTarget));
  if (!target) {
    return;
  }

  rv = target->AddEventListener(NS_LITERAL_STRING("focus"), eventListener, PR_TRUE);
  rv = target->AddEventListener(NS_LITERAL_STRING("blur"), eventListener, PR_TRUE);
  rv = target->AddEventListener(NS_LITERAL_STRING("DOMLinkAdded"), eventListener, PR_TRUE);
  rv = target->AddEventListener(NS_LITERAL_STRING("DOMPopupBlocked"), eventListener, PR_TRUE);
  rv = target->AddEventListener(NS_LITERAL_STRING("load"), eventListener, PR_TRUE);
  rv = target->AddEventListener(NS_LITERAL_STRING("DOMLinkXSLParsed"), eventListener, PR_TRUE);
  if (NS_FAILED(rv)) {
    NS_WARNING("Failed to add Mouse Motion listener\n");
    return;
  }
  // ok, all set.
  mListenersAttached = PR_TRUE;
}

void
EmbedPrivate::DetachListeners(void)
{
  if (!mListenersAttached || !mEventTarget)
    return;

  nsIDOMEventListener *eventListener =
    static_cast<nsIDOMEventListener *>(static_cast<nsIDOMKeyListener *>(mEventListener));

  nsresult rv;
  rv = mEventTarget->RemoveEventListenerByIID(eventListener,
						NS_GET_IID(nsIDOMKeyListener));
  if (NS_FAILED(rv)) {
    NS_WARNING("Failed to remove key listener\n");
    return;
  }

  rv =
    mEventTarget->RemoveEventListenerByIID(eventListener,
					     NS_GET_IID(nsIDOMMouseListener));
  if (NS_FAILED(rv)) {
    NS_WARNING("Failed to remove mouse listener\n");
    return;
  }

  rv = mEventTarget->RemoveEventListenerByIID(eventListener,
						NS_GET_IID(nsIDOMUIListener));
  if (NS_FAILED(rv)) {
    NS_WARNING("Failed to remove UI listener\n");
    return;
  }

  nsCOMPtr<nsIDOMEventTarget> target(do_QueryInterface(mEventTarget));
  if (!target) {
    return;
  }

  rv = target->RemoveEventListener(NS_LITERAL_STRING("focus"), eventListener, PR_TRUE);
  rv = target->RemoveEventListener(NS_LITERAL_STRING("blur"), eventListener, PR_TRUE);
  rv = target->RemoveEventListener(NS_LITERAL_STRING("DOMLinkAdded"), eventListener, PR_TRUE);
  rv = target->RemoveEventListener(NS_LITERAL_STRING("DOMPopupBlocked"), eventListener, PR_TRUE);
  rv = target->RemoveEventListener(NS_LITERAL_STRING("load"), eventListener, PR_TRUE);
  rv = target->RemoveEventListener(NS_LITERAL_STRING("DOMLinkXSLParsed"), eventListener, PR_TRUE);
  mListenersAttached = PR_FALSE;
}

nsresult
EmbedPrivate::GetFocusController(nsIFocusController * *controller)
{
  nsresult rv;
  if (!controller) {
    return NS_ERROR_FAILURE;
  }
  nsCOMPtr<nsPIDOMWindow> piWin;
  rv = GetPIDOMWindow(getter_AddRefs(piWin));
  if (!piWin || NS_FAILED(rv)) {
    return NS_ERROR_FAILURE;
  }
  nsIFocusController *focusController = piWin->GetRootFocusController();
  if (!focusController)
    return NS_ERROR_FAILURE;

  *controller = focusController;
  return NS_OK;
}

nsresult
EmbedPrivate::GetPIDOMWindow(nsPIDOMWindow **aPIWin)
{
  *aPIWin = nsnull;

  nsresult rv;
  // get the web browser
  nsCOMPtr<nsIWebBrowser> webBrowser;

  rv = mWindow->GetWebBrowser(getter_AddRefs(webBrowser));
  if (NS_FAILED(rv) || !webBrowser)
    return NS_ERROR_FAILURE;
  // get the content DOM window for that web browser
  nsCOMPtr<nsIDOMWindow> domWindow;
  rv = webBrowser->GetContentDOMWindow(getter_AddRefs(domWindow));
  if (NS_FAILED(rv) || !domWindow)
    return NS_ERROR_FAILURE;

  // get the private DOM window
  nsCOMPtr<nsPIDOMWindow> domWindowPrivate = do_QueryInterface(domWindow, &rv);
  // and the root window for that DOM window
  if (NS_FAILED(rv) || !domWindowPrivate)
    return NS_ERROR_FAILURE;
  *aPIWin = domWindowPrivate->GetPrivateRoot();

  if (*aPIWin) {
    NS_ADDREF(*aPIWin);
    return NS_OK;
  }

  return NS_ERROR_FAILURE;

}

#ifdef MOZ_ACCESSIBILITY_ATK
void *
EmbedPrivate::GetAtkObjectForCurrentDocument()
{
  if (!mNavigation)
    return nsnull;

  nsCOMPtr<nsIAccessibilityService> accService =
    do_GetService("@mozilla.org/accessibilityService;1");
  if (accService) {
    //get current document
    nsCOMPtr<nsIDOMDocument> domDoc;
    mNavigation->GetDocument(getter_AddRefs(domDoc));
    NS_ENSURE_TRUE(domDoc, nsnull);

    nsCOMPtr<nsIDOMNode> domNode(do_QueryInterface(domDoc));
    NS_ENSURE_TRUE(domNode, nsnull);

    nsCOMPtr<nsIAccessible> acc;
    accService->GetAccessibleFor(domNode, getter_AddRefs(acc));
    NS_ENSURE_TRUE(acc, nsnull);

    void *atkObj = nsnull;
    if (NS_SUCCEEDED(acc->GetNativeInterface(&atkObj)))
      return atkObj;
  }
  return nsnull;
}
#endif /* MOZ_ACCESSIBILITY_ATK */

/* static */
nsresult
EmbedPrivate::RegisterAppComponents(void)
{
  nsCOMPtr<nsIComponentRegistrar> cr;
  nsresult rv = NS_GetComponentRegistrar(getter_AddRefs(cr));
  NS_ENSURE_SUCCESS(rv, rv);

  nsCOMPtr<nsIComponentManager> cm;
  rv = NS_GetComponentManager (getter_AddRefs (cm));
  NS_ENSURE_SUCCESS (rv, rv);

  for (int i = 0; i < sNumAppComps; ++i) {
    nsCOMPtr<nsIGenericFactory> componentFactory;
    rv = NS_NewGenericFactory(getter_AddRefs(componentFactory),
                              &(sAppComps[i]));
    if (NS_FAILED(rv)) {
      NS_WARNING("Unable to create factory for component");
      continue;  // don't abort registering other components
    }

    rv = cr->RegisterFactory(sAppComps[i].mCID, sAppComps[i].mDescription,
                             sAppComps[i].mContractID, componentFactory);
    NS_ASSERTION(NS_SUCCEEDED(rv), "Unable to register factory for component");

    // Call the registration hook of the component, if any
    if (sAppComps[i].mRegisterSelfProc) {
      rv = sAppComps[i].mRegisterSelfProc(cm, nsnull, nsnull, nsnull,
                                          &(sAppComps[i]));
      NS_ASSERTION(NS_SUCCEEDED(rv), "Unable to self-register component");
    }
  }

  return rv;
}

char*
EmbedPrivate::GetEncoding()
{
  char *encoding;
  nsCOMPtr<nsIWebBrowser> webBrowser;
  mWindow->GetWebBrowser(getter_AddRefs(webBrowser));
  nsCOMPtr<nsIDocCharset> docCharset = do_GetInterface(webBrowser);
  docCharset->GetCharset(&encoding);
  return encoding;
}

nsresult
EmbedPrivate::SetEncoding(const char *encoding)
{
  nsCOMPtr<nsIWebBrowser> webBrowser;
  mWindow->GetWebBrowser(getter_AddRefs(webBrowser));
  nsCOMPtr<nsIContentViewer> contentViewer;
  GetContentViewer(webBrowser, getter_AddRefs(contentViewer));
  NS_ENSURE_TRUE (contentViewer, NS_ERROR_FAILURE);
  nsCOMPtr<nsIMarkupDocumentViewer> mDocViewer = do_QueryInterface(contentViewer);
  NS_ENSURE_TRUE (mDocViewer, NS_ERROR_FAILURE);
  return mDocViewer->SetForceCharacterSet(nsDependentCString(encoding));
}

PRBool
EmbedPrivate::FindText(const char *exp, PRBool  reverse,
                       PRBool  whole_word, PRBool  case_sensitive,
                       PRBool  restart)
{
  PRUnichar *text;
  PRBool match;
  nsresult rv;
  nsCOMPtr<nsIWebBrowser> webBrowser;
  mWindow->GetWebBrowser(getter_AddRefs(webBrowser));
  nsCOMPtr<nsIWebBrowserFind> finder(do_GetInterface(webBrowser));
  g_return_val_if_fail(finder != NULL, FALSE);
  text = ToNewUnicode(NS_ConvertUTF8toUTF16(exp));
  finder->SetSearchString(text);
  finder->SetFindBackwards(reverse);
  finder->SetWrapFind(restart); //DoWrapFind
  finder->SetEntireWord(whole_word);
  finder->SetSearchFrames(TRUE); //SearchInFrames
  finder->SetMatchCase(case_sensitive);
  rv = finder->FindNext(&match);
  NS_Free(text);
  if (NS_FAILED(rv))
    return FALSE;

  return match;
}

static nsresult
GetMarkupViewerByWindow(nsIDOMWindow *aDOMWindow,
                        nsIMarkupDocumentViewer * *aMarkupDocViewver)
{
  nsresult rv;
  NS_ENSURE_ARG_POINTER(aMarkupDocViewver);
  nsCOMPtr<nsPIDOMWindow> window(do_QueryInterface(aDOMWindow, &rv));
  NS_ENSURE_SUCCESS(rv, rv);
  nsCOMPtr<nsIDocShell> docShell;
  if (window)
    docShell = window->GetDocShell();
  NS_ENSURE_TRUE(docShell, NS_ERROR_FAILURE);
  nsCOMPtr<nsIContentViewer> contentViewer;
  rv = docShell->GetContentViewer(getter_AddRefs(contentViewer));
  NS_ENSURE_SUCCESS(rv, rv);
  nsCOMPtr<nsIMarkupDocumentViewer> markupViewer(do_QueryInterface(contentViewer, &rv));
  NS_ENSURE_TRUE(markupViewer, NS_ERROR_FAILURE);
  *aMarkupDocViewver = markupViewer;
  NS_IF_ADDREF(*aMarkupDocViewver);
  return rv;
}

nsresult
EmbedPrivate::GetZoom(PRInt32 *aZoomLevel, nsISupports *aContext)
{

  NS_ENSURE_ARG_POINTER(aZoomLevel);

  nsresult rv;
  *aZoomLevel = 100;

  nsCOMPtr<nsIDOMWindow> DOMWindow;
  if (aContext) {
    nsCOMPtr<nsIDOMNode> node = do_QueryInterface(aContext, &rv);
    NS_ENSURE_SUCCESS(rv, rv);

    rv = GetDOMWindowByNode(node, getter_AddRefs(DOMWindow));
  } else {
    nsCOMPtr<nsIWebBrowser> webBrowser;
    rv = mWindow->GetWebBrowser(getter_AddRefs(webBrowser));
    NS_ENSURE_SUCCESS(rv, rv);

    rv = webBrowser->GetContentDOMWindow(getter_AddRefs(DOMWindow));
    NS_ENSURE_SUCCESS(rv, rv);
  }

  float zoomLevelFloat = 100;
  if (DOMWindow) {
    nsCOMPtr<nsIMarkupDocumentViewer> markupViewer;
    rv = GetMarkupViewerByWindow(DOMWindow, getter_AddRefs(markupViewer));
    NS_ENSURE_SUCCESS(rv, rv);
    gboolean bval = FALSE;
    if (gtk_moz_embed_common_get_pref (G_TYPE_BOOLEAN, (gchar*)"gtkmozembed.textzoom", &bval) && bval)
      rv = markupViewer->GetTextZoom(&zoomLevelFloat);
    else
      rv = markupViewer->GetFullZoom(&zoomLevelFloat);
  }
  NS_ENSURE_SUCCESS(rv, rv);

  *aZoomLevel = (int)round(zoomLevelFloat * 100.);
  return rv;
}
nsresult
EmbedPrivate::SetZoom(PRInt32 aZoomLevel, nsISupports *aContext)
{
  nsresult rv;
  nsCOMPtr<nsIDOMWindow> DOMWindow;

  if (0 && aContext) {
    nsCOMPtr<nsIDOMNode> node = do_QueryInterface(aContext, &rv);
    NS_ENSURE_SUCCESS(rv, rv);

    rv = GetDOMWindowByNode(node, getter_AddRefs(DOMWindow));
  } else {
    nsCOMPtr<nsIWebBrowser> webBrowser;
    rv = mWindow->GetWebBrowser(getter_AddRefs(webBrowser));
    NS_ENSURE_SUCCESS(rv, rv);

    rv = webBrowser->GetContentDOMWindow(getter_AddRefs(DOMWindow));
    NS_ENSURE_SUCCESS(rv, rv);
  }

  float zoomLevelFloat;
  zoomLevelFloat = (float) aZoomLevel / 100.;

  if (DOMWindow) {
    nsCOMPtr<nsIMarkupDocumentViewer> markupViewer;
    rv = GetMarkupViewerByWindow(DOMWindow, getter_AddRefs(markupViewer));
    NS_ENSURE_SUCCESS(rv, rv);
    gboolean bval = FALSE;
    if (gtk_moz_embed_common_get_pref (G_TYPE_BOOLEAN, (gchar*)"gtkmozembed.textzoom", &bval) && bval)
      rv = markupViewer->SetTextZoom(zoomLevelFloat);
    else
      rv = markupViewer->SetFullZoom(zoomLevelFloat);
    gtk_widget_queue_draw(GTK_WIDGET(mOwningWidget));
  }
  //update
  mWindow->mBaseWindow->SetPosition(0, 0);
  
  if (aContext) {
    nsCOMPtr<nsIDOMNSHTMLElement> nodeElement = do_QueryInterface(aContext, &rv);
    if (NS_SUCCEEDED(rv) && nodeElement)
       nodeElement->ScrollIntoView(PR_TRUE);
  }
  
  return rv;
}

nsresult
EmbedPrivate::HasFrames  (PRUint32 *numberOfFrames)
{
  NS_ENSURE_ARG_POINTER(numberOfFrames);
  // setting default value.
  *numberOfFrames = 0;
  nsCOMPtr<nsIWebBrowser> webBrowser;
  nsresult rv = mWindow->GetWebBrowser(getter_AddRefs(webBrowser));
  if (NS_FAILED(rv)) return NS_ERROR_FAILURE;
  // get main dom window
  nsCOMPtr<nsIDOMWindow> DOMWindow;
  rv = webBrowser->GetContentDOMWindow(getter_AddRefs(DOMWindow));
  if (NS_FAILED(rv)) return NS_ERROR_FAILURE;
  // get frames.
  nsCOMPtr<nsIDOMWindowCollection> frameCollection;
  rv = DOMWindow->GetFrames(getter_AddRefs(frameCollection));
  if (NS_FAILED(rv)) return NS_ERROR_FAILURE;
  // comparing frames' zoom level
  rv = frameCollection->GetLength(numberOfFrames);
  return rv;
}

nsresult
EmbedPrivate::GetMIMEInfo(const char **aMime, nsIDOMNode *aDOMNode)
{
  NS_ENSURE_ARG_POINTER(aMime);
  nsresult rv;
  if (aDOMNode && mEventListener) {
    nsCOMPtr<imgIRequest> request;
    GetImageRequest(getter_AddRefs(request), aDOMNode);
    if (request)
      return request->GetMimeType((char**)aMime);
  }

  nsCOMPtr<nsIWebBrowser> webBrowser;
  rv = mWindow->GetWebBrowser(getter_AddRefs(webBrowser));

  nsCOMPtr<nsIDOMWindow> DOMWindow;
  rv = webBrowser->GetContentDOMWindow(getter_AddRefs(DOMWindow));

  nsCOMPtr<nsIDOMDocument> doc;
  rv = DOMWindow->GetDocument(getter_AddRefs(doc));

  nsCOMPtr<nsIDOMNSDocument> nsDoc = do_QueryInterface(doc);

  nsString nsmime;
  if (nsDoc)
    rv = nsDoc->GetContentType(nsmime);
  if (!NS_FAILED(rv) && !nsmime.IsEmpty())
    *aMime = g_strdup((char*)NS_LossyConvertUTF16toASCII(nsmime).get());
  return rv;
}

nsresult
EmbedPrivate::GetCacheEntry(const char *aStorage,
                             const char *aKeyName,
                             PRUint32 aAccess,
                             PRBool aIsBlocking,
                             nsICacheEntryDescriptor **aDescriptor)
{
  nsCOMPtr<nsICacheSession> session;
  nsresult rv;

  if (!sCacheService) {
    nsCOMPtr<nsICacheService> cacheService
      = do_GetService("@mozilla.org/network/cache-service;1", &rv);
    if (NS_FAILED(rv) || !cacheService) {
      NS_WARNING("do_GetService(kCacheServiceCID) failed\n");
      return rv;
    }
    NS_ADDREF(sCacheService = cacheService);
  }

  rv = sCacheService->CreateSession("HTTP", 0, PR_TRUE,
                                    getter_AddRefs(session));

  if (NS_FAILED(rv)) {
    NS_WARNING("nsCacheService::CreateSession() failed\n");
    return rv;
  }
  rv = session->OpenCacheEntry(nsCString(aKeyName),
                               nsICache::ACCESS_READ,
                               PR_FALSE,
                               aDescriptor);

  if (rv != NS_ERROR_CACHE_KEY_NOT_FOUND)
    NS_WARNING("OpenCacheEntry(ACCESS_READ) returned error for non-existent entry\n");
  return rv;
}
