/*
 * GStreamer
 * Copyright (C) 2008 Nokia Corporation <multimedia@maemo.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
/*
 * This application runs various tests and messures how long it takes.
 */

/*
 * Includes
 */
#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <gst/gst.h>
#include <gst/interfaces/xoverlay.h>
#include <gtk/gtk.h>
#include <gdk/gdkx.h>
#include <glade/glade-xml.h>
#include <string.h>

#include <sys/time.h>
#include <time.h>

#define DEFAULT_GLADE_FILE "gst-camera-perf.glade"
#define SHARED_GLADE_FILE CAMERA_APPS_GLADEDIR"/"DEFAULT_GLADE_FILE

/*
 * enums, typedefs and defines
 */


#define GET_TIME(t)                                     \
do {                                                    \
  t = gst_util_get_timestamp ();                        \
  GST_INFO("----------------------------------------"); \
} while(0)

#define DIFF_TIME(e,s,d) d=GST_CLOCK_DIFF(s,e)

#define CONT_SHOTS 10

#define TEST_CASES 9

typedef struct _ResultType
{
  GstClockTime avg;
  GstClockTime min;
  GstClockTime max;
  guint32 times;
} ResultType;

/*
 * Global vars
 */

static GladeXML *ui_glade_xml = NULL;
static GtkWidget *ui_wndMain = NULL;
static GtkWidget *ui_daMain = NULL;
static GtkLabel *ui_lbMain = NULL;

static GstElement *camera_bin = NULL;

static GString *filename = NULL;
static GString *rs_txt = NULL;
static guint32 num_pics = 0;
static guint32 num_pics_cont = 0;
//static guint32 num_vids = 0;

static GstClockTime t_initial = G_GUINT64_CONSTANT (0);
static GstClockTime t_final[CONT_SHOTS] = { G_GUINT64_CONSTANT (0), };

static GstClockTimeDiff diff;

static const GstClockTime target[TEST_CASES] = {
  1000 * GST_MSECOND,
  0,                            /* 1500 * GST_MSECOND, not tested */
  1500 * GST_MSECOND,
  2000 * GST_MSECOND,
  500 * GST_MSECOND,
  0,                            /* 2000 * GST_MSECOND, not tested */
  3500 * GST_MSECOND,
  1000 * GST_MSECOND,
  0                             /* 1000 * GST_MSECOND, not tested */
};

static const gchar *test_names[TEST_CASES] = {
  "Camera OFF to VF on",
  "(3A latency)",
  "Shot to snapshot",
  "Shot to shot",
  "Serial shooting",
  "(Shutter lag)",
  "Image saved",
  "Mode change",
  "(Video recording)"
};


static ResultType result[TEST_CASES] = { {0,}, };

static gboolean signal_sink = FALSE;
static gboolean signal_shot = FALSE;
static gboolean signal_cont = FALSE;
//static gboolean signal_save = FALSE;

/*
 * Static function declaration
 */

static gboolean ui_create (void);

static void ui_connect_signals (void);

static void
on_wndMain_delete_event (GtkWidget * widget, GdkEvent * event, gpointer data);

static void on_btnStart_clicked (GtkButton * button, gpointer user_data);

static void cleanup_pipeline (void);

static gboolean
xv_sink_has_buffer (GstPad * pad, GstBuffer * buf, gpointer user_data);

static gboolean setup_pipeline_video_sink (void);

static gboolean setup_pipeline (void);

static GstBusSyncReply
bus_sync_callback (GstBus * bus, GstMessage * message, gpointer data);

static gboolean
bus_callback (GstBus * bus, GstMessage * message, gpointer data);

static gboolean
img_capture_done (GstElement * camera, GString * fname, gpointer user_data);

static void set_next_cont_file_name (GString * filename);

/*
 * Static function implementation
 */

static gboolean
ui_create (void)
{
  gchar *gladefile = DEFAULT_GLADE_FILE;

  if (!g_file_test (gladefile, G_FILE_TEST_EXISTS)) {
    gladefile = SHARED_GLADE_FILE;
  }

  ui_glade_xml = glade_xml_new (gladefile, NULL, NULL);

  if (!ui_glade_xml) {
    fprintf (stderr, "glade_xml_new failed for %s\n", gladefile);
    fflush (stderr);
    goto done;
  }

  ui_wndMain = glade_xml_get_widget (ui_glade_xml, "wndMain");
  ui_daMain = glade_xml_get_widget (ui_glade_xml, "daMain");
  ui_lbMain = (GtkLabel *) glade_xml_get_widget (ui_glade_xml, "lbMain");

  if (!(ui_wndMain && ui_daMain && ui_lbMain)) {
    fprintf (stderr, "Some widgets couldn't be created\n");
    fflush (stderr);
    goto done;
  }

  gtk_widget_set_double_buffered (ui_daMain, FALSE);
  ui_connect_signals ();
  gtk_widget_show_all (ui_wndMain);
  return TRUE;
done:
  return FALSE;
}

static void
ui_connect_signals (void)
{
  glade_xml_signal_connect (ui_glade_xml, "on_wndMain_delete_event",
      (GCallback) on_wndMain_delete_event);

  glade_xml_signal_connect (ui_glade_xml, "on_btnStart_clicked",
      (GCallback) on_btnStart_clicked);

}

static void
on_wndMain_delete_event (GtkWidget * widget, GdkEvent * event, gpointer data)
{
  cleanup_pipeline ();
  gtk_main_quit ();
}

static void
head_result (void)
{
  g_string_append (rs_txt, "Results in:\n");
  g_string_append (rs_txt, "<span foreground=\"red\">");
  g_string_append (rs_txt, "Red means the result is bigger than target");
  g_string_append (rs_txt, "</span>\n");
  g_string_append (rs_txt, "<span foreground=\"orange\">");
  g_string_append (rs_txt,
      "Orange means the result is between 85% and 100% of the target");
  g_string_append (rs_txt, "</span>\n");
  g_string_append (rs_txt, "<span foreground=\"green\">");
  g_string_append (rs_txt,
      "Green means the result is between 70% and 85% of the target");
  g_string_append (rs_txt, "</span>\n");
  g_string_append (rs_txt, "<span foreground=\"blue\">");
  g_string_append (rs_txt, "Blue means the result is below 70% of the target");
  g_string_append (rs_txt, "</span>\n\n");
}

static void
start_result (GstClockTime result, GstClockTime target)
{
  if (result > target) {
    g_string_append (rs_txt, "<span foreground=\"red\">");
  } else if (result > (target * 0.85)) {
    g_string_append (rs_txt, "<span foreground=\"orange\">");
  } else if (result > (target * 0.70)) {
    g_string_append (rs_txt, "<span foreground=\"green\">");
  } else {
    g_string_append (rs_txt, "<span foreground=\"blue\">");
  }

  g_string_append_printf (rs_txt,
      "Result: %" GST_TIME_FORMAT
      "\nTarget: %" GST_TIME_FORMAT
      "\nRate: %.02f%% \n\n",
      GST_TIME_ARGS (result), GST_TIME_ARGS (target),
      (result * 100.0) / target);

}

static void
end_result (void)
{
  g_string_append (rs_txt, "</span>");
}

static void
print_table (void)
{
  gint i;

  puts ("");
  puts ("+---------------------------------------------------------------------------------------+");
  puts ("| test |  rate   | target  |   avg   |   min   |   max   | trials |     description     |");
  puts ("+---------------------------------------------------------------------------------------+");

  for (i = 0; i < TEST_CASES; ++i) {
    printf ("|  %02d  ", i + 1);
    if (target[i] == 0) {
      printf ("|                  test not implemented                    ");
    } else {
      printf ("| %6.02f%% ",
          100.0f * (float) result[i].max / (float) target[i]);
      printf ("|%5u ms ", (guint) GST_TIME_AS_MSECONDS (target[i]));
      printf ("|%5u ms ", (guint) GST_TIME_AS_MSECONDS (result[i].avg));
      printf ("|%5u ms ", (guint) GST_TIME_AS_MSECONDS (result[i].min));
      printf ("|%5u ms ", (guint) GST_TIME_AS_MSECONDS (result[i].max));
      printf ("|  %3d   ", result[i].times);
    }
    printf ("| %-19s |\n", test_names[i]);
  }
  puts ("+---------------------------------------------------------------------------------------+");
  puts ("");

  fflush (stdout);
}

static void
test_01 (void)
{
  g_string_append (rs_txt, "<b>01) Camera OFF to VF On</b>\n\n");

  g_string_append (rs_txt, "This only tests the time it takes to create the ");
  g_string_append (rs_txt, "pipeline and CameraBin element and have the ");
  g_string_append (rs_txt, "first video frame available in ViewFinder. ");
  g_string_append (rs_txt, "It is not testing the real init time. ");
  g_string_append (rs_txt, "To do it, the timer must start before the app.\n");

  GET_TIME (t_initial);
  setup_pipeline ();

  /* MAKE SURE THE PIPELINE IS IN PLAYING STATE BEFORE START TAKING PICTURES
     AND SO ON (otherwise it will deadlock) */
  gst_element_get_state (camera_bin, NULL, NULL, GST_CLOCK_TIME_NONE);

  GET_TIME (t_final[0]);
  DIFF_TIME (t_final[0], t_initial, diff);

  result[0].avg = result[0].min = result[0].max = diff;
  result[0].times = 1;

  start_result (diff, target[0]);

  end_result ();
}

#if 0
// FIXME: implement such test,
// how do we know when the photo has been shown in view finder?
static void
test_03 (void)
{
  g_string_append (rs_txt, "<b>03) Shot to snapshot</b>\n\n");

  g_string_append (rs_txt, "It tests the time between pressing the Shot ");
  g_string_append (rs_txt,
      "button and having the photo shown in ViewFinder.\n");

  signal_shot = TRUE;
  GET_TIME (t_initial);
  g_signal_emit_by_name (camera_bin, "user-start", 0);
  /* the second time is taken inside the callback, here we call "user-stop"
     just to go back to initial state (view-finder) again */
  g_signal_emit_by_name (camera_bin, "user-stop", 0);
  DIFF_TIME (t_final[0], t_initial, diff);

  result[2].avg = result[2].min = result[2].max = diff;
  result[2].times = 1;

  start_result (diff, target[2]);

  end_result ();
}
#endif

static void
test_04 (void)
{
  g_string_append (rs_txt, "<b>04) Shot to shot</b>\n\n");

  g_string_append (rs_txt,
      "It tests the time for being able to take a second ");
  g_string_append (rs_txt, "shot after the first one.\n");

  GET_TIME (t_initial);
  g_signal_emit_by_name (camera_bin, "user-start", 0);
  /* camerabin will be blocked in "user-start" until the previous shot is finished */
  g_signal_emit_by_name (camera_bin, "user-start", 0);
  GET_TIME (t_final[0]);
  DIFF_TIME (t_final[0], t_initial, diff);

  result[3].avg = result[3].min = result[3].max = diff;
  result[3].times = 1;

  start_result (diff, target[3]);

  end_result ();
}

static void
test_05 (void)
{
  gint i;
  GstClockTime max = 0;
  GstClockTime min = -1;
  GstClockTime total = 0;
  GstClockTime first_shot = 0;
  GstClockTime snd_shot = 0;

  g_string_append (rs_txt, "<b>05) Serial shooting</b>\n\n");

  g_string_append (rs_txt, "It tests the time between shots in continuous ");
  g_string_append (rs_txt, "mode.\n");

  signal_cont = TRUE;
  GET_TIME (t_initial);
  g_signal_emit_by_name (camera_bin, "user-start", 0);
  /* this second shot command is just a easy way to block until the previous
     shot finishes (otherwise we have to put a g_cond_wait here and a
     cond_wakeup in the last continous captured */
  g_signal_emit_by_name (camera_bin, "user-start", 0);
  /* the times are taken inside the callback, here we call "user-stop"
     just to go back to initial state (view-finder) again */
  g_signal_emit_by_name (camera_bin, "user-stop", 0);

  DIFF_TIME (t_final[0], t_initial, diff);
  max < diff ? max = diff : max;
  min > diff ? min = diff : min;
  first_shot = diff;
  total += diff;

  DIFF_TIME (t_final[1], t_final[0], diff);
  max < diff ? max = diff : max;
  min > diff ? min = diff : min;
  snd_shot = diff;
  total += diff;

  for (i = 2; i < CONT_SHOTS; ++i) {
    DIFF_TIME (t_final[i], t_final[i - 1], diff);

    max < diff ? max = diff : max;
    min > diff ? min = diff : min;
    total += diff;
  }

  result[4].avg = total / CONT_SHOTS;
  result[4].min = min;
  result[4].max = max;
  result[4].times = CONT_SHOTS;

  start_result (max, target[4]);

  g_string_append_printf (rs_txt, "First shot:     %" GST_TIME_FORMAT
      "  ->  %4.2f shots per sec\n", GST_TIME_ARGS (first_shot),
      (1.0 * GST_SECOND) / first_shot);
  g_string_append_printf (rs_txt,
      "Second shot:  %" GST_TIME_FORMAT "  ->  %4.2f shots per sec\n",
      GST_TIME_ARGS (snd_shot), (1.0 * GST_SECOND) / snd_shot);
  g_string_append_printf (rs_txt,
      "Min shot:     %" GST_TIME_FORMAT "  ->  %4.2f shots per sec\n",
      GST_TIME_ARGS (min), (1.0 * GST_SECOND) / min);
  g_string_append_printf (rs_txt,
      "Max shot:     %" GST_TIME_FORMAT "  ->  %4.2f shots per sec\n",
      GST_TIME_ARGS (max), (1.0 * GST_SECOND) / max);
  g_string_append_printf (rs_txt,
      "Avg shot:     %" GST_TIME_FORMAT "  ->  %4.2f shots per sec\n",
      GST_TIME_ARGS (total / CONT_SHOTS),
      CONT_SHOTS * (1.0 * GST_SECOND) / total);
  g_string_append_printf (rs_txt, "%u pictures have been taken\n", CONT_SHOTS);

  end_result ();

  g_string_append_printf (rs_txt, "\n");
}


static void
test_07 (void)
{
  g_string_append (rs_txt, "<b>07) Image saved</b>\n\n");

  g_string_append (rs_txt, "It tests the time between pressing the Shot ");
  g_string_append (rs_txt, "and the final image is saved to file system.\n");

  //  signal_save = TRUE;
  signal_shot = TRUE;

  GET_TIME (t_initial);
  g_signal_emit_by_name (camera_bin, "user-start", 0);
  /* call "user-stop" just to go back to initial state (view-finder) again */
  g_signal_emit_by_name (camera_bin, "user-stop", 0);
  DIFF_TIME (t_final[0], t_initial, diff);

  result[6].avg = result[6].min = result[6].max = diff;
  result[6].times = 1;

  start_result (diff, target[6]);

  end_result ();
}


static void
test_08 (void)
{
  GstClockTime total = 0;
  GstClockTime max = 0;
  GstClockTime min = -1;
  const gint count = 6;
  gint i;

  g_string_append (rs_txt, "<b>08) Mode change</b>\n\n");

  g_string_append (rs_txt, "It tests the time it takes to change between ");
  g_string_append (rs_txt, "still image and video recording mode ");
  g_string_append (rs_txt, "(In this test we change the mode few times).\n");

  for (i = 0; i < count; ++i) {
    GET_TIME (t_final[i]);
    g_object_set (camera_bin, "mode", (i + 1) & 1, NULL);
    GET_TIME (t_final[i + 1]);
  }

  for (i = 0; i < count; ++i) {
    DIFF_TIME (t_final[i + 1], t_final[i], diff);
    total += diff;
    if (diff > max)
      max = diff;
    if (diff < min)
      min = diff;
  }

  result[7].avg = total / count;
  result[7].min = min;
  result[7].max = max;
  result[7].times = count;

  start_result (max, target[7]);

  for (i = 0; i < count; ++i) {
    DIFF_TIME (t_final[i + 1], t_final[i], diff);
    if ((i + 1) & 1) {
      g_string_append_printf (rs_txt, "image->video %" GST_TIME_FORMAT "\n",
          GST_TIME_ARGS (diff));
    } else {
      g_string_append_printf (rs_txt, "video->image %" GST_TIME_FORMAT "\n",
          GST_TIME_ARGS (diff));
    }
  }
  g_string_append_printf (rs_txt, "Average: %" GST_TIME_FORMAT "\n",
      GST_TIME_ARGS (total / count));

  /* just make sure we are back to still image mode again */
  g_object_set (camera_bin, "mode", 0, NULL);

  end_result ();

  g_string_append_printf (rs_txt, "\n");
}

static void
on_btnStart_clicked (GtkButton * button, gpointer user_data)
{
  gint i;

  cleanup_pipeline ();
  g_string_assign (rs_txt, "");

  for (i = 0; i < TEST_CASES; ++i)
    memset (&result[i], 0x00, sizeof (ResultType));

  head_result ();

  test_01 ();
  //  test_03();
  test_04 ();
  test_05 ();
  test_07 ();
  test_08 ();

  gtk_label_set_markup (ui_lbMain, rs_txt->str);

  print_table ();
}

static void
cleanup_pipeline (void)
{
  if (camera_bin) {
    gst_element_set_state (camera_bin, GST_STATE_NULL);
    gst_element_get_state (camera_bin, NULL, NULL, GST_CLOCK_TIME_NONE);
    gst_object_unref (camera_bin);
    camera_bin = NULL;
  }
}

static gboolean
xv_sink_has_buffer (GstPad * pad, GstBuffer * buf, gpointer user_data)
{
  if (signal_sink) {
    signal_sink = FALSE;
    GET_TIME (t_final[0]);
  }
  return TRUE;
}

static gboolean
setup_pipeline_video_sink (void)
{
  gboolean ret = TRUE;
  GstElement *xv = NULL;
  GstPad *pad = NULL;

  xv = gst_element_factory_make ("xvimagesink", NULL);
  if (NULL == xv) {
    ret = FALSE;
    goto done;
  }

  if (NULL == (pad = gst_element_get_static_pad (xv, "sink"))) {
    ret = FALSE;
    goto done;
  }

  gst_pad_add_buffer_probe (pad, (GCallback) xv_sink_has_buffer, NULL);

  g_object_set (camera_bin, "vfsink", xv, NULL);
  xv = NULL;                    /* ownership taken by camerabin */

done:
  if (pad)
    gst_object_unref (pad);
  if (xv)
    gst_object_unref (xv);
  return ret;
}

static gboolean
setup_pipeline (void)
{
  GstBus *bus;

  cleanup_pipeline ();

  g_string_printf (filename, "test_%04u.jpg", num_pics);

  camera_bin = gst_element_factory_make ("camerabin", NULL);
  if (NULL == camera_bin) {
    goto done;
  }

  g_signal_connect (camera_bin, "img-done", (GCallback) img_capture_done, NULL);

  bus = gst_pipeline_get_bus (GST_PIPELINE (camera_bin));
  gst_bus_add_watch (bus, bus_callback, NULL);
  gst_bus_set_sync_handler (bus, bus_sync_callback, NULL);
  gst_object_unref (bus);

  if (!setup_pipeline_video_sink ()) {
    goto done;
  }

  /* set properties */
  g_object_set (camera_bin, "filename", filename->str, NULL);
#ifdef USE_MP4
  GstElement *videoenc = gst_element_factory_make ("omx_mpeg4enc", NULL);
  GstElement *videomux = gst_element_factory_make ("hantromp4mux", NULL);
  GstElement *audioenc = gst_element_factory_make ("omx_amrnbenc", NULL);
  g_object_set (camera_bin,
      "videoenc", videoenc, "videomux", videomux, "audioenc", audioenc, NULL);
#endif

  if (GST_STATE_CHANGE_FAILURE ==
      gst_element_set_state (camera_bin, GST_STATE_READY)) {
    goto done;
  }

  if (GST_STATE_CHANGE_FAILURE ==
      gst_element_set_state (camera_bin, GST_STATE_PLAYING)) {
    goto done;
  }
  return TRUE;
done:
  fprintf (stderr, "error to create pipeline\n");
  fflush (stderr);
  cleanup_pipeline ();
  return FALSE;
}

static GstBusSyncReply
set_xwindow (GstMessage ** message, gpointer data)
{
  GstBusSyncReply ret = GST_BUS_PASS;
  const GstStructure *s = gst_message_get_structure (*message);

  if (!s || !gst_structure_has_name (s, "prepare-xwindow-id")) {
    goto done;
  }

  gst_x_overlay_set_xwindow_id (GST_X_OVERLAY (GST_MESSAGE_SRC (*message)),
      GDK_WINDOW_XWINDOW (ui_daMain->window));

  gst_message_unref (*message);
  *message = NULL;
  ret = GST_BUS_DROP;
done:
  return ret;
}

static GstBusSyncReply
bus_sync_callback (GstBus * bus, GstMessage * message, gpointer data)
{
  GstBusSyncReply ret = GST_BUS_PASS;

  switch (GST_MESSAGE_TYPE (message)) {
    case GST_MESSAGE_ELEMENT:
      ret = set_xwindow (&message, data);
      break;
    default:
      /* unhandled message */
      break;
  }
  return ret;
}

static gboolean
bus_callback (GstBus * bus, GstMessage * message, gpointer data)
{
  switch (GST_MESSAGE_TYPE (message)) {
    case GST_MESSAGE_ERROR:{
      GError *err;
      gchar *debug;

      gst_message_parse_error (message, &err, &debug);
      g_print ("Error: %s\n", err->message);
      g_error_free (err);
      g_free (debug);

      gtk_main_quit ();
      break;
    }
    case GST_MESSAGE_EOS:
      /* end-of-stream */
      gtk_main_quit ();
      break;
    default:
      /* unhandled message */
      break;
  }
  return TRUE;
}

static gboolean
img_capture_done (GstElement * camera, GString * fname, gpointer user_data)
{
  gboolean ret = FALSE;

  if (signal_shot) {
    GET_TIME (t_final[num_pics_cont]);
    signal_shot = FALSE;
    return FALSE;
  }

  if (signal_cont) {
    if (num_pics_cont < CONT_SHOTS) {
      GET_TIME (t_final[num_pics_cont]);
      num_pics_cont++;
      set_next_cont_file_name (fname);
      ret = TRUE;
    } else {
      num_pics_cont = 0;
      signal_cont = FALSE;
    }
  }
  return ret;
}

static void
set_next_cont_file_name (GString * filename)
{
  /* FIXME: better file naming (possible with signal) */
  if (G_UNLIKELY (num_pics_cont == 1)) {
    gint i;
    for (i = filename->len - 1; i > 0; --i) {
      if (filename->str[i] == '.')
        break;
    }
    g_string_insert (filename, i, "_0001");
  } else {
    gchar tmp[6];
    gint i;
    for (i = filename->len - 1; i > 0; --i) {
      if (filename->str[i] == '_')
        break;
    }
    snprintf (tmp, 6, "_%04d", num_pics_cont);
    memcpy (filename->str + i, tmp, 5);
  }
}

int
main (int argc, char *argv[])
{
  int ret = 0;

  gst_init (&argc, &argv);
  gtk_init (&argc, &argv);

  rs_txt = g_string_new_len ("", 2048);
  filename = g_string_new_len ("", 16);

  /* create UI */
  if (!ui_create ()) {
    ret = -1;
    goto done;
  }

  gtk_main ();

done:
  cleanup_pipeline ();

  g_string_free (filename, TRUE);
  g_string_free (rs_txt, TRUE);
  return ret;
}
