/* GStreamer
 *
 * Copyright (C) 2001-2002 Ronald Bultje <rbultje@ronald.bitfreak.net>
 *               2006 Edgard Lima <edgard.lima@indt.org.br>
 *               2008 Nokia Corporation <multimedia@maemo.org>
 *
 * gstv4l2camsrc.h: BT8x8/V4L2 source element
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifndef __GST_V4L2CAMSRC_H__
#define __GST_V4L2CAMSRC_H__

/* Because of some really cool feature in video4linux1, also known as
 * 'not including sys/types.h and sys/time.h', we had to include it
 * ourselves. In all their intelligence, these people decided to fix
 * this in the next version (video4linux2) in such a cool way that it
 * breaks all compilations of old stuff...
 * The real problem is actually that linux/time.h doesn't use proper
 * macro checks before defining types like struct timeval. The proper
 * fix here is to either fuck the kernel header (which is what we do
 * by defining _LINUX_TIME_H, an innocent little hack) or by fixing it
 * upstream, which I'll consider doing later on. If you get compiler
 * errors here, check your linux/time.h && sys/time.h header setup.
 */
#include <sys/types.h>
#ifndef __sun
#include <linux/types.h>
#define _LINUX_TIME_H
#define __user
#include <linux/videodev2.h>
#else
#include <sys/videodev2.h>
#endif

#include <gst/gst.h>
#include <gst/base/gstpushsrc.h>
#include <gst/interfaces/photography.h>

#include "gstv4l2camdriver.h"

GST_DEBUG_CATEGORY_EXTERN (v4l2camsrc_debug);

#define GST_V4L2_MAX_BUFFERS 16
#define GST_V4L2_MIN_BUFFERS 1
#define GST_V4L2_DEFAULT_BUFFERS 4
#define GST_V4L2_MAX_SIZE (1<<15) /* 2^15 == 32768 */

G_BEGIN_DECLS

#define GST_TYPE_V4L2CAMSRC \
  (gst_v4l2camsrc_get_type())
#define GST_V4L2CAMSRC(obj) \
  (G_TYPE_CHECK_INSTANCE_CAST((obj),GST_TYPE_V4L2CAMSRC,GstV4l2CamSrc))
#define GST_V4L2CAMSRC_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_CAST((klass),GST_TYPE_V4L2CAMSRC,GstV4l2CamSrcClass))
#define GST_IS_V4L2CAMSRC(obj) \
  (G_TYPE_CHECK_INSTANCE_TYPE((obj),GST_TYPE_V4L2CAMSRC))
#define GST_IS_V4L2CAMSRC_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_TYPE((klass),GST_TYPE_V4L2CAMSRC))

#define GST_TYPE_V4L2CAMSRC_CAPTURE_MODE (gst_v4l2camsrc_capture_mode_get_type())

typedef enum {
  GST_V4L2CAMSRC_CAPTURE_MODE_VIEWFINDER,
  GST_V4L2CAMSRC_CAPTURE_MODE_STILL,
  GST_V4L2CAMSRC_CAPTURE_MODE_VIDEO
} GstV4L2CamSrcCaptureMode;

/**
 * GST_PAD_GET_STREAM_LOCK:
 * @camsrc: a #GstV4l2CamSrc
 *
 * Get the capture state lock of @camsrc. The capture state lock is protecting
 * the internal photo capturing state of @camsrc.
 */
#define GST_V4L2CAM_GET_STATE_LOCK(camsrc) (GST_V4L2CAMSRC(camsrc)->state_rec_lock)
/**
 * GST_V4L2CAM_STATE_LOCK:
 * @camsrc: a #GstV4l2CamSrc
 *
 * Lock the capture state lock of @camsrc.
 */
#define GST_V4L2CAM_STATE_LOCK(camsrc) (g_static_rec_mutex_lock(GST_V4L2CAM_GET_STATE_LOCK(camsrc)))
/**
 * GST_V4L2CAM_STATE_UNLOCK:
 * @camsrc: a #GstV4l2CamSrc
 *
 * Unlock the capture state lock of @camsrc.
 */
#define GST_V4L2CAM_STATE_UNLOCK(camsrc) (g_static_rec_mutex_unlock(GST_V4L2CAM_GET_STATE_LOCK(camsrc)))

typedef struct _GstV4l2BufferPool GstV4l2BufferPool;
typedef struct _GstV4l2CamSrc GstV4l2CamSrc;
typedef struct _GstV4l2CamSrcClass GstV4l2CamSrcClass;


/* global info */
struct _GstV4l2BufferPool
{
  GstMiniObject parent;

  GMutex *lock;
  gboolean running; /* with lock */
  gint num_live_buffers; /* with lock */
  gint video_fd; /* a dup(2) of the v4l2object's video_fd */
  guint buffer_count;
  GstBuffer **buffers; /* with lock; buffers[n] is NULL that buffer has been
                            * dequeued and pushed out */
  struct v4l2_buffer *vbuffers;
  guint *queued;
};

/**
 * GstV4l2CamSrc:
 * @pushsrc: parent #GstPushSrc.
 *
 * Opaque object.
 */
struct _GstV4l2CamSrc
{
  GstPushSrc pushsrc;

  /* the video device */
  char *videodev;

  /* the video-device's file descriptor */
  gint video_fd;

  GstPoll * poll;

  /* the video buffer (mmap()'ed) */
  guint8 **buffer;

  /* the video device's capabilities */
  struct v4l2_capability vcap;

  /* cropping & scaling capabilities. Available if crop_supported == TRUE */
  struct v4l2_cropcap vcrop;
  gboolean crop_supported;
  gfloat max_zoom_factor;

  /* lists... */
  GList *colors;

  GstCaps *probed_caps;
  GstCaps *vf_caps;

  /* internal lists */
  GSList *formats;              /* list of available capture formats */

  /* buffers */
  GstV4l2BufferPool *pool;

  guint32 num_buffers;
  guint32 frame_byte_size;

  /* if the buffer will be or not used from directly mmap */
  gboolean always_copy;

  /* True if we want to stop */
  gboolean quit;
  gboolean is_capturing;

  guint64 offset;

  gint     fps_d, fps_n;          /* framerate if device is open */
  gint    current_w, current_h;   /* current capture frame size */

  /* Photo interface -related */
  GstPhotoSettings  photoconf;
  GstPhotoShakeRisk shake_risk;
  GStaticRecMutex   *state_rec_lock;
  guint             photo_capture_phase;
  gint              capture_w;
  gint              capture_h;
  gint              capture_fps_n;
  gint              capture_fps_d;

  GstPhotoCapturePrepared prep_func;
  gpointer                prep_udata;

  GstV4L2CamSrcCaptureMode capture_mode;

  /* the video device driver */
  char *driver_name;

  GstV4l2CamDriver *driver;
};


/**
 * GstV4l2CamSrcClass:
 * @parent_class: the parent class structure.
 *
 * GstCamSrc class object.
 */
struct _GstV4l2CamSrcClass
{
  GstPushSrcClass parent_class;
};


GType gst_v4l2camsrc_get_type (void);

G_END_DECLS

#endif /* __GST_V4L2CAMSRC_H__ */
