/* GStreamer
 *
 * Copyright (C) 2008 Nokia Corporation <multimedia@maemo.org>
 *
 * driver_wrapper.c: Abstraction layer for camera driver implementations
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <gst/gst.h>
#include <gst/interfaces/photography.h>

#include "gstv4l2camsrc.h"

#define CAM_DRIVER_MODULE_PATH "/usr/lib/camdriver"

GST_DEBUG_CATEGORY_EXTERN (v4l2camsrc_debug);
#define GST_CAT_DEFAULT v4l2camsrc_debug


/******************************************************************************
 * Private functions
 *
 ******************************************************************************/

/* Default dummy implementations */

static GstPhotoCaps
wrapper_default_capabilities (GstV4l2CamDriver *driver)
{
  return GST_PHOTOGRAPHY_CAPS_ZOOM;
}

static gboolean
wrapper_default_ret_true (GstV4l2CamDriver *driver)
{
  return TRUE;
}

static gboolean
wrapper_default_ret_false (GstV4l2CamDriver *driver)
{
  return FALSE;
}

static gboolean
wrapper_default_ret_true_with_settings (GstV4l2CamDriver *driver,
                                        GstPhotoSettings *photoconf)
{
  return TRUE;
}

static gboolean
wrapper_default_write_settings (GstV4l2CamDriver *driver,
                                GstPhotoSettings *photoconf,
                                gboolean scene_mode_override)
{
  return TRUE;
}

static gboolean
wrapper_default_set_onoff (GstV4l2CamDriver *driver, gboolean on_off)
{
  return TRUE;
}

static gboolean
wrapper_default_set_float_false (GstV4l2CamDriver *driver, gfloat value)
{
  return FALSE;
}

static gboolean
wrapper_default_get_guint32 (GstV4l2CamDriver *driver, guint32 *value)
{
  *value = 0;
  return TRUE;
}

static gboolean
wrapper_default_init (GstV4l2CamDriver *driver, gint fd)
{
  return TRUE;
}

static gboolean
wrapper_default_set_guint32 (GstV4l2CamDriver *driver, guint32 value)
{
  return TRUE;
}

static GstFocusStatus
wrapper_default_get_focus_status (GstV4l2CamDriver *driver,
                                  guint32 *focused_windows_bitmask,
                                  guint8 *focus_rows,
                                  guint8 *focus_columns)
{
  *focused_windows_bitmask = *focus_rows = *focus_columns = 0;
  return GST_PHOTOGRAPHY_FOCUS_STATUS_NONE;
}

static gboolean
wrapper_default_get_shake_risk (GstV4l2CamDriver *driver,
                                GstPhotoShakeRisk *risk)
{
  *risk = GST_PHOTOGRAPHY_SHAKE_RISK_LOW;
  return FALSE;
}

static gboolean
wrapper_default_process (GstV4l2CamDriver *driver, GstBuffer **buf,
                         guint image_w, guint image_h)
{
  GST_DEBUG ("DUMMY_DRIVER: Process wrapper");
  return TRUE;
}

static GstCaps *
wrapper_default_get_caps (GstV4l2CamDriver *driver)
{
  GST_DEBUG ("DUMMY_DRIVER: Returning NULL caps");

  return NULL;
}

static gboolean
wrapper_default_set_capture_fmt (GstV4l2CamDriver *driver,
                                 guint32 output_fourcc, guint output_outsize,
                                 guint32 *internal_fourcc,
                                 guint *internal_outsize)
{
  GST_DEBUG ("DUMMY_DRIVER: Returning default (output) capture format");

  *internal_fourcc = output_fourcc;
  *internal_outsize = output_outsize;

  return TRUE;
}

static gboolean
wrapper_default_set_capture_res (GstV4l2CamDriver *driver,
                                 guint width, guint height)
{
  GST_DEBUG ("DUMMY_DRIVER: Setting resolution: %d x %d", width, height);

  return TRUE;
}

static gboolean
wrapper_default_get_makernote (GstV4l2CamDriver *driver,
                               guint8 **maker_note,
                               guint *maker_note_size)
{
  GST_DEBUG ("DUMMY_DRIVER: Makernote wrapper");
  *maker_note = NULL;
  *maker_note_size = 0;

  return TRUE;
}

static gboolean
wrapper_default_start_capture (GstV4l2CamDriver *driver,
                               gboolean *need_init)
{
  GST_DEBUG ("DUMMY_DRIVER: start capture");
  *need_init = TRUE;

  return TRUE;
}

static gboolean
wrapper_default_get_postproc_params (GstV4l2CamDriver *driver,
                                     GstStructure **pp_struct)
{
  GST_DEBUG ("DUMMY_DRIVER: get postproc params");
  *pp_struct = NULL;

  return FALSE;
}

static gboolean
wrapper_default_get_image_params (GstV4l2CamDriver *driver,
                                  GstV4l2CapturedImageParams **params_struct)
{
  GST_DEBUG ("DUMMY_DRIVER: get captured image params");
  *params_struct = NULL;

  return FALSE;
}

static gboolean
wrapper_default_set_capture_mode (GstV4l2CamDriver *driver,
                                  GstCamCaptureMode mode)
{
  return TRUE;
}


static void
wrapper_default_functions_init (GstV4l2CamDriver *driver)
{
  driver->init                = wrapper_default_init;
  driver->update_fd           = wrapper_default_init;
  driver->deinit              = wrapper_default_ret_true;
  driver->get_capabilities    = wrapper_default_capabilities;
  driver->get_raw_caps        = wrapper_default_get_caps;
  driver->get_capture_caps    = wrapper_default_get_caps;
  driver->set_capture_fmt     = wrapper_default_set_capture_fmt;
  driver->set_capture_res     = wrapper_default_set_capture_res;
  driver->set_capture_mode    = wrapper_default_set_capture_mode;
  driver->start               = wrapper_default_ret_true;
  driver->stop                = wrapper_default_ret_true;
  driver->set_autofocus       = wrapper_default_set_onoff;
  driver->set_privacy_light   = wrapper_default_set_onoff;
  driver->set_flash           = wrapper_default_set_onoff;
  driver->get_flash_intensity = wrapper_default_get_guint32;
  driver->set_flash_intensity = wrapper_default_set_guint32;
  driver->set_flash_duration  = wrapper_default_set_guint32;
  driver->get_focus_status    = wrapper_default_get_focus_status;
  driver->get_shake_risk      = wrapper_default_get_shake_risk;
  driver->process_excl_access = wrapper_default_ret_false;
  driver->process             = wrapper_default_process;
  driver->start_capture       = wrapper_default_start_capture;
  driver->stop_capture        = wrapper_default_ret_true;
  driver->read_settings       = wrapper_default_ret_true_with_settings;
  driver->write_settings      = wrapper_default_write_settings;
  driver->get_makernote       = wrapper_default_get_makernote;
  driver->get_postproc_params = wrapper_default_get_postproc_params;
  driver->get_hq_image_params = wrapper_default_get_image_params;
  driver->set_zoom            = wrapper_default_set_float_false;

  GST_DEBUG ("Default functions set");
}

/**
 * gst_v4l2camsrc_driver_wrapper_setup:
 * @v4l2camsrc: #GstV4l2CamSrc object
 *
 * Initialize the driver module by setting up necessary parameters.
 *
 */
void
gst_v4l2camsrc_driver_wrapper_setup (GstV4l2CamSrc *v4l2camsrc,
                                     GstV4l2CamDriver *driver)
{
  driver->set_capture_mode (driver, v4l2camsrc->capture_mode);
  driver->set_zoom (driver, v4l2camsrc->photoconf.zoom);
}

/******************************************************************************
 * Public functions
 *
 ******************************************************************************/

/**
 * gst_v4l2camsrc_driver_wrapper_load:
 * @v4l2camsrc: #GstV4l2CamSrc object
 *
 * Create a camera driver. If "driver-name" property was set, try to load
 * the driver module dynamically and initialize it.
 *
 * Returns: a new #GstV4l2CamDriver instance.
 */
GstV4l2CamDriver *
gst_v4l2camsrc_driver_wrapper_load (GstV4l2CamSrc *v4l2camsrc)
{
  gboolean (*init_function)(GstV4l2CamDriver *driver);
  GstV4l2CamDriver *driver;

  GST_DEBUG_OBJECT (v4l2camsrc, "Creating a driver object");

  driver = g_new0 (GstV4l2CamDriver, 1);
  driver->device_mutex = g_mutex_new ();

  /* Init the driver to default dummy functions first */
  wrapper_default_functions_init (driver);

  GST_DEBUG_OBJECT (v4l2camsrc, "Driver name: %s", v4l2camsrc->driver_name);

  /* Initialize driver with default functions first */
  wrapper_default_functions_init (driver);

  if (g_module_supported () && v4l2camsrc->driver_name) {
    gchar *module_file;

    module_file = g_module_build_path (CAM_DRIVER_MODULE_PATH,
                                       v4l2camsrc->driver_name);

    GST_DEBUG_OBJECT (v4l2camsrc, "DRIVER: Module name: %s", module_file);

    driver->module = g_module_open (module_file, G_MODULE_BIND_LOCAL);
    if (driver->module) {
      gboolean ret;

      ret = g_module_symbol (driver->module, "cam_driver_init",
                             (gpointer) &init_function);
      if (ret) {
        if (init_function (driver) == FALSE) {
          GST_WARNING_OBJECT (v4l2camsrc,
                              "DRIVER: Driver failed to initialize");

          g_module_close (driver->module);
          driver->module = NULL;
        }
      }
      else {
        GST_WARNING_OBJECT (v4l2camsrc, "DRIVER: Init symbol not found");
        g_module_close (driver->module);
        driver->module = NULL;
      }
    }
    else {
      GST_DEBUG_OBJECT (v4l2camsrc, "DRIVER: Opening the module failed: %s",
                      g_module_error());
    }
    g_free (module_file);

    GST_DEBUG_OBJECT (v4l2camsrc, "DRIVER: Camera driver initialized");
  }

  /* Set up the driver state */
  gst_v4l2camsrc_driver_wrapper_setup (v4l2camsrc, driver);

  return driver;
}


/**
 * gst_v4l2camsrc_driver_wrapper_unload:
 * @driver: #GstV4l2CamDriver instance to be unloaded.
 *
 * Unload the camera driver and free its resources.
 */
void
gst_v4l2camsrc_driver_wrapper_unload (GstV4l2CamDriver *driver)
{
  if (driver->module) {
    void (*deinit_function)(GstV4l2CamDriver *driver);
    gboolean ret;

    GST_DEBUG ("Unloading driver");

    ret = g_module_symbol (driver->module, "cam_driver_deinit",
                           (gpointer) &deinit_function);
    if (ret) {
      deinit_function (driver);
    }
    g_module_close (driver->module);
    driver->module = NULL;
  }

  if (driver->device_mutex) {
    g_mutex_free (driver->device_mutex);
  }
  g_free (driver);
}

