/*
 * This file is part of Functracer.
 * 
 * Copyright (C) 2008 by Nokia Corporation
 * 
 * Contact: Eero Tamminen <eero.tamminen@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 */

#include <assert.h>
#include <libiberty.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <errno.h>
#include <time.h>

#include "arch-defs.h"
#include "backtrace.h"
#include "config.h"
#include "debug.h"
#include "report.h"
#include "options.h"

#define FNAME_FMT "%s/allocs-%d.%d.trace"

void rp_write_backtraces(struct process *proc)
{
	int bt_depth, j;
	char *backtrace[MAX_BT_DEPTH];
	struct rp_data *rd = proc->rp_data;

	bt_depth = bt_backtrace(proc->bt_data, backtrace, arguments.depth);

	debug(3, "rp_write_backtraces(pid=%d)", rd->pid);
	
	for (j = 0; j < bt_depth; j++) {
		fprintf(rd->fp, "   %s\n", backtrace[j]);
		free(backtrace[j]);
	}
}

void rp_event(struct process *proc, const char *fmt, ...)
{
	struct rp_data *rd = proc->rp_data;
	va_list args;

	assert(rd != NULL);
	va_start(args, fmt);
	vfprintf(rd->fp, fmt, args);
	va_end(args);
}

static char *rp_date(void)
{
	time_t t;
	struct tm *tm;
	static char buf[11];

	time(&t);
	tm = localtime(&t);
	snprintf(buf, sizeof(buf), "%04d-%02d-%02d", 1900 + tm->tm_year,
		 tm->tm_mon + 1, tm->tm_mday);

	return buf;
}

static char *rp_timestamp(void)
{
	struct timeval tv;
	struct tm *tm;
	static char buf[16];

	gettimeofday(&tv, NULL);
	tm = localtime(&tv.tv_sec);
	snprintf(buf, sizeof(buf), "%02d:%02d:%02d.%06d", tm->tm_hour,
		 tm->tm_min, tm->tm_sec, (int)tv.tv_usec);

	return buf;
}

static int rp_write_header(struct process *proc)
{
	struct rp_data *rd = proc->rp_data;
	char path[256], *buf;

	if (arguments.save_to_file) {
		struct stat buf;
		/* Do not overwrite existing trace file so it will use a
		 * non-existing path.
		 */
		do {
			snprintf(path, sizeof(path), FNAME_FMT,
				 arguments.path ? : getenv("HOME"), rd->pid,
				 rd->step++);
		} while (stat(path, &buf) == 0);
		rd->step--;

		rd->fp = fopen(path, "a");
		if (rd->fp == NULL) {
			error_exit("rp_init(): fopen");
			bt_finish(proc->bt_data);
			free(rd);
			proc->rp_data = rd = NULL;											
			return -1;
		}
	} else
		rd->fp = stdout;
	if (arguments.resolve_name)
		fprintf(rd->fp, "generated by %s, with resolved names, %s, %s"
			" %s\n", PACKAGE_STRING, BUILD_ARCH, rp_date(),
			rp_timestamp());
	else
		fprintf(rd->fp, "generated by %s, %s, %s %s\n", PACKAGE_STRING,
			BUILD_ARCH, rp_date(), rp_timestamp());
	buf = cmd_from_pid(proc->pid, 1);
	fprintf(rd->fp, "PID %d, %s\n", proc->pid, buf);
	free(buf);

	return 0;
}

void rp_alloc(struct process *proc, int rp_number, const char *name,
	      size_t arg0, size_t arg1)
{
	struct rp_data *rd = proc->rp_data;

	assert(rd != NULL);
	if (arguments.time)
		rp_event(proc, "%d. [%s] %s(%d) = 0x%08x\n", rp_number,
			 rp_timestamp(), name, arg0, arg1);
	else
		rp_event(proc, "%d. %s(%d) = 0x%08x\n", rp_number, name, arg0, arg1);
}

void rp_alloc_details(struct process *proc, int rp_number, const char *name,
				size_t arg0, size_t arg1, char *details)
{
	struct rp_data *rd = proc->rp_data;

	assert(rd != NULL);
	if (arguments.time)
		rp_event(proc, "%d. [%s] %s(%d) = 0x%08x (%s)\n", rp_number,
			 rp_timestamp(), name, arg0, arg1, details);
	else
		rp_event(proc, "%d. %s(%d) = 0x%08x (%s)\n", rp_number, name, arg0, arg1, details);
}

void rp_free(struct process *proc, int rp_number, const char *name, size_t arg)
{
	struct rp_data *rd = proc->rp_data;

	assert(rd != NULL);
	if (arguments.time)
		rp_event(proc, "%d. [%s] %s(0x%08x)\n", rp_number,
			 rp_timestamp(), name, arg);
	else
		rp_event(proc, "%d. %s(0x%08x)\n", rp_number, name, arg);
}

int rp_init(struct process *proc)
{
	struct rp_data *rd;

	debug(3, "pid=%d", proc->pid);

	if (proc->parent != NULL)
		rd = proc->parent->rp_data;
	else
		rd = proc->rp_data;
	if (rd == NULL) {
		rd = xcalloc(1, sizeof(struct rp_data));
		if (proc->parent != NULL)
			rd->pid = proc->parent->pid;
		else
			rd->pid = proc->pid;
	}
	if (proc->parent != NULL)
		proc->parent->rp_data = rd;
	proc->rp_data = rd;
	if (rd->refcnt++ == 0) {
		int ret = rp_write_header(proc);
		if (ret < 0)
			return ret;
	}
	proc->bt_data = bt_init(proc->pid);
	if (arguments.verbose)
		fprintf(stderr, "Started tracing %d\n", proc->pid);
	return 0;
}

void rp_finish(struct process *proc)
{
	struct rp_data *rd = proc->rp_data;

	assert(rd != NULL);
	bt_finish(proc->bt_data);
	assert(rd->refcnt > 0);
	if (--rd->refcnt == 0) {
		rd->step++;
		if (arguments.save_to_file)
			fclose(rd->fp);
	}
	if (arguments.verbose) {
		char fname[256];
		if (arguments.save_to_file)
			snprintf(fname, sizeof(fname), FNAME_FMT,
				 arguments.path ? : getenv("HOME"), rd->pid,
				 rd->step - 1);
		else
			snprintf(fname, sizeof(fname), "stdout");
		fprintf(stderr, "Stopped tracing %d, trace saved to "
			"%s\n", proc->pid, fname);
	}
}
