/*
 * This file is part of functracer-postproc.
 *
 * Copyright (C) 2008 by Nokia Corporation
 *
 * Contact: Eero Tamminen <eero.tamminen@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <config.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "list.h"
#include "maps.h"
#include "trace_file.h"

static int parse_header_version(const char *version, int *maj, int *min,
	int *rev)
{
	int ret;

	ret = sscanf(version, "%d.%d.%d", maj, min, rev);
	if (ret != 3) {
		ret = sscanf(version, "%d.%d", maj, min);
		if (ret != 2) {
			fprintf(stderr, "invalid header version format\n");
			return -EINVAL;
		}
		*rev = 0;
	}

	return 0;
}

static int check_header_version(const char *version)
{
	int ret, maj1, min1, rev1, maj2, min2, rev2;

	ret = parse_header_version(version, &maj1, &min1, &rev1);
	if (ret < 0)
		return ret;
	ret = parse_header_version(FT_VERSION, &maj2, &min2, &rev2);
	if (ret < 0)
		return ret;

	if (maj1 < maj2 || min1 < min2 || rev1 < rev2) {
		fprintf(stderr, "error: trace header version (%s) not "
			"supported\n", version);
		return -EINVAL;
	}

	return 0;
}

/* Reads header information from .trace file */
static int read_header_info (t_trace_file *trace_data, char line[LINE_MAX])
{
	t_header *header = &trace_data->header;
	int ret;

	header->is_resolved = 0;

	ret = sscanf(line, "generated by functracer %10[^,], with resolved names, %[^,]",
		header->version, header->arch);
	if (header->version[sizeof(header->version) - 1] != '\0') {
		fprintf(stderr, "error: not enough space to hold version string\n");
		return -EINVAL;
	} else if (ret != 2) {
		ret = sscanf(line, "generated by functracer %10[^,], %[^,]",
			header->version, header->arch);
		if (header->version[sizeof(header->version) - 1] != '\0') {
			fprintf(stderr, "error: not enough space to hold version string\n");
			return -EINVAL;
		} else if (ret != 2) {
			fprintf(stderr, "error: invalid header format\n");
			return -EINVAL;
		}
	} else header->is_resolved = 1;

	if (strcmp(header->arch, BUILD_ARCH)) {
		fprintf(stderr, "error: trace architecture (%s) not supported\n",
			header->arch);
		return -EINVAL;
	}

	ret = check_header_version(header->version);
	if (ret < 0)
		return ret;

	return 0;
}

void print_header_file(t_trace_file *trace_data)
{
	FILE *trace_file = trace_data->trace_file;
	char line[LINE_MAX];

	/* Print header file */
	if (fseek(trace_file, 0L, SEEK_SET) < 0) {
		fprintf(stderr, "warning: could not set the file position \n");
	} else if (fgets(line, LINE_MAX, trace_file) != NULL) {
		printf("%s\n", line);
	}
}

t_address read_backtrace_lines(t_trace_file *trace_data, long offset,
			       char *line, size_t len)
{
	static t_address address;
	t_header header = trace_data->header;
	FILE *trace_file = trace_data->trace_file;
	int ret;

	if ((ret = fseek(trace_file, offset, SEEK_SET) < 0)) {
		fprintf(stderr, "error: could not set the file position indicator\n");
		ret = -EINVAL;
		goto error;
	}

	if (fgets(line, len, trace_file) == NULL) {
		ret = 0;
		goto error;
	}

	if (line[strlen(line)-1] == '\n')
		line[strlen(line)-1] = '\0';

	if (header.is_resolved) {
		ret = sscanf(line, "   0x%08lx: %*s", &address);
		if (ret != 1)
			goto error;
	} else {
		ret = sscanf(line, "   0x%08lx", &address);
		if (ret != 1)
			goto error;
	}

	ret = address;
error:
	return ret;
}

static int fill_alloc_info(char *line, t_line *line_info)
{
	int ret;
	
	/* capture size and adress from line */
	ret = sscanf(line, "%*d. %*[^(](%d) = %lx",  &line_info->size,
		     &line_info->address);
	if (ret != 2) {
		ret = sscanf(line, "%*d. %*[^(](%lx)", &line_info->address);
		if (ret != 1) {
			fprintf(stderr, "error: invalid trace line format: \
				%s\n", line);
			return -EINVAL;
		} else
			line_info->type = TYPE_FREE;
	} else
		line_info->type = TYPE_ALLOC;
	strcpy(line_info->header, line);
	return 0;
}

/* Read .trace file and store entries on trace (t_list) list */
int read_trace_file (t_trace_file *trace_data)
{
	char line[LINE_MAX], pathname[LINE_MAX];
	static t_line *line_info;
	t_list *list = &trace_data->list;
	t_maplist *maplist = &trace_data->maplist;
	FILE *trace_file = trace_data->trace_file;
	t_address start_addr, end_addr;
	int ret;

	line_info = (t_line *) malloc(sizeof(t_line));
	if (!line_info) {
		perror("malloc");
		return -ENOMEM;
	}

	/* read header info */
	fgets(line, LINE_MAX, trace_file);
	if (line[strlen(line)-1] == '\n')
		line[strlen(line)-1] = '\0';

	if ((ret = read_header_info(trace_data, line)))
		return ret;

	/* read traces (incl. backtraces) */
	while (fgets(line, LINE_MAX, trace_file)) {
		int discard;

		if (line[strlen(line)-1] == '\n')
			line[strlen(line)-1] = '\0';

		if (sscanf(line, "%d. %*[^(](%d)", &discard, &discard) == 2) {
			/* reset line_info structure */
			memset(line_info, 0, sizeof(t_line));

			/* get line offset from .trace file */
			line_info->offset = ftell(trace_file);

			if ((ret = fill_alloc_info(line, line_info)) < 0)
				return ret;

			if ((ret = list_add(list, line_info)))
				return ret;
		} else if ((ret = sscanf(line, "%s => %lx-%lx", pathname,
			   &start_addr, &end_addr)) == 3) {
			if ((ret = maplist_fill(maplist, start_addr, end_addr,
			    pathname)))
				return ret;
		}
	}

	free(line_info);

	return 0;
}
