/*
 * This file is part of functracer-postproc.
 *
 * Copyright (C) 2008 by Nokia Corporation
 *
 * Contact: Eero Tamminen <eero.tamminen@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "list.h"
#include "maps.h"
#include "options.h"
#include "resolve.h"

/* Initialize map list */
int maplist_init (t_maplist *maplist)
{
	maplist->first = (t_map *) malloc(sizeof(t_map));
	if (!maplist->first) {
		perror("malloc");
		return -ENOMEM;
	}

	maplist->last = maplist->first;
	maplist->first->next = NULL;

	return 0;
}

/* Verify if the map list is empty */
int maplist_is_empty (t_maplist *maplist)
{
	return (maplist->first == maplist->last);
}

/* Free map list memory allocations */
int maplist_free (t_maplist *maplist)
{
	t_map *map_current, *map_aux;

	map_current = maplist->first->next; /* skip sentinel */

	free(maplist->first); /* free sentinel */

	while (map_current)
	{
		map_aux = map_current;
		map_current = map_current->next;

		free(map_aux);
	}

	return 0;
}

/* Verify if the map entry already exists */
static t_map *maplist_search (t_maplist *maplist, char *pathname,
	t_address addr_init)
{
	t_map *map_current;

	map_current = maplist->first->next; /* skip sentinel */

	while (map_current && strcmp(pathname, map_current->pathname) != 0 &&
	       map_current->dyn_addr_init != addr_init)
		map_current = map_current->next;

	return map_current;
}

/* Fill out a map member */
static int maplist_fill_item (t_map *map_aux,
        t_address dyn_addr_init,
        t_address dyn_addr_end,
        char pathname[LINE_MAX])
{
	int ret;

	if (map_aux == NULL)
		return -ENOMEM;

	map_aux->dyn_addr_init = dyn_addr_init;
	map_aux->dyn_addr_end = dyn_addr_end;
	strcpy(map_aux->pathname, pathname);

	ret = is_absolute(map_aux->pathname);
	if (ret == 0 || ret == 1)
		map_aux->is_absolute = ret;
	else return ret;

	return 0;	
}

/* Alloc a new member to map list */
static t_map *maplist_add (t_maplist *maplist)
{
	maplist->last->next = (t_map *) malloc(sizeof(t_map));
	if (!maplist->last->next) {
		perror("malloc");
		return NULL;
	}

	maplist->last = maplist->last->next;
	maplist->last->next = NULL;

	return maplist->last;

}

/* Add a new member to map list or update an existent one */
int maplist_fill (t_maplist *maplist,
        t_address dyn_addr_init,
        t_address dyn_addr_end,
        char pathname[LINE_MAX])
{
	int ret;
	t_map *map_aux;

	map_aux = maplist_search(maplist, pathname, dyn_addr_init);
	if (map_aux == NULL)
		map_aux = maplist_add(maplist);

	ret = maplist_fill_item(map_aux, dyn_addr_init, dyn_addr_end, pathname);

	return ret;
}

/* Print the current maps list */
void maplist_print (t_maplist *maplist)
{
	t_map *map_current;

	map_current = maplist->first->next; /* skip sentinel */
	while (map_current) {
		printf("%s => 0x%08lx-0x%08lx\n", map_current->pathname,
		       map_current->dyn_addr_init,  map_current->dyn_addr_end);
		map_current = map_current->next;
	}
}
