/*
 * This file is part of functracer-postproc.
 *
 * Copyright (C) 2008 by Nokia Corporation
 *
 * Contact: Eero Tamminen <eero.tamminen@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <config.h>
#include <errno.h>
#include <regex.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "compress.h"
#include "list.h"
#include "maps.h"
#include "options.h"
#include "resolve.h"
#include "trace_file.h"

/* Initialize trace list */
int list_init (t_list *list)
{
	list->first = (t_line *) malloc(sizeof(t_line));
	if (!list->first) {
		perror("malloc");
		return -ENOMEM;
	}

	list->last = list->first;
	list->first->next = NULL;
	list->first->prev = NULL;

	/* sanity check */
	list->last->type = TYPE_INVALID;

	return 0;
}

/* Verify if the list is empty */
int list_is_empty (t_list *list)
{
	return (list->first == list->last);
}

/* Print backtrace list contents */
static int btlist_print (t_trace_file *trace_data, long offset)
{
	static t_address address, cut_addr = 0;
	static t_address_info *addr_info;
	static t_map *map;
	t_maplist *maplist = &trace_data->maplist;
	char line[LINE_MAX];
	int ret, is_empty = maplist_is_empty(maplist);

	addr_info = (t_address_info *) malloc (sizeof(t_address_info));
	if (!addr_info) {
		perror("malloc");
		return -ENOMEM;
	}

	if (is_empty && arguments.resolve) {
		fprintf(stderr, "error: map list is empty.\n");
		return -EINVAL;
	}

	while (1) {
		address = read_backtrace_lines(trace_data, offset, line,
					       sizeof(line));
		if (address <= 0)
			break;
		offset = ftell(trace_data->trace_file);

		if(!is_empty) {
			map = find_map_entry_from_address(maplist, address);
			if (map == NULL) {
				printf("   0x%08lx: <address unknown>\n", address);
				continue;
			}
		}

		if (arguments.resolve) {
			if ((ret = resolve_address(map, address, addr_info))) {
				fprintf(stderr, "error: invalid backtrace \
					address\n");
				return ret;
			}

			/* don't print backtraces below main() */
			if ((!cut_addr) && !strcmp(addr_info->function, "__libc_start_main")) {
				cut_addr = address;
				break;
			} else if (cut_addr == address)
				break;

			if (strcmp(addr_info->filename, "<undefined>") != 0) {
				if (addr_info->line == 0)
					printf ("   0x%08lx: %s (%s in %s)\n",
						address,
						addr_info->function,
						basename(addr_info->filename),
						basename(map->pathname));
				else printf ("   0x%08lx: %s (%s:%d in %s)\n",
						address,
						addr_info->function,
						basename(addr_info->filename),
						addr_info->line,
						basename(map->pathname));
			} else
				printf ("%s (in %s)\n", line, basename(map->pathname));
		} else {
			if (is_empty)
				printf ("%s\n", line);
			else
				printf ("%s (in %s)\n", line, basename(map->pathname));
		}
	}

	free(addr_info);

	return 0;
}

static int list_print_header (t_line *aux)
{
	if ((arguments.compress || arguments.leak || arguments.freed) &&
	    aux->type == TYPE_FREE)
		return -1;
	printf ("%s\n",	aux->header);
	return 0;
}

/* Print list contents */
int list_print (t_trace_file *trace_data) {
	t_line *aux;
	int ret;

	if (list_is_empty(&trace_data->list))
		return 0;

	aux = trace_data->list.first->next; /* skip sentinel */

	/* Maps information must be printed when the name resolution is not
 	 * enabled. */
	if (!arguments.resolve)
		maplist_print (&trace_data->maplist);
	while (aux)
	{
		if (!(arguments.blocks || arguments.size)) {
			if (list_print_header(aux) < 0)
				goto end;
		}

		if (aux->another_stack) {
			if (arguments.blocks || arguments.size) {
				if (aux->type == TYPE_FREE ||
				    aux->blocks < arguments.blocklimit)
					goto end;
				if (arguments.blocks)
					printf ("\n%d blocks, total %d bytes\n",
						aux->blocks, aux->total_size);
				else
					printf ("\n%d bytes in %d blocks\n",
						aux->total_size, aux->blocks);
			}
			if ((ret = btlist_print(trace_data, aux->offset)))
				return ret;
		}

end:
		aux = aux->next;
	}

	return 0;
}

/* Free list memory allocations */
int list_free (t_list *list)
{

	t_line *current, *aux;

	current = list->first->next; /* skip sentinel */

	free(list->first); /* free sentinel */

	while (current)
	{
		aux = current;
		current = current->next;

		free(aux);
	}

	return 0;
}

/* Add a member to trace list */
int list_add (t_list *list,
	t_line *line_info) {

	t_line *aux;

	list->last->next = (t_line *) malloc(sizeof(t_line));
	if (!list->last->next) {
		perror("malloc");
		return -ENOMEM;
	}

	aux = list->last;
	list->last = list->last->next;
	list->last->prev = aux;
	list->last->next = NULL;

	strcpy(list->last->header, line_info->header);
	list->last->type = line_info->type;
	list->last->address = line_info->address;
	list->last->size = line_info->size;
	list->last->another_stack = 1;
	list->last->total_size = 0;
	list->last->blocks = 0;

	list->last->offset = line_info->offset;

	return 0;
}

/* Remove last member from trace list */
int list_remove_last (t_list *list)
{
	t_line *aux;

	aux = list->last;
	list->last = list->last->prev;

	free(aux);

	return 0;
}

/* Remove one member from trace list */
int list_remove_item (t_list *list, t_line *item)
{
	if (item == NULL)
		return 0;
	if (list->last == item) /* last item */
		list->last = item->prev;

	if (list->first->next == item) /* first item, skip sentinel */
		list->first->next = item->next;

	item->prev->next = item->next;
	
	if (item->next != NULL)
		item->next->prev = item->prev;

	free(item);

	return 0;
}

/* Filter trace information (freed allocations and leaks detection) */
int list_filter (t_trace_file *trace_data) {

	t_line *current, *aux, *old_current;
	int found = 0, leaks_count, freed_count;
	t_address free_addr;
	t_list *list = &trace_data->list;

	leaks_count = freed_count = 0;

	current = list->first->next; /* skip sentinel */

	while (current)
	{
		aux = NULL;

		if (current->type != TYPE_FREE) { /* alloc functions only */
			aux = current->next;
			found = 0;

			free_addr = current->address;

			while (aux)
			{
				if (aux->type == TYPE_FREE &&
				    free_addr == aux->address) {
					if (arguments.freed)
						freed_count++;
					found = 1;
					break;
				}
				aux = aux->next;
			}

			if (!found && arguments.leak)
				leaks_count++;
		} else {
			/* unmatched free of address allocated before trace was
			 * started 
			 */
			old_current = current;
			current = current->next;
			list_remove_item(list, old_current);
			continue;
		}
		old_current = current;
		current = current->next;
		if (found) {
			/* remove allocation item if in leak detection mode */
			if (arguments.leak)
				list_remove_item(list, old_current);
			/* remove deallocation item */
			if (aux->type == TYPE_FREE) {
				if (current == aux && current != NULL)
					current = current->next;
				list_remove_item(list, aux);
			}
		} else if (arguments.freed) {
			/* remove leaks if in freed allocation detection mode */
			list_remove_item(list, old_current);
		}
	}

	if (arguments.leak)
		printf("%d non-freed resource allocations (possible leaks) "
		       "found\n\n", leaks_count);
	else if (arguments.freed)
		printf("%d freed resource allocations found\n\n", freed_count);

	return 0;
}

void list_sort(t_trace_file *trace_data, int (*cmp)(t_trace_file *trace_data,
	       struct t_line* item1, struct t_line* item2))
{
	t_line *p, *q, *e, *list, *tail;
	int insize, nmerges, psize, qsize, i;
	t_list *list_orig = &trace_data->list;

	if (list_is_empty(list_orig))
		return;
	/* remove sentinel from sort */
	list = list_orig->first->next;
	list->prev = list_orig->first->prev;
	
	insize = 1;

	for (;;) {
		p = list;
		list = tail = NULL;
		nmerges = 0;

		while (p) {
			nmerges++;
			q = p;
			psize = 0;
			for (i = 0; i < insize; i++) {
				psize++;
				q = q->next;
				if (!q)
					break;
			}

			qsize = insize;

			while (psize > 0 || (qsize > 0 && q)) {

				if (!psize) {
					e = q;
					q = q->next;
					qsize--;
				} else if (!qsize || !q) {
					e = p;
					p = p->next;
					psize--;
				} else if (cmp(trace_data, p, q) <= 0) {
					e = p;
					p = p->next;
					psize--;
				} else {
					e = q;
					q = q->next;
					qsize--;
				}
				if (tail)
					tail->next = e;
				else
					list = e;
				e->prev = tail;
				tail = e;
			}
			p = q;
		}

		tail->next = NULL;

		if (nmerges <= 1)
			break;

		insize *= 2;
	}

	list_orig->first->next = list;
	list_orig->first->prev = list->prev;
	list->prev = list_orig->first; /* sentinel */
	list_orig->last = tail;
}
