/*
 * This file is a part of control-plugin-loader
 *
 * Copyright (c) 2008 Nokia Corporation
 * Contact: Eero Tamminen <eero.tamminen@nokia.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "control_plugin_loader.h"
#include <unistd.h>
#include <string.h>
#include <libgen.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <signal.h>
#include <string.h>
#include <dirent.h>
#include <unistd.h>
#include <stdio.h>
#include <gtk/gtk.h>
#include <hildon/hildon-program.h>

#define PLUGINS_DIR  "/usr/lib/control-plugin-loader"
#define CONFIG_FILE "/usr/share/control-plugin-loader/plugin_loader.conf"

static GList* plugin_list = NULL;
static gint last_index = -1;
static GtkWidget* notebook = NULL;
static HildonProgram* hilProgram = NULL;
static HildonWindow* window = NULL;

/* Prototypes */
static int one(const struct dirent*);
static int collectGarbage(void *data);
static gboolean delete_event(GtkWidget *widget,
			     GdkEvent *event,
			     gpointer   data);
static gchar *get_plugin_name(const gchar *path);
static void showUi(void);
static void attachPlugin(CtrlPlugin* plug, const gchar *label);
static gboolean removePlugin(GtkWidget *, GdkEvent *);
static gboolean addPlugin (GtkWidget *, GdkEvent*);
static gboolean plugin_unload(CtrlPlugin *plugin);
static GtkWidget* newTab(const gchar *label);
static void loadPlugins(void);
static gboolean parse_config_line(gchar* buffer, gchar* plugin_path, guint path_size);


static int one(const struct dirent* notused)
{
    return 1;
}

static int collectGarbage(void *data)
{
    GList* li;
    GList* next_li;

    for (li = plugin_list; li != NULL; li = next_li) {
        next_li = g_list_next(li);
        CtrlPlugin* plug = li->data;
        if (plug) {
            int close=0;
            int status=0;
            pid_t pid;
            pid = waitpid(plug->pid, &status, WNOHANG|WUNTRACED);
            if (status!=0) {
                if (WIFEXITED(status) || WIFSIGNALED(status) ||
                    WCOREDUMP(status) || WIFSTOPPED(status)) {
                    close=1;
                }
            }

            if (pid<0) {
                close=1;
            }
            if (close) {
                gtk_notebook_remove_page(GTK_NOTEBOOK(notebook), plug->index);
                plugin_list = g_list_remove(plugin_list,plug);
            }

        }
    }
    g_list_free(li);

    return 1;
}

static gboolean 
delete_event( GtkWidget *widget,
	      GdkEvent *event,
	      gpointer   data )
{
    /* callback for "Close" menu command */
    GList* li;

    for (li = plugin_list; li != NULL; li = g_list_next(li)){
        CtrlPlugin* plug = li->data;
        if (plug) {
            plugin_unload(plug);
        }
    }
    g_list_free(plugin_list);
    gtk_main_quit();
    return FALSE;
}

static gchar *
get_plugin_name(const gchar *path){
    /* Make short name for plugin from its path.
     * Like "/usr/lib/control-plugin-loader/control-plugin-example"
     * becomes "Example"
     */
    gchar *fname;
    static char short_name[256];
    gchar prefix[] = "control-plugin-";
    gchar postfix[] = "-not_used";
    gchar *pre;
    gchar *post;
    gchar *start;
    gint len;

    /* first get just application name, remove directories */
    fname = g_path_get_basename(path);
    /* then remove common parts */
    pre = g_strstr_len(fname, strlen(fname), prefix);
    post = g_strstr_len(fname, strlen(fname), postfix);
    if (pre == NULL) {
        /* common prefix not found, use from beginning */
        start = fname;
    } else {
        /* name found, jump over prefix */
        start = pre + strlen(prefix);
    }
    if (post == NULL) {
        /* postfix not found, use rest of the name */
        len = strlen(start);
    } else {
        len = strlen(start) - strlen(postfix);
    }
    g_strlcpy(short_name, start, len+1);
    /* make first char uppercase */
    short_name[0] = g_ascii_toupper(short_name[0]);
    g_free(fname);
    return (short_name);
}

static CtrlPlugin *
plugin_load (const gchar *path)
{
    CtrlPlugin *plugin;
    pid_t branch;
    gchar *plug_name;

    g_return_val_if_fail (path != NULL, NULL);
    g_return_val_if_fail (g_file_test(path, G_FILE_TEST_IS_EXECUTABLE), NULL);
    plug_name = get_plugin_name(path);
    GtkWidget *socket = gtk_socket_new();
    if (socket == NULL) {
        return NULL;
    }

    plugin = g_new0 (CtrlPlugin, 1);
    g_assert (plugin);
    plugin->widget = socket;

    plugin_list = g_list_append(plugin_list, plugin);
    gtk_widget_show_all(socket);
    attachPlugin (plugin, plug_name);
    gtk_widget_realize(socket);

    branch = fork();

    if (branch == 0) {
        /* Plugin */
        char buffer[256];
        snprintf(buffer,255,"%i",gtk_socket_get_id(GTK_SOCKET(socket)));
        buffer[255]=0;
        execl(path,path,buffer,NULL);
        printf("Exec failed: %s\n",path);
        _exit(0);
    } else if (branch < 0) {
        return NULL;
    } else {
        plugin->pid = branch;
        return plugin;
    } 
}

static gboolean plugin_unload (CtrlPlugin *plugin)
{
    if (plugin != NULL) {
        kill(plugin->pid, SIGTERM);
        return TRUE;
    }
    return FALSE;
}

static gboolean 
addPlugin (GtkWidget *widget, GdkEvent *event)
{
    /* Callback for "Add Plugin" menu command. */
    GtkWidget* dialog;

    dialog = gtk_file_chooser_dialog_new ("Open File",
                                          GTK_WINDOW (window),
                                          GTK_FILE_CHOOSER_ACTION_OPEN,
                                          GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                          GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
                                          NULL);

    if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT) {
        gchar *filename;
        filename = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER (dialog));
        CtrlPlugin* plug = plugin_load(filename);
        if (plug == NULL) {
            g_print("Plugin load failed!\n");
        }
        g_free(filename);
    }

    gtk_widget_destroy(dialog);

    return TRUE;

}

static gboolean 
removePlugin (GtkWidget *widget, GdkEvent *event)
{
    /* Callback for "Remove plugin" menu command */

    gtk_notebook_remove_page(GTK_NOTEBOOK(notebook), 
                             gtk_notebook_get_current_page(GTK_NOTEBOOK(notebook)));
    return TRUE;
}

static GtkWidget* newTab(const gchar *label)
{
    GtkWidget* frame;   
    GtkWidget* tab_label = gtk_label_new(label);
    frame = gtk_frame_new (NULL);
    gtk_frame_set_shadow_type(GTK_FRAME(frame),GTK_SHADOW_NONE);
    if (notebook != NULL) {
        last_index = gtk_notebook_append_page (GTK_NOTEBOOK(notebook), frame, 
                                               GTK_WIDGET(tab_label));
        gtk_widget_show_all(notebook);
    }
    return frame;
}

static void attachPlugin (CtrlPlugin* plug, const gchar *label)
{    
  
    if (plug == NULL) return;
    GtkWidget* frame = newTab(label);
    GtkWidget* plugin = plug->widget;
    if (frame != NULL && plugin != NULL) {
        plug->index=last_index;
        gtk_container_add (GTK_CONTAINER (frame), plugin);    
        gtk_widget_show_all(plugin);
    }      
}

static gboolean 
parse_config_line(gchar* buffer, gchar* plugin_path, guint path_size)
{
    /* This function parses one line of config file (buffer).
     * Each line has name of the plugin to be loaded.
     * Empty lines and lines starting with '#' are ignored.
     * Plugin name can include full path or it can be bare base-name.
     * If name is bare base-name then plugin must be located in the default
     * directory for plugins.
     * Return TRUE for good line, FALSE for empty and comment lines.
     * Plugin name and it's path is written into plugin_path.
     */

    gchar* basename;
   
    /* Trim end and beginning of the line */
    buffer = g_strstrip(buffer);

    /* Skip empty and comment lines */
    if ((strlen(buffer) <= 0) || buffer[0] == '#') {
        return FALSE;  
    }

    basename = g_path_get_basename(buffer);
    if (strlen(buffer) > strlen(basename)) {
        /* Line already contains directories, use as it is */
        g_strlcpy(plugin_path, buffer, path_size);
    } else {
        /* This is just file name. Add default directory */
        snprintf(plugin_path, path_size, "%s/%s", PLUGINS_DIR, buffer);
    }
    g_free(basename);
    return TRUE;

}

static void loadPlugins(void)
{
    /* Load all plugins that are mentioned in the config file.
     * If config file doesn't exists, then load all plugins from
     * our default directory.
     */
    int n;
    int i;
    struct dirent** plugin_list;
    gchar plugpath[NAME_MAX + sizeof(PLUGINS_DIR) + 2];
    gchar buffer[NAME_MAX];
    CtrlPlugin* plugin;
    FILE* f;

    /* Open config file */
    f = fopen(CONFIG_FILE, "r");
    if (f != NULL){
        /* Go thru config file and load each plugin */
        while(fgets(buffer, sizeof(buffer), f) != NULL) {
            if (!parse_config_line(buffer, plugpath, sizeof(plugpath))) {
                continue;
            }
            plugin = plugin_load(plugpath);
            if (plugin == NULL) {
                g_print("Failed to load plugin %s\n", plugpath);
            }
        }
        fclose(f);  	
    } else {
        /* Config file not present, load all from the default directory */
        n = scandir(PLUGINS_DIR, &plugin_list, one, alphasort);
        /* Seek through list of directory entries and load plugins */
        for (i = 0; i < n; i++) {
            if (plugin_list[i]->d_name[0]=='.') continue;
            snprintf(plugpath, sizeof(plugpath), "%s/%s", PLUGINS_DIR, plugin_list[i]->d_name); 
            plugin = plugin_load(plugpath);
            if (plugin == NULL) {
                g_print("Failed to load plugin %s\n", plugpath);
            }
            free(plugin_list[i]);
        }
        free(plugin_list);
    }
}


static void showUi(void)
{ 
    GtkMenu* menu;
    GtkMenuItem* miAddPlugin;
    GtkMenuItem* miClosePlugin;
    GtkMenuItem* miQuit;
    GtkMenuItem* miSeparator;

    plugin_list = g_list_alloc();
    hilProgram = HILDON_PROGRAM(hildon_program_get_instance());
    g_set_application_name("Control Plugin Loader");
    window = HILDON_WINDOW(hildon_window_new());
    hildon_program_add_window(hilProgram, HILDON_WINDOW(window));

    notebook = gtk_notebook_new();   
    gtk_container_add(GTK_CONTAINER(window),notebook);
    gtk_notebook_set_show_tabs(GTK_NOTEBOOK (notebook), TRUE);
    gtk_notebook_set_show_border(GTK_NOTEBOOK (notebook), FALSE);
    

    /* Create the menu items. */
    miAddPlugin = GTK_MENU_ITEM(gtk_menu_item_new_with_label("Add plugin"));
    miClosePlugin = GTK_MENU_ITEM(gtk_menu_item_new_with_label("Remove plugin"));
    miSeparator = GTK_MENU_ITEM(gtk_separator_menu_item_new());
    miQuit = GTK_MENU_ITEM(gtk_menu_item_new_with_label("Close"));
    menu = g_object_new(GTK_TYPE_MENU, NULL);
    gtk_container_add(GTK_CONTAINER(menu), GTK_WIDGET(miAddPlugin));
    gtk_container_add(GTK_CONTAINER(menu), GTK_WIDGET(miClosePlugin));
    gtk_container_add(GTK_CONTAINER(menu), GTK_WIDGET(miSeparator));
    gtk_container_add(GTK_CONTAINER(menu), GTK_WIDGET(miQuit));
    hildon_program_set_common_menu(hilProgram, menu);


    /* Connect signals to menus */
    g_signal_connect (G_OBJECT (window), "delete_event",
                      G_CALLBACK (delete_event), NULL);
    g_signal_connect (G_OBJECT (miQuit), "activate",
                      G_CALLBACK (delete_event), NULL);
    g_signal_connect (G_OBJECT (miAddPlugin), "activate",
                      G_CALLBACK (addPlugin), NULL);
    g_signal_connect (G_OBJECT (miClosePlugin), "activate",
                      G_CALLBACK (removePlugin), NULL);

    gtk_widget_show_all (GTK_WIDGET(window));
    g_timeout_add(1000, collectGarbage, NULL);
}

static int init_load(void *data)
{
    loadPlugins();
    return 0;
}

int main (int argc, char* argv[])
{	    
    gtk_init (&argc, &argv);	    
    showUi();	
    g_timeout_add(200, init_load, NULL);
    gtk_main ();
	
    return 0;
}
