/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the EAL package.
 *
 * The Initial Developer of the Original Code is Nokia Corporation.
 * Portions created by the Initial Developer are Copyright (C) 2005
 * the Initial Developer. All Rights Reserved.
 *
 * Contact: Leonid Zolotarev <leonid.zolotarev@nokia.com>
 *
 * ***** END LICENSE BLOCK ***** */

#ifndef __G_WEB_BUS_H__
#define __G_WEB_BUS_H__

#include <sys/un.h>
#include <glib-object.h>

G_BEGIN_DECLS

//Bus names
#define G_WEB_BUS_EAL_NAME                      "WEB_BUS_EAL_NAME"
#define G_WEB_BUS_UI_NAME                       "WEB_BUS_UI_NAME"

/// Maximum size of a message to be delivered through bus
#define G_WEB_BUS_MAX_MSG_SIZE 1024
#define G_WEB_BUS_OBJECT_DATA_KEYWORD "web-bus"

/*
 * Type macros.
 */
#define G_TYPE_WEB_BUS                  (g_web_bus_get_type ())
#define G_WEB_BUS(obj)                  (G_TYPE_CHECK_INSTANCE_CAST ((obj), G_TYPE_WEB_BUS, GWebBus))
#define G_IS_WEB_BUS(obj)               (G_TYPE_CHECK_INSTANCE_TYPE ((obj), G_TYPE_WEB_BUS))
#define G_WEB_BUS_CLASS(klass)          (G_TYPE_CHECK_CLASS_CAST ((klass), G_TYPE_WEB_BUS, GWebBusClass))
#define G_IS_WEB_BUS_CLASS(klass)       (G_TYPE_CHECK_CLASS_TYPE ((klass), G_TYPE_WEB_BUS))
#define G_WEB_BUS_GET_CLASS(obj)        (G_TYPE_INSTANCE_GET_CLASS ((obj), G_TYPE_WEB_BUS, GWebBusClass))

typedef struct _GWebBus      GWebBus;
typedef struct _GWebBusClass GWebBusClass;

/**
 * Prototype of the 'pre' handler
 * @param msg_id Message identifier that is to be handled
 * @param data Received message data
 * @param data_len Length of the received data block
 * @param timeval Time of activity derived from corresponding GSource
 * @param user_data User data supplied with a registration function
 */
typedef void (*GWebBusPreHandler)(guint msg_id,
                                  gpointer data,
                                  guint data_len,
                                  GTimeVal *timeval,
                                  gpointer user_data);

struct _GWebBus {
    GObject parent_instance;
// private:
    /// Our address
    struct sockaddr_un address;
    /// IO channel from messages
    GIOChannel* channel;
    /// Array of callback elements
    GArray* callbacks;
    /// Our socket name
    char *name;
    /// Event source IDs for active watch handler callbacks.
    GHashTable * callback_event_sources;
    gpointer * presenddata;
    gpointer * prerecvdata;
    GWebBusPreHandler presendhandler;
    GWebBusPreHandler prerecvhandler;
    GSource *source;
    // Are we shutting down currently?
    // Needed for callbacks which are called
    // before shutdown and during it.
    gboolean shutdown;

};

/**
 * Prototype of message handler callback.
 * @param msg_id Message identifier that is to be handled
 * @param data Received message data
 * @param data_len Length of the received data block
 * @param user_data User data supplied with g_web_bus_register_callback()
 */
typedef void (*GWebBusCallback)(guint msg_id,
                                gpointer data,
                                guint data_len,
                                gpointer user_data);

/**
 * GWebBus interface declaration.
 */
struct _GWebBusClass {
    GObjectClass parent_class;
};

/**
 * @return Type instance for GWebBus
 */
GType g_web_bus_get_type();

/**
 * Creates a new bus endpoint with given name
 * @param name Logical name of this party, e.g. "server"
 * @return Pointer to a new GWebBus instance
 */
GWebBus* g_web_bus_new(const gchar* name);

/**
 * Registers a callback function for the given message ID
 * @param self Pointer to GWebBus instance
 * @param msg_id Message identifier
 * @param callback Message handler function with prototype g_web_bus_callback_t
 * @param user_data Pointer that will be passed to callback as-is
 * @return TRUE if the callback was successfully registered
 */
gboolean g_web_bus_register_callback(GWebBus* self,
                                     guint msg_id,
                                     GWebBusCallback callback,
                                     gpointer user_data);

/**
 * Check whether a callback for the given message ID has been registered.
 * @param self Pointer to GWebBus instance
 * @param msg_id Message identifier
 * @return TRUE if a callback for msg_id has already been registered
 */
gboolean g_web_bus_is_registered(GWebBus* self, guint msg_id);

/**
 * Sends a message with given ID and data.
 * This function allows a priority to be set for the message handler call.
 * Predefined priority values for GLib are G_PRIORITY_HIGH, G_PRIORITY_DEFAULT,
 * G_PRIORITY_HIGH_IDLE, G_PRIORITY_DEFAULT_IDLE and G_PRIORITY_LOW. See
 * GLib documentation for The Main Event Loop more information.
 * @param self Pointer to GWebBus instance
 * @param name Logical name of the receiving party, e.g. "client"
 * @param msg_id Message identifier
 * @param priority Priority at which the message is handled 
 * @param data Message data
 * @param data_len Length of message data
 * @return TRUE if the message was successfully sent.
 */
gboolean g_web_bus_send_message_with_priority(GWebBus* self,
                                              const gchar* name,
                                              guint msg_id,
                                              gint priority,
                                              const gpointer data,
                                              guint data_len);

/**
 * Sends a message with given ID and data at G_PRIORITY_DEFAULT_IDLE priority.
 * @param self Pointer to GWebBus instance
 * @param name Logical name of the receiving party, e.g. "client"
 * @param msg_id Message identifier
 * @param data Message data
 * @param data_len Length of message data
 * @return TRUE if the message was successfully sent.
 */
gboolean g_web_bus_send_message(GWebBus* self,
                                const gchar* name,
                                guint msg_id,
                                const gpointer data,
                                guint data_len);

/**
 * By some unknown reasons we can lose working socket
 * this method will just reinitialize socket and channel
 */
gboolean g_web_bus_update_channel(GWebBus* self);

/**
 * Register a handler which is executed before messages are sent
 * @param self Pointer to GWebBus instance
 * @param handler the handler
 */
void
g_web_bus_register_pre_send(GWebBus *self,
                            GWebBusPreHandler handler,
                            gpointer data);

/**
 * Register a handler which is executed before messages are received and processed
 * @param self Pointer to GWebBus instance
 * @param handler the handler
 */
void
g_web_bus_register_pre_recv(GWebBus *self, 
                            GWebBusPreHandler handler,
                            gpointer data);


/**
 * Increase the reference count of a GWebBus instance.
 */
#define g_web_bus_ref(object)      g_object_ref(object)

/**
 * Increase the reference count of a GWebBus instance and possibly remove 
 * the floating reference, if object has a floating reference.
 */
#define g_web_bus_ref_sink(object) g_object_ref_sink(object)

/**
 * Decrease the reference count of a GWebBus instance.
 * When its reference count drops to 0, the object is finalized
 * (i.e. its memory is freed).
 */
#define g_web_bus_unref(object)    g_object_unref(object)

G_END_DECLS

#endif // __G_WEB_BUS_H__
