/* $Id: shmalloc.h,v 1.5 2005/02/03 11:02:20 jlaako Exp $ */

/*

    Shared memory allocator.
    
    Copyright (C) 2005-2007 Nokia Corporation.

    Contact: Jussi Laako <jussi.laako@nokia.com>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

*/


#ifndef SHMALLOC_H
#define SHMALLOC_H


#include "sharedmem.h"


/** Align has to be big enough to hold block flags */
#define SHAREDMEM_ALIGN                 8

/** Flags for blocks. */
#define SHAREDMEM_BLOCK_INUSE           0x1

/** Mask for flags. */
#define SHAREDMEM_FLAGS_MASK            0x1


#ifdef __cplusplus
extern "C" {
#endif


/**
    Allocation block control structure.
*/
typedef struct _sharedmem_block_t
{
    size_t size;                        /**< Size of the allocation block */
    long next;                          /**< Offset to the next block (slist) */
} sharedmem_block_t;


/**
    Allocator descriptor.
*/
typedef struct _sharedmem_alloc_t
{
    sharedmem_t *shminst;               /**< Pointer to the block instance */
} sharedmem_alloc_t;


/** Get pointer to shared memory object. */
#define SHMALLOC_SHAREDMEM(i)           ((i)->shminst)


/**
    Initialize allocator on shared memory.
    
    \param inst Instance of shared memory allocator object
    \param shminst Instance of shared memory object
    \return 0 on success
*/
int sharedmem_alloc_create (sharedmem_alloc_t *inst, sharedmem_t *shminst);
/**
    Allocate arbitrary sized block from shared memory.
    
    \param inst Instance of shared memory allocator object
    \param size Size of block to allocate
    \return offset to shared memory
*/
long sharedmem_alloc (sharedmem_alloc_t *inst, size_t size);
/**
    Reallocate block from shared memory.
    
    \note The offset may change depending on availability of free space.
    
    \param inst Instance of shared memory allocator object
    \param old_offset Offset to reallocate
    \param size New size for the block
    \return offset to reallocated block
*/
long sharedmem_realloc (sharedmem_alloc_t *inst, long old_offset, size_t size);
/**
    Free previously allocated shared memory block.
    
    \param inst Instance of shared memory allocator object.
    \param offset Offset to free
    \return 0 on success
*/
int sharedmem_free (sharedmem_alloc_t *inst, long offset);
/**
    Get mapped pointer for offset.
    
    \param inst Instance of shared memory allocator object.
    \param offset Offset to get pointer for
    \return mapped pointer
*/
void * sharedmem_alloc_get_ptr (sharedmem_alloc_t *inst, long offset);
/**
    Grow the parent shared memory block and reflect the change in allocator.
    
    \param inst Instance of sharedmemory allocator object.
    \param new_size New size of the block
    \return 0 on success
*/
int sharedmem_alloc_grow (sharedmem_alloc_t *inst, size_t new_size);
/**
    Print list of allocations to stderr.
    
    \param inst Instance of shared memory allocator object.
*/
void sharedmem_alloc_list_allocs (sharedmem_alloc_t *inst);
/**
    Print list of free areas to stderr.
    
    \param inst Instance of shared memory allocator object.
*/
void sharedmem_alloc_list_frees (sharedmem_alloc_t *inst);


#ifdef __cplusplus
}
#endif

#endif
