/* $Id: shmalloc.c,v 1.5 2005/02/03 11:02:20 jlaako Exp $ */

/*

    Shared memory allocator.
    
    Copyright (C) 2005-2007 Nokia Corporation.

    Contact: Jussi Laako <jussi.laako@nokia.com>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

*/


/* specify source conformance to include necessary functions */
#define _XOPEN_SOURCE 600
#define _POSIX_C_SOURCE 200112L


#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <limits.h>
#include <sys/types.h>

#include "shmalloc.h"
#include "sharedmem_private.h"


/* --- internal */

/** Size also contains flags, so we need a macro to mask the flags out. */
#define BL_SZ(b) ((b)->size & ~SHAREDMEM_FLAGS_MASK)


/* set error message */
static void set_error (sharedmem_alloc_t *inst, int error_code,
    const char *error_str)
{
    _sharedmem_set_error(SHMALLOC_SHAREDMEM(inst), error_code, error_str);
}


/* get size rounded to block aligned boundary */
static size_t aligned_block_size (size_t size)
{
    size_t block_size;
    
    block_size = sizeof(sharedmem_block_t) + size;
    block_size += SHAREDMEM_ALIGN - block_size % SHAREDMEM_ALIGN;
    return block_size;
}


/* is the block reserved? */
static inline int block_is_reserved (sharedmem_block_t *block)
{
    if (block->size & SHAREDMEM_BLOCK_INUSE)
        return 1;
    else
        return 0;
}


/* is the block free? */
static inline int block_is_free (sharedmem_block_t *block)
{
    if (block->size & SHAREDMEM_BLOCK_INUSE)
        return 0;
    else
        return 1;
}


/* get offset mapped to process' address space */
static inline sharedmem_block_t * get_offs_ptr (sharedmem_alloc_t *inst,
    long offset)
{
    char *base_ptr;
    
    base_ptr = (char *) SHAREDMEM_PTR(SHMALLOC_SHAREDMEM(inst));
    return ((sharedmem_block_t *) (base_ptr + offset));
}


/* get data offset for specific block offset */
static inline long get_data_offs (long offset)
{
    return (offset + sizeof(sharedmem_block_t));
}


/* find space for block allocation on first-fit basis */
static long find_free (sharedmem_alloc_t *inst, size_t block_size)
{
    long block = 0;
    sharedmem_block_t *block_ptr;

    while (block >= 0)
    {
        block_ptr = get_offs_ptr(inst, block);
        if (block_is_free(block_ptr))
        {
            if (BL_SZ(block_ptr) >= block_size)
                return block;
        }
        block = block_ptr->next;
    }

    return -1;
}


/* if the last block is free, return it's offset, otherwise add new after it */
static long get_last_free (sharedmem_alloc_t *inst)
{
    long block = 0;
    long this;
    long next;
    sharedmem_block_t *block_ptr;
    sharedmem_block_t *next_ptr;

    do {
        this = block;
        block_ptr = get_offs_ptr(inst, block);
        block = block_ptr->next;
    } while (block >= 0);
    if (!block_is_free(block_ptr))
    {
        next = this + BL_SZ(block_ptr);
        next_ptr = get_offs_ptr(inst, next);
        next_ptr->size = 0;
        next_ptr->next = -1;
        block_ptr->next = next;
        return next;
    }
    return this;
}


/* expand the current block if there's enough space available */
static long expand_block (sharedmem_alloc_t *inst, long oldblock,
    size_t block_size)
{
    long block = oldblock;
    long next;
    long flags;
    sharedmem_block_t *block_ptr;
    sharedmem_block_t *next_ptr;

    block_ptr = get_offs_ptr(inst, block);
    next = block_ptr->next;
    if (next >= 0)
    {
        next_ptr = get_offs_ptr(inst, next);
        if (block_is_free(next_ptr))
        {
            /* next block has enough room to expand current */
            if ((BL_SZ(block_ptr) + BL_SZ(next_ptr)) >= block_size)
            {
                flags = next_ptr->size & SHAREDMEM_FLAGS_MASK;
                /* resize next */
                next_ptr->size = BL_SZ(next_ptr) - 
                    (block_size - BL_SZ(block_ptr));
                next_ptr->size |= flags;
                /* next ended up being zero sized, remove */
                if (BL_SZ(next_ptr) == 0)
                    block_ptr->next = next_ptr->next;
                /* current was just resized, return current */
                return block;
            }
        }
    }

    /* no free space */
    return -1;
}


/* shrink the current block by splitting out the freed space */
static long shrink_block (sharedmem_alloc_t *inst, long block,
    size_t block_size)
{
    long next;
    long flags;
    sharedmem_block_t *block_ptr;
    sharedmem_block_t *next_ptr;
    size_t next_size;

    block_ptr = get_offs_ptr(inst, block);
    /* create new next, zeroed flags */
    next_size = BL_SZ(block_ptr) - block_size;
    next = block + block_size;
    next_ptr = get_offs_ptr(inst, next);
    memset(next_ptr, 0x00, sizeof(sharedmem_block_t));
    /* move old next one ahead */
    next_ptr->next = block_ptr->next;
    next_ptr->size = next_size;
    /* the new size, new next, retain flags */
    flags = block_ptr->size & SHAREDMEM_FLAGS_MASK;
    block_ptr->size = block_size;
    block_ptr->size |= flags;
    block_ptr->next = next;
    
    return block;
}


static int _coalesce_blocks (sharedmem_alloc_t *inst)
{
    int retval = 0;
    long block;
    long next;
    sharedmem_block_t *block_ptr;
    sharedmem_block_t *next_ptr;
    
    block = 0;
    while (block >= 0)
    {
        /*printf("block = %x\n", (size_t) block);*/
        block_ptr = get_offs_ptr(inst, block);
        /* block is free and next exists */
        if (block_is_free(block_ptr) && block_ptr->next >= 0)
        {
            next = block_ptr->next;
            /*printf("next = %x\n", (size_t) next);*/
            next_ptr = get_offs_ptr(inst, next);
            /* two consecutive free blocks? coalesce... */
            if (block_is_free(next_ptr))
            {
                /*printf("combine %x && %x\n", (size_t) block, (size_t) next);*/
                block_ptr->size = BL_SZ(block_ptr) + BL_SZ(next_ptr);
                block_ptr->next = next_ptr->next;
                memset(next_ptr, 0x00, sizeof(sharedmem_block_t));
                retval = 1;
            }
        }
        block = block_ptr->next;
    }

    return retval;
}


/* block coalescence, is done on free and realloc */
static void coalesce_blocks (sharedmem_alloc_t *inst)
{
    while (_coalesce_blocks(inst));
}


static long sharedmem_alloc_nl (sharedmem_alloc_t *inst, size_t size)
{
    long block;
    long next;
    long offset = -1;
    size_t block_size;
    sharedmem_block_t *block_ptr;
    sharedmem_block_t *next_ptr;
    
    block_size = aligned_block_size(size);
    /* find free space */
    block = find_free(inst, block_size);
    if (block >= 0)
    {
        /* allocate and split the free space if needed */
        block_ptr = get_offs_ptr(inst, block);
        /* free space is larger than allocation request, split */
        if ((BL_SZ(block_ptr) - block_size) > sizeof(sharedmem_block_t))
        {
            /* next block starts after this */
            next = block + block_size;
            next_ptr = get_offs_ptr(inst, next);
            /* was there this->next already? */
            if (block_ptr->next >= 0)
            {
                /* newnext->next = this->next */
                next_ptr->next = block_ptr->next;
                /* this->next = newnext */
                block_ptr->next = next;
            }
            else
            {
                /* newnext->next = none (last) */
                next_ptr->next = -1;
                /* this->next = newnext */
                block_ptr->next = next;
            }
            /* next->size = this->size - newsize */
            next_ptr->size = BL_SZ(block_ptr) - block_size;
            /* this->size = newsize */
            block_ptr->size = (block_size | SHAREDMEM_BLOCK_INUSE);
        }
        else
        {
            block_ptr->size |= SHAREDMEM_BLOCK_INUSE;
        }
        offset = get_data_offs(block);
    }
    else
    {
        set_error(inst, 0, "sharedmem_alloc(): no space available");
    }

    return offset;
}


static int sharedmem_free_nl (sharedmem_alloc_t *inst, long offset)
{
    long block;
    sharedmem_block_t *block_ptr;

    block = offset - sizeof(sharedmem_block_t);
    if (block < 0)
        return -1;

    block_ptr = get_offs_ptr(inst, block);
    if (block_is_reserved(block_ptr))
    {
        block_ptr->size ^= SHAREDMEM_BLOCK_INUSE;
        coalesce_blocks(inst);
    }
    else
    {
        set_error(inst, 0, "sharedmem_free(): double free?");
        return -1;
    }

    return 0;
}


/* --- public */


int sharedmem_alloc_create (sharedmem_alloc_t *inst, sharedmem_t *shminst)
{
    sharedmem_block_t *block_ptr;

    if (inst == NULL || shminst == NULL)
        return -1;
    memset(inst, 0x00, sizeof(sharedmem_alloc_t));

    SHMALLOC_SHAREDMEM(inst) = shminst;
    block_ptr = (sharedmem_block_t *) SHAREDMEM_PTR(SHMALLOC_SHAREDMEM(inst));
    if (block_ptr == NULL)
    {
        set_error(inst, 0, "sharedmem_alloc_create(): sharedmem_t * == NULL");
        return -1;
    }

    if (!sharedmem_lock(SHMALLOC_SHAREDMEM(inst)))
        return -1;

    block_ptr->size = SHAREDMEM_SIZE(SHMALLOC_SHAREDMEM(inst));
    block_ptr->next = -1;

    if (!sharedmem_unlock(SHMALLOC_SHAREDMEM(inst)))
        return -1;

    return 0;
}


long sharedmem_alloc (sharedmem_alloc_t *inst, size_t size)
{
    long offset = -1;
    
    if (inst == NULL)
        return -1;
    if (size == 0)
        return -1;

    if (!sharedmem_lock(SHMALLOC_SHAREDMEM(inst)))
        return -1;

    offset = sharedmem_alloc_nl(inst, size);

    sharedmem_unlock(SHMALLOC_SHAREDMEM(inst));
    
    return offset;
}


long sharedmem_realloc (sharedmem_alloc_t *inst, long old_offset, size_t size)
{
    long old;
    long block = -1;
    long offset = -1;
    size_t block_size;
    size_t old_size;
    size_t copy_size;
    sharedmem_block_t *block_ptr;
    
    if (inst == NULL)
        return -1;

    /* new size is 0, so memory is just freed */
    if (size == 0)
        return sharedmem_free(inst, old_offset);

    old = old_offset - sizeof(sharedmem_block_t);
    if (old < 0)
        return -1;

    if (!sharedmem_lock(SHMALLOC_SHAREDMEM(inst)))
        return -1;

    block_ptr = get_offs_ptr(inst, old);
    old_size = BL_SZ(block_ptr);

    block_size = aligned_block_size(size);
    /* block is same size? */
    if (block_size == old_size)
    {
        offset = old_offset;
        goto realloc_out;
    }
    else if (block_size > old_size)
    {
        /* try to expand the current block */
        block = expand_block(inst, old, block_size);
    }
    else
    {
        /* shrink the current block */
        block = shrink_block(inst, old, block_size);
        coalesce_blocks(inst);
    }
    /* resize was successfull */
    if (block >= 0)
    {
        /*block_ptr = get_offs_ptr(inst, block);*/
        offset = get_data_offs(block);
    }
    else /* go to full allocate & copy & free */
    {
        offset = sharedmem_alloc_nl(inst, size);
        if (offset < 0)
        {
            set_error(inst, 0, "sharedmem_realloc(): no space available");
            goto realloc_out;
        }
        copy_size = (BL_SZ(block_ptr) <= old_size) ?
            BL_SZ(block_ptr) : old_size;
        copy_size -= sizeof(sharedmem_block_t);
        memcpy(sharedmem_alloc_get_ptr(inst, offset),
            sharedmem_alloc_get_ptr(inst, old_offset),
            copy_size);
        sharedmem_free_nl(inst, old_offset);
    }

realloc_out:
    sharedmem_unlock(SHMALLOC_SHAREDMEM(inst));
    
    return offset;
}


int sharedmem_free (sharedmem_alloc_t *inst, long offset)
{
    long res = 0;

    if (inst == NULL)
        return -1;

    if (!sharedmem_lock(SHMALLOC_SHAREDMEM(inst)))
        return -1;

    res = sharedmem_free_nl(inst, offset);

    sharedmem_unlock(SHMALLOC_SHAREDMEM(inst));

    return res;
}


void * sharedmem_alloc_get_ptr (sharedmem_alloc_t *inst, long offset)
{
    char *base_ptr;
    
    base_ptr = (char *) SHAREDMEM_PTR(SHMALLOC_SHAREDMEM(inst));
    return ((void *) (base_ptr + offset));
}


int sharedmem_alloc_grow (sharedmem_alloc_t *inst, size_t new_size)
{
    long last;
    size_t old_size;
    size_t size_diff;
    sharedmem_block_t *last_ptr;

    if (inst == NULL)
        return -1;
    if (new_size <= SHAREDMEM_SIZE(SHMALLOC_SHAREDMEM(inst)))
    {
        set_error(inst, 0, "sharedmem_alloc_grow(): new_size <= old_size");
        return -1;
    }
    old_size = SHAREDMEM_SIZE(SHMALLOC_SHAREDMEM(inst));
    sharedmem_resize(SHMALLOC_SHAREDMEM(inst), new_size);
    size_diff = SHAREDMEM_SIZE(SHMALLOC_SHAREDMEM(inst)) - old_size;
    if (size_diff < sizeof(sharedmem_block_t))
    {
        set_error(inst, 0,
            "sharedmem_alloc_grow(): size_diff < sizeof(sharedmem_block_t)");
        return -1;
    }

    if (!sharedmem_lock(SHMALLOC_SHAREDMEM(inst)))
        return -1;

    last = get_last_free(inst);
    last_ptr = get_offs_ptr(inst, last);
    last_ptr->size += size_diff;

    sharedmem_unlock(SHMALLOC_SHAREDMEM(inst));

    return 0;
}


void sharedmem_alloc_list_allocs (sharedmem_alloc_t *inst)
{
    long block = 0;
    sharedmem_block_t *block_ptr;

    if (inst == NULL)
        return;

    fprintf(stderr, "sharedmem_alloc_list_allocs()\n");
    fprintf(stderr, "-----------------------------\n");
    while (block >= 0)
    {
        block_ptr = get_offs_ptr(inst, block);
        if (!block_is_free(block_ptr))
        {
            fprintf(stderr, "0x%0x: offset = 0x%0x, size = %u\n",
                (size_t) block_ptr, (size_t) block, BL_SZ(block_ptr));
        }
        block = block_ptr->next;
    }
    fprintf(stderr, "-----------------------------\n");
}


void sharedmem_alloc_list_frees (sharedmem_alloc_t *inst)
{
    long block = 0;
    sharedmem_block_t *block_ptr;

    if (inst == NULL)
        return;

    fprintf(stderr, "sharedmem_alloc_list_frees()\n");
    fprintf(stderr, "----------------------------\n");
    while (block >= 0)
    {
        block_ptr = get_offs_ptr(inst, block);
        if (block_is_free(block_ptr))
        {
            fprintf(stderr, "0x%0x: offset = 0x%0x, size = %u\n",
                (size_t) block_ptr, (size_t) block, BL_SZ(block_ptr));
        }
        block = block_ptr->next;
    }
    fprintf(stderr, "----------------------------\n");
}
