/* $Id: sharedmem.h,v 1.3 2005/02/03 11:02:19 jlaako Exp $ */

/** \mainpage
    \author Jussi Laako

    Shared memory object for POSIX/IEEE-1003.1 compliant systems.

    Copyright (C) 2005-2007 Nokia Corporation.

    Contact: Jussi Laako <jussi.laako@nokia.com>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA


    \par Overview
    Main purpose for this library is to provide easy way to share data
    using standard method of shared memory. It is divided to two parts,
    shared memory handling and allocation.
    
    \par Shared memory handling
    Instance of the sharedmem_t type represents single instance of shared
    memory block. This is of some specified size rounded up to next page
    boundary. This block is meant to be fixed size for reasonably long
    periods of time, preferably for the duration of it's use. It can be
    resized though. In addition to basic lock/unlock, each block can optionally
    contain synchronization primitives through pthread_mutex_t and
    pthread_cond_t.
    
    \par Shared memory allocation
    Instance of the sharedmem_alloc_t type represents single instance of
    allocatable block of shared memory. It's always based on instance of
    sharedmem_t type object. Size of the parent block defines the size
    available for allocations within this object. Each allocator object
    can contain number of malloc(3) like allocations, total size limited by
    the size of the sharedmem_t type. These can be (re)allocated and freed
    independent of the original process context. Each allocation provides
    address offset which can be casted to process-local pointer by use of
    provided macro, or passed to another process for consumption.

*/

/**
    First example shows just basic functionality of bare sharedmem_t object.
    Second one shows basic functionality of sharedmem_alloc_t allocator object.
    Third example shows really simple use case.

    \example test.c
    \example test3.c
    \example test4.c
*/


#ifndef SHAREDMEM_H
#define SHAREDMEM_H


#include "shm_config.h"

#include <limits.h>
#include <sys/types.h>
#ifdef HAVE_PSHARED
#include <pthread.h>
#endif
#include <semaphore.h>


#define SHAREDMEM_ERROR_SIZE            256
#define SHAREDMEM_NAME_LENGTH           _POSIX_PATH_MAX


#if ((__GNUC__ >= 3) && (__GNUC_MINOR__ >= 1) && !defined(likely))
#define likely(x)           __builtin_expect(x, 1)
#define unlikely(x)         __builtin_expect(x, 0)
#define prefetch(x, w, l)   __builtin_prefetch(x, w, l)
#else
#define likely(x)           (x)
#define unlikely(x)         (x)
#define prefetch(x, w, l)
#endif


#ifdef __cplusplus
extern "C" {
#endif


/**
    Shared memory control structure. Separately mapped.
*/
typedef struct _sharedmem_ctrl_t
{
#ifdef HAVE_PSHARED
    pthread_mutex_t mutex;              /**< Shared mutex */
    pthread_cond_t cond;                /**< Shared condition */
#endif
    volatile size_t size;               /**< Size of the block */
    volatile int ref;                   /**< Reference count */
} sharedmem_ctrl_t;


/**
    Error information structure.
*/
typedef struct _sharedmem_error_t
{
    int code;                           /**< Error code */
    char string[SHAREDMEM_ERROR_SIZE];  /**< Error string */
} sharedmem_error_t;


/**
    Shared memory object instance structure.
*/
typedef struct _sharedmem_t
{
    int fd;                             /**< File descriptor for the object */
    sem_t *sem;                         /**< Semaphore used for locking */
    sharedmem_ctrl_t *ctrl;             /**< Pointer to the mapped control structure */
    size_t mapped_size;                 /**< Mapped size */
    void *pdata;                        /**< Pointer to the mapped data area */
    char name[SHAREDMEM_NAME_LENGTH];   /**< Name of the object */
    sharedmem_error_t error;            /**< Error information */
} sharedmem_t;


/** Get pointer to shared memory mutex. */
#define SHAREDMEM_MUTEX(i)              (&(i)->ctrl->mutex)
/** Get pointer to shared memory condition variable. */
#define SHAREDMEM_COND(i)               (&(i)->ctrl->cond)
/** Get pointer to shared memory semaphore. */
#define SHAREDMEM_SEM(i)                ((i)->sem)
/** Get pointer to shared memory. */
#define SHAREDMEM_PTR(i)                ((i)->pdata)
/** Get size of the shared memory. */
#define SHAREDMEM_SIZE(i)               ((i)->mapped_size)


/**
    Create named shared memory of specified size.
    
    \param inst Instance of shared memory object
    \param name Name of the shared memory (with leading slash)
    \param size Size of the shared memory
    \return 0 on success
*/
int sharedmem_create (sharedmem_t *inst, const char *name, size_t size);
/**
    Open named shared memory.
    
    \param inst Instance of shared memory object
    \param name Name of shared memory
    \return 0 on success
*/
int sharedmem_open (sharedmem_t *inst, const char *name);
/**
    Close named shared memory.
    
    \param inst Instance of shared memory object
    \return 0 on success
*/
int sharedmem_close (sharedmem_t *inst);
/**
    Resize named shared memory.
    
    \param inst Instance of shared memory object
    \param size New size of shared memory
    \return 0 on success
*/
int sharedmem_resize (sharedmem_t *inst, size_t size);
/**
    Resize named shared memory.
    
    \note This updates the size of the mapping to a new size in shared memory.
    
    \param inst Instance of shared memory object
    \return 0 on success
*/
int sharedmem_resize2 (sharedmem_t *inst);
/**
    Check for changes in named shared memory size.
    
    \note This function does not acquire mutex!
    
    \param inst Instance of shared memory object
    \return 0 for no changes, 1 if changed and -1 on error
*/
int sharedmem_size_changed (sharedmem_t *inst);
/**
    Lock shared memory.
    
    \param inst Instance of shared memory object
    \return success?
*/
int sharedmem_lock (sharedmem_t *inst);
/**
    Unlock shared memory.
    
    \param inst Instance of shared memory object
    \return success?
*/
int sharedmem_unlock (sharedmem_t *inst);
/**
    Get error code.
    
    \param inst Instance of shared memory object
    \return error code
*/
int sharedmem_get_error_code (sharedmem_t *inst);
/**
    Get error string.
    
    \param inst Instance of shared memory object
    \return pointer to error string
*/
const char * sharedmem_get_error_string (sharedmem_t *inst);


#ifdef __cplusplus
}
#endif

#endif
