/* $Id: sharedmem.c,v 1.3 2005/02/03 11:02:19 jlaako Exp $ */

/*

    Shared memory object for POSIX/IEEE-1003.1 compliant systems.

    Copyright (C) 2005-2007 Nokia Corporation.

    Contact: Jussi Laako <jussi.laako@nokia.com>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

*/


/* specify source conformance to include necessary functions */
#define _XOPEN_SOURCE 600
#define _POSIX_C_SOURCE 200112L


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <time.h>
#include <signal.h>
#include <unistd.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/mman.h>
#ifdef HAVE_PSHARED
#include <pthread.h>
#else
#include <semaphore.h>
#endif

#include "sharedmem.h"
#include "sharedmem_private.h"


/* --- internal */

long sharedmem_page_size = 0;


/* set error code and message */
void _sharedmem_set_error (sharedmem_t *inst, int error_code,
    const char *error_str)
{
    char errcodestr[SHAREDMEM_ERROR_SIZE / 2];

    if (inst == NULL)
        return;
    inst->error.code = error_code;
    if (error_str != NULL)
    {
        if (error_code > 0)
        {
            strerror_r(error_code, errcodestr, SHAREDMEM_ERROR_SIZE / 2);
            snprintf(inst->error.string, SHAREDMEM_ERROR_SIZE, "%s: %s",
                error_str, errcodestr);
        }
        else
        {
            snprintf(inst->error.string, SHAREDMEM_ERROR_SIZE, "%s",
                error_str);
        }
    }
    else
    {
        inst->error.string[0] = '\0';
    }
}


static void set_error (sharedmem_t *inst, int error_code, const char *error_str)
{
    _sharedmem_set_error(inst, error_code, error_str);
}


/* get system page size */
static int init_page_size (sharedmem_t *inst)
{
    if (sharedmem_page_size > 0)
        return 0;
    sharedmem_page_size = sysconf(_SC_PAGE_SIZE);
    if (sharedmem_page_size <= 0)
    {
        set_error(inst, errno, "init_page_size(): sysconf()");
        return -1;
    }
    return 0;
}


/* round size to nearest page boundary */
static size_t round_to_page_boundary (size_t size)
{
    size_t size_res;
    
    size_res = sharedmem_page_size * 
        ((size + (sharedmem_page_size - 1)) / 
        sharedmem_page_size);
    
    return size_res;
}


/* map the shared memory blocks */
static int do_maps (sharedmem_t *inst, size_t size)
{
    size_t size_ctrl;
    size_t size_data;
    size_t size_data_n;

    size_ctrl = round_to_page_boundary(sizeof(sharedmem_ctrl_t));
    /* map the control block, this is thus protected from the data */
    inst->ctrl = (sharedmem_ctrl_t *) mmap(NULL, size_ctrl,
        PROT_READ|PROT_WRITE, MAP_SHARED, inst->fd, 0);
    if (inst->ctrl == NULL)
    {
        set_error(inst, errno, "do_maps(): mmap()");
        return -1;
    }

    size_data_n = (size > 0) ? size : inst->ctrl->size;
    size_data = round_to_page_boundary(size_data_n);
    /* map the data block */
    inst->pdata = mmap(NULL, size_data, PROT_READ|PROT_WRITE, MAP_SHARED,
        inst->fd, size_ctrl);
    if (inst->pdata == NULL)
    {
        munmap(inst->ctrl, size_ctrl);
        set_error(inst, errno, "do_maps(): mmap()");
        return -1;
    }
    inst->mapped_size = size_data_n;

    return 0;
}


/* unmap the shared memory blocks */
static int undo_maps (sharedmem_t *inst)
{
    int res, res2;
    size_t size_ctrl;
    size_t size_data;
    
    size_ctrl = round_to_page_boundary(sizeof(sharedmem_ctrl_t));
    size_data = round_to_page_boundary(inst->mapped_size);
    /* unmap the data block */
    res = munmap(inst->pdata, size_data);
    inst->mapped_size = 0;
    inst->pdata = NULL;
    /* unmap the control block */
    res2 = munmap(inst->ctrl, size_ctrl);
    inst->ctrl = NULL;
    if (res != 0 || res2 != 0)
    {
        set_error(inst, errno, "undo_maps(): munmap()");
        return -1;
    }
    
    return 0;
}


static int init_mutex (sharedmem_t *inst)
{
#ifdef HAVE_PSHARED
    pthread_mutexattr_t mutex_attr;

    /* initialize mutex object in shared memory for process shared access */
    if (pthread_mutexattr_init(&mutex_attr) != 0)
    {
        set_error(inst, errno, "init_mutex(): pthread_mutexattr_init()");
        return -1;
    }
    if (pthread_mutexattr_setpshared(&mutex_attr, PTHREAD_PROCESS_SHARED) != 0)
    {
        set_error(inst, errno, "init_mutex(): pthread_mutexattr_setpshared()");
        return -1;
    }
    if (pthread_mutex_init(&inst->ctrl->mutex, &mutex_attr) != 0)
    {
        set_error(inst, errno, "init_mutex(): pthread_mutex_init()");
        return -1;
    }
    if (pthread_mutexattr_destroy(&mutex_attr) != 0)
    {
        set_error(inst, errno, "init_mutex(): pthread_mutexattr_destroy()");
        return -1;
    }
#endif  /* HAVE_PSHARED */
    
    return 0;
}


static int destroy_mutex (sharedmem_t *inst)
{
#ifdef HAVE_PSHARED
    return pthread_mutex_destroy(&inst->ctrl->mutex);
#else
    return 0;
#endif
}


static int init_cond (sharedmem_t *inst)
{
#ifdef HAVE_PSHARED
    pthread_condattr_t cond_attr;

    /* initialize condition variable object in shared memory for process shared
       access */
    if (pthread_condattr_init(&cond_attr) != 0)
    {
        set_error(inst, errno, "init_cond(): pthread_condattr_init()");
        return -1;
    }
    if (pthread_condattr_setpshared(&cond_attr, PTHREAD_PROCESS_SHARED) != 0)
    {
        set_error(inst, errno, "init_cond(): pthread_condattr_setpshared()");
        return -1;
    }
    if (pthread_cond_init(&inst->ctrl->cond, &cond_attr) != 0)
    {
        set_error(inst, errno, "init_cond(): pthread_cond_init()");
        return -1;
    }
    if (pthread_condattr_destroy(&cond_attr) != 0)
    {
        set_error(inst, errno, "init_cond(): pthread_condattr_destroy()");
        return -1;
    }
#endif  /* HAVE_PSHARED */
    
    return 0;
}


static int destroy_cond (sharedmem_t *inst)
{
#ifdef HAVE_PSHARED
    return pthread_cond_destroy(&inst->ctrl->cond);
#else
    return 0;
#endif
}


static inline int lock_sem (sharedmem_t *inst)
{
    if (unlikely(inst == NULL))
        return 0;
    if (unlikely(inst->sem == NULL))
        return 0;
    if (unlikely(sem_wait(SHAREDMEM_SEM(inst)) != 0))
        return 0;
    return 1;
}


static inline int unlock_sem (sharedmem_t *inst)
{
    if (unlikely(inst == NULL))
        return 0;
    if (unlikely(inst->sem == NULL))
        return 0;
    if (unlikely(sem_post(SHAREDMEM_SEM(inst)) != 0))
        return 0;
    return 1;
}


/* --- public */


int sharedmem_create (sharedmem_t *inst, const char *name, size_t size)
{
    size_t size_real;
    
    if (inst == NULL)
        return -1;
    memset(inst, 0x00, sizeof(sharedmem_t));
    inst->fd = -1;

    /* initialize page size */
    if (init_page_size(inst) != 0)
        return -1;
    strncpy(inst->name, name, SHAREDMEM_NAME_LENGTH - 1);
    /* create named semaphore in locked state */
    sem_unlink(inst->name);
    inst->sem = sem_open(inst->name, O_CREAT|O_EXCL,
        S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP, 0);
    if (inst->sem == NULL)
    {
        set_error(inst, errno, "sharedmem_create(): sem_open()");
        return -1;
    }
    /* create named shared memory */
    shm_unlink(inst->name);
    inst->fd = shm_open(inst->name, O_RDWR|O_CREAT|O_TRUNC, 
        S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP);
    if (inst->fd < 0)
    {
        set_error(inst, errno, "sharedmem_create(): shm_open()");
        goto create_exit;
    }
    /* set the size of the shared memory */
    size_real = round_to_page_boundary(sizeof(sharedmem_ctrl_t)) +
        round_to_page_boundary(size);
    if (ftruncate(inst->fd, size_real) != 0)
    {
        set_error(inst, errno, "sharedmem_create(): ftruncate()");
        goto create_exit;
    }
    /* map it to process address space */
    if (do_maps(inst, size) != 0)
        goto create_exit;

    /* initialize mutex object */
    if (init_mutex(inst) != 0)
        goto create_exit;
    /* initialize condition variable object */
    if (init_cond(inst) != 0)
        goto create_exit;

    /* set size and increase reference count */
    inst->ctrl->size = size;
    inst->ctrl->ref++;

    set_error(inst, 0, NULL);

    /* unlock the semaphore */
    unlock_sem(inst);
    return 0;

create_exit:
    /* unlock the semaphore */
    unlock_sem(inst);
    return -1;
}


int sharedmem_open (sharedmem_t *inst, const char *name)
{
    if (inst == NULL)
        return -1;
    memset(inst, 0x00, sizeof(sharedmem_t));
    inst->fd = -1;

    /* initialize page size */
    if (init_page_size(inst) != 0)
        return -1;
    strncpy(inst->name, name, SHAREDMEM_NAME_LENGTH - 1);
    /* open named semaphore */
    inst->sem = sem_open(inst->name, 0);
    if (inst->sem == NULL)
    {
        set_error(inst, errno, "sharedmem_open(): sem_open()");
        return -1;
    }
    /* lock the semaphore */
    if (!lock_sem(inst))
    {
        set_error(inst, errno, "sharedmem_open(): lock_sem()");
        return -1;
    }
    /* open named shared memory */
    inst->fd = shm_open(inst->name, O_RDWR, 0);
    if (inst->fd < 0)
    {
        set_error(inst, errno, "sharedmem_open(): shm_open()");
        goto open_exit;
    }
    /* map it to process address space */
    if (do_maps(inst, 0) != 0)
        goto open_exit;
    
    /* increase reference count */
    inst->ctrl->ref++;

    set_error(inst, 0, NULL);

    /* unlock the semaphore */
    unlock_sem(inst);
    return 0;

open_exit:
    /* unlock the semaphore */
    unlock_sem(inst);
    return -1;
}


int sharedmem_close (sharedmem_t *inst)
{
    int retval = 0;
    int destruct = 0;

    if (inst == NULL)
        return -1;
    if (inst->ctrl == NULL)
        return -1;

    set_error(inst, 0, NULL);
    /* decrease reference count */
    if (!lock_sem(inst))
    {
        set_error(inst, errno, "sharedmem_close(): lock_sem()");
        retval--;
    }
    inst->ctrl->ref--;
    if (inst->ctrl->ref <= 0)
        destruct = 1;
    if (!unlock_sem(inst))
    {
        set_error(inst, errno, "sharedmem_close(): unlock_sem()");
        retval--;
    }
    if (!destruct)
    {
        /* unmap the shared memory */
        if (undo_maps(inst) != 0)
            retval--;
        if (close(inst->fd) != 0)
        {
            set_error(inst, errno, "sharedmem_close(): close()");
            retval--;
        }
        inst->fd = -1;
        if (sem_close(inst->sem) != 0)
        {
            set_error(inst, errno, "sharedmem_close(): sem_close()");
            retval--;
        }

        if (retval == 0)
            set_error(inst, 0, NULL);
        return retval;
    }

    /* destroy mutex and condition variable objects */
    if (destroy_mutex(inst) != 0)
    {
        set_error(inst, errno, "sharedmem_close(): destroy_mutex()");
        retval--;
    }
    if (destroy_cond(inst) != 0)
    {
        set_error(inst, errno, "sharedmem_close(): destroy_cond()");
        retval--;
    }
    /* unmap the shared memory */
    if (undo_maps(inst) != 0)
        retval--;
    /* close the shared memory */
    if (close(inst->fd) != 0)
    {
        set_error(inst, errno, "sharedmem_close(): close()");
        retval--;
    }
    inst->fd = -1;
    /* remove named shared memory */
    if (shm_unlink(inst->name) != 0)
    {
        set_error(inst, errno, "sharedmem_close(): shm_unlink()");
        retval--;
    }
    /* close the named semaphore */
    if (sem_close(inst->sem) != 0)
    {
        set_error(inst, errno, "sharedmem_close(): sem_close()");
        retval--;
    }
    /* remove the named semaphore */
    if (sem_unlink(inst->name) != 0)
    {
        set_error(inst, errno, "sharedmem_close(): sem_unlink()");
        retval--;
    }

    return retval;    

    if (retval == 0)
        set_error(inst, 0, NULL);
    return retval;
}


int sharedmem_resize (sharedmem_t *inst, size_t size)
{
    size_t size_ctrl;
    size_t size_data;
    size_t size_real;

    if (inst == NULL)
        return -1;
    if (inst->ctrl == NULL)
        return -1;

    /* lock the shared memory object */
    if (!lock_sem(inst))
    {
        set_error(inst, errno, "sharedmem_resize(): lock_sem()");
        return -1;
    }

    /* unmap the data block */
    if (munmap(inst->pdata, round_to_page_boundary(inst->mapped_size)) != 0)
    {
        unlock_sem(inst);
        set_error(inst, errno, "sharedmem_resize(): munmap()");
        return -1;
    }
    inst->mapped_size = 0;
    inst->pdata = NULL;
    /* set the size of the shared memory */
    size_ctrl = round_to_page_boundary(sizeof(sharedmem_ctrl_t));
    size_data = round_to_page_boundary(size);
    size_real = size_ctrl + size_data;
    if (ftruncate(inst->fd, size_real) != 0)
    {
        unlock_sem(inst);
        set_error(inst, errno, "sharedmem_resize(): ftruncate()");
        return -1;
    }
    /* map the new data block */
    inst->pdata = mmap(NULL, size_data, PROT_READ|PROT_WRITE, MAP_SHARED,
        inst->fd, size_ctrl);
    if (inst->pdata == NULL)
    {
        unlock_sem(inst);
        set_error(inst, errno, "sharedmem_resize(): mmap()");
        return -1;
    }
    inst->mapped_size = size;
    /* set new size to shared memory */
    inst->ctrl->size = size;
    
    /* unlock shared memory */
    if (!unlock_sem(inst))
    {
        set_error(inst, errno, "sharedmem_resize(): unlock_sem()");
        return -1;
    }

    set_error(inst, 0, NULL);
    return 0;
}


int sharedmem_resize2 (sharedmem_t *inst)
{
    size_t size_ctrl;
    size_t size_data;

    if (inst == NULL)
        return -1;
    if (inst->ctrl == NULL)
        return -1;

    /* lock the shared memory object */
    if (!lock_sem(inst))
    {
        set_error(inst, errno, "sharedmem_resize2(): lock_sem()");
        return -1;
    }

    /* unmap the data block */
    if (munmap(inst->pdata, round_to_page_boundary(inst->mapped_size)) != 0)
    {
        unlock_sem(inst);
        set_error(inst, errno, "sharedmem_resize2(): munmap()");
        return -1;
    }
    inst->mapped_size = 0;
    inst->pdata = NULL;
    size_ctrl = round_to_page_boundary(sizeof(sharedmem_ctrl_t));
    size_data = round_to_page_boundary(inst->ctrl->size);
    /* map the new data block */
    inst->pdata = mmap(NULL, size_data, PROT_READ|PROT_WRITE, MAP_SHARED,
        inst->fd, size_ctrl);
    if (inst->pdata == NULL)
    {
        unlock_sem(inst);
        set_error(inst, errno, "sharedmem_resize2(): mmap()");
        return -1;
    }
    inst->mapped_size = inst->ctrl->size;
    
    /* unlock shared memory */
    if (!unlock_sem(inst))
    {
        set_error(inst, errno, "sharedmem_resize2(): unlock_sem()");
        return -1;
    }

    set_error(inst, 0, NULL);
    return 0;
}


int sharedmem_size_changed (sharedmem_t *inst)
{
    int res;

    if (inst == NULL)
        return -1;
    if (inst->ctrl == NULL)
        return -1;

    /* check for changes in size */
    res = (inst->mapped_size == inst->ctrl->size) ? 0 : 1;

    return res;
}


int sharedmem_lock (sharedmem_t *inst)
{
    return lock_sem(inst);
}


int sharedmem_unlock (sharedmem_t *inst)
{
    return unlock_sem(inst);
}


int sharedmem_get_error_code (sharedmem_t *inst)
{
    if (inst == NULL)
        return -1;
    
    return inst->error.code;
}


const char * sharedmem_get_error_string (sharedmem_t *inst)
{
    if (inst == NULL)
        return NULL;
    
    return inst->error.string;
}
