/**
    Gnome-VFS module for browsing UPnP AV MediaServers
    
    Copyright 2006 Nokia Corporation. All rights reserved.
	
    Contact: Aapo Makela <aapo.makela@nokia.com>
    
    This library is free software; you can redistribute it and/or modify it
    under the terms of the GNU Lesser General Public License version 2.1 as 
    published by the Free Software Foundation.
  
    This library is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
    General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this library; if not, write to the Free Software Foundation,
    Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#define _XOPEN_SOURCE /* glibc2 needs this */
#include <time.h>
#include <ctype.h>

#include <libgnomevfs/gnome-vfs-module.h>
#include <libgnomevfs/gnome-vfs-module-shared.h>
#include <libgnomevfs/gnome-vfs-mime.h>
#include <libgnomevfs/gnome-vfs-mime-handlers.h>

#ifdef GUPNP_USE_MAEMO
#define DBUS_API_SUBJECT_TO_CHANGE
#include <conic.h>
#include <hildon-mime-patterns.h>
#endif

#include "gnomevfs-upnpav-utils.h"

#include <cybergarage/net/cinterface.h>
#include <cybergarage/contentdirectory/ccdscontrolpoint.h>
#include <cybergarage/avcontrol/cavcontrolpoint.h>
#include <cybergarage/contentdirectory/ccontentdirectory.h>
#include <cybergarage/contentdirectory/cdidllite.h>

#define DEVICE_DIRECTORY_MIME_TYPE	"x-directory/special"
#define NORMAL_DIRECTORY_MIME_TYPE	"x-directory/normal"

#define MAX_ALLOWED_FILE_INFO_TIME	10
#define DEVICE_OBJECT	"0"
#define SEARCH_WAIT	(CG_UPNP_CONTROLPOINT_SSDP_DEFAULT_SEARCH_MX * 2 + 2)
#define NUM_PER_BROWSE	30
#define SUBSCRIPTION_TIMEOUT	1800
#define CDS_SYSTEM_UPDATE_ID "SystemUpdateID"
#define CDS_CONTAINER_UPDATE_IDS "ContainerUpdateIDs"

#define RES_SIZE_PROPERTY	"size"
#define RES_RESOLUTION_PROPERTY	"resolution"
#define RES_PREFERRED_PROTOCOL	"http-get"
#define UPNPAV_METHOD		"upnpav://"
#define HTTP_METHOD		"http://"

#define DATE_FORMAT_ISO8601	"%Y-%m-%d"
#define DATE_FORMAT_DATETIME	"%Y-%m-%dT%H:%M:%S"
#define ST_MEDIASERVER_V1	CG_UPNP_DEVICE_UPNP_ROOTDEVICE 
	/* "urn:schemas-upnp-org:device:MediaServer:1" */

#define CONTAINER_ID_DELIM		"::"

/*----- Internal API -----*/

typedef enum {
	NODE_REMOVE_EVENT = 1 << 0,
	NODE_REMOVE_UPDATE_CACHE = 1 << 1	
} NodeRemoveFlags;

/**
	Tree representing the found mediaservers and content
*/
typedef struct _UPnPAVTree
{
	CgUpnpControlPoint *cp;
	CgSysTime search_complete_date;
	GNode *root;
	gchar *iap_name;
	guint search_source;
	GHashTable *container_cache;
#ifdef USE_HTTP_FILE_INFO
	GHashTable *host_file_info_blacklist;
#endif
	time_t file_info_start;
	gboolean perform_search;
#ifdef GUPNP_USE_MAEMO
	ConIcConnection *conn;
#endif
} UPnPAVTree;


/**
	Removes control point, if it exists. Removes also all device nodes 
	and their children.
	Tree must be locked!
*/
static gboolean upnpav_destroy_controlpoint(void);
/**
	Tries to create UPnP control point.
	Tree must be locked!
*/
static gboolean upnpav_create_controlpoint(void);
/**
	Adds given UPnP MediaServer under the given AV node.
	Tree must be locked!
*/
static gboolean upnpav_get_device_to_node(CgUpnpDevice *device, GNode *node);
/**
 * Returns newly allocated string
 */
static gchar *upnpav_get_extension_from_uri(const gchar *uri);
/**
	Adds the given content nodelist as children for the given parent
	AV node. Overwrites previous children if necessary.
	Tree must be locked!
*/
static gint upnpav_get_content_nodelist_to_node(
					GNode *parent_node,
					CgXmlNodeList* nodelist,
					gboolean overwrite,
					gulong* returned,
					GnomeVFSCancellation *cancellation);
/**
	Destroys the given AV node.
	Tree must be locked!
	
	@param node Node to be destroyed
	@param data Non-null to event changes
*/
static void upnpav_destroy_node(GNode *node,
				gpointer data);
/**
	(Recursively) get path for the given node.
	Tree must be locked!
*/
static gchar *upnpav_get_path_for_node(GNode *node, gchar *lower_part);

/**
	Fill in the file info common for all files accessed via this plugin
*/
static void upnpav_set_global_file_info(GnomeVFSFileInfo* file_info);

#ifdef USE_HTTP_FILE_INFO
/**
	This function resolves metadata, which is not provided  by the media
	server (e.g. file size and modification date).
	Tree must be locked!
	
	@param data	UPnPAVNodeData to be filled.
*/
static GnomeVFSResult upnpav_resolve_missing_metadata(UPnPAVNodeData *data);
#endif

/**
	Determines the best resource for the given AV node data according to 
	the given content node.
	Tree must be locked!
	
	@param data	Node data for the media node
	@param content	XML node describing the node
*/
static gboolean upnpav_determine_resource(UPnPAVNodeData *data, 
					  CgXmlNode *content);

static void upnpav_node_data_destroy(UPnPAVNodeData *node_data);

/**
	Handles a container node change. This means 
	
	1) Rebrowsing the container and recording difference between the
	   direct child (childs added/removed)	   
	2) Eventing for the changed childs
	3) Replacing the original node's childs with the new childs.
	
	@param node		Container node to be checked
	@param recursive	Recursively check the nodes (really heavy!)
	@param cancellation	Cancellation object for cancelling
*/
static void upnpav_node_handle_container_change(GNode *node,
						gboolean recursive);

/**
	Handle container changes with the given CSV string.
	
	@param dev_udn	UDN of the device, which evented change.
	@param csv	(ContainerUpdateIDs value). See the string format in
			ContentDirectory spec.
*/
static void upnpav_node_handle_container_change_by_csv(const gchar *dev_udn, 
						       const gchar *csv);

/**
	Notify monitors about the given event for the given node. 
	Tree must be locked!
*/
static void upnpav_notify_monitor(GNode *node,
				  GnomeVFSMonitorEventType event_type);

/**
	Device listener.
*/
static void upnpav_device_listener(char* udn, CgUpnpDeviceStatus status);

/**

*/
static void upnpav_event_listener(CgUpnpProperty *prop);

#ifdef USE_HTTP_FILE_INFO
static gboolean upnpav_node_update_host_file_info_blacklist(
							UPnPAVNodeData *data);
static gboolean upnpav_node_has_safe_uri(UPnPAVNodeData *data);
#endif

#ifdef GUPNP_USE_MAEMO
/**
	OSSO-IC API callback.
*/
static void upnpav_iap_listener(ConIcConnection *connection,
				ConIcConnectionEvent *event,
				gpointer user_data);
#endif

/*----- Implementation -----*/

/**
	File tree
*/
UPnPAVTree *tree = NULL;

/**
	Locking of file tree for multiple threads
*/
GStaticMutex tree_mutex = G_STATIC_MUTEX_INIT;

/**
	Locking of monitor database for multiple threads
*/
GStaticMutex monitor_mutex = G_STATIC_MUTEX_INIT;

/**
	Database for monitors 
*/
GHashTable *monitor_hash;

/**
	Nodecount
*/
static glong nodecount = 0;

static const struct {
	const char *klass;
	const char *mime_type;
} folder_class_mapping[] = 
	{{"object.container.person.musicArtist", "x-directory/audio"},
	 {"object.container.playlistContainer", "x-directory/audio"},
	 {"object.container.album.musicAlbum", "x-directory/audio"},
	 {"object.container.album.photoAlbum", "x-directory/image"},
	 {"object.container.genre.musicGenre", "x-directory/audio"},
	 {"object.container.genre.movieGenre", "x-directory/video"},
	 {"object.container.storageSystem", NORMAL_DIRECTORY_MIME_TYPE},
	 {"object.container.storageVolume", NORMAL_DIRECTORY_MIME_TYPE},
	 {"object.container.storageFolder", NORMAL_DIRECTORY_MIME_TYPE},
	 {"object.container.musicContainer", "x-directory/audio"},
	 {NULL, NULL}};

	 
GNode *upnpav_init_tree(void)
{
	UPnPAVNodeData *node_data;	
	CgNetworkInterfaceList *interfaces = NULL;
	CgNetworkInterface *interface = NULL;
	
	debug("upnpav_init_tree");
	
	g_static_mutex_lock(&tree_mutex);
	
	tree = g_new0(UPnPAVTree, 1);

	tree->iap_name = NULL;
	tree->cp = NULL;
	tree->file_info_start = 0;
	
	/* Init root directory */
	node_data = g_slice_new0(UPnPAVNodeData);
	node_data->name = g_strdup("/");
	node_data->mime_type = g_strdup(NORMAL_DIRECTORY_MIME_TYPE);
	node_data->container = TRUE;
	node_data->browsed = MAX_BROWSED;
	
	tree->root = g_node_new(node_data);
	nodecount = 1;
	
	g_static_mutex_lock(&monitor_mutex);
	monitor_hash = g_hash_table_new_full ((GHashFunc) gnome_vfs_uri_hash,
					      (GEqualFunc) gnome_vfs_uri_equal,
					      (GDestroyNotify) gnome_vfs_uri_unref,
					      (GDestroyNotify) NULL);
	g_static_mutex_unlock(&monitor_mutex);
	
	tree->container_cache = g_hash_table_new_full(g_str_hash, g_str_equal,
						      g_free, NULL);
#ifdef USE_HTTP_FILE_INFO
	tree->host_file_info_blacklist = g_hash_table_new_full(g_str_hash,
							       g_str_equal,
							       g_free, NULL);
#endif
	g_static_mutex_unlock(&tree_mutex);
	
#ifdef GUPNP_USE_MAEMO
	/* Initialize internet connection */
	debug("Initializing IC");
	tree->conn = con_ic_connection_new ();
	g_assert(tree->conn);
	
	g_signal_connect(G_OBJECT(tree->conn),
			 "connection-event",
			 G_CALLBACK(upnpav_iap_listener),
			 NULL);
	g_object_set(G_OBJECT(tree->conn), 
		     "automatic-connection-events", TRUE,
		     NULL);
#endif	
	/* Start control point right away only, if we have eth0 or wlan0 up */
	interfaces = cg_net_interfacelist_new();
	if (interfaces != NULL &&
	    cg_net_gethostinterfaces(interfaces) > 0 &&
	    ((interface = cg_net_interfacelist_get(interfaces, "eth0")) 
		!= NULL ||
	     (interface = cg_net_interfacelist_get(interfaces, "wlan0"))
		!= NULL))
	{
		debug("Interface %s address: %s", 
			cg_net_interface_getname(interface),
			cg_net_interface_getaddress(interface));
		g_static_mutex_lock(&tree_mutex);
		upnpav_create_controlpoint();
		g_static_mutex_unlock(&tree_mutex);
	}
	
	if (interfaces != NULL) cg_net_interfacelist_delete(interfaces);
	
	return tree->root;
}


void upnpav_free_tree(void)
{
	UPnPAVNodeData *node_data;
	
	debug("upnpav_free_tree");
	
	g_static_mutex_lock(&tree_mutex);
	
	if (tree == NULL)
	{
		g_static_mutex_unlock(&tree_mutex);
		return;
	}
	
	if (tree->search_source > 0)
	{
		g_source_remove(tree->search_source);
		tree->search_source = 0;
	}
	
	/* Stop and delete UPnP control point */
	upnpav_destroy_controlpoint();
	
	/* Remove root */
	if (tree->root != NULL)
	{
		node_data = (UPnPAVNodeData*)tree->root->data;
		
		if (node_data != NULL)
		{
			g_free(node_data->name);
			g_slice_free(UPnPAVNodeData, node_data);
		}
		
		g_node_destroy(tree->root);
		nodecount--;
		debug("Nodecount: %ld", nodecount);
		
		tree->root = NULL;
	}
	
	g_hash_table_destroy(tree->container_cache);
	tree->container_cache = NULL;
	
#ifdef USE_HTTP_FILE_INFO
	g_hash_table_destroy(tree->host_file_info_blacklist);
	tree->host_file_info_blacklist = NULL;
#endif
	
	g_static_mutex_lock(&monitor_mutex);
	if (monitor_hash != NULL)
	{
		g_hash_table_destroy(monitor_hash);
		monitor_hash = NULL;
	}
	g_static_mutex_unlock(&monitor_mutex);
	
#ifdef GUPNP_USE_MAEMO
	if (tree->conn != NULL)
	{
		if (tree->iap_name != NULL)
		{
			g_free(tree->iap_name);
			tree->iap_name = NULL;
		}	
		g_object_unref(G_OBJECT(tree->conn));
		tree->conn = NULL;
	}
#endif
	
	g_static_mutex_unlock(&tree_mutex);
	
	return;
}


static gboolean upnpav_delayed_search(gpointer user_data)
{
	g_static_mutex_lock(&tree_mutex);
	tree->search_source = 0;
	debug("Performing another M-SEARCH..");
	if (tree->cp != NULL)
		cg_upnp_controlpoint_search(tree->cp, 
					    ST_MEDIASERVER_V1);
	g_static_mutex_unlock(&tree_mutex);
	
	return FALSE;
}


static gboolean upnpav_destroy_controlpoint_idle(gpointer cp_ref)
{
	CgUpnpControlPoint *ctrlPoint = (CgUpnpControlPoint *)cp_ref;
	
	debug("Stopping UPnP control point..");
	cg_upnp_controlpoint_stop(ctrlPoint);
	debug("Deleting UPnP control point..");
	cg_upnp_controlpoint_delete(ctrlPoint);
	
	return FALSE;
}


static gboolean upnpav_destroy_controlpoint(void)
{
	GNode *child = NULL, *next_child = NULL;
	
	/* Remove listener's from control point */
	if (tree->cp != NULL) 
	{
		/* TODO: Here we should check if we have an IAP name and do
		   unsubscribe to all devices, if we have */
		debug("Removing UPnP listeners..");
			
		cg_upnp_controlpoint_lock(tree->cp);
		cg_upnp_controlpoint_setdevicelistener(
					tree->cp,
					NULL);
		cg_upnp_controlpoint_removeeventlistener(
					tree->cp,
					upnpav_event_listener);
		cg_upnp_controlpoint_unlock(tree->cp);
	}
		
	/* Remove all device nodes */
	child = g_node_first_child(tree->root);
	while (child != NULL)
	{
		next_child = g_node_next_sibling(child);
		upnpav_destroy_node(child, GINT_TO_POINTER(
			NODE_REMOVE_EVENT | NODE_REMOVE_UPDATE_CACHE) );
		child = next_child;
	}
	
	/* Stop and delete UPnP control point */
	if (tree->cp != NULL) {
		/* Control point must not be locked, when stopped
		   The most reliable way to stop it is without tree_mutex
		   (as we may still have handlers ongoing), so we install
		   idle handler, which will take care of the deletion.
		*/
		g_idle_add(upnpav_destroy_controlpoint_idle, tree->cp);
		tree->cp = NULL;
	}
		
	return TRUE;
}


static gboolean upnpav_create_controlpoint(void)
{
	gboolean result = FALSE;
	
	if (tree->cp != NULL)
	{
		/* Control point creation was requested but we have already a
		   control point. Signal that we might have changed 
		   IP addresses */
		debug("Check control point IP addresses");
		cg_upnp_controlpoint_ipchanged(tree->cp);
		return TRUE;
	}
	
	/* Create new control point */
	tree->cp = cg_upnp_controlpoint_new();
	
	if (tree->cp != NULL) {
		debug("Starting control point..");
		
		/* Start the control point */
		if (cg_upnp_controlpoint_start(tree->cp) == FALSE)
		{
			debug("Unable to create control point!");
			
			/* Unable to start the control point, assume
			   offline mode, and remove cp */
			cg_upnp_controlpoint_delete(tree->cp);
			tree->cp = NULL;
			
			debug("Control point removed.");
		} else {
			/* start searching for devices */
			cg_upnp_controlpoint_lock(tree->cp);
			cg_upnp_controlpoint_setdevicelistener(
					tree->cp,
					upnpav_device_listener);
			
			cg_upnp_controlpoint_addeventlistener(
					tree->cp,
					upnpav_event_listener);
			
			cg_upnp_controlpoint_unlock(tree->cp);
			result = TRUE;
			tree->perform_search = TRUE;
		}
	
	} 
	
	return result;
}


#ifdef USE_HTTP_FILE_INFO
static gboolean upnpav_node_update_host_file_info_blacklist(
							UPnPAVNodeData *data)
{
	if (tree->file_info_start == 0) return FALSE;
	
	/* Check if host has already been blacklisted */
	if (g_hash_table_lookup_extended(tree->host_file_info_blacklist,
					 data->dev_udn, NULL, NULL) == TRUE) 
		return FALSE;
	
	if ((data->size == 0 && data->mtime == 0) ||
	    (time(NULL) - tree->file_info_start) > MAX_ALLOWED_FILE_INFO_TIME)
	{
		/* It took more than MAX_ALLOWED_FILE_INFO_TIME to resolve
		   file info for a host URL or the host did not give as the
		   info, we need => blacklist the host */
		g_hash_table_insert(tree->host_file_info_blacklist,
				    g_strdup(data->dev_udn), NULL);
		tree->file_info_start = 0;
		error("Device %s got blacklisted for resolving file info", 
		      data->dev_udn);
		return TRUE;
	}
	
	tree->file_info_start = 0;
	return FALSE;
}


static gboolean upnpav_node_has_safe_uri(UPnPAVNodeData *data)
{
	CgUpnpDevice *dev = NULL;
	GnomeVFSURI *dev_uri = NULL, *data_uri = NULL;
	
	/* Sanity checks */
	if (data->uri == NULL) return FALSE;
	if (data->dev_udn == NULL) return FALSE;
	
	/* Check if host has already been blacklisted */
	if (g_hash_table_lookup_extended(tree->host_file_info_blacklist,
					 data->dev_udn, NULL, NULL) == TRUE) 
		return FALSE;
	
	cg_upnp_controlpoint_lock(tree->cp);
	dev = cg_upnp_controlpoint_getdevicebyudn(tree->cp, data->dev_udn);
	dev = cg_upnp_device_getrootdevice(dev);
	if (dev != NULL && 
	    cg_upnp_device_getlocationfromssdppacket(dev) != NULL)
	{
		/* At least node has corresponding device, get device URI */
		data_uri = gnome_vfs_uri_new(data->uri);
		dev_uri  = gnome_vfs_uri_new(
			cg_upnp_device_getlocationfromssdppacket(dev));
		
		if (data_uri != NULL && dev_uri != NULL &&
		    gnome_vfs_uri_get_host_name(data_uri) != NULL && 
		    gnome_vfs_uri_get_host_name(dev_uri) != NULL &&
		    strcmp(gnome_vfs_uri_get_host_name(data_uri),
			   gnome_vfs_uri_get_host_name(dev_uri)) == 0 &&
		    gnome_vfs_uri_get_host_port(data_uri) == 
			gnome_vfs_uri_get_host_port(dev_uri))
		{
			/* device host and port match to resource host and 
			   port - assume that URI is safe */
			gnome_vfs_uri_unref(dev_uri); dev_uri = NULL;
			gnome_vfs_uri_unref(data_uri); data_uri = NULL;
			cg_upnp_controlpoint_unlock(tree->cp);
			
			/* Get timing, how much it will take to resolve 
			   file info for this host */
			tree->file_info_start = time(NULL);
			return TRUE;
		}
		
		if (dev_uri) gnome_vfs_uri_unref(dev_uri); 
		dev_uri = NULL;
		
		if (data_uri) gnome_vfs_uri_unref(data_uri); 
		data_uri = NULL;
	}
	cg_upnp_controlpoint_unlock(tree->cp);
	
	/* We got this far, assume that URI is not safe */
	return FALSE;
}
#endif /* USE_HTTP_FILE_INFO */


GnomeVFSResult upnpav_determine_fileinfo(UPnPAVNodeData *data, GnomeVFSFileInfo* file_info,
				      GnomeVFSFileInfoOptions *options)
{	
	GnomeVFSResult result = GNOME_VFS_OK;
	
	g_return_val_if_fail(data != NULL, GNOME_VFS_ERROR_NOT_FOUND);
	
	file_info->valid_fields = GNOME_VFS_FILE_INFO_FIELDS_NONE;
	upnpav_set_global_file_info(file_info);
	
	if (data->container == TRUE) 
	{
		file_info->type = GNOME_VFS_FILE_TYPE_DIRECTORY;
		file_info->valid_fields |= 
			GNOME_VFS_FILE_INFO_FIELDS_TYPE;
#ifndef USE_EXTENDED_DIRECTORY_TYPES
		file_info->mime_type = g_strdup (NORMAL_DIRECTORY_MIME_TYPE);
#else
		file_info->mime_type = g_strdup(data->mime_type);
#endif
		file_info->valid_fields |= 
			GNOME_VFS_FILE_INFO_FIELDS_MIME_TYPE;
		
		file_info->permissions |= GNOME_VFS_PERM_ACCESS_EXECUTABLE |
					  GNOME_VFS_PERM_USER_EXEC |
					  GNOME_VFS_PERM_GROUP_EXEC |
					  GNOME_VFS_PERM_OTHER_EXEC;
		
		file_info->size = 0;
		file_info->valid_fields |= 
			GNOME_VFS_FILE_INFO_FIELDS_SIZE;
	} else {
		file_info->type = GNOME_VFS_FILE_TYPE_REGULAR;
		file_info->valid_fields |= 
			GNOME_VFS_FILE_INFO_FIELDS_TYPE;
		
#ifdef USE_HTTP_FILE_INFO
		/* Fill in missing info, if required */
		if (data->size == 0)
			result = upnpav_resolve_missing_metadata(data);
#endif		
		
		if (data->size > 0)
		{
			file_info->size = data->size;
			file_info->valid_fields |= 
				GNOME_VFS_FILE_INFO_FIELDS_SIZE;
		}
		
		/* Determine file_info mime-type according to URI/content */
		if (data->uri != NULL)
		{
#ifdef MODEL_AS_SYMLINKS
			file_info->type = GNOME_VFS_FILE_TYPE_SYMBOLIC_LINK;
			GNOME_VFS_FILE_INFO_SET_SYMLINK(file_info, TRUE);
			file_info->valid_fields |= GNOME_VFS_FILE_INFO_FIELDS_FLAGS;
			
#ifdef USE_HTTP_FILE_INFO
			if (((*options) & GNOME_VFS_FILE_INFO_FOLLOW_LINKS) > 0 &&
			    upnpav_node_has_safe_uri(data))
			{
				debug("Following link %s", data->uri);
				result = gnome_vfs_get_file_info(data->uri,
								 file_info,
							         *options);
				upnpav_node_update_host_file_info_blacklist(
					data);
			}
#endif			
			file_info->link_count = 1;
			file_info->valid_fields |=
				GNOME_VFS_FILE_INFO_FIELDS_LINK_COUNT;
			
			file_info->symlink_name = g_strdup(data->uri);
			file_info->valid_fields |= 
				GNOME_VFS_FILE_INFO_FIELDS_SYMLINK_NAME;
#endif
		}
		
		/*
		    Some MediaServers report completely wrong MIME, but it
		    should be fixed in determine_resource.
		*/
		if (data->mime_type != NULL && file_info->mime_type == NULL)
		{
			file_info->mime_type = g_strdup (data->mime_type);
			file_info->valid_fields |= 
				GNOME_VFS_FILE_INFO_FIELDS_MIME_TYPE;
		}
	}
	
	/* Set modification date */
	if (data->mtime > 0)
	{
		file_info->mtime = data->mtime;
		file_info->ctime = data->mtime;
		
		file_info->valid_fields |= GNOME_VFS_FILE_INFO_FIELDS_MTIME;
		file_info->valid_fields |= GNOME_VFS_FILE_INFO_FIELDS_CTIME;
	}
	
	file_info->name = g_strdup(data->name);
	
	return result;
}


static void upnpav_set_global_file_info(GnomeVFSFileInfo* file_info)
{
	file_info->permissions = GNOME_VFS_PERM_ACCESS_READABLE |
				 GNOME_VFS_PERM_USER_READ |
				 GNOME_VFS_PERM_GROUP_READ |
				 GNOME_VFS_PERM_OTHER_READ;

	file_info->valid_fields |= 
		GNOME_VFS_FILE_INFO_FIELDS_PERMISSIONS;
}


#ifdef USE_HTTP_FILE_INFO
static GnomeVFSResult upnpav_resolve_missing_metadata(UPnPAVNodeData *data)
{
	/* TODO: There is many different possibilities to get the desired
	   metadata:
		1) Make manually HTTP HEAD request with clinkc stack
			=> Pros: we could benefit from the already open
			   HTTP connection
			=> Cons: more difficult to implement (require 
			   header parsing etc.)
		2) Query file info through GnomeVFS from HTTP URI
			=> Pros: easy to implement.
			=> Cons: Requires new HTTP connection for at least
			   1st file. There exists 2 HTTP connections
			   (clinkc stack and GnomeVFS)
	  I'll go with 2) for now.
	*/
	GnomeVFSFileInfo *file_info = NULL;
	GnomeVFSResult result = GNOME_VFS_OK;
	
	if (!upnpav_node_has_safe_uri(data))
	{
		debug("URI %s is not safe, not resolving missing metadata",
		      data->uri);
		return GNOME_VFS_OK;
	}
	
	file_info = gnome_vfs_file_info_new();
	result = gnome_vfs_get_file_info(data->uri, file_info,
					 GNOME_VFS_FILE_INFO_DEFAULT);
	
	if (result == GNOME_VFS_OK)
	{
		/* Resolve size */
		if (data->size == 0 &&
		    (file_info->valid_fields & 
		     GNOME_VFS_FILE_INFO_FIELDS_SIZE) > 0)
		{
			data->size = file_info->size;
		}
		
		/* Resolve modification time */
		if (data->mtime == 0 &&
		    ((file_info->valid_fields & 
		      GNOME_VFS_FILE_INFO_FIELDS_MTIME) > 0 ||
		     (file_info->valid_fields & 
		      GNOME_VFS_FILE_INFO_FIELDS_CTIME) > 0))
		{
			data->mtime = (file_info->valid_fields & 
				GNOME_VFS_FILE_INFO_FIELDS_MTIME) > 0 ?
					file_info->mtime :
					file_info->ctime;
		}
	} else {
		error("Unable to get file info for URI %s : %s", 
		      data->uri, gnome_vfs_result_to_string(result));
	}
	
	gnome_vfs_file_info_unref(file_info); file_info = NULL;
	upnpav_node_update_host_file_info_blacklist(data);
	return result;
}
#endif


gboolean upnpav_get_content_directory(GNode *node, 
				      GnomeVFSCancellation *cancellation,
				      gint *upnp_status)
{
	UPnPAVNodeData *node_data = NULL;
	UPnPAVNodeData *device_node_data = NULL;
	GNode *device_node = NULL;
	
	gchar *object_id = NULL, *browse_flag = NULL, *filter = NULL;
	gchar *starting_index = NULL, *requested_count = NULL;
	gchar *sort_criteria = NULL;
	gchar *browse_result = NULL;
	gchar *error_string = NULL;
	gchar *number_returned = NULL;
	gchar *total_matches = NULL;
	gchar *update_id = NULL;
	gchar *cache_key = NULL;
	
	glong update_id_value = 0;
	CgXmlNodeList* nodelist = NULL;
	CgUpnpDevice *device = NULL;
	CgUpnpService *service = NULL;
	gboolean bool_result = TRUE;
	gint upnp_status_code = 0;
	gulong returned = 0;
	
	/* This fails automatically, if there is no control point */
	if ( tree->cp == NULL ) return FALSE;
	g_return_val_if_fail(node != NULL, FALSE);
	
	node_data = (UPnPAVNodeData*)node->data;
	g_return_val_if_fail(node_data != NULL, FALSE);
	g_return_val_if_fail(node_data->container == TRUE, FALSE);
	
	/* Everything has already been browsed, return TRUE */
	if (node_data->browsed == MAX_BROWSED) return TRUE;
	
	cg_upnp_controlpoint_lock(tree->cp);
	device = cg_upnp_controlpoint_getdevicebyudn(tree->cp, 
						     node_data->dev_udn);
	
	if (device == NULL)
	{
		error("Unable to find device for node '%s'", node_data->name);
		cg_upnp_controlpoint_unlock(tree->cp);
		return FALSE;
	}
	
	/* Search for the device node corresponding the browsed node */
	device_node = g_node_first_child(tree->root);
	while (device_node != NULL)
	{
		device_node_data = (UPnPAVNodeData*)device_node->data;
		if (device_node_data != NULL && 
		    device_node_data->dev_udn != NULL &&
		    strcmp(node_data->dev_udn, 
			   device_node_data->dev_udn) == 0)
		{
			/* Device node was found */
			break;
		}
		
		device_node = g_node_next_sibling(device_node);
	}
	
	if ( device_node == NULL || device_node_data == NULL )
	{
		error("Unable to find device node for node '%s'", 
		      node_data->name);
		cg_upnp_controlpoint_unlock(tree->cp);
		return FALSE;
	}
	
	/* Subscribe to service, if not yet subscribed */
	if ( device_node_data->uri == NULL )
	{
		service = cg_upnp_device_getservicebytype(
						device, 
						CG_UPNP_CDS_SERVICE_TYPE);
		
		if (cg_upnp_controlpoint_subscribe(tree->cp, service, 
						   SUBSCRIPTION_TIMEOUT) &&
		    cg_upnp_service_getsubscriptionsid(service) != NULL)
		{
			/* Save subscription ID as URI */
			device_node_data->uri = 
			  g_strdup(
			    cg_upnp_service_getsubscriptionsid(service));
			debug("Subscribed to service %s with SSID %s", 
				cg_upnp_service_getserviceid(service), 
				cg_upnp_service_getsubscriptionsid(service));
		} else {
			error("Subscription to %s::%s failed!\n",
			      device_node_data->name,
			      cg_upnp_service_getserviceid(service));
			device_node_data->uri = g_strdup("");
		}
	}
	
	/* Browse arguments */
	
	/* Object ID */
	object_id = node_data->object_id;
	browse_flag = CG_UPNP_AV_CDS_BROWSE_FLAG_DIRECTCHILDREN;
	filter = CG_UPNP_AV_CDS_BROWSE_FILTER_ALL;
	sort_criteria = CG_UPNP_AV_CDS_BROWSE_SORTCRITERIA_NONE;
	
	starting_index = g_strdup_printf("%lu", node_data->browsed);
	if (starting_index == NULL)
	{
		cg_upnp_controlpoint_unlock(tree->cp);
		return FALSE;
	}
	
	requested_count = g_strdup_printf("%d", NUM_PER_BROWSE);
	if (requested_count == NULL)
	{
		g_free(starting_index);
		cg_upnp_controlpoint_unlock(tree->cp);
		return FALSE;
	}

#ifdef USE_CANCELLATION
	/* Check cancellation */
	if (cancellation != NULL &&
	    gnome_vfs_cancellation_check(cancellation) == TRUE)
	{
		g_free(starting_index);
		g_free(requested_count);
		cg_upnp_controlpoint_unlock(tree->cp);
		error("Browse operation was cancelled");
		return FALSE;
	}
#endif
	
	debug("Browse(%s, %s, %s, %s, %s, %s, %s)",
	      cg_upnp_device_getudn(device), object_id, browse_flag,
	      filter, starting_index, requested_count, sort_criteria);
	
	/* Get directory structure */
	upnp_status_code = cg_upnp_av_cds_control_browse(
                device,
                &error_string,
                &object_id,
                &browse_flag,
                &filter,
                &starting_index,
                &requested_count,
                &sort_criteria,
                &browse_result,
                &number_returned,
                &total_matches,
                &update_id);
	
	if ( upnp_status != NULL ) *upnp_status = upnp_status_code;
	
	/* Free allocated arguments */
	g_free(requested_count); requested_count = NULL;
	g_free(starting_index);  starting_index = NULL;
	g_free(number_returned); number_returned = NULL;
	g_free(total_matches);  total_matches = NULL;
	
	if (update_id != NULL)
	{
		update_id_value = cg_str2long(update_id);
		g_free(update_id); update_id = NULL;
	}
	
	if (node_data->browsed == 0 && update_id != NULL &&
	    update_id_value == node_data->update_id)
	{
		/* We start browsing from beginning, but have already 
		   update_id assigned and it matches => don't have to 
		   browse */
		debug("Browsing from start and update id matches: "
		      "%ld == %ld", update_id_value, node_data->update_id );
		g_free(error_string); error_string = NULL;
		cg_upnp_controlpoint_unlock(tree->cp);
		return FALSE;
	}
	
	/* Update update ID for all other objects but not device objects */
	if (node_data->object_id != NULL &&
	    strcmp(node_data->object_id, DEVICE_OBJECT) != 0)
	{
		node_data->update_id = update_id_value;
	}
	
	if (error_string != NULL) 
	{
		error("Browse error: %s", error_string);
		g_free(error_string); error_string = NULL;
	}
	
	cg_upnp_controlpoint_unlock(tree->cp);
	
	/* Add this container to container cache for quick retrieval */
	cache_key = g_strconcat(node_data->dev_udn, CONTAINER_ID_DELIM, 
				node_data->object_id, NULL);
	if (g_hash_table_lookup(tree->container_cache, cache_key) == NULL)
	{
		g_hash_table_insert(tree->container_cache, cache_key, node);
	} else {
		g_free(cache_key);
	}
	cache_key = NULL;
	
	if (cg_upnp_av_control_iserrorcodesuccessful(upnp_status_code) &&
	    browse_result != NULL)
	{
		nodelist = cg_xml_nodelist_new();
		if (nodelist == NULL || 
		    cg_upnp_av_cds_create_cg_xml(browse_result, nodelist) == FALSE)
		{
			error("Unable to parse browse result XML");
			debug(browse_result);
			bool_result = FALSE;
		}
		
	} else {
		if (browse_result != NULL)
			error("UPnP error %d occured", upnp_status_code);
		else
			error("No browse results");
		bool_result = FALSE;
		
		/* Mark as browsed here */
		node_data->browsed = MAX_BROWSED;
	}
	
	/* Get the contents of the individual CDS */
	if (nodelist != NULL && bool_result == TRUE)
	{
		bool_result = upnpav_get_content_nodelist_to_node(
							node, nodelist, 
							FALSE, &returned,
							cancellation);	
		if (bool_result == TRUE)
		{
			debug("%ld nodes were returned for browse", returned);
			/* Check how many we got */
			if (returned < NUM_PER_BROWSE)
			{
				/* Rest was returned, mark as browsed */
				node_data->browsed = MAX_BROWSED;
			} else {
				node_data->browsed += returned;
			}
		} else {
			error("Converting XML nodes to item nodes failed");
		}
	}
		
	/* We don't need this anymore */
	g_free(browse_result); browse_result = NULL;
	
	/* We don't need this anymore */
	if (nodelist != NULL)
		cg_xml_nodelist_delete(nodelist);
	nodelist = NULL;
	
	debug("Nodecount: %ld", nodecount);
	
	return bool_result;
}


static gboolean upnpav_is_unique_name(GNode *parent_node,
				      const gchar *name)
{
	GNode *node = NULL;
	
	g_return_val_if_fail(parent_node != NULL, FALSE);
	
	for (node =  g_node_first_child(parent_node);
	     node != NULL;
	     node = g_node_next_sibling(node))
	{
		if (strcmp(((UPnPAVNodeData*)node->data)->name, name) == 0)
		{
			/* The name matches with existing node,return FALSE */
			return FALSE;
		}
	}
	
	return TRUE;
}


static gchar *upnpav_get_extension_from_uri(const gchar *uri)
{
	gchar *start = NULL, *end = NULL;
	gchar *extension = NULL;
	
	if (uri == NULL) return NULL;
	
	start = g_strrstr(uri, "/");
	end = g_strrstr(uri, "?");
			
	if (start != NULL)
	{
		/* Start was found, try to find end */
		if (end == NULL) end = g_strrstr(uri, "#");
				
		if (end != NULL)
		{
			/* End was found, get extension 
			   from the middle */
			start = g_strrstr_len(start, end-start, ".");
			
			if (start != NULL && (end-start) > 0)
				extension = g_strndup(start, end-start);
		} else {
			/* End was not found, get extension from the end */
			start = g_strrstr(start, ".");
			if (start != NULL)
				extension = g_strdup(start);
		}
	}
	
	return extension;
}


static gchar *upnpav_get_extension_from_mime_type(const gchar *mime_type)
{
	struct _upnpav_extension_guess {
		const gchar *mime;
		const gchar *extension;
	} ext_guess_map[] = {
		{ .mime = "audio/mpeg", .extension = ".mp3" },
		{ .mime = "audio/mp3", .extension = ".mp3" },
		{ .mime = "audio/x-mp3", .extension = ".mp3" },
		{ .mime = "audio/x-ms-wma", .extension = ".wma" },
		{ .mime = "audio/x-ms-wmv", .extension = ".wmv" },
		{ .mime = "audio/wav", .extension = ".wav" },
		{ .mime = "audio/x-wav", .extension = ".wav" },
		{ .mime = "audio/mpegurl", .extension = ".m3u" },
		{ .mime = "audio/x-scpls", .extension = ".pls" },
		{ .mime = "audio/x-ms-asf", .extension = ".asf" },
		{ .mime = "audio/ogg", .extension = ".ogg" },
		{ .mime = "video/ogg", .extension = ".ogv" },
		{ .mime = "video/avi", .extension = ".avi" },
		{ .mime = "video/mpeg", .extension = ".mpg" },
		{ .mime = NULL, .extension = NULL }
	};
	gchar *extension = NULL;
	gint idx = 0;
#ifdef GUPNP_USE_MAEMO
	GSList *patterns = hildon_mime_patterns_get_for_mime_type(mime_type,
								 NULL);
	gchar *start = NULL;
	
	if (patterns != NULL)
	{
		extension = (gchar *)patterns->data;
		start = strstr(extension, ".");
		
		if (start != NULL)
			extension = g_strdup(start);
		else
			extension = NULL;
		
		g_slist_foreach (patterns, (GFunc) g_free, NULL);
		g_slist_free (patterns); patterns = NULL;
	}
#endif
	
	if (!extension)
	{
		/* Last resolt: try to guess correct extension with table */
		for (idx = 0; ext_guess_map[idx].mime != NULL; idx++)
		{
			if (strcmp(ext_guess_map[idx].mime, mime_type) == 0)
			{
				extension = g_strdup(
					ext_guess_map[idx].extension);
				break;
			}
		}
	}
	
	return extension;
}


static gboolean upnpav_get_content_nodelist_to_node(
					GNode *parent_node,
					CgXmlNodeList* nodelist,
					gboolean overwrite,
					gulong* returned,
					GnomeVFSCancellation *cancellation)
{
	UPnPAVNodeData *node_data = NULL, *parent_node_data = NULL;
	CgXmlNode *node = NULL;
	CgXmlNodeList* children = NULL;
	gchar *extension = NULL;
	
	g_return_val_if_fail(parent_node != NULL, FALSE);
	g_return_val_if_fail(nodelist != NULL, FALSE);
	
	parent_node_data = (UPnPAVNodeData*)parent_node->data;
	g_return_val_if_fail(parent_node_data != NULL, FALSE);
	g_return_val_if_fail(parent_node_data->container == TRUE, FALSE);
	
	if (returned != NULL) (*returned) = 0;
		
	/* Check that this is a valid DIDL-Lite document */
	node = cg_upnp_av_cds_didllite_nodelist_gets(nodelist);
	
	if (cg_upnp_av_cds_didllite_node_getname(node) != NULL &&
	    cg_strcmp(cg_upnp_av_cds_didllite_node_getname(node),
		      DIDL_LITE_NAME) == 0)
	{
		children = cg_upnp_av_cds_didllite_node_getchildnodelist(node);
		if (children == NULL)
		{
			error("No children found in DIDL-Lite document");
			return FALSE;
		}
	} else {
		error("The browse result is not a valid DIDL-Lite document!");
		return FALSE;
	}
	
	/* Remove old children */
	if (overwrite)
	{
		g_node_children_foreach (
				parent_node,
				G_TRAVERSE_ALL,
				(GNodeForeachFunc)upnpav_destroy_node,
				GINT_TO_POINTER(NODE_REMOVE_UPDATE_CACHE));
	}
	
	/* Put the result to the tree */
	for (node = cg_upnp_av_cds_didllite_nodelist_gets(children);
	     node != NULL;
	     node = cg_upnp_av_cds_didllite_node_next(node))
	{
		if (returned != NULL)
		{
			/* Update returned value already here as we might
			   discard the node later */
			(*returned)++;
		}
		
#ifdef USE_CANCELLATION
		/* Check cancellation */
		if (cancellation != NULL && 
		    gnome_vfs_cancellation_check(cancellation) == TRUE)
		{
			error("Browse operation was cancelled when creating "
			      "nodes");
			return FALSE;
		}
#endif
		
		if (!(cg_upnp_av_cds_didllite_node_isname(node,
							  DIDL_LITE_ITEM) ||
		      cg_upnp_av_cds_didllite_node_isname(node,
							DIDL_LITE_CONTAINER)))
		{
			/* The node is not item nor container
			    => We're not interested */
			continue;
		}
		
		node_data = g_slice_new0(UPnPAVNodeData);
		node_data->container = cg_upnp_av_cds_didllite_node_isname(
						node, DIDL_LITE_CONTAINER);
				
		if (node_data == NULL ||
		    !upnpav_determine_resource(node_data, node))
		{
			/* There wasn't compatible resource for this file,
			   ignore it */
			g_slice_free(UPnPAVNodeData, node_data);
			node_data = NULL;
			continue;
		}
		
		node_data->name = cg_xml_node_getchildnodevalue(node, DIDL_LITE_DC_TITLE);
		
		/* Some mediaservers do not return title - isn't that illegal?! */
		if (node_data->name == NULL || 
		    strcmp(node_data->name, "") == 0)
		{
			/* Name was illegal, continue with next one */
			g_slice_free(UPnPAVNodeData, node_data);
			node_data = NULL;
			continue;
		}
				
		node_data->object_id = g_strdup(
				cg_upnp_av_cds_didllite_node_getattribute(
					node, DIDL_LITE_ITEM_ID));
		node_data->dev_udn = parent_node_data->dev_udn;
		node_data->browsed = 0;
		node_data->update_id = -1;
		
#ifdef MODEL_AS_PLAYLISTS
		if (node_data->container == TRUE)
		{
			node_data->name = g_strdup(node_data->name);
		} else {
			node_data->name = g_strconcat(node_data->name, ".m3u", NULL);
		}
#else
		/* Check that node name has the right extension */
		if (node_data->container == TRUE)
		{
			/* Node is directory => no need for extension */
			node_data->name = g_strdup(node_data->name);
			
		} else {
			/* Get extension from the URI */
			extension = upnpav_get_extension_from_uri(node_data->uri);
			
			/* Get extension from MIME, if it is not possible to
			   get it from URI */
			if (extension == NULL) 
				extension = 
					upnpav_get_extension_from_mime_type(
							node_data->mime_type);
			
			if (extension != NULL)
			{
				/* Extension was found, concatenate it to name, if
				   it isn't found there already */
				if (!g_str_has_suffix(node_data->name, extension))
					node_data->name = g_strconcat(node_data->name, 
								      extension, 
								      NULL);
				else
					node_data->name = g_strdup(node_data->name);
				
				g_free(extension); extension = NULL;
			} else {
				/* Extension was not found, just copy name */
				error("No extension found for %s",
				      node_data->name);
				node_data->name = g_strdup(node_data->name);
			}
		}
#endif			
		if (node_data->name == NULL || node_data->object_id == NULL ||
		    !upnpav_is_unique_name(parent_node, node_data->name))
		{
			/* We are missing either name or object id - bail out */
			g_free(node_data->name);
			g_free(node_data->object_id);
			g_free(node_data->uri);
			g_free(node_data->mime_type);
			g_slice_free(UPnPAVNodeData, node_data); 
			node_data = NULL;
			continue;
		}
		
		/* Append node */
		if (g_node_append_data(parent_node, node_data) != NULL)
		{
			nodecount++;
		}
	}
	
	return TRUE;
}


static gboolean upnpav_is_mime_type_supported(const gchar *mime_type)
{
	GnomeVFSMimeApplication* app = NULL;
	
	if (mime_type == NULL) return FALSE;
	
	app = gnome_vfs_mime_get_default_application(mime_type);
	if (app != NULL) 
	{
		gnome_vfs_mime_application_free(app);
		return TRUE;
	}
	return FALSE;
}


static gboolean upnpav_determine_resource(UPnPAVNodeData *data, CgXmlNode *content)
{
	CgXmlNodeList* nodelist = NULL;
	CgXmlNode *prop = NULL, *best = NULL;
	CgXmlAttribute *attr = NULL;
	gboolean result = FALSE;
	gchar *protocol_info = NULL;
	gchar *protocol = NULL;
	gchar *mime_type = NULL;
	gchar **list = NULL;
	gchar *klass = NULL;
	gchar *date = NULL;
	struct tm tm;
	
	gint idx = 0;
	gint best_choice = 0;
	gint good_choice = 0;
	
	if (data->container == TRUE)
	{
		/* This node is a container and only MIME type must be
		   determined by checking class */
		klass = cg_xml_node_getchildnodevalue(content, DIDL_LITE_UPNP_CLASS);
		mime_type = NORMAL_DIRECTORY_MIME_TYPE;
		
		for (idx = 0; 
		     klass != NULL &&
		     folder_class_mapping[idx].klass != NULL && 
		     folder_class_mapping[idx].mime_type != NULL; idx++)
		{
			if (g_str_has_prefix(klass, 
				folder_class_mapping[idx].klass) == TRUE &&
				folder_class_mapping[idx].mime_type != NULL)
			{
				mime_type = 
				  (gchar*)folder_class_mapping[idx].mime_type;
				break;
			}
		}
		
		data->mime_type = g_strdup(mime_type);
		return TRUE;
	}
	
	nodelist = cg_upnp_av_cds_didllite_node_getchildnodelist(content);
	if (nodelist == NULL)
	{
		return FALSE;
	}
	
	/* There could be multiple RES, we must select the most
	   appropriate */
	for (prop = cg_upnp_av_cds_didllite_nodelist_gets(nodelist);
	     prop != NULL;
	     prop = cg_upnp_av_cds_didllite_node_next(prop))
	{
		/* If it isn't RES, then we are not interested.. */
		if (!cg_upnp_av_cds_didllite_node_isname(prop,
						     DIDL_LITE_RES)) continue;
		
		/* If it is empty RES, then we are not interested either */
		if (cg_xml_node_getvalue(prop) == NULL) continue;
		
		protocol_info = cg_xml_node_getattributevalue(
						prop, 
						DIDL_LITE_RES_PROTOCOLINFO);
		if (protocol_info == NULL) continue;
		
		good_choice = 1;
		
		list = g_strsplit(protocol_info, ":", 0);
		if (list == NULL || g_strv_length(list) < 3)
		{
			g_strfreev(list); list = NULL;
			continue;
		}
		
		protocol = list[0];
		mime_type = list[2];
		
		/* Check if protocol is the preferred one */
		if (protocol == NULL || 
		    strcmp(protocol, RES_PREFERRED_PROTOCOL) != 0 ||
		    cg_xml_node_getvalue(prop) == NULL ||
		    !g_str_has_prefix(cg_xml_node_getvalue(prop), HTTP_METHOD))
		{
			/* We do not support any other protocols than HTTP */
			g_strfreev(list); list = NULL;
			mime_type = NULL;
			protocol = NULL;
			continue;
		}
		
		/* Check if MIME type is supported */
		if (mime_type != NULL && 
		    upnpav_is_mime_type_supported(mime_type))
		{
			good_choice++;
		}
		
		/* Select the best choice */
		if (good_choice > best_choice)
		{
			best_choice = good_choice;
			best = prop;
				
#ifndef MODEL_AS_PLAYLISTS
			/* Get mime type already here */
			if (mime_type != NULL)
			{
				if (data->mime_type != NULL)
					g_free(data->mime_type);
				
				data->mime_type = g_strdup(mime_type);
			}
#endif
		}
		
		g_strfreev(list); list = NULL;
		mime_type = NULL;
		protocol = NULL;
	}

	/* No resource found for this item */
	if (best == NULL) return FALSE;
	
	/* If we found some resource, then assign its data to node, mime is
	   assigned beforehand */
	result = TRUE;
	data->uri = g_strdup(cg_xml_node_getvalue(best));
	
#ifndef MODEL_AS_PLAYLISTS
	/* If MIME wasn't assigned or it was not supported, try again */
	if (!upnpav_is_mime_type_supported(data->mime_type))
	{
		/* We do not have application for this MIME type */
		g_free(data->mime_type); data->mime_type = NULL;
		
		data->mime_type = g_strdup(
			gnome_vfs_mime_type_from_name_or_default(data->uri, 
								NULL));
		
		if (!upnpav_is_mime_type_supported(data->mime_type) &&
		    cg_xml_node_getchildnodevalue(content,
						  DIDL_LITE_DC_TITLE) != NULL)
		{
			/* Last resort - Try to get mime type out of title */
			g_free(data->mime_type); data->mime_type = NULL;
			data->mime_type = g_strdup(
				gnome_vfs_mime_type_from_name_or_default (
					cg_xml_node_getchildnodevalue(
							content,
							DIDL_LITE_DC_TITLE), 
					NULL));
			
			if (!upnpav_is_mime_type_supported(data->mime_type))
			{
				/* If even this is not supported - give up */
				debug("Unable to determine MIME type for %s!",
				      data->uri);
				g_free(data->uri); data->uri = NULL;
				
				g_free(data->mime_type); 
				data->mime_type = NULL;
				return FALSE;
			}
		}
	}
	
	attr = cg_xml_node_getattribute(best, RES_SIZE_PROPERTY);
	if (attr != NULL)
	{
		/* Get size for the data */
		data->size = g_ascii_strtoull(cg_xml_node_getvalue(attr),
					      NULL, 10);
	}
	/* TODO: we should check also storageUsed property! */
#else
	/* Handle items modelled as playlists */
	data->size = cg_strlen(data->uri) + 1;
	g_free(data->mime_type);
	data->mime_type = g_strdup("audio/x-mpegurl");	
#endif
	
	date = cg_xml_node_getchildnodevalue(content, DIDL_LITE_DC_DATE);
	data->mtime = 0;
	if (date != NULL)
	{
		/* Get mtime for the data */
		memset(&tm, 0, sizeof(tm));
		if (strptime(date, DATE_FORMAT_DATETIME, &tm) ||
		    strptime(date, DATE_FORMAT_ISO8601, &tm))
		{
			data->mtime = mktime(&tm);
		}
	}
	
	return result;
}


GNode *upnpav_search_for_path(GNode *parent, const gchar *path, 
			      GnomeVFSCancellation *cancellation)
{
	GNode *node = NULL, *last_visited_node = NULL;
	UPnPAVNodeData *node_data, *parent_data;
	const gchar *next_path = NULL;
	CgSysTime mtime;
	GNode *result = NULL;
	guint node_count = 0;
	
	debug("upnpav_search_for_path (%s): %p", 
		    path, upnpav_search_for_path);
	
	g_return_val_if_fail(path != NULL && parent != NULL, NULL);
	
	next_path = strchr(path, '/');
    
	if (parent == NULL || next_path == NULL) 
	{
		return NULL;
	}
	
	/* Skip separator */
	next_path++;
	
	/* This is the root directory case */
	if (next_path[0] == 0) 
	{
		return parent;
	}
	
	parent_data = (UPnPAVNodeData*)parent->data;
	if (parent_data == NULL) {
		return NULL;
	}
	
	/* If we are not container, we must get exact match */
	if (parent_data->container == FALSE)
	{
		if (strcmp(next_path, parent_data->name) == 0)
		{
			return parent;
		} else {
			return NULL;
		}
	}
	
	/* Take first node */
	node = g_node_first_child(parent);
	
	do {
		while (node != NULL)
		{
#ifdef USE_CANCELLATION
			/* First check cancellation */
			if (cancellation != NULL && 
			    gnome_vfs_cancellation_check(cancellation) == TRUE)
			{
				error("Node search was cancelled");
				return NULL;
			}
#endif			
			/* Get node data and check it */
			node_data = (UPnPAVNodeData*)node->data;
			
			if (node_data != NULL && node_data->name != NULL &&
			    g_str_has_prefix(next_path, node_data->name))
			{
				if (strcmp(next_path, node_data->name) == 0)
				{
					/* Return node, if they are exact match */
					return node;
				}
				
				if (node_data->container == TRUE &&
				    strlen(next_path) > strlen(node_data->name) &&
				    (next_path + strlen(node_data->name))[0] == '/')
				{
					/* If there is separator after the node path, 
					   then, search for child paths */
				
					result = upnpav_search_for_path(node, 
						(next_path + strlen(node_data->name)), 
						cancellation);
					if (result != NULL) 
					{
						return result;
					}
				}
			}
		
			/* Get next node */
			last_visited_node = node;
			node = g_node_next_sibling(node);
		}
		
		if (tree->cp != NULL && tree->perform_search)
		{
			tree->perform_search = FALSE;
		
			debug("Performing M-SEARCH..");
			cg_upnp_controlpoint_search(tree->cp, 
						    ST_MEDIASERVER_V1);
			tree->search_complete_date = 
				cg_getcurrentsystemtime() + SEARCH_WAIT;
		
			/* WLAN may drop packets, so search again after 4
			   seconds */
			tree->search_source = g_timeout_add(4000,
						upnpav_delayed_search,
						NULL);
		}
	
		/* Here we might wait if our search has not completed */
		if (cg_getcurrentsystemtime() < tree->search_complete_date &&
		    tree->cp != NULL && parent == tree->root)
		{
			mtime = (tree->search_complete_date - 
				 cg_getcurrentsystemtime()) * 1000;
			debug("Waiting for SSDP search: %lu => %lu = %lu", 
			      cg_getcurrentsystemtime(), 
			      tree->search_complete_date,
			      mtime);
			if (mtime > SEARCH_WAIT*1000) mtime = SEARCH_WAIT*1000;
			
			node_count = g_node_n_children(parent);
			while (mtime > 0 && 
			       g_node_n_children(parent) <= node_count)
			{
				g_static_mutex_unlock(&tree_mutex);
				
				/* invalidate variables here as tree may
				   change when the mutex is unlocked */
				last_visited_node = NULL;
				node = NULL;
				node_data = NULL;
				
				cg_wait(100);
				g_static_mutex_lock(&tree_mutex);
				if (tree == NULL || tree->root == NULL) 
					return NULL;
				
				/* Decrease the waiting period */
				if (mtime > 100) 
				    mtime = mtime - 100;
				else 
				    mtime = 0;
				
#ifdef USE_CANCELLATION
				/* Cancel if required */
				if (cancellation != NULL && 
				    gnome_vfs_cancellation_check(cancellation) 
					== TRUE)
				{
					error("Node search was cancelled");
					return NULL;
				}
#endif
			}
			
			/* set complete date to 0 */
			if (mtime == 0)
				tree->search_complete_date = 0;
			
			/* Start iteration from the beginning as mutex has
			   been unlocked and tree may have changed. 
			   (parent == root, so it is safe) */
			node = g_node_first_child(parent);
		}
		
		/* Node was not found in parent, browse more, if there would 
		   be a match */
		if (node == NULL && parent_data->container == TRUE && 
		    parent != tree->root &&
		    parent_data->browsed < MAX_BROWSED && tree->cp != NULL)
		{
			if (upnpav_get_content_directory(
						parent, 
						cancellation,
						NULL) == FALSE)
			{
				return NULL;
			}

			/* Get first child or next from last visited */
			if (last_visited_node == NULL)
			{
				node = g_node_first_child(parent);
			} else {
				node = g_node_next_sibling(last_visited_node);
			}
			
		} 
		
	} while (node != NULL);
	
	return NULL;
}


static gchar *upnpav_get_path_for_node(GNode *node, gchar *lower_part)
{
	gchar *result = NULL, *tmp = NULL;
	UPnPAVNodeData *node_data = NULL;
	
	g_return_val_if_fail(node != NULL, NULL);
	
	node_data = (UPnPAVNodeData*)node->data;
	
	/* Path has to be escaped, so it is handled properly */
	tmp = gnome_vfs_escape_string(node_data->name);
	
	/* Concatenate lower part to path, if it exists */
	if (lower_part != NULL)
	{
		if (strcmp(node_data->name, "/") != 0)
		{
			result = g_strconcat(tmp, "/", lower_part, NULL);
		} else {
			/* root path is a special case */
			result = g_strconcat("/", lower_part, NULL);
		}
		g_free(lower_part); lower_part = NULL;
	} else {
		/* lower part didn't exist, just strdup node name */
		result = tmp;
		tmp = NULL;
	}
	g_free(tmp); tmp = NULL;
	
	if (node->parent != NULL)
	{
		/* Recursively get the string also for parent */
		result = upnpav_get_path_for_node(node->parent, result);
	}
	
	return result;
}


static void upnpav_destroy_node(GNode *node,
				gpointer data)
{
	UPnPAVNodeData *node_data = NULL;
	gint flags = GPOINTER_TO_INT(data);
	gchar *cache_key = NULL;
	
	g_return_if_fail(node != NULL);
	
	/* First delete children */
	debug("Deleting all node children..");
	g_node_children_foreach(node,
				G_TRAVERSE_ALL,
				(GNodeForeachFunc)upnpav_destroy_node,
				data);
	
	node_data = (UPnPAVNodeData*)node->data;
	
	if ((flags & NODE_REMOVE_EVENT) && node_data != NULL)
	{
		upnpav_notify_monitor(node, GNOME_VFS_MONITOR_EVENT_DELETED);
	}

	/* Remove node from container cache */
	if (node_data != NULL)
	{
		if (node_data->container == TRUE && 
		    (flags & NODE_REMOVE_UPDATE_CACHE))
		{
			cache_key = g_strconcat(node_data->dev_udn, 
						CONTAINER_ID_DELIM, 
						node_data->object_id, NULL);
			g_hash_table_remove(tree->container_cache, cache_key);
			g_free(cache_key); cache_key = NULL;
		}
	
		upnpav_node_data_destroy(node_data);
	
		node_data = NULL;
		node->data = NULL;
	}
		
	/* Remove from parent and free it */
	g_node_unlink(node);
	g_node_destroy(node); node = NULL;
	nodecount--;
}


static void upnpav_node_data_destroy(UPnPAVNodeData *node_data)
{
	GSList *tmp;
	
	if (node_data->object_id != NULL &&
	    strcmp(node_data->object_id, DEVICE_OBJECT) == 0)
	{
		/* If this is device node, then UDN must also be freed */
		/* Note that all child will have invalid udn after this! */
		g_free(node_data->dev_udn);
		node_data->dev_udn = NULL;
	}
	
	/* Invalidate directory handles */
	tmp = node_data->directory_handles;
	while (tmp != NULL)
	{
		error("Invalidating directory handle");
		((UPnPAVDirectoryHandle*)tmp->data)->directory_node = NULL;
		tmp = tmp->next;
	}
	g_slist_free(node_data->directory_handles);
	node_data->directory_handles = NULL;
	
	if (node_data->mime_type != NULL) g_free(node_data->mime_type);
	if (node_data->uri != NULL) g_free(node_data->uri);
	if (node_data->name != NULL) g_free(node_data->name);
	if (node_data->object_id != NULL) g_free(node_data->object_id);
	
	node_data->container = FALSE;
	node_data->browsed = 0;
	node_data->size = 0;
	node_data->update_id = -1;
	
	g_slice_free(UPnPAVNodeData, node_data);
}


/**
	Returns list of child nodes, which are node1's childs, but NOT
	node2's childs. Differentation is based on object IDs.

	@param node1
	@param node2
	@param unparent	Unlink (from parent) node1's child nodes, which are
			not node2's childs.
	@return
 */
static GSList *upnpav_different_childs(GNode* node1, GNode* node2, 
				       gboolean unparent)
{
	GNode *node1_iter = NULL, *node2_iter = NULL, *tmp = NULL;
	GSList *result_list = NULL;
	gboolean found = FALSE;
	
	node1_iter = g_node_first_child(node1);
	while (node1_iter != NULL)
	{
		node2_iter = g_node_first_child(node2);
		
		found = FALSE;
		while (node2_iter != NULL)
		{
			if (strcmp(
			    ((UPnPAVNodeData*)node1_iter->data)->object_id,
			    ((UPnPAVNodeData*)node2_iter->data)->object_id)
				== 0)
			{
				found = TRUE; break;
			}
			
			node2_iter = g_node_next_sibling(node2_iter);
		}
		
		/* If the node in question wasn't found in nodes2, then
		   it is added to the result list */
		if (!found)
		{
			result_list = g_slist_append(result_list, 
						     node1_iter);
			if (unparent)
				tmp = node1_iter;
		}
		
		node1_iter = g_node_next_sibling(node1_iter);
		
		if (tmp)
		{
			/* Unparent must not happen before iterating forward */
			g_node_unlink(tmp);
			tmp = NULL;
		}
	}
	
	return result_list;
}


static void upnpav_node_handle_container_change(GNode *node,
						gboolean recursive)
{
	GNode *new_node = NULL, *node_iter = NULL;
	UPnPAVNodeData new_node_data;
	UPnPAVNodeData *node_data = NULL;
	GSList *added_nodes = NULL, *removed_nodes = NULL, *tmp = NULL;
	gchar *cache_key = NULL;

	g_return_if_fail(node != NULL && node->data != NULL);
	
	debug("upnpav_node_handle_container_change");
	
	node_data = ((UPnPAVNodeData*)node->data);
	
	/* Special cases: this is not a container or we don't have this in 
	   container cache, so we are not interested */
	if (node_data->container != TRUE) return;
	
	cache_key = g_strconcat(node_data->dev_udn, CONTAINER_ID_DELIM, 
				node_data->object_id, NULL);
	if (g_hash_table_lookup(tree->container_cache, cache_key) == NULL)
		return;
	
	new_node = g_node_new(&new_node_data);
	
	/* Make shallow copy of nodes data */
	new_node_data = *node_data;
	new_node_data.directory_handles = NULL;
	new_node_data.browsed = 0;
	
	/* Browse children for the new node */
	while (new_node_data.browsed < node_data->browsed &&
	       upnpav_get_content_directory(new_node, NULL, NULL));
	
	if (new_node_data.browsed < node_data->browsed)
	{
		/* get_content_directory failed, bail out */
		error("Get Content directory failed! (%lu < %lu)",
		      new_node_data.browsed, node_data->browsed);
		new_node->data = NULL;
		upnpav_destroy_node(new_node, GINT_TO_POINTER(0));
		return;
	}
	
	node_data->browsed = new_node_data.browsed;
	debug("Children: Old dir: %d, New dir: %d", 
	      g_node_n_children(node),
	      g_node_n_children(new_node));
	
	/* Invalidate directory handles */
	tmp = node_data->directory_handles;
	while (tmp != NULL)
	{
		error("Invalidating directory handle");
		((UPnPAVDirectoryHandle*)tmp->data)->directory_node = NULL;
		tmp = tmp->next;
	}
	g_slist_free(node_data->directory_handles);
	node_data->directory_handles = NULL;
	
	/* Check the difference */
	removed_nodes = upnpav_different_childs(node, new_node, FALSE);
	added_nodes = upnpav_different_childs(new_node, node, TRUE);
	
	/* Destroy removed nodes with eventing */
	tmp = removed_nodes;
	while (tmp != NULL)
	{
		debug("Removing node %s", ((UPnPAVNodeData*)((GNode*)tmp->data)->data)->name);
		
		upnpav_destroy_node((GNode*)tmp->data, 
				    GINT_TO_POINTER(
					NODE_REMOVE_EVENT | 
					NODE_REMOVE_UPDATE_CACHE));
		tmp = tmp->next;
	}
	g_slist_free(removed_nodes); removed_nodes = NULL;
	
	/* Add new nodes */
	tmp = added_nodes;
	while (tmp != NULL)
	{
		debug("Adding node %s", ((UPnPAVNodeData*)((GNode*)tmp->data)->data)->name);
		
		g_node_append(node, (GNode*)tmp->data);
		upnpav_notify_monitor((GNode*)tmp->data, 
				      GNOME_VFS_MONITOR_EVENT_CREATED);
		tmp = tmp->next;
	}
	g_slist_free(added_nodes); added_nodes = NULL;
	
	/* Destroy new_node (and children) without eventing */
	new_node->data = NULL;
	upnpav_destroy_node(new_node, GINT_TO_POINTER(0));
	new_node = NULL;
	
	if (recursive)
	{
		/* Go through all the browsed container children and handle 
		   change in them too */
		node_iter = g_node_first_child(node);
		while (node_iter != NULL)
		{
			node_data = (UPnPAVNodeData*)node_iter->data;
			if (node_data != NULL && node_data->container &&
			    node_data->browsed > 0)
			{
				upnpav_node_handle_container_change(node_iter, 
								    recursive);
			}
			node_iter = g_node_next_sibling(node_iter);
		}
	}
}


static void upnpav_node_handle_container_change_by_csv(const gchar *dev_udn, 
						       const gchar *csv)
{
	const gchar *end = csv;
	gchar *cache_key = NULL, *tmp = NULL;
	glong update_id = -1;
	GNode *node = NULL;

	debug("upnpav_node_handle_container_change_by_csv: %s", csv);
	
	while (end != NULL && end >= csv && 
	       end < csv + strlen(csv) && end[0] != 0)
	{
		end = strchr(end, ',');
		if (end > csv && end < csv + strlen(csv) && 
		    *(end - 1) == '\\')
		{
			end++;
			continue;
		} else if (end == NULL) 
		{
			tmp = g_strdup(csv);
		} else {
			tmp = g_strndup(csv, end-csv);
			end++;
		}
		
		if (cache_key == NULL)
		{
			cache_key = g_strconcat(dev_udn, CONTAINER_ID_DELIM, 
						tmp, NULL);
		} else {
			update_id = cg_str2long(tmp);
			
			/* Get corresponding node from container cache */
			node = (GNode *)g_hash_table_lookup(
						tree->container_cache,
						cache_key);
			
			/* Check if there is a need to update */
			if (node != NULL && node->data != NULL &&
			    ((UPnPAVNodeData*)node->data)->update_id 
				!= update_id)
			{
				upnpav_node_handle_container_change(node, 
								    FALSE);
			}
			g_free(cache_key); cache_key = NULL;
		}
		
		g_free(tmp); tmp = NULL;
		csv = end;
	}
}


static void upnpav_notify_monitor(GNode *node,
				  GnomeVFSMonitorEventType event_type)
{
	UPnPAVMonitorList *monitors = NULL;
	GList *l = NULL;
	UPnPAVMonitorHandle *handle = NULL;
	GnomeVFSURI *parent = NULL, *grand_parent = NULL, *uri = NULL;
	gchar *path = NULL, *uri_str = NULL;
	
	g_return_if_fail(node != NULL);
	
	path = upnpav_get_path_for_node(node, NULL);
	
	g_return_if_fail(path != NULL);
	
	uri_str = g_strconcat(UPNPAV_METHOD, path, NULL);
	uri = gnome_vfs_uri_new(uri_str);
	debug("upnpav_notify_monitor: %s", uri_str);
	g_free(path); path = NULL;
	g_free(uri_str); uri_str = NULL;
	
	/* Notify all registered monitors (for the given URI) */
	g_static_mutex_lock(&monitor_mutex);
	
	monitors = g_hash_table_lookup(monitor_hash, uri);
	if (monitors) {
		for (l = monitors->handles; l; l = l->next)
		{
			debug("Notifying info URI");
			handle = (UPnPAVMonitorHandle *)l->data;
			gnome_vfs_monitor_callback(
					(GnomeVFSMethodHandle *)handle,
					uri, event_type);
		}
	}
	
	parent = gnome_vfs_uri_get_parent(uri);
	if (parent == NULL)
	{
		gnome_vfs_uri_unref(uri); uri = NULL;
		g_static_mutex_unlock(&monitor_mutex);
		debug("No parent. Exiting upnpav_notify_monitor..");
		return;
	}
	
	monitors = g_hash_table_lookup(monitor_hash, parent);
	if (monitors) {
		for (l = monitors->handles; l; l = l->next)
		{
			handle = (UPnPAVMonitorHandle *)l->data;
			
			if (handle->monitor_type == GNOME_VFS_MONITOR_DIRECTORY) 
			{
				debug("Notifying parent URI");
				gnome_vfs_monitor_callback(
					(GnomeVFSMethodHandle *)handle,
					uri, event_type);				
			}
		}
	}
	
	gnome_vfs_uri_unref(uri); uri = NULL;
	grand_parent = gnome_vfs_uri_get_parent (parent);
        if (grand_parent == NULL ||
	    (event_type != GNOME_VFS_MONITOR_EVENT_CREATED &&
	     event_type != GNOME_VFS_MONITOR_EVENT_DELETED))
	{
		if (grand_parent != NULL) 
			gnome_vfs_uri_unref(grand_parent);
		
		gnome_vfs_uri_unref(parent);
		g_static_mutex_unlock(&monitor_mutex);
		debug("No grand parent. Exiting upnpav_notify_monitor..");
		return;
	}
	
	monitors = g_hash_table_lookup(monitor_hash, grand_parent);
	if (monitors) {
		for (l = monitors->handles; l; l = l->next)
		{
			handle = (UPnPAVMonitorHandle *)l->data;
			
			if (handle->monitor_type == GNOME_VFS_MONITOR_DIRECTORY) 
			{
				debug("Notifying grand parent URI");
				gnome_vfs_monitor_callback(
					(GnomeVFSMethodHandle *)handle,
					parent, GNOME_VFS_MONITOR_EVENT_CHANGED);
			}
		}
	}
	
	gnome_vfs_uri_unref(parent);
	gnome_vfs_uri_unref(grand_parent);
	
	g_static_mutex_unlock(&monitor_mutex);
	debug("Exiting upnpav_notify_monitor..");
}


static gboolean upnpav_get_device_to_node(CgUpnpDevice *dev, GNode *node)
{
	UPnPAVNodeData *node_data = NULL, *parent_node_data;
	CgUpnpService *service = NULL;
	GNode *new_node = NULL;
	
	g_return_val_if_fail(tree->cp != NULL, FALSE);
		
        /* Only append such servers that support CDS */
	service = cg_upnp_device_getservicebytype(dev, 
						  CG_UPNP_CDS_SERVICE_TYPE);
	if (service != NULL)
	{
		node_data = g_slice_new0(UPnPAVNodeData);
		node_data->name = g_strdup(cg_upnp_device_getfriendlyname(dev));
		node_data->object_id = g_strdup(DEVICE_OBJECT);
		node_data->dev_udn = g_strdup(cg_upnp_device_getudn(dev));
		node_data->container = TRUE;
		node_data->browsed = 0;
		node_data->mime_type = g_strdup(DEVICE_DIRECTORY_MIME_TYPE);
		node_data->uri = NULL;
		
		debug ( "Adding device %s to tree", node_data->name );
		new_node = g_node_append_data(node, node_data);
		nodecount++;
		
		/* Notify monitors only if nothing is iterating it */
		parent_node_data = (UPnPAVNodeData *)node->data;
		if (parent_node_data == NULL ||
		    parent_node_data->directory_handles == NULL)
		{
			upnpav_notify_monitor(new_node, 
					      GNOME_VFS_MONITOR_EVENT_CREATED);
		}
		
		/* Do subscribtion when browsing the device first time */
	}
	
	return TRUE;
}


static void upnpav_device_listener(char* udn, CgUpnpDeviceStatus status)
{
	GNode *child = NULL;
	GNode *dev_node = NULL;
	UPnPAVNodeData *node_data = NULL;
	CgUpnpDevice *device = NULL;
	
	g_return_if_fail(udn != NULL);
	
	/* Ignore events we are not interested in */
	if (status != CgUpnpDeviceStatusAdded &&
	    status != CgUpnpDeviceStatusRemoved &&
	    status != CgUpnpDeviceStatusInvalid)
		return;
	
	debug("upnpav_device_listener: %p", upnpav_device_listener);
	
	g_static_mutex_lock(&tree_mutex);
	
	/* Sanity checks */
	if (tree == NULL || tree->root == NULL || tree->cp == NULL) {
		error("upnpav_device_listener called without tree or CP");
		g_static_mutex_unlock(&tree_mutex);
		return;
	}
	
	/* Initial check, if we are interested in this device */
	cg_upnp_controlpoint_lock(tree->cp);
	device = cg_upnp_controlpoint_getdevicebyudn(tree->cp, udn);
	if (cg_upnp_controlpoint_getdevicelistener(tree->cp) == NULL ||
	    (device != NULL &&
	     cg_upnp_device_getservicebytype(
				device, CG_UPNP_CDS_SERVICE_TYPE) == NULL))
	{
		cg_upnp_controlpoint_unlock(tree->cp);
		g_static_mutex_unlock(&tree_mutex);
		return;
	}
	device = NULL;
	cg_upnp_controlpoint_unlock(tree->cp);
	
	/* Check if device already exists in the tree */
	child = g_node_first_child(tree->root);
	while (child != NULL)
	{
		node_data = (UPnPAVNodeData*)child->data;
		if (node_data->dev_udn != NULL &&
		    strcmp(node_data->dev_udn, udn) == 0)
		{
			/* Device node was found */
			dev_node = child;
			break;
		}
		
		child = g_node_next_sibling(child);
	}
	
	/* If device was not found and packet is not byebye, add it */
	if (status == CgUpnpDeviceStatusAdded &&
	    dev_node == NULL)
	{
		cg_upnp_controlpoint_lock(tree->cp);
		
		/* Now we have the udn, get the device */
		device = cg_upnp_controlpoint_getdevicebyudn(tree->cp, udn);
		if (cg_upnp_controlpoint_getdevicelistener(tree->cp) == NULL || 
		    device == NULL)
		{
			/* Device not found */
			cg_upnp_controlpoint_unlock(tree->cp);
			g_static_mutex_unlock(&tree_mutex);
			debug("Could not get the device. "
				    "Exiting upnpav_device_listener..");
			return;
		}
	
		/* Add device node to root */
		upnpav_get_device_to_node(device, tree->root);
		cg_upnp_controlpoint_unlock(tree->cp);
		
	} else if ((status == CgUpnpDeviceStatusRemoved ||
		    status == CgUpnpDeviceStatusInvalid) &&
		    dev_node != NULL)
	
	{
		upnpav_destroy_node(dev_node, GINT_TO_POINTER(
			NODE_REMOVE_EVENT | NODE_REMOVE_UPDATE_CACHE) );
	}
	
	g_static_mutex_unlock(&tree_mutex);
	
	debug("Exiting upnpav_device_listener..");
}


static void upnpav_event_listener(CgUpnpProperty *prop)
{
	const gchar *name = NULL;
	const gchar *ssid = NULL;
	const gchar *value = NULL;
	glong seq = 0;
	glong prop_value = -1;
	
	const gchar *udn = NULL;
	GNode *child = NULL, *dev_node = NULL;
	UPnPAVNodeData *node_data = NULL;
	
	debug("upnpav_event_listener: %p", upnpav_event_listener);
	
	g_return_if_fail(prop != NULL);
	
	name = cg_upnp_property_getname(prop);
	ssid = cg_upnp_property_getsid(prop);
	seq = cg_upnp_property_getseq(prop);
	value = cg_upnp_property_getvalue(prop);
	
	g_return_if_fail(name != NULL && ssid != NULL);
	
	/* If there is no value, we are not interested */
	if (value == NULL || strlen(value) < 1)
		return;
	
	if (strlen(value) > 0 && isdigit(value[0]))
		prop_value = cg_str2long(value);
		
	g_static_mutex_lock(&tree_mutex);
	
	/* Sanity checks */
	if (tree == NULL || tree->cp == NULL || tree->root == NULL)
	{
		error("Control point or tree does not exist");
		g_static_mutex_unlock(&tree_mutex);
		return;
	}
	
	/* Get device node */
	child = g_node_first_child(tree->root);
	while (child != NULL)
	{
		node_data = (UPnPAVNodeData*)child->data;
		if (node_data != NULL && node_data->uri != NULL &&
		    strcmp(node_data->uri, ssid) == 0)
		{
			/* Device node was found */
			dev_node = child;
			udn = node_data->dev_udn;
			break;
		}
		
		child = g_node_next_sibling(child);
	}
		
	/* Dev node not found */
	if (dev_node == NULL || udn == NULL)
	{
		error("Device for SSID %s not found!", ssid);
		g_static_mutex_unlock(&tree_mutex);
		return;
	}
	
	if (seq == 0)
	{
		/* This is initial event */
		if (strcmp(name, CDS_SYSTEM_UPDATE_ID) == 0 &&
		    ((UPnPAVNodeData*)dev_node->data)->update_id >= 0)
		{
			/* Save system update ID as dev node's update_id */
			((UPnPAVNodeData*)dev_node->data)->update_id = 
				prop_value;
		} else if (strcmp(name, CDS_CONTAINER_UPDATE_IDS) == 0) {
			/* Device supports ContainerUpdateIDs eventing, don't
			   use system update id */
			((UPnPAVNodeData*)dev_node->data)->update_id = -1;
		}		
		g_static_mutex_unlock(&tree_mutex);
		return;
	}
	
	/* Check that SystemUpdateID has been updated and it is not
	   an initial update (seq > 0) */
	if (strcmp(name, CDS_SYSTEM_UPDATE_ID) == 0 && 
	    ((UPnPAVNodeData*)dev_node->data)->update_id >= 0 &&
	    ((UPnPAVNodeData*)dev_node->data)->update_id != prop_value)
	{
		debug("SystemUpdateID was updated");
		
		/* System update id was evented and we don't have 
		   ContainerUpdateID support. Go through device 
		   recursively */
		upnpav_node_handle_container_change(dev_node, TRUE);
		((UPnPAVNodeData*)dev_node->data)->update_id = prop_value; 
	} else if (strcmp(name, CDS_CONTAINER_UPDATE_IDS) == 0) 
	{
		debug("ContainerUpdateIDs was updated");
		
		upnpav_node_handle_container_change_by_csv(
			((UPnPAVNodeData*)dev_node->data)->dev_udn,
			value);
	}
	
	g_static_mutex_unlock(&tree_mutex);
	debug("Exiting Event handler..");
}

#ifdef GUPNP_USE_MAEMO
static void upnpav_iap_listener(ConIcConnection *connection,
				ConIcConnectionEvent *event,
				gpointer user_data)
{
	const gchar *bearer = NULL;
	ConIcConnectionStatus status;
	
	g_assert(event != NULL);
	
	status = con_ic_connection_event_get_status(event);

	/* Check if bearer is supported */
	bearer = (const gchar *)con_ic_event_get_bearer_type(CON_IC_EVENT(event));
	if (bearer == NULL ||
	    (strcmp(bearer, CON_IC_BEARER_WLAN_INFRA) != 0 &&
	     strcmp(bearer, CON_IC_BEARER_WLAN_ADHOC) != 0))
	{
		debug("Bearer not supported: %s", bearer);
		return;
	}
	
	g_static_mutex_lock(&tree_mutex);
	
	/* Sanity checks */
	if (tree == NULL) 
	{
		g_static_mutex_unlock(&tree_mutex);
		return;
	}
	
	debug("upnpav_iap_listener: %p", upnpav_iap_listener);
	
	if (status == CON_IC_STATUS_CONNECTED && tree->iap_name == NULL)
	{
		/* Save IAP name */
		tree->iap_name = g_strdup(
			con_ic_event_get_iap_id(CON_IC_EVENT(event)));
		
		debug("Saving IAP name %s for further use..",
			    tree->iap_name);
		
		debug("Creating UPnP control point..");
		
		/* Start control point */
		upnpav_create_controlpoint();
		
		/* Do the search */
		if (tree->cp != NULL)
		{
			debug("Performing M-SEARCH..");
			cg_upnp_controlpoint_search(tree->cp, 
						    ST_MEDIASERVER_V1);
			tree->search_complete_date = 
				cg_getcurrentsystemtime() + SEARCH_WAIT;
		}
	} else if (status == CON_IC_STATUS_DISCONNECTED || 
		   status == CON_IC_STATUS_DISCONNECTING)
	{
		tree->perform_search = FALSE;
			
		if (tree->search_source > 0)
		{
			g_source_remove(tree->search_source);
			tree->search_source = 0;
		}
	
		if (tree->iap_name != NULL)
		{
			g_free(tree->iap_name);
			tree->iap_name = NULL;
		}
		
		upnpav_destroy_controlpoint();
	}
	
	g_static_mutex_unlock(&tree_mutex);
	
	debug("Exiting upnpav_iap_listener..");
}
#endif
