/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* vim:set ts=2 sw=2 sts=2 tw=80 et cindent: */
/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the mozilla.org code.
 *
 * The Initial Developer of the Original Code is
 * Oleg Romashin <romaxa@gmail.com>
 *
 * Portions created by the Initial Developer are Copyright (C) 2005
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

#include <stdio.h>
#include "gtkmozembed_maemo.h"
#include "EmbedPrivate.h"
#include "malloc.h"
#include "nsITimer.h"
#include "nsComponentManagerUtils.h"
#include <fcntl.h>
#include <unistd.h>
#include "gtkmozembed_download.h"

#define MEM_LEVEL_1 50
#define MEM_LEVEL_2 42
#define MEM_LEVEL_3 30
const char *kMemTopic = "memory-pressure";
const char *kMemData = "h\0e\0a\0p\0-\0m\0i\0n\0i\0m\0i\0z\0e\0\0";
#ifdef __arm__
const char *kLowMark = "/sys/kernel/low_watermark";
const char *kHighMark = "/sys/kernel/high_watermark";
const PRInt32 kMult = 10;
#else
const char *kLowMark = "/tmp/low_watermark";
const char *kHighMark = "/tmp/high_watermark";
const PRInt32 kMult = 1;
#endif
const char *kSuspendDisable = "/tmp/.flash_lock.";
static PRBool sSuspendEnabled = PR_TRUE;

#include <sys/types.h>
#include <signal.h>
#ifdef OSSO_ENABLE
#include <libosso.h>
#endif
#include "nsMemory.h"

static GThread* gMemThread = nsnull;

/* Original sources comes from nsMemoryImpl.cpp */
/**
 * A class that is used to periodically check the status of the system,
 * determine if too much memory is in use, and if so, trigger a "memory flush".
*/
class EmbedMemoryFlusher : public nsITimerCallback
{
public:
    // We don't use the generic macros because we are a special static object
    NS_IMETHOD QueryInterface(REFNSIID aIID, void** aResult);
    NS_IMETHOD_(nsrefcnt) AddRef(void) {
        return 2;
    }
    NS_IMETHOD_(nsrefcnt) Release(void) {
        return 1;
    }

    NS_DECL_NSITIMERCALLBACK

    nsresult Init();
    void StopAndJoin();
    static PRBool GetWatermarkValue (const char *path);
    static nsresult CheckWatermarkNotifier (const char *path);
    nsresult CheckMemoryState(PRBool aLowMem = PR_FALSE);

    enum {
        // milliseconds
        MEM_CHECK_DEFAULT_INTERVAL = 2000 * kMult,
        MEM_CHECK_LOWMEM_INTERVAL  = 1000 * kMult,
        MEM_CHECK_HIGHMEM_INTERVAL = 500 * kMult
    };

    PRBool mMemLevel;

private:

    nsresult HandleLowMemory(PRBool aEnable = PR_TRUE);
    nsresult HandleHighMemory(PRBool aEnable = PR_TRUE);
    nsresult TempImageDisable(PRBool aEnable = PR_TRUE);
    nsresult TempJsDisable(PRBool aEnable = PR_TRUE);


    PRBool mIsHighMemory;
    PRBool mIsLowMemory;
    PRInt32 mPrevImgPolicy;
    PRInt32 mPrevJsStatus;
    nsCOMPtr<nsITimer> mTimer;
    void * mContext;
    PRInt32 mImageEnableLevel;
};

static EmbedMemoryFlusher sGlobalEmbedMemoryFlusher;
NS_IMPL_QUERY_INTERFACE1(EmbedMemoryFlusher, nsITimerCallback)

static void *
high_mem_thread_func(void *data)
{
    PRBool high = PR_FALSE;
    PRBool low = PR_FALSE;
    EmbedMemoryFlusher* flusher = static_cast<EmbedMemoryFlusher *>(data);
    nsCOMPtr<nsIMemory> mMem;
    NS_GetMemoryManager(getter_AddRefs(mMem));
    while (TRUE) {
      PR_Sleep(EmbedMemoryFlusher::MEM_CHECK_HIGHMEM_INTERVAL);
      //PR_Sleep(200);
      low = EmbedMemoryFlusher::GetWatermarkValue(kLowMark);
      if (!low) { flusher->mMemLevel = MEM_LEVEL_1; continue; };
      high = EmbedMemoryFlusher::GetWatermarkValue(kHighMark);
      //g_debug("--bcx:l:%i,h:%i", low, high);
      mMem->SetLowMemory(low);
      if (high && low) {
        flusher->mMemLevel--;
        if (flusher->mMemLevel == MEM_LEVEL_2) {
          g_debug("MEM_LEVEL_2: Disable images to avoid OOM restart: pid:1");
          EmbedCommon::SuspendNative(PR_FALSE);
        } else if (flusher->mMemLevel == MEM_LEVEL_3) {
          g_debug("MEM_LEVEL_3: Disable something else");
          EmbedCommon::SuspendNative(PR_FALSE);
        } else if (!flusher->mMemLevel) {
          int pid = getpid();
          g_debug("MEM_LEVEL_OOM: Restart engine to avoid OOM kill: pid:%d", pid);
          // Device survive functionality.
          //gint retval = 0;
          //EmbedCommon * common = EmbedCommon::GetInstance();
          //if (common) g_signal_emit_by_name (G_OBJECT (common->mCommon), "on-service-notify", common->mCommon, "memory", "restart_required", &retval);
          kill (pid, SIGTERM);
        }
      } else {
        flusher->mMemLevel = MEM_LEVEL_1;
      }
    }
    EmbedCommon::SuspendNative(PR_FALSE);
    gMemThread = nsnull;
    return NULL;
}

#ifdef OSSO_ENABLE
static void osso_browser_hw_cb_f(void * aState, void * aData)
{
  osso_hw_state_t *state = static_cast<osso_hw_state_t *>(aState);
  EmbedMemoryFlusher* flusher = static_cast<EmbedMemoryFlusher *>(aData);
static PRBool inactivity = PR_FALSE;
  if (state->shutdown_ind) {
    g_debug("osso_browser_hw_cb_f: shutdown_ind");
  }
  else if (state->memory_low_ind) {
    g_debug("memory_low_ind = %i, ", state->memory_low_ind);
    flusher->CheckMemoryState(PR_TRUE);
  } else if (state->system_inactivity_ind) {
    g_debug("osso_browser_hw_cb_f: system_inactivity_ind");
    if (gtk_moz_embed_download_get_latest_object())
      return;
    static char *kSuspendDisablePid = nsnull;
    if (!kSuspendDisablePid) {
      kSuspendDisablePid = g_strdup_printf("%s%i", kSuspendDisable, getpid());
    }
    if (!sSuspendEnabled || NS_SUCCEEDED(EmbedMemoryFlusher::CheckWatermarkNotifier(kSuspendDisablePid)))
      return;
    if (!inactivity)
      EmbedCommon::SuspendNative(state->system_inactivity_ind);
    inactivity = state->system_inactivity_ind;
  } else {
    g_debug("memory_low_ind = %i,  system_inactivity_ind: %i", state->memory_low_ind, state->system_inactivity_ind);
    flusher->CheckMemoryState(PR_FALSE);
    if (inactivity)
      EmbedCommon::SuspendNative(state->system_inactivity_ind);
    inactivity = state->system_inactivity_ind;
  }
}
#endif

nsresult
EmbedMemoryFlusher::CheckWatermarkNotifier (const char *path)
{
  int fd = open (path, O_RDONLY);
  if (fd == -1) return NS_ERROR_FAILURE;
  close (fd);
  return NS_OK;
}

PRBool
EmbedMemoryFlusher::GetWatermarkValue (const char *path)
{
  int fd = open (path, O_RDONLY);
  if (fd == -1) return PR_FALSE;
  int c = 0;
  read (fd, &c, 1);
  close (fd);
  return c == '1';
}

nsresult
EmbedMemoryFlusher::CheckMemoryState(PRBool aLowMem)
{
  //g_debug("Checking memory status: Func:%s::%d, low:%i, high:%i\n", __PRETTY_FUNCTION__, __LINE__, mIsLowMemory, mIsHighMemory);
  if (!mIsHighMemory) {
    if (aLowMem || GetWatermarkValue (kLowMark)) {
      if (!mIsLowMemory) {
        g_debug ("BROWSER: Run in low memory situation!!!\n");
        mIsLowMemory = PR_TRUE;
        HandleLowMemory();
      }
      mImageEnableLevel = 0;
    } else if (mIsLowMemory) {
      g_debug ("BROWSER: Run out low memory situation!!!\n");
      mIsLowMemory = PR_FALSE;
      HandleLowMemory(mIsLowMemory);
      mImageEnableLevel = 3;
      TempJsDisable(PR_FALSE);
    }
  }
  if (mImageEnableLevel) {
    mImageEnableLevel--;
    if (!mImageEnableLevel)
      TempImageDisable(PR_FALSE);
  }

  if (mIsLowMemory || aLowMem) {
    if (GetWatermarkValue (kHighMark)) {
      if (!mIsHighMemory) {
        g_debug ("BROWSER: Run in very low memory situation!!!\n");
        mIsHighMemory = PR_TRUE;
        HandleHighMemory();
      }
      if (mMemLevel == MEM_LEVEL_2) {
        TempImageDisable(PR_TRUE);
      }
      if (mMemLevel == MEM_LEVEL_3) {
        TempJsDisable(PR_TRUE);
      }
      gtk_moz_embed_common_observe(nsnull, nsnull, kMemTopic, (gunichar *)kMemData);
    } else if (mIsHighMemory) {
      g_debug ("BROWSER: Run out very low memory situation!!!\n");
      mIsHighMemory = PR_FALSE;
      HandleHighMemory(mIsHighMemory);
    }
  }
  return NS_OK;
}

NS_IMETHODIMP
EmbedMemoryFlusher::Notify(nsITimer *timer)
{
  CheckMemoryState();
  PRUint32 delay = mIsHighMemory?MEM_CHECK_HIGHMEM_INTERVAL:(mIsLowMemory?MEM_CHECK_LOWMEM_INTERVAL:MEM_CHECK_DEFAULT_INTERVAL);
  //g_debug ("BROWSER: SetDelay to %i\n", delay);
  mTimer->SetDelay(delay);
  return NS_OK;
}

nsresult
EmbedMemoryFlusher::HandleLowMemory(PRBool aEnable)
{
  //g_debug("Checking memory status: Func:%s::%d, low:%i, high:%i\n", __PRETTY_FUNCTION__, __LINE__, mIsLowMemory, mIsHighMemory);
  gtk_moz_embed_common_observe(nsnull, nsnull, kMemTopic, (gunichar *)kMemData);
  malloc_trim (0);
  return NS_OK;
}

nsresult
EmbedMemoryFlusher::HandleHighMemory(PRBool aEnable)
{
  //g_debug("Checking memory status: Func:%s::%d, low:%i, high:%i\n", __PRETTY_FUNCTION__, __LINE__, mIsLowMemory, mIsHighMemory);
  EmbedCommon::SuspendNative(aEnable);
  if (!aEnable) return NS_OK;
  gtk_moz_embed_common_observe(nsnull, nsnull, kMemTopic, (gunichar *)kMemData);
  malloc_trim (0);
  return NS_OK;
}

nsresult
EmbedMemoryFlusher::TempJsDisable(PRBool aDisable)
{
  g_debug("Func:%s::%d, aDisable:%i, mPrevImgPolicy:%i, \n", __PRETTY_FUNCTION__, __LINE__, aDisable, mPrevJsStatus);
  PRBool policy = PR_TRUE;
  if (aDisable) {
    gtk_moz_embed_common_get_pref(G_TYPE_BOOLEAN, "javascript.enabled", &mPrevJsStatus);
    policy = PR_FALSE;
  } else if (mPrevJsStatus >= 0) {
    policy = mPrevJsStatus;
    mPrevJsStatus = -1;
    if (!gtk_moz_embed_common_set_pref(G_TYPE_BOOLEAN, "javascript.enabled", &policy))
      return NS_ERROR_FAILURE;
  }

  return NS_OK;
}

nsresult
EmbedMemoryFlusher::TempImageDisable(PRBool aDisable)
{
  /* Bug83382 Browser crashs in Low_memory state. 
     So, must not do Reload operation or any other memory related stuff.
   */
  return NS_OK;
  g_debug("Func:%s::%d, aDisable:%i, mPrevImgPolicy:%i, \n", __PRETTY_FUNCTION__, __LINE__, aDisable, mPrevImgPolicy);
  int policy = -1;
  if (aDisable) {
    gtk_moz_embed_common_get_pref(G_TYPE_INT, "permissions.default.image", &mPrevImgPolicy);
    policy = GTK_MOZ_EMBED_IMAGE_POLICY_NEVER;
    if (!gtk_moz_embed_common_set_pref(G_TYPE_INT, "permissions.default.image", &policy))
      return NS_ERROR_FAILURE;
  } else if (mPrevImgPolicy >= 0) {
    policy = mPrevImgPolicy;
    mPrevImgPolicy = -1;
    if (!gtk_moz_embed_common_set_pref(G_TYPE_INT, "permissions.default.image", &policy))
      return NS_ERROR_FAILURE;
  }

  if (!aDisable) return NS_OK;

  PRInt32 winCount = 0;
  if (EmbedPrivate::sWindowList)
      winCount = EmbedPrivate::sWindowList->Count();

  for (gint i = 0; i < winCount; i++) {
    EmbedPrivate *tmpPrivate = static_cast<EmbedPrivate *>
                                   (EmbedPrivate::sWindowList->ElementAt(i));
    tmpPrivate->Reload(nsIWebNavigation::LOAD_FLAGS_CHARSET_CHANGE);
  }
  gtk_moz_embed_common_observe(nsnull, nsnull, kMemTopic, (gunichar *)kMemData);
  malloc_trim (0);
  return NS_OK;
}

nsresult
EmbedMemoryFlusher::Init()
{
  nsresult rv;
  StopAndJoin();
  mTimer = do_CreateInstance(NS_TIMER_CONTRACTID);
  NS_ENSURE_STATE(mTimer);
  mPrevImgPolicy = -1;
  mPrevJsStatus = -1;
  mImageEnableLevel = -1;

  PRUint32 timeout = MEM_CHECK_DEFAULT_INTERVAL;
  rv = CheckWatermarkNotifier(kLowMark);
  NS_ENSURE_SUCCESS(rv, rv);
  mIsLowMemory = GetWatermarkValue (kLowMark);
  mMemLevel = MEM_LEVEL_1;
  if (mIsLowMemory) {
    mIsHighMemory = GetWatermarkValue (kHighMark);
    timeout = mIsHighMemory?MEM_CHECK_HIGHMEM_INTERVAL:MEM_CHECK_LOWMEM_INTERVAL;
  }
  gMemThread = g_thread_create (high_mem_thread_func, (void*)this, TRUE, nsnull);
#ifdef OSSO_ENABLE
  mContext = (osso_context_t*)osso_initialize ("gtkmozembed", "0.0.1", TRUE, NULL);
  PRBool suspendEnabled = PR_TRUE;
  if (gtk_moz_embed_common_get_pref(G_TYPE_BOOLEAN, "mozembed.suspend.disabled", &suspendEnabled) && suspendEnabled)
    sSuspendEnabled = PR_FALSE;
  osso_hw_set_event_cb(static_cast<osso_context_t*>(mContext), NULL, *(osso_hw_cb_f *) osso_browser_hw_cb_f, this);
#endif
  return mTimer->InitWithCallback(this, timeout,
                                  nsITimer::TYPE_REPEATING_SLACK);
}

void
EmbedMemoryFlusher::StopAndJoin()
{
  if (mTimer)
    mTimer->Cancel();
#ifdef OSSO_ENABLE
  if (mContext) {
    osso_hw_unset_event_cb(static_cast<osso_context_t*>(mContext), NULL);
    osso_deinitialize(static_cast<osso_context_t*>(mContext));
  }
  mContext = nsnull;
#endif
}

nsresult
EmbedMaemoRunMemoryHandling(void)
{
  PRBool policy = PR_TRUE;
  if (gtk_moz_embed_common_get_pref(G_TYPE_BOOLEAN, "mozembed.memhandler.disabled", &policy) && policy) {
    g_debug("Low memory handling disabled: mozembed.memhandler.disabled\n");
    return NS_OK;
  }
  return sGlobalEmbedMemoryFlusher.Init();
}

nsresult
EmbedMaemoStopMemoryHandling(void)
{
  sGlobalEmbedMemoryFlusher.StopAndJoin();
}
