/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* vim:set ts=2 sw=2 sts=2 tw=80 et cindent: */
/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is mozilla.org code.
 *
 * The Initial Developer of the Original Code is
 * Netscape Communications Corporation.
 * Portions created by the Initial Developer are Copyright (C) 1998
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *   Conrad Carlen <ccarlen@netscape.com>
 *   Oleg Romashin <romaxa@gmail.com>
 *   Andre Pedralho <apedralho@gmail.com>
 *   (from original mozilla/embedding/lite/nsEmbedGlobalHistory.cpp)
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */
/* XXX This file is probably out of sync with its original.
 *     This is a bad thing
 */
#include "EmbedGlobalHistory.h"
#include "nsIObserverService.h"
#include "nsAutoPtr.h"
#include "nsIURI.h"
#include "nsInt64.h"
#include "nsIIOService.h"
#include "nsNetUtil.h"
#include "gtkmozembed_common.h"
#include "nsISeekableStream.h"
#ifndef MOZILLA_INTERNAL_API
#include "nsCRT.h"
#endif
#include "nsILineInputStream.h"

// Constants
#define defaultSeparator 1
// Number of changes in history before automatic flush
static const PRInt32 kNewEntriesBetweenFlush = 20;
static const PRInt32 kMaxSafeReadEntriesCount = 2000;
// Default expiration interval: used if can't get preference service value
static const PRUint32 kDefaultExpirationIntervalDays = 7;
// Mozilla and EAL standard are different each other
static const PRInt64 kMSecsPerDay = LL_INIT(0, 60 * 60 * 24 * 1000);
static const PRInt64 kOneThousand = LL_INIT(0, 1000);
// The history list and the entries counter
static GList *mURLList;                 /** < The history list */
static PRInt64 mExpirationInterval;     /** < Expiration interval time */
static EmbedGlobalHistory *sEmbedGlobalHistory = nsnull;

//*****************************************************************************
// HistoryEntryOld: an entry object and its methods
//*****************************************************************************
class HistoryEntry
{
public:
  PRInt64         mLastVisitTime;     // Millisecs
  PRPackedBool    mWritten;           // TRUE if ever persisted
  nsCString       mTitle;             // The entry title
  nsCString       mUrl;               // The url itself
};

#define CLOSE_FILE_HANDLE(file_handle) \
  PR_BEGIN_MACRO \
    if (file_handle) { \
      close_output_stream(file_handle); \
      NS_RELEASE(file_handle); \
    } \
  PR_END_MACRO

static void close_output_stream(OUTPUT_STREAM *file_handle)
{
  file_handle->Close();
  return;
}

static bool file_handle_uri_exists(LOCAL_FILE *uri)
{
  g_return_val_if_fail(uri, false);
  PRBool exists = PR_FALSE;
  uri->Exists(&exists);
  return exists;
}

static LOCAL_FILE* file_handle_uri_new(const char *uri)
{
  g_return_val_if_fail(uri, nsnull);
  nsresult rv;
  LOCAL_FILE *historyFile = nsnull;
  rv = NS_NewNativeLocalFile(nsDependentCString(uri), 1, &historyFile);
  return historyFile;
}

static void file_handle_uri_release(LOCAL_FILE *uri)
{
  g_return_if_fail(uri);
  NS_RELEASE(uri);
}


static bool file_handle_create_uri(OUTPUT_STREAM **file_handle, LOCAL_FILE *uri)
{
  g_return_val_if_fail(file_handle, false);
  nsresult rv;
  rv = NS_NewLocalFileOutputStream(file_handle, uri, PR_RDWR | PR_APPEND | PR_CREATE_FILE, 0660);

  return NS_SUCCEEDED(rv);
}

static bool file_handle_open_uri(OUTPUT_STREAM **file_handle, LOCAL_FILE *uri)
{
  g_return_val_if_fail(file_handle, false);
  nsresult rv;
  rv = NS_NewLocalFileOutputStream(file_handle, uri, PR_RDWR, 0660);

  return NS_SUCCEEDED(rv);
}

static bool file_handle_seek(OUTPUT_STREAM *file_handle, gboolean end)
{
  g_return_val_if_fail(file_handle, false);
  nsresult rv;
  nsCOMPtr<nsISeekableStream> seekable = do_QueryInterface(file_handle, &rv);
  rv = seekable->Seek(nsISeekableStream::NS_SEEK_SET, end ? -1 : 0);
  return NS_SUCCEEDED(rv);
}

static bool file_handle_truncate(OUTPUT_STREAM *file_handle)
{
  g_return_val_if_fail(file_handle, false);
  nsresult rv;
  nsCOMPtr<nsISeekableStream> seekable = do_QueryInterface(file_handle, &rv);
  rv = seekable->SetEOF();
  return NS_SUCCEEDED(rv);
}

static bool file_handle_write(OUTPUT_STREAM *file_handle, gpointer line)
{
  g_return_val_if_fail(file_handle, 0);
  PRUint32 amt = 0;
  nsresult rv;
  rv = file_handle->Write((char*)line, strlen((char*)line), &amt);
  return NS_SUCCEEDED(rv);
}

// Static Routine Prototypes
static nsresult writeEntry(OUTPUT_STREAM *file_handle, HistoryEntry *entry);

// when an entry is visited
nsresult OnVisited(HistoryEntry *entry)
{
  NS_ENSURE_ARG(entry);
  entry->mLastVisitTime = PR_Now();
  LL_DIV(entry->mLastVisitTime, entry->mLastVisitTime, kOneThousand);
  return NS_OK;
}

// Return the last time an entry was visited
PRInt64 GetLastVisitTime(HistoryEntry *entry)
{
  NS_ENSURE_ARG(entry);
  return entry->mLastVisitTime;
}

// Change the last time an entry was visited
nsresult SetLastVisitTime(HistoryEntry *entry, const PRInt64& aTime)
{
  NS_ENSURE_ARG(entry);
  NS_ENSURE_ARG_POINTER(aTime);
  entry->mLastVisitTime = aTime;
  return NS_OK;
}

// Return TRUE if an entry has been written
PRBool GetIsWritten(HistoryEntry *entry)
{
  NS_ENSURE_ARG(entry);
  return entry->mWritten;
}

// Set TRUE when an entry is visited
nsresult SetIsWritten(HistoryEntry *entry)
{
  NS_ENSURE_ARG(entry);
  entry->mWritten = PR_TRUE;
  return NS_OK;
}

// Change the entry title
nsresult SET_URL(HistoryEntry *aEntry, const char *aUrl)
{
  NS_ENSURE_ARG(aEntry);
  NS_ENSURE_ARG(aUrl);
  aEntry->mUrl.Assign(aUrl);
  return NS_OK;
}

// Return the entry url
const char* GET_URL(HistoryEntry *aEntry)
{
  return (aEntry && !aEntry->mUrl.IsEmpty()) ? aEntry->mUrl.get() : "";
}

// Return the entry url
const char* GET_HOST(HistoryEntry *aEntry)
{
  nsCOMPtr<nsIURI> aURI;
  nsresult rv = NS_NewURI(getter_AddRefs(aURI), GET_URL(aEntry));
  if (NS_FAILED(rv))
    return GET_URL(aEntry);
  aURI->GetHost(aEntry->mTitle);
  if (aEntry->mTitle.IsEmpty())
    return GET_URL(aEntry);
  return aEntry->mTitle.get();
}

// Change the entry title
#define SET_TITLE(entry, aTitle) (entry->mTitle.Assign(aTitle))

// Return the entry title
#define GET_TITLE(entry) (entry && !entry->mTitle.IsEmpty() ? entry->mTitle.get() : GET_HOST(entry))

// Traverse the history list trying to find a frame
int history_entry_find_exist(gconstpointer a, gconstpointer b)
{
  return g_ascii_strcasecmp((char*)GET_URL((HistoryEntry *)a), (char *) b);
}

// Traverse the history list looking for the correct place to add a new item
int find_insertion_place(gconstpointer a, gconstpointer b)
{
  PRInt64 lastVisitTime = GetLastVisitTime((HistoryEntry *) a);
  PRInt64 tempTime = GetLastVisitTime((HistoryEntry *) b);
  return LL_CMP(lastVisitTime, <, tempTime);
}

// Check whether an entry has expired
PRBool entryHasExpired(HistoryEntry *entry)
{
  // convert "now" from microsecs to millisecs
  PRInt64 nowInMilliSecs = PR_Now();
  LL_DIV(nowInMilliSecs, nowInMilliSecs, kOneThousand);
  // determine when the entry would have expired
  PRInt64 expirationIntervalAgo;
  LL_SUB(expirationIntervalAgo, nowInMilliSecs, mExpirationInterval);
  PRInt64 lastVisitTime = GetLastVisitTime(entry);
  return LL_CMP(lastVisitTime, <, expirationIntervalAgo);
}

// Traverse the history list to get all the entry data
void history_entry_foreach_to_remove(gpointer data, gpointer user_data)
{
  HistoryEntry *entry = (HistoryEntry *) data;
  if (entry) {
    entry->mLastVisitTime = 0;
    delete entry;
  }
}

//*****************************************************************************
// EmbedGlobalHistory - Creation/Destruction
//*****************************************************************************
NS_IMPL_ISUPPORTS2(EmbedGlobalHistory, nsIGlobalHistory2, nsIObserver)
/* static */
EmbedGlobalHistory*
EmbedGlobalHistory::GetInstance()
{
  if (!sEmbedGlobalHistory)
  {
    sEmbedGlobalHistory = new EmbedGlobalHistory();
    if (!sEmbedGlobalHistory)
      return nsnull;
    NS_ADDREF(sEmbedGlobalHistory);   // addref the global
    if (NS_FAILED(sEmbedGlobalHistory->Init()))
    {
      NS_RELEASE(sEmbedGlobalHistory);
      sEmbedGlobalHistory = nsnull;
      return nsnull;
    }
  }
  else
    NS_ADDREF(sEmbedGlobalHistory);   // addref the return result
  return sEmbedGlobalHistory;
}

/* static */
void
EmbedGlobalHistory::DeleteInstance()
{
  if (sEmbedGlobalHistory)
    delete sEmbedGlobalHistory;
  sEmbedGlobalHistory = nsnull;
}

// The global history component constructor
EmbedGlobalHistory::EmbedGlobalHistory()
: mFileHandle(nsnull)
{
  if (!mURLList) {
    mDataIsLoaded = PR_FALSE;
    mFlushModeFullWriteNeeded = PR_FALSE;
    mEntriesAddedSinceFlush = 0;
    mHistoryFile = nsnull;
    LL_I2L(mExpirationInterval, kDefaultExpirationIntervalDays);
    LL_MUL(mExpirationInterval, mExpirationInterval, kMSecsPerDay);
  }
}

// The global history component destructor
EmbedGlobalHistory::~EmbedGlobalHistory()
{
  LoadData();
  mFlushModeFullWriteNeeded = PR_TRUE;
  FlushData();
  if (mURLList) {
    g_list_foreach(mURLList, (GFunc) history_entry_foreach_to_remove, NULL);
    g_list_free(mURLList);
    mURLList = NULL;
  }
  if (mFileHandle) {
    CLOSE_FILE_HANDLE(mFileHandle);
  }
  if (mHistoryFile) {
    NS_Free(mHistoryFile);
    mHistoryFile = nsnull;
  }
  if (sEmbedGlobalHistory)
    sEmbedGlobalHistory = nsnull;
}

// Initialize the global history component
NS_IMETHODIMP EmbedGlobalHistory::Init()
{
  if (mURLList)
    return NS_OK;
  // Get Pref and convert to millisecs

  PRInt32 expireDays;
  int success = gtk_moz_embed_common_get_pref(G_TYPE_INT, EMBED_HISTORY_PREF_EXPIRE_DAYS, &expireDays);
  if (success) {
    LL_I2L(mExpirationInterval, expireDays);
    LL_MUL(mExpirationInterval, mExpirationInterval, kMSecsPerDay);
  }
  // register to observe profile changes
  nsCOMPtr<nsIObserverService> observerService =
    do_GetService(NS_OBSERVERSERVICE_CONTRACTID);
  NS_ASSERTION(observerService, "failed to get observer service");
  if (observerService) {
    observerService->AddObserver(this, "quit-application", PR_FALSE);
    observerService->AddObserver(this, "history-item-removed", PR_FALSE);
  }
  nsresult rv = InitFile();
  if (NS_FAILED(rv))
    return NS_ERROR_FAILURE;
  rv = LoadData();
  NS_ENSURE_SUCCESS(rv, rv);
  return rv;
}

#define BROKEN_RV_HANDLING_CODE(rv) PR_BEGIN_MACRO                        \
  if (NS_FAILED(rv)) {                                                    \
    /* OOPS the coder (not timeless) didn't handle this case well at all. \
     * unfortunately the coder will remain anonymous.                     \
     * XXX please fix me.                                                 \
     */                                                                   \
  }                                                                       \
  PR_END_MACRO

#define BROKEN_STRING_GETTER(out) PR_BEGIN_MACRO                      \
  /* OOPS the coder (not timeless) decided not to do anything in this \
   * method, but to return NS_OK anyway. That's not very polite.      \
   */                                                                 \
  out.Truncate();                                                     \
  PR_END_MACRO

#define BROKEN_STRING_BUILDER(var) PR_BEGIN_MACRO \
  /* This is just wrong */                        \
  PR_END_MACRO

#define UNACCEPTABLE_CRASHY_GLIB_ALLOCATION(newed) PR_BEGIN_MACRO \
  /* OOPS this code is using a glib allocation function which     \
   * will cause the application to crash when it runs out of      \
   * memory. This is not cool. either g_try methods should be     \
   * used or malloc, or new (note that gecko /will/ be replacing  \
   * its operator new such that new will not throw exceptions).   \
   * XXX please fix me.                                           \
   */                                                             \
  if (!newed) {                                                   \
  }                                                               \
  PR_END_MACRO

#define ALLOC_NOT_CHECKED(newed) PR_BEGIN_MACRO               \
  /* This might not crash, but the code probably isn't really \
   * designed to handle it, perhaps the code should be fixed? \
   */                                                         \
  if (!newed) {                                               \
  }                                                           \
  PR_END_MACRO

//*****************************************************************************
// EmbedGlobalHistory::nsIGlobalHistory
//*****************************************************************************
// Add a new URI to the history
NS_IMETHODIMP EmbedGlobalHistory::AddURI(nsIURI *aURI, PRBool aRedirect, PRBool aToplevel, nsIURI *aReferrer)
{
  NS_ENSURE_ARG(aURI);
  nsCAutoString URISpec;
  aURI->GetSpec(URISpec);
  const char *aURL = URISpec.get();
  if (!aToplevel)
    return NS_OK;
  PRBool isHTTP, isHTTPS;
  nsresult rv = NS_OK;
  rv |= aURI->SchemeIs("http", &isHTTP);
  rv |= aURI->SchemeIs("https", &isHTTPS);
  NS_ENSURE_SUCCESS(rv, NS_ERROR_FAILURE);
  // Only get valid uri schemes
  if (!isHTTP && !isHTTPS)
  {
    /* the following blacklist is silly.
     * if there's some need to whitelist http(s) + ftp,
     * that's what we should do.
     */
    PRBool isAbout, isImap, isNews, isMailbox, isViewSource, isChrome, isData, isJavascript, isLocal;
    rv  = aURI->SchemeIs("about", &isAbout);
    rv |= aURI->SchemeIs("imap", &isImap);
    rv |= aURI->SchemeIs("news", &isNews);
    rv |= aURI->SchemeIs("file", &isLocal);
    rv |= aURI->SchemeIs("mailbox", &isMailbox);
    rv |= aURI->SchemeIs("view-source", &isViewSource);
    rv |= aURI->SchemeIs("chrome", &isChrome);
    rv |= aURI->SchemeIs("data", &isData);
    rv |= aURI->SchemeIs("javascript", &isJavascript);
    NS_ENSURE_SUCCESS(rv, NS_ERROR_FAILURE);
    if (isAbout ||
        isImap ||
        isNews ||
        isLocal ||
        isMailbox ||
        isViewSource ||
        isChrome ||
        isData ||
        isJavascript) {
      return NS_OK;
    }
  }

  rv = LoadData();
  if (NS_FAILED(rv))
    return rv;

  GList *node = g_list_find_custom(mURLList, aURL, (GCompareFunc) history_entry_find_exist);
  HistoryEntry *entry = NULL;
  if (node && node->data)
    entry = (HistoryEntry *)(node->data);
  nsCAutoString hostname;
  aURI->GetHost(hostname);

  // It is not in the history: add a new entry
  if (!entry) {
    entry = new HistoryEntry;
    if (!entry)
      return NS_ERROR_OUT_OF_MEMORY;

    rv = OnVisited(entry);
    if (NS_FAILED(rv)) {
      NS_Free(entry);
      return rv;
    }

    SET_TITLE(entry, hostname);
    rv = SET_URL(entry, aURL);
    if (NS_FAILED(rv)) {
      NS_Free(entry);
      return rv;
    }

    unsigned int listSize = g_list_length(mURLList);
    if (listSize+1 > kDefaultMaxSize) {
      GList *last = g_list_last(mURLList);
      mURLList = g_list_remove(mURLList, last->data);
    }
    mURLList = g_list_insert_sorted(mURLList, entry,
                                    (GCompareFunc) find_insertion_place);

    // Flush after kNewEntriesBetweenFlush changes
    if (++mEntriesAddedSinceFlush >= kNewEntriesBetweenFlush) {
      rv = FlushData();
      if (NS_FAILED(rv))
        return rv;
    }
  }
  // It is already in the history: just update the entry
  else {
    // update the last visited time
    rv = OnVisited(entry);
    if (NS_FAILED(rv))
      return rv;

    SET_TITLE(entry, hostname);
    // Move the element to the start of the list
    mURLList = g_list_remove(mURLList, entry);
    mURLList = g_list_insert_sorted(mURLList, entry, (GCompareFunc) find_insertion_place);

    // Flush after kNewEntriesBetweenFlush changes
    mFlushModeFullWriteNeeded = PR_TRUE;
    if (++mEntriesAddedSinceFlush >= kNewEntriesBetweenFlush) {
      rv = FlushData();
      if (NS_FAILED(rv))
        return rv;
    }
  }
  return NS_OK;
}

// Return TRUE if the url is already in history
NS_IMETHODIMP EmbedGlobalHistory::IsVisited(nsIURI *aURI, PRBool *_retval)
{
  NS_ENSURE_ARG(aURI);
  NS_ENSURE_ARG_POINTER(_retval);
  nsCAutoString URISpec;
  aURI->GetSpec(URISpec);
  const char *aURL = URISpec.get();
  nsresult rv = LoadData();
  if (NS_FAILED(rv))
    return rv;

  // find the element in the history list
  GList *node = g_list_find_custom(mURLList, aURL,
                                   (GCompareFunc) history_entry_find_exist);
  *_retval = (node && node->data);
  return NS_OK;
}

// It is called when Mozilla get real name of a URL
NS_IMETHODIMP EmbedGlobalHistory::SetPageTitle(nsIURI *aURI,
                                               const nsAString & aTitle)
{
  NS_ENSURE_ARG(aURI);
  nsresult rv;
  // skip about: URIs to avoid reading in the db (about:blank, especially)
  PRBool isAbout;
  rv = aURI->SchemeIs("about", &isAbout);
  NS_ENSURE_SUCCESS(rv, rv);
  if (isAbout)
    return NS_OK;

  // load the stored history
  rv = LoadData();
  NS_ENSURE_SUCCESS(rv, rv);

  nsCAutoString URISpec;
  aURI->GetSpec(URISpec);
  const char *aURL = URISpec.get();

  // find the entry in the history list
  GList *node = g_list_find_custom(mURLList, aURL,
                                   (GCompareFunc) history_entry_find_exist);
  HistoryEntry *entry = NULL;

  // entry not found
  if (!node)
    return NS_ERROR_FAILURE;

  entry = (HistoryEntry *)(node->data);

  // change the entry content
  if (!entry) {
    return NS_ERROR_FAILURE;
  }
  else {
    SET_TITLE(entry, NS_ConvertUTF16toUTF8(aTitle).get());

    // flush after kNewEntriesBetweenFlush entries found
    mFlushModeFullWriteNeeded = PR_TRUE;
    if (++mEntriesAddedSinceFlush >= kNewEntriesBetweenFlush) {
      rv = FlushData();
      if (NS_FAILED(rv))
        return NS_ERROR_FAILURE;
    }
  }

  return NS_OK;
}

nsresult EmbedGlobalHistory::RemoveEntries(const PRUnichar *url, int time)
{
  nsresult rv;

  // there is no history list
  if (!mURLList)
    return NS_ERROR_FAILURE;

  // the item to be removed is a URL
  if (url) {
    GList *node = g_list_find_custom(mURLList, NS_ConvertUTF16toUTF8(url).get(), (GCompareFunc) history_entry_find_exist);
    if (!node)
      return NS_ERROR_FAILURE;
    if (!(node->data)) {
      return NS_ERROR_FAILURE;
    }
    else {
      HistoryEntry *entry = static_cast<HistoryEntry *>
                                           (node->data);

      entry->mLastVisitTime = 0;
      mURLList = g_list_remove_link(mURLList, node);
      delete entry;
    }
  }
  // the item to be removed is a folder
  else {
    g_list_foreach (mURLList, (GFunc) history_entry_foreach_to_remove, NULL);
    g_list_free(mURLList);
    mURLList = NULL;
  }

  // flush after kNewEntriesBetweenFlush changes
  mFlushModeFullWriteNeeded = PR_TRUE;
  if (++mEntriesAddedSinceFlush >= kNewEntriesBetweenFlush) {
    rv = FlushData();
    if (NS_FAILED(rv))
      return NS_ERROR_FAILURE;
  }
  return NS_OK;
}

//*****************************************************************************
// EmbedGlobalHistory::nsIObserver
//*****************************************************************************
NS_IMETHODIMP EmbedGlobalHistory::Observe(nsISupports *aSubject,
                                          const char *aTopic,
                                          const PRUnichar *aData)
{
  nsresult rv;

  // used when the browser is closed and the EmbedGlobalHistory destructor is not called
  if (strcmp(aTopic, "quit-application") == 0) {
    // load the history data to ensure that all the history items stored in the file were properly loaded
    rv = LoadData();
    if (NS_FAILED(rv))
      return NS_ERROR_FAILURE;
    // flush the history data
    rv = FlushData();
    if (NS_FAILED(rv))
      return NS_ERROR_FAILURE;
    // free the history list
    if (mURLList) {
      g_list_foreach(mURLList, (GFunc) history_entry_foreach_to_remove, NULL);
      g_list_free(mURLList);
      mURLList = NULL;
    }
    // close the history file
    if (mFileHandle) {
      CLOSE_FILE_HANDLE(mFileHandle);
    }
  }
  // used when the user wants to remove a history item
  else if (strcmp(aTopic, "history-item-removed") == 0) {
    rv = RemoveEntries(aData, 0);
    if(NS_FAILED(rv))
      return NS_ERROR_FAILURE;
  }
  return NS_OK;
}

static nsresult
GetHistoryFileName(char **aHistoryFile)
{
  NS_ENSURE_ARG_POINTER(aHistoryFile);
  // Get the history file in our profile dir.
  // Notice we are not just getting NS_APP_HISTORY_50_FILE
  // because it is used by the "real" global history component.
  nsCAutoString tempDirString;
  // there is a profile set
  if (EmbedPrivate::sProfileDir) {
    nsCString path;
    EmbedPrivate::sProfileDir->GetNativePath(path);

    tempDirString.Assign(path.get());
  }
  // there is no profile set
  else
  {
    // get the temporary directory
    tempDirString.Assign(g_get_tmp_dir());
  }
  // the history file name
  tempDirString.Append(NS_LITERAL_CSTRING("/history.txt"));
  // return the full history path
  *aHistoryFile = ToNewCString(tempDirString);
  if (!(*aHistoryFile))
    return NS_ERROR_OUT_OF_MEMORY;
  return NS_OK;
}

//*****************************************************************************
// EmbedGlobalHistory
//*****************************************************************************
// Open/Create the history.txt file if it does not exist
nsresult EmbedGlobalHistory::InitFile()
{
  // the history file is not initialized
  if (!mHistoryFile)
    if (NS_FAILED(GetHistoryFileName(&mHistoryFile)))
      return NS_ERROR_FAILURE;

  LOCAL_FILE *uri = file_handle_uri_new(mHistoryFile);
  if (!uri)
    return NS_ERROR_FAILURE;

  gboolean rs = FALSE;
  // there is not a history file
  if (!file_handle_uri_exists(uri)) {
    // create the history file
    if (!file_handle_create_uri(&mFileHandle, uri)) {
      NS_WARNING("Could not create a history file");
      file_handle_uri_release(uri);
      return NS_ERROR_FAILURE;
    }
    // close the history file
    CLOSE_FILE_HANDLE(mFileHandle);
  }
  // open the history file
  rs = file_handle_open_uri(&mFileHandle, uri);

  file_handle_uri_release(uri);

  if (!rs) {
    NS_WARNING("Could not open a history file");
    return NS_ERROR_FAILURE;
  }
  return NS_OK;
}

// Get the data from history.txt file
nsresult EmbedGlobalHistory::LoadData()
{
  nsresult rv;
  // the data is not loaded
  if (!mDataIsLoaded) {
    // create the history file handle
    LOCAL_FILE *uri = file_handle_uri_new(mHistoryFile);
    // could not create a history file handle
    if (!uri) {
      return NS_ERROR_FAILURE;
    }
    else {
      rv = ReadEntries(uri);
      // could not read the history entries
      if (NS_FAILED(rv))
        return NS_ERROR_FAILURE;
      file_handle_uri_release(uri);
    }
    mDataIsLoaded = PR_TRUE;
  }
  return NS_OK;
}

nsresult EmbedGlobalHistory::WriteEntryIfWritten(GList *list, OUTPUT_STREAM *file_handle)
{
  NS_ENSURE_ARG_POINTER(file_handle);

  // get the history length;
  unsigned int counter = g_list_length(list);

  // for each history entry
  while (counter > 0) {
    // get a history entry
    HistoryEntry *entry = static_cast<HistoryEntry*>(g_list_nth_data(list, counter-1));
    counter--;

    // could not get a history entry
    if (!entry)
      return NS_ERROR_OUT_OF_MEMORY;

    // the entry has expired so does not store it in the history file
    if (entryHasExpired(entry))
      continue;

    // write the entry in the history file
    writeEntry(file_handle, entry);
  }
  return NS_OK;
}

nsresult EmbedGlobalHistory::WriteEntryIfUnwritten(GList *list, OUTPUT_STREAM *file_handle)
{
  NS_ENSURE_ARG_POINTER(file_handle);

  // get the history length;
  unsigned int counter = g_list_length(list);

  // for each history entry
  while (counter > 0) {
    // get a history entry
    HistoryEntry *entry = static_cast<HistoryEntry*>(g_list_nth_data(list, counter-1));
    counter--;

    // could not get a history entry
    if (!entry)
      return NS_ERROR_OUT_OF_MEMORY;

    // the entry has expired so don't store it in the history file
    if(entryHasExpired(entry))
      continue;

    // the entry has not been written
    if (!GetIsWritten(entry))
      writeEntry(file_handle, entry);
  }
  return NS_OK;
}

nsresult EmbedGlobalHistory::FlushData()
{
  nsresult rv;
  // there are no history entries to be added
  if (mEntriesAddedSinceFlush == 0)
    return NS_OK;

  // there is no history file handle
  if (!mHistoryFile)
  {
    // create the history file
    rv = InitFile();
    NS_ENSURE_SUCCESS(rv, rv);

    mFlushModeFullWriteNeeded = PR_TRUE;
    rv = FlushData();
    NS_ENSURE_SUCCESS(rv, rv);
  }

  if (mFlushModeFullWriteNeeded)
  {
    if (!file_handle_seek(mFileHandle, FALSE))
      return NS_ERROR_FAILURE;
    if (!file_handle_truncate(mFileHandle))
      return NS_ERROR_FAILURE;
    WriteEntryIfWritten(mURLList, mFileHandle);
    mFlushModeFullWriteNeeded = PR_FALSE;
  }
  // just need to append some entries
  else
  {
    // go to the end of the history file
    if (!file_handle_seek(mFileHandle, TRUE))
      return NS_ERROR_FAILURE;
    WriteEntryIfUnwritten(mURLList, mFileHandle);
  }
  mEntriesAddedSinceFlush = 0;
  return NS_OK;
}

// Split an entry in last visit time, title and url.
// Add a stored entry in the history.txt file in the history list
nsresult EmbedGlobalHistory::GetEntry(nsCString& string)
{
  char *entry = string.BeginWriting();
  PRInt32 str_len = string.Length();
  char separator = (char) defaultSeparator;
  int pos = 0;
  nsInt64 outValue = 0;
  // get the last visit time from the given entry
  while (PR_TRUE) {
    PRInt32 digit;
    // it is a separator: get the next digit
    if (entry[pos] == separator) {
      pos++;
      break;
    }
    // it is the end of a word or is not a number
    if (str_len <= pos || entry[pos] == '\0' || !isdigit(entry[pos]))
      return NS_ERROR_FAILURE;
    digit = entry[pos] - '0';
    // add the new digit found to the previous ones
    outValue *= nsInt64(10);
    outValue += nsInt64(digit);
    pos++;
  }
  char *url=&entry[pos];
  int urlLength= 0, titleLength= 0, numStrings=1;
  // get the url and title from the given entry
  // FIXME
  while(PR_TRUE) {
    if (str_len <= pos) break;
    // it is a separator: get the next character
    if (entry[pos] == separator) {
      numStrings++;
      pos++;
      continue;
    }
    // there are no more strings to be found
    if (numStrings > 2)
      break;
    // it is the first string found: the url
    if (numStrings == 1)
      urlLength++;
    // it is the second string found: the title
    else
      titleLength++;
    pos++;
  }
  // terminate the strings properly
  if (!urlLength || !titleLength)
    return NS_ERROR_FAILURE;

  url[urlLength]='\0';
  char *title=&url[urlLength + 1];
  title[titleLength]='\0';

  // create a new history entry
  HistoryEntry *newEntry = new HistoryEntry;
  if (!newEntry)
    return NS_ERROR_OUT_OF_MEMORY;

  nsresult rv;
  // initialize the title of the new entry
  SET_TITLE(newEntry, title);
  // initialize the last visit time of the new entry
  rv = SetLastVisitTime(newEntry, outValue);
  if (NS_FAILED(rv))
    return NS_ERROR_FAILURE;
  // initialize the isWritten flag of the new entry: it is in the history file
  rv = SetIsWritten(newEntry);
  if (NS_FAILED(rv))
    return NS_ERROR_FAILURE;
  // initialize the url of the new entry
  rv = SET_URL(newEntry, url);
  if (NS_FAILED(rv))
    return NS_ERROR_FAILURE;

  // Check whether the entry has expired and then add it to the history list
  if (!entryHasExpired(newEntry))
    mURLList = g_list_prepend(mURLList, newEntry);
  return NS_OK;
}

// Get the history entries from history.txt file
nsresult EmbedGlobalHistory::ReadEntries(LOCAL_FILE *file_uri)
{
  // the history file handle has not been initialized
  if (!file_uri)
    return NS_ERROR_FAILURE;

  nsresult rv = NS_OK;
  // create an input stream
  nsCOMPtr<nsIInputStream> fileStream;
  NS_NewLocalFileInputStream(getter_AddRefs(fileStream), file_uri);
  if (!fileStream)
    return NS_ERROR_OUT_OF_MEMORY;
  nsCOMPtr<nsILineInputStream> lineStream = do_QueryInterface(fileStream, &rv);
  NS_ASSERTION(lineStream, "File stream is not an nsILineInputStream");
  // Read the header
  nsCString utf8Buffer;
  PRBool moreData = PR_FALSE;

  PRInt32 safe_limit = 0;
  // get the file content line by line
  do {
    rv = lineStream->ReadLine(utf8Buffer, &moreData);
    safe_limit++;
    if (NS_FAILED(rv))
      break;

    // go to the next line: might be a blank line in the file
    if (utf8Buffer.IsEmpty())
      continue;

    // add the entry to the history
    rv = GetEntry(utf8Buffer);
    if (NS_FAILED(rv))
      break;
  } while (moreData && safe_limit < kMaxSafeReadEntriesCount);
  // close the file stream
  fileStream->Close();
  return rv;
}

//*****************************************************************************
// Static Functions
//*****************************************************************************
// Transform the last visit time from a PRInt64 to a string
static nsresult writePRInt64(char time[14], const PRInt64& inValue)
{
  nsInt64 value(inValue);
  // the inValue is 0, then return a "0" string
  if (value == nsInt64(0)) {
    strncpy(time, "0", 1);
    return NS_OK;
  }
  nsCAutoString tempString;
  // transform the PRInt64 in a nsCAutoString, digit by digit
  while (value != nsInt64(0)) {
    PRInt32 ones = PRInt32(value % nsInt64(10));
    value /= nsInt64(10);
    tempString.Insert(char('0' + ones), 0);
  }
  // transform the nsCAutoString to a string
  strncpy(time,(char *) tempString.get(), 14);
  return NS_OK;
}

// Write an entry in the history.txt file
nsresult writeEntry(OUTPUT_STREAM *file_handle, HistoryEntry *entry)
{
  nsresult rv;
  char time[14];
  nsCAutoString tempString;

  // get the string equivalent to a PRInt64 value
  writePRInt64(time, GetLastVisitTime(entry));

  // add the time string to the entry line
  tempString.Assign(time);
  // add a separator to the entry line
  tempString.Append(defaultSeparator);
  // add the url string to the entry line
  tempString.Append(GET_URL(entry));
  // add a separator to the entry line
  tempString.Append(defaultSeparator);
  // add the title string to the entry line
  tempString.Append(GET_TITLE(entry));
  // add a separator to the entry line
  tempString.Append(defaultSeparator);
  // add the line separator to the entry line
  tempString.Append("\n");

  // write the new line to the file
  guint64 size = file_handle_write(file_handle, (gpointer) tempString.get());

  // could not write all the line to the file
  if (size != tempString.Length())
    rv = NS_ERROR_OUT_OF_MEMORY;
  rv = SetIsWritten(entry);
  if (NS_FAILED(rv))
    return rv;
  return NS_OK;
}

nsresult EmbedGlobalHistory::GetContentList(GtkMozHistoryItem **aHistoryVector, int *aCounter)
{
  // the history list has not been initialized
  if (!mURLList)
    return NS_ERROR_FAILURE;

  // create a new history vector
  *aHistoryVector = g_try_new0(GtkMozHistoryItem, g_list_length(mURLList));
  if (!(*aHistoryVector))
    return NS_ERROR_OUT_OF_MEMORY;

  GtkMozHistoryItem * aItem = (GtkMozHistoryItem *)*aHistoryVector;
  // get the history items one by one
  while ((uint)*aCounter < g_list_length(mURLList)) {
    HistoryEntry *entry = static_cast<HistoryEntry*>
                                         (g_list_nth_data(mURLList, *aCounter));
    // verify if the entry has expired and discard it
    if (entryHasExpired(entry))
      break;

    glong accessed;
    // transform the last visit time to a readable value in microsecs
    PRInt64 temp, outValue;
    LL_MUL(outValue, GetLastVisitTime(entry), kOneThousand);
    LL_DIV(temp, outValue, PR_USEC_PER_SEC);
    LL_L2I(accessed, temp);
    // set history vector title value
    aItem[*aCounter].title = (const char *) GET_TITLE(entry);

    if (!aItem[*aCounter].title)
      return NS_ERROR_OUT_OF_MEMORY;
    // set history vector url value
    aItem[*aCounter].url = (const char *)GET_URL(entry);
    if (!aItem[*aCounter].url)
      return NS_ERROR_OUT_OF_MEMORY;
    // set history vector accessed flag
    aItem[*aCounter].accessed = accessed;
    // count one more history item
    (*aCounter)++;
  }
  return NS_OK;
}
