/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* vim:set ts=2 sw=2 sts=2 tw=80 et cindent: */
/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is mozilla.org code.
 *
 * The Initial Developer of the Original Code is
 * Christopher Blizzard.
 * Portions created by the Initial Developer are Copyright (C) 2001
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *   Christopher Blizzard <blizzard@mozilla.org>
 *   Oleg Romashin <romaxa@gmail.com>
 *   Antonio Gomes <tonikitoo@gmail.com>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */
/**
 * Derived from GContentHandler http://landfill.mozilla.org/mxr-test/gnome/source/galeon/mozilla/ContentHandler.cpp
 */
#include "EmbedDownloadMgr.h"
#include "EmbedGtkTools.h"
#ifdef MOZILLA_INTERNAL_API
#include "nsXPIDLString.h"
#else
#include "nsComponentManagerUtils.h"
#endif
#include "nsIChannel.h"
#include "nsIWebProgress.h"
#include "nsIDOMWindow.h"
#include "nsCRT.h"
#include "nsIPromptService.h"
#include "nsIWebProgressListener2.h"
#include "nsCOMPtr.h"
#include "nsIServiceManager.h"
#include "nsIInterfaceRequestorUtils.h"
#include "nsIURI.h"
#include "nsIURL.h"
#include "nsIFile.h"
#include "nsIDOMWindow.h"
#include "nsIExternalHelperAppService.h"
#include "nsCExternalHandlerService.h"
#include "nsMemory.h"
#include "nsNetError.h"
#include "nsIStreamListener.h"
#include "nsIFile.h"
#include "nsILocalFile.h"
#include "nsNetCID.h"
#include <unistd.h>
#include "gtkmozembed_download.h"
#include "nsIIOService.h"
#include "nsIProgressEventSink.h"
#include "nsNetUtil.h"
#include "nsIDOMWindowInternal.h"
#include "nsIPrompt.h"
#ifdef MOZ_RDF
#include "nsIExtensionManager.h"
#endif
#ifdef MOZ_XPINSTALL
#include "nsIXPIProgressDialog.h"
#endif
#include "gtkmozembed_hildon.h"
#include "nsICategoryManager.h"
#include "nsIPrefBranch2.h"
#include "nsIPrefService.h"
#include <sys/types.h>
#include <signal.h>

#define UNKNOWN_FILE_SIZE -1
#ifdef MOZ_TRUNK_BUILD
#undef OUTPUT_HANDLER_IMPL
#endif

#include "nsIStringBundle.h"
static const char kEMPropertiesURL[] = "chrome://mozapps/locale/extensions/extensions.properties";
static const char kXIPropertiesURL[] = "chrome://global/locale/xpinstall/xpinstall.properties";
static nsIStringBundle* sEMBundle = nsnull;
static nsIStringBundle* sXIBundle = nsnull;

class EmbedDownloadMgr;

static void ObserveData(nsISupports *aObject, const char *aTopic, const PRUnichar *aData)
{
    nsCOMPtr <nsIObserverService> observerService = do_GetService("@mozilla.org/observer-service;1");
    if (observerService)
        observerService->NotifyObservers(aObject, aTopic, aData);
}

class ProgressListener : public nsIWebProgressListener2
#ifdef MOZ_RDF
#ifndef MOZ_TRUNK_BUILD
                       , public nsIAddonUpdateListener
#else
                       , public nsIAddonInstallListener
#endif
                       , public nsIAddonUpdateCheckListener
#endif
{
public:
    ProgressListener(EmbedDownload *aDownload):mDownload(aDownload),mObserveIndex(-1),mProgressState(-1)
    {
      //g_critical("Func:%s::%d\n", __PRETTY_FUNCTION__, __LINE__);
      mUpdatesGlist = nsnull;
      mProgressDialog = nsnull;
      mProgressBar = nsnull;
      mTimer_id = 0;
      mPulseCount = 0;
      mIsInstall = PR_TRUE;
    }

    ~ProgressListener(void)
    {
      //g_critical("Func:%s::%d\n", __PRETTY_FUNCTION__, __LINE__);
    }

    NS_DECL_ISUPPORTS
    NS_DECL_NSIWEBPROGRESSLISTENER
    NS_DECL_NSIWEBPROGRESSLISTENER2
#ifdef MOZ_RDF
#ifndef MOZ_TRUNK_BUILD
    NS_DECL_NSIADDONUPDATELISTENER
#else
    NS_DECL_NSIADDONINSTALLLISTENER
#endif
    NS_DECL_NSIADDONUPDATECHECKLISTENER
#endif
    void SetupDownload(EmbedDownload *aDownload) { mDownload = aDownload; };

    GtkWidget *mProgressDialog;
    GtkWidget *mProgressBar;
    guint mTimer_id;
    guint mPulseCount;
    EmbedDownload *mDownload;
    PRInt32 mObserveIndex;
    PRInt32 mProgressState;
    PRBool mIsInstall;
private:
    GList *mUpdatesGlist;
};

class FetchObserver : public nsIRequestObserver
                      ,public nsIProgressEventSink
{
public:
    FetchObserver(EmbedDownload *aDownload):mDownload(aDownload)
    {
    }

    ~FetchObserver(void)
    {
    }

    NS_DECL_ISUPPORTS
    NS_DECL_NSIREQUESTOBSERVER
    NS_DECL_NSIPROGRESSEVENTSINK
    EmbedDownload *mDownload;
};

NS_IMPL_ISUPPORTS2(FetchObserver, nsIRequestObserver, nsIProgressEventSink)

NS_IMPL_ADDREF(ProgressListener)
NS_IMPL_RELEASE(ProgressListener)
NS_INTERFACE_MAP_BEGIN(ProgressListener)
NS_INTERFACE_MAP_ENTRY(nsIWebProgressListener2)
NS_INTERFACE_MAP_ENTRY(nsIWebProgressListener)
#ifdef MOZ_RDF
#ifndef MOZ_TRUNK_BUILD
NS_INTERFACE_MAP_ENTRY_AMBIGUOUS(nsISupports, nsIAddonUpdateListener)
NS_INTERFACE_MAP_ENTRY(nsIAddonUpdateListener)
#else
NS_INTERFACE_MAP_ENTRY_AMBIGUOUS(nsISupports, nsIAddonInstallListener)
NS_INTERFACE_MAP_ENTRY(nsIAddonInstallListener)
#endif
NS_INTERFACE_MAP_ENTRY(nsIAddonUpdateCheckListener)
#endif
NS_INTERFACE_MAP_END

NS_IMPL_ADDREF(EmbedDownloadMgr)
NS_IMPL_RELEASE(EmbedDownloadMgr)
NS_INTERFACE_MAP_BEGIN(EmbedDownloadMgr)
NS_INTERFACE_MAP_ENTRY(nsIHelperAppLauncherDialog)
#ifdef MOZ_XPINSTALL
NS_INTERFACE_MAP_ENTRY(nsIXPIDialogService)
#endif
NS_INTERFACE_MAP_END

static void destroy_progress_dialog(ProgressListener *listener);
static gboolean progress_bar_pulse(void *bar);
static gboolean find_updates_timeout(void *bar);
static gboolean install_progress_timeout(void *bar);
static gboolean select_updates_timeout(void *bar);


EmbedDownload::EmbedDownload(void)
  : parent(nsnull), gtkMozEmbedParentWidget(nsnull), launcher(nsnull), request(nsnull), isCanceled(PR_FALSE)
{
}

EmbedDownload::~EmbedDownload(void)
{
  isCanceled = PR_TRUE;
  request = nsnull;
}

EmbedDownloadMgr::EmbedDownloadMgr(void)
{
  mDownload = nsnull;
}

EmbedDownloadMgr::~EmbedDownloadMgr(void)
{
  // if (!mDownload || !mDownload->parent)
  //   return;
  // GtkMozEmbedDownload *temp = (GtkMozEmbedDownload *) mDownload->parent;
  // temp->is_paused = TRUE;
  // temp->data = NULL;
}

static char *
RemoveSchemeFromFilePath(const char *path)
{
  return g_strdup(g_str_has_prefix(path, "file://")?&path[7]:path);
}

NS_IMETHODIMP
EmbedDownloadMgr::Show(nsIHelperAppLauncher *aLauncher,
                       nsISupports *aContext,
                       PRUint32 aForced)
{
  nsresult rv;
  GtkObject* instance;

  /* create a Download object */
  instance = gtk_moz_embed_download_get_restart_flag () ?
                gtk_moz_embed_download_get_latest_object () : gtk_moz_embed_download_new();

  // XXX reseting download restart flag ...
  gtk_moz_embed_download_set_restart_flag (FALSE);

  if (!instance)
    return NS_ERROR_FAILURE;

  mDownload = (EmbedDownload *) GTK_MOZ_EMBED_DOWNLOAD(instance)->data;
  if (!mDownload)
    return NS_ERROR_FAILURE;
  mDownload->parent = instance;

  rv = GetDownloadInfo(aLauncher, aContext);

  /* Retrieve GtkMozEmbed object from DOM Window */
  nsCOMPtr<nsIDOMWindow> parentDOMWindow = do_GetInterface(aContext);
  mDownload->gtkMozEmbedParentWidget = GetGtkWidgetForDOMWindow(parentDOMWindow);

  // avoiding some casts.
  GtkMozEmbedDownload *temp = (GtkMozEmbedDownload *) mDownload->parent;

  ObserveData(nsnull, "dl-ask", NS_ConvertUTF8toUTF16(temp->server).get());

  PRBool useDownloadDir = PR_FALSE;
  nsCOMPtr<nsIPrefBranch2> prefs(do_GetService(NS_PREFSERVICE_CONTRACTID, &rv));
  rv = prefs->GetBoolPref("browser.download.useDownloadDir", &useDownloadDir);

  gtk_signal_emit(GTK_OBJECT(mDownload->gtkMozEmbedParentWidget),
                  moz_embed_signals[DOWNLOAD_REQUEST],
                  temp->server,
                  temp->file_name,
                  temp->file_type,
                  (gulong) temp->file_size,
                   useDownloadDir?2:1);

  gtk_signal_emit(GTK_OBJECT(mDownload->parent),
                  moz_embed_download_signals[DOWNLOAD_STARTED_SIGNAL],
                  & temp->file_name_with_path);

  if (!temp->file_name_with_path) {
    gtk_moz_embed_download_do_command(GTK_MOZ_EMBED_DOWNLOAD(mDownload->parent),
                                      GTK_MOZ_EMBED_DOWNLOAD_CANCEL);
    return NS_OK;
  }
  ObserveData(nsnull, "dl-start", nsnull);

  char * path = RemoveSchemeFromFilePath(temp->file_name_with_path);
  NS_Free(temp->file_name_with_path);
  temp->file_name_with_path = path;

  return aLauncher->SaveToDisk(nsnull, PR_FALSE);
}

NS_METHOD
EmbedDownloadMgr::GetDownloadInfo(nsIHelperAppLauncher *aLauncher,
                                  nsISupports *aContext)
{
  /* File type */
  nsCOMPtr<nsIMIMEInfo> mimeInfo;
  nsresult rv = aLauncher->GetMIMEInfo(getter_AddRefs(mimeInfo));
  if (NS_FAILED(rv))
    return NS_ERROR_FAILURE;

  nsCAutoString mimeType;
  rv = mimeInfo->GetMIMEType(mimeType);
  if (NS_FAILED(rv))
    return NS_ERROR_FAILURE;

  /* File name */
  nsCAutoString tempFileName;
  nsAutoString suggestedFileName;
  rv = aLauncher->GetSuggestedFileName(suggestedFileName);

  if (NS_FAILED(rv))
    return NS_ERROR_FAILURE;

  tempFileName = NS_ConvertUTF16toUTF8(suggestedFileName);

  /* Complete source URL */
  nsCOMPtr<nsIURI> uri;
  rv = aLauncher->GetSource(getter_AddRefs(uri));
  if (NS_FAILED(rv))
    return NS_ERROR_FAILURE;

  nsCAutoString spec;
  rv = uri->Resolve(NS_LITERAL_CSTRING("."), spec);
  if (NS_FAILED(rv))
    return NS_ERROR_FAILURE;

  PRInt64 mContentLength;
#ifdef GET_CONTENT_LENGTH
  rv = aLauncher->GetContentLength(&mContentLength);
  if (NS_FAILED(rv))
    return rv;
#endif

  /* avoiding some casts */
  GtkMozEmbedDownload *temp = (GtkMozEmbedDownload *) mDownload->parent;

  /* Sets download object to keep control of each download. */
  mDownload->launcher = aLauncher;
  mDownload->incr_launcher = nsnull;
  temp->downloaded_size = -1;
  temp->file_name = g_strdup((gchar *) tempFileName.get());
  temp->server = g_strconcat(spec.get(), (gchar *) temp->file_name, NULL);
  temp->file_type = g_strdup(mimeType.get());
  temp->file_size = (mContentLength > 0)? mContentLength : UNKNOWN_FILE_SIZE;

  return NS_OK;
}

NS_IMETHODIMP EmbedDownloadMgr::PromptForSaveToFile(nsIHelperAppLauncher *aLauncher,
                                                    nsISupports *aWindowContext,
                                                    const PRUnichar *aDefaultFile,
                                                    const PRUnichar *aSuggestedFileExtension,
                                                    nsILocalFile **_retval)
{
  *_retval = nsnull;

  nsCAutoString filePath;
  filePath.Assign(((GtkMozEmbedDownload *) mDownload->parent)->file_name_with_path);

  GtkMozEmbedDownload *temp = (GtkMozEmbedDownload *) mDownload->parent;
  if (temp) {
    if (temp->file_name && !strlen(temp->file_name)) {
      NS_Free(temp->file_name);
      if (!NS_ConvertUTF16toUTF8(aDefaultFile).IsEmpty())
        temp->file_name = NS_strdup(NS_ConvertUTF16toUTF8(aDefaultFile).get());
      else
        temp->file_name = g_strdup_printf("_temp_file%s", NS_ConvertUTF16toUTF8(aSuggestedFileExtension).get());
      if (temp->file_name_with_path) {
        filePath.Append(temp->file_name);
        NS_Free(temp->file_name_with_path);
        temp->file_name_with_path = NS_strdup(filePath.get());
      }
    }
  }

  nsCOMPtr<nsILocalFile> destFile;
  NS_NewNativeLocalFile(filePath,
                        PR_TRUE,
                        getter_AddRefs(destFile));
  if (!destFile)
    return NS_ERROR_OUT_OF_MEMORY;

  /* Progress listener to follow the download and connecting it to
     the launcher which controls the download. */
  nsCOMPtr<nsIWebProgressListener2> listener = new ProgressListener(mDownload);
  if (!listener)
    return NS_ERROR_OUT_OF_MEMORY;
  
  nsresult rv = aLauncher->SetWebProgressListener(listener);
  if (NS_FAILED(rv))
    return NS_ERROR_FAILURE;

  NS_ADDREF(*_retval = destFile);
  return NS_OK;
}

#ifdef OUTPUT_HANDLER_IMPL
NS_IMETHODIMP EmbedDownloadMgr::PromptForSaveToUri(nsIHelperAppLauncher *aLauncher,
                                                   nsISupports *aWindowContext,
                                                   const PRUnichar *aDefaultFile,
                                                   const PRUnichar *aSuggestedFileExtension,
                                                   nsIURI **_retval)
{
  nsresult rv;

  nsCAutoString file_path;
  file_path.Assign (((GtkMozEmbedDownload *) mDownload->parent)->file_name_with_path);
  nsCOMPtr<nsIIOService> ios (do_GetService(NS_IOSERVICE_CONTRACTID));
  if (!ios)
    return FALSE;

  nsCOMPtr<nsIURI> uri;
  rv = ios->NewURI(file_path, "", nsnull, getter_AddRefs(uri));
  if (!uri)
    return FALSE;

  NS_ADDREF (*_retval = uri);

  /* Progress listener to follow the download and connecting it to
     the launcher which controls the download. */
  nsCOMPtr<nsIWebProgressListener2> listener = new ProgressListener(mDownload);
  rv = aLauncher->SetWebProgressListener (listener);
  if (NS_FAILED (rv))
    return NS_ERROR_FAILURE;

  return NS_OK;
}
#endif

/* nsIWebProgressListener Functions
   all these methods must be here due to nsIWebProgressListener/2 inheritance */
NS_IMETHODIMP ProgressListener::OnStatusChange(nsIWebProgress *aWebProgress,
                                               nsIRequest *aRequest,
                                               nsresult aStatus,
                                               const PRUnichar *aMessage)
{
  switch (aStatus)
  {
    case NS_ERROR_OUT_OF_MEMORY:     // No memory
    case NS_ERROR_FILE_DISK_FULL:    // Out of space on target volume.
    case NS_ERROR_FILE_NO_DEVICE_SPACE:
    case NS_ERROR_FILE_READ_ONLY:     // Attempt to write to read/only file.
    case NS_ERROR_FILE_ACCESS_DENIED: // Attempt to write without sufficient permissions.
    case NS_ERROR_FILE_NOT_FOUND:     // Helper app not found, let's verify this happened on launch
    case NS_ERROR_FILE_TARGET_DOES_NOT_EXIST:
    case NS_ERROR_FILE_UNRECOGNIZED_PATH:
    default:
    {
      gtk_signal_emit(GTK_OBJECT(mDownload->parent), moz_embed_download_signals[DOWNLOAD_DESTROYED_SIGNAL]);
      ObserveData(nsnull, "dl-failed", nsnull);
    }
      break;
  }

  if (NS_SUCCEEDED(aStatus))
      return NS_OK;

  return NS_ERROR_FAILURE;
}

NS_IMETHODIMP ProgressListener::OnStateChange(nsIWebProgress *aWebProgress,
                                              nsIRequest *aRequest, PRUint32 aStateFlags,
                                              nsresult aStatus)
{
  if (NS_FAILED(aStatus))
    return NS_ERROR_FAILURE;

  if (aStateFlags & STATE_STOP) {
    // We can crash here if parent or mDownload is null
    ObserveData(nsnull, "dl-done", NS_ConvertUTF8toUTF16(((GtkMozEmbedDownload *) mDownload->parent)->file_name_with_path).get());
    gtk_signal_emit(GTK_OBJECT(mDownload->parent),
                    moz_embed_download_signals[DOWNLOAD_COMPLETED_SIGNAL]);
  }

  return NS_OK;
}

NS_IMETHODIMP ProgressListener::OnProgressChange(nsIWebProgress *aWebProgress,
                                                 nsIRequest *aRequest, PRInt32 aCurSelfProgress,
                                                 PRInt32 aMaxSelfProgress, PRInt32 aCurTotalProgress,
                                                 PRInt32 aMaxTotalProgress)
{
  return OnProgressChange64(aWebProgress,
                            aRequest,
                            aCurSelfProgress,
                            aMaxSelfProgress,
                            aCurTotalProgress,
                            aMaxTotalProgress);
}

NS_IMETHODIMP ProgressListener::OnLocationChange(nsIWebProgress *aWebProgress,
                                                 nsIRequest *aRequest, nsIURI *location)
{
  return NS_ERROR_NOT_IMPLEMENTED;
}

NS_IMETHODIMP ProgressListener::OnSecurityChange(nsIWebProgress *aWebProgress,
                                                 nsIRequest *aRequest, PRUint32 state)
{
  return NS_ERROR_NOT_IMPLEMENTED;
}

/* nsIWebProgressListener2 method */
NS_IMETHODIMP ProgressListener::OnProgressChange64(nsIWebProgress *aWebProgress,
                                                   nsIRequest *aRequest, PRInt64 aCurSelfProgress,
                                                   PRInt64 aMaxSelfProgress, PRInt64 aCurTotalProgress,
                                                   PRInt64 aMaxTotalProgress)
{
  mDownload->request = aRequest;

  if (aMaxSelfProgress != UNKNOWN_FILE_SIZE) {
    gtk_signal_emit(GTK_OBJECT(mDownload->parent),
                    moz_embed_download_signals[DOWNLOAD_PROGRESS_SIGNAL],
                    (gulong) aCurSelfProgress, (gulong) aMaxSelfProgress, 1);
  }
  else {
    gtk_signal_emit(GTK_OBJECT(mDownload->parent),
                    moz_embed_download_signals[DOWNLOAD_PROGRESS_SIGNAL],
                    (gulong) aCurSelfProgress, 0, 1);
  }


  /* storing current downloaded size. */
  ((GtkMozEmbedDownload *) mDownload->parent)->downloaded_size = (gulong) aCurSelfProgress;

  return NS_OK;
}

NS_IMETHODIMP ProgressListener::OnRefreshAttempted(nsIWebProgress *aWebProgress,
                                                   nsIURI *aUri, PRInt32 aDelay,
                                                   PRBool aSameUri,
                                                   PRBool *allowRefresh)
{
  *allowRefresh = PR_TRUE;
  return NS_OK;
}

NS_IMETHODIMP
EmbedDownloadMgr::CreateIncrementalDownload(const char *aUrl,
                                            const char *aDestination,
                                            PRBool aHidden)
{
  //printf("EmbedDownloadMgr.cpp, Line:%d, Func:%s\n",  __LINE__, __FUNCTION__);

  GtkObject* instance;
  /* create a Download object */
  instance = gtk_moz_embed_download_get_restart_flag () ?
                gtk_moz_embed_download_get_latest_object () : gtk_moz_embed_download_new();
  // XXX reseting download restart flag ...
  gtk_moz_embed_download_set_restart_flag (FALSE);
  if (!instance)
    return NS_ERROR_FAILURE;

  EmbedDownload *Download = (EmbedDownload *) GTK_MOZ_EMBED_DOWNLOAD(instance)->data;
  if (!Download)
    return NS_ERROR_FAILURE;
  Download->parent = instance;
  /* Sets download object to keep control of each download. */
  Download->launcher = nsnull;

  GtkMozEmbedDownload *temp = (GtkMozEmbedDownload *) Download->parent;
  temp->downloaded_size = -1;
  temp->file_size = UNKNOWN_FILE_SIZE;
  temp->server = NS_strdup(aUrl);
  temp->file_name_with_path = NS_strdup(aDestination);
  temp->file_name = RemoveSchemeFromFilePath(aDestination);

  return EmbedDownloadMgr::CreateIncrementalDownload(Download);
}

NS_IMETHODIMP
EmbedDownloadMgr::CreateIncrementalDownload(EmbedDownload *Download)
{
  //printf("EmbedDownloadMgr.cpp, Line:%d, Func:%s\n",  __LINE__, __FUNCTION__);

  GtkObject* instance = Download->parent;
  // XXX reseting download restart flag ...
  gtk_moz_embed_download_set_restart_flag (FALSE);
  if (!instance)
    return NS_ERROR_FAILURE;

  /* Sets download object to keep control of each download. */
  Download->launcher = nsnull;

  GtkMozEmbedDownload *temp = (GtkMozEmbedDownload *) Download->parent;

  nsCOMPtr<nsILocalFile> resultFile;
  char *new_path = RemoveSchemeFromFilePath(temp->file_name_with_path);
  nsresult rv = NS_NewNativeLocalFile(nsDependentCString(new_path),
                                      PR_FALSE, getter_AddRefs(resultFile));
  NS_ENSURE_SUCCESS(rv, rv);
  nsCOMPtr<nsIURI> uri;
  rv = NS_NewURI(getter_AddRefs(uri), temp->server);
  NS_ENSURE_SUCCESS(rv, rv);

  nsCOMPtr <nsIRequestObserver> observer = new FetchObserver(Download);
  NS_ENSURE_TRUE(observer, rv);

  nsCOMPtr<nsIIncrementalDownload> download =
      do_CreateInstance(NS_INCREMENTALDOWNLOAD_CONTRACTID, &rv);
  NS_ENSURE_SUCCESS(rv, rv);

  Download->incr_launcher = download;

  rv = download->Init(uri, resultFile, -1, 0);
  NS_ENSURE_SUCCESS(rv, rv);

  rv = download->Start(observer, nsnull);
  NS_ENSURE_SUCCESS(rv, rv);
  return NS_OK;
}


NS_IMETHODIMP
FetchObserver::OnStartRequest(nsIRequest *request, nsISupports *context)
{
  //printf("FetchObserver::OnStartRequest\n");
  NS_ENSURE_TRUE(mDownload, NS_OK);
  mDownload->request = request;
  GtkMozEmbedDownload *temp = (GtkMozEmbedDownload *) mDownload->parent;
  gtk_signal_emit(GTK_OBJECT(mDownload->parent),
                  moz_embed_download_signals[DOWNLOAD_STARTED_SIGNAL], temp->file_name);
  return NS_OK;
}

NS_IMETHODIMP
FetchObserver::OnProgress(nsIRequest *request, nsISupports *context,
                          PRUint64 progress, PRUint64 progressMax)
{
  //printf("FetchObserver::OnProgress [%lu/%lu]\n",PRUint32(progress), PRUint32(progressMax));
  NS_ENSURE_TRUE(mDownload, NS_OK);
  mDownload->request = request;
  if (mDownload->isCanceled)
    return NS_OK;
  if (progressMax != UNKNOWN_FILE_SIZE)
    gtk_signal_emit(GTK_OBJECT(mDownload->parent),
                    moz_embed_download_signals[DOWNLOAD_PROGRESS_SIGNAL],
                    (gulong) progress, (gulong) progressMax, 1);
  else
    gtk_signal_emit(GTK_OBJECT(mDownload->parent),
                    moz_embed_download_signals[DOWNLOAD_PROGRESS_SIGNAL],
                    (gulong) progress, 0, 1);

  GtkMozEmbedDownload *temp = (GtkMozEmbedDownload *) mDownload->parent;
  temp->downloaded_size = progress;
  temp->file_size = progressMax;

  /* storing current downloaded size. */
  ((GtkMozEmbedDownload *) mDownload->parent)->downloaded_size = (gulong) progress;
  return NS_OK;
}

NS_IMETHODIMP
FetchObserver::OnStatus(nsIRequest *request, nsISupports *context,
                        nsresult status, const PRUnichar *statusText)
{
  //printf("gtkmozembed_download.cpp, Line:%d, Func:%s\n",  __LINE__, __FUNCTION__);
  NS_ENSURE_TRUE(mDownload, NS_OK);
  mDownload->request = request;
  return NS_OK;
}

NS_IMETHODIMP
FetchObserver::OnStopRequest(nsIRequest *request, nsISupports *context,
                             nsresult status)
{
  //printf("gtkmozembed_download.cpp, Line:%d, Func:%s, [status=%x]\n",  __LINE__, __FUNCTION__, status);
  NS_ENSURE_TRUE(mDownload, NS_OK);
  mDownload->request = nsnull;
  GtkMozEmbedDownload *parent = (GtkMozEmbedDownload *) mDownload->parent;
  if (mDownload->isCanceled)
    gtk_signal_emit(GTK_OBJECT(mDownload->parent), moz_embed_download_signals[DOWNLOAD_DESTROYED_SIGNAL]);
  else if (!parent->is_paused) {
    if (NS_FAILED(status)) {
      gtk_signal_emit(GTK_OBJECT(mDownload->parent),
                      moz_embed_download_signals[DOWNLOAD_FAILED_SIGNAL]);    //error
      // DOWNLOAD_FAILED_SIGNAL, DOWNLOAD_DESTROYED_SIGNAL, DOWNLOAD_COMPLETED_SIGNAL, DOWNLOAD_STOPPED_SIGNAL,
      gtk_moz_embed_load_url (EmbedCommon::GetAnyLiveWidget(), parent->server);
    }
    else
      gtk_signal_emit(GTK_OBJECT(mDownload->parent),
                      moz_embed_download_signals[DOWNLOAD_COMPLETED_SIGNAL]);
  }

  return NS_OK;
}

#ifdef MOZ_XPINSTALL

/* static */ void
EmbedDownloadMgr::GetEMLocalizedString(const nsAString& key,
                                       nsAString& aResult,
                                       PRBool aIsFormatted,
                                       const PRUnichar** aFormatArgs,
                                       PRUint32 aFormatArgsLength)
{
    if (!sEMBundle) {
        nsCOMPtr<nsIStringBundleService> bundleService = do_GetService(NS_STRINGBUNDLE_CONTRACTID);
        bundleService->CreateBundle(kEMPropertiesURL,
                                    &sEMBundle);
        if (!sEMBundle) {
            NS_ERROR("string bundle not present");
            return;
        }
    }
    nsString str;
    if (aIsFormatted)
        sEMBundle->FormatStringFromName(PromiseFlatString(key).get(),
                                        aFormatArgs, aFormatArgsLength,
                                        getter_Copies(str));
    else
        sEMBundle->GetStringFromName(PromiseFlatString(key).get(),
                                     getter_Copies(str));
    aResult.Assign(str);
}

/* static */ void
EmbedDownloadMgr::GetXILocalizedString(const nsAString& key,
                                       nsAString& aResult,
                                       PRBool aIsFormatted,
                                       const PRUnichar** aFormatArgs,
                                       PRUint32 aFormatArgsLength)
{
    if (!sXIBundle) {
        nsCOMPtr<nsIStringBundleService> bundleService = do_GetService(NS_STRINGBUNDLE_CONTRACTID);
        bundleService->CreateBundle(kXIPropertiesURL,
                                    &sXIBundle);
        if (!sXIBundle) {
            NS_ERROR("string bundle not present");
            return;
        }
    }
    nsString str;
    if (aIsFormatted)
        sXIBundle->FormatStringFromName(PromiseFlatString(key).get(),
                                        aFormatArgs, aFormatArgsLength,
                                        getter_Copies(str));
    else
        sXIBundle->GetStringFromName(PromiseFlatString(key).get(),
                                     getter_Copies(str));
    aResult.Assign(str);
}

// nsIXPIDialogService.idl
NS_IMETHODIMP
EmbedDownloadMgr::ConfirmInstall(nsIDOMWindow *aParent, const PRUnichar **aPackageList, PRUint32 aCount, PRBool *aRetval)
{
  *aRetval = PR_FALSE;
  nsresult rv = NS_OK;
  PRInt32 selection;
  selection = gtkmozembed_hildon_install_dialog(GetGtkWindowForDOMWindow(mParentWindow), aPackageList, aCount, PR_TRUE);
  *aRetval = selection ? PR_TRUE : PR_FALSE;
#if 0
  nsCOMPtr<nsIDOMWindowInternal> mParentWindow( do_QueryInterface(aParent) );
  if (NS_FAILED(rv) && mParentWindow)
  {
    PRInt32 selection;
    nsCOMPtr<nsIPrompt> prompt;
    rv = mParentWindow->GetPrompter(getter_AddRefs(prompt));
    if (NS_SUCCEEDED(rv)) {
      nsString str;
      str.Append(NS_LITERAL_STRING("Do you want to install:\n"));
      for (PRUint32 i = 0; i < aCount; i++)
      {
        if (!aPackageList[i][0]) continue;
        str.Append(aPackageList[i]);
        str.Append(NS_LITERAL_STRING("\n"));
      }
      rv = prompt->ConfirmEx(NS_LITERAL_STRING("Installation confirm").get(), str.get(),
                             (nsIPrompt::BUTTON_DELAY_ENABLE) + (nsIPrompt::BUTTON_POS_0_DEFAULT)
                             + (nsIPrompt::BUTTON_TITLE_OK * nsIPrompt::BUTTON_POS_0)
                             + (nsIPrompt::BUTTON_TITLE_CANCEL * nsIPrompt::BUTTON_POS_1),
                             nsnull, nsnull, nsnull, nsnull, nsnull, &selection);

      if (NS_SUCCEEDED(rv))
      {
        //Now get which button was pressed from the ParamBlock
        *aRetval = selection ? PR_TRUE : PR_FALSE;
      }
    }
  }
#endif
  return rv;
}

NS_IMETHODIMP
EmbedDownloadMgr::OpenProgressDialog(const PRUnichar **aPackageList, PRUint32 aCount, nsIObserver *aObserver)
{
  nsresult rv;
  nsCOMPtr<nsIExtensionManager>
    extensionManager(do_GetService("@mozilla.org/extensions/manager;1", &rv));
  NS_ENSURE_SUCCESS(rv, rv);

  nsCOMPtr<nsIIOService> ios(do_GetService(NS_IOSERVICE_CONTRACTID));

  PRUint32 itemscount = aCount/4;
  nsIUpdateItem **items = new nsIUpdateItem*[itemscount];
  memset(items, 0, sizeof(nsIUpdateItem *) * itemscount);
  PRUint32 j = 0;
  for (PRUint32 i = 0; i < aCount; i++) {
    nsString displayName(aPackageList[i++]);
    nsString url(aPackageList[i++]);
    nsString iconURL(aPackageList[i++]);
    PRBool isTheme = PR_FALSE;

    if (ios) {
      nsCOMPtr<nsIURI> uri;
      ios->NewURI(NS_ConvertUTF16toUTF8(url), "", nsnull, getter_AddRefs(uri));
      nsCOMPtr<nsIURL> nsURL(do_QueryInterface(uri));
      if (nsURL) {
        nsCString extension;
        nsURL->GetFileExtension(extension);
        isTheme = extension.Equals("jar");
      }
    }
    if (iconURL.IsEmpty())
      iconURL = isTheme ? NS_LITERAL_STRING("chrome://mozapps/skin/extensions/themeGeneric.png") :
        NS_LITERAL_STRING("chrome://mozapps/skin/xpinstall/xpinstallItemGeneric.png");

    nsCOMPtr<nsIUpdateItem> item = do_CreateInstance("@mozilla.org/updates/item;1", &rv);
    NS_ENSURE_SUCCESS(rv, rv);
    rv = item->Init(url, NS_LITERAL_STRING(" "), NS_LITERAL_STRING("app-profile"), NS_LITERAL_STRING(""), NS_LITERAL_STRING(""),
                    displayName, url, NS_LITERAL_STRING(""), iconURL, NS_LITERAL_STRING(""),
#ifdef MOZ_TRUNK_BUILD
                    NS_LITERAL_STRING(""), isTheme?nsIUpdateItem::TYPE_THEME:nsIUpdateItem::TYPE_EXTENSION, NS_LITERAL_STRING(""));
#else
                                           isTheme?nsIUpdateItem::TYPE_THEME:nsIUpdateItem::TYPE_EXTENSION);
#endif

    if (NS_SUCCEEDED(rv))
      NS_ADDREF(items[j++] = item);

    if (StringBeginsWith(url, NS_LITERAL_STRING("file://"))) continue;

    GtkObject* instance = gtk_moz_embed_download_get_restart_flag () ?
      gtk_moz_embed_download_get_latest_object () : gtk_moz_embed_download_new();
    gtk_moz_embed_download_set_restart_flag (FALSE);
    if (!instance) return NS_OK;
    mDownload = (EmbedDownload *) GTK_MOZ_EMBED_DOWNLOAD(instance)->data;
    if (!mDownload) return NS_OK;
    mDownload->parent = instance;
    mDownload->gtkMozEmbedParentWidget = GetGtkWidgetForDOMWindow(nsnull);
    GtkMozEmbedDownload *temp = (GtkMozEmbedDownload *) mDownload->parent;
    gtk_signal_emit(GTK_OBJECT(mDownload->gtkMozEmbedParentWidget),
                    moz_embed_signals[DOWNLOAD_REQUEST],
                    NS_ConvertUTF16toUTF8(url).get(),
                    NS_ConvertUTF16toUTF8(displayName).get(),
                    "application/x-xpinstall",
                    (gulong) 1,
                    2);
    gtk_signal_emit(GTK_OBJECT(mDownload->parent),
                    moz_embed_download_signals[DOWNLOAD_STARTED_SIGNAL],
                    &temp->file_name_with_path);

  }

#ifdef MOZ_TRUNK_BUILD
  rv = extensionManager->AddDownloads(items, j, nsnull);

  nsIAddonInstallListener *listener = new ProgressListener(mDownload);
  if (listener)
    extensionManager->AddInstallListener(static_cast<nsIAddonInstallListener*>(listener),
                                        &static_cast<ProgressListener*>(listener)->mObserveIndex);
#else
  rv = extensionManager->AddDownloads(items, j, PR_FALSE);

  nsIAddonUpdateListener *listener = new ProgressListener(mDownload);
  if (listener)
    extensionManager->AddUpdateListener(static_cast<nsIAddonUpdateListener*>(listener),
                                        &static_cast<ProgressListener*>(listener)->mObserveIndex);
#endif

  return rv;
}
#endif

/* static */
void destroy_progress_dialog(ProgressListener *listener)
{
  if (GTK_IS_WIDGET(listener->mProgressDialog))
    gtk_widget_destroy(listener->mProgressDialog);
  listener->mProgressDialog = nsnull;
  listener->mProgressBar = nsnull;
  if (listener->mTimer_id)
    g_source_remove(listener->mTimer_id);
  listener->mTimer_id = 0;
  listener->mPulseCount = 0;

}

gboolean progress_bar_pulse(void *bar)
{
  ProgressListener *listener = static_cast<ProgressListener*>(bar);
  //g_critical("Func:%s::%d, bar:%p, dlg:%p\n", __PRETTY_FUNCTION__, __LINE__, listener->mProgressBar, listener->mProgressDialog);
  if (!listener->mProgressBar) return FALSE;
  gtk_progress_bar_pulse(GTK_PROGRESS_BAR(listener->mProgressBar));
  listener->mPulseCount++;
  if (listener->mPulseCount > 10 && !listener->mIsInstall) {
    destroy_progress_dialog(listener);
    gint retval = 0;
    EmbedCommon * common = EmbedCommon::GetInstance();
    if (common)
      g_signal_emit_by_name (G_OBJECT (common->mCommon), "on-service-notify", common->mCommon, "components", "no_updates", &retval);
    return FALSE;
  } 
  return TRUE;
}

gboolean
install_progress_timeout(void *bar)
{
  ProgressListener *listener = static_cast<ProgressListener*>(bar);
  listener->mProgressDialog = gtkmozembed_hildon_install_progress_banner(GetGtkWindowForDOMWindow(nsnull), &listener->mProgressBar);
  listener->mIsInstall = PR_TRUE;
  listener->mTimer_id = g_timeout_add(500, progress_bar_pulse, listener);
  return FALSE;
}

#ifdef MOZ_RDF
// nsIAddonUpdateListener
#ifndef MOZ_TRUNK_BUILD
NS_IMETHODIMP
ProgressListener::OnStateChange(nsIUpdateItem *addon, PRInt16 state, PRInt32 value)
{
  // printf("nsIAddonUpdateListener:OnStateChange idx:%p, state:%x, val:%i, mProgressState:%i\n", addon, state, value, mProgressState);
  if (mDownload)
    mDownload->request = nsnull;
#ifdef MOZ_XPINSTALL  
  switch (state) {
    case nsIXPIProgressDialog::DOWNLOAD_START:
        break;
    case nsIXPIProgressDialog::DOWNLOAD_DONE:
    {
        NS_ENSURE_TRUE(mDownload, NS_OK);
        GtkMozEmbedDownload *parent = (GtkMozEmbedDownload *) mDownload->parent;
        gtk_signal_emit(GTK_OBJECT(mDownload->parent),
                        moz_embed_download_signals[DOWNLOAD_COMPLETED_SIGNAL]);
        mDownload = nsnull;
        break;
    }
    case nsIXPIProgressDialog::INSTALL_START:
    {
        if (mDownload) {
          GtkMozEmbedDownload *parent = (GtkMozEmbedDownload *) mDownload->parent;
          gtk_signal_emit(GTK_OBJECT(mDownload->parent),
                          moz_embed_download_signals[DOWNLOAD_COMPLETED_SIGNAL]);
          mDownload = nsnull;
        }
        g_idle_add(install_progress_timeout, this);
        break;
    }
    case nsIXPIProgressDialog::INSTALL_DONE:
    {
        // From nsInstall.h
        // SUCCESS        = 0
        // REBOOT_NEEDED  = 999
        // USER_CANCELLED = -210
        destroy_progress_dialog(this);
        if (value == 30) {
          mProgressState = 30;
          return NS_OK;
        }
        mProgressState = value;
        break;
    }
    case nsIXPIProgressDialog::DIALOG_CLOSE:
    {
        nsCOMPtr<nsIExtensionManager>
          extensionManager(do_GetService("@mozilla.org/extensions/manager;1"));
        if (extensionManager)
          extensionManager->RemoveUpdateListenerAt(mObserveIndex);

        mDownload = nsnull;
        mObserveIndex = -1;

        if (mProgressState < -1) {
          nsresult rv;
          nsCOMPtr<nsIWindowWatcher> watcher = do_GetService(NS_WINDOWWATCHER_CONTRACTID, &rv);
          NS_ENSURE_SUCCESS(rv, rv);

          nsCOMPtr<nsIDOMWindow> DOMWindow;
          watcher->GetActiveWindow(getter_AddRefs(DOMWindow));
          nsCOMPtr<nsIDOMWindowInternal> winint = do_QueryInterface(DOMWindow, &rv);
          NS_ENSURE_SUCCESS(rv, rv);

          nsCOMPtr<nsIPrompt> prompt;
          winint->GetPrompter(getter_AddRefs(prompt));
          if (!prompt)
            return NS_ERROR_FAILURE;
          nsAutoString checkMsg, checkTitle;
          EmbedDownloadMgr::GetEMLocalizedString(NS_LITERAL_STRING("errorInstallTitle"), checkTitle);
          nsString url, name, error;
          addon->GetXpiURL(url);// GetXpiURL GetId GetVersion GetObjectSource
          name.Assign(NS_LITERAL_STRING("MicroB"));
          nsString err;
          err.Assign(NS_LITERAL_STRING("error"));
          err.AppendInt(mProgressState);
          EmbedDownloadMgr::GetXILocalizedString(err, error);
          const PRUnichar *strings[] = { name.get(), url.get(), error.get() };
          EmbedDownloadMgr::GetEMLocalizedString(NS_LITERAL_STRING("errorInstallMsg"), checkMsg, PR_TRUE, strings, 3);
          prompt->Alert(checkTitle.get(), checkMsg.get());
        }
        else {
          PRBool aNeedsRestart = PR_FALSE;
          nsresult rv;
          if (extensionManager)
            extensionManager->Start(nsnull, &aNeedsRestart);
          nsCOMPtr<nsIComponentRegistrar> cr;
          rv = NS_GetComponentRegistrar(getter_AddRefs(cr));
          NS_ENSURE_SUCCESS(rv, rv);
          cr->AutoRegister(nsnull);
          if (aNeedsRestart) {
            EmbedCommon * common = EmbedCommon::GetInstance();
            gint retval = 0;
            g_signal_emit_by_name (G_OBJECT (common->mCommon), "on-service-notify", common->mCommon, "components", "restart_required", &retval);
          }
        }
        break;
    }
    default:
        break;
  }
#endif

  return NS_OK;
}

NS_IMETHODIMP
ProgressListener::OnProgress(nsIUpdateItem *addon, PRUint32 value, PRUint32 maxValue)
{
  // printf("nsIAddonUpdateListener:OnProgress idx:%p, val:%i, maxval:%i\n", addon, value, maxValue);
  NS_ENSURE_TRUE(mDownload, NS_OK);
  if (mDownload->isCanceled) {
    nsCOMPtr<nsIExtensionManager>
      extensionManager(do_GetService("@mozilla.org/extensions/manager;1"));
    nsString url;
    addon->GetXpiURL(url);// GetXpiURL GetId GetVersion GetObjectSource
    if (extensionManager)
      extensionManager->RemoveDownload(url);
    return NS_OK;
  }
  gtk_signal_emit(GTK_OBJECT(mDownload->parent),
                  moz_embed_download_signals[DOWNLOAD_PROGRESS_SIGNAL],
                  (gulong) value, (gulong) maxValue, 1);

  GtkMozEmbedDownload *temp = (GtkMozEmbedDownload *) mDownload->parent;
  temp->downloaded_size = value;
  temp->file_size = maxValue;

  /* storing current downloaded size. */
  ((GtkMozEmbedDownload *) mDownload->parent)->downloaded_size = (gulong) value;

  return NS_OK;
}
#endif /* MOZ_TRUNK_BUILD */

NS_IMETHODIMP ProgressListener::OnUpdateStarted()
{
  //g_critical("Func:%s::%d\n", __PRETTY_FUNCTION__, __LINE__);
  return NS_OK;
}

gboolean select_updates_timeout(void *bar)
{
  //g_critical("Func:%s::%d\n", __PRETTY_FUNCTION__, __LINE__);
  return FALSE;
}

NS_IMETHODIMP ProgressListener::OnUpdateEnded()
{
  //g_critical("Func:%s::%d\n", __PRETTY_FUNCTION__, __LINE__);
  destroy_progress_dialog(this);
  //g_timeout_add(500, select_updates_timeout, (void*)this);
  if (!mUpdatesGlist) {
     gint retval = 0;
     EmbedCommon * common = EmbedCommon::GetInstance();
     if (common)
       g_signal_emit_by_name (G_OBJECT (common->mCommon), "on-service-notify", common->mCommon, "components", "no_updates", &retval);
     
     return NS_OK;
  }
  if (!gtkmozembed_hildon_select_updates_dialog(GetGtkWindowForDOMWindow(NULL), &mUpdatesGlist)) return NS_OK;

  PRUint32 itemscount = g_list_length(mUpdatesGlist);
  nsIUpdateItem **items = new nsIUpdateItem*[itemscount];
  memset(items, 0, sizeof(nsIUpdateItem *) * itemscount);
  PRUint32 j = 0;
  EmbedDownload *mDownload = NULL;
  
  for (PRUint32 i = 0; i < itemscount; i++) {
    nsIUpdateItem *extension = static_cast<nsIUpdateItem *>(g_list_nth_data(mUpdatesGlist, i));
    //if (NS_SUCCEEDED(rv))
    NS_ADDREF(items[j++] = extension);

    nsString str, url;
    extension->GetName(str);// GetXpiURL GetId GetVersion GetObjectSource
    extension->GetXpiURL(url);// GetXpiURL GetId GetVersion GetObjectSource

    GtkObject* instance = gtk_moz_embed_download_get_restart_flag () ?
      gtk_moz_embed_download_get_latest_object () : gtk_moz_embed_download_new();
    gtk_moz_embed_download_set_restart_flag (FALSE);
    if (!instance) return NS_OK;
    mDownload = (EmbedDownload *) GTK_MOZ_EMBED_DOWNLOAD(instance)->data;
    if (!mDownload) return NS_OK;
    mDownload->parent = instance;
    mDownload->gtkMozEmbedParentWidget = GetGtkWidgetForDOMWindow(nsnull);
    GtkMozEmbedDownload *temp = (GtkMozEmbedDownload *) mDownload->parent;
    gtk_signal_emit(GTK_OBJECT(mDownload->gtkMozEmbedParentWidget),
                    moz_embed_signals[DOWNLOAD_REQUEST],
                    NS_ConvertUTF16toUTF8(url).get(),
                    NS_ConvertUTF16toUTF8(str).get(),
                    "application/x-xpinstall",
                    (gulong) 1,
                    2);
    gtk_signal_emit(GTK_OBJECT(mDownload->parent),
                    moz_embed_download_signals[DOWNLOAD_STARTED_SIGNAL],
                    &temp->file_name_with_path);
  }
  nsresult rv;
  nsCOMPtr<nsIExtensionManager>
    extensionManager(do_GetService("@mozilla.org/extensions/manager;1", &rv));
  if (NS_FAILED(rv)) {
    printf("Fail to initialize extension manager\n");
    return NS_OK;
  }

#ifdef MOZ_TRUNK_BUILD
  rv = extensionManager->AddDownloads(items, j, nsnull);
  SetupDownload(mDownload);
  extensionManager->AddInstallListener(static_cast<nsIAddonInstallListener*>(this),
                                      &mObserveIndex);
#else
  rv = extensionManager->AddDownloads(items, j, PR_TRUE);
  SetupDownload(mDownload);
  extensionManager->AddUpdateListener(static_cast<nsIAddonUpdateListener*>(this),
                                      &mObserveIndex);
#endif

  g_list_free(mUpdatesGlist);
  mUpdatesGlist = nsnull;
  return NS_OK;
}

NS_IMETHODIMP ProgressListener::OnAddonUpdateStarted(nsIUpdateItem *addon)
{
  //g_critical("Func:%s::%d: addon:%p\n", __PRETTY_FUNCTION__, __LINE__, addon);
  return NS_OK;
}

NS_IMETHODIMP ProgressListener::OnAddonUpdateEnded(nsIUpdateItem *addon, PRInt32 status)
{
  //g_critical("Func:%s::%d: addon:%p, status:%i\n", __PRETTY_FUNCTION__, __LINE__, addon, status);
  if (status == 1)
    mUpdatesGlist = g_list_append(mUpdatesGlist, addon);
  return NS_OK;
}

nsresult
EmbedDownloadObserver::Create(nsISupports *aOuter, REFNSIID aIID, void **aResult)
{
  //g_critical("Func:%s::%d\n", __PRETTY_FUNCTION__, __LINE__);
  nsresult rv;

  if (aOuter) {
    rv = NS_ERROR_NO_AGGREGATION;
    return rv;
  }

  EmbedDownloadObserver* adapter = new EmbedDownloadObserver();
  if (!adapter) {
    rv = NS_ERROR_OUT_OF_MEMORY;
    return rv;
  }

  NS_ADDREF(adapter);
  rv = adapter->Init();
  if (NS_SUCCEEDED(rv)) {
    rv = adapter->QueryInterface(aIID, aResult);
  }
  NS_RELEASE(adapter);

  return rv;
}

nsresult
EmbedDownloadObserver::RegisterSelf(nsIComponentManager* aCompMgr,
                                    nsIFile* aPath,
                                    const char* aLoaderStr,
                                    const char* aType,
                                    const nsModuleComponentInfo *aInfo)
{
  //g_critical("Func:%s::%d\n", __PRETTY_FUNCTION__, __LINE__);
  nsresult rv;
  nsCOMPtr<nsICategoryManager> catMan = do_GetService(NS_CATEGORYMANAGER_CONTRACTID, &rv);
  if (!catMan) return NS_ERROR_FAILURE;

  rv = catMan->AddCategoryEntry("app-startup",
                                "EmbedDownloadObserver",
                                NS_EMBED_DOWNLOAD_CONTRACTID,
                                PR_TRUE, PR_TRUE, nsnull);
  return rv;
}

nsresult
EmbedDownloadObserver::UnregisterSelf(nsIComponentManager* aCompMgr,
                                      nsIFile* aPath,
                                      const char *registryLocation,
                                      const nsModuleComponentInfo *aInfo)
{
  //g_critical("Func:%s::%d\n", __PRETTY_FUNCTION__, __LINE__);
  nsresult rv;
  nsCOMPtr<nsICategoryManager> catMan = do_GetService(NS_CATEGORYMANAGER_CONTRACTID, &rv);
  if (!catMan) return NS_ERROR_FAILURE;

  rv = catMan->DeleteCategoryEntry("app-startup",
                                   "EmbedDownloadObserver",
                                   PR_TRUE);
  return rv;
}


EmbedDownloadObserver::EmbedDownloadObserver()
{
  //g_critical("Func:%s::%d\n", __PRETTY_FUNCTION__, __LINE__);
}

EmbedDownloadObserver::~EmbedDownloadObserver()
{
  //g_critical("Func:%s::%d\n", __PRETTY_FUNCTION__, __LINE__);
}

nsresult
EmbedDownloadObserver::Init()
{
  //g_critical("Func:%s::%d\n", __PRETTY_FUNCTION__, __LINE__);
  nsresult rv = NS_OK;
  return rv;
}

/* static */
gboolean find_updates_timeout(void *bar)
{
  //g_critical("Func:%s::%d\n", __PRETTY_FUNCTION__, __LINE__);
  nsresult rv;
  nsCOMPtr<nsIExtensionManager>
    extensionManager(do_GetService("@mozilla.org/extensions/manager;1", &rv));
  if (NS_FAILED(rv)) {
    printf("Fail to initialize extension manager\n");
    return FALSE;
  }
  PRUint32 itemCount = 0;
  nsIUpdateItem **items;
  rv = extensionManager->GetItemList(nsIUpdateItem::TYPE_ADDON, &itemCount, &items);
  
  if (!itemCount) {
    gint retval = 0;
    EmbedCommon * common = EmbedCommon::GetInstance();
    if (common)
      g_signal_emit_by_name (G_OBJECT (common->mCommon), "on-service-notify", common->mCommon, "components", "no_updates", &retval);
    return FALSE;
  }

  nsCOMPtr <nsIAddonUpdateCheckListener> listener;
  ProgressListener *ilistener = new ProgressListener(NULL);
  listener = ilistener;
  ilistener->mProgressDialog = gtkmozembed_hildon_check_updates_dialog(GetGtkWindowForDOMWindow(nsnull), &ilistener->mProgressBar);
  ilistener->mTimer_id = g_timeout_add(500, progress_bar_pulse, ilistener);
  ilistener->mIsInstall = FALSE;
  rv = extensionManager->Update(items, itemCount, PR_FALSE, listener);
  guint response = gtk_dialog_run (GTK_DIALOG (ilistener->mProgressDialog));
  destroy_progress_dialog(ilistener);
  return FALSE;
}

nsresult
EmbedDownloadObserver::FindUpdates(const PRUnichar *aData)
{
  //g_critical("Func:%s::%d\n", __PRETTY_FUNCTION__, __LINE__);
  g_idle_add(find_updates_timeout, (void*)aData);
  return NS_OK;
}

nsresult
EmbedDownloadObserver::UninstallComponent(const PRUnichar *aData)
{
  nsresult rv;
  //g_critical("Func:%s::%d, comp:name:%s\n", __PRETTY_FUNCTION__, __LINE__, NS_ConvertUTF16toUTF8(aData).get());
  nsCOMPtr<nsIExtensionManager>
    extensionManager(do_GetService("@mozilla.org/extensions/manager;1", &rv));
  if (NS_FAILED(rv)) {
    printf("Fail to initialize extension manager\n");
    return NS_ERROR_FAILURE;
  }
  PRUint32 itemCount = 0;
  nsIUpdateItem **items;
  rv = extensionManager->GetItemList(nsIUpdateItem::TYPE_ADDON, &itemCount, &items);
  for (PRUint32 i = 0; i < itemCount; i++) {
     nsIUpdateItem *extension = items[i];
     nsString str, id;
     extension->GetName(str);// GetXpiURL GetId GetVersion GetObjectSource
     extension->GetId(id);// GetXpiURL GetId GetVersion GetObjectSource
     if (str.Equals(aData)) {
       extensionManager->UninstallItem(id);
       gint retval = 0;
       EmbedCommon * common = EmbedCommon::GetInstance();
       if (common)
         g_signal_emit_by_name (G_OBJECT (common->mCommon), "on-service-notify", common->mCommon, "components", "restart_required", &retval);
     }
  }
  return rv;
}


NS_IMETHODIMP
EmbedDownloadObserver::Observe(nsISupports *aSubject, const char *aTopic, const PRUnichar *aData)
{
  //g_critical("Func:%s::%d, aTopic:%s\n", __PRETTY_FUNCTION__, __LINE__, aTopic);
  if (!strcmp(aTopic, "find_updates")) {
    FindUpdates(aData);
  } else  if (!strcmp(aTopic, "uninstall_component")) {
    UninstallComponent(aData);
  }
  return NS_OK;
}

NS_IMPL_ISUPPORTS1(EmbedDownloadObserver, nsIObserver)

#endif

