/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* vim:set ts=2 sw=2 sts=2 tw=80 et cindent: */
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is mozilla.org code.
 *
 * The Initial Developer of the Original Code is
 * Oleg Romashin.
 * Portions created by the Initial Developer are Copyright (C) 2006
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *   Oleg Romashin <romaxa@gmail.com>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

#include "EmbedContextMenuInfo2.h"
#include "EmbedGtkTools.h"

#include "gtkmozembed_context.h"
#include "gtkmozembed.h"
#include "gtkmozembed_internal.h"

#include "nsIContent.h"

#include "nsIDOMDocument.h"
#include "nsIDOMDocumentView.h"
#include "nsIDOMElement.h"
#include "nsIDOMEvent.h"
#include "nsIDOMEventTarget.h"
#include "nsIDOMHTMLAnchorElement.h"
#include "nsIDOMHTMLAreaElement.h"
#include "nsIDOMHTMLImageElement.h"
#include "nsIDOMHTMLInputElement.h"
#include "nsIDOMHTMLTextAreaElement.h"
#include "nsIDOMMouseEvent.h"
#include "nsIDOMNSEvent.h"
#include "nsIDOMNSHTMLDocument.h"
#include "nsIDOMNSEditableElement.h"
#include "nsIDOMNode.h"
#include "nsIDOMWindow.h"
#include "nsIDOMWindowCollection.h"
#include "nsIDOMNSHTMLInputElement.h"
#include "nsIDOMNSHTMLTextAreaElement.h"

#include "nsISelectionController.h"
#include "nsIFormControl.h"
#include "nsIWebBrowser.h"
#include "nsIWebNavigation.h"

#include "gfxIImageFrame.h"
// imglib2
#include "imgIRequest.h"
#include "imgIContainer.h"
#include "nsIImageLoadingContent.h"


#include "nsPIDOMWindow.h"
#include "nsIDocShell.h"
#include "nsIInterfaceRequestorUtils.h"

// composer
#include "nsIEditingSession.h"

// editor
#include "nsIPlaintextEditor.h"
#include "nsIEditor.h"
#include "nsIDocument.h"


//#include ".h"
#include "nsIDOMNSElement.h"
#include "nsIDOMHTMLObjectElement.h"
#include "nsIDOMNodeList.h"
#include "nsIDOMCSSStyleDeclaration.h"
#include "nsIDOMCSSValue.h"
#include "nsIDOMCSSPrimitiveValue.h"
#include "nsIDOM3Document.h"
#include "nsIDOMViewCSS.h"

#ifndef MOZ_TRUNK_BUILD
#include "nsIDOMTextRectangle.h"
#else
#include "nsIDOMClientRect.h"
#endif

#include "nsIDOMNSHTMLElement.h"
#include "nsIWebBrowserFocus.h"

//*****************************************************************************
// class EmbedContextMenuInfo2
//*****************************************************************************

EmbedContextMenuInfo2::EmbedContextMenuInfo2(GtkMozContext *ctx, const void *aEmbed)
: mCtx(ctx),
  mEmbed(aEmbed),
  mEventNode(nsnull),
  mIsScrollableElement(PR_FALSE)
{
}

EmbedContextMenuInfo2::~EmbedContextMenuInfo2(void)
{
}

NS_IMPL_ADDREF(EmbedContextMenuInfo2)
NS_IMPL_RELEASE(EmbedContextMenuInfo2)
NS_INTERFACE_MAP_BEGIN(EmbedContextMenuInfo2)
NS_INTERFACE_MAP_ENTRY(nsISupports)
NS_INTERFACE_MAP_END

static PRBool
IsXULNode(nsIDOMNode *aNode, PRUint32 *aType = 0)
{
  PRBool retval = PR_FALSE;
  if (!aNode) return retval;

  nsString sorigNode;
  aNode->GetNodeName(sorigNode);
  if (sorigNode.EqualsLiteral("#document"))
    return retval;
  retval = StringBeginsWith(sorigNode, NS_LITERAL_STRING("xul:"));

  if (!aType) return retval;

  if (sorigNode.EqualsLiteral("xul:thumb")
      || sorigNode.EqualsLiteral("xul:vbox")
      || sorigNode.EqualsLiteral("xul:spacer"))
    *aType = PR_FALSE; // Magic
  else if (sorigNode.EqualsLiteral("xul:slider"))
    *aType = 2; // Magic
  else if (sorigNode.EqualsLiteral("xul:scrollbarbutton"))
    *aType = 3; // Magic

  return retval;
}

nsresult
EmbedContextMenuInfo2::SetFramePosition(void)
{
  nsresult rv;

  //FIXME canot get frame info without main window
  if (!GTK_IS_MOZ_EMBED(mEmbed)) return NS_ERROR_FAILURE;

  nsCOMPtr<nsIDOMWindow> mainWindow;
  {
    nsIWebBrowser *webBrowser = nsnull;
    gtk_moz_embed_get_nsIWebBrowser(GTK_MOZ_EMBED(mEmbed), &webBrowser);
    NS_ENSURE_TRUE(webBrowser, NS_ERROR_FAILURE);
    rv = webBrowser->GetContentDOMWindow(getter_AddRefs(mainWindow));
  }
  NS_ENSURE_SUCCESS(rv, rv);

  nsCOMPtr<nsIDOMWindow> ctxWindow;
  rv = GetDOMWindowByNode(mEventNode, getter_AddRefs(ctxWindow));

  mCtx->docindex = -1;
  if (!ctxWindow || ctxWindow == mainWindow)
    return NS_OK;

  mCtx->type |= GTK_MOZ_EMBED_CONTEXT_IFRAME;

  nsCOMPtr<nsIDOMWindowCollection> frames;
  rv = mainWindow->GetFrames(getter_AddRefs(frames));
  NS_ENSURE_SUCCESS(rv, rv);

  PRUint32 frameCount = 0;
  frames->GetLength(&frameCount);
  nsCOMPtr<nsIDOMWindow> currentWindow;
  NS_ENSURE_SUCCESS(rv, rv);
  for (unsigned int i= 0; i < frameCount; i++) {
    frames->Item(i, getter_AddRefs(currentWindow));
    if (currentWindow != ctxWindow) continue;
    mCtx->docindex = i;
    break;
  }
  return NS_OK;
}

nsresult
EmbedContextMenuInfo2::CheckRichEdit(void)
{
  nsresult rv;
  nsCOMPtr<nsIDOMElement> targetDOMElement;
  rv = mCtxDocument->GetDocumentElement(getter_AddRefs(targetDOMElement));
  if (!targetDOMElement) return rv;

  nsCOMPtr<nsIDOMNSHTMLDocument> htmlDoc = do_QueryInterface(mCtxDocument, &rv);
  if (!htmlDoc) return rv;

  nsString DMode;
  htmlDoc->GetDesignMode(DMode);
  if (!DMode.EqualsLiteral("on")) return rv;

  mCtx->type |= GTK_MOZ_EMBED_CONTEXT_INPUT_TEXT | GTK_MOZ_EMBED_CONTEXT_RICHEDIT | GTK_MOZ_EMBED_CONTEXT_MULTILINE;
  mIsScrollableElement = PR_FALSE;
  return rv;
}

nsresult
EmbedContextMenuInfo2::CheckFormControl(nsIDOMNode* aNode)
{
  nsresult rv;
  PRBool rdonly = PR_FALSE;
  nsCOMPtr<nsIDOMEventTarget> eventTarget;
  if (aNode) {
    eventTarget = do_QueryInterface(aNode, &rv);
  } else
    rv = mCtxEvent->GetTarget(getter_AddRefs(eventTarget));
  NS_ENSURE_SUCCESS(rv, rv);

  nsCOMPtr<nsIContent> tgContent = do_QueryInterface(eventTarget, &rv);
  NS_ENSURE_SUCCESS(rv, rv);
  mCtxFormType = 0;
  if (!tgContent->IsNodeOfType(nsIContent::eHTML_FORM_CONTROL))
    return NS_ERROR_FAILURE;

  nsCOMPtr<nsIFormControl> formControl(do_QueryInterface(tgContent, &rv));
  NS_ENSURE_SUCCESS(rv, rv);
  mCtxFormType = formControl->GetType();
  switch (mCtxFormType) {
  case NS_FORM_BUTTON_BUTTON:
    break;
  case NS_FORM_BUTTON_RESET:
    break;
  case NS_FORM_BUTTON_SUBMIT:
    break;
  case NS_FORM_INPUT_BUTTON:
    break;
  case NS_FORM_INPUT_CHECKBOX:
    break;
  case NS_FORM_INPUT_FILE:
    break;
  case NS_FORM_INPUT_HIDDEN:
    break;
  case NS_FORM_INPUT_RESET:
    break;
  case NS_FORM_INPUT_IMAGE:
    break;
  case NS_FORM_INPUT_PASSWORD:
  {
    nsCOMPtr<nsIDOMHTMLInputElement> input;
    input = do_QueryInterface(mEventNode);
    if (input) input->GetReadOnly(&rdonly);
    if (!rdonly) {
        mCtx->type |= GTK_MOZ_EMBED_CONTEXT_INPUT_TEXT;
        mCtx->type |= GTK_MOZ_EMBED_CONTEXT_IPASSWORD;
    }
    break;
  }    
  case NS_FORM_INPUT_RADIO:
    break;
  case NS_FORM_INPUT_SUBMIT:
    break;
  case NS_FORM_INPUT_TEXT:
  {
    nsCOMPtr<nsIDOMHTMLInputElement> input;
    input = do_QueryInterface(mEventNode);
    if (input) input->GetReadOnly(&rdonly);
    if (!rdonly)
        mCtx->type |= GTK_MOZ_EMBED_CONTEXT_INPUT_TEXT;
    break;
  }    
  case NS_FORM_LABEL:
    break;
  case NS_FORM_OPTION:
    break;
  case NS_FORM_OPTGROUP:
    break;
  case NS_FORM_LEGEND:
    break;
  case NS_FORM_SELECT:
    mIsScrollableElement = PR_FALSE;
    break;
  case NS_FORM_TEXTAREA:
  {
    nsCOMPtr<nsIDOMHTMLTextAreaElement> input;
    input = do_QueryInterface(mEventNode);
    if (input) input->GetReadOnly(&rdonly);
    if (!rdonly) 
        mCtx->type |= GTK_MOZ_EMBED_CONTEXT_INPUT_TEXT | GTK_MOZ_EMBED_CONTEXT_MULTILINE;	
    break;
  }
  case NS_FORM_OBJECT:
    break;
  default:
    break;
  }
  if (mCtx->type & GTK_MOZ_EMBED_CONTEXT_INPUT_TEXT)
    mIsScrollableElement = rdonly;

  return rv;
}

nsresult
EmbedContextMenuInfo2::ResolveBaseURL (const nsAString &relurl, nsACString &url)
{
  nsCString cRelURL(NS_ConvertUTF16toUTF8(relurl).get());
  return mBaseURI->Resolve (cRelURL, url);
}

nsresult
EmbedContextMenuInfo2::ScrollToSelectedNode(nsISupports *element)
{
  nsresult rv = NS_ERROR_FAILURE;
  nsCOMPtr<nsIDOMNSHTMLElement> nodeElement = do_QueryInterface(element);
  if (!nodeElement) {
    nsCOMPtr<nsIWebBrowser> webBrowser = do_QueryInterface(element);
    if (!webBrowser)
       return rv;
    nsCOMPtr<nsIWebBrowserFocus> focus(do_GetInterface(webBrowser));
    if (!focus)
      return rv;
    nsCOMPtr<nsIDOMElement> domElement;
    rv = focus->GetFocusedElement(getter_AddRefs(domElement));
    nodeElement = do_QueryInterface(domElement);
  }
  if (nodeElement) {
      nodeElement->ScrollIntoView(PR_FALSE);
  }
  return rv;
}

nsresult
EmbedContextMenuInfo2::CheckDomHtmlNode(nsIDOMNode *aNode, PRBool &aHasImage)
{
  nsresult rv;

  nsCOMPtr<nsIDOMNode> node = do_QueryInterface(aNode, &rv);
  NS_ENSURE_TRUE(node, NS_ERROR_FAILURE);

  nsString uTag;
  nsString tmp;
  PRUint16 dnode_type;
  rv = node->GetNodeType(&dnode_type);
  NS_ENSURE_SUCCESS(rv, rv);

  if (nsIDOMNode::ELEMENT_NODE != dnode_type)
    return rv;

  node->GetLocalName(uTag);
  if (uTag.LowerCaseEqualsLiteral("object")) {
    nsCOMPtr<nsIDOMHTMLObjectElement> object;
    object = do_QueryInterface (node);
    if (!object) return NS_ERROR_FAILURE;
    nsString value;
    object->GetType(value);
    if (StringBeginsWith(value, NS_LITERAL_STRING("/image"))) {
      nsString img;
      rv = object->GetData (img);
      if (NS_FAILED (rv)) return NS_ERROR_FAILURE;
      nsCString cImg;
      rv = ResolveBaseURL (img, cImg);
      if (NS_FAILED (rv)) return NS_ERROR_FAILURE;
      SetStringProperty ("image", cImg.get());
      mCtx->type |= GTK_MOZ_EMBED_CONTEXT_IMAGE;
      aHasImage = PR_TRUE;
    }
  }
  else if (uTag.LowerCaseEqualsLiteral("html")) {
    nsCOMPtr<nsIDOMNodeList> nodeList;
    rv = mCtxDocument->GetElementsByTagName (NS_LITERAL_STRING("body"),
                                             getter_AddRefs (nodeList));
    if (NS_SUCCEEDED (rv) && nodeList) {
      nsCOMPtr<nsIDOMNode> bodyNode;
      nodeList->Item (0, getter_AddRefs (bodyNode));
      nsString cssurl;
      rv = GetCSSBackground (bodyNode, cssurl);
      if (NS_SUCCEEDED (rv)) {
        nsCString bgimg;
        rv = ResolveBaseURL (cssurl, bgimg);
        if (NS_FAILED (rv)) return NS_ERROR_FAILURE;
        SetStringProperty ("image", bgimg.get());
        mCtx->type |= GTK_MOZ_EMBED_CONTEXT_IMAGE;
        aHasImage = PR_TRUE;
      }
    }
  }
  else if (uTag.LowerCaseEqualsLiteral("a")) {

    nsCOMPtr<nsIDOMHTMLAnchorElement> anchor = do_QueryInterface(node, &rv);
    NS_ENSURE_SUCCESS(rv, rv);
    mCtx->type |= GTK_MOZ_EMBED_CONTEXT_LINK;
    anchor->GetHref(tmp);
    
    if (StringBeginsWith(tmp, NS_LITERAL_STRING("mailto:"))) {
      nsCString href;
      NS_UTF16ToCString (tmp, NS_CSTRING_ENCODING_UTF8, href);
      href.Cut (0, 7);
      SetStringProperty ("email", href.get());
      mCtx->type |= GTK_MOZ_EMBED_CONTEXT_MAILTO;
    }
    
    if (!tmp.IsEmpty()) {
      SetStringProperty ("link", tmp);
      if (tmp.LowerCaseEqualsLiteral("text/smartbookmark")) {

        nsCOMPtr<nsIDOMNode> childNode;
        node->GetFirstChild(getter_AddRefs(childNode));
        if (childNode) {
          PRInt32 width, height;
          tmp.Truncate();
          rv = CheckDomImageElement(node, tmp, width, height);
          if (NS_SUCCEEDED(rv)) {
            mCtx->type |= GTK_MOZ_EMBED_CONTEXT_IMAGE;
            SetStringProperty ("image", tmp);
          }
        }
      } else {
        nsCOMPtr<nsIDOMNode> childNode;
        node->GetFirstChild(getter_AddRefs(childNode));
        if (childNode) {
          childNode->GetLocalName(uTag);
          mIsScrollableElement = !uTag.LowerCaseEqualsLiteral("div");
        }
      }
    }
  }
  else if (uTag.LowerCaseEqualsLiteral("option")) {
    mCtx->type |= GTK_MOZ_EMBED_CONTEXT_NONE;
    rv = NS_OK;
  }
  else if (uTag.LowerCaseEqualsLiteral("area")) {
    nsCOMPtr<nsIDOMHTMLAreaElement> area = do_QueryInterface(node, &rv);
    mCtx->type |= GTK_MOZ_EMBED_CONTEXT_LINK;
    if (NS_SUCCEEDED(rv) && area) {
      PRBool aNoHref = PR_FALSE;
      rv = area->GetNoHref(&aNoHref);
      rv = aNoHref?area->GetTarget(tmp):area->GetHref(tmp);
      SetStringProperty ("link", tmp);
      rv = NS_OK;
    }
  }
  else if (uTag.LowerCaseEqualsLiteral("img")) {
    PRInt32 width, height;
    rv = CheckDomImageElement(node, tmp, width, height);
    if (NS_SUCCEEDED(rv)) {
      mCtx->type |= GTK_MOZ_EMBED_CONTEXT_IMAGE;
      SetStringProperty ("image", tmp);
    }
  } else
    rv = NS_ERROR_FAILURE;
  if (!aHasImage)
  {
    nsString cssurl;
    rv = GetCSSBackground (node, cssurl);
    if (NS_SUCCEEDED (rv))
    {
      nsCString bgimg;

      rv = ResolveBaseURL (cssurl, bgimg);
      if (NS_FAILED (rv))
        return NS_ERROR_FAILURE;
      SetStringProperty ("image", bgimg.get());
      mCtx->type |= GTK_MOZ_EMBED_CONTEXT_IMAGE;
      aHasImage = PR_TRUE;
    }
  }
  return rv;
}

nsresult
EmbedContextMenuInfo2::SetIntProperty (const char *name, int value)
{
  GValue *val = g_new0 (GValue, 1);
  g_value_init (val, G_TYPE_INT);
  g_value_set_int (val, value);
  gtk_moz_context_set_prop(mCtx, name, val);
  return NS_OK;
}

nsresult
EmbedContextMenuInfo2::SetStringProperty (const char *name, const char *value)
{
  GValue *val = g_new0 (GValue, 1);
  g_value_init (val, G_TYPE_STRING);
  g_value_set_string (val, value);
  gtk_moz_context_set_prop(mCtx, name, val);
  return NS_OK;
}

nsresult
EmbedContextMenuInfo2::SetStringProperty (const char *name, const nsAString &value)
{
  nsCString cValue(NS_ConvertUTF16toUTF8(value).get());
  return SetStringProperty (name, cValue.get());
}

/* static */
nsresult
EmbedContextMenuInfo2::GetCSSBackground (nsIDOMNode *node, nsAString& url)
{

  nsresult rv;

  NS_ENSURE_TRUE (mViewCSS, NS_ERROR_FAILURE);

  nsCOMPtr<nsIDOMElement> element = do_QueryInterface (node);
  NS_ENSURE_TRUE (element, NS_ERROR_FAILURE);

  nsCOMPtr<nsIDOMCSSStyleDeclaration> decl;
  mViewCSS->GetComputedStyle (element, nsString(),
                              getter_AddRefs (decl));
  NS_ENSURE_TRUE (decl, NS_ERROR_FAILURE);

  nsCOMPtr<nsIDOMCSSValue> CSSValue;
  decl->GetPropertyCSSValue (NS_LITERAL_STRING("background-image"), getter_AddRefs (CSSValue));

  nsCOMPtr<nsIDOMCSSPrimitiveValue> primitiveValue =
    do_QueryInterface (CSSValue);
  if (!primitiveValue) return NS_ERROR_FAILURE;

  PRUint16 type;
  rv = primitiveValue->GetPrimitiveType (&type);
  NS_ENSURE_SUCCESS (rv, NS_ERROR_FAILURE);

  if (type != nsIDOMCSSPrimitiveValue::CSS_URI) return NS_ERROR_FAILURE;

  rv = primitiveValue->GetStringValue (url);
  NS_ENSURE_SUCCESS (rv, NS_ERROR_FAILURE);

  return NS_OK;
}

/* static */
nsresult
EmbedContextMenuInfo2::GetImageRequest(imgIRequest **aRequest, nsIDOMNode *aDOMNode)
{
  NS_ENSURE_ARG(aDOMNode);
  NS_ENSURE_ARG_POINTER(aRequest);

  // Get content
  nsCOMPtr<nsIImageLoadingContent> content(do_QueryInterface(aDOMNode));
  NS_ENSURE_TRUE(content, NS_ERROR_FAILURE);

  return content->GetRequest(nsIImageLoadingContent::CURRENT_REQUEST,
                             aRequest);
}

/* static */
nsresult
EmbedContextMenuInfo2::CheckDomImageElement(nsIDOMNode *node, nsString& aHref,
                                           PRInt32 &aWidth, PRInt32 &aHeight, PRUint32 *aSize)
{
  nsresult rv;
  nsCOMPtr<nsIDOMHTMLImageElement> image =
    do_QueryInterface(node, &rv);
  NS_ENSURE_TRUE(image, rv);
  rv = image->GetSrc(aHref);
  NS_ENSURE_SUCCESS(rv, rv);
  image->GetWidth(&aWidth);
  image->GetHeight(&aHeight);
  if (!aSize) return rv;
  nsCOMPtr<imgIRequest> request;
  GetImageRequest(getter_AddRefs(request), node);
  NS_ENSURE_TRUE(request, rv);
  nsCOMPtr<imgIContainer> imgCont;
  request->GetImage(getter_AddRefs(imgCont));
  NS_ENSURE_TRUE(imgCont, rv);
  nsCOMPtr<gfxIImageFrame> currentFrame;
  imgCont->GetCurrentFrame(getter_AddRefs(currentFrame));
  NS_ENSURE_TRUE(currentFrame, rv);
  currentFrame->GetImageDataLength(aSize);
  return rv;
}

/* static */
nsresult
EmbedContextMenuInfo2::GetJSHandledInfo(nsIDOMEvent *aEvent, PRInt32 *aIsJSHandled)
{
  if (!aEvent && !aIsJSHandled)
    return NS_ERROR_FAILURE;
  nsresult rv = NS_ERROR_FAILURE;
#ifdef MOZ_JS_HANDLER_INFO
  if (aEvent) {
    PRUint32 flags;
    rv = aEvent->GetPrivateFlags(&flags);
    if (NS_SUCCEEDED(rv) && (flags & NS_PRIV_EVENT_FLAG_SCRIPT)) {
      mIsJSHandledElement = aIsJSHandled?*aIsJSHandled:1;
    } else {
      mIsJSHandledElement = 0;
      rv = NS_ERROR_FAILURE;
    }
  }
  if (aIsJSHandled)
    *aIsJSHandled = mIsJSHandledElement;
#endif
  return rv;
}

nsresult
EmbedContextMenuInfo2::InsertTextToNode(nsIDOMNode *aDOMNode, nsAString &aString, PRInt32 surr1, PRInt32 surr2)
{
  nsresult rv;

  nsCOMPtr<nsIDOMNode> targetNode = do_QueryInterface(aDOMNode, &rv);
  if (targetNode) {
    if (!(!mEventNode || mEventNode == targetNode) && mInsertNode != targetNode) {
      rv = CheckFormControl(aDOMNode);
      if (NS_SUCCEEDED(rv))
        mInsertNode = targetNode;
    }
  }
  else
    targetNode = do_QueryInterface(mEventNode, &rv);
  NS_ENSURE_SUCCESS(rv, rv);

  nsString nodeName, buffer;
  PRInt32 selectionStart = 0, selectionEnd = 0, textLength = 0;
  targetNode->GetNodeName(nodeName);
  if (mCtxFormType == NS_FORM_TEXTAREA) {
    nsCOMPtr<nsIDOMHTMLTextAreaElement> input;
    input = do_QueryInterface(targetNode, &rv);
    NS_ENSURE_SUCCESS(rv, rv);
    PRBool rdonly = PR_FALSE;
    input->GetReadOnly(&rdonly);
    if (rdonly) return NS_ERROR_FAILURE;

    nsCOMPtr<nsIDOMNSHTMLTextAreaElement> nsinput;
    nsinput = do_QueryInterface(targetNode, &rv);
    NS_ENSURE_SUCCESS(rv, rv);
    rv = input->GetValue(buffer);
    if (!buffer.IsEmpty()) {
      NS_ENSURE_SUCCESS(rv, rv);
      rv = input->GetValue(buffer);
      nsinput->GetSelectionStart(&selectionStart);
      nsinput->GetSelectionEnd(&selectionEnd);

      if (selectionStart != selectionEnd)
        buffer.Cut(selectionStart, selectionEnd - selectionStart);
      if (surr1 != 0 || surr2 != 0) {
        buffer.Cut(selectionStart + surr1, surr2);
        buffer.Insert(aString, selectionStart + surr1);
      } else
        buffer.Insert(aString, selectionStart);
    } else
      buffer.Assign(aString);

    input->SetValue(buffer);
    int len = aString.Length() + surr1;
    nsinput->SetSelectionRange(selectionStart + len, selectionStart + len);

    nsCOMPtr<nsIDOMNSEditableElement> nseditable;
    nseditable = do_QueryInterface(targetNode, &rv);
    NS_ENSURE_SUCCESS(rv, rv);

    nsCOMPtr<nsIEditor> theEditor;
    nseditable->GetEditor(getter_AddRefs(theEditor));
    NS_ENSURE_TRUE(theEditor, NS_OK);

    nsCOMPtr<nsISelectionController> selectionController;
    theEditor->GetSelectionController(getter_AddRefs(selectionController));
    PRBool sync = PR_TRUE;
    short selectionType = nsISelectionController::SELECTION_NORMAL;
    selectionController->ScrollSelectionIntoView(selectionType, selectionType, sync);
  }
  else if ((mCtx->type & GTK_MOZ_EMBED_CONTEXT_INPUT_TEXT) && !(mCtx->type & GTK_MOZ_EMBED_CONTEXT_RICHEDIT)) {
    nsCOMPtr<nsIDOMHTMLInputElement> input;
    input = do_QueryInterface(targetNode, &rv);
    NS_ENSURE_SUCCESS(rv, rv);
    PRBool rdonly = PR_FALSE;
    input->GetReadOnly(&rdonly);
    if (rdonly) return NS_ERROR_FAILURE;

    nsCOMPtr<nsIDOMNSHTMLInputElement> nsinput;
    nsinput = do_QueryInterface(targetNode, &rv);
    NS_ENSURE_SUCCESS(rv, rv);
    rv = input->GetValue(buffer);
    if (!buffer.IsEmpty()) {
      nsinput->GetSelectionStart(&selectionStart);
      nsinput->GetSelectionEnd(&selectionEnd);

      if (selectionStart != selectionEnd)
        buffer.Cut(selectionStart, selectionEnd - selectionStart);

      if (surr1 != 0 || surr2 != 0) {
        buffer.Cut(selectionStart + surr1, surr2);
        buffer.Insert(aString, selectionStart + surr1);
      } else
        buffer.Insert(aString, selectionStart);
    } else
      buffer.Assign(aString);

    input->SetValue(buffer);
    nsinput->SetSelectionRange(selectionStart + surr1 + aString.Length(), selectionStart + surr1 + aString.Length());

    nsCOMPtr<nsIDOMNSEditableElement> nseditable;
    nseditable = do_QueryInterface(targetNode, &rv);
    NS_ENSURE_SUCCESS(rv, rv);

    nsCOMPtr<nsIEditor> theEditor;
    nseditable->GetEditor(getter_AddRefs(theEditor));
    NS_ENSURE_TRUE(theEditor, NS_OK);

    nsCOMPtr<nsISelectionController> selectionController;
    theEditor->GetSelectionController(getter_AddRefs(selectionController));
    PRBool sync = PR_TRUE;
    short selectionType = nsISelectionController::SELECTION_NORMAL;
    selectionController->ScrollSelectionIntoView(selectionType, selectionType, sync);
  }
  else {
    nsCOMPtr<nsIWebBrowser> webBrowser;
    gtk_moz_embed_get_nsIWebBrowser(GTK_MOZ_EMBED(mEmbed), getter_AddRefs(webBrowser));
    NS_ENSURE_TRUE(webBrowser, NS_ERROR_FAILURE);
    nsCOMPtr<nsIEditingSession> editingSession = do_GetInterface(webBrowser);
    if (!editingSession)
      return NS_ERROR_FAILURE;

    nsCOMPtr<nsIEditor> theEditor;
    nsCOMPtr<nsPIDOMWindow> piWin;
    nsCOMPtr<nsIDocument> doc = do_QueryInterface(mCtxDocument);
    NS_ENSURE_TRUE(doc, NS_ERROR_FAILURE);
    piWin = doc->GetWindow();
    editingSession->GetEditorForWindow(piWin, getter_AddRefs(theEditor));
    NS_ENSURE_TRUE(theEditor, NS_ERROR_FAILURE);

    nsCOMPtr<nsIPlaintextEditor> textEditor;
    textEditor = do_QueryInterface(theEditor, &rv);
    if (NS_FAILED(rv) || !textEditor)
      return NS_ERROR_FAILURE;
    if (surr1 != 0 || surr2 != 0) {
      DoCommand(aDOMNode?(nsISupports*)aDOMNode:(nsISupports*)webBrowser, "cmd_delete", surr2);
    }
    textEditor->InsertText(aString);
  }
  return NS_OK;
}

nsresult
EmbedContextMenuInfo2::GetEventData()
{
  nsresult rv;

  nsCOMPtr<nsIDOMMouseEvent> mouseEvent(do_QueryInterface(mCtxEvent, &rv));
  if (!mouseEvent) {
    nsCOMPtr<nsIDOMNSElement> nselement;
    nselement = do_QueryInterface(mOrigNode, &rv);
    if (nselement) {
#ifndef MOZ_TRUNK_BUILD
      nsCOMPtr<nsIDOMTextRectangle> rect;
#else
      nsCOMPtr<nsIDOMClientRect> rect;
#endif
      nselement->GetBoundingClientRect(getter_AddRefs(rect));
      if (rect) {
        float l = 0, t = 0;
        //float r = 0, b = 0;
        rect->GetLeft((float*)&l);
        rect->GetTop((float*)&t);
        mCtx->x = (guint)trunc(l);
        mCtx->y = (guint)trunc(t);
      }
    }
  } else {
    PRUint16 btn = 1729;
    mouseEvent->GetButton (&btn);
    switch (btn) {
    case 0:
      mCtx->button = 1;
      break;
    case 1:
      mCtx->button = 2;
      break;
    case 2:
      mCtx->button = 3;
      break;
    case (PRUint16) -1:
    default:
      mCtx->button = 0;
      break;
    }
    if (mCtx->button != 0) {
      mouseEvent->GetClientX((PRInt32*)&mCtx->x);
      mouseEvent->GetClientY((PRInt32*)&mCtx->y);
      mouseEvent->GetScreenX((PRInt32*)&mCtx->x_root);
      mouseEvent->GetScreenY((PRInt32*)&mCtx->y_root);
    }
  }
  return NS_OK;
}

nsresult
EmbedContextMenuInfo2::UpdateContextData(nsIDOMEvent *aDOMEvent)
{
  NS_ENSURE_ARG_POINTER(aDOMEvent);

  nsresult rv;

  if (mCtxEvent == aDOMEvent)
    return NS_OK;

  mCtxEvent = do_QueryInterface(aDOMEvent, &rv);
  NS_ENSURE_SUCCESS(rv , NS_OK);

  PRUint16 eventphase;
  mCtxEvent->GetEventPhase(&eventphase);
  if (!eventphase) {
    mCtxEvent = nsnull;
    return NS_ERROR_FAILURE;
  }

  // Get mOrigNode
  {
    nsCOMPtr<nsIDOMNode> origNode;
    nsCOMPtr<nsIDOMNSEvent> aEvent = do_QueryInterface(mCtxEvent, &rv);
    NS_ENSURE_SUCCESS(rv , NS_OK);

    nsCOMPtr<nsIDOMEventTarget> origTarget;
    rv = aEvent->GetOriginalTarget(getter_AddRefs(origTarget));
    origNode = do_QueryInterface(origTarget, &rv);
    NS_ENSURE_SUCCESS(rv , NS_ERROR_NULL_POINTER);
    if (origNode == mOrigNode)
      return NS_OK;
    mOrigNode = origNode;
  }

  GetEventData();

  mEventNode = nsnull;
  mCtxDocument = nsnull;
  mCtx->docindex = -1;
  mCtx->type = GTK_MOZ_EMBED_CONTEXT_NONE;

  // Get mEventNode
  {
    nsCOMPtr<nsIDOMEventTarget> eventTarget;
    rv = mCtxEvent->GetTarget(getter_AddRefs(eventTarget));
    NS_ENSURE_SUCCESS(rv , NS_OK);
    mEventNode = do_QueryInterface(eventTarget, &rv);
  }
  NS_ENSURE_SUCCESS(rv , NS_OK);

  if (IsXULNode(mOrigNode, &mIsScrollableElement)) {
    mCtx->type |= GTK_MOZ_EMBED_CONTEXT_XUL;
    return NS_OK;
  }
  mIsScrollableElement = PR_TRUE;

  // Get mCtxDocument
  {
    nsCOMPtr<nsIDOMDocument> domDoc;
    rv = mEventNode->GetOwnerDocument(getter_AddRefs(domDoc));
    if (!NS_SUCCEEDED(rv) || !domDoc)
      rv = mOrigNode->GetOwnerDocument(getter_AddRefs(domDoc));
    if (NS_SUCCEEDED(rv) && domDoc && mCtxDocument != domDoc) {
      mCtxDocument = domDoc;
      nsCOMPtr<nsIDOM3Document> docuri = do_QueryInterface(mCtxDocument);
      nsString uri;
      docuri->GetDocumentURI(uri);
      SetStringProperty("docuri", uri);
      nsCOMPtr<nsIDocument> doc = do_QueryInterface(mCtxDocument);
      if (doc) {
        uri = doc->GetDocumentTitle();
        SetStringProperty("doctitle", uri);
      }
    }
  }
  SetStringProperty("link", "");
  SetStringProperty("image", "");

  nsCOMPtr<nsIDOMDocumentView> docView (do_QueryInterface (mCtxDocument));
  NS_ENSURE_TRUE (docView, NS_ERROR_FAILURE);
  nsCOMPtr<nsIDOMAbstractView> abstractView;
  docView->GetDefaultView (getter_AddRefs (abstractView));
  NS_ENSURE_TRUE (abstractView, NS_ERROR_FAILURE);
  mViewCSS = do_QueryInterface (abstractView);
  NS_ENSURE_TRUE (mViewCSS, NS_ERROR_FAILURE);

  nsCOMPtr<nsIWebNavigation> webNav (do_GetInterface (abstractView, &rv));
  NS_ENSURE_SUCCESS (rv, rv);

  rv = webNav->GetCurrentURI (getter_AddRefs (mBaseURI));
  NS_ENSURE_SUCCESS (rv, rv);

  SetFramePosition();
  CheckRichEdit();
  CheckFormControl();

  nsCOMPtr<nsIDOMNode> node;
  nsCOMPtr<nsIDOMNode> parentNode;

  nsCOMPtr<nsIDOMHTMLElement> element = do_QueryInterface(mEventNode, &rv);
  if (element)
    node = do_QueryInterface(mEventNode, &rv);
  else /* Fallback to XML/XSLT content */
    node = do_QueryInterface(mOrigNode, &rv);
  NS_ENSURE_SUCCESS(rv, rv);

  PRBool has_image = PR_FALSE;
  CheckDomHtmlNode(node, has_image);
  node->GetParentNode(getter_AddRefs(parentNode));
  node = parentNode;
  while (node) {
    if (NS_FAILED(CheckDomHtmlNode(node, has_image)))
      break;
    node->GetParentNode(getter_AddRefs(parentNode));
    node = parentNode;
  }
  mCtx->type |= GTK_MOZ_EMBED_CONTEXT_DOCUMENT;

  return NS_OK;
}

nsresult
EmbedContextMenuInfo2::GetContextText(nsAString &aString, PRBool aSelection, PRInt32 *aSelStart)
{
  nsresult rv;
  if (mCtxFormType != 0 && mEventNode) {
    PRInt32 selStart = 0, selEnd = 0;
    if (mCtxFormType == NS_FORM_INPUT_TEXT || mCtxFormType == NS_FORM_INPUT_FILE) {
      nsCOMPtr<nsIDOMHTMLInputElement> input = do_QueryInterface(mEventNode, &rv);
      if (NS_SUCCEEDED(rv) && input)
        rv = input->GetValue(aString);
      if (aString.IsEmpty()) return NS_ERROR_FAILURE;
      nsCOMPtr<nsIDOMNSHTMLInputElement> nsinput = do_QueryInterface(mEventNode, &rv);
      if (NS_SUCCEEDED(rv) && nsinput) {
        nsinput->GetSelectionEnd(&selEnd);
        nsinput->GetSelectionStart(&selStart);
      }
    } else if (mCtxFormType == NS_FORM_TEXTAREA) {
      nsCOMPtr<nsIDOMHTMLTextAreaElement> input = do_QueryInterface(mEventNode, &rv);
      if (NS_SUCCEEDED(rv) && input)
        rv = input->GetValue(aString);
      if (aString.IsEmpty()) return NS_ERROR_FAILURE;
      nsCOMPtr<nsIDOMNSHTMLTextAreaElement> nsinput = do_QueryInterface(mEventNode, &rv);
      if (NS_SUCCEEDED(rv) && nsinput) {
        nsinput->GetSelectionEnd(&selEnd);
        nsinput->GetSelectionStart(&selStart);
      }
    }
    if (selStart < selEnd && aSelection) {
      PRInt32 textLength = aString.Length();
      aString.Cut(0, selStart);
      aString.Cut(selEnd-selStart, textLength);
    }
    if (aSelStart) *aSelStart = selStart;
  } else if (mCtxDocument && aSelection) {
    nsCOMPtr<nsIDOMNSHTMLDocument> htmlDoc = do_QueryInterface(mCtxDocument, &rv);
    NS_ENSURE_SUCCESS(rv, rv);
    rv = htmlDoc->GetSelection(aString);
    if (NS_FAILED(rv) || aString.IsEmpty())
      return NS_ERROR_FAILURE;
  }
  return NS_OK;
}
