/*
 * tlv320aic33.c -- ALSA driver for Texas Instruments TLV320AIC33 Audio Codec
 *
 * Copyright (C) 2007 Nokia Corporation.
 *
 * Contact: Jarkko Nikula <jarkko.nikula@nokia.com>
 *          Eero Nurkkala
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/device.h>
#include <linux/platform_device.h>
#include <linux/delay.h>
#include <linux/errno.h>
#include <linux/i2c.h>
#include <linux/tlv320aic33.h>

#include <sound/driver.h>
#include <sound/initval.h>
#include <sound/control.h>

#define DRIVER_NAME			"tlv320aic33"

/* AIC33 Registers */
#define PLL_PROGRAMMING_A		3
#define PLL_PROGRAMMING_B		4
#define PLL_PROGRAMMING_C		5
#define PLL_PROGRAMMING_D		6
#define CODEC_DATAPATH			7
#define AUDIO_SERIAL_A			8
#define AUDIO_CODEC_OVERFLOW		11
#define AUDIO_DIGITAL_FILTER		12
#define HEADSET_DETECTION_B		14
#define LEFT_ADC_PGA_GAIN		15
#define RIGHT_ADC_PGA_GAIN		16
#define MIC3LR_TO_LEFT_ADC		17
#define MIC3LR_TO_RIGHT_ADC		18
#define LINE1L_TO_LEFT_ADC		19
#define LINE2L_TO_LEFT_ADC		20
#define LINE1R_TO_LEFT_ADC		21
#define LINE1R_TO_RIGHT_ADC		22
#define LINE2R_TO_RIGHT_ADC		23
#define LINE1L_TO_RIGHT_ADC		24
#define MICBIAS_CONTROL			25

#define DAC_POWER_CTRL			37
#define HPR_DRIVER_CTRL			38
#define HPR_STAGE_CTRL			40
#define DAC_OUTPUT_SWITCH		41

#define LEFT_DAC_DIGITAL_VOLUME		43
#define RIGHT_DAC_DIGITAL_VOLUME	44

#define LINE2L_TO_HPLOUT_VOLUME		45
#define PGA_L_TO_HPLOUT_VOLUME		46
#define DAC_L1_TO_HPLOUT_VOLUME		47
#define LINE2R_TO_HPLOUT_VOLUME		48
#define PGA_R_TO_HPLOUT_VOLUME		49
#define DAC_R1_TO_HPLOUT_VOLUME		50
#define HPLOUT_OUTPUT_CTRL		51

#define LINE2L_TO_HPLCOM_VOLUME		52
#define PGA_L_TO_HPLCOM_VOLUME		53
#define DAC_L1_TO_HPLCOM_VOLUME		54
#define LINE2R_TO_HPLCOM_VOLUME		55
#define PGA_R_TO_HPLCOM_VOLUME		56
#define DAC_R1_TO_HPLCOM_VOLUME		57
#define HPLCOM_OUTPUT_CTRL		58

#define LINE2L_TO_HPROUT_VOLUME		59
#define PGA_L_TO_HPROUT_VOLUME		60
#define DAC_L1_TO_HPROUT_VOLUME		61
#define LINE2R_TO_HPROUT_VOLUME		62
#define PGA_R_TO_HPROUT_VOLUME		63
#define DAC_R1_TO_HPROUT_VOLUME		64
#define HPROUT_OUTPUT_CTRL		65

#define LINE2L_TO_HPRCOM_VOLUME		66
#define PGA_L_TO_HPRCOM_VOLUME		67
#define DAC_L1_TO_HPRCOM_VOLUME		68
#define LINE2R_TO_HPRCOM_VOLUME		69
#define PGA_R_TO_HPRCOM_VOLUME		70
#define DAC_R1_TO_HPRCOM_VOLUME		71
#define HPRCOM_OUTPUT_CTRL		72

#define LINE2L_TO_MONO_VOLUME		73
#define PGA_L_TO_MONO_VOLUME		74
#define DAC_L1_TO_MONO_VOLUME		75
#define LINE2R_TO_MONO_VOLUME		76
#define PGA_R_TO_MONO_VOLUME		77
#define DAC_R1_TO_MONO_VOLUME		78
#define MONO_OUTPUT_CTRL		79

#define LINE2_TO_L_LO_VOLUME		80
#define PGA_L_TO_L_LO_VOLUME		81
#define DAC_L1_TO_L_LO_VOLUME		82
#define LINE2R_TO_L_LO_VOLUME		83
#define PGA_R_TO_L_LO_VOLUME		84
#define DAC_R1_TO_L_LO_VOLUME		85
#define LEFT_LO_OUTPUT_CTRL		86

#define LINE2L_TO_R_LO_VOLUME		87
#define PGA_L_TO_R_LO_VOLUME		88
#define DAC_L1_TO_R_LO_VOLUME		89
#define LINE2R_TO_R_LO_VOLUME		90
#define PGA_R_TO_R_LO_VOLUME		91
#define DAC_R1_TO_R_LO_VOLUME		92
#define RIGHT_LO_OUTPUT_CTRL		93

#define GPIO1_CONTROL			98
#define GPIO2_CONTROL			99

/* PLL definitions */
#define PLL_VAL_P(val)			((val & 0x7) << 0)
#define PLL_VAL_J(val)			((val & 0x3f) << 2)
#define PLL_VAL_D_8MSB(val)		(((val >> 6) & 0xff) << 0)
#define PLL_VAL_D_6LSB(val)		((val & 0x3f) << 0)
#define PLL_VAL_R(val)			((val & 0x07) << 0)
#define PLL_ENABLED			(1 << 7)

/* output stage level control register bits */
#define OUTPUT_POWERED			(1 << 0)
#define OUTPUT_UNMUTED			(1 << 3)
#define HP_OUTPUT_TRI_STATED		(1 << 2) /* only in HP control regs */

/* DAC definitions */
#define DAC_L1R1_ROUTE_ENABLE		(1 << 7)
#define DAC_LEFT_POWER			(1 << 7)
#define DAC_RIGHT_POWER			(1 << 6)
#define DAC_LEFT_MUTE			(1 << 7)
#define DAC_RIGHT_MUTE			(1 << 7)
#define DAC_LEFT_PLAYS_LEFT_CHANNEL	(1 << 3)
#define DAC_RIGHT_PLAYS_RIGHT_CHANNEL	(1 << 1)

/* ADC definitions */
#define ADC_LEFT_HIGHPASS(v)		(((v) & 3) << 6)
#define ADC_RIGHT_HIGHPASS(v)		(((v) & 3) << 4)
#define ADC_PGA_MUTE			(1 << 7)
#define ADC_POWER			(1 << 2)

/* Miscellaneous definitions */
#define DMIC_OVERSAMPLING_MASK		(0x3 << 0)
#define DMIC_OVERSAMPLING(v)		(((v) & 0x3) << 0)
#define DMIC_OVERSAMPLING_VAL(v)	(((v) >> 0) & 0x3)
#define GPIO1_CTRL(v)			(((v) & 0xf) << 4)
#define GPIO2_CTRL(v)			(((v) & 0xf) << 4)
#define MICBIAS_MASK			(0x3 << 6)
#define MICBIAS(v)			(((v) & 0x3) << 6)
#define MASTER_BCLK			(1 << 7)
#define MASTER_WCLK			(1 << 6)
#define HP_OUTPUT_AC_COUPLED		(1 << 7)

/* kcontrol builders */
#define AIC33_DOUBLE_R(xname, left_reg, right_reg, shift, maskbits, max, inv) \
{\
	.access = SNDRV_CTL_ELEM_ACCESS_READWRITE, \
	.iface = SNDRV_CTL_ELEM_IFACE_MIXER, \
	.name = xname, \
	.index = 0, \
	.info = aic33_info_double_r, \
	.get = aic33_get_double_r, \
	.put = aic33_put_double_r, \
	.private_value = left_reg | (right_reg << 8) | (max << 16) | \
			 (maskbits << 24) | (shift << 27) | (inv << 30) \
}
#define AIC33_DOUBLE_R_LREG(v)		((v) & 0x7f)
#define AIC33_DOUBLE_R_RREG(v)		(((v) >> 8) & 0x7f)
#define AIC33_DOUBLE_R_MAX(v)		(((v) >> 16) & 0x7f)
#define AIC33_DOUBLE_R_MASKBITS(v)	(((v) >> 24) & 0x7)
#define AIC33_DOUBLE_R_MASK(v)		~(~0 << AIC33_DOUBLE_R_MASKBITS(v))
#define AIC33_DOUBLE_R_SHIFT(v)		(((v) >> 27) & 0x7)
#define AIC33_DOUBLE_R_INV(v)		(((v) >> 30) & 0x1)

#define IG_MUTE				0xf
#define ig_index(reg)			((reg) - MIC3LR_TO_LEFT_ADC)

#define AIC33_IGAIN_DOUBLE(xname, reg, left_shift, right_shift, max) \
{\
	.access = SNDRV_CTL_ELEM_ACCESS_READWRITE, \
	.iface = SNDRV_CTL_ELEM_IFACE_MIXER, \
	.name = xname, \
	.index = 0, \
	.info = aic33_info_igain_double, \
	.get = aic33_get_igain_double, \
	.put = aic33_put_igain_double, \
	.private_value = reg | (left_shift << 8) | (right_shift << 12) | \
			 (max << 16) \
}
#define AIC33_IGAIN_DOUBLE_REG(v)		((v) & 0x7f)
#define AIC33_IGAIN_DOUBLE_LSHIFT(v)		(((v) >> 8) & 0x7)
#define AIC33_IGAIN_DOUBLE_RSHIFT(v)		(((v) >> 12) & 0x7)
#define AIC33_IGAIN_DOUBLE_MAX(v)		(((v) >> 16) & 0xf)

#define AIC33_IGAIN_DOUBLE_R(xname, left_reg, right_reg, shift, max) \
{\
	.access = SNDRV_CTL_ELEM_ACCESS_READWRITE, \
	.iface = SNDRV_CTL_ELEM_IFACE_MIXER, \
	.name = xname, \
	.index = 0, \
	.info = aic33_info_igain_double_r, \
	.get = aic33_get_igain_double_r, \
	.put = aic33_put_igain_double_r, \
	.private_value = left_reg | (right_reg << 8) | (shift << 16) | \
			 (max << 20) \
}
#define AIC33_IGAIN_DOUBLE_R_LREG(v)		((v) & 0x7f)
#define AIC33_IGAIN_DOUBLE_R_RREG(v)		(((v) >> 8) & 0x7f)
#define AIC33_IGAIN_DOUBLE_R_SHIFT(v)		(((v) >> 16) & 0x7)
#define AIC33_IGAIN_DOUBLE_R_MAX(v)		(((v) >> 20) & 0xf)

#define TLV320AIC33_BOOL(xname, func) \
{ \
	.access = SNDRV_CTL_ELEM_ACCESS_READWRITE,\
	.iface = SNDRV_CTL_ELEM_IFACE_MIXER, \
	.name = xname, \
	.index = 0, \
	.info = aic33_info_bool, \
	.get = aic33_get_bool, \
	.put = aic33_put_bool, \
	.private_value = (unsigned long)func \
}
#define	AIC33_BOOL_FUNC(v)		((int (*)(struct aic33 *, \
						  int set, int *p))v)

enum aic33_gpio1_ctrl {
	AIC33_GPIO1_DISABLED,
	AIC33_GPIO1_DMIC_CLK = 10,
};

enum aic33_gpio2_ctrl {
	AIC33_GPIO2_DISABLED,
	AIC33_GPIO2_DMIC_INPUT = 5,
};

static unsigned short normal_i2c[] = {
	0x18, 0x19, 0x1a, 0x1b, I2C_CLIENT_END
};

I2C_CLIENT_INSMOD;

static struct aic33_platform_data	*aic33_pdata;
static struct i2c_driver		aic33_i2c_driver;

static void aic33_setreg(struct aic33 *aic, u8 reg, u8 val)
{
	BUG_ON(reg > 127);

	aic->cache[reg] = val;
}
#define setreg(aic, eg, val)	aic33_setreg(aic, reg, val)

static u8 aic33_getreg(struct aic33 *aic, u8 reg)
{
	BUG_ON(reg > 127);

	return aic->cache[reg];
}
#define getreg(aic, reg)	aic33_getreg(aic, reg)

int aic33_write(struct aic33 *aic, u8 reg, u8 value)
{
	int err;
	u8 buffer[2];

	setreg(aic, reg, value);
	buffer[0] = reg;
	buffer[1] = value;
	err = i2c_master_send(&aic->client, &buffer[0], 2);
	if (err < 0) {
		dev_err(&aic->client.dev, "i2c error!\n");
		return err;
	}
	return 0;
}

static int aic33_read_regs(struct aic33 *aic, u8 reg, u8 *buf, int len)
{
	int err;
	struct i2c_adapter *adap = aic->client.adapter;
	struct i2c_msg msg[] = {
		{
			.addr	= aic->client.addr,
			.flags	= 0,
			.len	= 1,
			.buf	= &reg,
		},
		{
			.addr	= aic->client.addr,
			.flags	= I2C_M_RD,
			.len	= len,
			.buf	= buf,
		},
	};

	BUG_ON(reg + len > 127 + 1);
	err = i2c_transfer(adap, msg, 2);

	return (err < 0) ? err : len;
}

static int aic33_check_output(struct aic33 *aic, u8 reg, int *active)
{
	int changed = 0;
	int muted, powered, used;

	muted = !(aic->cache[reg] & OUTPUT_UNMUTED);
	powered = (aic->cache[reg] & OUTPUT_POWERED);
	*active = used = (aic->dac_enabled && !muted);

	if (!used && powered) {
		changed = 1;
	} else if (used && !powered) {
		changed = 1;
	}

	return changed;
}

static void aic33_change_output(struct aic33 *aic, u8 reg, int active)
{
	u8 temp;

	if (active) {
		aic33_write(aic, reg, (aic->cache[reg] | OUTPUT_POWERED) &
			    ~OUTPUT_UNMUTED);
		aic33_write(aic, reg, aic->cache[reg] | OUTPUT_UNMUTED);
	} else {
		/*
		 * preserve mute state set by ALSA control in cache even we
		 * mute the output stage in chip
		 */
		temp = aic->cache[reg];
		aic33_write(aic, reg, aic->cache[reg] & ~OUTPUT_UNMUTED);
		aic33_write(aic, reg, aic->cache[reg] & ~OUTPUT_POWERED);
		aic->cache[reg] |= (temp & OUTPUT_UNMUTED);
	}
}

static void aic33_output_power_ctrl(struct aic33 *aic)
{
	struct device *dev = &aic->client.dev;
	int l_changed, r_changed, l_act, r_act;
	int ext_act;

	/* high-power output power control */
	l_changed = aic33_check_output(aic, HPLOUT_OUTPUT_CTRL, &l_act);
	r_changed = aic33_check_output(aic, HPROUT_OUTPUT_CTRL, &r_act);
	if (l_changed || r_changed) {
		ext_act = (l_act || r_act);
		if (!ext_act && aic->ext_hp_amplifier_ctrl)
			aic->ext_hp_amplifier_ctrl(dev, 0);

		if (l_changed)
			aic33_change_output(aic, HPLOUT_OUTPUT_CTRL, l_act);
		if (r_changed)
			aic33_change_output(aic, HPROUT_OUTPUT_CTRL, r_act);

		if (ext_act && aic->ext_hp_amplifier_ctrl)
			aic->ext_hp_amplifier_ctrl(dev, 1);
	}

	/* line output power control */
	l_changed = aic33_check_output(aic, LEFT_LO_OUTPUT_CTRL, &l_act);
	r_changed = aic33_check_output(aic, RIGHT_LO_OUTPUT_CTRL, &r_act);
	if (l_changed || r_changed) {
		ext_act = (l_act || r_act);
		if (!ext_act && aic->ext_lo_amplifier_ctrl)
			aic->ext_lo_amplifier_ctrl(dev, 0);

		if (l_changed)
			aic33_change_output(aic, LEFT_LO_OUTPUT_CTRL, l_act);
		if (r_changed)
			aic33_change_output(aic, RIGHT_LO_OUTPUT_CTRL, r_act);

		if (ext_act && aic->ext_lo_amplifier_ctrl)
			aic->ext_lo_amplifier_ctrl(dev, 1);
	}
}

static void aic33_dmic_enable(struct aic33 *aic, int enable)
{
	enum aic33_dmic_rate rate;
	enum aic33_micbias bias;
	u8 val;

	if (enable) {
		/* activate bias if needed */
		if (aic->dmic_data->needs_bias) {
			bias = aic->dmic_data->dmic_bias;
			val = getreg(aic, MICBIAS_CONTROL) & ~MICBIAS_MASK;
			aic33_write(aic, MICBIAS_CONTROL, val | MICBIAS(bias));
		}
		/*
		 * enable digital mic. This must be done before GPIO
		 * configuration in order to get correct DMIC CLK. Otherwise
		 * ADC clock will show in DMIC CLK output before the rate is
		 * set
		 */
		rate = aic->dmic_data->dmic_rate;
		val = getreg(aic, AUDIO_SERIAL_A) & ~DMIC_OVERSAMPLING_MASK;
		aic33_write(aic, AUDIO_SERIAL_A,
			    val | DMIC_OVERSAMPLING(rate));
		/* configure GPIO1 and GPIO2 for digital mic */
		aic33_write(aic, GPIO1_CONTROL,
			    GPIO1_CTRL(AIC33_GPIO1_DMIC_CLK));
		aic33_write(aic, GPIO2_CONTROL,
			    GPIO2_CTRL(AIC33_GPIO2_DMIC_INPUT));
	} else {
		/* de-activate GPIO's and digital mic */
		aic33_write(aic, GPIO1_CONTROL,
			    GPIO1_CTRL(AIC33_GPIO1_DISABLED));
		aic33_write(aic, GPIO2_CONTROL,
			    GPIO2_CTRL(AIC33_GPIO2_DISABLED));
		val = getreg(aic, AUDIO_SERIAL_A) & ~DMIC_OVERSAMPLING_MASK;
		aic33_write(aic, AUDIO_SERIAL_A,
			    val | DMIC_OVERSAMPLING(AIC33_DMIC_DISABLED));
		/* de-activate bias */
		if (aic->dmic_data->needs_bias) {
			bias = AIC33_MICBIAS_OFF;
			val = getreg(aic, MICBIAS_CONTROL) & ~MICBIAS_MASK;
			aic33_write(aic, MICBIAS_CONTROL, val | MICBIAS(bias));
		}
	}
}

static void aic33_dmic_power_ctrl(struct aic33 *aic)
{
	int enable, changed = 0;
	u8 val;

	val = DMIC_OVERSAMPLING_VAL(getreg(aic, AUDIO_SERIAL_A));
	if (aic->adc_enabled &&
	    aic->dmic_enabled && (val == AIC33_DMIC_DISABLED)) {
		/*
		 * ADC enabled and digital mic enabled but it is not active
		 * so activate it now
		 */
		enable = 1;
		changed = 1;
	} else if (val != AIC33_DMIC_DISABLED &&
		   (!aic->dmic_enabled || !aic->adc_enabled)) {
		/*
		 * digital mic is active but either it or ADC are enabled
		 * so de-activate it now
		 */
		enable = 0;
		changed = 1;
	}

	if (changed)
		aic33_dmic_enable(aic, enable);
}

void aic33_power_ctrl(struct aic33 *aic)
{
	aic33_output_power_ctrl(aic);

	if (aic->dmic_data)
		aic33_dmic_power_ctrl(aic);

	if (aic->adc_enabled) {
		aic33_write(aic, LINE1L_TO_LEFT_ADC,
			    getreg(aic, LINE1L_TO_LEFT_ADC) | ADC_POWER);
		aic33_write(aic, LINE1R_TO_RIGHT_ADC,
			    getreg(aic, LINE1R_TO_RIGHT_ADC) | ADC_POWER);
	} else {
		aic33_write(aic, LINE1L_TO_LEFT_ADC,
			    getreg(aic, LINE1L_TO_LEFT_ADC) & ~ADC_POWER);
		aic33_write(aic, LINE1R_TO_RIGHT_ADC,
			    getreg(aic, LINE1R_TO_RIGHT_ADC) & ~ADC_POWER);
	}
}

static int aic33_info_double_r(struct snd_kcontrol *kcontrol,
			       struct snd_ctl_elem_info *uinfo)
{
	unsigned long priv = kcontrol->private_value;
	int mask = AIC33_DOUBLE_R_MASK(priv); /* switch: mask == 1 */
	int max = AIC33_DOUBLE_R_MAX(priv);

	uinfo->type = mask == 1 ? SNDRV_CTL_ELEM_TYPE_BOOLEAN :
				  SNDRV_CTL_ELEM_TYPE_INTEGER;
	uinfo->count = 2;
	uinfo->value.integer.min = 0;
	uinfo->value.integer.max = max;

	return 0;
}

static int aic33_get_double_r(struct snd_kcontrol *kcontrol,
			      struct snd_ctl_elem_value *ucontrol)
{
	struct aic33 *aic = kcontrol->private_data;
	u8 *cache = aic->cache;
	unsigned long priv = kcontrol->private_value;
	int lr = AIC33_DOUBLE_R_LREG(priv);
	int rr = AIC33_DOUBLE_R_RREG(priv);
	int mask = AIC33_DOUBLE_R_MASK(priv); /* switch: mask == 1 */
	int max = AIC33_DOUBLE_R_MAX(priv);
	int s = AIC33_DOUBLE_R_SHIFT(priv);
	int inv = AIC33_DOUBLE_R_INV(priv);

	mutex_lock(&aic->mutex);
	if (inv) {
		/*
		 * inverted control
		 * switch controls: 0 == active path, 1 == muted
		 * volume controls: 0 == maximum, mask == minimum
		 */
		ucontrol->value.integer.value[0] = max -
						   ((cache[lr] >> s) & mask);
		ucontrol->value.integer.value[1] = max -
						   ((cache[rr] >> s) & mask);
	} else {
		/*
		 * switch controls: 0 == muted, 1 == active path
		 * volume controls: 0 == minimum, mask == maximum
		 */
		ucontrol->value.integer.value[0] = (cache[lr] >> s) & mask;
		ucontrol->value.integer.value[1] = (cache[rr] >> s) & mask;
	}
	mutex_unlock(&aic->mutex);

	return 0;
}

static int aic33_put_double_r(struct snd_kcontrol *kcontrol,
			      struct snd_ctl_elem_value *ucontrol)
{
	struct aic33 *aic = kcontrol->private_data;
	u8 *cache = aic->cache;
	u8 lold, rold;
	unsigned long priv = kcontrol->private_value;
	int lr = AIC33_DOUBLE_R_LREG(priv);
	int rr = AIC33_DOUBLE_R_RREG(priv);
	int mask = AIC33_DOUBLE_R_MASK(priv); /* switch: mask == 1 */
	int max = AIC33_DOUBLE_R_MAX(priv);
	int s = AIC33_DOUBLE_R_SHIFT(priv);
	int inv = AIC33_DOUBLE_R_INV(priv);
	int changed = 0;

	mutex_lock(&aic->mutex);
	lold = cache[lr];
	rold = cache[rr];
	cache[lr] &= ~(mask << s);
	cache[rr] &= ~(mask << s);
	if (inv) {
		/*
		 * inverted control
		 * switch controls: 0 == active path, 1 == muted
		 * volume controls: 0 == maximum, mask == minimum
		 */
		cache[lr] |= ((max - ucontrol->value.integer.value[0]) &
			      mask) << s;
		cache[rr] |= ((max - ucontrol->value.integer.value[1]) &
			      mask) << s;
	} else {
		/*
		 * switch controls: 0 == muted, 1 == active path
		 * volume controls: 0 == minimum, mask == maximum
		 */
		cache[lr] |= (ucontrol->value.integer.value[0] & mask) << s;
		cache[rr] |= (ucontrol->value.integer.value[1] & mask) << s;
	}

	changed |= ((cache[lr] != lold) || (cache[rr] != rold));
	if (changed) {
		aic33_power_ctrl(aic);
		aic33_write(aic, lr, cache[lr]);
		aic33_write(aic, rr, cache[rr]);
	}
	mutex_unlock(&aic->mutex);

	return changed;
}

static int aic33_info_igain_double_r(struct snd_kcontrol *kcontrol,
				     struct snd_ctl_elem_info *uinfo)
{
	unsigned long priv = kcontrol->private_value;
	int max = AIC33_IGAIN_DOUBLE_R_MAX(priv); /* switch: max == 0xf */

	uinfo->type = max == IG_MUTE ? SNDRV_CTL_ELEM_TYPE_BOOLEAN :
				       SNDRV_CTL_ELEM_TYPE_INTEGER;
	uinfo->count = 2;
	uinfo->value.integer.min = 0;
	uinfo->value.integer.max = max == IG_MUTE ? 1 : max;

	return 0;

}

static int aic33_get_igain_double_r(struct snd_kcontrol *kcontrol,
				    struct snd_ctl_elem_value *ucontrol)
{
	struct aic33 *aic = kcontrol->private_data;
	u8 *cache = aic->cache;
	u8 *igain_cache = aic->igain_cache;
	u8 lval, rval;
	unsigned long priv = kcontrol->private_value;
	int lr = AIC33_IGAIN_DOUBLE_R_LREG(priv);
	int rr = AIC33_IGAIN_DOUBLE_R_RREG(priv);
	int s = AIC33_IGAIN_DOUBLE_R_SHIFT(priv);
	int mask = IG_MUTE;
	int max = AIC33_IGAIN_DOUBLE_R_MAX(priv);
	int sw = max == IG_MUTE ? 1 : 0;

	mutex_lock(&aic->mutex);
	if (sw) {
		/* switch. Path is muted if gain value is 0xf */
		lval = ((cache[lr] >> s) & mask) != IG_MUTE;
		rval = ((cache[rr] >> s) & mask) != IG_MUTE;
	} else {
		/* volume. All input gain volumes are inverted */
		lval = max - ((igain_cache[ig_index(lr)] >> s) & mask);
		rval = max - ((igain_cache[ig_index(rr)] >> s) & mask);
	}
	mutex_unlock(&aic->mutex);

	ucontrol->value.integer.value[0] = lval;
	ucontrol->value.integer.value[1] = rval;

	return 0;
}

static int aic33_put_igain_double_r(struct snd_kcontrol *kcontrol,
				    struct snd_ctl_elem_value *ucontrol)
{
	struct aic33 *aic = kcontrol->private_data;
	u8 *cache = aic->cache;
	u8 *igain_cache = aic->igain_cache;
	u8 lold, rold;
	u8 lval, rval;
	unsigned long priv = kcontrol->private_value;
	int lr = AIC33_IGAIN_DOUBLE_R_LREG(priv);
	int rr = AIC33_IGAIN_DOUBLE_R_RREG(priv);
	int s = AIC33_IGAIN_DOUBLE_R_SHIFT(priv);
	int mask = IG_MUTE;
	int max = AIC33_IGAIN_DOUBLE_R_MAX(priv);
	int sw = max == IG_MUTE ? 1 : 0;
	int changed = 0;

	mutex_lock(&aic->mutex);
	if (sw) {
		/*
		 * switch. If unmuted path, then use value from gain cache for
		 * updating register cache and mute value otherwise
		 */
		lold = (igain_cache[ig_index(lr)] >> s) & mask;
		rold = (igain_cache[ig_index(rr)] >> s) & mask;
		lval = ucontrol->value.integer.value[0] == 1 ? lold : IG_MUTE;
		rval = ucontrol->value.integer.value[1] == 1 ? rold : IG_MUTE;
	} else {
		/* volume */
		lval = ucontrol->value.integer.value[0] & mask;
		rval = ucontrol->value.integer.value[1] & mask;
		/* do sanity check and value inversion to input gain volume */
		lval = lval > max ? 0 : max - lval;
		rval = rval > max ? 0 : max - rval;
		/*
		 * update gain cache. Do register update below only if path is
		 * not muted
		 */
		igain_cache[ig_index(lr)] = lval << s;
		igain_cache[ig_index(rr)] = rval << s;
	}

	lold = cache[lr];
	rold = cache[rr];
	if (sw) {
		cache[lr] &= ~(mask << s);
		cache[rr] &= ~(mask << s);
		cache[lr] |= lval << s;
		cache[rr] |= rval << s;
	} else {
		/*
		 * if changing volume, update it into register cache only if
		 * path is not muted
		 */
		if (((cache[lr] >> s) & mask) != IG_MUTE) {
			cache[lr] &= ~(mask << s);
			cache[lr] |= igain_cache[ig_index(lr)] & (mask << s);
		}
		if (((cache[rr] >> s) & mask) != IG_MUTE) {
			cache[rr] &= ~(mask << s);
			cache[rr] |= igain_cache[ig_index(rr)] & (mask << s);
		}
	}

	changed |= ((cache[lr] != lold) || (cache[rr] != rold));
	if (changed) {
		aic33_write(aic, lr, cache[lr]);
		aic33_write(aic, rr, cache[rr]);
	}
	aic33_power_ctrl(aic);

	mutex_unlock(&aic->mutex);

	return changed;
}

static int aic33_info_bool(struct snd_kcontrol *kcontrol,
			   struct snd_ctl_elem_info *uinfo)
{
	uinfo->type = SNDRV_CTL_ELEM_TYPE_BOOLEAN;
	uinfo->count = 1;
	uinfo->value.integer.min = 0;
	uinfo->value.integer.max = 1;

	return 0;
}

static int aic33_get_bool(struct snd_kcontrol *kcontrol,
			  struct snd_ctl_elem_value *ucontrol)
{
	struct aic33 *aic = kcontrol->private_data;
	unsigned long priv = kcontrol->private_value;
	int (*func)(struct aic33 *, int set, int *p) =
		AIC33_BOOL_FUNC(priv);

	mutex_lock(&aic->mutex);
	ucontrol->value.integer.value[0] = func(aic, 0, NULL);
	mutex_unlock(&aic->mutex);

	return 0;
}

static int aic33_put_bool(struct snd_kcontrol *kcontrol,
			  struct snd_ctl_elem_value *ucontrol)
{
	struct aic33 *aic = kcontrol->private_data;
	unsigned long priv = kcontrol->private_value;
	int (*func)(struct aic33 *, int set, int *p) =
		AIC33_BOOL_FUNC(priv);
	int changed;
	int val = ucontrol->value.integer.value[0];

	mutex_lock(&aic->mutex);
	changed = func(aic, 1, &val);
	mutex_unlock(&aic->mutex);

	return changed;
}

static struct snd_kcontrol_new aic33_controls[] = {
	/* ADC/DAC gains */
	AIC33_DOUBLE_R("Capture Switch",
		LEFT_ADC_PGA_GAIN, RIGHT_ADC_PGA_GAIN, 7, 1, 1, 1),
	AIC33_DOUBLE_R("Capture Volume",
		LEFT_ADC_PGA_GAIN, RIGHT_ADC_PGA_GAIN, 0, 7, 119, 0),

	AIC33_DOUBLE_R("PCM Playback Switch",
		LEFT_DAC_DIGITAL_VOLUME, RIGHT_DAC_DIGITAL_VOLUME, 7, 1, 1, 1),
	AIC33_DOUBLE_R("PCM Playback Volume",
		LEFT_DAC_DIGITAL_VOLUME, RIGHT_DAC_DIGITAL_VOLUME, 0, 7, 127, 1),

	/*
	 * FIXME: Input gain control for pin MIC1/LINE1L. These stereo switch
	 * and volume controls route it into left and right ADC input.
	 * Note that this is Nokia RX-44 specific implementation and actually
	 * input routing configuration and control names should come from
	 * platform configuration. However, (too) versatile AIC33 doesn't
	 * make such implementation too easy...
	 */
	AIC33_IGAIN_DOUBLE_R("Mic1 Capture Switch",
		LINE1L_TO_LEFT_ADC, LINE1L_TO_RIGHT_ADC, 3, 0xf),
	AIC33_IGAIN_DOUBLE_R("Mic1 Capture Volume",
		LINE1L_TO_LEFT_ADC, LINE1L_TO_RIGHT_ADC, 3, 8),

	/* bypass path after input PGA to output stages */
	AIC33_DOUBLE_R("Capture_to_HP Bypass Switch",
		PGA_L_TO_HPLOUT_VOLUME, PGA_R_TO_HPROUT_VOLUME, 7, 1, 1, 0),
	AIC33_DOUBLE_R("Capture_to_HP Bypass Volume",
		PGA_L_TO_HPLOUT_VOLUME, PGA_R_TO_HPROUT_VOLUME, 0, 7, 117, 1),
	AIC33_DOUBLE_R("Capture_to_Line Bypass Switch",
		PGA_L_TO_L_LO_VOLUME, PGA_R_TO_R_LO_VOLUME, 7, 1, 1, 0),
	AIC33_DOUBLE_R("Capture_to_Line Bypass Volume",
		PGA_L_TO_L_LO_VOLUME, PGA_R_TO_R_LO_VOLUME, 0, 7, 117, 1),

	/* output stage volume controls */
	AIC33_DOUBLE_R("DAC_to_HP Playback Switch",
		DAC_L1_TO_HPLOUT_VOLUME, DAC_R1_TO_HPROUT_VOLUME, 7, 1, 1, 0),
	AIC33_DOUBLE_R("DAC_to_HP Playback Volume",
		DAC_L1_TO_HPLOUT_VOLUME, DAC_R1_TO_HPROUT_VOLUME, 0, 7, 117, 1),
	AIC33_DOUBLE_R("DAC_to_Line Playback Switch",
		DAC_L1_TO_L_LO_VOLUME, DAC_R1_TO_R_LO_VOLUME, 7, 1, 1, 0),
	AIC33_DOUBLE_R("DAC_to_Line Playback Volume",
		DAC_L1_TO_L_LO_VOLUME, DAC_R1_TO_R_LO_VOLUME, 0, 7, 117, 1),

	/* output stage level controls */
	AIC33_DOUBLE_R("HP Playback Switch",
		HPLOUT_OUTPUT_CTRL, HPROUT_OUTPUT_CTRL, 3, 1, 1, 0),
	AIC33_DOUBLE_R("HP Playback Volume",
		HPLOUT_OUTPUT_CTRL, HPROUT_OUTPUT_CTRL, 4, 4, 9, 0),
	AIC33_DOUBLE_R("Line Playback Switch",
		LEFT_LO_OUTPUT_CTRL, RIGHT_LO_OUTPUT_CTRL, 3, 1, 1, 0),
	AIC33_DOUBLE_R("Line Playback Volume",
		LEFT_LO_OUTPUT_CTRL, RIGHT_LO_OUTPUT_CTRL, 4, 4, 9, 0),
};

/*
 * Sets or gets digital mic enable status
 * Must be called while aic->mutex locked
 */
static int aic33_dmic_switch(struct aic33 *aic, int set, int *p)
{
	int ret = 1;

	if (set) {
		aic->dmic_enabled = *p;
		aic33_power_ctrl(aic);
	} else {
		ret = aic->dmic_enabled;
	}

	return ret;
}

static int aic33_mixer_init(struct aic33 *aic, struct snd_card *card)
{
	unsigned int idx;
	int err = 0;

	snd_assert(card != NULL, return -EINVAL);

	/* add AIC33 common/fixed controls */
	for (idx = 0; idx < ARRAY_SIZE(aic33_controls); idx++) {
		if ((err = snd_ctl_add(card, snd_ctl_new1(&aic33_controls[idx],
				       aic))) < 0) {
			goto out;
		}
	}

	/* add control for digital mic if configured */
	if (aic->dmic_data) {
		struct snd_kcontrol_new ctrl =
			TLV320AIC33_BOOL("Digital Mic Switch",
					 aic33_dmic_switch);
		err = snd_ctl_add(card, snd_ctl_new1(&ctrl, aic));
	}

out:
	return err;
}

static int aic33_setup_chip(struct aic33 *aic,
			    struct aic33_platform_data *pdata)
{
	int err;

	err = aic33_read_regs(aic, 0, aic->cache, sizeof(aic->cache));
	if (err < 0)
		goto out;

	/*
	 * DAC is connected into L1/R1 path by default. Use it since it allow
	 * to route DAC into every output via their mixer controls
	 * Route left DAC into left outputs and right DAC into right outputs
	 * respectively
	 */
	aic33_write(aic, DAC_L1_TO_HPLOUT_VOLUME, DAC_L1R1_ROUTE_ENABLE);
	aic33_write(aic, DAC_R1_TO_HPROUT_VOLUME, DAC_L1R1_ROUTE_ENABLE);
	aic33_write(aic, DAC_L1_TO_L_LO_VOLUME, DAC_L1R1_ROUTE_ENABLE);
	aic33_write(aic, DAC_R1_TO_R_LO_VOLUME, DAC_L1R1_ROUTE_ENABLE);

	aic33_write(aic, CODEC_DATAPATH, DAC_LEFT_PLAYS_LEFT_CHANNEL |
		    DAC_RIGHT_PLAYS_RIGHT_CHANNEL);

	/* PLL setup */
	aic33_write(aic, PLL_PROGRAMMING_A, PLL_VAL_P(pdata->pll_p));
	aic33_write(aic, PLL_PROGRAMMING_B, PLL_VAL_J(pdata->pll_j));
	aic33_write(aic, PLL_PROGRAMMING_C, PLL_VAL_D_8MSB(pdata->pll_d));
	aic33_write(aic, PLL_PROGRAMMING_D, PLL_VAL_D_6LSB(pdata->pll_d));
	aic33_write(aic, AUDIO_CODEC_OVERFLOW, PLL_VAL_R(pdata->pll_r));

	/* configure bit and word clock direction */
	aic33_write(aic, AUDIO_SERIAL_A,
		(pdata->bclk_output ? MASTER_BCLK : 0) |
		(pdata->wclk_output ? MASTER_WCLK : 0));

	/* configure ADC highpass filter */
	aic33_write(aic, AUDIO_DIGITAL_FILTER,
		    ADC_LEFT_HIGHPASS(pdata->adc_hp_filter) |
		    ADC_RIGHT_HIGHPASS(pdata->adc_hp_filter));

	if (pdata->hp_data) {
		/* configure high-power output */
		aic33_write(aic, HEADSET_DETECTION_B,
			(pdata->hp_data->ac_coupled ?
			 HP_OUTPUT_AC_COUPLED : 0));

		aic33_write(aic, HPLOUT_OUTPUT_CTRL,
			(pdata->hp_data->pd_tri_stated ?
			 HP_OUTPUT_TRI_STATED : 0));
		aic33_write(aic, HPROUT_OUTPUT_CTRL,
			(pdata->hp_data->pd_tri_stated ?
			 HP_OUTPUT_TRI_STATED : 0));
	}

	err = 0;
out:
	return err;
}

static void aic33_enable_dacs(struct aic33 *aic)
{
	aic33_write(aic, DAC_POWER_CTRL,
		    getreg(aic, DAC_POWER_CTRL) |
		    DAC_LEFT_POWER | DAC_RIGHT_POWER);
}

static void aic33_disable_dacs(struct aic33 *aic)
{
	aic33_write(aic, DAC_POWER_CTRL,
		    getreg(aic, DAC_POWER_CTRL) &
		    ~(DAC_LEFT_POWER | DAC_RIGHT_POWER));

	/* Power off timeout before MCLK will be shut down */
	schedule_timeout_interruptible(msecs_to_jiffies(20));
}

static void aic33_enable_pll(struct aic33 *aic)
{
	aic33_write(aic, PLL_PROGRAMMING_A,
		    getreg(aic, PLL_PROGRAMMING_A) | PLL_ENABLED);
	/* FIXME: short hack to prevent strong audible pop when powering
	 * up the audio. It seems that pop comes when we power up the DAC and
	 * output stage before PLL is locked since pop is missing if we keep
	 * PLL on. Real correction, however, is to use correct power up/down
	 * order between PLL, DAC and output stages and proper delay between
	 * them
	 */
	msleep(15);
}

static void aic33_disable_pll(struct aic33 *aic)
{
	aic33_write(aic, PLL_PROGRAMMING_A,
		    getreg(aic, PLL_PROGRAMMING_A) & ~PLL_ENABLED);
}

/* External interface - nonstatic functions */

void aic33_mixer_set_power(struct device *dev, int dac, int adc)
{
	struct aic33 *aic = container_of(dev, struct aic33, client.dev);

	dev_dbg(&aic->client.dev, "%s\n", __func__);

	mutex_lock(&aic->mutex);
	aic->adc_enabled = adc;
	aic->dac_enabled = dac;
	if (dac | adc)
		aic33_enable_pll(aic);
	if (dac)
		aic33_enable_dacs(aic);

	aic33_power_ctrl(aic);

	if (!dac)
		aic33_disable_dacs(aic);
	if (!(dac | adc))
		aic33_disable_pll(aic);
	mutex_unlock(&aic->mutex);
}

int aic33_mixer_register_controls(struct device *dev, struct snd_card *card)
{
	struct aic33 *aic = container_of(dev, struct aic33, client.dev);

	aic33_mixer_init(aic, card);

	return 0;
}

int aic33_enable_mclk(struct device *dev)
{
	struct aic33 *aic = container_of(dev, struct aic33, client.dev);
	int err = 0;

	dev_dbg(&aic->client.dev, "%s\n", __func__);

	mutex_lock(&aic->mutex);
	if (aic->enable_clock != NULL)
		err = aic->enable_clock(dev);
	mutex_unlock(&aic->mutex);

	return err;
}

void aic33_disable_mclk(struct device *dev)
{
	struct aic33 *aic = container_of(dev, struct aic33, client.dev);

	dev_dbg(&aic->client.dev, "%s\n", __func__);

	mutex_lock(&aic->mutex);
	if (aic->disable_clock != NULL)
		aic->disable_clock(dev);
	mutex_unlock(&aic->mutex);
}

static int aic33_detect_client(struct i2c_adapter *adapter, int address,
			       int kind)
{
	int err = 0;
	struct aic33 *aic;
	struct i2c_client *new_client;
	struct aic33_platform_data *pdata = aic33_pdata;

	if (!(aic = kzalloc(sizeof(*aic), GFP_KERNEL))) {
		err = -ENOMEM;
		goto err1;
	}

	mutex_init(&aic->mutex);
	new_client = &aic->client;
	i2c_set_clientdata(new_client, aic);
	new_client->addr = address;
	new_client->adapter = adapter;
	new_client->driver = &aic33_i2c_driver;
	new_client->flags = 0;
	strcpy(new_client->name, DRIVER_NAME);

	if ((err = i2c_attach_client(new_client))) {
		dev_err(&adapter->dev,
			"can't attach %s to address %d, err %d\n",
			DRIVER_NAME, address, err);
		goto err2;
	}

	aic->enable_clock = pdata->enable_clock;
	aic->disable_clock = pdata->disable_clock;
	aic->platform_init = pdata->codec_init;
	aic->platform_cleanup = pdata->codec_cleanup;
	aic->ext_hp_amplifier_ctrl = pdata->ext_hp_amplifier_ctrl;
	aic->ext_lo_amplifier_ctrl = pdata->ext_lo_amplifier_ctrl;
	aic->dmic_data = pdata->dmic_data;

	if ((err = aic33_setup_chip(aic, pdata))) {
		dev_err(&new_client->dev,
			"chip initialization failed, err %d\n", err);
		goto err3;
	}

	if (aic->platform_init) {
		if ((err = aic->platform_init(&new_client->dev))) {
			dev_err(&new_client->dev,
				"platform specific initialization failed, "
				"err %d\n",
				err);
			goto err3;
		}
	}

	return 0;

err3:
	i2c_detach_client(new_client);
err2:
	kfree(aic);
err1:
	return err;
}

static int aic33_attach_adapter(struct i2c_adapter *adapter)
{
	return i2c_probe(adapter, &addr_data, &aic33_detect_client);
}

static int aic33_detach_client(struct i2c_client *client)
{
	int err;
	struct aic33 *aic = i2c_get_clientdata(client);

	if (aic->platform_cleanup)
		aic->platform_cleanup(&client->dev);

	if ((err = i2c_detach_client(client)))
		return err;

	kfree(aic);

	return 0;
}

static struct i2c_driver aic33_i2c_driver = {
	.driver = {
		.name	= DRIVER_NAME,
	},
	.id		= I2C_DRIVERID_MISC,
	.attach_adapter	= aic33_attach_adapter,
	.detach_client	= aic33_detach_client,
};

/* Platform driver interface */
static int aic33_probe(struct platform_device *pdev)
{
	struct aic33_platform_data *pdata = pdev->dev.platform_data;

	if (aic33_pdata) {
		dev_err(&pdev->dev,
			"only single %s platform device supported\n",
			DRIVER_NAME);
		return -EBUSY;
	}
	aic33_pdata = pdata;

	if (pdata->codec_reset) {
		/* perform chip reset */
		pdata->codec_reset(1);
		udelay(1); /* spec says reset must be active at least 10 ns */
		pdata->codec_reset(0);
		msleep(1); /* stetson guessed reset release delay */
	}

	return i2c_add_driver(&aic33_i2c_driver);
}

static int aic33_remove(struct platform_device *pdev)
{
	struct aic33_platform_data *pdata = pdev->dev.platform_data;

	i2c_del_driver(&aic33_i2c_driver);
	/* keep chip in reset */
	if (pdata->codec_reset)
		pdata->codec_reset(1);

	aic33_pdata = NULL;

	return 0;
}

static struct platform_driver aic33_driver = {
	.driver = {
		.owner	= THIS_MODULE,
		.name	= "aic33-driver",
	},
	.probe		= aic33_probe,
	.remove		= aic33_remove,
};

static int __init aic33_init(void)
{
	if (platform_driver_register(&aic33_driver)) {
		printk(KERN_ERR "Failed to register %s driver\n",
		       DRIVER_NAME);
		return -ENODEV;
	}

	return 0;
}

static void __exit aic33_exit(void)
{
	platform_driver_unregister(&aic33_driver);
}

MODULE_AUTHOR("Jarkko Nikula <jarkko.nikula@nokia.com>");
MODULE_DESCRIPTION("TLV320AIC33 audio codec driver");
MODULE_LICENSE("GPL");

module_init(aic33_init);
module_exit(aic33_exit);
