/*
 * drivers/usb/core/otg.c
 *
 * Copyright (C) 2004 Texas Instruments
 * Copyright (C) 2007 Nokia Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

#include <linux/module.h>
#include <linux/usb.h>
#include <linux/usb/otg.h>

static struct otg_transceiver *xceiv;

int otg_set_transceiver(struct otg_transceiver *x)
{
	if (xceiv && x)
		return -EBUSY;
	xceiv = x;
	return 0;
}
EXPORT_SYMBOL(otg_set_transceiver);

struct otg_transceiver *otg_get_transceiver(void)
{
	if (xceiv)
		get_device(xceiv->dev);
	return xceiv;
}
EXPORT_SYMBOL(otg_get_transceiver);

/*
 * OTG MESSAGE: report errors here, customize to match your product
 * See also otg_last_error_show().
 */
void otg_set_error(struct otg_transceiver *x, enum usb_otg_error errno)
{
	if (!x)
		return;

	x->last_error = errno;
	if (!x->host->root_hub) {
		printk(KERN_WARNING "OTG: root hub not yet initialized\n");
		return;
	}

	sysfs_notify(&x->host->root_hub->dev.kobj, NULL, "otg_last_error");
}
EXPORT_SYMBOL(otg_set_error);

/*-------------------------------------------------------------------------*/

#ifdef	CONFIG_USB_OTG_WHITELIST

/*
 * This OTG Whitelist is the OTG "Targeted Peripheral List".  It should
 * mostly use of USB_DEVICE() or USB_DEVICE_VER() entries..
 *
 * YOU _SHOULD_ CHANGE THIS LIST TO MATCH YOUR PRODUCT AND ITS TESTING!
 */

static struct usb_device_id whitelist_table [] __initdata = {

/* USB OPT HS A electrical test devices are always targeted */
{ USB_DEVICE(0x1a0a, 0x0101), },
{ USB_DEVICE(0x1a0a, 0x0102), },
{ USB_DEVICE(0x1a0a, 0x0103), },
{ USB_DEVICE(0x1a0a, 0x0104), },
{ USB_DEVICE(0x1a0a, 0x0105), },
{ USB_DEVICE(0x1a0a, 0x0106), },
{ USB_DEVICE(0x1a0a, 0x0107), },
{ USB_DEVICE(0x1a0a, 0x0108), },

/* hubs are optional in OTG, but very handy ... */
{ USB_DEVICE_INFO(USB_CLASS_HUB, 0, 0), },
{ USB_DEVICE_INFO(USB_CLASS_HUB, 0, 1), },

#ifdef	CONFIG_USB_PRINTER		/* ignoring nonstatic linkage! */
/* FIXME actually, printers are NOT supposed to use device classes;
 * they're supposed to use interface classes...
 */
{ USB_DEVICE_INFO(7, 1, 1) },
{ USB_DEVICE_INFO(7, 1, 2) },
{ USB_DEVICE_INFO(7, 1, 3) },
#endif

#ifdef	CONFIG_USB_NET_CDCETHER
/* Linux-USB CDC Ethernet gadget */
{ USB_DEVICE(0x0525, 0xa4a1), },
/* Linux-USB CDC Ethernet + RNDIS gadget */
{ USB_DEVICE(0x0525, 0xa4a2), },
#endif

#if	defined(CONFIG_USB_TEST) || defined(CONFIG_USB_TEST_MODULE)
/* gadget zero, for testing */
{ USB_DEVICE(0x0525, 0xa4a0), },
#endif

/*
 * Support known mass storage devices
 * REVISIT: Get this data from platform data
 */
{ USB_DEVICE(0x0421, 0x0431), }, /* N770 */
{ USB_DEVICE(0x0421, 0x04c3), }, /* N800 */
{ USB_DEVICE(0x0421, 0x0150), }, /* RX44 */
{ USB_DEVICE(0x0421, 0x003b), }, /* Lynn */
{ USB_DEVICE(0x0781, 0x9393), }, /* Sandisk ImageMate SD-MMC */

{ }	/* Terminating entry */
};

struct otg_device {
	struct list_head	list;
	struct usb_device_id	id;
};

static struct list_head	tpl_devices;

static void tpl_add_device(struct usb_device_id *id)
{
	struct otg_device	*otg;

	otg = kzalloc(sizeof(*otg), GFP_KERNEL);
	if (!otg)
		return;
	INIT_LIST_HEAD(&otg->list);
	memcpy(&otg->id, id, sizeof(struct usb_device_id));
	list_add_tail(&otg->list, &tpl_devices);
}

static void tpl_add_devices(struct usb_device_id *whitelist_table)
{
	struct usb_device_id	*id;

	for (id = whitelist_table; id->match_flags; id++)
		tpl_add_device(id);
}

ssize_t otg_print_whitelist(char *buf)
{
	struct otg_device	*otg;
	ssize_t			len = 0;

	list_for_each_entry(otg, &tpl_devices, list)
		len += sprintf(buf + len, "%04x:%04x\t%02x %02x %02x\n",
				le16_to_cpu(otg->id.idVendor),
				le16_to_cpu(otg->id.idProduct),
				otg->id.bDeviceClass,
				otg->id.bDeviceSubClass,
				otg->id.bDeviceProtocol);
	return len;
}

int otg_targeted(struct usb_device *dev)
{
	struct otg_device	*otg;

	if (!xceiv || !xceiv->tpl_enabled)
		return OTG_ERR_DEVICE_SUPPORTED;

	/* possible in developer configs only! */
	if (!dev->bus->otg_port)
		goto targeted;

	/* HNP test device is _never_ targeted (see OTG spec 6.6.6) */
	if ((le16_to_cpu(dev->descriptor.idVendor) == 0x1a0a &&
	     le16_to_cpu(dev->descriptor.idProduct) == 0xbadd))
		goto err;

	/* NOTE: can't use usb_match_id() since interface caches
	 * aren't set up yet. this is cut/paste from that code.
	 */
	list_for_each_entry(otg, &tpl_devices, list) {
		struct usb_device_id	*id = &otg->id;

		if ((id->match_flags & USB_DEVICE_ID_MATCH_VENDOR) &&
		    id->idVendor != le16_to_cpu(dev->descriptor.idVendor))
			continue;

		if ((id->match_flags & USB_DEVICE_ID_MATCH_PRODUCT) &&
		    id->idProduct != le16_to_cpu(dev->descriptor.idProduct))
			continue;

		/* No need to test id->bcdDevice_lo != 0, since 0 is never
		   greater than any unsigned number. */
		if ((id->match_flags & USB_DEVICE_ID_MATCH_DEV_LO) &&
		    (id->bcdDevice_lo > le16_to_cpu(dev->descriptor.bcdDevice)))
			continue;

		if ((id->match_flags & USB_DEVICE_ID_MATCH_DEV_HI) &&
		    (id->bcdDevice_hi < le16_to_cpu(dev->descriptor.bcdDevice)))
			continue;

		if ((id->match_flags & USB_DEVICE_ID_MATCH_DEV_CLASS) &&
		    (id->bDeviceClass != dev->descriptor.bDeviceClass))
			continue;

		if ((id->match_flags & USB_DEVICE_ID_MATCH_DEV_SUBCLASS) &&
		    (id->bDeviceSubClass!= dev->descriptor.bDeviceSubClass))
			continue;

		if ((id->match_flags & USB_DEVICE_ID_MATCH_DEV_PROTOCOL) &&
		    (id->bDeviceProtocol != dev->descriptor.bDeviceProtocol))
			continue;

		goto targeted;
	}

	/* add other match criteria here ... */

err:
	dev_err(&dev->dev, "device v%04x p%04x is not supported\n",
		le16_to_cpu(dev->descriptor.idVendor),
		le16_to_cpu(dev->descriptor.idProduct));
	otg_set_error(xceiv, OTG_ERR_DEVICE_NOT_SUPPORTED);
	return xceiv->last_error;

targeted:
	xceiv->last_error = OTG_ERR_DEVICE_SUPPORTED;
	return xceiv->last_error;
}

static int __init tpl_init(void)
{
	INIT_LIST_HEAD(&tpl_devices);
	tpl_add_devices(whitelist_table);
	if (xceiv)
		xceiv->tpl_enabled = 1;

	return 0;
}

static void __exit tpl_exit(void)
{
	struct otg_device	*otg;

	if (xceiv)
		xceiv->tpl_enabled = 0;

	list_for_each_entry(otg, &tpl_devices, list)
		kfree(otg);
}

module_init(tpl_init);
module_exit(tpl_exit);

#endif	/* CONFIG_USB_OTG_WHITELIST */
