/*
 * pc2400m_osal.c
 *
 * Copyright (C) 2007 Nokia Corporation
 * Author: Juuso Oikarinen <juuso.oikarinen@nokia.com>
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 */


#include <linux/kernel.h>
#include <linux/netdevice.h>
#include <linux/skbuff.h>
#include <linux/delay.h>
#include <linux/timer.h>
#include "pc2400m_com.h"
#include "pc2400m_if_netlink.h"
#include "pc2400m_osal.h"
#include "pc2400m_osal_trace.h"

/* trace and trace group string definitions */
#define TRACE_GROUP_BEGIN(name, descr, on) descr,
#define TRACE_BEGIN(name, descr)
#define TRACE_GROUP_END()
static u8 *wimax_osal_trace_group_str[] = {
#include PC2400M_OSAL_TRACE_INCLUDE
};
#undef TRACE_GROUP_BEGIN
#undef TRACE_GROUP_END
#undef TRACE_BEGIN

#define TRACE_GROUP_BEGIN(name, descr, on)
#define TRACE_BEGIN(name, descr) descr,
#define TRACE_GROUP_END()
static u8 *wimax_osal_trace_id_str[] = {
#include PC2400M_OSAL_TRACE_INCLUDE
};
#undef TRACE_GROUP_BEGIN
#undef TRACE_GROUP_END
#undef TRACE_BEGIN


struct wimax_osal_timer_data {
	struct timer_list *timer;
	wimax_osal_context *ctx;
	struct work_struct timer_work;
	wimax_osal_timer_cb cb;
	void *data;
	s32 timeout;
	s32 periodic;
	s32 resolution;
        u32 in_handler;
};


static void wimax_osal_timer_work(struct work_struct *work)
{

	struct wimax_osal_timer_data *td = 
	        container_of(work, 
			     struct wimax_osal_timer_data, 
			     timer_work);

	/* this is outside the if to guarantee sync with the 
	   wimax_osal_timer_cancel function */
	pc2400m_com_lock(td->ctx);
	td->in_handler = 1;
	if (td->cb && td->timeout != -1) {
		td->cb((s32)td, td->data);
	}
	kfree(td->timer);
	kfree(td);
	pc2400m_com_unlock();

}

void wimax_osal_timer_handler(unsigned long data) 
{

	struct wimax_osal_timer_data *td = (struct wimax_osal_timer_data*)data;
	struct work_struct *work = &td->timer_work;
	del_timer(td->timer);
	schedule_work(work);

}

/**
 * wimax_osal_timer_create - create a timer
 * @timeout: timeout in microseconds for the timer
 * @resolution: required accuracy of the timer (WIMAX_OSAL_TR_xxxx)
 * @periodic: if non-zero, a peridic timer, if zero a one shot timer is req'd
 * @data: a cookie passed to the call back
 * @cb: the callback function to invoke upon expiration
 *
 * Start a timer and register a call back function to be invoked upon
 * expiration. A resolution request may be provided to help the implementation
 * choose between possible timer implementations, but this is only advisory.
 */
s32 wimax_osal_timer_create(s32 timeout, 
			    enum WIMAX_OSAL_TR resolution, 
			    boolean periodic, 
			    void *data, 
			    wimax_osal_timer_cb cb)
{
  
        struct timer_list *timer = 
	        wimax_osal_mem_alloc(sizeof(struct timer_list));
	struct wimax_osal_timer_data *td = 
	        wimax_osal_mem_alloc(sizeof(struct wimax_osal_timer_data));

	memset(timer, 0x00, sizeof(struct timer_list));
	memset(td, 0x00, sizeof(struct wimax_osal_timer_data));

	/* negative timeouts are considered a special value */
        BUG_ON(timeout < 0);

	/* this implementation does not support periodic timers */
	BUG_ON(periodic);

	td->timer = timer;
	td->cb = cb;
	td->data = data;
	td->timeout = timeout;
	td->periodic = periodic;
	td->resolution = resolution;
	td->in_handler = 0;
	td->ctx = pc2400m_com_running_context;
	INIT_WORK(&td->timer_work, wimax_osal_timer_work);

	setup_timer(timer, wimax_osal_timer_handler, 
		    (unsigned long)td);
	mod_timer(timer, jiffies + (timeout / 1000) * HZ / 1000);

	return (s32)(td);

}

/**
 * wimax_osal_timer_cancel - cancel a timer
 * @handle: handle (returned upon creation) of the timer
 *
 * Cancel an ongoing timer. This function guarantees, that the timeout
 * call back will not be called.
 */
s32 wimax_osal_timer_cancel(s32 handle) 
{
	struct wimax_osal_timer_data *td = 
	        (struct wimax_osal_timer_data*)handle;

	if (td && !td->in_handler) {

		/* this guarantees that the timer expiration function has 
		   executed before return; this is safe, as the expiration does
		   not contain synchronization with the common lock */
		if (del_timer_sync(td->timer)) {
			kfree(td->timer);
			kfree(td);
		} else {
			/* note that it is quaranteed this is synched with
			   the timer_work function locks */
			td->timeout = -1;
		}

	}

	return 0;
}


/**
 * wimax_osal_packet_list_initialize - initialize a packet list
 * @list: pointer to the packet list
 *
 * This function initializes the provided packet list. If there is a list upon
 * calling it will be lost and the associated wimax_osal_packets corrupted 
 * (their next and prev vars will point although the list is gone.)
 */
void wimax_osal_packet_list_initialize(wimax_osal_packet_list *list)
{

	memset(list, 0x00, sizeof(wimax_osal_packet_list));
	return;

}

/**
 * wimax_osal_packet_list_append - append an osal packet to a packet list
 * @list: pointer to the packet list
 * @delete: zero to not free the list elements, nonzero to free them also
 *
 * Free the memory taken by the list, by removing the list items and freeing
 * them if requested.
 */
void wimax_osal_packet_list_release(wimax_osal_packet_list *list, s32 delete)
{

	struct sk_buff *skb = list->first;
	struct sk_buff *skb2;

	while (skb) {
		skb2 = skb->next;
		if (delete) {
			kfree_skb(skb);
		} else {
			skb->next = NULL;
			skb->prev = NULL;
		}
		skb = skb2;
	}

	list->len = 0;
	list->first = NULL;
	list->last = NULL;

	return;
}

/**
 * wimax_osal_packet_list_append - append an osal packet to a packet list
 * @list: pointer to the packet list
 * @packet: pointer to the packet
 *
 * Append the wimax_osal_packet to a linked list (it goes to the end of the 
 * list.) The osal_packet can be part of only one list at a time.
 */
void wimax_osal_packet_list_append(wimax_osal_packet_list *list, 
				   wimax_osal_packet *packet)
{

	BUG_ON(packet->next || packet->prev);

	if (list->last) {
		list->last->next = packet;
		packet->prev = list->last;
		list->last = packet;
	} else {
		list->first = packet;
		packet->prev = NULL;
		list->last = packet;
	}
	packet->next = NULL;

	list->len ++;

}

/**
 * wimax_osal_packet_list_push - push an osal packet to a packet list
 * @list: pointer to the packet list
 * @packet: pointer to the packet
 *
 * Push the wimax_osal_packet to a linked list (it goes to the top of the 
 * list.) The osal_packet can be part of only one list at a time.
 */
void wimax_osal_packet_list_push(wimax_osal_packet_list *list, 
				 wimax_osal_packet *packet)
{

	BUG_ON(packet->next || packet->prev);

	packet->next = list->first;
	packet->prev = NULL;
	list->first = packet;

	if (!list->last) list->last = packet;

	list->len ++;

}

/**
 * wimax_osal_packet_list_pop - get the topmost osal packet from the packet 
 * list
 * @list: pointer to the packet list
 *
 * Return the topmost packet from the packet list or NULL if the list is
 * empty. After append and push, the osal packet must be popped before it can
 * be put into another list.
 */
wimax_osal_packet *wimax_osal_packet_list_pop(wimax_osal_packet_list *list)
{

        wimax_osal_packet *pkt = list->first;

	if (pkt) {
		if (pkt->next) pkt->next->prev = NULL;
		list->first = list->first->next;
		if (!list->first) list->last = NULL;

		pkt->next = NULL;
		pkt->prev = NULL;

		list->len --;
	}

	return pkt;

}

inline static void osal_priority(u8 priority) 
{
	switch (priority) {
	case WIMAX_OSAL_PRIORITY_DEBUG:
	case WIMAX_OSAL_PRIORITY_DIAGNOSTIC:
		printk(KERN_DEBUG DRIVER_SUFFIX);
		break;
	case WIMAX_OSAL_PRIORITY_ERROR:
		printk(KERN_ERR DRIVER_SUFFIX);
		break;
	case WIMAX_OSAL_PRIORITY_INFO:
		printk(KERN_INFO DRIVER_SUFFIX);
		break;
	}
}
	

inline static void osal_trace_to_pipe(u32 type,
				      u32 group_id, 
				      u32 trace_id, 
				      u32 len,
				      u8 *ptr)
{

	wimax_osal_packet *msg;
	u8 *traceptr;
	struct netlink_trace_msg_str *trace;


	if (!pc2400m_netlink_trace_pipe_ready(
		    pc2400m_com_running_context)) goto out;

	/* forward the trace data to the trace pipe */
	msg = wimax_osal_packet_alloc(
		sizeof(struct netlink_trace_msg_str) + len);
	trace = (struct netlink_trace_msg_str*)
		wimax_osal_packet_put(
			msg, sizeof(struct netlink_trace_msg_str));
	trace->header.type = NETLINK_PIPE_TYPE_TRACE;
	trace->header.length = sizeof(struct netlink_trace_msg_str) + len;
	trace->header.version = NETLINK_PIPE_VERSION;
	trace->trace_type = type;
	trace->trace_group = group_id;
	trace->trace_id = trace_id;
	trace->trace_len = len;

	traceptr = (u8*)wimax_osal_packet_put(msg, len);
	wimax_osal_mem_cpy(traceptr, ptr, len);

	pc2400m_netlink_send_data(
		pc2400m_com_running_context, msg, NETLINK_PC2400M3);

 out:
	return;
}

void wimax_osal_trace(u32 group_id, u32 trace_id, u8 priority)
{

	osal_priority(priority);
	printk("%s: %s:\n", 
	       wimax_osal_trace_group_str[group_id],
	       wimax_osal_trace_id_str[trace_id]);
	osal_trace_to_pipe(
		NETLINK_TRACE_TYPE_EMPTY, group_id, trace_id, 0, NULL);

}

void wimax_osal_trace_byte(u32 group_id, u32 trace_id, u8 priority, u8 param)
{

	osal_priority(priority);
	printk("%s: %s: Byte %2.2x\n", 
	       wimax_osal_trace_group_str[group_id],
	       wimax_osal_trace_id_str[trace_id],
	       param);
	osal_trace_to_pipe(
		NETLINK_TRACE_TYPE_BYTE, group_id, trace_id, 1, &param);

}

void wimax_osal_trace_u16(u32 group_id, u32 trace_id, u8 priority, u16 param)
{

	osal_priority(priority);
	printk("%s: %s: Short %4.4x\n", 
	       wimax_osal_trace_group_str[group_id],
	       wimax_osal_trace_id_str[trace_id],
	       param);
	osal_trace_to_pipe(
		NETLINK_TRACE_TYPE_U16, group_id, trace_id, 2, (u8*)&param);
}

void wimax_osal_trace_u32(u32 group_id, u32 trace_id, u8 priority, u32 param)
{

	osal_priority(priority);
	printk("%s: %s: Long %8.8x\n", 
	       wimax_osal_trace_group_str[group_id],
	       wimax_osal_trace_id_str[trace_id],
	       param);
	osal_trace_to_pipe(
		NETLINK_TRACE_TYPE_U32, group_id, trace_id, 4, (u8*)&param);

}

void wimax_osal_trace_str(u32 group_id, u32 trace_id, u8 priority, u8 *param)
{

	osal_priority(priority);
	printk("%s: %s: Str %s\n", 
	       wimax_osal_trace_group_str[group_id],
	       wimax_osal_trace_id_str[trace_id],
	       param);
	osal_trace_to_pipe(
		NETLINK_TRACE_TYPE_STR, group_id, trace_id, 
		wimax_osal_str_len(param)+1, param);

}

void wimax_osal_trace_data(
	u32 group_id, u32 trace_id, u8 priority, u8 *param, u32 len)
{

	s32 j, k;

	if (priority != WIMAX_OSAL_PRIORITY_DIAGNOSTIC) {
		osal_priority(priority);

		printk("%s: %s: Data %d bytes:", 
		       wimax_osal_trace_group_str[group_id], 
		       wimax_osal_trace_id_str[trace_id], 
		       len);
		for (j=0, k=0; j<len; j++) {
			if (!k) {
				printk("\n");
				osal_priority(priority);
			}
			k++;
			printk("%2.2x ", param[j]);
			if (k==16) k=0;
		}
		printk("\n");
	}

	osal_trace_to_pipe(
		NETLINK_TRACE_TYPE_HEX, group_id, trace_id, len, param);

}
