/*
 * pc2400m_drv_dm.c
 *
 * Copyright (C) 2007 Nokia Corporation
 * Author: Juuso Oikarinen <juuso.oikarinen@nokia.com>
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 */


#include "pc2400m_drv_com.h"
#include "pc2400m_drv_dm.h"
#include "pc2400m_drv_cd.h"
#include "pc2400m_drv_hdi.h"
#include "pc2400m_drv_sllist.h"

#define DM_MSG_ENQUEUE(msg,flags) \
	     hnd->cd->msg_enqueue(hnd->cd, msg, pc2400m_drv_dm_resp_handler, \
             (void*)hnd, flags)

#define PAD4(x)    (((x) + (4-1)) & (~(4-1)))


/* sub states */
/* PC2400M_DRV_DM_GET_DEVICE_INFO */
#define DM_STATE_GET_DEVICE_INFO_NONE   0
#define DM_STATE_GET_DEVICE_INFO_L4M    1

/* PC2400M_DRV_DM_ENABLE_RADIO */
#define DM_STATE_RADIO_ON_MODE_REQ      0
#define DM_STATE_RADIO_ON_MODE_REQ_DONE 1
#define DM_STATE_RADIO_ON_REQ           2
#define DM_STATE_RADIO_ON_COMPLETE      3

/* PC2400M_DRV_DM_DISABLE_RADIO */
#define DM_STATE_RADIO_OFF_REQ          0
#define DM_STATE_RADIO_OFF_COMPLETE     1

/* PC2400M_DRV_DM_START_SCAN */
#define DM_STATE_START_SCAN_PARAMS      0
#define DM_STATE_START_SCAN_REQ         1

/* PC2400M_DRV_DM_INIT */
#define DM_STATE_INIT_WAITING_CONFIG    0
#define DM_STATE_INIT_REQ               1
#define DM_STATE_INIT_WAITING_CH        2

/* PC2400M_DRV_DM_TERMINATE */
#define DM_STATE_TERMINATE_REQ          0
#define DM_STATE_TERMINATE_WAITING_CH   1

/* PC2400M_DRV_DM_PRODUCTION */
#define DM_STATE_PRODUCTION_REQ         0
#define DM_STATE_PRODUCTION_WAITING_CH  1


/* transition state and status information */
struct pc2400m_drv_dm_trans {
	struct sllist list;

	/* owning instance */
	struct pc2400m_drv_dm_if *hnd;

	/* transition type */
	enum pc2400m_drv_dm_transitions id;

	/* parameter data */
	void *params;

	/* transition internal data */
	union {
		u32 state;
	} intdata;

        /* flags */
#define DM_TRANS_FLAG_COMPLETING 0x00000001
        u32 flags;

	/* transition completion callback */
	dm_transition_cb cb;
	void *cb_data;
	void *cb_cl_data;

	/* timeout timer handle */
	s32 timerhnd;

};


/* state macros */
#define GOTO_STATE(x,y)       { ((x) = (y)); wimax_osal_trace_byte(\
                                  PC2400M_GROUP_DM_DEBUG, \
                                  PC2400M_TRACE_DM_STATE_CHANGE, \
                                  WIMAX_OSAL_PRIORITY_DEBUG, (y)); }
                      
#define RESET_STATE(x)        ((x) = L3L4_SYSTEM_STATE_UNINITIALIZED)


/* state managing helper */
static void dm_trans_execute(struct pc2400m_drv_dm_if *hnd);
static void dm_trans_complete(struct pc2400m_drv_dm_if *hnd, s32 status);
static void dm_trans_abort(struct pc2400m_drv_dm_if *hnd);

/* state machine handlers */
static void pc2400m_drv_dm_timeout_handler(s32 id, void* data);
static void pc2400m_drv_dm_resp_handler(struct pc2400m_drv_cd_if*, 
					wimax_osal_packet*, void*);

/* indication handlers */
static void dm_ind_report_state(struct pc2400m_drv_cd_if *cdhnd, 
				u32 indid, 
				wimax_osal_packet *msg,
				void *data);
static void dm_ind_scan_result(struct pc2400m_drv_cd_if *cdhnd, 
			       u32 indid, 
			       wimax_osal_packet *msg,
			       void *data);

static u16 dm_header_status(wimax_osal_packet *msg) 
{
	struct pc2400m_drv_chipif_ctrl_msg *header = 
		(struct pc2400m_drv_chipif_ctrl_msg*)
		wimax_osal_packet_ptr(msg);
	
	return WIMAX_OSAL_LE_TO_U16(header->status);
}

static void dm_queue_get_req(struct pc2400m_drv_dm_if *hnd, u16 opcode)
{

	struct pc2400m_drv_chipif_ctrl_msg* header;
	wimax_osal_packet *msg;

	wimax_osal_assert(hnd->cd);

	msg = wimax_osal_packet_alloc(
		  PC2400M_DRV_CHIPIF_CTRL_MSG_LEN);
	header = (struct pc2400m_drv_chipif_ctrl_msg*)
		wimax_osal_packet_put(msg, PC2400M_DRV_CHIPIF_CTRL_MSG_LEN);
	hnd->setup_header(hnd, (u8*)header, opcode, 0);
	DM_MSG_ENQUEUE(msg, PC2400M_CD_FLAG_NONE);
	
	return;
}

static boolean dm_supported_ver_check(struct pc2400m_drv_dm_if *hnd) 
{

	int i;
        struct L3L4SupportedVersion versions[] = L3L4_SUPPORTED_VERSIONS;

	for (i=0; 
	     i<sizeof(versions)/sizeof(struct L3L4SupportedVersion); 
	     i++) {
	        if (hnd->chipset.cap.L4Mversion[0] == versions[i].major &&
		    hnd->chipset.cap.L4Mversion[1] == versions[i].minor &&
		    hnd->chipset.cap.L4Mversion[2] == versions[i].branch) {

		        /* supported version */
		        if (versions[i].note) {
			        wimax_osal_trace_str(
				        PC2400M_GROUP_DM_ERROR, 
					PC2400M_TRACE_VERSION_NOTE,
					WIMAX_OSAL_PRIORITY_ERROR,
					versions[i].note);
			}
			return TRUE;

		}
	}

	wimax_osal_trace_data(PC2400M_GROUP_DM_ERROR, 
			      PC2400M_TRACE_UNSUPPORTED_VERSION,
			      WIMAX_OSAL_PRIORITY_ERROR,
			      (u8*)hnd->chipset.cap.L4Mversion,
			      sizeof(hnd->chipset.cap.L4Mversion));
				
	return FALSE;

}

static wimax_osal_packet *dm_create_scan_param(
	struct pc2400m_drv_dm_if *hnd,
        struct pc2400m_drv_dm_trans *trans)
{
	
	struct L3L4_TLV_STR_CHANNEL_INFO *chinfo;
	struct pc2400m_drv_dm_start_scan_params *params;
	wimax_osal_packet *msg;
	s32 cnt, size, i;
	u8 *ptr;

	wimax_osal_assert(trans->params);
	params = (struct pc2400m_drv_dm_start_scan_params*)trans->params;

	cnt = params->req.search_plan_count;
	if (cnt == 0) cnt = 1;
	/* MISSING: to be removed when multiple scan parameters are possible*/
	if (cnt > 20) {
		wimax_osal_trace_u32(
		        PC2400M_GROUP_DM_ERROR, 
			PC2400M_TRACE_TOO_MANY_SEARCH_LIMITS,
			WIMAX_OSAL_PRIORITY_ERROR, cnt);

		cnt = 20;
	}

	size = PC2400M_DRV_CHIPIF_CTRL_MSG_LEN + 
		cnt * L3L4_TLV_SIZE_CHANNEL_INFO;

	msg = wimax_osal_packet_alloc(size);

	/* setup message header */
	ptr = wimax_osal_packet_put(msg, PC2400M_DRV_CHIPIF_CTRL_MSG_LEN);
	hnd->setup_header(hnd, ptr, L4_L3_OPCODE_SET_SCAN_PARAM,
			  size - PC2400M_DRV_CHIPIF_CTRL_MSG_LEN);

	/* MISSING: to be removed once multiple scan parameters 
	   become possible */
	if (params->req.search_plan_count) {

		for (i=0; i<cnt; i++) {
			chinfo = (struct L3L4_TLV_STR_CHANNEL_INFO*)
			        wimax_osal_packet_put(
				        msg, L3L4_TLV_SIZE_CHANNEL_INFO);

			chinfo->tlv.type = WIMAX_OSAL_U16_TO_LE(
				L3L4_TLV_TYPE_CHANNEL_INFO);
			chinfo->tlv.length = WIMAX_OSAL_U16_TO_LE(
				L3L4_TLV_SIZE_CHANNEL_INFO - L3L4_TLV_HDR_LEN);

			/* WARNING: assumption made, that same numbering is 
			   used on chipset if and WiHAL */
			wimax_osal_assert(L3L4_BANDWIDTH_10 == 
				    DRV_CHANNEL_BANDWIDTH_10MHZ);
			wimax_osal_assert(L3L4_FFT_1024 == DRV_FFT_SIZE_1024);

			chinfo->channel_id = WIMAX_OSAL_U16_TO_LE((u16)i);
			chinfo->frequency = WIMAX_OSAL_U32_TO_LE(
				params->req.p_search_plan[i].carrier_freq);
			chinfo->bandwidth = (u8)params->req.p_search_plan[i].
				channel_bandwidth;
			chinfo->fft = (u8)(params->req.p_search_plan[i].
					   fft_size);
			chinfo->max_tx_power = WIMAX_OSAL_U32_TO_LE(
				params->req.p_search_plan[i].max_tx_power);

			wimax_osal_mem_cpy(chinfo->preambles, 
				    params->req.p_search_plan[i].preamble, 
				    DRV_PREAMBLE_SIZE);

			chinfo->reserved = 0;

		}

	} else {

		/* fill one TLV with all parameters "empty" */
		chinfo = (struct L3L4_TLV_STR_CHANNEL_INFO*)
			wimax_osal_packet_put(msg, 
					L3L4_TLV_SIZE_CHANNEL_INFO);
		
		chinfo->tlv.type = WIMAX_OSAL_U16_TO_LE(
			L3L4_TLV_TYPE_CHANNEL_INFO);
		chinfo->tlv.length = WIMAX_OSAL_U16_TO_LE(
			L3L4_TLV_SIZE_CHANNEL_INFO - L3L4_TLV_HDR_LEN);
		
		chinfo->channel_id = WIMAX_OSAL_U16_TO_LE(0);
		chinfo->frequency = WIMAX_OSAL_U32_TO_LE(0);
		chinfo->bandwidth = (u8)L3L4_BANDWIDTH_NA;
		chinfo->fft = (u8)L3L4_FFT_NA;
		chinfo->max_tx_power = WIMAX_OSAL_U32_TO_LE(0);
	
		wimax_osal_mem_set(chinfo->preambles, 0x00, DRV_PREAMBLE_SIZE);
		chinfo->reserved = 0;
		
	}

	return msg;

}

static void dm_send_mode_of_operation_req(struct pc2400m_drv_dm_if *hnd,
					  u32 mode)
{

        struct L3L4_TLV_STR_MODE_OF_OPERATION *moo;
	wimax_osal_packet *msg;
	u8 *ptr;

	msg = wimax_osal_packet_alloc
	        (PC2400M_DRV_CHIPIF_CTRL_MSG_LEN +
		 L3L4_TLV_SIZE_MODE_OF_OPERATION);
		
	ptr = wimax_osal_packet_put(
		msg, PC2400M_DRV_CHIPIF_CTRL_MSG_LEN);
	hnd->setup_header(hnd, ptr, 
			  L4_L3_OPCODE_CMD_MODE_OF_OPERATION,
			  L3L4_TLV_SIZE_MODE_OF_OPERATION);
		
	moo = (struct L3L4_TLV_STR_MODE_OF_OPERATION*)
	        wimax_osal_packet_put(
		        msg, L3L4_TLV_SIZE_MODE_OF_OPERATION);
		
	moo->tlv.type = WIMAX_OSAL_U16_TO_LE(L3L4_TLV_TYPE_MODE_OF_OPERATION);
	moo->tlv.length =
	        WIMAX_OSAL_U16_TO_LE(L3L4_TLV_SIZE_MODE_OF_OPERATION-
				     L3L4_TLV_HDR_LEN);
	moo->mode = WIMAX_OSAL_U32_TO_LE(mode);

	DM_MSG_ENQUEUE(msg, PC2400M_CD_FLAG_NONE);

}

static void dm_send_rf_operation_req(struct pc2400m_drv_dm_if *hnd,
					u32 state)
{

        struct L3L4_TLV_STR_RF_OPERATION *rfop;
	wimax_osal_packet *msg;
	u8 *ptr;

	msg = wimax_osal_packet_alloc(
		PC2400M_DRV_CHIPIF_CTRL_MSG_LEN +
		L3L4_TLV_SIZE_RF_OPERATION);

	ptr = wimax_osal_packet_put(
		msg, PC2400M_DRV_CHIPIF_CTRL_MSG_LEN);
	hnd->setup_header(hnd, ptr, L4_L3_OPCODE_CMD_RF_CONTROL,
	        L3L4_TLV_SIZE_RF_OPERATION);

	/* fill the rf-operation message */
	rfop = (struct L3L4_TLV_STR_RF_OPERATION*)
	        wimax_osal_packet_put(msg, L3L4_TLV_SIZE_RF_OPERATION);
	rfop->tlv.type = WIMAX_OSAL_U16_TO_LE(L3L4_TLV_TYPE_RF_OPERATION);
	rfop->tlv.length = WIMAX_OSAL_U16_TO_LE(L3L4_TLV_SIZE_RF_OPERATION-
						L3L4_TLV_HDR_LEN);
	rfop->rf_operation = WIMAX_OSAL_U32_TO_LE(state);
		
	DM_MSG_ENQUEUE(msg, PC2400M_CD_FLAG_NONE);

}

static void dm_suspend_resp(struct pc2400m_drv_cd_if *cdhnd, void *data)
{

	struct pc2400m_drv_dm_if *hnd = (struct pc2400m_drv_dm_if*)data;

	/* once the control dispatcher has also suspended (completed any
	   ongoing operation) we are ready to go down */
	dm_trans_complete(hnd, ESABABA);

}

static void dm_trans_execute(struct pc2400m_drv_dm_if *hnd)
{

	struct pc2400m_private *priv = wimax_osal_ctx_priv_get(hnd->ctx);
	wimax_osal_packet *msg;
	s32 ret = ESABABA;
	u8 *ptr;
	u16 size;

	/* if a transition is ongoing or there are no items in the queue,
	   bail out */
	if (hnd->trans || !hnd->transq) goto out;

	hnd->trans = (struct pc2400m_drv_dm_trans*)sllist_pop(&(hnd->transq));

	/* if suspended, do not execute unless this is the resume operation */
	if (hnd->flags & PC2400M_DRV_DM_FLAGS_SUSPENDED &&
	    hnd->trans->id != PC2400M_DRV_DM_RESUME) {
		sllist_insert(&(hnd->transq), &(hnd->trans->list));
		hnd->trans = NULL;
		goto out;
	}

	wimax_osal_trace_u32(PC2400M_GROUP_DM_DEBUG, 
			     PC2400M_TRACE_TRANSITION_START,
			     WIMAX_OSAL_PRIORITY_DEBUG, hnd->trans->id);

	/* start the timeout timer */
#define DM_TRANSITION_TIMEOUT 4000000
	hnd->trans->timerhnd = 
		wimax_osal_timer_create(DM_TRANSITION_TIMEOUT,
					WIMAX_OSAL_TR_10MS,
					0,
					(void*)hnd,
					pc2400m_drv_dm_timeout_handler);

	/* start handling the transition */
	switch (hnd->trans->id) {
	case PC2400M_DRV_DM_GET_DEVICE_INFO:
		/* this transition is started by DM itself during init
		   only */
		wimax_osal_assert(hnd->state == L3L4_SYSTEM_STATE_INIT);

		hnd->trans->intdata.state = DM_STATE_GET_DEVICE_INFO_NONE;

		/* send the GET L4M Version request */
		dm_queue_get_req(hnd, L4_L3_OPCODE_GET_LM_VERSION);

		break;

	case PC2400M_DRV_DM_ENABLE_RADIO:
	{


		/* check prereq state */
		if (hnd->state != L3L4_SYSTEM_STATE_INIT &&
		    hnd->state != L3L4_SYSTEM_STATE_RF_OFF)
			goto err1;

		/* if in INIT state, quietly perform the mode of operation
		   request first to change state */
		if (hnd->state == L3L4_SYSTEM_STATE_INIT) {

		        /* enqueue request to select "normal" operation mode. 
			   RF enable will be requested once that completes. */
			hnd->trans->intdata.state = DM_STATE_RADIO_ON_MODE_REQ;
			dm_send_mode_of_operation_req
			        (hnd, L3L4_MODE_OF_OPERATION_NORMAL);
					
		} else {
		        /* enqueue request to switch on RF */
		        hnd->trans->intdata.state = DM_STATE_RADIO_ON_REQ;
		        dm_send_rf_operation_req(hnd, L3L4_RF_STATUS_ON);
		}

		break;
	}

	case PC2400M_DRV_DM_DISABLE_RADIO:
	{

		/* check prereq state */
		if (hnd->state == L3L4_SYSTEM_STATE_UNINITIALIZED || 
		    hnd->state == L3L4_SYSTEM_STATE_INIT ||
		    hnd->state == L3L4_SYSTEM_STATE_CONFIG ||
		    hnd->state == L3L4_SYSTEM_STATE_PRODUCTION ||
		    hnd->state == L3L4_SYSTEM_STATE_RF_SHUTDOWN)
  		        /* it is allowed to start this from RF OFF state to 
			   cancel a requested enable radio operation */
			goto err1;

		/* send radio off request to the chipset */
		hnd->trans->intdata.state = DM_STATE_RADIO_OFF_REQ;
		dm_send_rf_operation_req(hnd, L3L4_RF_STATUS_OFF);
		break;
	}

	case PC2400M_DRV_DM_START_SCAN:
	{
		struct L3L4_TLV_STR_SCAN_COMMAND *scan_cmd;

		/* check prereq state */
		if (hnd->state != L3L4_SYSTEM_STATE_READY)
			goto err1;

		hnd->trans->intdata.state = DM_STATE_START_SCAN_PARAMS;

		/* send the scan params request to the chipset */
		msg = dm_create_scan_param(hnd, hnd->trans);
		DM_MSG_ENQUEUE(msg, PC2400M_CD_FLAG_NONE);

		/* send the scan command request to the chipset */
		msg = wimax_osal_packet_alloc
			  (PC2400M_DRV_CHIPIF_CTRL_MSG_LEN +
			   L3L4_TLV_SIZE_SCAN_COMMAND);
		
		ptr = wimax_osal_packet_put(
			msg, PC2400M_DRV_CHIPIF_CTRL_MSG_LEN);
		hnd->setup_header(hnd, ptr, L4_L3_OPCODE_CMD_SCAN,
				  L3L4_TLV_SIZE_SCAN_COMMAND);
		
		scan_cmd = (struct L3L4_TLV_STR_SCAN_COMMAND*)
			wimax_osal_packet_put(
				msg, L3L4_TLV_SIZE_SCAN_COMMAND);
		
		scan_cmd->tlv.type = 
			WIMAX_OSAL_U16_TO_LE(
				L3L4_TLV_TYPE_SCAN_COMMAND);
		scan_cmd->tlv.length =
			WIMAX_OSAL_U16_TO_LE(
				L3L4_TLV_SIZE_SCAN_COMMAND-
				L3L4_TLV_HDR_LEN);
		scan_cmd->do_scan_command = 
			WIMAX_OSAL_U32_TO_LE(L3L4_SCAN_COMMAND_START);
		
		DM_MSG_ENQUEUE(msg, PC2400M_CD_FLAG_NONE);
		
		break;
	}

	case PC2400M_DRV_DM_STOP_SCAN:
	{
		struct L3L4_TLV_STR_SCAN_COMMAND *scan_cmd;

		/* check prereq state */
		if (hnd->state != L3L4_SYSTEM_STATE_SCAN &&
		    hnd->state != L3L4_SYSTEM_STATE_STANDBY &&
		    hnd->state != L3L4_SYSTEM_STATE_OUT_OF_ZONE)
			goto err1;
		
		/* send the scan command request to the chipset */
		msg = wimax_osal_packet_alloc(
                        PC2400M_DRV_CHIPIF_CTRL_MSG_LEN +
			L3L4_TLV_SIZE_SCAN_COMMAND);
		
		ptr = wimax_osal_packet_put(
  		         msg, PC2400M_DRV_CHIPIF_CTRL_MSG_LEN);
		hnd->setup_header(hnd, ptr, L4_L3_OPCODE_CMD_SCAN,
				  L3L4_TLV_SIZE_SCAN_COMMAND);
		
		scan_cmd = (struct L3L4_TLV_STR_SCAN_COMMAND*)
			wimax_osal_packet_put(msg,
					      L3L4_TLV_SIZE_SCAN_COMMAND);
		
		scan_cmd->tlv.type = 
			WIMAX_OSAL_U16_TO_LE(L3L4_TLV_TYPE_SCAN_COMMAND);
		scan_cmd->tlv.length =
			WIMAX_OSAL_U16_TO_LE(L3L4_TLV_SIZE_SCAN_COMMAND-
				       L3L4_TLV_HDR_LEN);
		scan_cmd->do_scan_command = 
			WIMAX_OSAL_U32_TO_LE(L3L4_SCAN_COMMAND_STOP);


		DM_MSG_ENQUEUE(msg, PC2400M_CD_FLAG_NONE);
		
		break;
	}

	case PC2400M_DRV_DM_CONNECT:
	{
		struct L3L4_TLV_STR_BS_CRITERIA *bscr;
		struct L3L4_TLV_STR_TH_CRITERIA *thcr;
		struct pc2400m_drv_dm_connect_params *connect_params;

		/* check prereq state */
		if (hnd->state != L3L4_SYSTEM_STATE_READY &&
		    hnd->state != L3L4_SYSTEM_STATE_SCAN &&
		    hnd->state != L3L4_SYSTEM_STATE_STANDBY)
			goto err1;

		if (hnd->state == L3L4_SYSTEM_STATE_SCAN ||
		    hnd->state == L3L4_SYSTEM_STATE_STANDBY) {
			/* NOTE: the chipset will automatically stop
			   scanning upon network entry */
			priv->send_search_resp(hnd->ctx, E_SUCCESS);
		}
			

		wimax_osal_assert(hnd->trans->params);

		connect_params = (struct pc2400m_drv_dm_connect_params*)
			hnd->trans->params;

		/* send connect command to the chipset */
		size = L3L4_TLV_SIZE_BS_CRITERIA + L3L4_TLV_SIZE_TH_CRITERIA;
		msg = wimax_osal_packet_alloc
			  (PC2400M_DRV_CHIPIF_CTRL_MSG_LEN + size);


		ptr = wimax_osal_packet_put(
                         msg, PC2400M_DRV_CHIPIF_CTRL_MSG_LEN);
		hnd->setup_header(hnd, ptr, L4_L3_OPCODE_CMD_CONNECT, size);

		/* fill in the th criteria tlv */
		thcr = (struct L3L4_TLV_STR_TH_CRITERIA*)
			wimax_osal_packet_put(msg, L3L4_TLV_SIZE_TH_CRITERIA);
		thcr->tlv.type = 
		        WIMAX_OSAL_U16_TO_LE(L3L4_TLV_TYPE_TH_CRITERIA);
		thcr->tlv.length = 
		        WIMAX_OSAL_U16_TO_LE(L3L4_TLV_SIZE_TH_CRITERIA-
					     L3L4_TLV_HDR_LEN);
		thcr->rssi = WIMAX_OSAL_U32_TO_LE(0);
		thcr->cinr = WIMAX_OSAL_U32_TO_LE(0); /* MISSING: values? */

		/* fill in the bs criteria tlv */
		bscr = (struct L3L4_TLV_STR_BS_CRITERIA*)
			wimax_osal_packet_put(msg, L3L4_TLV_SIZE_BS_CRITERIA);


		bscr->tlv.type = 
			WIMAX_OSAL_U16_TO_LE(L3L4_TLV_TYPE_BS_CRITERIA);
		bscr->tlv.length =
			WIMAX_OSAL_U16_TO_LE(L3L4_TLV_SIZE_BS_CRITERIA-
				       L3L4_TLV_HDR_LEN);

		if (connect_params->req.handover_scheme == 
		    DRV_HANDOVER_SCHEME_BS_LOCK) {
			/* BSID criterion */
			wimax_osal_mem_cpy(bscr->bsid_criterion, 
					   connect_params->req.bsid, 
					   DRV_BSID_LENGTH);
			
			/* BSID criterion mask */
			wimax_osal_mem_set(bscr->bsid_criterion_mask, 0xff, 
					   DRV_BSID_LENGTH);
			
			/* home operator */
			bscr->home_operator = WIMAX_OSAL_U32_TO_LE(0x00);
		} else {
			/* BSID criterion */
			bscr->bsid_criterion[0] = 
				(u8)((connect_params->req.nap_id>>16)&0xff);
			bscr->bsid_criterion[1] =
				(u8)((connect_params->req.nap_id>>8)&0xff);
			bscr->bsid_criterion[2] = 
				(u8)(connect_params->req.nap_id & 0xff);
			bscr->bsid_criterion[3] = 0x00;
			bscr->bsid_criterion[4] = 0x00;
			bscr->bsid_criterion[5] = 0x00;
			
			/* BSID criterion mask */
			bscr->bsid_criterion_mask[0] = 0xff;
			bscr->bsid_criterion_mask[1] = 0xff;
			bscr->bsid_criterion_mask[2] = 0xff;
			bscr->bsid_criterion_mask[3] = 0x00;
			bscr->bsid_criterion_mask[4] = 0x00;
			bscr->bsid_criterion_mask[5] = 0x00;

			/* home operator */
			bscr->home_operator = WIMAX_OSAL_U32_TO_LE(0x00);
		}

		wimax_osal_trace_data(PC2400M_GROUP_DM_DEBUG, 
				      PC2400M_TRACE_DM_BS_CRITERIA,
				      WIMAX_OSAL_PRIORITY_DEBUG, 
				      bscr->bsid_criterion, DRV_BSID_LENGTH);

		wimax_osal_trace_data(PC2400M_GROUP_DM_DEBUG, 
				      PC2400M_TRACE_DM_BS_CRITERIA_MASK,
				      WIMAX_OSAL_PRIORITY_DEBUG, 
				      bscr->bsid_criterion_mask, 
				      DRV_BSID_LENGTH);
		

		/* MISSING: need to get the end-up link status somewhere else
		 */
		wimax_osal_mem_cpy(hnd->chipset.link.bs_id, ptr+4, 
				   DRV_BSID_LENGTH);
		hnd->chipset.link.rssi = 0xff;

	        DM_MSG_ENQUEUE(msg, PC2400M_CD_FLAG_NONE);

		break;
	}

	case PC2400M_DRV_DM_DISCONNECT:
		/* check prereq state */
		if (hnd->state != L3L4_SYSTEM_STATE_CONNECTING &&
		    hnd->state != L3L4_SYSTEM_STATE_WIMAX_CONNECTED &&
		    hnd->state != L3L4_SYSTEM_STATE_DATA_PATH_CONNECTED &&
		    hnd->state != L3L4_SYSTEM_STATE_IDLE)
			goto err1;

		if (hnd->state != L3L4_SYSTEM_STATE_DATA_PATH_CONNECTED)
			hnd->chipset.trans.disconnect_status = 
				E_SUCCESS;
		else
			hnd->chipset.trans.disconnect_status = 
				E_SUCCESS;

		/* send the disconnect command to the chipset */
		dm_queue_get_req(hnd, L4_L3_OPCODE_CMD_DISCONNECT);
		break;

	case PC2400M_DRV_DM_INIT:
		/* init is performed by DM upon initialization ONLY. 
		   interrupts are enabled within the same scope as enabling
		   the INIT transaction, so the device cannot have shifted
		   to the CONFIG state before entering this function. */
		wimax_osal_assert(hnd->state == 
				  L3L4_SYSTEM_STATE_UNINITIALIZED);

		hnd->trans->intdata.state = DM_STATE_INIT_WAITING_CONFIG;
		break;

	case PC2400M_DRV_DM_TERMINATE:
		/* terminate can occur in any state */
	        if (hnd->state != L3L4_SYSTEM_STATE_UNINITIALIZED) {
		        hnd->trans->intdata.state = DM_STATE_TERMINATE_REQ;
			dm_queue_get_req(hnd, L4_L3_OPCODE_CMD_TERMINATE);
		} else {
		        dm_trans_complete(hnd, ESABABA);
		}
		break;

	case PC2400M_DRV_DM_PRODUCTION:
	{
	        if (hnd->state != L3L4_SYSTEM_STATE_INIT)
		        goto err1;

		dm_send_mode_of_operation_req
		        (hnd, L3L4_MODE_OF_OPERATION_PRODUCTION);


		break;
	}

	case PC2400M_DRV_DM_SUSPEND:
	{
		/* stop execution of the transition queue,
		   and then immediately complete */
		hnd->flags |= PC2400M_DRV_DM_FLAGS_SUSPENDED;

		/* suspend the control dispatcher to suspend */
		hnd->cd->suspend(hnd->cd, dm_suspend_resp, (void*)hnd); 
		break;
	}

	case PC2400M_DRV_DM_RESUME:
	{
		/* resume execution of the transition queue,
		   and then immediately complete */
		hnd->flags &= (~PC2400M_DRV_DM_FLAGS_SUSPENDED);
		hnd->cd->resume(hnd->cd);
		dm_trans_complete(hnd, ESABABA);
		break;
	}
	};


	goto out;
 err1:
	ret = -EINVAL;
	wimax_osal_trace_u32(PC2400M_GROUP_DM_ERROR, 
			     PC2400M_TRACE_INVALID_DM_STATE,
			     WIMAX_OSAL_PRIORITY_DEBUG, hnd->state);
	dm_trans_complete(hnd, ret);

 out:
	return;

}

static void dm_trans_complete(struct pc2400m_drv_dm_if *hnd, s32 status)
{

	struct pc2400m_drv_dm_trans *trans = hnd->trans;

	/* avoid recursive completions in error handling */
	if (trans->flags & DM_TRANS_FLAG_COMPLETING) goto out;
	trans->flags |= DM_TRANS_FLAG_COMPLETING;

	/* clear current transition data */
	wimax_osal_timer_cancel(hnd->trans->timerhnd);
	trans->timerhnd = 0;

	/* call the transition callback */
	if (trans->cb)
		hnd->trans->cb(hnd, status, trans->params, 
			       trans->cb_data, trans->cb_cl_data);
		
	/* cleanup transition */
	hnd->trans = NULL;
	wimax_osal_mem_free(&trans);

	/* start next queued transition */
	if (status != -ECANCELED)
		dm_trans_execute(hnd);

 out:
	return;
}


/**
 * pc2400m_drv_dm_abort_mission - go back to boot state
 * @hnd: pointer to the instance of the device manager
 *
 * Abort all ongoing actions immediately and go back to boot state
 * assuming the chip is dead or rebooted
 */
static s32 pc2400m_drv_dm_abort_mission(struct pc2400m_drv_dm_if *hnd) 
{

	struct pc2400m_private *priv = wimax_osal_ctx_priv_get(hnd->ctx);

	hnd->reset(hnd);
	priv->send_system_state_ind(
		hnd->ctx, E_SYSTEM_STATE_FATAL_ERROR);

	return 0;

}

static void dm_trans_abort(struct pc2400m_drv_dm_if *hnd)
{

	/* call the transition callback */
	if (!hnd->trans)
		goto err1;

	/* clear transition specific memory */
	switch (hnd->trans->id) {
	default:
		/* No transition specific memory in the remaining
		   transitions. */
		break;
	};
	
	/* handle the transition */
	dm_trans_complete(hnd, -ECANCELED);

 err1:
	return;
}

static void forcecancel_transition(void *ptr) {
	
	struct pc2400m_drv_dm_trans *trans = (struct pc2400m_drv_dm_trans*)ptr;

	/* get rid of the transaction without scheduling the next */
	trans->hnd->trans = trans;
	dm_trans_complete(trans->hnd, -ECANCELED);

}


/**
 * pc2400m_drv_dm_reset - clean up the device manager
 * @hnd: pointer to the instance of the device manager
 *
 * Free the memory and resources used by the cd cancelling all ongoing
 * operations and bringing it to an initial state
 */
static void pc2400m_drv_dm_reset(struct pc2400m_drv_dm_if *hnd)
{

	/* forcibly abort ongoing transition */
	dm_trans_abort(hnd);

	/* clean up the remaining queue */
	sllist_free(&(hnd->transq), forcecancel_transition);
		
	/* bring subsystems to an initial state (the subsystems always
	   exist when dm exists */
	hnd->cd->reset(hnd->cd);
	hnd->hdi->reset(hnd->hdi);

	/* clear flags (suspension etc) */
	hnd->flags = 0;

	/* clear device info cache */
	wimax_osal_mem_set(&(hnd->chipset), 0x00, 
	       sizeof(struct pc2400m_drv_dm_chipset));

	/* reset system state */
	RESET_STATE(hnd->state);
}

/**
 * pc2400m_drv_dm_transition - execute a transition
 * @hnd: handle to the dm instance
 * @id: identifier of the transition
 * @data: message structure containing transition parameters
 *
 * Initializes internal variables and prepares the dm for usage
 */
static void pc2400m_drv_dm_transition(struct pc2400m_drv_dm_if *hnd,
				      enum pc2400m_drv_dm_transitions id,
				      void *data,
				      dm_transition_cb trans_cb,
				      void *trans_data,
				      void *client_data)
{
       
	struct pc2400m_drv_dm_trans *trans_info;

	/* enqueue transition */
	trans_info = (struct pc2400m_drv_dm_trans*)wimax_osal_mem_alloc
		  (sizeof(struct pc2400m_drv_dm_trans));

	wimax_osal_mem_set(trans_info, 0x00, 
			   sizeof(struct pc2400m_drv_dm_trans));
	trans_info->hnd = hnd;
	trans_info->id = id;
	trans_info->params = data;
	trans_info->cb_data = trans_data;
	trans_info->cb_cl_data = client_data;
	trans_info->cb = trans_cb;

	/* the RESUME operation must go first, if there are multiple items
	   queued otherwise we will never wake up */
	if (id != PC2400M_DRV_DM_RESUME) {
		sllist_append(&(hnd->transq), &(trans_info->list));
	} else {
		sllist_insert(&(hnd->transq), &(trans_info->list));
	}

	/* execute if possible */
	dm_trans_execute(hnd);
	
	return;
}

static s32 pc2400m_drv_dm_get_link_state(struct pc2400m_drv_dm_if *hnd)
{

	return hnd->chipset.link.up;

}

/**
 * pc2400m_drv_get_capablities - return cached system capability values
 * @hnd: handle to the dm instance
 * @firmware_ver: pointer to store firmware version to
 * @hardware_ver: pointer to store hardware version to
 * @mac_addr: pointer to store the device MAC address to
 *
 * Retrieves capability information from the DM cache
 */
static s32 pc2400m_drv_dm_get_capabilities(struct pc2400m_drv_dm_if *hnd,
					 u8 *firmware_ver,
					 u8 *hardware_ver, 
					 u8 *mac_addr)
{

	wimax_osal_mem_cpy(firmware_ver, hnd->chipset.cap.fw_version, 
			   DRV_VERSION_STR_LEN);
	firmware_ver[DRV_VERSION_STR_LEN-1] = '\0';

	wimax_osal_mem_cpy(hardware_ver, hnd->chipset.cap.hw_version, 
			   DRV_VERSION_STR_LEN);
	hardware_ver[DRV_VERSION_STR_LEN-1] = '\0';

	wimax_osal_mem_cpy(mac_addr, hnd->chipset.cap.mac_addr, 
			   DM_MAC_ADDR_LEN);

	
	if (hnd->chipset.cap.set)
		return 0;
	else
		/* the data has been refreshed from the device */
		return -EFAULT;

}


static void pc2400m_drv_dm_setup_header(struct pc2400m_drv_dm_if *hnd,
					u8* ptr, 
					u16 type,
					u16 length)
{
	
	struct pc2400m_drv_chipif_ctrl_msg *hdr = 
		(struct pc2400m_drv_chipif_ctrl_msg*)ptr;

	hdr->type = WIMAX_OSAL_U16_TO_LE(type);
	hdr->length = WIMAX_OSAL_U16_TO_LE(length);
	hdr->version = WIMAX_OSAL_U16_TO_LE(L3L4_VERSION);
	hdr->accessibility = WIMAX_OSAL_U16_TO_LE(L3L4_ACCESSIBILITY);
	hdr->status = WIMAX_OSAL_U16_TO_LE(L3L4_RESPONSE_STATUS_SUCCESS_DONE);
	hdr->reserved = 0;
	return;
}


/**
 * pc2400m_drv_dm_get_tlv - find start of next tlv of specified type
 * @hnd: pointer to the instance of the device manager
 * @ptr: pointer to the start of the message (including the header)
 * @reqtype: type of the tlv to be found
 * @last: pointer to previous tlv of the type to find the next
 *
 * Search the message data for first instance of the requested tlv type. To
 * find multiple tlv's of the same type, multiple calls are used with the
 * pointer of the previous tlv as 'last'.
 */
static u8* pc2400m_drv_dm_get_tlv(struct pc2400m_drv_dm_if *hnd, 
				  u8 *ptr, 
				  u16 reqtype, 
				  u8 *last)
{
	s32 length;
	u16 tlvlen;

	length = WIMAX_OSAL_LE_TO_U16
		(((struct pc2400m_drv_chipif_ctrl_msg*)ptr)->length);
	ptr += PC2400M_DRV_CHIPIF_CTRL_MSG_LEN;
	
	if (last) {
		length -= (last-ptr);
		
		tlvlen = WIMAX_OSAL_GET_LE16(last+2);
		ptr = last + (tlvlen + L3L4_TLV_HDR_LEN);
		length -= (tlvlen + L3L4_TLV_HDR_LEN);
	}

	while (length) {
		wimax_osal_assert(length > 0);
		if (reqtype == WIMAX_OSAL_GET_LE16(ptr)) return ptr;

		tlvlen = WIMAX_OSAL_GET_LE16(ptr+2);
		wimax_osal_assert(tlvlen);

		wimax_osal_assert(tlvlen > 0);
		ptr += (tlvlen + L3L4_TLV_HDR_LEN);
		length -= (tlvlen + L3L4_TLV_HDR_LEN);
		
		if (length <= L3L4_TLV_HDR_LEN) length = 0;
	}

	return NULL;

}




static void pc2400m_drv_dm_interrupt(wimax_osal_context *ctx)
{
	struct pc2400m_private *priv  = wimax_osal_ctx_priv_get(ctx);
	wimax_osal_assert(priv->dm->cd);
	priv->dm->cd->interrupt(priv->dm->cd);
}

static void dm_initialize_info_cb(struct pc2400m_drv_dm_if *hnd, 
				  s32 status, 
				  void *params, 
				  void *cbdata, 
				  void *cldata)
{

        dm_initialize_cb cb = (dm_initialize_cb)cbdata;

	if (status == ESABABA) {

	        /* indicate success */
	        cb(hnd, ESABABA, (void*)cldata);

	} else {

	        /* clean up */
	        hnd->cd->reset(hnd->cd);
		hnd->hdi->reset(hnd->hdi);

	        /* indicate failure */
	        cb(hnd, -EFAULT, (void*)cldata);

	}


}

static void dm_initialize_init_cb(struct pc2400m_drv_dm_if *hnd, 
				  s32 status, 
				  void *params, 
				  void *cbdata, 
				  void *cldata)
{

        dm_initialize_cb cb = (dm_initialize_cb)cbdata;

	if (status == ESABABA) {

	        /* request system state */
	        hnd->transition(hnd, PC2400M_DRV_DM_GET_DEVICE_INFO, 
				NULL, dm_initialize_info_cb, (void*)cb, 
				cldata);

	} else {

	        /* clean up */
	        hnd->cd->reset(hnd->cd);
		hnd->hdi->reset(hnd->hdi);

	        /* indicate failure */
	        cb(hnd, -EFAULT, (void*)cldata);

	}

}

/**
 * pc2400m_drv_dm_initialize - initialize the dm instance
 * @hnd: handle to the dm instance
 *
 * Initializes internal variables and prepares the dm for usage
 */
static void pc2400m_drv_dm_initialize(struct pc2400m_drv_dm_if *hnd,
				      u8 *firmware, u32 len,
				      dm_initialize_cb cb,
				      void *cbdata) 
{

	s32 ret = 0;

	wimax_osal_assert(hnd->cd != NULL);
	wimax_osal_assert(hnd->hdi != NULL);
	wimax_osal_assert(cb);

	/* initialize the subsystems required for the dm */
	hnd->cd->initialize(hnd->cd, hnd->hdi);

	/* register global DM indications */
	hnd->cd->register_indication(hnd->cd, L3_L4_OPCODE_REPORT_STATE,
				     dm_ind_report_state, (void*)hnd);

	hnd->cd->register_indication(hnd->cd, L3_L4_OPCODE_REPORT_SCAN_RESULT,
				     dm_ind_scan_result, (void*)hnd);

	/* MISSING: initialize the IPCS and config adapter */

	if (hnd->hdi->initialize(hnd->hdi, firmware, len,
				 pc2400m_drv_dm_interrupt)) goto err3;

	/* clear flags (suspension etc) */
	hnd->flags = 0;

	/* upgrade state */
	RESET_STATE(hnd->state);

	/* initialize the chipset */
	hnd->transition(hnd, PC2400M_DRV_DM_INIT,
			NULL, dm_initialize_init_cb, (void*)cb, cbdata);
	
	goto out;

 err3:
	hnd->cd->reset(hnd->cd);
	hnd->hdi->reset(hnd->hdi);

	ret = -EFAULT;
	cb(hnd, ret, cbdata);
 out:
	return;

}

/**
 * pc2400m_drv_dm_cleanup - remove an instance of the device manager
 * @hnd: pointer to the instance of the device manager
 *
 * Free the memory and resources used up by the specified instance
 * of the device manager
 */
static void pc2400m_drv_dm_cleanup(struct pc2400m_drv_dm_if *hnd)
{
	
	hnd->reset(hnd);

	/* unregister indications */
	hnd->cd->register_indication(
		hnd->cd, L3_L4_OPCODE_REPORT_STATE, NULL, NULL);
	hnd->cd->register_indication(
                hnd->cd, L3_L4_OPCODE_REPORT_SCAN_RESULT, 
		NULL, NULL);

	/* get rid of the control dispatcher */
	hnd->cd->cleanup(hnd->cd);
	hnd->hdi->cleanup(hnd->hdi);

	wimax_osal_mem_free(&hnd);

}

/**
 * pc2400m_drv_dm_if_new - create new instance of the device manager
 * @ctx: the device context to bind this dm instance to
 *
 * Allocate and initialize a new instance of the device manager module,
 * bound to a specific device instance.
 */
struct pc2400m_drv_dm_if *pc2400m_drv_dm_if_new(wimax_osal_context* ctx) {

	struct pc2400m_drv_dm_if *iface;

	iface = (struct pc2400m_drv_dm_if*)
		  wimax_osal_mem_alloc(sizeof(struct pc2400m_drv_dm_if));
	wimax_osal_mem_set(
		iface, 0x00, sizeof(struct pc2400m_drv_dm_if));
	iface->ctx = ctx;
	iface->cleanup = pc2400m_drv_dm_cleanup;
	iface->reset = pc2400m_drv_dm_reset;
	iface->initialize = pc2400m_drv_dm_initialize;
	iface->abort_mission = pc2400m_drv_dm_abort_mission;
	iface->transition = pc2400m_drv_dm_transition;
	iface->get_capabilities = pc2400m_drv_dm_get_capabilities;
	iface->get_link_state = pc2400m_drv_dm_get_link_state;
	iface->get_tlv = pc2400m_drv_dm_get_tlv;
	iface->setup_header = pc2400m_drv_dm_setup_header;
	
	/* instance variables */
	wimax_osal_mem_set(&(iface->chipset), 0x00, 
			   sizeof(struct pc2400m_drv_dm_chipset));

	iface->cd = pc2400m_drv_cd_if_new(ctx);
	iface->hdi = pc2400m_drv_hdi_if_new(ctx);

	RESET_STATE(iface->state);

	return iface;

}

static void pc2400m_drv_dm_timeout_handler(s32 id, void* data)
{

	struct pc2400m_drv_dm_if *hnd = (struct pc2400m_drv_dm_if*)data;
	struct pc2400m_private *priv = wimax_osal_ctx_priv_get(hnd->ctx);

	wimax_osal_assert(hnd->trans);

	priv->ref(priv);

	wimax_osal_trace_u32(PC2400M_GROUP_DM_ERROR, 
			     PC2400M_TRACE_CHIP_TRANS_TOUT,
			     WIMAX_OSAL_PRIORITY_ERROR,
			     hnd->trans->id);

	switch (hnd->trans->id) {
	case PC2400M_DRV_DM_INIT:
		/* the chipset has failed to initialize itself (change to the
		   init state) */
		dm_trans_complete(hnd, -EFAULT);
		break;
	case PC2400M_DRV_DM_TERMINATE:
		/* this returns ESABABA, because the driver will be shutdown
		   anyway right after. */
		GOTO_STATE(hnd->state, 
			   L3L4_SYSTEM_STATE_UNINITIALIZED);
		dm_trans_complete(hnd, ESABABA);
		break;

	case PC2400M_DRV_DM_ENABLE_RADIO:
		if (hnd->trans->intdata.state == DM_STATE_RADIO_ON_COMPLETE) {
		        /* unable to switch radio on, set the sw
			   switch of the radio to off */
		        hnd->transition(hnd, PC2400M_DRV_DM_DISABLE_RADIO,
					NULL, NULL, NULL, NULL);
			wimax_osal_trace(
				PC2400M_GROUP_DM_ERROR,	 
				PC2400M_TRACE_HW_RF_SWITCH_OFF,
				WIMAX_OSAL_PRIORITY_ERROR);
		}
		dm_trans_complete(hnd, -EFAULT);
		break;

	default:
		/* normally the timeout indicates irrecoverable failure */
		hnd->abort_mission(hnd);
		break;

	};

	/* if the system was deleted during the handling of this, 
	   handle it now */
	priv->unref(priv);


}

static void pc2400m_drv_dm_resp_handler(struct pc2400m_drv_cd_if* cdhnd, 
					wimax_osal_packet *msg, 
					void *data)
{

	struct pc2400m_drv_dm_if *hnd = (struct pc2400m_drv_dm_if*)data;
	struct L3L4_TLV_STR_L4_MESSAGE_VERSION *l4mv;
	struct L3L4_TLV_STR_DEVICE_DETAILS *dd;
	u8 *ptr;
	u16 status = dm_header_status(msg);

	wimax_osal_assert(hnd->trans);

	/* process the response */
	switch (hnd->trans->id) {
	case PC2400M_DRV_DM_GET_DEVICE_INFO:

		/* first, the L4M version */
		wimax_osal_assert(hnd->trans->intdata.state ==
				  DM_STATE_GET_DEVICE_INFO_NONE ||
				  hnd->trans->intdata.state ==
				  DM_STATE_GET_DEVICE_INFO_L4M);
		switch (hnd->trans->intdata.state) {
		case DM_STATE_GET_DEVICE_INFO_NONE:
			l4mv = (struct L3L4_TLV_STR_L4_MESSAGE_VERSION*)
				hnd->get_tlv(hnd,
					     wimax_osal_packet_ptr(msg), 
					     L3L4_TLV_TYPE_L4_MESSAGE_VERSION,
					     NULL);
			if (l4mv) {
				wimax_osal_assert(
					WIMAX_OSAL_LE_TO_U16(
						l4mv->tlv.length) ==
					L3L4_TLV_SIZE_L4_MESSAGE_VERSION-
					L3L4_TLV_HDR_LEN);

				hnd->chipset.cap.L4Mversion[0] = 
					WIMAX_OSAL_LE_TO_U16(
					        l4mv->version_major);
				hnd->chipset.cap.L4Mversion[1] = 
					WIMAX_OSAL_LE_TO_U16(
                                                l4mv->version_minor);
				hnd->chipset.cap.L4Mversion[2] = 
					WIMAX_OSAL_LE_TO_U16(
						l4mv->version_branch);

			} else {
				/* no version information */
				hnd->chipset.cap.L4Mversion[0] = 0;
				hnd->chipset.cap.L4Mversion[1] = 0;
				hnd->chipset.cap.L4Mversion[2] = 0;

				wimax_osal_trace_u16(
					PC2400M_GROUP_DM_ERROR, 
					PC2400M_TRACE_CHIP_EXP_TLV_MISSING,
					WIMAX_OSAL_PRIORITY_DEBUG,
					L3L4_TLV_TYPE_L4_MESSAGE_VERSION);
			}
			
			/* enforce supported firmware version */
			if (!dm_supported_ver_check(hnd)) {
				dm_trans_complete(hnd, -EFAULT);
				break;
			}

			wimax_osal_trace_data(
                                PC2400M_GROUP_DM_DEBUG, 
				PC2400M_TRACE_CHIP_L4M_VERSION,
				WIMAX_OSAL_PRIORITY_DEBUG,
				(u8*)hnd->chipset.cap.L4Mversion,
				sizeof(hnd->chipset.cap.L4Mversion));
			
			/* send the GET Device Info request */
			dm_queue_get_req(hnd, L4_L3_OPCODE_GET_DEVICE_INFO);

			hnd->trans->intdata.state = 
				DM_STATE_GET_DEVICE_INFO_L4M;
			break;
		case DM_STATE_GET_DEVICE_INFO_L4M:
			ptr = hnd->get_tlv(hnd, wimax_osal_packet_ptr(msg),
					   L3L4_TLV_TYPE_VERSIONS_STRINGS,
					   NULL);
			if (ptr) {
				ptr += L3L4_TLV_HDR_LEN;
				wimax_osal_trace_str(
                                        PC2400M_GROUP_DM_INFO,
					PC2400M_TRACE_CHIP_ASIC_VER,
					WIMAX_OSAL_PRIORITY_DEBUG,
					ptr);
				ptr += wimax_osal_str_len(ptr) + 1;

				wimax_osal_trace_str(
				        PC2400M_GROUP_DM_INFO,
					PC2400M_TRACE_CHIP_ASIC_VER,
					WIMAX_OSAL_PRIORITY_DEBUG,
					ptr);
				wimax_osal_str_ncpy(
				        hnd->chipset.cap.hw_version,
					     ptr, DRV_VERSION_STR_LEN);
				hnd->chipset.cap.hw_version
					[DRV_VERSION_STR_LEN-1] = '\0';
				ptr += wimax_osal_str_len(ptr) + 1;

				wimax_osal_trace_str(
					PC2400M_GROUP_DM_INFO,
					PC2400M_TRACE_CHIP_BOARD_VER,
					WIMAX_OSAL_PRIORITY_DEBUG,
					ptr);
				ptr += wimax_osal_str_len(ptr) + 1;

				wimax_osal_trace_str(
					PC2400M_GROUP_DM_INFO,
					PC2400M_TRACE_CHIP_BOARD_VER,
					WIMAX_OSAL_PRIORITY_DEBUG,
					ptr);
				ptr += wimax_osal_str_len(ptr) + 1;

				wimax_osal_trace_str(
					PC2400M_GROUP_DM_INFO,
					PC2400M_TRACE_CHIP_FW_VER,
					WIMAX_OSAL_PRIORITY_DEBUG,
					ptr);
				wimax_osal_str_ncpy(
				        hnd->chipset.cap.fw_version,
					     ptr, DRV_VERSION_STR_LEN);
				hnd->chipset.cap.fw_version
					[DRV_VERSION_STR_LEN-1] = '\0';
				ptr += wimax_osal_str_len(ptr) + 1;

				wimax_osal_trace_str(
					PC2400M_GROUP_DM_INFO,
					PC2400M_TRACE_CHIP_L1_VER,
					WIMAX_OSAL_PRIORITY_DEBUG,
					ptr);
				ptr += wimax_osal_str_len(ptr) + 1;

				wimax_osal_trace_str(
				        PC2400M_GROUP_DM_INFO,
					PC2400M_TRACE_CHIP_L2_VER,
					WIMAX_OSAL_PRIORITY_DEBUG,
					ptr);
				ptr += wimax_osal_str_len(ptr) + 1;

				wimax_osal_trace_str(
					PC2400M_GROUP_DM_INFO,
					PC2400M_TRACE_CHIP_L3_VER,
					WIMAX_OSAL_PRIORITY_DEBUG,
					ptr);

			} else {
			        wimax_osal_trace_u16(
					PC2400M_GROUP_DM_ERROR, 
					PC2400M_TRACE_CHIP_EXP_TLV_MISSING,
					WIMAX_OSAL_PRIORITY_DEBUG,
					L3L4_TLV_TYPE_VERSIONS_STRINGS);
			}

			/* MAC address ? */
			dd = (struct L3L4_TLV_STR_DEVICE_DETAILS*)
				hnd->get_tlv(hnd, wimax_osal_packet_ptr(msg), 
					     L3L4_TLV_TYPE_DEVICE_DETAILS, 
					     NULL);

			if (dd) {
 
 			        /* MISSING: The extra magic numbers (8) and 
 				   skip code below are to support an intel 
 				   firmware version transition with two 
 				   different message formats. Remove the excess
 				   at fw3.0 level the latest. */
 			        if (WIMAX_OSAL_LE_TO_U16(
 					    dd->tlv.length) != 8) {
 				        wimax_osal_assert(
 					        WIMAX_OSAL_LE_TO_U16(
 							dd->tlv.length) ==
 						L3L4_TLV_SIZE_DEVICE_DETAILS-
						L3L4_TLV_HDR_LEN);
				
					wimax_osal_mem_cpy(
						hnd->chipset.cap.mac_addr, 
						dd->mac_address,
						DM_MAC_ADDR_LEN);


				} else {
				  
 				        /* MISSING: support the old MAC TLV */
 				        u8 *ptr = (u8*)dd;
 
 					wimax_osal_mem_cpy(
 						hnd->chipset.cap.mac_addr, 
 						ptr + 4,
 						DM_MAC_ADDR_LEN);
 					
 				}

				wimax_osal_trace_data(
				        PC2400M_GROUP_DM_INFO,
					PC2400M_TRACE_CHIP_MAC_ADDR,
					WIMAX_OSAL_PRIORITY_DEBUG,
					hnd->chipset.cap.mac_addr,
					DM_MAC_ADDR_LEN);

				/* ready to rock */
				hnd->chipset.cap.set = 1;
			} else {
			        wimax_osal_trace_u16(
					PC2400M_GROUP_DM_ERROR, 
					PC2400M_TRACE_CHIP_EXP_TLV_MISSING,
					WIMAX_OSAL_PRIORITY_DEBUG,
					L3L4_TLV_TYPE_DEVICE_DETAILS);

				/* clear the MAC address */
				wimax_osal_mem_set(hnd->chipset.cap.mac_addr, 
						   0x00,
						   DM_MAC_ADDR_LEN);

				/* do not set the capabilities as updated 
				   (cap.set) as we use that to indicate that
				   the info is invalid */
			}

			dm_trans_complete(hnd, ESABABA);
			break;
		}

		break;

	case PC2400M_DRV_DM_ENABLE_RADIO:
		if (status != L3L4_RESPONSE_STATUS_SUCCESS_IN_PROCESS &&
		    status != L3L4_RESPONSE_STATUS_SUCCESS_DONE) {
			wimax_osal_trace_u16(
				PC2400M_GROUP_DM_ERROR, 
				PC2400M_TRACE_REQUEST_FAILED,
				WIMAX_OSAL_PRIORITY_DEBUG, status);
			dm_trans_complete(hnd, -EFAULT);
		} else {
		        if (hnd->trans->intdata.state == 
			    DM_STATE_RADIO_ON_REQ &&
			    hnd->chipset.state == L3L4_SYSTEM_STATE_READY) {
				wimax_osal_trace(
					PC2400M_GROUP_DM_DEBUG,
					PC2400M_TRACE_RF_CONTROL_RADIO_ON,
					WIMAX_OSAL_PRIORITY_DEBUG);
				dm_trans_complete(hnd, 0);
			} else if (hnd->trans->intdata.state == 
				   DM_STATE_RADIO_ON_REQ) {
			        /* the request is complete, but the state has 
				   not yet changed */
			        hnd->trans->intdata.state = 
				        DM_STATE_RADIO_ON_COMPLETE;
			} else if (hnd->trans->intdata.state == 
				   DM_STATE_RADIO_ON_MODE_REQ &&
				   hnd->state == L3L4_SYSTEM_STATE_RF_OFF) {
			        /* the system is already in RF OFF, send radio
				   on request immediately */
			        hnd->trans->intdata.state =
				        DM_STATE_RADIO_ON_REQ;
				dm_send_rf_operation_req(
					hnd, L3L4_RF_STATUS_ON);
			} else if (hnd->trans->intdata.state ==
				   DM_STATE_RADIO_ON_MODE_REQ) {
			        /* the state has not yet changed to RF OFF */
			        hnd->trans->intdata.state =
				        DM_STATE_RADIO_ON_MODE_REQ_DONE;
			}
			  
		}
		break;

	case PC2400M_DRV_DM_DISABLE_RADIO:
		if (status != L3L4_RESPONSE_STATUS_SUCCESS_IN_PROCESS) {
			wimax_osal_trace_u16(
				PC2400M_GROUP_DM_ERROR, 
				PC2400M_TRACE_REQUEST_FAILED,
				WIMAX_OSAL_PRIORITY_DEBUG, status);
			dm_trans_complete(hnd, -EFAULT);
		} else if (hnd->chipset.state == 
			   L3L4_SYSTEM_STATE_RF_SHUTDOWN ||
			   hnd->chipset.state ==
			   L3L4_SYSTEM_STATE_RF_OFF) {
		        wimax_osal_trace(
				PC2400M_GROUP_DM_DEBUG,
				PC2400M_TRACE_RF_CONTROL_RADIO_OFF,
				WIMAX_OSAL_PRIORITY_DEBUG);
			dm_trans_complete(hnd, 0);
		} else {
			wimax_osal_trace_u16(
				PC2400M_GROUP_DM_ERROR, 
				PC2400M_TRACE_REQUEST_FAILED,
				WIMAX_OSAL_PRIORITY_DEBUG, status);
			dm_trans_complete(hnd, -EFAULT);		
		}
		break;

	case PC2400M_DRV_DM_START_SCAN:
		if (hnd->trans->intdata.state == DM_STATE_START_SCAN_PARAMS) {
			hnd->trans->intdata.state = DM_STATE_START_SCAN_REQ;
			
			/* NOTE! For error handling, it is certainly true that
			   the scan parameter phase (this phase right here)
			   might already fail. 

			   The return cause is not checked here, because the
			   actual scan request has (due to streamlining) 
			   already been sent, and will also fail if the 
			   parameters were not properly set.

			   The resulting error will be checked and reported
			   there. */

		} else {
			if (status != 
			    L3L4_RESPONSE_STATUS_SUCCESS_IN_PROCESS && 
			    status != L3L4_RESPONSE_STATUS_SUCCESS_DONE) {
				wimax_osal_trace_u16(
					PC2400M_GROUP_DM_ERROR, 
					PC2400M_TRACE_REQUEST_FAILED,
					WIMAX_OSAL_PRIORITY_DEBUG, status);
				dm_trans_complete(hnd, -EFAULT);
			} else {
			        wimax_osal_trace(
					PC2400M_GROUP_DM_DEBUG, 
					PC2400M_TRACE_START_SCAN,
					WIMAX_OSAL_PRIORITY_DEBUG);
				GOTO_STATE(hnd->state, 
					   L3L4_SYSTEM_STATE_SCAN);
				dm_trans_complete(hnd, 0);
			}
		}
		break;

	case PC2400M_DRV_DM_STOP_SCAN:
		if (status != L3L4_RESPONSE_STATUS_SUCCESS_IN_PROCESS && 
		    status != L3L4_RESPONSE_STATUS_SUCCESS_DONE) {
			wimax_osal_trace_u16(
				PC2400M_GROUP_DM_ERROR, 
				PC2400M_TRACE_REQUEST_FAILED,
				WIMAX_OSAL_PRIORITY_DEBUG, status);
			dm_trans_complete(hnd, -EFAULT);
		} else {
			wimax_osal_trace(
				PC2400M_GROUP_DM_DEBUG, 
				PC2400M_TRACE_STOP_SCAN,
				WIMAX_OSAL_PRIORITY_DEBUG);
			dm_trans_complete(hnd, 0);
		}
		

		break;

	case PC2400M_DRV_DM_CONNECT:

		if (status != L3L4_RESPONSE_STATUS_SUCCESS_IN_PROCESS &&
		    status != L3L4_RESPONSE_STATUS_SUCCESS_DONE) {
			wimax_osal_trace_u16(
				PC2400M_GROUP_DM_ERROR, 
				PC2400M_TRACE_REQUEST_FAILED,
				WIMAX_OSAL_PRIORITY_DEBUG, status);
			dm_trans_complete(hnd, -EFAULT);
		} else {
			GOTO_STATE(hnd->state, L3L4_SYSTEM_STATE_CONNECTING);
			dm_trans_complete(hnd, 0);
		}
		break;

	case PC2400M_DRV_DM_DISCONNECT:
		if (status != L3L4_RESPONSE_STATUS_SUCCESS_IN_PROCESS) {
			wimax_osal_trace_u16(
			        PC2400M_GROUP_DM_ERROR, 
				PC2400M_TRACE_REQUEST_FAILED,
				WIMAX_OSAL_PRIORITY_DEBUG, status);
			dm_trans_complete(hnd, -EFAULT);
		} else {
			GOTO_STATE(hnd->state, 
				   L3L4_SYSTEM_STATE_DISCONNECTING);
			dm_trans_complete(hnd, 0);
		}
		break;

	case PC2400M_DRV_DM_INIT:
		/* check response status code */
		if (status != L3L4_RESPONSE_STATUS_SUCCESS_IN_PROCESS &&
		    status != L3L4_RESPONSE_STATUS_SUCCESS_DONE) {
			wimax_osal_trace_u16(
				PC2400M_GROUP_DM_ERROR, 
				PC2400M_TRACE_UNABLE_TO_INITIALIZE,
				WIMAX_OSAL_PRIORITY_DEBUG, status);
			dm_trans_complete(hnd, -EFAULT);
		} else {
			if (hnd->chipset.state == L3L4_SYSTEM_STATE_INIT) {
				GOTO_STATE(hnd->state, L3L4_SYSTEM_STATE_INIT);
				dm_trans_complete(hnd, 0);
			} else {
				hnd->trans->intdata.state = 
					DM_STATE_INIT_WAITING_CH;
			}
		}		
		break;

	case PC2400M_DRV_DM_TERMINATE:
		/* terminate cannot fail */
		wimax_osal_assert(
			status == L3L4_RESPONSE_STATUS_SUCCESS_IN_PROCESS||
			status == L3L4_RESPONSE_STATUS_SUCCESS_DONE||
			status == L3L4_RESPONSE_STATUS_ILLEGAL_VALUE);
		
		if (hnd->chipset.state == L3L4_SYSTEM_STATE_UNINITIALIZED) {
			GOTO_STATE(hnd->state,
				   L3L4_SYSTEM_STATE_UNINITIALIZED);
			dm_trans_complete(hnd, 0);
		} else {
			hnd->trans->intdata.state = 
				DM_STATE_TERMINATE_WAITING_CH;
		}		
		break;

	case PC2400M_DRV_DM_PRODUCTION:
		if (status == L3L4_RESPONSE_STATUS_SUCCESS_IN_PROCESS ||
		    status == L3L4_RESPONSE_STATUS_SUCCESS_DONE ||
		    status == L3L4_RESPONSE_STATUS_ILLEGAL_VALUE) {

			if (hnd->chipset.state == 
			    L3L4_SYSTEM_STATE_PRODUCTION) {
				GOTO_STATE(hnd->state,
					   L3L4_SYSTEM_STATE_PRODUCTION);
				dm_trans_complete(hnd, 0);
			} else {
				hnd->trans->intdata.state = 
					DM_STATE_PRODUCTION_WAITING_CH;
			}
		} else {
		        wimax_osal_trace_u16(
				PC2400M_GROUP_DM_ERROR, 
				PC2400M_TRACE_CANNOT_GO_PRODUCTION,
				WIMAX_OSAL_PRIORITY_ERROR, status);
			dm_trans_complete(hnd, -EFAULT);
		}
		break;
	case PC2400M_DRV_DM_SUSPEND:
	case PC2400M_DRV_DM_RESUME:
		/* these don't send messages, and will not have responses */
		break;
	};

	wimax_osal_packet_free(&msg);
	return;

}


static void dm_ind_scan_result(struct pc2400m_drv_cd_if *cdhnd, 
			       u32 indid, 
			       wimax_osal_packet *msg,
			       void *data)
{
	struct pc2400m_drv_dm_if *hnd = (struct pc2400m_drv_dm_if*)data;
	struct pc2400m_private *priv = wimax_osal_ctx_priv_get(hnd->ctx);
	struct L3L4_TLV_STR_SCAN_STATUS *scstatus;
	struct L3L4_TLV_STR_NETWORK_SCAN_RESULT *scres;
#define DM_MAX_SCAN_RESULTS   16
	u32 nap[DM_MAX_SCAN_RESULTS];
	u8 bestrssi[DM_MAX_SCAN_RESULTS];
	u8 numbs;
	u8 cnt = 0;
	u8 status = L3L4_SCAN_STATUS_FAILURE;

	scstatus = (struct L3L4_TLV_STR_SCAN_STATUS*)
		hnd->get_tlv(hnd, wimax_osal_packet_ptr(msg), 
			     L3L4_TLV_TYPE_SCAN_STATUS, NULL);
	if (scstatus) {
		wimax_osal_assert(
			WIMAX_OSAL_LE_TO_U16(scstatus->tlv.length) == 
			L3L4_TLV_SIZE_SCAN_STATUS-L3L4_TLV_HDR_LEN);
		status = (u8)WIMAX_OSAL_LE_TO_U32(scstatus->scan_status);
	} else {
		wimax_osal_trace_u16(
			PC2400M_GROUP_DM_ERROR, 
			PC2400M_TRACE_CHIP_EXP_TLV_MISSING,
			WIMAX_OSAL_PRIORITY_DEBUG,
			L3L4_TLV_TYPE_SCAN_STATUS);
	}

	if (status == L3L4_SCAN_STATUS_DONE ||
	    status == L3L4_SCAN_STATUS_SCANING ||
	    status == L3L4_SCAN_STATUS_STOPPED) {
		scres = NULL;
		while ((scres = (struct L3L4_TLV_STR_NETWORK_SCAN_RESULT*)
			hnd->get_tlv(hnd, wimax_osal_packet_ptr(msg), 
				     L3L4_TLV_TYPE_NETWORK_SCAN_RESULT, 
				     (u8*)scres))) {
			wimax_osal_assert(
                                WIMAX_OSAL_LE_TO_U16(scres->tlv.length) == 
				L3L4_TLV_SIZE_NETWORK_SCAN_RESULT-
				L3L4_TLV_HDR_LEN);

			nap[cnt] = (((u32)scres->network_id[0])<<16) |
				    (((u32)scres->network_id[1])<<8) |
				    (((u32)scres->network_id[2]));
			bestrssi[cnt] = 
			        (u8)WIMAX_OSAL_LE_TO_U32(scres->best_rssi);
			numbs = scres->num_of_links;
			cnt++;
		}
		
		/* give results to the client */
		if (cnt) priv->send_search_ind(hnd->ctx, cnt, nap, bestrssi);

	}

	wimax_osal_assert(status == L3L4_SCAN_STATUS_DONE ||
			  status == L3L4_SCAN_STATUS_FAILURE ||
			  status == L3L4_SCAN_STATUS_STOPPED ||
			  status == L3L4_SCAN_STATUS_SCANING);

	switch (status) {
	case L3L4_SCAN_STATUS_DONE:
		/* MISSING: do we need to stop scanning to not get more
		   results? 
		   priv->send_search_resp(hnd->ctx, E_SUCCESS);*/
		/* this will cause a scan complete to be sent to the client */
		priv->dm->transition(priv->dm, PC2400M_DRV_DM_STOP_SCAN,
				     NULL, NULL, NULL, NULL);
		break;
	case L3L4_SCAN_STATUS_FAILURE:
		/* MISSING: do we need to stop scanning to not get more
		   results?
		   priv->send_search_resp(hnd->ctx, E_ERR_FAILURE); */
		/* this will cause a scan complete to be sent to the client */
		priv->dm->transition(priv->dm, PC2400M_DRV_DM_STOP_SCAN,
				     NULL, NULL, NULL, NULL);
		break;
	case L3L4_SCAN_STATUS_STOPPED:	  
	case L3L4_SCAN_STATUS_SCANING:
		break;
	}

	wimax_osal_packet_free(&msg);

	return;

}



static void dm_ind_report_state(struct pc2400m_drv_cd_if *cdhnd, 
				u32 indid, 
				wimax_osal_packet *msg,
				void *data)
{
	struct pc2400m_drv_dm_if *hnd = (struct pc2400m_drv_dm_if*)data;
	struct pc2400m_private *priv = wimax_osal_ctx_priv_get(hnd->ctx);
	struct L3L4_TLV_STR_REPORT_STATE_REASON *rsreason;
	struct L3L4_TLV_STR_SYSTEM_STATE *rsstate;
	struct L3L4_TLV_STR_RF_SWITCHES_STATUS *rfswitches;
	u8 reason = L3L4_REPORT_STATE_REASON_NORMAL;

	/* check the reason for the status message */
	rsreason = (struct L3L4_TLV_STR_REPORT_STATE_REASON*)
		hnd->get_tlv(hnd, wimax_osal_packet_ptr(msg), 
			     L3L4_TLV_TYPE_REPORT_STATE_REASON, NULL);
	if (rsreason) {
		wimax_osal_assert(
			WIMAX_OSAL_LE_TO_U16(rsreason->tlv.length) == 
			L3L4_TLV_SIZE_REPORT_STATE_REASON-
			L3L4_TLV_HDR_LEN);

		reason = (u8)WIMAX_OSAL_LE_TO_U32(rsreason->reason);
		wimax_osal_trace_byte(PC2400M_GROUP_DM_DEBUG, 
				      PC2400M_TRACE_CHIP_ST_CH_REASON,
				      WIMAX_OSAL_PRIORITY_DEBUG, 
				      reason);
	} else {
		wimax_osal_trace_u16(
			PC2400M_GROUP_DM_ERROR, 
			PC2400M_TRACE_CHIP_EXP_TLV_MISSING,
			WIMAX_OSAL_PRIORITY_DEBUG,
			L3L4_TLV_TYPE_REPORT_STATE_REASON);
	}

	/* get the new chipset state */
	rsstate = (struct L3L4_TLV_STR_SYSTEM_STATE*) 
		hnd->get_tlv(hnd, wimax_osal_packet_ptr(msg),
			     L3L4_TLV_TYPE_SYSTEM_STATE, NULL);
	if (rsstate) {
		wimax_osal_assert(WIMAX_OSAL_LE_TO_U16(rsstate->tlv.length) == 
			    L3L4_TLV_SIZE_SYSTEM_STATE - L3L4_TLV_HDR_LEN);
		hnd->chipset.state = (u8)WIMAX_OSAL_LE_TO_U32(rsstate->state);

		wimax_osal_trace_byte(PC2400M_GROUP_DM_DEBUG, 
				      PC2400M_TRACE_CHIP_ST_CH_STATE,
				      WIMAX_OSAL_PRIORITY_DEBUG, 
				      hnd->chipset.state);
	}

	/* check rf switches status */
	rfswitches = (struct L3L4_TLV_STR_RF_SWITCHES_STATUS*)
	        hnd->get_tlv(hnd, wimax_osal_packet_ptr(msg),
			     L3L4_TLV_TYPE_RF_SWITCHES_STATUS, NULL);
	if (rfswitches) {

	  /* update the chipset cache with the current status of the 
	     rf HW switch */
	  if (rfswitches->hw_rf_switch == L3L4_RF_SWITCH_ON)
	          hnd->chipset.rf.hw_switch = TRUE;
	  else
	          hnd->chipset.rf.hw_switch = FALSE;

	}

	/* check additional fields for specific states */
	switch (hnd->state) {
	case L3L4_SYSTEM_STATE_CONNECTING:
	{
		struct L3L4_TLV_STR_CONNECT_PROGRESS *rscprog;
		enum drv_network_status stat = E_NETWORK_STATUS_CONNECTED;

		rscprog = (struct L3L4_TLV_STR_CONNECT_PROGRESS*)
			hnd->get_tlv(hnd, wimax_osal_packet_ptr(msg), 
				     L3L4_TLV_TYPE_CONNECT_PROGRESS, NULL);
		if (rscprog) {
			wimax_osal_trace_u32(
			        PC2400M_GROUP_DM_DEBUG, 
				PC2400M_TRACE_CHIP_CONNECT_PROG,
				WIMAX_OSAL_PRIORITY_DEBUG,
				WIMAX_OSAL_LE_TO_U32(rscprog->progress));

			switch (WIMAX_OSAL_LE_TO_U32(rscprog->progress)) {
			case L3L4_CONNECT_PROGRESS_RANGING:
				stat = E_NETWORK_STATUS_RANGING;
				break;
			case L3L4_CONNECT_PROGRESS_SBC:
				break;
			case L3L4_CONNECT_PROGRESS_EAP_AUTHENTICATION:
				stat = E_NETWORK_STATUS_AUTHENTICATION;
				break;
			case L3L4_CONNECT_PROGRESS_3_WAY_HANDSHAKE:
				break;
			case L3L4_CONNECT_PROGRESS_REGISTRATION:
				stat = E_NETWORK_STATUS_REGISTERING;
				break;
			default:
				break;
			}

			/* send a status indication */
			if (stat != E_NETWORK_STATUS_CONNECTED)
				priv->send_network_status_ind(
					hnd->ctx,
					stat,
					E_SUCCESS,
					hnd->chipset.link.bs_id,
					hnd->chipset.link.rssi);
		}


		break;
	}
	default:
	{

		struct L3L4_TLV_STR_MEDIA_STATUS *mediastatus;
		struct L3L4_TLV_STR_LINK_STATUS *linkstatus;

		/* update the link status in the chipset cache */
		linkstatus = (struct L3L4_TLV_STR_LINK_STATUS*)
		        hnd->get_tlv(hnd, wimax_osal_packet_ptr(msg),
				     L3L4_TLV_TYPE_LINK_STATUS, NULL);
		if (linkstatus) {
			wimax_osal_assert(
				WIMAX_OSAL_LE_TO_U16(linkstatus->tlv.length) ==
				L3L4_TLV_SIZE_LINK_STATUS - L3L4_TLV_HDR_LEN);
			wimax_osal_mem_set(hnd->chipset.link.bs_id, 
					   0x00, DRV_BSID_LENGTH);
			wimax_osal_mem_cpy(hnd->chipset.link.bs_id, 
					   linkstatus->operator_id, 
					   L3L4_OPERATOR_ID_LEN);
			hnd->chipset.link.rssi = linkstatus->link_quality;
			
		}

		/* update the media status in the chipset cache */
		mediastatus = (struct L3L4_TLV_STR_MEDIA_STATUS*)
			hnd->get_tlv(hnd, wimax_osal_packet_ptr(msg),
				     L3L4_TLV_TYPE_MEDIA_STATUS, NULL);
		if (mediastatus) {
			wimax_osal_assert(
				WIMAX_OSAL_LE_TO_U16(
                                       mediastatus->tlv.length) == 
				L3L4_TLV_SIZE_MEDIA_STATUS - 
				L3L4_TLV_HDR_LEN);
			wimax_osal_trace_u32(
				PC2400M_GROUP_DM_DEBUG,
				PC2400M_TRACE_LINK_STATE,
				WIMAX_OSAL_PRIORITY_DEBUG,
				WIMAX_OSAL_LE_TO_U32(mediastatus->
						     media_status));

			switch(WIMAX_OSAL_LE_TO_U32(
				      mediastatus->media_status)) {
			case L3L4_MEDIA_STATUS_LINK_UP:
				hnd->chipset.link.up = 1;
				break;
			case L3L4_MEDIA_STATUS_LINK_DOWN:
				hnd->chipset.link.up = 0;
				break;
			default:
				wimax_osal_assert(0);
				break;
			}
		}

	}
	}

	wimax_osal_packet_free(&msg);


	/* process the new status according to transition */
	if (hnd->trans) switch (hnd->trans->id) {
	case PC2400M_DRV_DM_ENABLE_RADIO:
	        if (hnd->chipset.state == L3L4_SYSTEM_STATE_RF_OFF &&
		    hnd->trans->intdata.state == 
		    DM_STATE_RADIO_ON_MODE_REQ_DONE) {
		        hnd->trans->intdata.state = DM_STATE_RADIO_ON_REQ;
			dm_send_rf_operation_req(hnd, L3L4_RF_STATUS_ON);
		} else if (hnd->chipset.state == L3L4_SYSTEM_STATE_READY &&
			   hnd->trans->intdata.state == 
			   DM_STATE_RADIO_ON_COMPLETE) {
		        /* the state change completes the transition */
		        dm_trans_complete(hnd, ESABABA);		  
		}
		break;

	case PC2400M_DRV_DM_INIT:
		if (hnd->trans->intdata.state == 
		    DM_STATE_INIT_WAITING_CONFIG &&
		    hnd->chipset.state == L3L4_SYSTEM_STATE_CONFIG) {
			hnd->trans->intdata.state = DM_STATE_INIT_REQ;
			GOTO_STATE(hnd->state, 
				   L3L4_SYSTEM_STATE_CONFIG);
			dm_queue_get_req(hnd, L4_L3_OPCODE_CMD_INIT);
		} else if (hnd->trans->intdata.state ==
			   DM_STATE_INIT_WAITING_CH &&
			   hnd->chipset.state == L3L4_SYSTEM_STATE_INIT) {
			GOTO_STATE(hnd->state, L3L4_SYSTEM_STATE_INIT);
			dm_trans_complete(hnd, ESABABA);
		} 
		break;

	case PC2400M_DRV_DM_TERMINATE:
		if (hnd->chipset.state == L3L4_SYSTEM_STATE_UNINITIALIZED) {
			GOTO_STATE(hnd->state, 
				   L3L4_SYSTEM_STATE_UNINITIALIZED);
			if (hnd->trans->intdata.state == 
			    DM_STATE_TERMINATE_WAITING_CH) {
				dm_trans_complete(hnd, ESABABA);
			}
		} else {
			wimax_osal_trace_byte(
				PC2400M_GROUP_DM_ERROR,
				PC2400M_TRACE_UNKNOWN_STATUS_IN_TERMINATE,
				WIMAX_OSAL_PRIORITY_DEBUG,
				hnd->chipset.state);
		}
		break;

	case PC2400M_DRV_DM_PRODUCTION:
		if (hnd->chipset.state == L3L4_SYSTEM_STATE_PRODUCTION) {
			GOTO_STATE(hnd->state, 
				   L3L4_SYSTEM_STATE_PRODUCTION);
			if (hnd->trans->intdata.state == 
			    DM_STATE_PRODUCTION_WAITING_CH) {
				dm_trans_complete(hnd, ESABABA);
			}
		} else {
			wimax_osal_trace_byte(
				PC2400M_GROUP_DM_ERROR,
				PC2400M_TRACE_CHIP_UNEXP_STATE_IND,
				WIMAX_OSAL_PRIORITY_DEBUG,
				hnd->chipset.state);
		}
		break;

	default:
		break;
	}

	/* process unsolicited state changes only if state updated */
	if (rsstate) switch (hnd->chipset.state) {
	case L3L4_SYSTEM_STATE_RF_OFF:
	        switch (hnd->state) {
		case L3L4_SYSTEM_STATE_INIT:
		case L3L4_SYSTEM_STATE_RF_SHUTDOWN:
		        GOTO_STATE(hnd->state, L3L4_SYSTEM_STATE_RF_OFF);
			break;
		case L3L4_SYSTEM_STATE_RF_OFF:
		        /* no change */
		        break;
		default:
		        goto err1;
		}
		break;
	case L3L4_SYSTEM_STATE_RF_SHUTDOWN:
	        switch (hnd->state) {
		case L3L4_SYSTEM_STATE_READY:
		case L3L4_SYSTEM_STATE_DISCONNECTING:
		case L3L4_SYSTEM_STATE_SCAN:
		case L3L4_SYSTEM_STATE_STANDBY:
		case L3L4_SYSTEM_STATE_CONNECTING:
		case L3L4_SYSTEM_STATE_WIMAX_CONNECTED:
		case L3L4_SYSTEM_STATE_DATA_PATH_CONNECTED:
		case L3L4_SYSTEM_STATE_OUT_OF_ZONE:
		        /* the RF was switched off */
		        GOTO_STATE(hnd->state, L3L4_SYSTEM_STATE_RF_SHUTDOWN);
			break;
		case L3L4_SYSTEM_STATE_RF_SHUTDOWN:
		        /* no change */
		        break;
		default:
		        goto err1;
		}
		break;
	case L3L4_SYSTEM_STATE_SCAN:
		switch (hnd->state) {
		case L3L4_SYSTEM_STATE_READY:
		case L3L4_SYSTEM_STATE_STANDBY:
		case L3L4_SYSTEM_STATE_OUT_OF_ZONE:
			GOTO_STATE(hnd->state, L3L4_SYSTEM_STATE_SCAN);
			break;
		case L3L4_SYSTEM_STATE_SCAN:
			break;
		default:
			goto err1;
		}
		break;

	case L3L4_SYSTEM_STATE_STANDBY:
		switch (hnd->state) {
		case L3L4_SYSTEM_STATE_SCAN:
			GOTO_STATE(hnd->state, L3L4_SYSTEM_STATE_STANDBY);
			break;
		case L3L4_SYSTEM_STATE_STANDBY:
			break;
		default:
			goto err1;
		}
		break;

	case L3L4_SYSTEM_STATE_OUT_OF_ZONE:
		switch (hnd->state) {
		case L3L4_SYSTEM_STATE_SCAN:
			GOTO_STATE(hnd->state, L3L4_SYSTEM_STATE_OUT_OF_ZONE);
			break;
		default:
			goto err1;
		}
		break;

	case L3L4_SYSTEM_STATE_WIMAX_CONNECTED:
		switch (hnd->state) {
		case L3L4_SYSTEM_STATE_CONNECTING:
			GOTO_STATE(hnd->state,
				   L3L4_SYSTEM_STATE_WIMAX_CONNECTED);
			priv->send_network_status_ind(hnd->ctx,
						      E_NETWORK_STATUS_SF,
						      E_SUCCESS,
						      hnd->chipset.link.bs_id,
						      hnd->chipset.link.rssi);
			break;
		case L3L4_SYSTEM_STATE_WIMAX_CONNECTED:
			/* no change */
			break;
		default:
			goto err1;
		}
		break;

	case L3L4_SYSTEM_STATE_DATA_PATH_CONNECTED:
		switch (hnd->state) {
		case L3L4_SYSTEM_STATE_INIT:
		case L3L4_SYSTEM_STATE_READY:
			/* if creating the network connection implicitly,
			   during its own initialization */
		case L3L4_SYSTEM_STATE_CONNECTING:
			/* the device skips the WIMAX_CONNECTED state from
			   time to time */
		case L3L4_SYSTEM_STATE_WIMAX_CONNECTED:
			GOTO_STATE(hnd->state,
				   L3L4_SYSTEM_STATE_DATA_PATH_CONNECTED);
			priv->send_network_status_ind(
				hnd->ctx,
				E_NETWORK_STATUS_CONNECTED,
				E_SUCCESS,
				hnd->chipset.link.bs_id,
				hnd->chipset.link.rssi);
			break;
		case L3L4_SYSTEM_STATE_IDLE:
			GOTO_STATE(hnd->state,
				   L3L4_SYSTEM_STATE_DATA_PATH_CONNECTED);
		case L3L4_SYSTEM_STATE_DATA_PATH_CONNECTED:
			/* no change */
			break;
		default:
			goto err1;
		}
		break;

	case L3L4_SYSTEM_STATE_CONNECTING:
		switch (hnd->state) {
		case L3L4_SYSTEM_STATE_READY:
			GOTO_STATE(hnd->state,
				   L3L4_SYSTEM_STATE_CONNECTING);
			break;
		case L3L4_SYSTEM_STATE_CONNECTING:
			/* no change */
			break;
		default:
			goto err1;
		}
		break;				

	case L3L4_SYSTEM_STATE_READY:
		switch (hnd->state) {
		case L3L4_SYSTEM_STATE_RF_OFF:
		        /* RF switched on */
		        GOTO_STATE(hnd->state, L3L4_SYSTEM_STATE_READY);
			break;
		case L3L4_SYSTEM_STATE_SCAN:
		case L3L4_SYSTEM_STATE_STANDBY:
		case L3L4_SYSTEM_STATE_OUT_OF_ZONE:
			GOTO_STATE(hnd->state,
				   L3L4_SYSTEM_STATE_READY);
			/* indicate, that searching has completed */
			priv->send_search_resp(hnd->ctx, E_SUCCESS);
			break;
		case L3L4_SYSTEM_STATE_CONNECTING:
		case L3L4_SYSTEM_STATE_WIMAX_CONNECTED:
		case L3L4_SYSTEM_STATE_DATA_PATH_CONNECTED:
		case L3L4_SYSTEM_STATE_IDLE:
			GOTO_STATE(hnd->state,
				   L3L4_SYSTEM_STATE_READY);
			priv->send_network_status_ind(
				hnd->ctx,
				E_NETWORK_STATUS_DISCONNECTED,
				E_ERR_FAILURE,
				NULL,
				0);
			break;
		case L3L4_SYSTEM_STATE_READY:
			/* no change */
			break;

		case L3L4_SYSTEM_STATE_DISCONNECTING:
			GOTO_STATE(hnd->state,
				   L3L4_SYSTEM_STATE_READY);
			priv->send_network_status_ind(
				hnd->ctx,
				E_NETWORK_STATUS_DISCONNECTED,
				hnd->chipset.trans.disconnect_status,
				NULL,
				0);
			break;
		default:
			goto err1;
		}
		break;

	case L3L4_SYSTEM_STATE_DISCONNECTING:
		switch (hnd->state) {
		case L3L4_SYSTEM_STATE_CONNECTING:
		case L3L4_SYSTEM_STATE_WIMAX_CONNECTED:
		case L3L4_SYSTEM_STATE_DATA_PATH_CONNECTED:
		case L3L4_SYSTEM_STATE_IDLE:
			hnd->chipset.trans.disconnect_status = 
				E_ERR_FAILURE;
			GOTO_STATE(hnd->state,
				   L3L4_SYSTEM_STATE_DISCONNECTING);
			break;

		case L3L4_SYSTEM_STATE_DISCONNECTING:
			/* no change */
			break;
		default:
			goto err1;
			
		}
		break;

	case L3L4_SYSTEM_STATE_IDLE:
		switch (hnd->state) {
		case L3L4_SYSTEM_STATE_DATA_PATH_CONNECTED:
			GOTO_STATE(hnd->state, L3L4_SYSTEM_STATE_IDLE);
			break;
		case L3L4_SYSTEM_STATE_IDLE:
			break;
		default:
			goto err1;
		}
		break;

	case L3L4_SYSTEM_STATE_PRODUCTION:
		switch (hnd->state) {
		case L3L4_SYSTEM_STATE_PRODUCTION:
			/* from INIT to PRODUCTION is handled above */
			/* no change */
			break;
		default:
			goto err1;
		}
		break;

	case L3L4_SYSTEM_STATE_CONFIG:
		switch (hnd->state) {
		case L3L4_SYSTEM_STATE_CONFIG:
			/* no change */
			break;
		case L3L4_SYSTEM_STATE_UNINITIALIZED:
			GOTO_STATE(hnd->state, L3L4_SYSTEM_STATE_CONFIG);
			break;
		default:
			/* no way to config except via uninit */
			goto err1;
		}
		break;

	case L3L4_SYSTEM_STATE_UNINITIALIZED:
		switch (hnd->state) {
		case L3L4_SYSTEM_STATE_UNINITIALIZED:
			/* no change */
			break;
		default:
			/* can change to unitilialized only, if requested via
			   TERMINATE, which is handled above. */
			goto err1;
		}
		break;

	default:
		break;
	}
	
	goto out;
	
 err1:
	wimax_osal_trace_u16(PC2400M_GROUP_DM_ERROR, 
			     PC2400M_TRACE_CHIP_UNEXP_STATE_IND,
			     WIMAX_OSAL_PRIORITY_DEBUG,
			     hnd->chipset.state);

	/* the chipset will be out of sync in this event, irrecoverably */
	hnd->abort_mission(hnd);

 out:
	return;

}


