/*
 * pc2400m_drv.c
 *
 * Copyright (C) 2007 Nokia Corporation
 * Author: Juuso Oikarinen <juuso.oikarinen@nokia.com>
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 */


#include "pc2400m_drv_com.h"
#include "pc2400m_drv_spi.h"
#include "pc2400m_drv_if.h"
#include "pc2400m_drv_hdi.h"
#include "pc2400m_drv_cd.h"
#include "pc2400m_drv_dm.h"
#include "pc2400m_drv.h"

/* helper structure to maintain production testing data */
struct pc2400m_drv_prod_data {
	void *ptr;
	u32 len;
};

static void wimax_command(wimax_osal_context *ctx, struct drv_cmd* cmd, void*);
static void pc2400m_drv_send_network_status_ind(
	wimax_osal_context*,enum drv_network_status, enum drv_status, u8*, u8);
static void pc2400m_drv_send_system_state_ind(
	wimax_osal_context*, enum drv_system_state);
static void pc2400m_drv_send_search_ind(wimax_osal_context*, u8, u32*, u8*);
static void pc2400m_drv_send_search_resp(wimax_osal_context*, enum drv_status);

/* reference handling */
static void drv_ref(struct pc2400m_private *priv);
static void drv_unref(struct pc2400m_private *priv);
static void drv_unref_data(struct pc2400m_private *priv, drv_unref_cb cb,
		      void* data);

/* EAP subsystem */
static void eap_response(struct pc2400m_private *priv, wimax_osal_packet *msg);
static void eap_set_failure(struct pc2400m_private *priv);
static void eap_set_key(struct pc2400m_private *priv, u8 *msk);
static void eap_request(struct pc2400m_drv_cd_if *cdhnd, 
			u32 indid, 
			wimax_osal_packet *msg,
			void *data);
static s32 eap_send_key_resp(struct pc2400m_private *priv);
static void eap_key_request(struct pc2400m_drv_cd_if *cdhnd, 
			    u32 indid, 
			    wimax_osal_packet *msg,
			    void *data);
static void eap_alt_accept(struct pc2400m_drv_cd_if *cdhnd, 
			   u32 indid, 
			   wimax_osal_packet *msg,
			   void *data);


static struct wimax_driver_if pc2400m_drv_if = {
	wimax_command
};


/**
 * pc2400m_drv_wimax_open - open a new instance of the Pc2400m driver
 * @ctx: handle to identify the physical device for the driver
 *
 * Open a new instance of the pc2400m driver for a physical device
 */
struct wimax_driver_if *pc2400m_drv_wimax_open(wimax_osal_context *ctx) {

	struct pc2400m_private *priv;
	struct wimax_driver_if *iface = NULL;

	/* create the private data */
	priv = wimax_osal_mem_alloc(sizeof(struct pc2400m_private));
	wimax_osal_mem_set(priv, 0x00, sizeof(struct pc2400m_private));
	priv->ctx = ctx;

	wimax_osal_packet_list_initialize(&(priv->pkt_list));

	/* instantiate interface functions */
	priv->send_network_status_ind = pc2400m_drv_send_network_status_ind;
	priv->send_system_state_ind = pc2400m_drv_send_system_state_ind;
	priv->send_search_ind = pc2400m_drv_send_search_ind;
	priv->send_search_resp = pc2400m_drv_send_search_resp;

	priv->ref = drv_ref;
	priv->unref = drv_unref;
	priv->unref_data = drv_unref_data;

	/* start up with full data send quota */
	priv->send_credit = DRV_DATA_SEND_CREDIT_LEN;

	/* add the permanent reference to the instance */
	priv->ref(priv);

	/* instantiate subsystems */
	priv->dm = pc2400m_drv_dm_if_new(ctx);

	/* instance of the interface */
	iface = &pc2400m_drv_if;
	wimax_osal_ctx_priv_set(ctx, priv);

	/* NOTE: powering the chipset here causes problems as the
	   ctx variable is not fully initialized. */

	return iface;

}

static void pc2400m_drv_send_search_ind(
	wimax_osal_context *ctx, 
	u8 resc,
	u32 *res_nap,
	u8 *res_rssi)
{

	struct drv_ind ind;
	s32 i;

	ind.ind.search.p_search_result = 
		  (struct drv_search_result*)wimax_osal_mem_alloc(
			  resc * sizeof(struct drv_search_result));

	ind.ind_id = E_SEARCH_IND;
	ind.ind.search.response_type = DRV_RESP_SEARCH_TYPE_NETWORK;
	ind.ind.search.result_count = resc;
	ind.ind.search.search_progress = 0;

	for (i=0; i<resc; i++) {
		ind.ind.search.p_search_result[i].nap_id = res_nap[i];
		ind.ind.search.p_search_result[i].signal_strength = 
			res_rssi[i];
		ind.ind.search.p_search_result[i].number_of_nsp_id = 0;
		ind.ind.search.p_search_result[i].service_level_prediction = 
			DRV_SERVICE_LEVEL_PREDICTION_NA;

	}

	/* provide this information to the client */
	pc2400m_drv_adapt.ind(ctx, &ind);
	wimax_osal_mem_free(&(ind.ind.search.p_search_result));

	return;
}

static void pc2400m_drv_send_search_resp(
	wimax_osal_context *ctx, enum drv_status status)
{

	struct drv_resp resp;
	resp.cmd_resp_id = E_SEARCH_RESP;
	resp.cmd_status = status;
	

	/* provide this information to the client */
	pc2400m_drv_adapt.cmd_resp(ctx, &resp, NULL);

	return;
}
	

static void pc2400m_drv_send_system_state_ind(
	wimax_osal_context *ctx, enum drv_system_state system_state)
{

	struct drv_ind ind;

	ind.ind_id = E_SYSTEM_STATE_IND;
	ind.ind.system_state.system_state = system_state;

	/* power off the chipset, if it is a fatal error */
	if (system_state == E_SYSTEM_STATE_FATAL_ERROR)
		pc2400m_drv_adapt.set_power(ctx, FALSE);

	/* send indication */
	pc2400m_drv_adapt.ind(ctx, &ind);

}


static void pc2400m_drv_send_network_status_ind(
	wimax_osal_context *ctx,
	enum drv_network_status network_status,
	enum drv_status status,
	u8 *bs_id,
	u8 signal_strength)
{

	struct drv_ind ind;

	ind.ind_id = E_NETWORK_STATUS_IND;
	ind.ind.network_status.network_status = network_status;
	ind.ind.network_status.status = status;
	ind.ind.network_status.signal_strength = signal_strength;
	if (bs_id)
                wimax_osal_mem_cpy(ind.ind.network_status.bsid, bs_id, 
				    DRV_BSID_LENGTH);
	else
		wimax_osal_mem_set(ind.ind.network_status.bsid, 0xff, 
				   DRV_BSID_LENGTH);

	/* send indication */
	pc2400m_drv_adapt.ind(ctx, &ind);
	
}

static void configure_resp(struct pc2400m_drv_dm_if *dm, 
			   s32 status, 
			   void *data, 
			   void *cbdata,
			   void *cbdatacl)
{

	wimax_osal_context *ctx = (wimax_osal_context*)cbdata;
	struct drv_resp resp;	

	resp.cmd_resp_id = E_CONFIGURE_RESP;
	if (!status) 
		resp.cmd_status = E_SUCCESS;
	else if (status == -EINVAL)
		resp.cmd_status = E_ERR_INVALID_STATE;
	else
		resp.cmd_status = E_ERR_FAILURE;

	/* get hardware capabilities */
	if (dm->get_capabilities(
		    dm,
		    (resp.resp.configure.capabilities.p_fw_version),
		    (resp.resp.configure.capabilities.p_hw_version),
		    resp.resp.configure.capabilities.mac_address))
		resp.cmd_status = E_ERR_INVALID_STATE;

	/* send response */
	pc2400m_drv_adapt.cmd_resp(ctx, &resp, cbdatacl);

}

static void radio_state_resp(struct pc2400m_drv_dm_if *dm, 
			     s32 status, 
			     void *data, 
			     void *cbdata,
			     void *cbdatacl)
{

	wimax_osal_context *ctx = (wimax_osal_context*)cbdata;
	struct drv_resp resp;

	resp.cmd_resp_id = E_RADIO_STATE_RESP;
	if (!status) 
		resp.cmd_status = E_SUCCESS;
	else if (status == -EINVAL)
		resp.cmd_status = E_ERR_INVALID_STATE;
	else
		resp.cmd_status = E_ERR_FAILURE;

	/* send response */
	pc2400m_drv_adapt.cmd_resp(ctx, &resp, cbdatacl);

}

static void search_resp(struct pc2400m_drv_dm_if *dm, 
			s32 status, 
			void *data, 
			void *cbdata,
			void *cbdatacl)
{

	wimax_osal_context *ctx = (wimax_osal_context*)cbdata;
	struct pc2400m_private *priv = wimax_osal_ctx_priv_get(ctx);
	struct pc2400m_drv_dm_start_scan_params *params = data;

	/* free parameter data */
	wimax_osal_assert(params);
	wimax_osal_mem_free(&(params->req.p_search_plan));
	wimax_osal_mem_free(&(params->req.p_known_nap_info));
	wimax_osal_mem_free(&params);

	/* no response is sent, unless the scanning failed already */
	if (status == -EINVAL)
		priv->send_search_resp(ctx, E_ERR_INVALID_STATE);
	else if (status != ESABABA)
		priv->send_search_resp(ctx, E_ERR_FAILURE);

	/* otherwise, searching has commenced. */
	return;

}

static void search_stop_resp(struct pc2400m_drv_dm_if *dm, 
			     s32 status, 
			     void *data, 
			     void *cbdata,
			     void *cbdatacl)
{
	
	wimax_osal_context *ctx = (wimax_osal_context*)cbdata;
	struct drv_resp resp;

	resp.cmd_resp_id = E_SEARCH_STOP_RESP;
	if (!status) 
		resp.cmd_status = E_SUCCESS;
	else if (status == -EINVAL)
		resp.cmd_status = E_ERR_INVALID_STATE;
	else
		resp.cmd_status = E_ERR_FAILURE;
	
	/* send response */
	pc2400m_drv_adapt.cmd_resp(ctx, &resp, cbdatacl);

}

static void connect_resp(struct pc2400m_drv_dm_if *dm, 
			 s32 status, 
			 void *data, 
			 void *cbdata,
			 void *cbdatacl)
{
	
	wimax_osal_context *ctx = (wimax_osal_context*)cbdata;
	struct drv_resp resp;

	wimax_osal_mem_free(&data);

	resp.cmd_resp_id = E_ENTRY_RESP;
	if (!status) 
		resp.cmd_status = E_SUCCESS;
	else if (status == -EINVAL)
		resp.cmd_status = E_ERR_INVALID_STATE;
	else
		resp.cmd_status = E_ERR_FAILURE;
	
	/* send response */
	pc2400m_drv_adapt.cmd_resp(ctx, &resp, cbdatacl);

}

static void disconnect_resp(struct pc2400m_drv_dm_if *dm, 
			    s32 status, 
			    void *data, 
			    void *cbdata,
			    void *cbdatacl)
{
	
	wimax_osal_context *ctx = (wimax_osal_context*)cbdata;
	struct drv_resp resp;

	resp.cmd_resp_id = E_EXIT_RESP;
	if (!status) 
		resp.cmd_status = E_SUCCESS;
	else if (status == -EINVAL)
		resp.cmd_status = E_ERR_INVALID_STATE;
	else
		resp.cmd_status = E_ERR_FAILURE;
	
	/* send response */
	pc2400m_drv_adapt.cmd_resp(ctx, &resp, cbdatacl);

}

static void prod_test_resp(struct pc2400m_drv_cd_if *cd, 
			   wimax_osal_packet *msg, 
			   void *data) 
{
	
	wimax_osal_context *ctx = (wimax_osal_context*)data;
	struct drv_resp resp;

	resp.cmd_resp_id = E_PROD_TEST_RESP;
	resp.cmd_status = E_SUCCESS;

	resp.resp.prod_test.data_length = wimax_osal_packet_size(msg);
	resp.resp.prod_test.data = 
		  wimax_osal_mem_alloc(wimax_osal_packet_size(msg));
	wimax_osal_mem_cpy(
	        resp.resp.prod_test.data, wimax_osal_packet_ptr(msg),
		wimax_osal_packet_size(msg));
	wimax_osal_packet_free(&msg);
	
	/* send response */
	pc2400m_drv_adapt.cmd_resp(ctx, &resp, NULL);

}


static void production_resp(struct pc2400m_drv_dm_if *dm, 
			    s32 status, 
			    void *data, 
			    void *cbdata,
			    void *cbdatacl)
{
	
	wimax_osal_context *ctx = (wimax_osal_context*)cbdata;
	struct pc2400m_private *priv = wimax_osal_ctx_priv_get(ctx);
	struct pc2400m_drv_prod_data *pdata = 
		(struct pc2400m_drv_prod_data*)data;
	wimax_osal_packet *msg;

	if (!status)
		priv->mode_flags |= PC2400M_DRV_COM_PROD_TEST_MODE;

	/* forward message directly to the control dispatcher */
	if (priv->dm && priv->dm->cd &&
		(priv->mode_flags&PC2400M_DRV_COM_PROD_TEST_MODE)) {

		/* this operation will not maintain client data pointers */
		wimax_osal_assert(!cbdatacl);
			
		msg = wimax_osal_packet_alloc(pdata->len);
		wimax_osal_mem_cpy(wimax_osal_packet_put(msg, pdata->len),
				   pdata->ptr, pdata->len);
		
		priv->dm->cd->msg_enqueue(priv->dm->cd, msg, 
					  prod_test_resp, 
					  (void*)ctx, 
					  PC2400M_CD_FLAG_NONE);
	}

	wimax_osal_mem_free(&(pdata->ptr));
	wimax_osal_mem_free(&pdata);

}


static void trace_resp(struct pc2400m_drv_cd_if *cd, 
		       wimax_osal_packet *msg, 
		       void *data) 
{
	
        wimax_osal_context *ctx = (wimax_osal_context*)data;
	struct drv_resp resp;

	resp.cmd_resp_id = E_TRACE_RESP;
	resp.cmd_status = E_SUCCESS;

	resp.resp.trace.data_length = wimax_osal_packet_size(msg);
	resp.resp.trace.data = wimax_osal_mem_alloc(
		wimax_osal_packet_size(msg));
	wimax_osal_mem_cpy(resp.resp.trace.data, 
			   wimax_osal_packet_ptr(msg),
			   wimax_osal_packet_size(msg));
	wimax_osal_packet_free(&msg);

	/* send response */
	pc2400m_drv_adapt.cmd_resp(ctx, &resp, NULL);

}

static void statistics_get_resp(struct pc2400m_drv_cd_if *cd, 
				wimax_osal_packet *msg, 
				void *data) 
{
	
	wimax_osal_context *ctx = (wimax_osal_context*)data;
	struct drv_resp resp;

	resp.cmd_resp_id = E_STATISTICS_GET_RESP;
	resp.cmd_status = E_SUCCESS;

	resp.resp.statistics_get.data_length = wimax_osal_packet_size(msg);
	resp.resp.statistics_get.data = 
		  wimax_osal_mem_alloc(wimax_osal_packet_size(msg));
	wimax_osal_mem_cpy(resp.resp.statistics_get.data, 
			   wimax_osal_packet_ptr(msg),
			   wimax_osal_packet_size(msg));
	wimax_osal_packet_free(&msg);
	
	/* send response */
	pc2400m_drv_adapt.cmd_resp(ctx, &resp, NULL);

}

static void suspend_resp(struct pc2400m_drv_dm_if *dm, 
			 s32 status, 
			 void *data, 
			 void *cbdata,
			 void *cbdatacl)
{
	
	wimax_osal_context *ctx = (wimax_osal_context*)cbdata;
	struct drv_resp resp;

	/* suspend cannot fail, so status is ignored */
	resp.cmd_resp_id = E_SUSPEND_RESP;
	resp.cmd_status = E_SUCCESS;
	
	/* send response */
	pc2400m_drv_adapt.cmd_resp(ctx, &resp, cbdatacl);

}

static void resume_resp(struct pc2400m_drv_dm_if *dm, 
			s32 status, 
			void *data, 
			void *cbdata,
			void *cbdatacl)
{
	
	wimax_osal_context *ctx = (wimax_osal_context*)cbdata;
	struct drv_resp resp;
	
	/* resume cannot fail, so status is ignored */
	resp.cmd_resp_id = E_RESUME_RESP;
	resp.cmd_status = E_SUCCESS;
	
	/* send response */
	pc2400m_drv_adapt.cmd_resp(ctx, &resp, cbdatacl);

}

static void drv_ref(struct pc2400m_private *priv)
{

        priv->ref_cnt++;

}

static void drv_unref_data(struct pc2400m_private *priv, drv_unref_cb cb, 
		      void* data)
{

	wimax_osal_assert(cb);

	priv->unref_cb = cb;
	priv->unref_cb_data = data;

	priv->unref(priv);

	return;

}

static void drv_unref(struct pc2400m_private *priv)
{

	priv->ref_cnt--;

	if (!priv->ref_cnt) {
	        /* local copy of the unref cb pointers */
	        drv_unref_cb cb = priv->unref_cb;
		void *data = priv->unref_cb_data;
		wimax_osal_context *ctx = priv->ctx;

	        /* unregister EAP indications */			
	        priv->dm->cd->register_indication(
	                priv->dm->cd, L3_L4_OPCODE_REPORT_ALT_ACCEPT, 
			NULL, NULL);

		priv->dm->cd->register_indication(
			priv->dm->cd, L3_L4_OPCODE_REPORT_KEY_REQUEST, 
			NULL, NULL);

		priv->dm->cd->register_indication(
	                priv->dm->cd, L3_L4_OPCODE_REPORT_EAP_REQUEST, 
			NULL, NULL);

		/* free the subsystem instances */
		priv->dm->cleanup(priv->dm);
	
		/* free the packet buffer */
		wimax_osal_packet_list_release(&(priv->pkt_list), TRUE);

		/* indicate success */
		wimax_osal_trace_u32(PC2400M_GROUP_WIHAL_DEBUG, 
				     PC2400M_TRACE_DRIVER_UNLOADED, 
				     WIMAX_OSAL_PRIORITY_DIAGNOSTIC,
				     (u32)priv->ctx);

		/* free the instance itself */
		wimax_osal_ctx_priv_set(priv->ctx, NULL);
		wimax_osal_mem_free(&priv);

		/* call the unref cb to indicate that we're gone */
		if (cb) cb(ctx, data);

	}
	return;

}

static void send_close_resp(wimax_osal_context *ctx, void *data) 
{
	struct drv_resp resp;

	/* send notification */
	resp.cmd_resp_id = E_CLOSE_RESP;
	resp.cmd_status = E_SUCCESS;
	pc2400m_drv_adapt.cmd_resp(ctx, &resp, data);

}

static void close_resp(struct pc2400m_drv_dm_if *dm, 
			 s32 status, 
			 void *data, 
			 void *cbdata,
			 void *cb_cldata)
{
	
	wimax_osal_context *ctx = (wimax_osal_context*)cbdata;
	struct pc2400m_private *priv = wimax_osal_ctx_priv_get(ctx);

	/* power off the chipset */
	pc2400m_drv_adapt.set_power(ctx, FALSE);

	/* the unref operator will take care of cleaning the instance
	   once the stack has been cleared of references to it =>
	   it will then call our function and allow the thing to finalize */
	priv->unref_data(priv, send_close_resp, cb_cldata);

	return;

}


static void pkt_handler(struct pc2400m_drv_cd_if *cd, 
			wimax_osal_packet *msg, 
			s32 cs,
			void *data)
{

	struct pc2400m_private *priv = 
	       wimax_osal_ctx_priv_get((wimax_osal_context*)data);
	struct drv_ind ind;

	/* append the packet into the receive ring */
	wimax_osal_packet_list_append(&(priv->pkt_list), msg);
	
	/* poll for a transmission unless not already polled */
	if (!priv->recv_polled) {
		ind.ind_id = E_DATA_RECV_IND;
		ind.ind.data_receive.data_length = 
			wimax_osal_packet_list_len(&(priv->pkt_list));
		/* MISSING: do we get to know the transport id? */
		ind.ind.data_receive.transport_id = 0; /* Not used currently */
		ind.ind.data_receive.status = E_SUCCESS;
		priv->recv_polled = 1;
		pc2400m_drv_adapt.ind((wimax_osal_context*)data, &ind);
	}
	
	/* MISSING: a flow control mechanism towards the chipset is dearly
	   missed */
}


static void dbg_handler(struct pc2400m_drv_cd_if *cd, 
			wimax_osal_packet *msg, 
			void *data)
{

	/* trace the message out */
	wimax_osal_trace_data(PC2400M_GROUP_CHIPSET,
			      PC2400M_TRACE_UNSOLICITED, 
			      WIMAX_OSAL_PRIORITY_DIAGNOSTIC,
			      wimax_osal_packet_ptr(msg),
			      wimax_osal_packet_size(msg));

	wimax_osal_packet_free(&msg);	

}

static void initialize_resp(struct pc2400m_drv_dm_if *hnd, s32 ret, void *data)
{

        struct drv_resp resp;
	wimax_osal_context *ctx = hnd->ctx;
	struct pc2400m_private *priv = wimax_osal_ctx_priv_get(ctx);

        if (ret) {
	        priv->dm->reset(priv->dm);
	}
		
	if (!ret) {
	        /* register data handler */
	        priv->dm->cd->register_pkt_handler(
			priv->dm->cd, pkt_handler, (void*)ctx);

		/* register trace handler */
		priv->dm->cd->register_dbg_handler(
			priv->dm->cd, dbg_handler, (void*)ctx);

		/* register EAP handlers */
		priv->dm->cd->register_indication(
			priv->dm->cd, L3_L4_OPCODE_REPORT_EAP_REQUEST,
			eap_request, (void*)priv);
			
		priv->dm->cd->register_indication(
			priv->dm->cd, L3_L4_OPCODE_REPORT_ALT_ACCEPT,
			eap_alt_accept, (void*)priv);

		priv->dm->cd->register_indication(
			priv->dm->cd, L3_L4_OPCODE_REPORT_KEY_REQUEST,
			eap_key_request, (void*)priv);

	}

	/* generate response */
	resp.cmd_resp_id = E_INITIALIZE_RESP;
	if (ret<0) 
	        resp.cmd_status = E_ERR_FAILURE;
	else
	        resp.cmd_status = E_SUCCESS;

	pc2400m_drv_adapt.cmd_resp(ctx, &resp, data);

}



static void prod_handler(struct pc2400m_drv_cd_if *cd, 
			wimax_osal_packet *msg, 
			void *data)
{

	wimax_osal_context *ctx = (wimax_osal_context*)data;
	struct drv_ind ind;

	ind.ind_id = E_PROD_TEST_IND;

	ind.ind.prod_test.data_length = wimax_osal_packet_size(msg);
	ind.ind.prod_test.data = 
		  wimax_osal_mem_alloc(wimax_osal_packet_size(msg));
	wimax_osal_mem_cpy(
	        ind.ind.prod_test.data, wimax_osal_packet_ptr(msg),
		wimax_osal_packet_size(msg));
	wimax_osal_packet_free(&msg);
	
	/* send response */
	pc2400m_drv_adapt.ind(ctx, &ind);

}



static inline void transmit_packets(wimax_osal_context *ctx, u16 max, u8 tid, 
				    void *cldata)
{

	struct drv_ind ind;
	struct pc2400m_private *priv = wimax_osal_ctx_priv_get(ctx);
	struct drv_resp resp;
	s32 count, i;

	/* MISSING:
	   transport_id is ignored again, do we have the info? */
	count = max;
	if (count < wimax_osal_packet_list_len(&(priv->pkt_list)))
		count = wimax_osal_packet_list_len(&(priv->pkt_list));

	resp.resp.data_receive.p_data = (drv_data_block*)
		  wimax_osal_mem_alloc(count * sizeof(wimax_osal_packet*));
	resp.cmd_resp_id = E_DATA_RECEIVE_RESP;
	resp.resp.data_receive.transport_id = 0; /* Not used currently */
	resp.resp.data_receive.data_length = count;
	resp.resp.data_receive.last_packet = DRV_TRUE;
	
	for (i=0; i<count; i++) {
		resp.resp.data_receive.p_data[i] = 
			wimax_osal_packet_list_pop(&(priv->pkt_list));
	}

	priv->recv_polled = FALSE;
	
	pc2400m_drv_adapt.cmd_resp(ctx, &resp, cldata);

	wimax_osal_mem_free(&(resp.resp.data_receive.p_data));

	/* schedule another poll if there is still more data */
	if (wimax_osal_packet_list_len(&(priv->pkt_list))) {
		ind.ind_id = E_DATA_RECV_IND;
		ind.ind.data_receive.data_length = 
			wimax_osal_packet_list_len(&(priv->pkt_list));
		
		/* MISSING: do we get to know the transport id? */
		ind.ind.data_receive.transport_id = 0; /* Not used currently */
		ind.ind.data_receive.status = E_SUCCESS;
		priv->recv_polled = TRUE;
		pc2400m_drv_adapt.ind(ctx, &ind);
	}

}

/**
 * wimax_command - handler for WiMAX specific commands
 * @ctx: handle identifying the driver instance
 * @cmd: structure containing command parameters
 *
 * Start asynchronous handling of a WiMAX specific command.
 */
static void wimax_command(wimax_osal_context *ctx, 
			  struct drv_cmd *cmd, void *data) {
	
	struct pc2400m_private *priv = wimax_osal_ctx_priv_get(ctx);
	struct pc2400m_drv_dm_connect_params *connect_params;
	struct pc2400m_drv_dm_start_scan_params *start_scan_params;
	struct drv_resp resp;
	wimax_osal_packet *msg;
	s32 i;

	/* trace the message */
	if (cmd->cmd_id != E_DATA_RECEIVE && 
	    cmd->cmd_id != E_DATA_SEND) {
	        wimax_osal_trace_data(PC2400M_GROUP_WIHAL_DEBUG, 
				      PC2400M_TRACE_WIHAL_CTRL, 
				      WIMAX_OSAL_PRIORITY_DEBUG,
				      (void*)(cmd),
				      sizeof(struct drv_cmd));
	}

	/* accessing an uninstantiated instance of the driver? */
	wimax_osal_assert(priv);
	priv->ref(priv);

	/* accessing an unitialized instance of the driver? */
	wimax_osal_assert((priv->dm && priv->dm->cd) ||
			  cmd->cmd_id == E_INITIALIZE ||
			  cmd->cmd_id == E_CLOSE);

	switch (cmd->cmd_id) {

	case E_DATA_RECEIVE:
		/* transmit queued packets */
		transmit_packets(ctx, 
				 cmd->cmd.data_receive.data_length,
				 cmd->cmd.data_receive.transport_id,
				 data);
		break;

	case E_DATA_SEND:		
		if (priv->dm->get_link_state(priv->dm)) {
			/* not exceeding our credit are we? */

			wimax_osal_assert(priv->send_credit >=
					  cmd->cmd.data_send.data_length);

			priv->send_credit -= cmd->cmd.data_send.data_length;

			/* queue the data packet(s), and send the resp */
			for (i=0; i<cmd->cmd.data_send.data_length; i++) {
				if (i == cmd->cmd.data_send.data_length-1)
					priv->dm->cd->pkt_enqueue(
						priv->dm->cd,
						cmd->cmd.data_send.p_data[i], 
						-1 /*ruleindex=transport_id?*/,
						PC2400M_CD_FLAG_NONE);
				else
					priv->dm->cd->pkt_enqueue(
						priv->dm->cd, 
						cmd->cmd.data_send.p_data[i], 
						-1 /*ruleindex=transport_id?*/,
						PC2400M_CD_FLAG_AGGREGATE);

			}

			/* if the credit is exhausted and the chipset can
			   take more data, send response */
			if (!priv->send_credit) {
				priv->send_credit = DRV_DATA_SEND_CREDIT_LEN;
				resp.cmd_status = E_SUCCESS;
				resp.cmd_resp_id = E_DATA_SEND_RESP;
				resp.resp.data_send.transport_id = 
					cmd->cmd.data_send.transport_id;
				pc2400m_drv_adapt.cmd_resp(ctx, &resp, data);

				/* MISSING: currently, the chipset does not
				   have any form of flow control. Once it does,
				   and a condition is added here to handle it,
				   plese not the following:

				   It is possible that the credit is zero, 
				   with flow control on, when the link is 
				   closed. Once reopened in this case, a resp 
				   must be sent to the driver to allow it to 
				   once again send. */

			}

		} else {
			for (i=0; i<cmd->cmd.data_send.data_length; i++)
				wimax_osal_packet_free(
					&(cmd->cmd.data_send.p_data[i]));
				
			wimax_osal_trace(PC2400M_GROUP_WIHAL_ERROR, 
					 PC2400M_TRACE_LINK_DOWN, 
					 WIMAX_OSAL_PRIORITY_DEBUG);

			/* the response implies full credit to the client */
			priv->send_credit = DRV_DATA_SEND_CREDIT_LEN;
			resp.cmd_status = E_ERR_NO_CONNECTION;
			resp.cmd_resp_id = E_DATA_SEND_RESP;
			resp.resp.data_send.transport_id = 
				cmd->cmd.data_send.transport_id;
			pc2400m_drv_adapt.cmd_resp(ctx, &resp, data);
		}
		
		break;

	case E_INITIALIZE:

		/* Currently, only single block firmwares supported */
		wimax_osal_assert(cmd->cmd.initialize.block_count == 1);

		/* NOTE: power will go on during firmware download */

		/* the device manager will initialize the rest of the 
		   system */
		priv->dm->initialize(
			priv->dm, cmd->cmd.initialize.p_blocks[0].ptr,
			cmd->cmd.initialize.p_blocks[0].length,
			initialize_resp, (void*)data);

		break;

	case E_CONFIGURE:

		/* MISSING: ignoring configuration parameters */

	       		
		/* MISSING: maybe start the configuration transition? */
		configure_resp(priv->dm, ESABABA, NULL, ctx, data);
		break;

	case E_CLOSE:

		/* perform a soft reset on the chip before powering down 
		   unless the driver is in unitialized state, in which case
		   simply send the response and clean up */
	        priv->dm->transition(priv->dm, 
				     PC2400M_DRV_DM_TERMINATE,
				     NULL, 
				     close_resp, 
				     (void*)ctx,
				     data);
		break;

	case E_RADIO_STATE:
	        switch(cmd->cmd.radio_state.radio_state) {
		case DRV_RADIO_STATE_ON:
		        priv->dm->transition(
				priv->dm, 
				PC2400M_DRV_DM_ENABLE_RADIO,
				NULL, 
				radio_state_resp, 
				(void*)ctx,
				data);
				
			break;
		case DRV_RADIO_STATE_OFF:
		        priv->dm->transition(
				priv->dm, 
				PC2400M_DRV_DM_DISABLE_RADIO,
				NULL, 
				radio_state_resp, 
				(void*)ctx,
				data);
			
			break;
		}
		break;

	case E_SEARCH:
		wimax_osal_assert(data == NULL);
		
		/* copy the search command */
		start_scan_params = 
			  (struct pc2400m_drv_dm_start_scan_params*)
			  wimax_osal_mem_alloc
			  (sizeof(struct pc2400m_drv_dm_start_scan_params));
		wimax_osal_mem_cpy(&(start_scan_params->req),
				   &(cmd->cmd.search),
				   sizeof(struct drv_cmd_search));

		/* copy the search plan */
		start_scan_params->req.p_search_plan =
			  (struct drv_search_plan*)wimax_osal_mem_alloc
			   (sizeof(struct drv_search_plan) * 
			    cmd->cmd.search.search_plan_count);
		wimax_osal_mem_cpy(start_scan_params->req.p_search_plan,
				   cmd->cmd.search.p_search_plan,
				   (sizeof(struct drv_search_plan) * 
				    cmd->cmd.search.search_plan_count));

		/* copy the known nap info */
		start_scan_params->req.p_known_nap_info =
			  (struct drv_known_nap_info*)wimax_osal_mem_alloc
			   (sizeof(struct drv_known_nap_info) * 
			    cmd->cmd.search.known_nap_info_count);
		wimax_osal_mem_cpy(start_scan_params->req.p_known_nap_info,
				   cmd->cmd.search.p_known_nap_info,
				   (sizeof(struct drv_known_nap_info) * 
				    cmd->cmd.search.known_nap_info_count));

		priv->dm->transition(priv->dm, 
				     PC2400M_DRV_DM_START_SCAN,
				     (void*)start_scan_params, 
				     search_resp,
				     (void*)ctx,
				     data);
		break;

	case E_SEARCH_STOP:
	        priv->dm->transition(priv->dm, 
				     PC2400M_DRV_DM_STOP_SCAN,
				     NULL, 
				     search_stop_resp,
				     (void*)ctx,
				     data);
		break;
		
	case E_ENTRY:
	        connect_params = 
		        (struct pc2400m_drv_dm_connect_params*)
			wimax_osal_mem_alloc(
                                sizeof(struct pc2400m_drv_dm_connect_params));

		wimax_osal_mem_cpy(
			&(connect_params->req), &(cmd->cmd.entry),
			sizeof(struct pc2400m_drv_dm_connect_params));

		priv->dm->transition(priv->dm, 
				     PC2400M_DRV_DM_CONNECT,
				     (void*)connect_params, 
				     connect_resp,
				     (void*)ctx,
				     data);
		break;

	case E_EXIT:
	        priv->dm->transition(priv->dm, 
				     PC2400M_DRV_DM_DISCONNECT,
				     NULL, 
				     disconnect_resp,
				     (void*)ctx,
				     data);
		break;

	case E_PROD_TEST:
	{
		struct pc2400m_drv_prod_data *pdata;

		pdata = (struct pc2400m_drv_prod_data*)
			  wimax_osal_mem_alloc(
				  sizeof(struct pc2400m_drv_prod_data));
		pdata->ptr = cmd->cmd.prod_test.data;
		pdata->len = cmd->cmd.prod_test.data_length;

		/* upon the first production test message, change to
		   the production testing mode */
		if (!(priv->mode_flags &
		      PC2400M_DRV_COM_PROD_TEST_MODE)) {
		        priv->dm->transition(priv->dm, 
					     PC2400M_DRV_DM_PRODUCTION,
					     pdata, 
					     production_resp,
					     (void*)ctx,
					     data);
			/* register a handler for production testing 
			   reports */
			priv->dm->cd->register_prod_handler(
				priv->dm->cd, prod_handler, ctx);
		} else {
  		        production_resp(priv->dm, ESABABA, pdata, ctx, data);
		}
		break;
	}

	case E_TRACE:

		/* forward message directly to the control dispatcher */
	        wimax_osal_assert(!data);
			
		msg = wimax_osal_packet_alloc(
			cmd->cmd.trace.data_length);
		wimax_osal_mem_cpy(wimax_osal_packet_put(
			msg, cmd->cmd.trace.data_length),
			cmd->cmd.trace.data,
			cmd->cmd.trace.data_length);
		
		priv->dm->cd->msg_enqueue(
			priv->dm->cd, msg, 
			trace_resp, 
			(void*)ctx, 
			PC2400M_CD_FLAG_NONE);
		
		wimax_osal_mem_free(&(cmd->cmd.trace.data));
		       
		break;

	case E_STATISTICS_GET:
		/* forward message directly to the control dispatcher */
	        wimax_osal_assert(!data);
			
		msg = wimax_osal_packet_alloc(
			cmd->cmd.statistics_get.data_length);
		wimax_osal_mem_cpy(wimax_osal_packet_put(
				           msg, 
					   cmd->
					   cmd.statistics_get.data_length),
		        cmd->cmd.statistics_get.data,
			cmd->cmd.statistics_get.data_length);
			
		priv->dm->cd->msg_enqueue(
			priv->dm->cd, msg, 
			statistics_get_resp, 
			(void*)ctx, 
			PC2400M_CD_FLAG_NONE);
		
		wimax_osal_mem_free(&(cmd->cmd.statistics_get.data));
		break;

	case E_EAP_DATA:
		/* forward the data to the eap subsystem */
		eap_response(priv, cmd->cmd.eap_data.p_payload);

		resp.cmd_resp_id = E_EAP_DATA_RESP;
		resp.cmd_status = E_SUCCESS;
		pc2400m_drv_adapt.cmd_resp(ctx, &resp, data);
		break;

	case E_EAP_RESULT:
		/* either indicate failure or forward the key to the
		   eap subsystem */
		if (cmd->cmd.eap_result.result) {
			wimax_osal_assert(cmd->cmd.eap_result.msk_length ==
					  L3L4_MSK_LENGTH);
			eap_set_key(priv, cmd->cmd.eap_result.p_msk);

			/* there is an L3L4 for eap success as well, but
			   the spec specifies that it is optional, and can
			   just as well be deduced by the chipset itself. 
			   
			   we do not send it. */
		} else {
			eap_set_failure(priv);
		}

		resp.cmd_resp_id = E_EAP_RESULT_RESP;
		resp.cmd_status = E_SUCCESS;
		pc2400m_drv_adapt.cmd_resp(ctx, &resp, data);
		break;

	case E_SUSPEND:
		/* suspend device manager */
		priv->dm->transition(priv->dm, 
				     PC2400M_DRV_DM_SUSPEND,
				     NULL, 
				     suspend_resp,
				     (void*)ctx,
				     data);
		
		break;

	case E_RESUME:
		/* resume the device manager */
	        priv->dm->transition(priv->dm, 
				     PC2400M_DRV_DM_RESUME,
				     NULL, 
				     resume_resp,
				     (void*)ctx,
				     data);

		break;

	default:
	        wimax_osal_trace_u32(PC2400M_GROUP_WIHAL_ERROR, 
				     PC2400M_TRACE_UNKNOWN_CMD, 
				     WIMAX_OSAL_PRIORITY_ERROR, cmd->cmd_id);
	}

	priv->unref(priv);

	return;

}





/*
 * EAP functionality 
 */

static void eap_response(struct pc2400m_private *priv, wimax_osal_packet *msg)
{

	struct L3L4_TLV_STR_REPORT_EAP_RESPONSE *resp;
	wimax_osal_packet *newmsg;
	u32 len = wimax_osal_packet_size(msg);

	/* create a duplicate message with enough space to hold excess padding
	   at the end */
#define PAD4(x) (((x) + (4-1)) & (~(4-1)))
	newmsg = wimax_osal_packet_alloc(
		PC2400M_DRV_CHIPIF_CTRL_MSG_LEN +
		PAD4(L3L4_TLV_SIZE_REPORT_EAP_RESPONSE + len));

	/* insert the message header in front of the TLV */
	priv->dm->setup_header(
		priv->dm, 
		wimax_osal_packet_put(newmsg, PC2400M_DRV_CHIPIF_CTRL_MSG_LEN),
		L4_L3_OPCODE_CMD_SEND_EAP_RESPONSE,
		PAD4(L3L4_TLV_SIZE_REPORT_EAP_RESPONSE + len));       

	/* insert the TLV header in front of the data */
	resp = (struct L3L4_TLV_STR_REPORT_EAP_RESPONSE*)
		wimax_osal_packet_put(
			newmsg, L3L4_TLV_SIZE_REPORT_EAP_RESPONSE);
	resp->tlv.type = 
		WIMAX_OSAL_U16_TO_LE(L3L4_TLV_TYPE_REPORT_EAP_RESPONSE);
	resp->tlv.length = 
		WIMAX_OSAL_U16_TO_LE(PAD4(L3L4_TLV_SIZE_REPORT_EAP_RESPONSE + 
					  len) - L3L4_TLV_HDR_LEN);
	resp->length = WIMAX_OSAL_U16_TO_LE(len);
	
	/* copy the payload data into the message */
	wimax_osal_mem_cpy(wimax_osal_packet_put(newmsg, len),
			   wimax_osal_packet_ptr(msg),
			   len);

	/* trace now to make the "len" available for other use ;) */
	wimax_osal_trace_u32(PC2400M_GROUP_EAP_DEBUG, 
			     PC2400M_TRACE_EAP_RESP,
			     WIMAX_OSAL_PRIORITY_DEBUG, len);

	/* set the padding bytes to 0xff */
	len = PAD4(L3L4_TLV_SIZE_REPORT_EAP_RESPONSE + len) - 
		(len + L3L4_TLV_SIZE_REPORT_EAP_RESPONSE);
	wimax_osal_mem_set(wimax_osal_packet_put(newmsg, len), 0xff, len);

	/* remove the original payload */
	wimax_osal_packet_free(&msg);
	
	priv->dm->cd->msg_enqueue(priv->dm->cd, newmsg, NULL, NULL, 
				  PC2400M_CD_FLAG_NONE);	

}

static void eap_set_failure(struct pc2400m_private *priv)
{

	wimax_osal_packet *msg;

	/* send EAP failure message directly to chipset */
	msg = wimax_osal_packet_alloc(PC2400M_DRV_CHIPIF_CTRL_MSG_LEN);
	priv->dm->setup_header(
		priv->dm, 
		wimax_osal_packet_put(
			msg, PC2400M_DRV_CHIPIF_CTRL_MSG_LEN),
		L4_L3_OPCODE_SET_EAP_FAIL,
		0);

	priv->dm->cd->msg_enqueue(priv->dm->cd, msg, NULL, NULL, 
				  PC2400M_CD_FLAG_NONE);

}

static void eap_set_key(struct pc2400m_private *priv, u8 *msk)
{

	/* copy the key into our own buffer */
	wimax_osal_mem_cpy(priv->msk, msk, L3L4_MSK_LENGTH);
	priv->msk_flags |= PC2400M_DRV_MSK_FLAG_VALID;

	wimax_osal_trace(PC2400M_GROUP_EAP_DEBUG, 
			 PC2400M_TRACE_MSK_RECVD,
			 WIMAX_OSAL_PRIORITY_DEBUG);

	/* if a request is pending for the key from the device, dispatch
	   immediately */
	if (priv->msk_flags & PC2400M_DRV_MSK_FLAG_REQD) {
	        wimax_osal_timer_cancel(priv->msk_req_timer);
		priv->msk_req_timer = 0;
		
		eap_send_key_resp(priv);
	}

}

static void eap_request(struct pc2400m_drv_cd_if *cdhnd, 
			u32 indid, 
			wimax_osal_packet *msg,
			void *data)
{

	struct pc2400m_private *priv = (struct pc2400m_private*)data;
	struct drv_ind ind;
	struct L3L4_TLV_STR_REPORT_EAP_REQUEST *tlv;
	u16 len;

	/* extract the payload */
	tlv = (struct L3L4_TLV_STR_REPORT_EAP_REQUEST*)
		priv->dm->get_tlv(priv->dm, wimax_osal_packet_ptr(msg),
				  L3L4_TLV_TYPE_REPORT_EAP_REQUEST, NULL);
	
	wimax_osal_assert(tlv);
	
	/* get the actual payload length */
	len = WIMAX_OSAL_LE_TO_U16(tlv->length);

	/* forward the EAP payload as-is to the client, remove the message
	   headers in front */
	wimax_osal_packet_head_cut(msg, L3L4_TLV_SIZE_REPORT_EAP_REQUEST + 
				   PC2400M_DRV_CHIPIF_CTRL_MSG_LEN);

	/* remove excess padding and the end of the payload */
	wimax_osal_packet_trim(msg, len);

	ind.ind_id = E_EAP_DATA_IND;
	ind.ind.eap_data.p_payload = msg;
	
	wimax_osal_trace_u32(PC2400M_GROUP_EAP_DEBUG, 
			     PC2400M_TRACE_EAP_REQ,
			     WIMAX_OSAL_PRIORITY_DEBUG, 
			     WIMAX_OSAL_LE_TO_U16(tlv->tlv.length));
	
	pc2400m_drv_adapt.ind((wimax_osal_context*)data, &ind);
	
}

static s32 eap_send_key_resp(struct pc2400m_private *priv)
{

	struct L3L4_TLV_STR_SET_EAP_KEY_VALID *valid;
	struct L3L4_TLV_STR_SET_EAP_KEY_MSK *msk;
	wimax_osal_packet *msg;
	s32 ret = ESABABA;

	/* if there is a valid key, set it to the device */
	if (priv->msk_flags & PC2400M_DRV_MSK_FLAG_VALID) {
		msg = wimax_osal_packet_alloc(PC2400M_DRV_CHIPIF_CTRL_MSG_LEN +
					      L3L4_TLV_SIZE_SET_EAP_KEY_VALID +
					      L3L4_TLV_SIZE_SET_EAP_KEY_MSK);
		priv->dm->setup_header(
			priv->dm, 
			wimax_osal_packet_put(
				msg, PC2400M_DRV_CHIPIF_CTRL_MSG_LEN),
			L4_L3_OPCODE_SET_EAP_KEY,
			L3L4_TLV_SIZE_SET_EAP_KEY_VALID + 
			L3L4_TLV_SIZE_SET_EAP_KEY_MSK);

		/* the key is valid ... */
		valid = (struct L3L4_TLV_STR_SET_EAP_KEY_VALID*)
			wimax_osal_packet_put(
				msg, L3L4_TLV_SIZE_SET_EAP_KEY_VALID);
		valid->tlv.type = 
			WIMAX_OSAL_U16_TO_LE(L3L4_TLV_TYPE_SET_EAP_KEY_VALID);
		valid->tlv.length = 
			WIMAX_OSAL_U16_TO_LE(L3L4_TLV_SIZE_SET_EAP_KEY_VALID -
					     L3L4_TLV_HDR_LEN);
		valid->key_valid = 
			WIMAX_OSAL_U32_TO_LE(L3L4_EAP_KEY_VALID_VALID);

		/* ... and follows in the next tlv */
		msk = (struct L3L4_TLV_STR_SET_EAP_KEY_MSK*)
			wimax_osal_packet_put(
				msg, L3L4_TLV_SIZE_SET_EAP_KEY_MSK);
		msk->tlv.type = 
			WIMAX_OSAL_U16_TO_LE(L3L4_TLV_TYPE_SET_EAP_KEY_MSK);
		msk->tlv.length = 
			WIMAX_OSAL_U16_TO_LE(L3L4_TLV_SIZE_SET_EAP_KEY_MSK -
					     L3L4_TLV_HDR_LEN);

		wimax_osal_mem_cpy(msk->msk, priv->msk, L3L4_MSK_LENGTH);

		priv->dm->cd->msg_enqueue(priv->dm->cd, msg, NULL, NULL, 
					  PC2400M_CD_FLAG_NONE);

		wimax_osal_mem_set(priv->msk, 0x00, L3L4_MSK_LENGTH);
		priv->msk_flags &= (~PC2400M_DRV_MSK_FLAG_VALID);
		priv->msk_flags &= (~PC2400M_DRV_MSK_FLAG_REQD);

		wimax_osal_trace(PC2400M_GROUP_EAP_INFO, 
				 PC2400M_TRACE_MSK_SET,
				 WIMAX_OSAL_PRIORITY_INFO);

	} else {
		ret = -EFAULT;
	}

	return ret;

}

static void eap_key_request_timeout(s32 id, void* data)
{

	struct pc2400m_private *priv = (struct pc2400m_private*)data;	
	struct L3L4_TLV_STR_SET_EAP_KEY_VALID *valid;
	wimax_osal_packet *msg;

	/* trace */
	wimax_osal_trace(PC2400M_GROUP_EAP_ERROR, 
			 PC2400M_TRACE_MSK_NOT_RECVD,
			 WIMAX_OSAL_PRIORITY_ERROR);

	/* no eap key received, respond with no valid key */
	msg = wimax_osal_packet_alloc(PC2400M_DRV_CHIPIF_CTRL_MSG_LEN +
				      L3L4_TLV_SIZE_SET_EAP_KEY_VALID);
	priv->dm->setup_header(
		priv->dm, 
		wimax_osal_packet_put(msg, PC2400M_DRV_CHIPIF_CTRL_MSG_LEN),
		L4_L3_OPCODE_SET_EAP_KEY,
		L3L4_TLV_SIZE_SET_EAP_KEY_VALID);
	
	valid = (struct L3L4_TLV_STR_SET_EAP_KEY_VALID*)
		wimax_osal_packet_put(msg, L3L4_TLV_SIZE_SET_EAP_KEY_VALID);
	valid->tlv.type = 
		WIMAX_OSAL_U16_TO_LE(L3L4_TLV_TYPE_SET_EAP_KEY_VALID);
	valid->tlv.length = 
		WIMAX_OSAL_U16_TO_LE(L3L4_TLV_SIZE_SET_EAP_KEY_VALID -
				     L3L4_TLV_HDR_LEN);
	valid->key_valid = 
		WIMAX_OSAL_U32_TO_LE(L3L4_EAP_KEY_VALID_INVALID);

	priv->dm->cd->msg_enqueue(priv->dm->cd, msg, NULL, NULL, 
				  PC2400M_CD_FLAG_NONE);

	priv->msk_flags &= (~PC2400M_DRV_MSK_FLAG_REQD);

}

static void eap_key_request(struct pc2400m_drv_cd_if *cdhnd, 
			    u32 indid, 
			    wimax_osal_packet *msg,
			    void *data)
{

	struct pc2400m_private *priv = (struct pc2400m_private*)data;

	/* respond with the key if valid, otherwise wait for
	   it to become valid */
	if (eap_send_key_resp(priv) != ESABABA) {
		priv->msk_req_timer = 
			wimax_osal_timer_create(
				PC2400M_DRV_MSK_REQ_TIMEOUT,
				WIMAX_OSAL_TR_10MS,
				0,
				(void*)priv,
				eap_key_request_timeout);

		priv->msk_flags |= PC2400M_DRV_MSK_FLAG_REQD;

	}

	wimax_osal_packet_free(&msg);
}

static void eap_alt_accept(struct pc2400m_drv_cd_if *cdhnd, 
			   u32 indid, 
			   wimax_osal_packet *msg,
			   void *data)
{

	wimax_osal_trace(PC2400M_GROUP_EAP_ERROR, 
			 PC2400M_TRACE_ALT_ACCEPT_IGN,
			 WIMAX_OSAL_PRIORITY_ERROR);

	wimax_osal_packet_free(&msg);

}


