/*
 * omap24xxcam.c
 *
 * OMAP 2 camera block driver.
 *
 * Copyright (C) 2004 MontaVista Software, Inc.
 * Copyright (C) 2004 Texas Instruments.
 * Copyright (C) 2007 Nokia Corporation.
 *
 * Contact: Sakari Ailus <sakari.ailus@nokia.com>
 *
 * Based on code from Andy Lowe <source@mvista.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 */

#include <linux/module.h>
#include <linux/delay.h>
#include <linux/kernel.h>
#include <linux/interrupt.h>
#include <linux/videodev2.h>
#include <linux/pci.h>		/* needed for videobufs */
#include <linux/version.h>
#include <linux/platform_device.h>
#include <linux/clk.h>

#include <media/v4l2-common.h>

#include <asm/io.h>
#include <asm/arch/dvfs_notif.h>

#include "omap24xxcam.h"

#include "sensor_if.h"

#define RESET_TIMEOUT_NS 10000

static void omap24xxcam_reset(struct omap24xxcam_device *cam);
static int omap24xxcam_device_register(struct omap24xxcam_device *cam);
static void omap24xxcam_device_unregister(struct omap24xxcam_device *cam);
static int omap24xxcam_remove(struct platform_device *pdev);

/* module parameters */
static int video_nr = -1;	/* video device minor (-1 ==> auto assign) */
/* Maximum amount of memory to use for capture buffers.
 * Default is 4800KB, enough to double-buffer SXGA.
 */
static int capture_mem = 1280 * 960 * 2 * 2;

/*
 *
 * Clocks.
 *
 */

static void omap24xxcam_clock_put(struct omap24xxcam_device *cam)
{
	if (cam->ick != NULL && !IS_ERR(cam->ick))
		clk_put(cam->ick);
	if (cam->fck != NULL && !IS_ERR(cam->fck))
		clk_put(cam->fck);

	cam->ick = cam->fck = NULL;
}

static int omap24xxcam_clock_get(struct omap24xxcam_device *cam)
{
	int rval = 0;

	cam->fck = clk_get(cam->dev, "cam_fck");
	if (IS_ERR(cam->fck)) {
		dev_err(cam->dev, "can't get cam_fck");
		rval = PTR_ERR(cam->fck);
		omap24xxcam_clock_put(cam);
		return rval;
	}

	cam->ick = clk_get(cam->dev, "cam_ick");
	if (IS_ERR(cam->ick)) {
		dev_err(cam->dev, "can't get cam_ick");
		rval = PTR_ERR(cam->ick);
		omap24xxcam_clock_put(cam);
	}

	return rval;
}

static void omap24xxcam_clock_on(struct omap24xxcam_device *cam)
{
	clk_enable(cam->fck);
	clk_enable(cam->ick);
}

static void omap24xxcam_clock_off(struct omap24xxcam_device *cam)
{
	clk_disable(cam->fck);
	clk_disable(cam->ick);
}

/*
 *
 * Camera core
 *
 */

/*
 * Set xclk.
 *
 * If the given value is not usable, the next possible lower value
 * will be selected.
 *
 * To disable xclk, use value zero.
 *
 * The selected xclk value is returned.
 */
static u32 omap24xxcam_core_xclk_set(const struct omap24xxcam_device *cam,
				     u32 xclk)
{
	u32 divisor;

	if (xclk) {
		if (xclk > CAM_MCLK)
			xclk = CAM_MCLK;

		divisor = CAM_MCLK / xclk;
		if (xclk * divisor < CAM_MCLK)
			divisor += 1;
		if (divisor > 30)
			divisor = 30;
		xclk = CAM_MCLK / divisor;

		if (divisor == 1)
			omap24xxcam_reg_out(cam->mmio_base + CC_REG_OFFSET,
					    CC_CTRL_XCLK,
					    CC_CTRL_XCLK_DIV_BYPASS);
		else
			omap24xxcam_reg_out(cam->mmio_base + CC_REG_OFFSET,
					    CC_CTRL_XCLK, divisor);
	} else
		omap24xxcam_reg_out(cam->mmio_base + CC_REG_OFFSET,
				    CC_CTRL_XCLK, CC_CTRL_XCLK_DIV_STABLE_LOW);

	return xclk;
}

static void omap24xxcam_core_hwinit(const struct omap24xxcam_device *cam)
{
	/* Setting the camera core AUTOIDLE bit causes problems with frame
	 * synchronization, so we will clear the AUTOIDLE bit instead.
	 */
	//omap24xxcam_reg_out(cam, CC_SYSCONFIG, 0);
	omap24xxcam_reg_out(cam->mmio_base + CC_REG_OFFSET, CC_SYSCONFIG,
			    CC_SYSCONFIG_AUTOIDLE);

	/* program the camera interface DMA packet size */
	omap24xxcam_reg_out(cam->mmio_base + CC_REG_OFFSET, CC_CTRL_DMA,
			    CC_CTRL_DMA_EN | (DMA_THRESHOLD / 4 - 1));

	/* enable camera core error interrupts */
	omap24xxcam_reg_out(cam->mmio_base + CC_REG_OFFSET, CC_IRQENABLE,
			    CC_IRQENABLE_FW_ERR_IRQ
			    | CC_IRQENABLE_FSC_ERR_IRQ
			    | CC_IRQENABLE_SSC_ERR_IRQ
			    | CC_IRQENABLE_FIFO_OF_IRQ);
}

/*
 * Enable the camera core.
 *
 * Data transfer to the camera DMA starts from next starting frame.
 */
static void omap24xxcam_core_enable(const struct omap24xxcam_device *cam)
{
	omap24xxcam_reg_out(cam->mmio_base + CC_REG_OFFSET, CC_CTRL,
			    CC_CTRL_NOBT_SYNCHRO
			    | CC_CTRL_PAR_CLK_POL
			    | CC_CTRL_PAR_MODE_NOBT8
			    | CC_CTRL_CC_EN);
}

/*
 * Disable camera core.
 *
 * The data transfer will be stopped immediately (CC_CTRL_CC_RST). The
 * core internal state machines will be reset. Use
 * CC_CTRL_CC_FRAME_TRIG instead if you want to transfer the current
 * frame completely.
 */
static void omap24xxcam_core_disable(const struct omap24xxcam_device *cam)
{
	omap24xxcam_reg_out(cam->mmio_base + CC_REG_OFFSET, CC_CTRL,
			    CC_CTRL_CC_RST);
}

/* Interrupt service routine for camera core interrupts. */
static void omap24xxcam_core_isr(struct omap24xxcam_device *cam)
{
	u32 cc_irqstatus;
	const u32 cc_irqstatus_err =
		CC_IRQSTATUS_FW_ERR_IRQ
		| CC_IRQSTATUS_FSC_ERR_IRQ
		| CC_IRQSTATUS_SSC_ERR_IRQ
		| CC_IRQSTATUS_FIFO_UF_IRQ
		| CC_IRQSTATUS_FIFO_OF_IRQ;

	cc_irqstatus = omap24xxcam_reg_in(cam->mmio_base + CC_REG_OFFSET,
					  CC_IRQSTATUS);
	omap24xxcam_reg_out(cam->mmio_base + CC_REG_OFFSET, CC_IRQSTATUS,
			    cc_irqstatus);

	if (cc_irqstatus & cc_irqstatus_err
	    && !atomic_read(&cam->in_reset)) {
		dev_dbg(cam->dev, "resetting camera, cc_irqstatus 0x%x\n",
			cc_irqstatus);
		omap24xxcam_reset(cam);
	}
}

/*
 *
 * videobuf_buffer handling.
 *
 * Memory for mmapped videobuf_buffers is not allocated
 * conventionally, but by several kmalloc allocations and then
 * creating the scatterlist on our own. User-space buffers are handled
 * normally.
 *
 */

/*
 * Free the memory-mapped buffer memory allocated for a
 * videobuf_buffer and the associated scatterlist.
 */
static void omap24xxcam_vbq_free_mmap_buffer(struct videobuf_buffer *vb)
{
	int i;
	size_t alloc_size;
	struct page *page;

	if (vb->dma.sglist == NULL)
		return;

	i = vb->dma.sglen;
	while (i) {
		i--;
		alloc_size = vb->dma.sglist[i].length;
		page = vb->dma.sglist[i].page;
		do {
			ClearPageReserved(page++);
		} while (alloc_size -= PAGE_SIZE);
		__free_pages(vb->dma.sglist[i].page,
			     get_order(vb->dma.sglist[i].length));
	}

	kfree(vb->dma.sglist);
	vb->dma.sglist = NULL;
}

/* Release all memory related to the videobuf_queue. */
static void omap24xxcam_vbq_free_mmap_buffers(struct videobuf_queue *vbq)
{
	int i;

	mutex_lock(&vbq->lock);

	for (i = 0; i < VIDEO_MAX_FRAME; i++) {
		if (NULL == vbq->bufs[i])
			continue;
		if (V4L2_MEMORY_MMAP != vbq->bufs[i]->memory)
			continue;
		vbq->ops->buf_release(vbq, vbq->bufs[i]);
		omap24xxcam_vbq_free_mmap_buffer(vbq->bufs[i]);
		kfree(vbq->bufs[i]);
		vbq->bufs[i] = NULL;
	}

	mutex_unlock(&vbq->lock);

	videobuf_mmap_free(vbq);
}

/*
 * Allocate physically as contiguous as possible buffer for video
 * frame and allocate and build DMA scatter-gather list for it.
 */
static int omap24xxcam_vbq_alloc_mmap_buffer(struct videobuf_buffer *vb)
{
	unsigned int order;
	size_t alloc_size, size = vb->bsize; /* vb->bsize is page aligned */
	struct page *page;
	int max_pages, err = 0, i = 0;

	/* allocate maximum size scatter-gather list. Note this is overhead. We
	 * may not use as many entries as we allocate */
	max_pages = vb->bsize >> PAGE_SHIFT;
	vb->dma.sglist =
		kcalloc(max_pages, sizeof(*vb->dma.sglist), GFP_KERNEL);
	if (vb->dma.sglist == NULL) {
		err = -ENOMEM;
		goto out;
	}

	while (size) {
		order = get_order(size);
		/* do not over-allocate even if we would get larger contiguous
		 * chunk that way */
		if ((PAGE_SIZE << order) > size)
			order--;

		/* try to allocate as many contiguous pages as possible */
		page = alloc_pages(GFP_KERNEL | GFP_DMA, order);
		/* if allocation fails, try to allocate smaller amount */
		while (page == NULL) {
			order--;
			page = alloc_pages(GFP_KERNEL | GFP_DMA, order);
			if (page == NULL && !order) {
				err = -ENOMEM;
				goto out;
			}
		}
		size -= (PAGE_SIZE << order);

		/* append allocated chunk of pages into scatter-gather list */
		vb->dma.sglist[i].page = page;
		vb->dma.sglist[i].length = (PAGE_SIZE << order);
		vb->dma.sglen++;
		i++;

		alloc_size = (PAGE_SIZE << order);

		/* clear pages before giving them to user space */
		memset(page_address(page), 0, alloc_size);

		/* mark allocated pages reserved */
		do {
			SetPageReserved(page++);
		} while (alloc_size -= PAGE_SIZE);
	}
	/* REVISIT: not fully correct to assign nr_pages == sglen but video-buf
	 * is passing nr_pages for e.g. unmap_sg calls */
	vb->dma.nr_pages = vb->dma.sglen;
	vb->dma.direction = PCI_DMA_FROMDEVICE;

	return 0;

out:
	omap24xxcam_vbq_free_mmap_buffer(vb);
	return err;
}

static int omap24xxcam_vbq_alloc_mmap_buffers(struct videobuf_queue *vbq,
					      unsigned int count)
{
	int i, err = 0;
	struct omap24xxcam_fh *fh =
		container_of(vbq, struct omap24xxcam_fh, vbq);

	mutex_lock(&vbq->lock);

	for (i = 0; i < count; i++) {
		err = omap24xxcam_vbq_alloc_mmap_buffer(vbq->bufs[i]);
		if (err)
			goto out;
		dev_dbg(fh->cam->dev, "sglen is %d for buffer %d\n",
			vbq->bufs[i]->dma.sglen, i);
	}

	mutex_unlock(&vbq->lock);

	return 0;
out:
	while (i) {
		i--;
		omap24xxcam_vbq_free_mmap_buffer(vbq->bufs[i]);
	}

	mutex_unlock(&vbq->lock);

	return err;
}

/* This routine is called from interrupt context when a scatter-gather DMA
 * transfer of a videobuf_buffer completes.
 */
static void omap24xxcam_vbq_complete(struct omap24xxcam_sgdma *sgdma,
				     u32 csr, void *arg)
{
	struct omap24xxcam_device *cam =
		container_of(sgdma, struct omap24xxcam_device, sgdma);
	struct omap24xxcam_fh *fh = cam->streaming->private_data;
	struct videobuf_buffer *vb = (struct videobuf_buffer *)arg;
	const u32 csr_error = CAMDMA_CSR_MISALIGNED_ERR
		| CAMDMA_CSR_SUPERVISOR_ERR | CAMDMA_CSR_SECURE_ERR
		| CAMDMA_CSR_TRANS_ERR | CAMDMA_CSR_DROP;
	unsigned long flags;

	spin_lock_irqsave(&cam->core_enable_disable_lock, flags);
	if (--cam->sgdma_in_queue == 0)
		omap24xxcam_core_disable(cam);
	if (time_before(cam->next_reset_at, jiffies)
	    && !atomic_read(&cam->in_reset)) {
		spin_unlock_irqrestore(&cam->core_enable_disable_lock, flags);
		omap24xxcam_reset(cam);
	} else {
		spin_unlock_irqrestore(&cam->core_enable_disable_lock, flags);
	}

	do_gettimeofday(&vb->ts);
	vb->field_count = atomic_add_return(2, &fh->field_count);
	if (csr & csr_error) {
		vb->state = STATE_ERROR;
		if (!atomic_read(&cam->in_reset)) {
			dev_dbg(cam->dev, "resetting camera, csr 0x%x\n", csr);
			omap24xxcam_reset(cam);
		}
	} else
		vb->state = STATE_DONE;
	wake_up(&vb->done);
}

static void omap24xxcam_vbq_release(struct videobuf_queue *vbq,
				    struct videobuf_buffer *vb)
{
	struct videobuf_dmabuf *dma = &vb->dma;

	/* wait for buffer, especially to get out of the sgdma queue */
	videobuf_waiton(vb, 0, 0);
	if (vb->memory == V4L2_MEMORY_MMAP) {
		dma_unmap_sg(vbq->dev, dma->sglist, dma->sglen,
			     dma->direction);
		dma->direction = DMA_NONE;
	} else {
		videobuf_dma_unmap(vbq, &vb->dma);
		videobuf_dma_free(&vb->dma);
	}

	vb->state = STATE_NEEDS_INIT;
}

/* Limit the number of available kernel image capture buffers based on the
 * number requested, the currently selected image size, and the maximum
 * amount of memory permitted for kernel capture buffers.
 */
static int omap24xxcam_vbq_setup(struct videobuf_queue *vbq, unsigned int *cnt,
				 unsigned int *size)
{
	struct omap24xxcam_fh *fh = vbq->priv_data;

	if (*cnt <= 0)
		*cnt = VIDEO_MAX_FRAME;	/* supply a default number of buffers */

	if (*cnt > VIDEO_MAX_FRAME)
		*cnt = VIDEO_MAX_FRAME;

	*size = fh->pix.sizeimage;

	/* accessing fh->cam->capture_mem is ok, it's constant */
	while (*size * *cnt > fh->cam->capture_mem)
		(*cnt)--;

	return 0;
}

static int omap24xxcam_dma_iolock(struct videobuf_queue *vbq,
				  struct videobuf_dmabuf *dma)
{
	int err = 0;

	dma->direction = PCI_DMA_FROMDEVICE;
	if (!dma_map_sg(vbq->dev, dma->sglist, dma->sglen, dma->direction)) {
		kfree(dma->sglist);
		dma->sglist = NULL;
		dma->sglen = 0;
		err = -EIO;
	}

	return err;
}

static int omap24xxcam_vbq_prepare(struct videobuf_queue *vbq,
				   struct videobuf_buffer *vb,
				   enum v4l2_field field)
{
	struct omap24xxcam_fh *fh = vbq->priv_data;
	int err = 0;

	/* accessing pix here is okay since it's constant while
	 * streaming is on (and we only get called then). */

	if (vb->baddr) {
		/* This is a userspace buffer. */
		if (fh->pix.sizeimage > vb->bsize) {
			/* The buffer isn't big enough. */
			err = -EINVAL;
		} else
			vb->size = fh->pix.sizeimage;
	} else {
		if (vb->state != STATE_NEEDS_INIT) {
			/* We have a kernel bounce buffer that has already been
			 * allocated.
			 */
			if (fh->pix.sizeimage > vb->size) {
				/* The image size has been changed to a larger
				 * size since this buffer was allocated, so we
				 * need to free and reallocate it.
				 */
				omap24xxcam_vbq_release(vbq, vb);
				vb->size = fh->pix.sizeimage;
			}
		} else {
			/* We need to allocate a new kernel bounce buffer. */
			vb->size = fh->pix.sizeimage;
		}
	}

	if (err)
		return err;

	vb->width = fh->pix.width;
	vb->height = fh->pix.height;
	vb->field = field;

	if (vb->state == STATE_NEEDS_INIT) {
		if (vb->memory == V4L2_MEMORY_MMAP)
			/* we have built the scatter-gather list by ourself so
			 * do the scatter-gather mapping as well */
			err = omap24xxcam_dma_iolock(vbq, &vb->dma);
		else
			err = videobuf_iolock(vbq, vb, NULL);
	}

	if (!err)
		vb->state = STATE_PREPARED;
	else
		omap24xxcam_vbq_release(vbq, vb);

	return err;
}

static void omap24xxcam_vbq_queue(struct videobuf_queue *vbq,
				  struct videobuf_buffer *vb)
{
	struct omap24xxcam_fh *fh = vbq->priv_data;
	struct omap24xxcam_device *cam = fh->cam;
	enum videobuf_state state = vb->state;
	unsigned long flags;
	int err;

	/*
	 * FIXME: We're marking the buffer active since we have no
	 * pretty way of marking it active exactly when the
	 * scatter-gather transfer starts.
	 */
	vb->state = STATE_ACTIVE;

	err = omap24xxcam_sgdma_queue(&fh->cam->sgdma, vb->dma.sglist,
				      vb->dma.sglen, vb->size,
				      omap24xxcam_vbq_complete, vb);

	if (!err) {
		spin_lock_irqsave(&cam->core_enable_disable_lock, flags);
		if (++cam->sgdma_in_queue == 1
		    && !atomic_read(&cam->in_reset))
			omap24xxcam_core_enable(cam);
		spin_unlock_irqrestore(&cam->core_enable_disable_lock, flags);
	} else {
		/* Oops.  We're not supposed to get any errors here.  The only
		 * way we could get an error is if we ran out of scatter-gather
		 * DMA slots, but we are supposed to have at least as many
		 * scatter-gather DMA slots as video buffers so that can't
		 * happen.
		 */
		dev_err(cam->dev, "failed to queue a video buffer for dma!\n");
		dev_err(cam->dev, "likely a bug in the driver!\n");
		vb->state = state;
	}
}

static struct videobuf_queue_ops omap24xxcam_vbq_ops = {
	.buf_setup = omap24xxcam_vbq_setup,
	.buf_prepare = omap24xxcam_vbq_prepare,
	.buf_queue = omap24xxcam_vbq_queue,
	.buf_release = omap24xxcam_vbq_release,
};

/*
 *
 * OMAP main camera system
 *
 */

/*
 * Reset camera block to power-on state.
 */
static void omap24xxcam_poweron_reset(const struct omap24xxcam_device *cam)
{
	int max_loop = RESET_TIMEOUT_NS;

	/* Reset whole camera subsystem */
	omap24xxcam_reg_out(cam->mmio_base,
			    CAM_SYSCONFIG,
			    CAM_SYSCONFIG_SOFTRESET);

	/* Wait till it's finished */
	while (!(omap24xxcam_reg_in(cam->mmio_base, CAM_SYSSTATUS)
		 & CAM_SYSSTATUS_RESETDONE)
	       && --max_loop) {
		ndelay(1);
	}

	if (!(omap24xxcam_reg_in(cam->mmio_base, CAM_SYSSTATUS)
	      & CAM_SYSSTATUS_RESETDONE))
		dev_err(cam->dev, "camera soft reset timeout\n");
}

/*
 * (Re)initialise the camera block.
 */
static void omap24xxcam_hwinit(const struct omap24xxcam_device *cam)
{
	omap24xxcam_poweron_reset(cam);

	/* set the camera subsystem autoidle bit */
	omap24xxcam_reg_out(cam->mmio_base, CAM_SYSCONFIG,
			    CAM_SYSCONFIG_AUTOIDLE);

	/* set the camera MMU autoidle bit */
	omap24xxcam_reg_out(cam->mmio_base,
			    CAMMMU_REG_OFFSET + CAMMMU_SYSCONFIG,
			    CAMMMU_SYSCONFIG_AUTOIDLE);

	omap24xxcam_core_hwinit(cam);

	omap24xxcam_dma_hwinit(&cam->sgdma.dma);
}

/*
 * Callback for dma transfer stalling.
 */
static void omap24xxcam_stalled_dma_reset(unsigned long data)
{
	struct omap24xxcam_device *cam = (struct omap24xxcam_device *)data;

	if (!atomic_read(&cam->in_reset)) {
		dev_dbg(cam->dev, "dma stalled, resetting camera\n");
		omap24xxcam_reset(cam);
	}
}

/*
 * Stop capture. Mark we're doing a reset, stop DMA transfers and
 * core. (No new scatter-gather transfers will be queued whilst
 * in_reset is non-zero.)
 *
 * If omap24xxcam_capture_stop is called from several places at
 * once, only the first call will have an effect. Similarly, the last
 * call omap24xxcam_streaming_cont will have effect.
 *
 * Serialisation is ensured by using cam->core_enable_disable_lock.
 */
static void omap24xxcam_capture_stop(struct omap24xxcam_device *cam)
{
	unsigned long flags;

	spin_lock_irqsave(&cam->core_enable_disable_lock, flags);

	if (atomic_inc_return(&cam->in_reset) != 1) {
		spin_unlock_irqrestore(&cam->core_enable_disable_lock, flags);
		return;
	}

	omap24xxcam_core_disable(cam);

	spin_unlock_irqrestore(&cam->core_enable_disable_lock, flags);

	omap24xxcam_sgdma_sync(&cam->sgdma);
}

/*
 * Reset and continue streaming.
 *
 * Note: Resetting the camera FIFO via the CC_RST bit in the CC_CTRL
 * register is supposed to be sufficient to recover from a camera
 * interface error, but it doesn't seem to be enough. If we only do
 * that then subsequent image captures are out of sync by either one
 * or two times DMA_THRESHOLD bytes. Resetting and re-initializing the
 * entire camera subsystem prevents the problem with frame
 * synchronization.
 */
static void omap24xxcam_capture_cont(struct omap24xxcam_device *cam)
{
	unsigned long flags;
	struct omap24xxcam_fh *fh = cam->streaming->private_data;

	spin_lock_irqsave(&cam->core_enable_disable_lock, flags);

	if (atomic_read(&cam->in_reset) != 1)
		goto out;

	omap24xxcam_hwinit(cam);

	omap24xxcam_core_xclk_set(
		cam, cam->sdev->u.slave->vidioc_int_g_ext_clk(cam->sdev));

	omap24xxcam_sgdma_process(&cam->sgdma);

	if (cam->sgdma_in_queue)
		omap24xxcam_core_enable(cam);

	cam->next_reset_at = jiffies + RESET_INTERVAL;
	atomic_set(&cam->field_count_at_last_reset,
		   atomic_read(&fh->field_count) + 2);

out:
	atomic_dec(&cam->in_reset);
	spin_unlock_irqrestore(&cam->core_enable_disable_lock, flags);
}

static ssize_t
omap24xxcam_streaming_show(struct device *dev, struct device_attribute *attr,
		char *buf)
{
	struct omap24xxcam_device *cam = dev_get_drvdata(dev);

	return sprintf(buf, "%s\n", cam->streaming ?  "active" : "inactive");
}
static DEVICE_ATTR(streaming, S_IRUGO, omap24xxcam_streaming_show, NULL);

/*
 * Stop capture and restart it. I.e. reset the camera during use.
 */
static void omap24xxcam_reset(struct omap24xxcam_device *cam)
{
	omap24xxcam_capture_stop(cam);
	omap24xxcam_capture_cont(cam);
}

/*
 * The main interrupt handler.
 */
static irqreturn_t omap24xxcam_isr(int irq, void *arg)
{
	struct omap24xxcam_device *cam = (struct omap24xxcam_device *)arg;
	u32 irqstatus;
	unsigned int irqhandled = 0;

	irqstatus = omap24xxcam_reg_in(cam->mmio_base, CAM_IRQSTATUS);

	if (irqstatus &
	    (CAM_IRQSTATUS_DMA_IRQ2 | CAM_IRQSTATUS_DMA_IRQ1
	     | CAM_IRQSTATUS_DMA_IRQ0)) {
		omap24xxcam_dma_isr(&cam->sgdma.dma);
		irqhandled = 1;
	}
	if (irqstatus & CAM_IRQSTATUS_CC_IRQ) {
		omap24xxcam_core_isr(cam);
		irqhandled = 1;
	}
	if (irqstatus & CAM_IRQSTATUS_MMU_IRQ)
		dev_err(cam->dev, "unhandled camera MMU interrupt!\n");

	return IRQ_RETVAL(irqhandled);
}

/*
 *
 * Sensor handling.
 *
 */

/*
 * Initialise the sensor hardware.
 */
static int omap24xxcam_sensor_init(struct omap24xxcam_device *cam)
{
	int err = 0;
	struct v4l2_int_device *sdev = cam->sdev;

	omap24xxcam_clock_on(cam);
	omap24xxcam_core_xclk_set(cam,
				  sdev->u.slave->vidioc_int_g_ext_clk(sdev));

	/* power up sensor during sensor initialization */
	sdev->u.slave->power_set(sdev, 1);

	err = sdev->u.slave->dev_init(sdev);
	if (err) {
		dev_err(cam->dev, "cannot initialize sensor, error %d\n", err);
		/* Sensor init failed --- it's nonexistent to us! */
		cam->sdev = NULL;
		goto out;
	}

	dev_info(cam->dev, "sensor is %s\n", sdev->name);

out:
	omap24xxcam_core_xclk_set(cam, 0);
	omap24xxcam_clock_off(cam);

	sdev->u.slave->power_set(sdev, 0);

	return err;
}

static void omap24xxcam_sensor_exit(struct omap24xxcam_device *cam)
{
	if (cam->sdev)
		cam->sdev->u.slave->dev_exit(cam->sdev);
}

static void omap24xxcam_sensor_disable(struct omap24xxcam_device *cam)
{
	omap24xxcam_core_xclk_set(cam, 0);
	omap24xxcam_clock_off(cam);
	cam->sdev->u.slave->power_set(cam->sdev, 0);
}

/*
 * Power-up and configure camera sensor. It's ready for capturing now.
 */
static int omap24xxcam_sensor_enable(struct omap24xxcam_device *cam)
{
	int rval;

	omap24xxcam_clock_on(cam);

	omap24xxcam_core_xclk_set(
		cam, cam->sdev->u.slave->vidioc_int_g_ext_clk(cam->sdev));

	if ((rval = cam->sdev->u.slave->power_set(cam->sdev, 1)))
		goto out;

	if ((rval = cam->sdev->u.slave->vidioc_int_init(cam->sdev)))
		goto out;

	return 0;

out:
	omap24xxcam_sensor_disable(cam);
	return rval;
}

static void omap24xxcam_sensor_reset_work(struct work_struct *work)
{
	struct omap24xxcam_device *cam =
		container_of(work, struct omap24xxcam_device,
			     sensor_reset_work);

	if (atomic_read(&cam->reset_disable))
		return;

	down_read(&cam->dvfs_sem);

	omap24xxcam_capture_stop(cam);

	/* Can't reset it by vidioc_int_reset. */
	if (cam->sdev->u.slave->vidioc_int_reset(cam->sdev)) {
		omap24xxcam_sensor_disable(cam);
		omap24xxcam_sensor_enable(cam);
	}

	omap24xxcam_capture_cont(cam);

	up_read(&cam->dvfs_sem);
}

/*
 * Register sensor to the camera. Only works after the camera module
 * is initialised.
 */
static int omap24xxcam_sensor_attach(struct v4l2_int_device *ctl,
				     struct v4l2_int_device *s)
{
	struct omap24xxcam_device *cam = ctl->priv;
	int rval;

	if (cam->sdev != NULL)
		return -EBUSY;

	cam->sdev = s;

	rval = omap24xxcam_device_register(cam);
	if (rval)
		cam->sdev = NULL;

	return rval;
}

static void omap24xxcam_sensor_detach(struct v4l2_int_device *ctl)
{
	struct omap24xxcam_device *cam = ctl->priv;

	omap24xxcam_device_unregister(cam);

	cam->sdev = NULL;
}

static struct v4l2_int_master omap24xxcam_master = {
	.attach = &omap24xxcam_sensor_attach,
	.detach = &omap24xxcam_sensor_detach,
};

static struct v4l2_int_device omap24xxcam = {
	.module = THIS_MODULE,
	.name = CAM_NAME,
	.type = v4l2_int_type_master,
	.u = { .master = &omap24xxcam_master },
};

/*
 *
 * IOCTL interface.
 *
 */

static int vidioc_querycap(struct file *file, void *fh,
			   struct v4l2_capability *cap)
{
	struct omap24xxcam_fh *ofh = fh;
	struct omap24xxcam_device *cam = ofh->cam;

	strlcpy(cap->driver, CAM_NAME, sizeof(cap->driver));
	strlcpy(cap->card, cam->vfd->name, sizeof(cap->card));
	cap->version = KERNEL_VERSION(0, 0, 0);
	cap->capabilities = V4L2_CAP_VIDEO_CAPTURE | V4L2_CAP_STREAMING;

	return 0;
}

static int vidioc_enum_fmt_cap(struct file *file, void *fh,
			       struct v4l2_fmtdesc *f)
{
	struct omap24xxcam_fh *ofh = fh;
	struct omap24xxcam_device *cam = ofh->cam;

	return cam->sdev->u.slave->vidioc_int_enum_fmt_cap(cam->sdev, f);
}

static int vidioc_g_fmt_cap(struct file *file, void *fh,
			    struct v4l2_format *f)
{
	struct omap24xxcam_fh *ofh = fh;
	struct omap24xxcam_device *cam = ofh->cam;
	int rval;

	mutex_lock(&cam->mutex);
	rval = cam->sdev->u.slave->vidioc_int_g_fmt_cap(cam->sdev, f);
	mutex_unlock(&cam->mutex);

	return rval;
}

static int vidioc_s_fmt_cap(struct file *file, void *fh,
			    struct v4l2_format *f)
{
	struct omap24xxcam_fh *ofh = fh;
	struct omap24xxcam_device *cam = ofh->cam;
	int rval;

	down_read(&cam->dvfs_sem);
	mutex_lock(&cam->mutex);
	if (cam->streaming) {
		rval = -EBUSY;
		goto out;
	}

	rval = cam->sdev->u.slave->vidioc_int_s_fmt_cap(cam->sdev, f);

out:
	mutex_unlock(&cam->mutex);
	up_read(&cam->dvfs_sem);

	if (!rval) {
		mutex_lock(&ofh->vbq.lock);
		ofh->pix = f->fmt.pix;
		mutex_unlock(&ofh->vbq.lock);
	}

	memset(f, 0, sizeof(*f));
	vidioc_g_fmt_cap(file, fh, f);

	return rval;
}

static int vidioc_try_fmt_cap(struct file *file, void *fh,
			      struct v4l2_format *f)
{
	struct omap24xxcam_fh *ofh = fh;
	struct omap24xxcam_device *cam = ofh->cam;
	int rval;

	mutex_lock(&cam->mutex);
	rval = cam->sdev->u.slave->vidioc_int_try_fmt_cap(cam->sdev, f);
	mutex_unlock(&cam->mutex);

	return rval;
}

static int vidioc_reqbufs(struct file *file, void *fh,
			  struct v4l2_requestbuffers *b)
{
	struct omap24xxcam_fh *ofh = fh;
	struct omap24xxcam_device *cam = ofh->cam;
	int rval;

	mutex_lock(&cam->mutex);
	if (cam->streaming) {
		mutex_unlock(&cam->mutex);
		return -EBUSY;
	}

	omap24xxcam_vbq_free_mmap_buffers(&ofh->vbq);
	mutex_unlock(&cam->mutex);

	rval = videobuf_reqbufs(&ofh->vbq, b);

	/* Either videobuf_reqbufs failed or the buffers are not
	 * memory-mapped (which would need special attention). */
	if (rval || b->memory != V4L2_MEMORY_MMAP)
		goto out;

	rval = omap24xxcam_vbq_alloc_mmap_buffers(&ofh->vbq, b->count);
	if (rval)
		omap24xxcam_vbq_free_mmap_buffers(&ofh->vbq);

out:
	return rval;
}

static int vidioc_querybuf(struct file *file, void *fh,
			   struct v4l2_buffer *b)
{
	struct omap24xxcam_fh *ofh = fh;

	return videobuf_querybuf(&ofh->vbq, b);
}

static int vidioc_qbuf(struct file *file, void *fh, struct v4l2_buffer *b)
{
	struct omap24xxcam_fh *ofh = fh;

	return videobuf_qbuf(&ofh->vbq, b);
}

static int vidioc_dqbuf(struct file *file, void *fh, struct v4l2_buffer *b)
{
	struct omap24xxcam_fh *ofh = fh;
	struct omap24xxcam_device *cam = ofh->cam;
	struct videobuf_buffer *vb;
	int rval;

	rval = videobuf_dqbuf(&ofh->vbq, b, file->f_flags & O_NONBLOCK);
	if (rval)
		goto out;

	vb = ofh->vbq.bufs[b->index];

	/* Don't test frames immediately after reset. */
	if ((atomic_read(&cam->field_count_at_last_reset) + 2 > vb->field_count
	     || atomic_read(&cam->field_count_at_last_reset) < vb->field_count)
	    && cam->sdev->u.slave->vidioc_int_g_needs_reset) {
		mutex_lock(&cam->mutex);
		rval = cam->sdev->u.slave->vidioc_int_g_needs_reset(
			cam->sdev, (void *)vb->baddr);
		mutex_unlock(&cam->mutex);
		if (rval)
			schedule_work(&cam->sensor_reset_work);
	}
out:
	/* This is a hack. User space won't get the index of this
	 * buffer and does not want to requeue it so we requeue it
	 * here. */
	if (rval == -EIO)
		videobuf_qbuf(&ofh->vbq, b);

	return rval;
}

static int vidioc_streamon(struct file *file, void *fh, enum v4l2_buf_type i)
{
	struct omap24xxcam_fh *ofh = fh;
	struct omap24xxcam_device *cam = ofh->cam;
	int rval;

	down_read(&cam->dvfs_sem);
	mutex_lock(&cam->mutex);
	if (cam->streaming) {
		mutex_unlock(&cam->mutex);
		up_read(&cam->dvfs_sem);
		return -EBUSY;
	}

	if (!(rval = videobuf_streamon(&ofh->vbq))) {
		cam->streaming = file;
		sysfs_notify(&cam->dev->kobj, NULL, "streaming");
	}

	mutex_unlock(&cam->mutex);
	up_read(&cam->dvfs_sem);

	return rval;
}

static int vidioc_streamoff(struct file *file, void *fh, enum v4l2_buf_type i)
{
	struct omap24xxcam_fh *ofh = fh;
	struct omap24xxcam_device *cam = ofh->cam;
	struct videobuf_queue *q = &ofh->vbq;
	int rval;

	atomic_inc(&cam->reset_disable);

	flush_scheduled_work();

	if (!(rval = videobuf_streamoff(q))) {
		down_read(&cam->dvfs_sem);
		mutex_lock(&cam->mutex);
		cam->streaming = NULL;
		mutex_unlock(&cam->mutex);
		up_read(&cam->dvfs_sem);
		sysfs_notify(&cam->dev->kobj, NULL, "streaming");
	}

	atomic_dec(&cam->reset_disable);

	return rval;
}

static int vidioc_enum_input(struct file *file, void *fh,
			     struct v4l2_input *inp)
{
	if (inp->index > 0)
		return -EINVAL;

	strlcpy(inp->name, "camera", sizeof(inp->name));
	inp->type = V4L2_INPUT_TYPE_CAMERA;

	return 0;
}

static int vidioc_g_input(struct file *file, void *fh, unsigned int *i)
{
	*i = 0;

	return 0;
}

static int vidioc_s_input(struct file *file, void *fh, unsigned int i)
{
	if (i > 0)
		return -EINVAL;

	return 0;
}

static int vidioc_queryctrl(struct file *file, void *fh,
			    struct v4l2_queryctrl *a)
{
	struct omap24xxcam_fh *ofh = fh;
	struct omap24xxcam_device *cam = ofh->cam;

	return cam->sdev->u.slave->vidioc_int_queryctrl(cam->sdev, a);
}

static int vidioc_g_ctrl(struct file *file, void *fh,
			 struct v4l2_control *a)
{
	struct omap24xxcam_fh *ofh = fh;
	struct omap24xxcam_device *cam = ofh->cam;
	int rval;

	mutex_lock(&cam->mutex);
	rval = cam->sdev->u.slave->vidioc_int_g_ctrl(cam->sdev, a);
	mutex_unlock(&cam->mutex);

	return rval;
}

static int vidioc_s_ctrl(struct file *file, void *fh,
			 struct v4l2_control *a)
{
	struct omap24xxcam_fh *ofh = fh;
	struct omap24xxcam_device *cam = ofh->cam;
	int rval;

	mutex_lock(&cam->mutex);
	rval = cam->sdev->u.slave->vidioc_int_s_ctrl(cam->sdev, a);
	mutex_unlock(&cam->mutex);

	return rval;
}

static int vidioc_g_parm(struct file *file, void *fh,
			 struct v4l2_streamparm *a) {
	struct omap24xxcam_fh *ofh = fh;
	struct omap24xxcam_device *cam = ofh->cam;
	int rval;

	if (a->type != V4L2_BUF_TYPE_VIDEO_CAPTURE)
		return -EINVAL;

	mutex_lock(&cam->mutex);
	rval = cam->sdev->u.slave->vidioc_int_g_parm(cam->sdev, a);
	mutex_unlock(&cam->mutex);

	return rval;
}

static int vidioc_s_parm(struct file *file, void *fh,
			 struct v4l2_streamparm *a)
{
	struct omap24xxcam_fh *ofh = fh;
	struct omap24xxcam_device *cam = ofh->cam;
	int rval;
	u32 xclk, xclk_old;

	if (a->type != V4L2_BUF_TYPE_VIDEO_CAPTURE)
		return -EINVAL;

	down_read(&cam->dvfs_sem);
	mutex_lock(&cam->mutex);
	if (cam->streaming) {
		rval = -EBUSY;
		goto out;
	}

	xclk_old = cam->sdev->u.slave->vidioc_int_g_ext_clk(cam->sdev);

	rval = cam->sdev->u.slave->vidioc_int_s_parm(cam->sdev, a);

	xclk = omap24xxcam_core_xclk_set(
		cam, cam->sdev->u.slave->vidioc_int_g_ext_clk(cam->sdev));

	/*
	 * FIXME: if setting xclk fails, streaming parameters are not
	 * reverted. Also bad xclk is used for a short while.
	 */
	if (cam->sdev->u.slave->vidioc_int_s_ext_clk(cam->sdev, xclk)) {
		rval = -EINVAL;
		omap24xxcam_core_xclk_set(cam, xclk_old);
	}

out:
	mutex_unlock(&cam->mutex);
	up_read(&cam->dvfs_sem);

	return rval;
}

/*
 *
 * File operations.
 *
 */

static unsigned int omap24xxcam_poll(struct file *file,
				     struct poll_table_struct *wait)
{
	struct omap24xxcam_fh *fh = file->private_data;
	struct omap24xxcam_device *cam = fh->cam;
	struct videobuf_buffer *vb;

	mutex_lock(&cam->mutex);
	if (cam->streaming != file) {
		mutex_unlock(&cam->mutex);
		return POLLERR;
	}
	mutex_unlock(&cam->mutex);

	mutex_lock(&fh->vbq.lock);
	if (list_empty(&fh->vbq.stream)) {
		mutex_unlock(&fh->vbq.lock);
		return POLLERR;
	}
	vb = list_entry(fh->vbq.stream.next, struct videobuf_buffer, stream);
	mutex_unlock(&fh->vbq.lock);

	poll_wait(file, &vb->done, wait);

	if (vb->state == STATE_DONE || vb->state == STATE_ERROR)
		return POLLIN | POLLRDNORM;

	return 0;
}

static int omap24xxcam_mmap_buffers(struct file *file,
				    struct vm_area_struct *vma)
{
	struct omap24xxcam_fh *fh = file->private_data;
	struct omap24xxcam_device *cam = fh->cam;
	struct videobuf_queue *vbq = &fh->vbq;
	struct videobuf_buffer *vb;
	unsigned int first, last, size, i, j;
	int err = 0;

	mutex_lock(&cam->mutex);
	if (cam->streaming) {
		mutex_unlock(&cam->mutex);
		return -EBUSY;
	}
	mutex_unlock(&cam->mutex);
	mutex_lock(&vbq->lock);

	/* look for first buffer to map */
	for (first = 0; first < VIDEO_MAX_FRAME; first++) {
		if (NULL == vbq->bufs[first])
			continue;
		if (V4L2_MEMORY_MMAP != vbq->bufs[first]->memory)
			continue;
		if (vbq->bufs[first]->boff == (vma->vm_pgoff << PAGE_SHIFT))
			break;
	}

	/* look for last buffer to map */
	for (size = 0, last = first; last < VIDEO_MAX_FRAME; last++) {
		if (NULL == vbq->bufs[last])
			continue;
		if (V4L2_MEMORY_MMAP != vbq->bufs[last]->memory)
			continue;
		size += vbq->bufs[last]->bsize;
		if (size == (vma->vm_end - vma->vm_start))
			break;
	}

	size = 0;
	for (i = first; i <= last; i++) {
		vb = vbq->bufs[i];
		for (j = 0; j < vb->dma.sglen; j++) {
			err = remap_pfn_range(
				vma, vma->vm_start + size,
				page_to_pfn(vb->dma.sglist[j].page),
				vb->dma.sglist[j].length, vma->vm_page_prot);
			if (err)
				goto out;
			size += vb->dma.sglist[j].length;
		}
	}

out:
	mutex_unlock(&vbq->lock);

	return err;
}

static int omap24xxcam_mmap(struct file *file, struct vm_area_struct *vma)
{
	struct omap24xxcam_fh *fh = file->private_data;
	int rval;

	/* let the video-buf mapper check arguments and set-up structures */
	rval = videobuf_mmap_mapper(&fh->vbq, vma);
	if (rval)
		return rval;

	vma->vm_page_prot = pgprot_noncached(vma->vm_page_prot);

	/* do mapping to our allocated buffers */
	rval = omap24xxcam_mmap_buffers(file, vma);
	/* In case of error, free vma->vm_private_data allocated by
	 * videobuf_mmap_mapper. */
	if (rval)
		kfree(vma->vm_private_data);

	return rval;
}

static int omap24xxcam_open(struct inode *inode, struct file *file)
{
	int minor = iminor(inode);
	struct omap24xxcam_device *cam = omap24xxcam.priv;
	struct omap24xxcam_fh *fh;
	struct v4l2_format format;

	if (!cam || !cam->vfd || (cam->vfd->minor != minor))
		return -ENODEV;

	if ((fh = kzalloc(sizeof(*fh), GFP_KERNEL)) == NULL)
		return -ENOMEM;

	mutex_lock(&cam->mutex);
	if (cam->sdev == NULL || !try_module_get(cam->sdev->module))
		goto out_try_module_get;
	mutex_unlock(&cam->mutex);

	down_read(&cam->dvfs_sem);
	mutex_lock(&cam->mutex);
	if (atomic_inc_return(&cam->users) == 1) {
		omap24xxcam_hwinit(cam);
		if (omap24xxcam_sensor_enable(cam))
			goto out_omap24xxcam_sensor_enable;
	}
	mutex_unlock(&cam->mutex);
	up_read(&cam->dvfs_sem);

	fh->cam = cam;
	mutex_lock(&cam->mutex);
	cam->sdev->u.slave->vidioc_int_g_fmt_cap(cam->sdev, &format);
	mutex_unlock(&cam->mutex);
	/* FIXME: how about fh->pix when there are more users? */
	fh->pix = format.fmt.pix;

	file->private_data = fh;

	spin_lock_init(&fh->vbq_lock);

	videobuf_queue_init(&fh->vbq, &omap24xxcam_vbq_ops, NULL,
			    &fh->vbq_lock, V4L2_BUF_TYPE_VIDEO_CAPTURE,
			    V4L2_FIELD_NONE,
			    sizeof(struct videobuf_buffer), fh);

	return 0;

out_omap24xxcam_sensor_enable:
	omap24xxcam_poweron_reset(cam);
	up_read(&cam->dvfs_sem);
	module_put(cam->sdev->module);

out_try_module_get:
	mutex_unlock(&cam->mutex);
	kfree(fh);

	return -ENODEV;
}

static int omap24xxcam_release(struct inode *inode, struct file *file)
{
	struct omap24xxcam_fh *fh = file->private_data;
	struct omap24xxcam_device *cam = fh->cam;

	atomic_inc(&cam->reset_disable);

	flush_scheduled_work();

	mutex_lock(&cam->mutex);
	/* stop streaming capture */
	if (cam->streaming == file) {
		cam->streaming = NULL;
		mutex_unlock(&cam->mutex);
		videobuf_streamoff(&fh->vbq);
		sysfs_notify(&cam->dev->kobj, NULL, "streaming");
	} else {
		mutex_unlock(&cam->mutex);
	}

	atomic_dec(&cam->reset_disable);

	omap24xxcam_vbq_free_mmap_buffers(&fh->vbq);

	/* Make sure the reset work we might have scheduled is not
	 * pending! It may be run *only* if we have users. (And it may
	 * not be scheduled anymore since streaming is already
	 * disabled.)
	 */
	flush_scheduled_work();

	down_read(&cam->dvfs_sem);
	mutex_lock(&cam->mutex);
	if (atomic_dec_return(&cam->users) == 0) {
		omap24xxcam_sensor_disable(cam);
		omap24xxcam_poweron_reset(cam);
	}
	mutex_unlock(&cam->mutex);
	up_read(&cam->dvfs_sem);

	file->private_data = NULL;

	module_put(cam->sdev->module);
	kfree(fh);

	return 0;
}

static struct file_operations omap24xxcam_fops = {
	.owner = THIS_MODULE,
	.llseek = no_llseek,
	.ioctl = video_ioctl2,
	.poll = omap24xxcam_poll,
	.mmap = omap24xxcam_mmap,
	.open = omap24xxcam_open,
	.release = omap24xxcam_release,
};

/*
 *
 * Power management.
 *
 */

#ifdef CONFIG_PM
static int omap24xxcam_suspend(struct platform_device *pdev, pm_message_t state)
{
	struct omap24xxcam_device *cam = platform_get_drvdata(pdev);

	if (atomic_read(&cam->users) == 0)
		return 0;

	if (!atomic_read(&cam->reset_disable))
		omap24xxcam_capture_stop(cam);

	omap24xxcam_sensor_disable(cam);
	omap24xxcam_poweron_reset(cam);

	return 0;
}

static int omap24xxcam_resume(struct platform_device *pdev)
{
	struct omap24xxcam_device *cam = platform_get_drvdata(pdev);

	if (atomic_read(&cam->users) == 0)
		return 0;

	omap24xxcam_hwinit(cam);
	omap24xxcam_sensor_enable(cam);

	if (!atomic_read(&cam->reset_disable))
		omap24xxcam_capture_cont(cam);

	return 0;
}
#endif /* CONFIG_PM */

#ifdef CONFIG_MACH_OMAP2420_DVFS
static int omap24xxcam_dvfs_callback(struct notifier_block *nb,
				     unsigned long state, void *p)
{
	struct omap24xxcam_device *cam =
		container_of(DVFS_NOTIF(nb), struct omap24xxcam_device,
			     dvfs_notifier);

	switch (state) {
	case DVFS_PRE_NOTIFICATION:
		down_write(&cam->dvfs_sem);
		mutex_lock(&cam->mutex);
		/* cam->users musn't change during dvfs state transtion! */
		if (cam->streaming)
			omap24xxcam_capture_stop(cam);
		mutex_unlock(&cam->mutex);
		break;
	case DVFS_POST_NOTIFICATION:
		mutex_lock(&cam->mutex);
		if (cam->streaming)
			omap24xxcam_capture_cont(cam);
		mutex_unlock(&cam->mutex);
		up_write(&cam->dvfs_sem);
		break;
	}

	dvfs_client_notification_cb(DVFS_NOTIF(nb));

	return NOTIFY_DONE;
}
#else
#define omap24xxcam_dvfs_callback NULL
#endif

/*
 *
 * Camera device (i.e. /dev/video).
 *
 */

static int omap24xxcam_device_register(struct omap24xxcam_device *cam)
{
	struct video_device *vfd;
	int rval;

	dvfs_register_notifier(&cam->dvfs_notifier);

	if (device_create_file(cam->dev, &dev_attr_streaming) != 0) {
		dev_err(cam->dev, "could not register sysfs entry\n");
		rval = -EBUSY;
		goto err;
	}

	/* initialize the video_device struct */
	vfd = cam->vfd = video_device_alloc();
	if (!vfd) {
		dev_err(cam->dev, "could not allocate video device struct\n");
		rval = -ENOMEM;
		goto err;
	}
	vfd->release = video_device_release;

	vfd->dev = cam->dev;

	strlcpy(vfd->name, CAM_NAME, sizeof(vfd->name));
	vfd->type = VID_TYPE_CAPTURE | VID_TYPE_CHROMAKEY;
	/* need to register for a VID_HARDWARE_* ID in videodev.h */
	vfd->hardware = 0;
	vfd->fops = &omap24xxcam_fops;
	/* FIXME: need to use the full v4l2 API */
	vfd->priv = cam;
	vfd->minor = -1;

	vfd->vidioc_querycap = &vidioc_querycap;
	vfd->vidioc_enum_fmt_cap = &vidioc_enum_fmt_cap;
	vfd->vidioc_g_fmt_cap = &vidioc_g_fmt_cap;
	vfd->vidioc_s_fmt_cap = &vidioc_s_fmt_cap;
	vfd->vidioc_try_fmt_cap = &vidioc_try_fmt_cap;
	vfd->vidioc_reqbufs = &vidioc_reqbufs;
	vfd->vidioc_querybuf = &vidioc_querybuf;
	vfd->vidioc_qbuf = &vidioc_qbuf;
	vfd->vidioc_dqbuf = &vidioc_dqbuf;
	vfd->vidioc_streamon = &vidioc_streamon;
	vfd->vidioc_streamoff = &vidioc_streamoff;
	vfd->vidioc_enum_input = &vidioc_enum_input;
	vfd->vidioc_g_input = &vidioc_g_input;
	vfd->vidioc_s_input = &vidioc_s_input;
	vfd->vidioc_queryctrl = &vidioc_queryctrl;
	vfd->vidioc_g_ctrl = &vidioc_g_ctrl;
	vfd->vidioc_s_ctrl = &vidioc_s_ctrl;
	vfd->vidioc_g_parm = &vidioc_g_parm;
	vfd->vidioc_s_parm = &vidioc_s_parm;

	down_read(&cam->dvfs_sem);

	omap24xxcam_hwinit(cam);

	if ((rval = omap24xxcam_sensor_init(cam)))
		goto err_dvfs;

	if (video_register_device(vfd, VFL_TYPE_GRABBER, video_nr) < 0) {
		dev_err(cam->dev, "could not register V4L device\n");
		vfd->minor = -1;
		rval = -EBUSY;
		goto err_dvfs;
	}

	omap24xxcam_poweron_reset(cam);

	up_read(&cam->dvfs_sem);

	dev_info(cam->dev, "registered device video%d [v4l2]\n", vfd->minor);

	return 0;

err_dvfs:
	up_read(&cam->dvfs_sem);

err:
	omap24xxcam_device_unregister(cam);

	return rval;
}

static void omap24xxcam_device_unregister(struct omap24xxcam_device *cam)
{
	omap24xxcam_sensor_exit(cam);

	dvfs_unregister_notifier(&cam->dvfs_notifier);

	if (cam->vfd) {
		if (cam->vfd->minor == -1) {
			/* The device was never registered, so release the
			 * video_device struct directly.
			 */
			video_device_release(cam->vfd);
		} else {
			/* The unregister function will release the video_device
			 * struct as well as unregistering it.
			 */
			video_unregister_device(cam->vfd);
		}
		cam->vfd = NULL;
	}

	device_remove_file(cam->dev, &dev_attr_streaming);
}

/*
 *
 * Driver initialisation and deinitialisation.
 *
 */

static int omap24xxcam_probe(struct platform_device *pdev)
{
	struct omap24xxcam_device *cam;
	struct resource *mem;
	int irq;

	cam = kzalloc(sizeof(*cam), GFP_KERNEL);
	if (!cam) {
		dev_err(&pdev->dev, "could not allocate memory\n");
		goto err;
	}

	platform_set_drvdata(pdev, cam);

	cam->dev = &pdev->dev;

	/* Impose a lower limit on the amount of memory allocated for capture.
	 * We require at least enough memory to double-buffer QVGA (300KB).
	 */
	if (capture_mem < 320 * 240 * 2 * 2)
		capture_mem = 320 * 240 * 2 * 2;
	cam->capture_mem = capture_mem;

	/* request the mem region for the camera registers */
	mem = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!mem) {
		dev_err(cam->dev, "no mem resource?\n");
		goto err;
	}
	if (!request_mem_region(mem->start, (mem->end - mem->start) + 1,
				pdev->name)) {
		dev_err(cam->dev,
			"cannot reserve camera register I/O region\n");
		goto err;
	}
	cam->mmio_base_phys = mem->start;
	cam->mmio_size = (mem->end - mem->start) + 1;

	/* map the region */
	cam->mmio_base = (unsigned long)
		ioremap_nocache(cam->mmio_base_phys, cam->mmio_size);
	if (!cam->mmio_base) {
		dev_err(cam->dev, "cannot map camera register I/O region\n");
		goto err;
	}

	irq = platform_get_irq(pdev, 0);
	if (irq <= 0) {
		dev_err(cam->dev, "no irq for camera?\n");
		goto err;
	}

	/* install the interrupt service routine */
	if (request_irq(irq, omap24xxcam_isr, 0, CAM_NAME, cam)) {
		dev_err(cam->dev,
			"could not install interrupt service routine\n");
		goto err;
	}
	cam->irq = irq;

	if (omap24xxcam_clock_get(cam))
		goto err;

	INIT_WORK(&cam->sensor_reset_work, omap24xxcam_sensor_reset_work);

	dvfs_notif_set(&cam->dvfs_notifier, kobject_name(&cam->dev->kobj),
		       omap24xxcam_dvfs_callback, DVFS_CHAINED_NOTIF, 0);
	init_rwsem(&cam->dvfs_sem);

	mutex_init(&cam->mutex);
	spin_lock_init(&cam->core_enable_disable_lock);

	omap24xxcam_sgdma_init(&cam->sgdma,
			       cam->mmio_base + CAMDMA_REG_OFFSET,
			       &omap24xxcam_stalled_dma_reset,
			       (unsigned long)cam);

	omap24xxcam.priv = cam;

	if (v4l2_int_device_register(&omap24xxcam))
		goto err;

	return 0;

err:
	omap24xxcam_remove(pdev);
	return -ENODEV;
}

static int omap24xxcam_remove(struct platform_device *pdev)
{
	struct omap24xxcam_device *cam = platform_get_drvdata(pdev);

	if (!cam)
		return 0;

	if (omap24xxcam.priv != NULL)
		v4l2_int_device_unregister(&omap24xxcam);
	omap24xxcam.priv = NULL;

	omap24xxcam_device_unregister(cam);
	omap24xxcam_clock_put(cam);

	if (cam->irq) {
		free_irq(cam->irq, cam);
		cam->irq = 0;
	}

	if (cam->mmio_base) {
		iounmap((void *)cam->mmio_base);
		cam->mmio_base = 0;
	}

	if (cam->mmio_base_phys) {
		release_mem_region(cam->mmio_base_phys, cam->mmio_size);
		cam->mmio_base_phys = 0;
	}

	kfree(cam);

	return 0;
}

static struct platform_driver omap24xxcam_driver = {
	.probe = omap24xxcam_probe,
	.remove = omap24xxcam_remove,
#ifdef CONFIG_PM
	.suspend = omap24xxcam_suspend,
	.resume = omap24xxcam_resume,
#endif
	.driver = {
		.name = CAM_NAME,
	},
};

/*
 *
 * Module initialisation and deinitialisation
 *
 */

static int __init omap24xxcam_init(void)
{
	return platform_driver_register(&omap24xxcam_driver);
}

static void __exit omap24xxcam_cleanup(void)
{
	platform_driver_unregister(&omap24xxcam_driver);
}

MODULE_AUTHOR("Sakari Ailus <sakari.ailus@nokia.com>");
MODULE_DESCRIPTION("OMAP24xx Video for Linux camera driver");
MODULE_LICENSE("GPL");
module_param(video_nr, int, 0);
MODULE_PARM_DESC(video_nr,
		 "Minor number for video device (-1 ==> auto assign)");
module_param(capture_mem, int, 0);
MODULE_PARM_DESC(capture_mem,
		 "Maximum amount of memory for capture buffers (default 4800kiB)");

module_init(omap24xxcam_init);
module_exit(omap24xxcam_cleanup);
