/*
 * Cryptographic API for LZO compression.
 *
 * Copyright (C) 2007 Nokia Corporation. All rights reserved.
 *
 * Author: Richard Purdie <rpurdie@openedhand.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <linux/init.h>
#include <linux/module.h>
#include <linux/crypto.h>
#include <linux/vmalloc.h>
#include <linux/interrupt.h>
#include <linux/mm.h>
#include <linux/lzo.h>

struct lzo_ctx {
	void *lzo_mem;
};

static int lzo_init(struct crypto_tfm *tfm)
{
	struct lzo_ctx *ctx = crypto_tfm_ctx(tfm);

	ctx->lzo_mem = vmalloc(LZO1X_MEM_COMPRESS);

	if (!ctx->lzo_mem) {
		vfree(ctx->lzo_mem);
		return -ENOMEM;
	}

	return 0;
}

static void lzo_exit(struct crypto_tfm *tfm)
{
	struct lzo_ctx *ctx = crypto_tfm_ctx(tfm);

	vfree(ctx->lzo_mem);
}

static int lzo_compress(struct crypto_tfm *tfm, const u8 *src,
			    unsigned int slen, u8 *dst, unsigned int *dlen)
{
	struct lzo_ctx *ctx = crypto_tfm_ctx(tfm);
	size_t compress_size;
	int ret;

	/* Check if enough space in dst buffer for worst case expansion */
	if (*dlen < lzo1x_worst_compress(slen))
		return -EINVAL;

	ret = lzo1x_1_compress(src, slen, dst, &compress_size, ctx->lzo_mem);

	if (ret != LZO_E_OK)
		return -EINVAL;

	*dlen = compress_size;

	return 0;
}

static int lzo_decompress(struct crypto_tfm *tfm, const u8 *src,
			      unsigned int slen, u8 *dst, unsigned int *dlen)
{
	int ret;
	size_t dest_len = *dlen;

	ret = lzo1x_decompress_safe(src, slen, dst, &dest_len);

	*dlen = dest_len;

	if (ret != LZO_E_OK)
		return -EINVAL;

	return 0;
}

static struct crypto_alg alg = {
	.cra_name		= "lzo1x",
	.cra_flags		= CRYPTO_ALG_TYPE_COMPRESS,
	.cra_ctxsize		= sizeof(struct lzo_ctx),
	.cra_module		= THIS_MODULE,
	.cra_list		= LIST_HEAD_INIT(alg.cra_list),
	.cra_init		= lzo_init,
	.cra_exit		= lzo_exit,
	.cra_u			= { .compress = {
	.coa_compress 		= lzo_compress,
	.coa_decompress  	= lzo_decompress } }
};

static int __init init(void)
{
	return crypto_register_alg(&alg);
}

static void __exit fini(void)
{
	crypto_unregister_alg(&alg);
}

module_init(init);
module_exit(fini);

MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("LZO Compression Algorithm");
MODULE_AUTHOR("Richard Purdie <rpurdie@openedhand.com>");

