/*
 * Copyright (C) 2006, Nomad Global Solutions, Inc.
 * Author: Eugeny S. Mints <eugeny.mints@gmail.com>
 *
 * Copyright (C) 2007 Nokia Corporation.
 * Rewritten by: Igor Stoppa <igor.stoppa@nokia.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

#include <linux/errno.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/kmod.h>
#include <linux/module.h>
#include <linux/stat.h>
#include <linux/string.h>
#include <linux/device.h>
#include <linux/sched.h>
#include <linux/param.h>
#include <linux/freezer.h>
#include <asm/arch/dvfs_notif.h>

/* FIXME: Eventually it should it be 0 */
#define MIN_NOTIFICATIONS_TIMEOUT 100

static LIST_HEAD(drivers_list);
static BLOCKING_NOTIFIER_HEAD(notifier_list);
static DECLARE_WAIT_QUEUE_HEAD(queue);
static DEFINE_MUTEX(dvfs_notif_mutex);
static atomic_t missing_cb;
static atomic_t failing_drivers;
static struct completion nf_completion;
static int dvfs_event;
static int clients_count;
static int target_state;
static unsigned long timeout;

static inline int set_starting_state(int event) { return event - 1;}

static void update_target_state(int event)
{
	struct dvfs_notif *curs;

	target_state = event;
	list_for_each_entry(curs, &drivers_list, list) {
		curs->err = 0;
		curs->state = set_starting_state(event);
	}
}

static void update_timeout(void)
{
	struct dvfs_notif *curs;
	unsigned long tmp;

	timeout = msecs_to_jiffies(MIN_NOTIFICATIONS_TIMEOUT);
	list_for_each_entry(curs, &drivers_list, list) {
		tmp = msecs_to_jiffies(curs->timeout);
		timeout = timeout >= tmp ? timeout : tmp;
	}
}

static void list_erratic_drivers(void)
{
	struct dvfs_notif *curs;

	list_for_each_entry(curs, &drivers_list, list) {
		if (curs->state != target_state)
			printk(KERN_ERR "DVFS: driver %s - %s\n",
			       curs->err ? "error" : "timeout", curs->name);
	}
}

int dvfs_run_notifications(enum dvfs_notifications event)
{
	int ret = 0;

	if (event == DVFS_PRE_NOTIFICATION)
		mutex_lock(&dvfs_notif_mutex);

	dvfs_event = event;

	init_completion(&nf_completion);
	update_target_state(event);
	atomic_set(&missing_cb, clients_count);
	atomic_set(&failing_drivers, 0);

	preempt_disable();
	wake_up_all(&queue);
	blocking_notifier_call_chain(&notifier_list, dvfs_event, NULL);
	preempt_enable();

	wait_for_completion_timeout(&nf_completion, timeout);

	if (atomic_read(&missing_cb) || atomic_read(&failing_drivers)) {
		if (atomic_read(&missing_cb))
			printk(KERN_ERR "DVFS: Timeout!\n");
		if (atomic_read(&failing_drivers))
			printk(KERN_ERR
			       "DVFS: Driver(s) failing notification!\n");
		list_erratic_drivers();
		ret = -EIO;
	}

	if (event == DVFS_POST_NOTIFICATION)
		mutex_unlock(&dvfs_notif_mutex);

	return ret;

}
EXPORT_SYMBOL_GPL(dvfs_run_notifications);

void dvfs_client_notification_cb(struct dvfs_notif *n)
{
	atomic_dec(&missing_cb);

	if (n->err)
		atomic_inc(&failing_drivers);
	else
		n->state = target_state;

	if (unlikely(atomic_read(&missing_cb) == 0))
		complete(&nf_completion);
}
EXPORT_SYMBOL_GPL(dvfs_client_notification_cb);

static int notifier_thread(void *arg)
{
	struct dvfs_notif *n = (struct dvfs_notif *)arg;
	DEFINE_WAIT(__wait);

	for (;;) {
		prepare_to_wait(&queue, &__wait, TASK_INTERRUPTIBLE);
		schedule();
		if (signal_pending(current)) {
			if (try_to_freeze())
				continue;
			break;
		}
		n->nb.notifier_call(&n->nb, dvfs_event, n);
	}
	finish_wait(&queue, &__wait);

	return 0;
}

int dvfs_notif_set(struct dvfs_notif *target, const char *name,
		   int (*callback)(struct notifier_block *,
		   unsigned long dvfs_notifications, void *),
		   enum dvfs_threaded threaded, unsigned long timeout)
{
	if (!target)
		return -EINVAL;

	target->name = name;
	target->nb.notifier_call = callback;
	target->timeout = timeout;
	target->threaded = threaded;

	return 0;
}

int dvfs_register_notifier(struct dvfs_notif *n)
{
	if (unlikely((!n) || (!n->name) || (!n->nb.notifier_call) ||
		     ((n->threaded == DVFS_THREADED_NOTIF) && !n->timeout) ||
		     ((n->threaded == DVFS_CHAINED_NOTIF) && n->timeout)))
		return -EINVAL;

	mutex_lock(&dvfs_notif_mutex);

	if (n->threaded == DVFS_THREADED_NOTIF) {
		n->task = kthread_run(notifier_thread, n, n->name);
		if (IS_ERR(n->task)) {
			mutex_unlock(&dvfs_notif_mutex);
			return -ENOMEM;
		}
		n->err = 0;
	} else {
		blocking_notifier_chain_register(&notifier_list, &n->nb);
	}

	list_add(&n->list, &drivers_list);
	clients_count++;
	update_timeout();

	pr_info("DVFS - Registered %s with timeout %u ms - "
		"notification global timeout: %d ms\n",
		n->name, n->timeout, jiffies_to_msecs(timeout));
	mutex_unlock(&dvfs_notif_mutex);

	return 0;
}
EXPORT_SYMBOL_GPL(dvfs_register_notifier);


int dvfs_unregister_notifier(struct dvfs_notif *n)
{
	if (unlikely(!n))
		return -EINVAL;

	mutex_lock(&dvfs_notif_mutex);

	if (n->task)
		kthread_stop(n->task);
	else
		blocking_notifier_chain_unregister(&notifier_list, &n->nb);

	list_del(&n->list);
	clients_count--;
	update_timeout();

	pr_info("DVFS - Unregistered %s with timeout %u ms - "
		"notification global timeout: %d ms\n",
		n->name, n->timeout, jiffies_to_msecs(timeout));
	mutex_unlock(&dvfs_notif_mutex);

	return 0;
}
EXPORT_SYMBOL_GPL(dvfs_unregister_notifier);
