/*
 * File: linux/arch/arm/mach-omap2/cpu.c
 *
 * Copyright (C) 2007 Nokia Corporation.
 *
 * Klaus Pedersen <klaus.k.pedersen@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/cpufreq.h>
#include <linux/delay.h>
#include <linux/init.h>
#include <linux/err.h>
#include <linux/clk.h>

#include <asm/hardware.h>
#include <asm/io.h>
#include <asm/system.h>

#include "dvfs.h"

#define dprintk(msg...) \
	cpufreq_debug_printk(CPUFREQ_DEBUG_DRIVER, "omap-dvfs", msg)

static struct cpufreq_driver omap_driver;

static struct cpu_model *omap_model;

#define OP(mhz, op) { \
	.frequency = (mhz) * 1000,  \
	.index = op,                \
}

static struct cpufreq_frequency_table omap242x_std[] = {
	OP(165,  3),
	OP(266,  2),
	OP(330,  1),
	OP(400,  0),
	{ .frequency = CPUFREQ_TABLE_END }
};

struct cpu_model {
	unsigned	cpu_id;
	const char	*model_name;
	unsigned	max_freq; /* max clock in kHz */
	struct cpufreq_frequency_table *op_points;  /* clock/OP pairs */
};

#define _OMAP(cpuid, max, op_s, name) { \
	.cpu_id         = cpuid,	                          \
	.model_name     = "Texas(R) OMAP(R) " name " processor ", \
	.max_freq       = (max)*1000,	                          \
	.op_points      = op_s,	                                  \
}

/* CPU models, their operating frequency range, and freq/voltage
   operating points */
static struct cpu_model models[] = {
	_OMAP(2420, 400, omap242x_std, "242x-std"),
	{ .cpu_id = UINT_MAX, }
};
#undef _OMAP
#undef OP

static int omap_verify_cpu_id(int model_id)
{
    return 1;  /* Just say yes */
}

static int omap_cpu_init_table(struct cpufreq_policy *policy)
{
	struct cpu_model *model;

	for (model = models; model->cpu_id != UINT_MAX; model++)
		if (omap_verify_cpu_id(model->cpu_id))
			break;

	omap_model = model;

	dprintk("found \"%s\": max frequency: %dkHz\n",
	        model->model_name, model->max_freq);

	return 0;
}

static unsigned extract_clock(unsigned op)
{
	int i;
	struct cpufreq_frequency_table *p;

	if (!omap_model || !omap_model->op_points)
		return 0;

	p = omap_model->op_points;
	for (i = 0; p[i].frequency != CPUFREQ_TABLE_END; i++) {
		if (op == p[i].index)
			return p[i].frequency;
	}
	BUG_ON(i == 0);
	return p[i-1].frequency;
}

/**
 * omap_verify_speed - verifies a new CPUFreq policy
 * @policy: new policy
 *
 * Limit must be within this model's frequency range at least one
 * border included.
 */
static int omap_verify_speed(struct cpufreq_policy *policy)
{
	return cpufreq_frequency_table_verify(policy, omap_model->op_points);
}

/**
 * omap_get_speed - get the current frequency
 * @cpu: for cpu
 *
 * Return the current clock freq of cpu: @cpu
 */
static unsigned int omap_get_speed(unsigned int cpu)
{
	if (cpu)
		return 0;
	return extract_clock(omap242x_dvfs_get_op());
}

/**
 * omap_setpolicy - set a new CPUFreq policy
 * @policy: new policy
 * @target_freq: the target frequency
 * @relation: how that frequency relates to achieved frequency 
 *            (CPUFREQ_RELATION_L or CPUFREQ_RELATION_H)
 *
 * Sets a new CPUFreq policy.
 */
static int omap_target(struct cpufreq_policy *policy,
		       unsigned int target_freq,
		       unsigned int relation)
{
	struct cpufreq_freqs freqs;
	unsigned int         newstate = 0;
	unsigned int         op;

	if (policy->cpu != 0)
		return -ENODEV;

	if (unlikely(cpufreq_frequency_table_target(policy,
		     omap_model->op_points,
		     target_freq,
		     relation,
		     &newstate))) {
		return -EINVAL;
	}

	op = omap_model->op_points[newstate].index;

	if (op == omap242x_dvfs_get_op()) {
		dprintk("no change needed - OP was and needs to be %x\n", op);
		return 0;
	}

	freqs.cpu = 0;
	freqs.old = extract_clock(omap242x_dvfs_get_op());
	freqs.new = extract_clock(op);

	dprintk("target=%dkHz old=%d new=%d op=%d\n",
		target_freq, freqs.old, freqs.new, op);

	cpufreq_notify_transition(&freqs, CPUFREQ_PRECHANGE);

	omap242x_dvfs_set_op(op);

	cpufreq_notify_transition(&freqs, CPUFREQ_POSTCHANGE);

	return 0;
}

static int __init omap_cpu_init(struct cpufreq_policy *policy)
{
	unsigned freq;
	int ret;

	if (omap_cpu_init_table(policy))
		return -ENODEV;

	freq = omap_get_speed(policy->cpu);

	policy->governor = CPUFREQ_DEFAULT_GOVERNOR;
	policy->cpuinfo.transition_latency = 100000;  /* 100us */
	policy->cur = freq;

	dprintk("omap_cpu_init: cur=%dkHz\n", policy->cur);

	ret = cpufreq_frequency_table_cpuinfo(policy, omap_model->op_points);
	if (ret)
		return (ret);

	cpufreq_frequency_table_get_attr(omap_model->op_points, policy->cpu);

	return 0;
}

static struct freq_attr *omap_attr[] = {
	&cpufreq_freq_attr_scaling_available_freqs,
	NULL,
};

static struct cpufreq_driver omap_driver = {
	.name           = "omap",
	.flags          = CPUFREQ_STICKY,
	.init           = omap_cpu_init,
	.verify         = omap_verify_speed,
	.target         = omap_target,
	.get            = omap_get_speed,
	.attr           = omap_attr,
};

static int __init omap_cpufreq_init(void)
{
	return cpufreq_register_driver(&omap_driver);
}

arch_initcall(omap_cpufreq_init);
