/*
 * This file is part of libgst0.10-dsp
 *
 * Copyright (C) 2006 Nokia Corporation. All rights reserved.
 *
 * Contact: Stefan Kost <stefan.kost@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <gst/gst.h>
#include <gst/base/gstbasesink.h>

#include "gstdsppcmsink.h"
#include "dspmixer.h"

GST_DEBUG_CATEGORY_STATIC (dsppcmsink_debug);
#define GST_CAT_DEFAULT dsppcmsink_debug

static const gchar *pcm_sink_device ="/dev/dsptask/pcm2";


static GstStaticPadTemplate sink_template =
    GST_STATIC_PAD_TEMPLATE ("sink",
    GST_PAD_SINK,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS ("audio/x-raw-int, "
        "endianness = (int) " G_STRINGIFY (G_BYTE_ORDER) ", "
        "signed = (boolean) { TRUE, FALSE }, "
        "width = (int) { 8, 16 }, "
        "depth = (int) { 8, 16 }, "
            "rate = (int) {  8000, 11025, 12000, "
                          " 16000, 22050, 24000, "
                          " 32000, 44100, 48000, "
                          " 64000, 88200, 96000 }, "
        "channels = (int) [ 1, 2 ]; "

        "audio/x-alaw, "
            "rate = (int) {  8000, 11025, 12000, "
                          " 16000, 22050, 24000, "
                          " 32000, 44100, 48000, "
                          " 64000, 88200, 96000 }, "
        "channels = (int) [ 1, 2 ]; "

        "audio/x-mulaw, "
            "rate = (int) {  8000, 11025, 12000, "
                          " 16000, 22050, 24000, "
                          " 32000, 44100, 48000, "
                          " 64000, 88200, 96000 }, "
        "channels = (int) [ 1, 2 ]"
      )
    );

// Function prototypes
static void gst_dsppcmsink_base_init (gpointer g_class);
static void gst_dsppcmsink_class_init (GstDSPPCMSinkClass * klass);
static void gst_dsppcmsink_init (GstDSPPCMSink * dsp, GstDSPPCMSinkClass *g_class);
static void gst_dsppcmsink_dispose (GObject * object);
static void gst_dsppcmsink_set_property (GObject * object, guint prop_id, const GValue * value, GParamSpec * pspec);
static void gst_dsppcmsink_get_property (GObject * object, guint prop_id, GValue * value, GParamSpec * pspec);

// For GstElement
static GstStateChangeReturn gst_dsppcmsink_change_state (GstElement * element, GstStateChange transition);
static gboolean gst_dsppcmsink_query (GstElement * element, GstQuery *query);
static GstClock *gst_dsppcmsink_provide_clock (GstElement * elem);

// For GstBaseSink
static GstCaps *gst_dsppcmsink_getcaps (GstBaseSink *sink);
static gboolean gst_dsppcmsink_setcaps (GstBaseSink *sink, GstCaps *caps);
static GstFlowReturn gst_dsppcmsink_render(GstBaseSink *sink, GstBuffer * buffer);
static gboolean gst_dsppcmsink_event (GstBaseSink *sink, GstEvent *event);
static GstClockTime gst_dsppcmsink_get_time (GstClock * clock, gpointer sink);
static GstStateChangeReturn gst_dsppcmsink_async_play (GstBaseSink *sink);

GST_IMPLEMENT_DSPMIXER_METHODS(GstDSPPCMSink, gst_dsppcmsink);

/*
 * gst_dsppcmsink_do_init:
 * @type:
 *
 *
 */
static void
gst_dsppcmsink_do_init (GType type)
{
  static const GInterfaceInfo iface_info = {
    (GInterfaceInitFunc) gst_dsppcmsink_interface_init,
    NULL,
    NULL,
  };
  static const GInterfaceInfo mixer_iface_info = {
    (GInterfaceInitFunc) gst_dsppcmsink_mixer_interface_init,
    NULL,
    NULL,
  };

  gst_dspaudio_init();
  GST_DEBUG_CATEGORY_INIT (dsppcmsink_debug, "dsppcmsink", 0,
                           "DSP PCM audio sink");
  g_type_add_interface_static (type, GST_TYPE_IMPLEMENTS_INTERFACE, &iface_info);
  g_type_add_interface_static (type, GST_TYPE_MIXER, &mixer_iface_info);
}

GST_BOILERPLATE_FULL (GstDSPPCMSink, gst_dsppcmsink, GstBaseSink,
                      GST_TYPE_BASE_SINK, gst_dsppcmsink_do_init);


/*
 * gst_dsppcmsink_dispose:
 * @object: GObject pointer to element to be deleted
 *
 * Deletes PCM sink element instance. Called automatically by
 * GLib framework when element needs to be disposed.
 */
static void
gst_dsppcmsink_dispose (GObject * object)
{
  GST_INFO("gst_dsppcmsink_dispose");
  GstDSPPCMSink *dsp = (GstDSPPCMSink *) object;

  if (dsp->provided_clock) {
    gst_object_unref (dsp->provided_clock);
    dsp->provided_clock = NULL;
  }

  if (dsp->audio) {
    gst_dspaudio_destroy (dsp->audio);
    dsp->audio = NULL;
  }

  G_OBJECT_CLASS (parent_class)->dispose (object);
}


/*
 * gst_dsppcmsink_base_init:
 * @g_class: PCM sink class
 *
 * Does the basic initialization of PCM sink element class. This is
 * called by GStreamer framework.
 */
static void
gst_dsppcmsink_base_init (gpointer g_class)
{
  GstElementClass *gstelement_class = GST_ELEMENT_CLASS (g_class);
  static const GstElementDetails details = GST_ELEMENT_DETAILS (
    "DSP PCM Sink",
    "Sink/Audio",
    "Raw DSP PCM audio sink",
    "Makoto Sugano <makoto.sugano@nokia.com>");

  gst_element_class_add_pad_template (gstelement_class,
      gst_static_pad_template_get (&sink_template));
  gst_element_class_set_details (gstelement_class, &details);

}


/*
 * gst_dsppcmsink_class_init:
 * @klass: PCM Sink class object
 *
 * Initializes PCM sink element class. This is called by GStreamer
 * framework.
 */
static void
gst_dsppcmsink_class_init (GstDSPPCMSinkClass * klass)
{
  GObjectClass *gobject_class;
  GstElementClass *gstelement_class;
  GstBaseSinkClass *gstbasesink_class;

  gobject_class = (GObjectClass *) klass;
  gstelement_class = (GstElementClass *) klass;
  gstbasesink_class = (GstBaseSinkClass *) klass;

  gobject_class->set_property = GST_DEBUG_FUNCPTR (gst_dsppcmsink_set_property);
  gobject_class->get_property = GST_DEBUG_FUNCPTR (gst_dsppcmsink_get_property);
  gobject_class->dispose      = GST_DEBUG_FUNCPTR (gst_dsppcmsink_dispose);

  gstelement_class->provide_clock = GST_DEBUG_FUNCPTR (gst_dsppcmsink_provide_clock);
  gstelement_class->change_state  = GST_DEBUG_FUNCPTR (gst_dsppcmsink_change_state);
  gstelement_class->query         = GST_DEBUG_FUNCPTR (gst_dsppcmsink_query);

  // Use dspaudio utility to install default properties
  gst_dspaudio_install_sink_properties (G_OBJECT_CLASS (klass));

  gstbasesink_class->render   = GST_DEBUG_FUNCPTR (gst_dsppcmsink_render);
  gstbasesink_class->set_caps = GST_DEBUG_FUNCPTR (gst_dsppcmsink_setcaps);
  gstbasesink_class->get_caps = GST_DEBUG_FUNCPTR (gst_dsppcmsink_getcaps);
  gstbasesink_class->event    = GST_DEBUG_FUNCPTR (gst_dsppcmsink_event);
  gstbasesink_class->async_play = GST_DEBUG_FUNCPTR (gst_dsppcmsink_async_play);
}


/*
 * gst_dsppcmsink_init:
 * @dsp: DSP PCM sink object
 * @g_class: PCM sink Class object
 *
 * Initializes the given PCM sink element instance and allocates required
 * resources. This also creates an audio stream object. This is
 * called by GStreamer framework.
 */
static void
gst_dsppcmsink_init (GstDSPPCMSink * dsp, GstDSPPCMSinkClass *g_class)
{
  dsp->audio = gst_dspaudio_new();
  dsp->provided_clock = gst_audio_clock_new (
                            "DSPClock",
                            (GstAudioClockGetTimeFunc) gst_dsppcmsink_get_time,
                            dsp);
  // DSP sinks do not need syncing
  gst_base_sink_set_sync (GST_BASE_SINK (dsp), FALSE);
}


/*
 * gst_dsppcmsink_getcaps:
 * @sink: GstBaseSink object
 *
 */
static GstCaps *
gst_dsppcmsink_getcaps (GstBaseSink *sink)
{
  return gst_caps_copy (
    gst_pad_get_pad_template_caps (GST_BASE_SINK_PAD(sink)));
}


/*
 * gst_dsppcmsink_initialize:
 * @dsp: DSP PCM sink element
 *
 */
static gboolean
gst_dsppcmsink_initialize(GstDSPPCMSink *dsp)
{
  // Read the junk out...
  gst_dspaudio_flush (dsp->audio);

  GST_DEBUG("AEC off");
  GST_DEBUG ("Init data: fmt=%d, rate=%d (%d), channels=%d",
             dsp->format, dsp->rate_fmt, dsp->rate,
             dsp->channels);

  if (!gst_dspaudio_set_audioparams (dsp->audio,
    dsp->format, dsp->rate_fmt, dsp->channels_fmt)) {
    GST_WARNING ("SET_PARAMS error");
    GST_ELEMENT_ERROR (dsp, RESOURCE, SETTINGS, (NULL),
                       ("gst_dspaudio_setparams"));
    return FALSE;
  }

  // this tells max. amount of bytes we want to put in mmap buffer
  dsp->buf_limit = dsp->bps / 20;        // 50 millisecs per buffer
  dsp->buf_limit = dsp->buf_limit &~ 1;  // make it even

  GST_DEBUG ("buf_limit: %d bytes", dsp->buf_limit);
  GST_INFO ("DSP PCM configured");

  dsp->audio->mode = DSP_MODE_CONFIGURED;
  // DSP throws an error if volume is not set before PLAY
  dsp->audio->volume_changed = TRUE;
  return TRUE;
}


/*
 * gst_dsppcmsink_setcaps:
 * @src: GstBaseSink object
 * @caps: GstCaps to be used.
 *
 * Returns: TRUE if the caps are accepted, otherwise FALSE
 */
static gboolean
gst_dsppcmsink_setcaps (GstBaseSink *src, GstCaps *caps)
{
  GST_INFO("gst_dsppcmsink_setcaps");
  GstDSPPCMSink *dsp = GST_DSPPCMSINK(src);
  guint bps, codec;

  GstStructure *structure = gst_caps_get_structure (caps, 0);
  const gchar *mimetype = gst_structure_get_name (structure);

  if (!strncmp (mimetype, "audio/x-raw-int", 15)) {
    gst_structure_get_int (structure, "width", &dsp->width);
    gst_structure_get_int (structure, "depth", &dsp->depth);
    gst_structure_get_int (structure, "endianness", &dsp->endianness);
    gst_structure_get_boolean (structure, "signed", &dsp->sign);

    if (dsp->width != dsp->depth)
      return FALSE;

    dsp->buf_limit = 160; // 10 milliseconds
    codec = GST_DSPAUDIO_CODEC_RAW;
  }
  else if(!strncmp (mimetype, "audio/x-mulaw", 13)) {
    dsp->buf_limit = 80; // 10 milliseconds
    codec = GST_DSPAUDIO_CODEC_MULAW;
  }
  else if(!strncmp (mimetype, "audio/x-alaw", 12)) {
    dsp->buf_limit = 80; // 10 milliseconds
    codec = GST_DSPAUDIO_CODEC_ALAW;
  }
  else {
    GST_ERROR("Unknown mimetype");
    return FALSE;
  }

  if (!gst_dspaudio_map_format (dsp->audio, codec, dsp->endianness, dsp->sign, dsp->width,
       &dsp->format, &bps)) {
    GST_ERROR("Cannot determine audio format");
    return FALSE;
  }

  // These are common for all formats
  gst_structure_get_int (structure, "channels", &dsp->channels);
  gst_structure_get_int (structure, "rate", &dsp->rate);

  if(!gst_dspaudio_map_channels (dsp->channels, &dsp->channels_fmt)) {
    GST_ELEMENT_ERROR (dsp, STREAM, FORMAT,
        (NULL), ("Not supported channel number : %d!", dsp->channels));
    return FALSE;
  }
  if(!gst_dspaudio_map_samplerate (dsp->rate, &dsp->rate_fmt) ||
      dsp->rate > 48000)
  {
    GST_ELEMENT_ERROR (dsp, STREAM, FORMAT,
        (NULL), ("Not supported sample rate : %d!", dsp->rate));
    return FALSE;
  }

  dsp->bps = bps * dsp->channels * dsp->rate;

  if (gst_dsppcmsink_initialize (dsp) == FALSE) {
    GST_ELEMENT_ERROR (dsp, RESOURCE, SETTINGS,
                       (NULL), ("gst_dsppcmsink_initialize"));
    return FALSE;
  }

  return TRUE;
}


/*
 * gst_dsppcmsink_render:
 * @base: GstBaseSink object
 * @buffer: GstBuffer to be rendered
 *
 */
static GstFlowReturn
gst_dsppcmsink_render (GstBaseSink *sink, GstBuffer * buffer)
{
  DSPWaitStatus status = DSP_WAIT_ERROR;
  GstDSPPCMSink *dsp = GST_DSPPCMSINK (sink);
  guint8 *data = (guint8 *) GST_BUFFER_DATA(buffer);
  guint length = GST_BUFFER_SIZE(buffer);
  guint dataIndex = 0, tw, tw1;
  GstFlowReturn retval = GST_FLOW_OK;

  GST_LOG ("render: %d bytes", GST_BUFFER_SIZE(buffer));

  while (length > 0) {

    if(dsp->audio->mode == DSP_MODE_ERROR) {
      GST_ERROR ("error. Leaving...");

      GST_ELEMENT_ERROR (dsp, RESOURCE, WRITE,
                         ("dsp error cmd: %d, status: %d",
                          dsp->audio->error_cmd,
                          dsp->audio->error_status), GST_ERROR_SYSTEM);
      retval = GST_FLOW_ERROR;
      break;
    }
    else if (dsp->audio->mode == DSP_MODE_CONFIGURED) {
      gst_dspaudio_flush (dsp->audio);
      gst_dspaudio_update_dsp_settings (dsp->audio);
      gst_dspaudio_play (dsp->audio);

      GstClockTime latency = gst_base_sink_get_latency (sink);
      if (latency) {
        GST_INFO ("Latency: %"GST_TIME_FORMAT, GST_TIME_ARGS (latency));
        g_usleep (latency / GST_USECOND);
      }
    }

    status = gst_dspaudio_wait_buffer (dsp->audio);

    if (status == DSP_WAIT_OK) {
      tw1 = MIN ((dsp->audio->codec.mmap_buffer_size - dsp->outIndex), length);
      tw = MIN (dsp->buf_limit, tw1);
      memcpy (dsp->audio->codec.mmap_buffer + dsp->outIndex, data+dataIndex, tw);
      dsp->outIndex += tw;
      dataIndex += tw;
      length -= tw;
    }
    else if (status == DSP_WAIT_INTERRUPT) {
      GST_DEBUG ("DSP_WAIT_INTERRUPT");
      retval = gst_base_sink_wait_preroll (sink);
      if (retval != GST_FLOW_OK)
        break;
      continue;
    }
    else {
      GST_DEBUG ("Wait_buffer error");
      dsp->audio->mode = DSP_MODE_ERROR;
      retval = GST_FLOW_ERROR;
      break;
    }

    if (dsp->outIndex >= dsp->buf_limit) {
      gst_dspaudio_write_data (dsp->audio, dsp->outIndex / sizeof(short int),
                               FALSE);
      GST_LOG ("PCM WRITE %d bytes to DSP", dsp->outIndex);
      dsp->outIndex = 0;
    }
  }
  GST_LOG ("render exit");
  return retval;
}


/*
 * gst_dsppcmsink_set_property:
 * @object: GObject where the property should be retrieved
 * @prop_id: Unique identifier of the desired property
 * @value: Pointer to a variable where the value should be put
 * @pspec: Parameter specification
 *
 * Stores the property value into GObject
 */
static void
gst_dsppcmsink_set_property (GObject * object,
                             guint prop_id,
                             const GValue * value,
                             GParamSpec * pspec)
{
  GstDSPPCMSink *dsp = GST_DSPPCMSINK (object);

  if (!gst_dspaudio_set_property(dsp->audio, prop_id, value)) {
    GST_WARNING ("Trying to set illegal property");
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
  }
}


/*
 * gst_dsppcmsink_get_property:
 * @object: GObject where the property should be retrieved
 * @prop_id: Unique identifier of the desired property
 * @value: Pointer to a variable where the value should be put
 * @pspec: Parameter specification
 *
 * Retrieves the desired parameter value from object
 */
static void
gst_dsppcmsink_get_property (GObject * object,
                             guint prop_id,
                             GValue * value,
                             GParamSpec * pspec)
{
  GstDSPPCMSink *dsp = GST_DSPPCMSINK (object);

  if (!gst_dspaudio_get_property (dsp->audio, prop_id, value)) {
    GST_WARNING ("Trying to get illegal property");
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
  }
}


/*
 * gst_dsppcmsink_event:
 *
 */
static gboolean
gst_dsppcmsink_event (GstBaseSink *sink, GstEvent *event)
{
  GST_DEBUG("Event");
  GstDSPPCMSink *dsp = GST_DSPPCMSINK (sink);

  GstEventType type = event ? GST_EVENT_TYPE (event) : GST_EVENT_UNKNOWN;

  if(type == GST_EVENT_EOS) {
    DSPWaitStatus status;

    GST_DEBUG("EOS event");

    // Write the rest of the buffer to DSP
    if (dsp->audio->mode == DSP_MODE_PLAYING) {
      GST_DEBUG ("Waiting data_write in handle_event()");
      status = gst_dspaudio_wait_buffer (dsp->audio);

      if (status == DSP_WAIT_OK) {
        GST_DEBUG ("Got the data_write");
        gst_dspaudio_write_data (dsp->audio, dsp->outIndex / sizeof(short int),
                                 TRUE);
        GST_DEBUG ("Wrote DSP_CMD_EOF to DSP");
        dsp->audio->mode = DSP_MODE_EOS;
      }
    }
  }
  else if (type == GST_EVENT_FLUSH_STOP) {
    GST_DEBUG("flush stop");

    if (dsp->audio->mode != DSP_MODE_EOS) {
      gst_dspaudio_discont (dsp->audio);
      dsp->outIndex = 0;
    }
  }
  return TRUE;
}


/*
 * gst_dsppcmsink_query:
 * @element: GstElement that received this query
 * @type: Type of the query, e.g. TOTAL, POSITION
 * @format: The format of the response, e.g. BYTES, TIME
 * @value: Pointer where the return value should be stored
 *
 * Returns: TRUE, iff the query was successful.
 */
static gboolean
gst_dsppcmsink_query (GstElement * element, GstQuery *query)
{
  GstBaseSink *bsink = GST_BASE_SINK_CAST (element);
  GstQueryType type = GST_QUERY_TYPE (query);
  gboolean res = FALSE;

  if (type == GST_QUERY_LATENCY) {
    GstDSPPCMSink *dsp = GST_DSPPCMSINK (element);
    gboolean live, us_live;
    GstClockTime min_l, max_l;
    GST_DEBUG ("latency query");

    if (dsp->audio->mode < DSP_MODE_CONFIGURED) {
      GST_DEBUG ("we are not yet negotiated, can't report latency yet");
      res = FALSE;
    }
    else if ((res = gst_base_sink_query_latency (bsink, &live, &us_live,
                                                 &min_l, &max_l)))
    {
      GstClockTime min_latency, max_latency;
      /* upstream is live, adjust the min_latency */
      if (us_live) {
        /* FIXME: Calculate this somehow */
        min_latency = min_l + 500 * GST_MSECOND;
        max_latency = min_latency + (max_l == -1 ? 0 : max_l);
      } else {
        GST_DEBUG ("peer is not live, don't care about latency");
        min_latency = 0;
        max_latency = -1;
      }
      gst_query_set_latency (query, live, min_latency, max_latency);
    }
  }
  else {
    res = GST_ELEMENT_CLASS (parent_class)->query (element, query);
  }
  return res;
}


/*
 * gst_dsppcmsink_async_play:
 *
 */
static GstStateChangeReturn
gst_dsppcmsink_async_play (GstBaseSink *sink)
{
  GstDSPPCMSink *dsppcmsink = GST_DSPPCMSINK (sink);
  GstClock *clock = GST_ELEMENT_CLOCK (sink);

  GstClockTime itime, etime;
  GstClockTime rate_num, rate_denom;

  GST_DEBUG ("Async play");

  if (clock == dsppcmsink->provided_clock)
    goto done;

  /* if we are slaved to a clock, we need to set the initial
   * calibration */
  /* get external and internal time to set as calibration params */
  etime = gst_clock_get_time (clock);
  itime = gst_clock_get_internal_time (dsppcmsink->provided_clock);

  GST_DEBUG ("internal time: %" GST_TIME_FORMAT " external time: %" GST_TIME_FORMAT,
             GST_TIME_ARGS (itime), GST_TIME_ARGS (etime));

  gst_clock_get_calibration (dsppcmsink->provided_clock, NULL, NULL, &rate_num,
                             &rate_denom);
  gst_clock_set_calibration (dsppcmsink->provided_clock, itime, etime,
                             rate_num, rate_denom);

done:
  if(dsppcmsink->audio->mode == DSP_MODE_PAUSED ||
     dsppcmsink->audio->mode == DSP_MODE_EOS)
  {
    gst_dspaudio_play (dsppcmsink->audio);
    GST_DEBUG("async play success");
  }
  return GST_STATE_CHANGE_SUCCESS;
}


/*
 * gst_dsppcmsink_change_state:
 * @element: The GstElement whose state to be changed
 *
 * Changes the state of given element. The desired state information
 * is contained by the element itself, and this method is just
 * a callback, so the plugin knows when the state is changing.
 * This method is called by GStreamer framework
 *
 * Returns: TRUE if operation was successful, otherwise FALSE
 */
static GstStateChangeReturn
gst_dsppcmsink_change_state(GstElement * element, GstStateChange transition)
{
  GstDSPPCMSink *dsp = GST_DSPPCMSINK (element);
  GstStateChangeReturn ret;

  GST_DEBUG("Transition: %d", transition);

  switch (transition) {

    case GST_STATE_CHANGE_NULL_TO_READY:
      GST_DEBUG("NULL TO READY");
      if(gst_dspaudio_open(dsp->audio, pcm_sink_device) == FALSE) {
        GST_ELEMENT_ERROR (dsp, RESOURCE, OPEN_WRITE,
                           (NULL), ("gst_dspaudio_open"));
        return GST_STATE_CHANGE_FAILURE;
      }
      dsp->outIndex = 0;
      break;

    case GST_STATE_CHANGE_READY_TO_PAUSED:
      GST_DEBUG("READY TO PAUSED");
      break;

    case GST_STATE_CHANGE_PAUSED_TO_PLAYING:
      GST_DEBUG("PAUSED TO PLAYING");
      gst_dsppcmsink_async_play (GST_BASE_SINK_CAST (element));
      GST_DEBUG("PAUSED TO PLAYING OK");
      break;

    case GST_STATE_CHANGE_PLAYING_TO_PAUSED:
      GST_DEBUG("PLAYING TO PAUSED 1");
      gst_dspaudio_interrupt_render(dsp->audio);

    default:
      break;
  }

  gst_dspaudio_clock_change_state (dsp->audio, transition);
  ret = GST_ELEMENT_CLASS (parent_class)->change_state (element, transition);

  switch (transition) {

    case GST_STATE_CHANGE_PLAYING_TO_PAUSED:
      gst_dspaudio_remove_interrupt (dsp->audio);
      GST_DEBUG ("PLAYING TO PAUSED 2");
      if(dsp->audio->mode == DSP_MODE_EOS) {
        gst_dspaudio_wait_eos (dsp->audio);
        GST_DEBUG ("Got eos response");
      }
      else {
        gst_dspaudio_pause (dsp->audio);
      }
      GST_DEBUG ("PLAYING TO PAUSED 2 OK");
      break;

    case GST_STATE_CHANGE_PAUSED_TO_READY:
      GST_DEBUG ("PAUSED TO READY");
      gst_dspaudio_stop(dsp->audio);
      GST_DEBUG ("PAUSED TO READY OK");
      break;

    case GST_STATE_CHANGE_READY_TO_NULL:
      GST_DEBUG ("READY To NULL");
      gst_dspaudio_close (dsp->audio);
      gst_dspaudio_reset (dsp->audio);
      break;

    default:
      break;
  }

  GST_DEBUG ("State change successful");
  return ret;
}


/*
 * gst_dsppcmsink_provide_clock:
 *
 */
static GstClock *
gst_dsppcmsink_provide_clock (GstElement * elem)
{
  GST_INFO("gst_dsppcmsink_provide_clock");
  GstDSPPCMSink *dsp = GST_DSPPCMSINK (elem);
  return GST_CLOCK_CAST (gst_object_ref (dsp->provided_clock));
}


/*
 * gst_dsppcmsink_get_time:
 *
 */
static GstClockTime
gst_dsppcmsink_get_time (GstClock * clock, gpointer sink)
{
  GstDSPPCMSink *dsp = GST_DSPPCMSINK (sink);
  return gst_dspaudio_get_dsp_clock (dsp->audio, TRUE);
}

