/*
 * This file is part of libgst0.10-dsp
 *
 * Copyright (C) 2006 Nokia Corporation. All rights reserved.
 *
 * Contact: Stefan Kost <stefan.kost@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <gst/gst.h>

#include "gstdspilbcsink.h"
#include "dspmixer.h"

// iLBC SID_UPDATE frame size in bytes
#define ILBC_DTX_UPDATE_FRAME_SIZE 22

GST_DEBUG_CATEGORY_STATIC (dspilbcsink_debug);
#define GST_CAT_DEFAULT dspilbcsink_debug

static const gchar *devname = "/dev/dsptask/ilbc_dec";

static GstStaticPadTemplate dspilbcsink_sink_template =
GST_STATIC_PAD_TEMPLATE ("sink",
    GST_PAD_SINK,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS ( "audio/x-iLBC, "
                      "mode = (int) { 20, 30 }"
                    ) );

// Function prototypes

static void gst_dspilbcsink_class_init (GstDSPILBCSinkClass * klass);
static void gst_dspilbcsink_base_init (gpointer g_class);
static void gst_dspilbcsink_init (GstDSPILBCSink *dspmp3sink, GstDSPILBCSinkClass *g_class);
static void gst_dspilbcsink_set_property (GObject * object, guint prop_id, const GValue * value, GParamSpec * pspec);
static void gst_dspilbcsink_get_property (GObject * object, guint prop_id, GValue * value, GParamSpec * pspec);
static gboolean gst_dspilbcsink_start (GstBaseSink * bsrc);
static gboolean gst_dspilbcsink_stop (GstBaseSink * bsrc);
static GstFlowReturn gst_dspilbcsink_render(GstBaseSink *sink, GstBuffer * buffer);
static gboolean gst_dspilbcsink_setcaps (GstBaseSink * basesink, GstCaps * caps);
static gboolean gst_dspilbcsink_unlock (GstBaseSink *src);
static gboolean gst_dspilbcsink_unlock_stop (GstBaseSink *src);

GST_IMPLEMENT_DSPMIXER_METHODS (GstDSPILBCSink, gst_dspilbcsink);


/*
 * gst_dspilbcsink_do_init:
 * @type:
 *
 *
 */
static void
gst_dspilbcsink_do_init (GType type)
{
  static const GInterfaceInfo iface_info = {
    (GInterfaceInitFunc) gst_dspilbcsink_interface_init,
    NULL,
    NULL,
  };
  static const GInterfaceInfo mixer_iface_info = {
    (GInterfaceInitFunc) gst_dspilbcsink_mixer_interface_init,
    NULL,
    NULL,
  };

  gst_dspaudio_init();
  GST_DEBUG_CATEGORY_INIT (dspilbcsink_debug, "dspilbcsink", 0,
                           "DSP iLBC audio sink");
  g_type_add_interface_static (type, GST_TYPE_IMPLEMENTS_INTERFACE, &iface_info);
  g_type_add_interface_static (type, GST_TYPE_MIXER, &mixer_iface_info);
}

GST_BOILERPLATE_FULL (GstDSPILBCSink, gst_dspilbcsink, GstBaseSink,
                      GST_TYPE_BASE_SINK, gst_dspilbcsink_do_init);


/*
 * gst_dspilbcsink_dispose:
 * @object: GObject pointer to element to be deleted
 *
 * Deletes ILBC sink element instance. Called automatically by
 * GLib framework when element needs to be disposed.
 */
static void
gst_dspilbcsink_dispose (GObject * object)
{
  GstDSPILBCSink *dsp = (GstDSPILBCSink *) object;

  if(dsp->audio) {
    gst_dspaudio_destroy(dsp->audio);
    dsp->audio = NULL;
  }

  G_OBJECT_CLASS (parent_class)->dispose (object);
}


/*
 * gst_dspilbcsink_base_init:
 * @g_class: ILBC sink class
 *
 * Does the basic initialization of ILBC sink element class. This is
 * called by GStreamer framework.
 */
static void
gst_dspilbcsink_base_init (gpointer g_class)
{
  GstElementClass *gstelement_class = GST_ELEMENT_CLASS (g_class);
  static const GstElementDetails details = GST_ELEMENT_DETAILS (
    "DSP ILBC Sink",
    "Sink/Audio",
    "ILBC audio sink",
    "Makoto Sugano <makoto.sugano@nokia.com>");

  gst_element_class_add_pad_template (gstelement_class,
      gst_static_pad_template_get (&dspilbcsink_sink_template));
  gst_element_class_set_details (gstelement_class, &details);

}


/*
 * gst_dspilbcsink_class_init:
 * @klass: ILBC Sink class object
 *
 * Initializes ILBC sink element class. This is called by GStreamer
 * framework.
 */
static void
gst_dspilbcsink_class_init (GstDSPILBCSinkClass * klass)
{
  GObjectClass *gobject_class;
  GstBaseSinkClass *gstbase_sink_class;

  gobject_class = (GObjectClass *) klass;
  gstbase_sink_class = (GstBaseSinkClass *) klass;

  gobject_class->set_property = GST_DEBUG_FUNCPTR (gst_dspilbcsink_set_property);
  gobject_class->get_property = GST_DEBUG_FUNCPTR (gst_dspilbcsink_get_property);
  gobject_class->dispose      = GST_DEBUG_FUNCPTR (gst_dspilbcsink_dispose);

  // Use dspaudio utility to install default properties
  gst_dspaudio_install_sink_properties (G_OBJECT_CLASS (klass));

  gstbase_sink_class->start       = GST_DEBUG_FUNCPTR (gst_dspilbcsink_start);
  gstbase_sink_class->stop        = GST_DEBUG_FUNCPTR (gst_dspilbcsink_stop);
  gstbase_sink_class->render      = GST_DEBUG_FUNCPTR (gst_dspilbcsink_render);
  gstbase_sink_class->set_caps    = GST_DEBUG_FUNCPTR (gst_dspilbcsink_setcaps);
  gstbase_sink_class->unlock      = GST_DEBUG_FUNCPTR (gst_dspilbcsink_unlock);
  gstbase_sink_class->unlock_stop = GST_DEBUG_FUNCPTR (gst_dspilbcsink_unlock_stop);
}


/*
 * gst_dspilbcsink_init:
 * @dsp: DSP ILBC sink object
 * @g_class: ILBC sink Class object
 *
 * Initializes the given ILBC sink element instance and allocates required
 * resources. This also creates an audio stream object. This is
 * called by GStreamer framework.
 */
static void
gst_dspilbcsink_init (GstDSPILBCSink * dspilbcsink, GstDSPILBCSinkClass *g_class)
{
  dspilbcsink->audio = gst_dspaudio_new();

  // DSP sinks do not need syncing
  gst_base_sink_set_sync (GST_BASE_SINK (dspilbcsink), FALSE);
}


/*
 * gst_dspilbcsink_setcaps:
 * @basesink: GstBaseSink
 * @caps: GstCaps that were offered by the connecting element
 *
 * Returns: TRUE if caps can be accepted, otherwise FALSE
 */
static gboolean
gst_dspilbcsink_setcaps(GstBaseSink * basesink, GstCaps * caps)
{
  GST_DEBUG("gst_dspilbcsink_setcaps");
  GstDSPILBCSink *dsp = GST_DSPILBCSINK (basesink);
  guint format, sample_rate, frame_samples;
  GstStructure *structure = gst_caps_get_structure (caps, 0);
  
  gst_structure_get_int (structure, "mode", &dsp->mode);

  if (dsp->mode == 20) {
    frame_samples = 20*8;      // samples
    dsp->framesize = 38;       // bytes
  }
  else {
    frame_samples = 30*8;      // samples
    dsp->framesize = 50;       // bytes
  }

  // FIXME: is dsp->audio->dtxmode == GST_DSPAUDIO_DTX_MODE_ON
  //format = DSP_AFMT_ILBC_DTX;
  if(!gst_dspaudio_map_format (dsp->audio, GST_DSPAUDIO_CODEC_ILBC,
    G_LITTLE_ENDIAN, TRUE, 0, &format, NULL)) {
    GST_ERROR("Cannot determine audio format");
    return FALSE;
  }
  if(!gst_dspaudio_map_samplerate (8000,&sample_rate)) {
    GST_WARNING ("unsupported sample rate: 8000");
    return FALSE;
  }

  GST_DEBUG("Sink frame size: %d bytes", dsp->framesize);

  // Read the junk out...
  gst_dspaudio_flush (dsp->audio);

  if (!gst_dspaudio_set_speechparams (dsp->audio,
    format, sample_rate, frame_samples)) {
    return FALSE;
  }
  dsp->audio->mode = DSP_MODE_INITIALIZED;
  GST_DEBUG("gst_dspilbcsink_setcaps OK");
  return TRUE;
}


/*
 * gst_dspilbcsink_render:
 * sink:
 * @buffer:
 *
 */
static GstFlowReturn
gst_dspilbcsink_render (GstBaseSink *sink, GstBuffer * buffer)
{
  GstFlowReturn ret = GST_FLOW_OK;
  GstDSPILBCSink *dsp = GST_DSPILBCSINK (sink);

  GST_LOG ("Render: %d bytes, timestamp %lld, discont = %s",
           GST_BUFFER_SIZE(buffer), GST_BUFFER_TIMESTAMP(buffer),
           GST_BUFFER_FLAG_IS_SET(buffer, GST_BUFFER_FLAG_DISCONT) ?
                "TRUE" : "FALSE");

  if (dsp->audio->mode == DSP_MODE_ERROR) {
    GST_ELEMENT_ERROR (dsp, RESOURCE, WRITE, (NULL),
                       ("error cmd: %d status: %d",
                        dsp->audio->error_cmd,
                        dsp->audio->error_status));
    return GST_FLOW_ERROR;
  }
  else if (dsp->audio->mode != DSP_MODE_PLAYING) {
    GST_DEBUG ("Setting dsp to play");

    // Force the volume setting
    dsp->audio->volume_changed = TRUE;
    gst_dspaudio_update_dsp_settings (dsp->audio);
    gst_dspaudio_play (dsp->audio);
  }

retry:
  ret = gst_dspaudio_write_frame (dsp->audio, buffer, dsp->framesize,
                                  ILBC_DTX_UPDATE_FRAME_SIZE, TRUE);

  if (ret == GST_FLOW_WRONG_STATE && dsp->audio->mode != DSP_MODE_ERROR) {
    ret = gst_base_sink_wait_preroll (sink);
    if (ret == GST_FLOW_OK) goto retry;
  }

  GST_LOG("Render exit");
  return ret;
}


/*
 * gst_dspilbcsink_start:
 * @bsink: GstBaseSink
 *
 */
static gboolean
gst_dspilbcsink_start (GstBaseSink * bsink)
{
  GST_INFO("gst_dspilbcsink_start");
  GstDSPILBCSink *dsp = GST_DSPILBCSINK (bsink);

  if (gst_dspaudio_open(dsp->audio, devname) == FALSE) {
    GST_ELEMENT_ERROR (dsp, RESOURCE,
                       OPEN_READ_WRITE, (NULL),
                       ("gst_dspaudio_open"));
    return FALSE;
  }
  GST_DEBUG("gst_dspilbcsink_start OK");
  return TRUE;
}


/*
 * gst_dspilbcsink_stop:
 * @bsink: GstBaseSink
 *
 */
static gboolean
gst_dspilbcsink_stop (GstBaseSink * bsink)
{
  GST_INFO("gst_dspilbcsink_stop");
  GstDSPILBCSink *dsp = GST_DSPILBCSINK (bsink);

  gst_dspaudio_remove_interrupt (dsp->audio);
  gst_dspaudio_stop (dsp->audio);
  gst_dspaudio_close (dsp->audio);
  gst_dspaudio_reset (dsp->audio);
  GST_DEBUG("gst_dspilbcsink_stop OK");
  return TRUE;
}


/*
 * gst_dspilbcsink_set_property:
 * @object: GObject where the property should be retrieved
 * @prop_id: Unique identifier of the desired property
 * @value: Pointer to a variable where the value should be put
 * @pspec: Parameter specification
 *
 * Stores the property value into GObject
 */
static void
gst_dspilbcsink_set_property(GObject * object,
                            guint prop_id,
                            const GValue * value,
                            GParamSpec * pspec)
{
  GstDSPILBCSink *dsp = GST_DSPILBCSINK (object);

  if(!gst_dspaudio_set_property(dsp->audio, prop_id, value)) {
    GST_WARNING("Trying to set illegal property");
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
  }
}


/*
 * gst_dspilbcsink_get_property:
 * @object: GObject where the property should be retrieved
 * @prop_id: Unique identifier of the desired property
 * @value: Pointer to a variable where the value should be put
 * @pspec: Parameter specification
 *
 * Retrieves the desired parameter value from object
 */
static void
gst_dspilbcsink_get_property (GObject * object,
                              guint prop_id,
                              GValue * value,
                              GParamSpec * pspec)
{
  GstDSPILBCSink *dsp = GST_DSPILBCSINK (object);

  if(!gst_dspaudio_get_property (dsp->audio, prop_id, value)) {
    GST_WARNING ("Trying to get illegal property");
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
  }
}


/*
 * gst_dspilbcsink_unlock:
 * @src: GstBaseSink
 *
 * Interrupts the blocking :render() function.
 *
 * Returns: TRUE
 */
static gboolean
gst_dspilbcsink_unlock (GstBaseSink *src)
{
    GST_DEBUG ("unlock");
    GstDSPILBCSink *dsp = GST_DSPILBCSINK (src);
    gst_dspaudio_interrupt_render (dsp->audio);
    return TRUE;
}


/*
 * gst_dspilbcsink_unlock_stop:
 * @src: GstBaseSink
 *
 * Removes :render() unlocking.
 *
 * Returns: TRUE
 */
static gboolean
gst_dspilbcsink_unlock_stop (GstBaseSink *src)
{
    GST_DEBUG ("unlock_stop");
    GstDSPILBCSink *dsp = GST_DSPILBCSINK (src);
    gst_dspaudio_remove_interrupt (dsp->audio);
    return TRUE;
}
