/*
 * This file is part of libgst0.10-dsp
 *
 * Copyright (C) 2006 Nokia Corporation. All rights reserved.
 *
 * Contact: Stefan Kost <stefan.kost@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <unistd.h>
#include <string.h>
#include <gst/gst.h>
#include <gst/audio/gstaudioclock.h>

#include "gstdspg729src.h"
#include "dspmixer.h"

// Duration of one G.729 voice frame in milliseconds
#define G729_FRAMETIMEMILLIS 10

// This is the frame size (in bytes) produced by DSP (parametric format).
#define G729_FRAMESIZE       22

GST_DEBUG_CATEGORY_STATIC (dspg729src_debug);
#define GST_CAT_DEFAULT dspg729src_debug

static gchar *devname = "/dev/dsptask/g729_enc";

static GstStaticPadTemplate dspg729src_src_template =
GST_STATIC_PAD_TEMPLATE ("src",
    GST_PAD_SRC,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS ( "audio/g729, "
                      "rate = (int) 8000, "
                      "channels = (int) 1"
                    ) );

// Function prototypes

static void
gst_dspg729src_class_init           (GstDSPG729SrcClass * klass);

static void
gst_dspg729src_base_init            (gpointer g_class);

static void
gst_dspg729src_init                 (GstDSPG729Src *src,
                                     GstDSPG729SrcClass *g_class);

static void
gst_dspg729src_set_property         (GObject * object,
                                     guint prop_id,
                                     const GValue * value,
                                     GParamSpec * pspec);

static void
gst_dspg729src_get_property         (GObject * object,
                                     guint prop_id,
                                     GValue * value,
                                     GParamSpec * pspec);

static GstClock *
gst_dspg729src_provide_clock        (GstElement * elem);

static GstClockTime
gst_dspg729src_get_time             (GstClock * clock,
                                     GstDSPG729Src * src);

static GstCaps *
gst_dspg729src_getcaps              (GstBaseSrc * src);

static GstFlowReturn
gst_dspg729src_create               (GstPushSrc * psrc,
                                     GstBuffer ** buf);

static gboolean
gst_dspg729src_start                (GstBaseSrc * bsrc);

static gboolean
gst_dspg729src_stop                 (GstBaseSrc * bsrc);

static gboolean
gst_dspg729src_event                (GstBaseSrc *src,
                                     GstEvent *event);

static GstBuffer *
gst_dspg729src_voice_frame_handler  (short int *mmap_ptr,
                                     guint fsize);

static GstBuffer *
gst_dspg729src_sid_frame_handler    (short int *mmap_ptr,
                                     guint fsize);

static gboolean
gst_dspg729src_unlock               (GstBaseSrc *src);

static gboolean
gst_dspg729src_unlock_stop          (GstBaseSrc *src);


GST_IMPLEMENT_DSPMIXER_METHODS (GstDSPG729Src, gst_dspg729src);


/*
 * gst_dspg729src_do_init:
 * @type:
 *
 *
 */
static void
gst_dspg729src_do_init (GType type)
{
  static const GInterfaceInfo iface_info = {
    (GInterfaceInitFunc) gst_dspg729src_interface_init,
    NULL,
    NULL,
  };
  static const GInterfaceInfo mixer_iface_info = {
    (GInterfaceInitFunc) gst_dspg729src_mixer_interface_init,
    NULL,
    NULL,
  };

  gst_dspaudio_init();
  GST_DEBUG_CATEGORY_INIT (dspg729src_debug, "dspg729src", 0,
                           "DSP G.729 audio source");
  g_type_add_interface_static (type, GST_TYPE_IMPLEMENTS_INTERFACE, &iface_info);
  g_type_add_interface_static (type, GST_TYPE_MIXER, &mixer_iface_info);
}

GST_BOILERPLATE_FULL (GstDSPG729Src, gst_dspg729src, GstPushSrc,
                      GST_TYPE_PUSH_SRC, gst_dspg729src_do_init);


/*
 * gst_dspg729src_dispose:
 * @object: GObject pointer to element to be deleted
 *
 * Deletes G729 src element instance. Called automatically by
 * GLib framework when element needs to be disposed.
 */
static void
gst_dspg729src_dispose (GObject * object)
{
  GstDSPG729Src *dsp = (GstDSPG729Src *) object;

  if (dsp->clock) {
    gst_object_unref (dsp->clock);
    dsp->clock = NULL;
  }

  if(dsp->audio) {
    gst_dspaudio_destroy(dsp->audio);
    dsp->audio = NULL;
  }

  G_OBJECT_CLASS (parent_class)->dispose (object);
}


/*
 * gst_dspg729src_base_init:
 * @g_class: G729 src class
 *
 * Does the basic initialization of G729 src element class. This is
 * called by GStreamer framework.
 */
static void
gst_dspg729src_base_init (gpointer g_class)
{
  GstElementClass *gstelement_class = GST_ELEMENT_CLASS (g_class);
  static const GstElementDetails details = GST_ELEMENT_DETAILS (
    "DSP G.729 Src",
    "Source/Audio",
    "G.729 audio src",
    "Makoto Sugano <makoto.sugano@nokia.com>");

  gst_element_class_add_pad_template (gstelement_class,
      gst_static_pad_template_get (&dspg729src_src_template));
  gst_element_class_set_details (gstelement_class, &details);

}

/*
 * gst_dspg729src_class_init:
 * @klass: G729 Src class object
 *
 * Initializes G729 src element class. This is called by GStreamer
 * framework.
 */
static void
gst_dspg729src_class_init (GstDSPG729SrcClass * klass)
{
  GObjectClass *gobject_class;
  GstElementClass *gstelement_class;
  GstBaseSrcClass *gstbasesrc_class;
  GstPushSrcClass *gstpushsrc_class;

  gobject_class = (GObjectClass *) klass;
  gstelement_class = (GstElementClass *) klass;
  gstbasesrc_class = (GstBaseSrcClass *) klass;
  gstpushsrc_class = (GstPushSrcClass *) klass;

  gobject_class->set_property = GST_DEBUG_FUNCPTR (gst_dspg729src_set_property);
  gobject_class->get_property = GST_DEBUG_FUNCPTR (gst_dspg729src_get_property);
  gobject_class->dispose      = GST_DEBUG_FUNCPTR (gst_dspg729src_dispose);

  // Use dspaudio utility to install default properties
  gst_dspaudio_install_src_properties (G_OBJECT_CLASS (klass));

  gstelement_class->provide_clock = GST_DEBUG_FUNCPTR (gst_dspg729src_provide_clock);

  gstbasesrc_class->start       = GST_DEBUG_FUNCPTR (gst_dspg729src_start);
  gstbasesrc_class->stop        = GST_DEBUG_FUNCPTR (gst_dspg729src_stop);
  gstbasesrc_class->get_caps    = GST_DEBUG_FUNCPTR (gst_dspg729src_getcaps);
  gstbasesrc_class->event       = GST_DEBUG_FUNCPTR (gst_dspg729src_event);
  gstbasesrc_class->unlock      = GST_DEBUG_FUNCPTR (gst_dspg729src_unlock);
  gstbasesrc_class->unlock_stop = GST_DEBUG_FUNCPTR (gst_dspg729src_unlock_stop);

  gstpushsrc_class->create   = GST_DEBUG_FUNCPTR (gst_dspg729src_create);
}


/*
 * gst_dspg729src_init:
 * @dsp: DSP G729 src object
 * @g_class: G729 src Class object
 *
 * Initializes the given G729 src element instance and allocates required
 * resources. This also creates an audio stream object. This is
 * called by GStreamer framework.
 */
static void
gst_dspg729src_init (GstDSPG729Src * dspg729src, GstDSPG729SrcClass *g_class)
{
  dspg729src->audio = gst_dspaudio_new();
  gst_base_src_set_live (GST_BASE_SRC(dspg729src), TRUE);

  // Override default frame handlers with our own
  dspg729src->audio->handle_voice_frame = gst_dspg729src_voice_frame_handler;
  dspg729src->audio->handle_sid_frame = gst_dspg729src_sid_frame_handler;

  dspg729src->clock = gst_audio_clock_new ("clock",
                                           (GstAudioClockGetTimeFunc)
                                                gst_dspg729src_get_time,
                                           dspg729src);
  GST_INFO("Clock created");
}


/*
 * gst_dspg729src_getcaps:
 * @src: GstBaseSrc
 *
 * Returns: Output caps of source pad
 */
static GstCaps *
gst_dspg729src_getcaps (GstBaseSrc * src)
{
  GstCaps *caps = gst_caps_copy (
      gst_pad_get_pad_template_caps (GST_BASE_SRC_PAD(src)));

  return caps;
}


/*
 * gst_dspg729src_create:
 * @psrc: GstPushSrc
 * @buffer: pointer where the newly created buffer should be stored
 *
 * Retrieves audio data from DSP and creates a GstBuffer of it.
 *
 * Returns: GstFlowReturn
 */
static GstFlowReturn
gst_dspg729src_create (GstPushSrc * psrc, GstBuffer ** buf)
{
  GST_LOG("gst_dspg729src_create");
  GstDSPG729Src *dsp = GST_DSPG729SRC (psrc);
  GstFlowReturn ret = GST_FLOW_OK;

  if(dsp->audio->mode == DSP_MODE_ERROR) {
    GST_ELEMENT_ERROR (dsp, RESOURCE, READ, (NULL),
                       ("error cmd: %d status: %d",
                        dsp->audio->error_cmd,
                        dsp->audio->error_status));
    return GST_FLOW_ERROR;
  }

  for (;;) {
    *buf = gst_dspaudio_read_frame (dsp->audio, G729_FRAMESIZE, 8,
                                    G729_FRAMETIMEMILLIS);

    if (!*buf) {
      if (dsp->audio->mode == DSP_MODE_ERROR) {
        ret = GST_FLOW_ERROR;
        break;
      }
      ret = gst_base_src_wait_playing (GST_BASE_SRC_CAST (psrc));
      if (ret != GST_FLOW_OK) break;
      else continue;
    }

    if (GST_BUFFER_SIZE (*buf) == 2 && dsp->cn_counter-- != 0) {
        GST_LOG ("Dropping duplicate CN frame, counter = %d", dsp->cn_counter);
        continue;
    }
    else {
      gst_buffer_set_caps (*buf, GST_PAD_CAPS (GST_BASE_SRC_PAD (psrc)));
      if (GST_BUFFER_FLAG_IS_SET (*buf, GST_BUFFER_FLAG_LAST)) {
        GST_BUFFER_FLAG_UNSET (*buf, GST_BUFFER_FLAG_LAST);
        GST_LOG ("CN buffer created");
      }
      else {
        GST_LOG ("Audio buffer created");
      }
      dsp->cn_counter = 4;
      break;
    }
  }

  return ret;
}


/*
 * gst_dspg729src_start:
 * @bsrc: GstBaseSrc
 *
 * Start the operation. Initializes DSP node
 *
 * Returns: TRUE on success
 */
static gboolean
gst_dspg729src_start (GstBaseSrc * bsrc)
{
  GST_INFO("gst_dspg729src_start");
  GstDSPG729Src *dsp = GST_DSPG729SRC(bsrc);
  guint format, sample_rate;

  if (gst_dspaudio_ed_microphone(dsp->audio, TRUE) == FALSE) {
    GST_ELEMENT_ERROR (dsp, RESOURCE,
                       OPEN_WRITE, (NULL),
                       ("gst_dspaudio_ed_microphone"));
    return FALSE;
  }

  if (gst_dspaudio_aep_initialize(dsp->audio) == FALSE) {
    gst_dspaudio_ed_microphone(dsp->audio, FALSE);
    GST_ELEMENT_ERROR (dsp, RESOURCE,
                       OPEN_READ_WRITE, (NULL),
                       ("gst_dspaudio_aep_initialize"));
    return FALSE;
  }

  if (gst_dspaudio_open(dsp->audio, devname) == FALSE) {
    gst_dspaudio_ed_microphone(dsp->audio, FALSE);
    GST_ELEMENT_ERROR (dsp, RESOURCE,
                       OPEN_READ_WRITE, (NULL),
                       ("gst_dspaudio_open"));
    return FALSE;
  }

  if(!gst_dspaudio_map_format (dsp->audio, GST_DSPAUDIO_CODEC_G729,
    G_LITTLE_ENDIAN, TRUE, 0, &format, NULL)) {
    GST_ERROR("Cannot determine audio format");
    return FALSE;
  }
  if(!gst_dspaudio_map_samplerate (8000,&sample_rate)) {
    GST_WARNING ("unsupported sample rate: 8000");
    return FALSE;
  }

  // Read the junk out...
  gst_dspaudio_flush (dsp->audio);

  if (!gst_dspaudio_set_speechparams (dsp->audio,
    format, sample_rate, 80 /* 10ms */)) {
    return FALSE;
  }
  dsp->audio->mode = DSP_MODE_INITIALIZED;
  dsp->cn_counter = 0;
  gst_dspaudio_play(dsp->audio);
  GST_DEBUG("gst_dspg729src_start OK");
  return TRUE;
}


/*
 * gst_dspg729src_stop:
 * @bsrc: GstBaseSrc
 *
 * Stop the operation.
 *
 * Returns: TRUE on success
 */
static gboolean
gst_dspg729src_stop (GstBaseSrc * bsrc)
{
  GST_INFO ("gst_dspg729src_stop");
  GstDSPG729Src *dsp = GST_DSPG729SRC (bsrc);

  gst_dspaudio_remove_interrupt (dsp->audio);

  if (dsp->audio->all_frames_read || dsp->audio->read_sent) {
    GST_DEBUG("waiting reply for CMD_DATA_READ");
    gst_dspaudio_wait_buffer (dsp->audio);
    GST_DEBUG("got it");
  }

  gst_dspaudio_dtmf_tone_stop (dsp->audio);

  gst_dspaudio_stop(dsp->audio);
  gst_dspaudio_ed_microphone (dsp->audio, FALSE);
  gst_dspaudio_close (dsp->audio);
  gst_dspaudio_aep_close (dsp->audio);
  gst_dspaudio_reset (dsp->audio);
  GST_DEBUG ("gst_dspg729src_stop OK");
  return TRUE;
}


/*
 * gst_dspg729src_set_property:
 * @object: GObject where the property should be retrieved
 * @prop_id: Unique identifier of the desired property
 * @value: Pointer to a variable where the value should be put
 * @pspec: Parameter specification
 *
 * Stores the property value into GObject
 */
static void
gst_dspg729src_set_property (GObject * object,
                             guint prop_id,
                             const GValue * value,
                             GParamSpec * pspec)
{
  GstDSPG729Src *dsp = GST_DSPG729SRC (object);

  if (!gst_dspaudio_set_property (dsp->audio, prop_id, value)) {
    GST_WARNING ("Trying to set illegal property");
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
  }
}


/*
 * gst_dspg729src_get_property:
 * @object: GObject where the property should be retrieved
 * @prop_id: Unique identifier of the desired property
 * @value: Pointer to a variable where the value should be put
 * @pspec: Parameter specification
 *
 * Retrieves the desired parameter value from object
 */
static void
gst_dspg729src_get_property (GObject * object,
                             guint prop_id,
                             GValue * value,
                             GParamSpec * pspec)
{
  GstDSPG729Src *dsp = GST_DSPG729SRC (object);

  if (!gst_dspaudio_get_property (dsp->audio, prop_id, value)) {
    GST_WARNING ("Trying to get illegal property");
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
  }
}

/*
 * gst_dspg729src_event:
 * @src: GstBaseSrc object
 * @event: GstEvent to be handled
 *
 * Handles incoming events. Checks if the event is special "custom upstream
 * event" and in this case forwards it to dspaudio library for checking if
 * it is DTMF event.
 *
 * Returns: TRUE if the event was handled.
 */
static gboolean
gst_dspg729src_event (GstBaseSrc *src, GstEvent *event)
{
  GstEventType type = event ? GST_EVENT_TYPE (event) : GST_EVENT_UNKNOWN;
  GstDSPG729Src *dsp = GST_DSPG729SRC (src);
  gboolean ret = FALSE;

  if(type == GST_EVENT_CUSTOM_UPSTREAM) {
    ret = gst_dspaudio_check_upstream_event (dsp->audio, event);
  }
  return ret;
}


/*
 * gst_dspg729src_provide_clock:
 *
 */
static GstClock *
gst_dspg729src_provide_clock (GstElement * elem)
{
  GST_INFO("gst_dspg729src_provide_clock");
  GstDSPG729Src *dsp = GST_DSPG729SRC (elem);
  return GST_CLOCK_CAST (gst_object_ref (dsp->clock));
}


/*
 * gst_dspg729src_get_time:
 *
 */
static GstClockTime
gst_dspg729src_get_time (GstClock * clock, GstDSPG729Src * dsp)
{
  GstClockTime result;
  result = dsp->audio->in_framecount * G729_FRAMETIMEMILLIS * GST_MSECOND;
  return result;
}


/*
 * gst_dspg729src_voice_parametric_to_bitstream:
 * @in: input frame pointer
 * @out: pointer where to write the re-formatted frame
 *
 * Convert incoming DSP G.729 SID_UPDATE (parametric representation) into
 * "normal" G.729 Comfort Noise frame format (big endian).
 */
static inline void
gst_dspg729src_cn_parametric_to_bitstream (const short int *in, char *out)
{
#if G_BYTE_ORDER == G_LITTLE_ENDIAN
    out[0] = (in[0] << 7) |(in[1] << 2) | (in[2] >> 2);
    out[1] = ((in[2] & 3) << 6) |(in[3] << 1);
#else
    out[1] = (in[0] << 7) |(in[1] << 2) | (in[2] >> 2);
    out[0] = ((in[2] & 3) << 6) |(in[3] << 1);
#endif
}


/*
 * gst_dspg729src_voice_parametric_to_bitstream:
 * @in: input frame pointer
 * @out: pointer where to write the re-formatted frame
 *
 * Convert incoming DSP G.729 frame (parametric representation) into
 * "normal" G.729 voice frame format (big endian).
 */
static inline void
gst_dspg729src_voice_parametric_to_bitstream (const short int *in, char *out)
{
#if G_BYTE_ORDER == G_LITTLE_ENDIAN
    out[0] = in[0];
    out[1] = in[1] >> 2;
    out[2] = ((in[1] & 3) << 6) | (in[2] >> 2);
    out[3] = ((in[2] & 3) << 6) | (in[3] << 5) | (in[4] >> 8);
    out[4] = in[4] & 0xff;
    out[5] = (in[5] << 4) | (in[6] >> 3);
    out[6] = ((in[6] & 0x07) << 5) | in[7];
    out[7] = in[8] >> 5;
    out[8] = ((in[8] & 0x1f) << 3) | (in[9] >> 1);
    out[9] = ((in[9] & 1) << 7) | in[10];
#else
    out[3] = in[0];
    out[2] = in[1] >> 2;
    out[1] = ((in[1] & 3) << 6) | (in[2] >> 2);
    out[0] = ((in[2] & 3) << 6) | (in[3] << 5) | (in[4] >> 8);
    out[7] = in[4] & 0xff;
    out[6] = (in[5] << 4) | (in[6] >> 3);
    out[5] = ((in[6] & 0x07) << 5) | in[7];
    out[4] = in[8] >> 5;
    out[9] = ((in[8] & 0x1f) << 3) | (in[9] >> 1);
    out[8] = ((in[9] & 1) << 7) | in[10];
#endif
}


/*
 * gst_dspg729src_voice_frame_handler:
 * @mmap_ptr: Pointer to audio frame data
 * @fsize: Size of the incoming frame, in BYTES
 *
 * VOICE_FRAME frame handler function for G.729
 * Returns: GstBuffer containing the handled frame
 */
static GstBuffer *
gst_dspg729src_voice_frame_handler (short int *mmap_ptr, guint fsize)
{
    GstBuffer *newbuf = gst_buffer_new_and_alloc(10);
    char *data = GST_BUFFER_DATA (newbuf);
    gst_dspg729src_voice_parametric_to_bitstream (mmap_ptr, data);
    return newbuf;
}


/*
 * gst_dspg729src_sid_frame_handler:
 * @mmap_ptr: Pointer to audio frame data
 * @fsize: Size of the incoming frame, in BYTES
 *
 * SID_UPDATE frame handler function for G.729
 * Returns: GstBuffer containing the handled frame
 */
static GstBuffer *
gst_dspg729src_sid_frame_handler (short int *mmap_ptr, guint fsize)
{
    GstBuffer *newbuf = gst_buffer_new_and_alloc(2);
    char *data = GST_BUFFER_DATA (newbuf);
    gst_dspg729src_cn_parametric_to_bitstream (mmap_ptr, data);
    return newbuf;
}


/*
 * gst_dspg729src_unlock:
 * @src: GstBaseSrc
 *
 * Interrupts the blocking :create() function.
 *
 * Returns: TRUE
 */
static gboolean
gst_dspg729src_unlock (GstBaseSrc *src)
{
    GST_DEBUG ("unlock");
    GstDSPG729Src *dsp = GST_DSPG729SRC (src);
    gst_dspaudio_interrupt_render (dsp->audio);
    return TRUE;
}


/*
 * gst_dspg729src_unlock_stop:
 * @src: GstBaseSrc
 *
 * Removes :create() unlocking.
 *
 * Returns: TRUE
 */
static gboolean
gst_dspg729src_unlock_stop (GstBaseSrc *src)
{
    GST_DEBUG ("unlock_stop");
    GstDSPG729Src *dsp = GST_DSPG729SRC (src);
    gst_dspaudio_remove_interrupt (dsp->audio);
    return TRUE;
}
