/*
 * This file is part of libgst0.10-dsp
 *
 * Copyright (C) 2006 Nokia Corporation. All rights reserved.
 *
 * Contact: Stefan Kost <stefan.kost@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <unistd.h>
#include <string.h>
#include <gst/gst.h>

#include "gstdspg729sink.h"
#include "dspmixer.h"

// G.729 VOICE_FRAME size in bytes
#define G729_VOICE_FRAME_SIZE 10


GST_DEBUG_CATEGORY_STATIC (dspg729sink_debug);
#define GST_CAT_DEFAULT dspg729sink_debug

static gchar *devname = "/dev/dsptask/g729_dec";

static GstStaticPadTemplate dspg729sink_sink_template =
GST_STATIC_PAD_TEMPLATE ("sink",
    GST_PAD_SINK,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS ( "audio/g729, "
                      "rate = (int) 8000, "
                      "channels = (int) 1"
                    ) );

// Function prototypes

static void
gst_dspg729sink_class_init      (GstDSPG729SinkClass *klass);

static void
gst_dspg729sink_base_init       (gpointer g_class);

static void
gst_dspg729sink_init            (GstDSPG729Sink *dspmp3sink,
                                 GstDSPG729SinkClass *g_class);

static void
gst_dspg729sink_set_property    (GObject *object,
                                 guint prop_id,
                                 const GValue *value,
                                 GParamSpec *pspec);

static void
gst_dspg729sink_get_property    (GObject *object,
                                 guint prop_id,
                                 GValue *value,
                                 GParamSpec *pspec);

static gboolean
gst_dspg729sink_start           (GstBaseSink *bsrc);

static gboolean
gst_dspg729sink_stop            (GstBaseSink *bsrc);

static GstFlowReturn
gst_dspg729sink_render          (GstBaseSink *sink,
                                 GstBuffer *buffer);

static gboolean
gst_dspg729sink_unlock          (GstBaseSink *src);

static gboolean
gst_dspg729sink_unlock_stop     (GstBaseSink *src);


GST_IMPLEMENT_DSPMIXER_METHODS (GstDSPG729Sink, gst_dspg729sink);


/*
 * gst_dspg729sink_do_init:
 * @type:
 *
 *
 */
static void
gst_dspg729sink_do_init (GType type)
{
  static const GInterfaceInfo iface_info = {
    (GInterfaceInitFunc) gst_dspg729sink_interface_init,
    NULL,
    NULL,
  };
  static const GInterfaceInfo mixer_iface_info = {
    (GInterfaceInitFunc) gst_dspg729sink_mixer_interface_init,
    NULL,
    NULL,
  };

  gst_dspaudio_init();
  GST_DEBUG_CATEGORY_INIT (dspg729sink_debug, "dspg729sink", 0,
                           "DSP G.729 audio sink");
  g_type_add_interface_static (type, GST_TYPE_IMPLEMENTS_INTERFACE, &iface_info);
  g_type_add_interface_static (type, GST_TYPE_MIXER, &mixer_iface_info);
}

GST_BOILERPLATE_FULL (GstDSPG729Sink, gst_dspg729sink, GstBaseSink,
                      GST_TYPE_BASE_SINK, gst_dspg729sink_do_init);


/*
 * gst_dspg729sink_dispose:
 * @object: GObject pointer to element to be deleted
 *
 * Deletes G729 sink element instance. Called automatically by
 * GLib framework when element needs to be disposed.
 */
static void
gst_dspg729sink_dispose (GObject * object)
{
  GstDSPG729Sink *dsp = (GstDSPG729Sink *) object;

  if(dsp->audio) {
    gst_dspaudio_destroy(dsp->audio);
    dsp->audio = NULL;
  }

  G_OBJECT_CLASS (parent_class)->dispose (object);
}


/*
 * gst_dspg729sink_base_init:
 * @g_class: G729 sink class
 *
 * Does the basic initialization of G729 sink element class. This is
 * called by GStreamer framework.
 */
static void
gst_dspg729sink_base_init (gpointer g_class)
{
  GstElementClass *gstelement_class = GST_ELEMENT_CLASS (g_class);
  static const GstElementDetails details = GST_ELEMENT_DETAILS (
    "DSP G.729 Sink",
    "Sink/Audio",
    "G.729 audio sink",
    "Makoto Sugano <makoto.sugano@nokia.com>");

  gst_element_class_add_pad_template (gstelement_class,
      gst_static_pad_template_get (&dspg729sink_sink_template));
  gst_element_class_set_details (gstelement_class, &details);

}

/*
 * gst_dspg729sink_class_init:
 * @klass: G729 Sink class object
 *
 * Initializes G729 sink element class. This is called by GStreamer
 * framework.
 */
static void
gst_dspg729sink_class_init (GstDSPG729SinkClass * klass)
{
  GObjectClass *gobject_class;
  GstBaseSinkClass *gstbase_sink_class;

  gobject_class = (GObjectClass *) klass;
  gstbase_sink_class = (GstBaseSinkClass *) klass;

  gobject_class->set_property = GST_DEBUG_FUNCPTR (gst_dspg729sink_set_property);
  gobject_class->get_property = GST_DEBUG_FUNCPTR (gst_dspg729sink_get_property);
  gobject_class->dispose      = GST_DEBUG_FUNCPTR (gst_dspg729sink_dispose);

  // Use dspaudio utility to install default properties
  gst_dspaudio_install_sink_properties (G_OBJECT_CLASS (klass));

  gstbase_sink_class->start       = GST_DEBUG_FUNCPTR (gst_dspg729sink_start);
  gstbase_sink_class->stop        = GST_DEBUG_FUNCPTR (gst_dspg729sink_stop);
  gstbase_sink_class->render      = GST_DEBUG_FUNCPTR (gst_dspg729sink_render);
  gstbase_sink_class->unlock      = GST_DEBUG_FUNCPTR (gst_dspg729sink_unlock);
  gstbase_sink_class->unlock_stop = GST_DEBUG_FUNCPTR (gst_dspg729sink_unlock_stop);
}


/*
 * gst_dspg729sink_init:
 * @dsp: DSP G729 sink object
 * @g_class: G729 sink Class object
 *
 * Initializes the given G729 sink element instance and allocates required
 * resources. This also creates an audio stream object. This is
 * called by GStreamer framework.
 */
static void
gst_dspg729sink_init (GstDSPG729Sink * dspg729sink, GstDSPG729SinkClass *g_class)
{
  dspg729sink->audio = gst_dspaudio_new();

  // DSP sinks do not need syncing
  gst_base_sink_set_sync (GST_BASE_SINK (dspg729sink), FALSE);
}


/*
 *
 *
 */
static inline void
gst_dspg729sink_voice_bitstream_to_parametric (const char *in, short int *out)
{
#if G_BYTE_ORDER == G_LITTLE_ENDIAN
    out[0] = in[0];                                              // L0, L1
    out[1] = ((short int)in[1] << 2) | (in[2] & 0xc0) >> 6;      // L2, L3
    out[2] = ((in[2] & 0x3f) << 2) | (in[3] >> 6);               // P1
    out[3] = (in[3] & 0x20) >> 5;                                // P0
    out[4] = (((short int)(in[3] & 0x1f)) << 8) | in[4];         // C1
    out[5] = (in[5] & 0xf0) >> 4;                                // S1
    out[6] = ((in[5] & 0x0f) << 3) | ((in[6] & 0xe0) >> 5);      // GA1, GB1
    out[7] = in[6] & 0x1f;                                       // P2
    out[8] = (((short int) in[7]) << 5) | ((in[8] & 0xf8) >> 3); // C2
    out[9] = ((in[8] & 7) << 1) | ((in[9] & 0x80) >> 7);         // S2
    out[10] = (in[9] & 0x7F);                                    // GA2, GB2
#else
    out[0] = in[3];                                              // L0, L1
    out[1] = ((short int)in[2] << 2) | (in[1] & 0xc0) >> 6;      // L2, L3
    out[2] = ((in[1] & 0x3f) << 2) | (in[0] >> 6);               // P1
    out[3] = (in[0] & 0x20) >> 5;                                // P0
    out[4] = (((short int)(in[0] & 0x1f)) << 8) | in[7];         // C1
    out[5] = (in[6] & 0xf0) >> 4;                                // S1
    out[6] = ((in[6] & 0x0f) << 3) | ((in[5] & 0xe0) >> 5);      // GA1, GB1
    out[7] = in[5] & 0x1f;                                       // P2
    out[8] = (((short int) in[4]) << 5) | ((in[9] & 0xf8) >> 3); // C2
    out[9] = ((in[9] & 7) << 1) | ((in[8] & 0x80) >> 7);         // S2
    out[10] = (in[8] & 0x7F);                                    // GA2, GB2
#endif
}


/*
 *
 *
 */
static inline void
gst_dspg729sink_cn_bitstream_to_parametric (const char *in, short int *out)
{
#if G_BYTE_ORDER == G_LITTLE_ENDIAN
    out[0] = (in[0] & 0x80) >> 7;
    out[1] = (in[0] & 0x7c) >> 2;
    out[2] = ((in[0] & 3) << 2) | ((in[1] & 0xc0) >> 6);
    out[3] = (in[1] & 0x3e) >> 1;
#else
    out[0] = (in[1] & 0x80) >> 7;
    out[1] = (in[1] & 0x7c) >> 2;
    out[2] = ((in[1] & 3) << 2) | ((in[0] & 0xc0) >> 6);
    out[3] = (in[0] & 0x3e) >> 1;
#endif
}


/*
 * gst_dspg729sink_render:
 * sink: GstBaseSink
 * @buffer: GstBuffer to be consumed
 *
 * Plays the given audio buffer
 */
static GstFlowReturn
gst_dspg729sink_render(GstBaseSink *sink, GstBuffer * buffer)
{
  GstFlowReturn ret = GST_FLOW_OK;
  GstDSPG729Sink *dsp = GST_DSPG729SINK (sink);
  const guint8 *data = GST_BUFFER_DATA(buffer);
  guint avail = GST_BUFFER_SIZE (buffer);
  DSPWaitStatus status;
  guint dataindex = 0, data_size = 0;

  GST_LOG ("Render: %d bytes, timestamp %lld, discont = %s",
           GST_BUFFER_SIZE(buffer), GST_BUFFER_TIMESTAMP(buffer),
           GST_BUFFER_FLAG_IS_SET(buffer, GST_BUFFER_FLAG_DISCONT) ?
                "TRUE" : "FALSE");

  // FIXME: Try to replace this with gst_dspaudio_write_frame()

  do {
    status = gst_dspaudio_wait_buffer (dsp->audio);

    if (status == DSP_WAIT_INTERRUPT) {
      GST_DEBUG ("DSP_WAIT_INTERRUPT");
      ret = gst_base_sink_wait_preroll (sink);
      if (ret != GST_FLOW_OK)
        break;
    }
    else if (status == DSP_WAIT_ERROR) {
      GST_ELEMENT_ERROR (dsp, RESOURCE, WRITE, (NULL),
                         ("loop2: error cmd: %d status: %d",
                          dsp->audio->error_cmd,
                          dsp->audio->error_status));
      ret = GST_FLOW_ERROR;
      break;
    }

copyframe:

    dsp->mmap_ptr[data_size + 0] = 0; // BFI (Bad Frame Indication)

    if (avail >= G729_VOICE_FRAME_SIZE) {
        dsp->mmap_ptr[data_size + 1] = VOICE_FRAME; // DTX mode
        gst_dspg729sink_voice_bitstream_to_parametric (data+dataindex,
                &(dsp->mmap_ptr[data_size + 2]));
        data_size += (2 + 11); // BFI + DTX + data
        dataindex += G729_VOICE_FRAME_SIZE;
        avail -= G729_VOICE_FRAME_SIZE;
        goto copyframe;
    }
    else if (avail >= 2) {
        if (data[dataindex+0] == dsp->lastcn[0] &&
            data[dataindex+1] == dsp->lastcn[1])
        {
//            dsp->mmap_ptr[data_size + 1] = SID_FRAME_NO_UPDATE;
//            data_size += 2;
        }
        else {
//             dsp->mmap_ptr[data_size + 1] = SID_FRAME_UPDATE;
//             gst_dspg729sink_cn_bitstream_to_parametric (data+dataindex,
//                     &(dsp->mmap_ptr[data_size + 2]));
//             data_size += (2 + 4);        // BFI + DTX + SID
            dsp->lastcn[0] = data[dataindex+0];
            dsp->lastcn[1] = data[dataindex+1];
        }
        dataindex += 2;
        avail -= 2;
        goto copyframe;
    }

    GST_LOG ("copied data to mmap");

    if (data_size) {
      if (gst_dspaudio_write_data (dsp->audio, data_size, FALSE) == FALSE) {
        GST_ELEMENT_ERROR (dsp, RESOURCE, WRITE, (NULL), ("DSP write failed"));
        ret = GST_FLOW_ERROR;
        break;
      }
      GST_LOG ("%d bytes written to DSP", data_size * sizeof(short int));
    }
  }
  while (avail >= 2);

  return ret;
}


/*
 * gst_dspg729sink_start:
 * @bsink: GstBaseSink
 *
 * Open DSP node and start the playback
 *
 * Returns: TRUE on success
 */
static gboolean
gst_dspg729sink_start (GstBaseSink * bsink)
{
  GST_INFO ("gst_dspg729sink_start");
  GstDSPG729Sink *dsp = GST_DSPG729SINK (bsink);
  guint format, sample_rate;

  if (gst_dspaudio_open (dsp->audio, devname) == FALSE) {
    GST_ELEMENT_ERROR (dsp, RESOURCE,
                       OPEN_READ_WRITE, (NULL),
                       ("gst_dspaudio_open"));
    return FALSE;
  }

  // FIXME: is dsp->audio->dtxmode == GST_DSPAUDIO_DTX_MODE_ON
  //format = DSP_AFMT_G729_DTX;
  if(!gst_dspaudio_map_format (dsp->audio, GST_DSPAUDIO_CODEC_G729,
    G_LITTLE_ENDIAN, TRUE, 0, &format, NULL)) {
    GST_ERROR("Cannot determine audio format");
    return FALSE;
  }
  if(!gst_dspaudio_map_samplerate (8000,&sample_rate)) {
    GST_WARNING ("unsupported sample rate: 8000");
    return FALSE;
  }

  // Read the junk out...
  gst_dspaudio_flush (dsp->audio);

  if (!gst_dspaudio_set_speechparams (dsp->audio,
    format, sample_rate, 80 /* 10 ms */)) {
    return FALSE;
  }
  dsp->lastcn[0] = dsp->lastcn[1] = 0;
  dsp->mmap_ptr = (short int *) dsp->audio->codec.mmap_buffer;
  dsp->audio->mode = DSP_MODE_INITIALIZED;
  gst_dspaudio_update_dsp_settings (dsp->audio);
  gst_dspaudio_play (dsp->audio);
  GST_DEBUG ("gst_dspg729sink_start OK");
  return TRUE;
}


/*
 * gst_dspg729sink_stop:
 * @bsink: GstBaseSink
 *
 * Stop the operation. Uninitialize and close DSP.
 *
 * Returns: TRUE on success
 */
static gboolean
gst_dspg729sink_stop (GstBaseSink * bsink)
{
  GST_INFO ("gst_dspg729sink_stop");
  GstDSPG729Sink *dsp = GST_DSPG729SINK (bsink);

  gst_dspaudio_remove_interrupt (dsp->audio);
  gst_dspaudio_stop (dsp->audio);
  gst_dspaudio_close (dsp->audio);
  gst_dspaudio_reset (dsp->audio);
  GST_DEBUG ("gst_dspg729sink_stop OK");
  return TRUE;
}


/*
 * gst_dspg729sink_set_property:
 * @object: GObject where the property should be retrieved
 * @prop_id: Unique identifier of the desired property
 * @value: Pointer to a variable where the value should be put
 * @pspec: Parameter specification
 *
 * Stores the property value into GObject
 */
static void
gst_dspg729sink_set_property(GObject * object,
                            guint prop_id,
                            const GValue * value,
                            GParamSpec * pspec)
{
  GstDSPG729Sink *dsp = GST_DSPG729SINK (object);

  if(!gst_dspaudio_set_property(dsp->audio, prop_id, value)) {
    GST_WARNING("Trying to set illegal property");
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
  }
}


/*
 * gst_dspg729sink_get_property:
 * @object: GObject where the property should be retrieved
 * @prop_id: Unique identifier of the desired property
 * @value: Pointer to a variable where the value should be put
 * @pspec: Parameter specification
 *
 * Retrieves the desired parameter value from object
 */
static void
gst_dspg729sink_get_property(GObject * object,
                            guint prop_id,
                            GValue * value,
                            GParamSpec * pspec)
{
  GstDSPG729Sink *dsp = GST_DSPG729SINK (object);

  if(!gst_dspaudio_get_property(dsp->audio, prop_id, value)) {
    GST_WARNING("Trying to get illegal property");
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
  }
}


/*
 * gst_dspg729sink_unlock:
 * @src: GstBaseSink
 *
 * Interrupts the blocking :render() function.
 *
 * Returns: TRUE
 */
static gboolean
gst_dspg729sink_unlock (GstBaseSink *src)
{
    GST_DEBUG ("unlock");
    GstDSPG729Sink *dsp = GST_DSPG729SINK (src);
    gst_dspaudio_interrupt_render (dsp->audio);
    return TRUE;
}


/*
 * gst_dspg729sink_unlock_stop:
 * @src: GstBaseSink
 *
 * Removes :render() unlocking.
 *
 * Returns: TRUE
 */
static gboolean
gst_dspg729sink_unlock_stop (GstBaseSink *src)
{
    GST_DEBUG ("unlock_stop");
    GstDSPG729Sink *dsp = GST_DSPG729SINK (src);
    gst_dspaudio_remove_interrupt (dsp->audio);
    return TRUE;
}

