/*
 * This file is part of libgst0.10-dsp
 *
 * Copyright (C) 2007 Nokia Corporation. All rights reserved.
 *
 * Contact: Stefan Kost <stefan.kost@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

/*
 * gst_dspaudio_open_node:
 * @dsp: DSP node info object
 * @devname: device name to be opened & initialized
 *
 * Initializes given DSP node.
 *
 * Returns: TRUE if operation was successful, otherwise FALSE
 */
static gboolean
gst_dspaudio_open_node (DSPNodeInfo *info, const gchar *devname)
{
  g_return_val_if_fail (info, FALSE);
  g_return_val_if_fail (devname, FALSE);

  info->fd = open ("/dev/zero", O_RDWR);
  if (info->fd == -1) {
    GST_ERROR ("Opening %s failed", devname);
    return FALSE;
  }

  info->bridge_buffer_size = 1024;
  info->mmap_buffer_size = 1024 * sizeof (short int);

  info->mmap_buffer = (char *)
      mmap ((void *) 0, info->mmap_buffer_size, PROT_READ | PROT_WRITE,
            MAP_SHARED, info->fd, 0);

  if (info->mmap_buffer == NULL) {
    GST_ERROR ("Cannot mmap data buffer");
    return FALSE;
  }

  return TRUE;
}

/*
 * gst_dspaudio_close_node:
 * @dsp: DSP node info object
 *
 * Closes the specific DSP node
 *
 * Returns: TRUE if operation was successful, otherwise FALSE
 */
static gboolean
gst_dspaudio_close_node (DSPNodeInfo *info)
{
  g_return_val_if_fail (info, FALSE);

  // We may get called even when DSP is not opened yet
  if (info->fd != -1) {
    if (info->mmap_buffer) {
      munmap (info->mmap_buffer, info->mmap_buffer_size);
      info->mmap_buffer = NULL;
    }

    // we need to give DSP some time to send the last messages
    usleep (200000);

    // flush the message buffer once again to make sure it is empty
    gst_dspaudio_flush_unlocked (info);

    //ioctl (info->fd, OMAP_DSP_TASK_IOCTL_UNLOCK);
    close (info->fd);
    info->fd = -1;
  }
  return TRUE;
}


/*
 * gst_dspaudio_flush_unlocked:
 * @dsp: DSP node info object
 *
 * Reads all unread data from node. Caller needs to grab DSP lock first.
 */
void
gst_dspaudio_flush_unlocked (DSPNodeInfo *info)
{
}


/*
 * gst_dspaudio_set_audioparams:
 * @dsp: DSP audio object
 * @format: audio format constant
 * @sample_rate: sample rate constant
 * @channels: number of audio channels
 *
 * This sets the given audio parameters to audio node
 *
 * Returns: TRUE if operation was successful, otherwise FALSE
 */
gboolean
gst_dspaudio_set_audioparams (GstDSPAudio *dsp, guint format, guint sample_rate,
  guint channels)
{
  return TRUE;
}


/*
 * gst_dspaudio_set_speechparams:
 * @dsp: DSP audio object
 * @format: audio format constant
 * @sample_rate: sample rate constant
 * @frame_size: size of sample frames
 *
 * This sets the given audio parameters to audio node
 *
 * Returns: TRUE if operation was successful, otherwise FALSE
 */
gboolean
gst_dspaudio_set_speechparams (GstDSPAudio *dsp, guint format, guint sample_rate,
  guint frame_size)
{
  return TRUE;
}


/*
 * gst_dspaudio_discont:
 * @dsp: DSP audio object
 *
 * Informs the DSP about discontinuity of the stream. Must be used
 * when a seek operation is performed on a compressed stream.
 *
 * Returns: TRUE if the operation was successful, otherwise FALSE
 */
gboolean
gst_dspaudio_discont (GstDSPAudio *dsp)
{
  return TRUE;
}


/*
 * gst_dspaudio_set_volume:
 * @dsp: DSP audio object
 *
 * Sends the volume command to DSP. Volume value is retrieved from
 * "volume" attribute of the audio object. The value can be between
 * 0 - 0xFFFF.
 *
 * Returns: TRUE if the operation was successful, otherwise FALSE
 */
static gboolean
gst_dspaudio_set_volume (GstDSPAudio *dsp)
{
  return TRUE;
}


/*
 * gst_dspaudio_set_panning:
 * @dsp: DSP audio object
 *
 * Sends the balance command to DSP. Panning value is retrieved from
 * "panning" attribute of the audio object. This value must be between
 * -0x4000 (left panning) and +0x4000 (right panning)
 *
 * Returns: TRUE if the operation was successful, otherwise FALSE
 */
static gboolean
gst_dspaudio_set_panning (GstDSPAudio *dsp)
{
  return TRUE;
}


/*
 * gst_dspaudio_set_mute:
 * @dsp: DSP audio object
 *
 * Send MUTE/UNMUTE command to DSP. The value is retrieved from
 * "mute" attribute of the audio object.
 *
 * Returns: TRUE if the operation was successful, otherwise FALSE
 */
static gboolean
gst_dspaudio_set_mute (GstDSPAudio *dsp)
{
  return TRUE;
}


/*
 * gst_dspaudio_play:
 * @dsp: DSP audio object
 *
 * Sends a PLAY command to DSP. After this the DSP will start sending
 * DSP_CMD_DATA_WRITE requests, that can be read with wait_buffer ()
 * method.
 *
 * Returns: TRUE if the operation was successful, otherwise FALSE
 */
gboolean
gst_dspaudio_play (GstDSPAudio *dsp)
{
  return TRUE;
}


/*
 * gst_dspaudio_pause:
 * @dsp: DSP audio object
 *
 * Sends a PAUSE command to DSP.
 *
 * Returns: TRUE if the operation was successful, otherwise FALSE
 */
gboolean
gst_dspaudio_pause (GstDSPAudio *dsp)
{
  return TRUE;
}


/*
 * gst_dspaudio_stop:
 * @dsp: DSP audio object
 *
 * Send a STOP command to DSP. This will stop the audio playback and
 * set the audio object mode into DSP_MODE_STOPPED.
 *
 * Returns: TRUE if the operation was successful, otherwise FALSE
 */
gboolean
gst_dspaudio_stop (GstDSPAudio *dsp)
{
  return TRUE;
}


/*
 * gst_dspaudio_map_samplerate:
 * @rate: Source rate to be converted
 * @dest_rate: Pointer where to store DSP-specific rate definition
 *
 * Converts sample rate value into DSP-specific enumeration value
 *
 * Returns: TRUE if conversion was successful, otherwise FALSE
 */
gboolean
gst_dspaudio_map_samplerate (guint rate, guint *dest_rate)
{
  *dest_rate = 0;
  return TRUE;
}


/*
 * gst_dspaudio_map_channels:
 * @channels: Number of channels to convert
 * @dest_channels: Pointer where to store DSP-specific channels definition
 *
 * Converts channel number value into DSP-specific enumeration value
 *
 * Returns: TRUE if conversion was successful, otherwise FALSE
 */
gboolean
gst_dspaudio_map_channels (guint channels, guint *dest_channels)
{
  *dest_channels = 1;

  return TRUE;
}


/*
 * gst_dspaudio_map_format:
 * @dsp: DSP audio object
 * @codec: codec type
 * @endianness: endianness of the sample (G_BIG_ENDIAN / G_LITTLE_ENDIAN)
 * @sign: sign mode. FALSE=unsigned, TRUE=signed
 * @width: Width of the samples. Should be 8 or 16
 * @format: Pointer to a variable where format value should be placed
 * @bps: Pointer to a variable where to put bytes-per-sample value (or %NULL if not needed)
 *
 * Translates various audio-related information into one format definition
 *
 * Returns: TRUE if operation was successful, otherwise FALSE
 */
gboolean
gst_dspaudio_map_format (GstDSPAudio *dsp, GstDSPAudioCodec codec, gint endianness,
    gboolean sign, gint width, guint * format, guint * bps)
{
  if(bps) *bps = 1;
  *format = 0;
  return TRUE;
}

/*
 * gst_dspaudio_get_info_unlocked:
 * @dsp: DSP audio object
 *
 * Send a status query to DSP. The status reply needs to be
 * read with _wait_buffer () or _read_cmd () call.
 *
 * Returns: TRUE, if operation was successful. Otherwise FALSE
 */
gboolean
gst_dspaudio_get_info_unlocked (GstDSPAudio *dsp)
{
  return TRUE;
}


/*
 * gst_dspaudio_read_data:
 * @dsp: DSPAudio object
 * @data_size: Amount of data in MMAP buffer (number of short ints)
 * @eof_mode: Inform DSP that this is the last audio data chunk 
 *
 * Informs DSP that application wants to read more data by sending
 * a DSP_CMD_DATA_READ command. If eof_mode was selected, DSP_CMD_EOF will
 * be sent instead.
 *
 * Returns: TRUE if operation was successful
 */
gboolean
gst_dspaudio_read_data (GstDSPAudio *dsp, guint data_size, guint8 *ptr)
{
  g_return_val_if_fail (dsp, FALSE);
  g_return_val_if_fail (dsp->mode != DSP_MODE_ERROR, FALSE);

  return TRUE;
}


/*
 * gst_dspaudio_read_frame:
 * @dsp: DSPAudio object
 * @voice_framesize: size of the VOICE_FRAME to be read
 * @update_framesize: size of the SID_FRAME_UPDATE (CN) frame
 * @frame_duration_millis: duration of one frame in milliseconds
 *
 * Reads one DTX audio frame from DSP. CN frames will be marked with
 * GST_BUFFER_FLAG_LAST-flag.
 *
 * Returns: GstBuffer that contains the frame, or NULL. If NULL is
 * returned, it means that either an error is occured, or pipeline is
 * set to PAUSED. On error the GstDSPAudio::mode will be set to DSP_MODE_ERROR.
 */
GstBuffer *
gst_dspaudio_read_frame (GstDSPAudio *dsp,
                         guint voice_framesize,
                         guint update_framesize,
                         guint frame_duration_millis)
{
  g_return_val_if_fail (dsp, FALSE);
  g_return_val_if_fail (dsp->mode != DSP_MODE_ERROR, FALSE);

  return NULL;
}


/*
 * gst_dspaudio_write_data:
 * @dsp: DSPAudio object
 * @data_size: Amount of data in MMAP buffer (number of short ints)
 * @eof_mode: Inform DSP that this is the last audio data chunk 
 *
 * Informs DSP that new audio data has been copied to MMAP buffer by sending
 * a DSP_CMD_DATA_WRITE command. If eof_mode was selected, DSP_CMD_EOF will
 * be sent instead.
 *
 * Returns: TRUE if operation was successful
 */
gboolean
gst_dspaudio_write_data (GstDSPAudio *dsp, guint data_size, gboolean eof_mode)
{
  g_return_val_if_fail (dsp, FALSE);
  g_return_val_if_fail (dsp->mode != DSP_MODE_ERROR, FALSE);

  return TRUE;
}


/*
 * gst_dspaudio_write_frame:
 * @dsp: DSPAudio object
 * @buffer: Audio data buffer to be rendered
 * @framesize: size of the VOICE_FRAME to be read
 * @update_framesize: size of the SID_FRAME_UPDATE frame
 * @use_lfi: Set to TRUE if codec requires Last Frame Indicator flags
 *
 * Writes DTX audio frame(s) to DSP.
 *
 * Returns: GST_FLOW_OK if operation was successful, GST_FLOW_WRONG_STATE if
 * interrupted and GST_FLOW_ERROR otherwise
 */
GstFlowReturn
gst_dspaudio_write_frame (GstDSPAudio *dsp, GstBuffer *buffer,
                          guint framesize, guint update_framesize,
                          gboolean use_lfi)
{
  g_return_val_if_fail (dsp, FALSE);
  g_return_val_if_fail (buffer, FALSE);

  return GST_FLOW_OK;
}


/*
 * gst_dspaudio_get_dsp_time:
 * @dsp: DSPAudio object
 *
 * Reads the current time from DSP.
 *
 * Returns: GstClockTime representing DSP time, or GST_CLOCK_TIME_NONE if
 * something went wrong.
 */
static GstClockTime
gst_dspaudio_get_dsp_time (GstDSPAudio *dsp)
{
  return GST_CLOCK_TIME_NONE;
}


/*
 * gst_dspaudio_map_dtmf:
 * @tone_id: DTMF id to be mapped.
 * @dsp_tone_id: output parameter for DSP tone ID
 *
 * Map the given character to DSP tone ID. The DTMF codes can be found
 * from http://www.iana.org/assignments/audio-telephone-event-registry
 *
 * Returns: TRUE if mapping found, otherwise FALSE.
 */
gboolean
gst_dspaudio_map_dtmf(gint tone_id, unsigned short int *dsp_tone_id)
{
  return TRUE;
}


/*
 * gst_dspaudio_dtmf_start_unlocked:
 * @dsp: DSP audio object
 * @tone_char: DTMF tone character to be sent (start) or -1 (stop)
 *
 * Send DTMF_START command to DSP. Caller need to grab DSP lock first
 *
 * Returns: TRUE if the operation was successful, otherwise FALSE
 */
gboolean
gst_dspaudio_dtmf_tone_start_unlocked (GstDSPAudio *dsp, gint tone_id)
{
  return TRUE;
}


/*
 * gst_dspaudio_dtmf_stop_unlocked:
 * @dsp: DSP audio object
 *
 * Send DTMF_STOP command to DSP. Caller need to grab DSP lock first
 *
 * Returns: TRUE if the operation was successful, otherwise FALSE
 */
gboolean
gst_dspaudio_dtmf_tone_stop_unlocked (GstDSPAudio *dsp)
{
  return TRUE;
}


