/*
 * This file is part of libgst0.10-dsp
 *
 * Copyright (C) 2007 Nokia Corporation. All rights reserved.
 *
 * Contact: Stefan Kost <stefan.kost@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#ifndef __GST_DSP_AUDIO_H__
#define __GST_DSP_AUDIO_H__

#include <glib.h>
#include <gst/gst.h>
#include <gst/audio/gstaudioclock.h>
#include <gst/interfaces/mixer.h>
#include <sys/select.h>

#include "omapdsp.h"
#ifdef HAVE_DSP_H
#include <dsp/audio_dsptask.h>
#endif

G_BEGIN_DECLS

#ifndef MIN
#define MIN(a,b) ((a)<(b)) ? (a) : (b)
#endif

#define DSP_VOLUME_DEFAULT_INT  0xFFFF
#define DSP_VOLUME_MAX_INT      0xFFFF

#define DSP_VOLUME_DEFAULT_FLOAT  1.0
#define DSP_VOLUME_MAX_FLOAT      1.0

#define DSP_PANNING_MIN     -0x4000
#define DSP_PANNING_MAX      0x4000
#define DSP_PANNING_DEFAULT  0

#define DSP_PRIORITY_MIN     0
#define DSP_PRIORITY_MAX     5
#define DSP_PRIORITY_DEFAULT 0


#define SID_FRAME_NO_UPDATE       0           /* SID params no update        */
#define VOICE_FRAME               1           /* data is encoded speech      */
#define SID_FRAME_UPDATE          2           /* SID params to be updated    */

// Internal pipe communication "protocol" commands
#define PIPE_CMD_PROP_UPDATE      1
#define PIPE_CMD_PAUSE            2
#define PIPE_CMD_INTERRUPT        3
#define PIPE_CMD_NEED_LOCK        4

#define AEP_DEVICE_NAME "/dev/dsptask/aep"

#define GST_TYPE_DSPAUDIO_DTX_MODE (gst_dspaudio_dtx_mode_get_type())

typedef enum {
  GST_DSPAUDIO_DTX_MODE_OFF,
  GST_DSPAUDIO_DTX_MODE_ON_NOSEND,
  GST_DSPAUDIO_DTX_MODE_ON_SEND_EMPTY,
  GST_DSPAUDIO_DTX_MODE_ON_SEND_CN,
} GstDSPAudioDTXMode;


typedef enum
{
  DSP_WAIT_ERROR,
  DSP_WAIT_OK,
  DSP_WAIT_INTERRUPT,
  DSP_WAIT_RESET,
} DSPWaitStatus;


typedef enum
{
  DSPAUDIO_PROP_0,
  DSPAUDIO_PROP_VOLUME,
  DSPAUDIO_PROP_VOLUME_FLOAT,
  DSPAUDIO_PROP_PANNING,
  DSPAUDIO_PROP_MUTE,
  DSPAUDIO_PROP_PRIORITY,
  DSPAUDIO_PROP_DTXMODE,
  DSPAUDIO_PROP_LAST
} DSPAudioProperties;

typedef enum
{
  GST_DSPAUDIO_CODEC_RAW=0,
  GST_DSPAUDIO_CODEC_MULAW,
  GST_DSPAUDIO_CODEC_ALAW,
  GST_DSPAUDIO_CODEC_ILBC,
  GST_DSPAUDIO_CODEC_G729,
  GST_DSPAUDIO_CODEC_AMR,
  /*
#define DSP_AFMT_MP2          0x0C // MP2 stream.
#define DSP_AFMT_MP3          0x09 // MP3 stream.
#define DSP_AFMT_AAC          0x0A // AAC stream.
*/
} GstDSPAudioCodec;

typedef struct _GstDSPAudio GstDSPAudio;
typedef struct _DSPNodeInfo DSPNodeInfo;


struct _DSPNodeInfo {
  int            fd;
  guint          bridge_buffer_size;  // in bytes
  guint          mmap_buffer_size;    // in bytes
  char *         mmap_buffer;
};


struct _GstDSPAudio {
  GMutex         *dsp_mutex;
  GMutex         *prop_mutex;
  GMutex         *clock_mutex;
  GCond          *dsp_cond;      // When some other thread "borrows" DSP lock,
                                 // this cond is used to signal the main thread
                                 // that it can continue
  DSPNodeInfo    codec;
  DSPNodeInfo    aep;

  /* clock */
  GstClockTime   clock_time;
  GstClockTime   last_dsp_update;
  guint64        in_framecount;

  /* Property-related */
  guint          volume;
  gdouble        volume_f;
  gboolean       volume_changed;
  gint           panning;
  guint          priority;
  gboolean       panning_changed;
  gboolean       mute;
  gboolean       mute_changed;

  /* DSP communication */
  guint          rw_pending;
  guint          out_index;
  gint           error_status;
  guint          error_cmd;
  guint          mode;
  gboolean       discont_sent;
  gboolean       read_sent;
  gboolean       interrupted;
  gboolean       dtmf_playing;
  gboolean       eof_received;
  gboolean       reset_needed;

  // DTX-mode specific
  GstBuffer          *lastcn;
  GstDSPAudioDTXMode dtxmode;
  gboolean           all_frames_read;
  gboolean           silence_detected;
  guint              readindex;

#ifdef HAVE_DSP_H
  AUDIO_STATUS_INFO info;
  DSP_DATA_WRITE    outbuf;
#endif

  short int      dsp_buffer_free;
  fd_set         read_fdset;
  int            pipe_fds[2];

  struct {
    unsigned short int frame_size;
    unsigned int stream_time_ms;
  } readinfo;

  GList          *tracklist;

  /* For resource management */
  gpointer       *dbus_connection;

  /* Audio frame handlers */
  GstBuffer *    (*handle_voice_frame) (short int *, guint);
  GstBuffer *    (*handle_sid_frame) (short int *, guint);
};

// Member functions

GType
gst_dspaudio_dtx_mode_get_type ();

void
gst_dspaudio_init (void);

GstDSPAudio*
gst_dspaudio_new ();

void
gst_dspaudio_reset (GstDSPAudio *dsp);

void
gst_dspaudio_destroy (GstDSPAudio *dsp);

gboolean
gst_dspaudio_open (GstDSPAudio *dsp, const gchar *devname);

gboolean
gst_dspaudio_close (GstDSPAudio *dsp);

//gboolean
//gst_dspaudio_setparams (GstDSPAudio *dsp, const char *data, guint datasize);

gboolean
gst_dspaudio_set_audioparams (GstDSPAudio *dsp, guint format, guint sample_rate,
  guint channels);

gboolean
gst_dspaudio_set_speechparams (GstDSPAudio *dsp, guint format, guint sample_rate,
  guint frame_size);

gboolean
gst_dspaudio_play (GstDSPAudio *dsp);

gboolean
gst_dspaudio_pause (GstDSPAudio *dsp);

gboolean
gst_dspaudio_stop (GstDSPAudio *dsp);

gboolean
gst_dspaudio_discont (GstDSPAudio *dsp);

gboolean
gst_dspaudio_map_samplerate (guint rate, guint *dest_rate);

gboolean
gst_dspaudio_map_channels (guint channels, guint *dest_channels);

gboolean
gst_dspaudio_map_format (GstDSPAudio *dsp, GstDSPAudioCodec codec, gint endianness,
                         gboolean sign, gint width, guint * format, guint * bps);

DSPWaitStatus
gst_dspaudio_wait_buffer (GstDSPAudio *dsp);

DSPWaitStatus
gst_dspaudio_read_cmd (GstDSPAudio *dsp, gint timeout_millis);

void
gst_dspaudio_flush (GstDSPAudio *dsp);

void
gst_dspaudio_wait_eos (GstDSPAudio *dsp);

void
gst_dspaudio_install_general_properties (GObjectClass *klass);

void
gst_dspaudio_install_sink_properties (GObjectClass *klass);

void
gst_dspaudio_install_src_properties (GObjectClass *klass);

gboolean
gst_dspaudio_set_property (GstDSPAudio *dsp,
                           guint prop_id,
                           const GValue *value);

gboolean
gst_dspaudio_get_property (GstDSPAudio *dsp,
                           guint prop_id,
                           GValue *value);

gboolean
gst_dspaudio_get_info (GstDSPAudio *dsp);

void
gst_dspaudio_update_dsp_settings (GstDSPAudio *dsp);

gboolean
gst_dspaudio_ed_microphone (GstDSPAudio *dsp, gboolean mic_enable);

void
gst_dspaudio_interrupt_render (GstDSPAudio *dsp);

void
gst_dspaudio_remove_interrupt (GstDSPAudio *dsp);

gboolean
gst_dspaudio_dtmf_tone_start (GstDSPAudio *dsp, gint tone_id);

gboolean
gst_dspaudio_dtmf_tone_stop (GstDSPAudio *dsp);

gboolean
gst_dspaudio_check_upstream_event (GstDSPAudio *dsp, GstEvent *event);

guint
gst_dspaudio_round_bsize (guint bsize, guint min, guint max);

gboolean
gst_dspaudio_activate_push (GstPad * pad, gboolean active);

// ===================================================
// == AEP ============================================
// ===================================================

gboolean
gst_dspaudio_aep_initialize (GstDSPAudio *dsp);

gboolean
gst_dspaudio_aep_close (GstDSPAudio *dsp);

// ===================================================
// == Clock Helper Function ==========================
// ===================================================

GstClockTime
gst_dspaudio_info_to_time (GstDSPAudio *dsp);

extern GstStateChangeReturn
gst_dspaudio_clock_change_state (GstDSPAudio *dsp, GstStateChange transition);

extern GstClockTime
gst_dspaudio_get_dsp_clock (GstDSPAudio *dsp, gboolean use_jitter);

// ===================================================
// == Reading/Writing audio  =========================
// ===================================================

gboolean
gst_dspaudio_read_data  (GstDSPAudio *dsp,
                         guint data_size,
                         guint8 *ptr);

GstBuffer *
gst_dspaudio_read_frame (GstDSPAudio *dsp,
                         guint framesize,
                         guint update_frame_size,
                         guint frame_duration_millis);

GstFlowReturn
gst_dspaudio_write_frame (GstDSPAudio *dsp,
                          GstBuffer *buffer,
                          guint framesize,
                          guint update_framesize,
                          gboolean use_lfi);

gboolean
gst_dspaudio_write_data  (GstDSPAudio *dsp,
                          guint data_size,
                          gboolean eof_mode);

// ===================================================
// == GstMixer interface related =====================
// ===================================================

const GList *
gst_dspaudio_create_track_list (GstDSPAudio *dsp);


G_END_DECLS


#endif
