/*
 * This file is part of libgst0.10-dsp
 *
 * Copyright (C) 2006 Nokia Corporation. All rights reserved.
 *
 * Contact: Stefan Kost <stefan.kost@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <unistd.h>
#include <string.h>
#include <gst/gst.h>

#include "gstdspaacsink.h"
#include "dspmixer.h"

#define AUDIO_SEND_CHUNK_SIZE 4000

#define ADIF_MAX_SIZE 40 /* Should be enough */
#define ADTS_MAX_SIZE 10 /* Should be enough */

#define AAC_OBJECT_TYPE_LC    1
#define AAC_OBJECT_TYPE_LTP   3

GST_DEBUG_CATEGORY_STATIC (dspaac_debug);
#define GST_CAT_DEFAULT dspaac_debug


static const guint aac_sample_rates[] = {
  96000,
  88200,
  64000,
  48000,
  44100,
  32000,
  24000,
  22050,
  16000,
  12000,
  11025,
  8000
};

static const gchar *devname = "/dev/dsptask/aacdec";

static GstStaticPadTemplate dspaacsink_sink_template =
GST_STATIC_PAD_TEMPLATE ("sink",
    GST_PAD_SINK,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS ( "audio/mpeg, "
        "mpegversion = (int) { 2, 4 } ") );

// Function prototypes
static void gst_dspaacsink_class_init (GstDSPAACSinkClass * klass);
static void gst_dspaacsink_base_init (gpointer g_class);
static void gst_dspaacsink_init (GstDSPAACSink *dspmp3sink, GstDSPAACSinkClass *g_class);
static gboolean gst_dspaacsink_initialize (GstDSPAACSink *dsp);
static GstClockTime gst_dspaacsink_get_time (GstClock * clock, gpointer sink);

// For GObject
static void gst_dspaacsink_set_property (GObject * object, guint prop_id, const GValue * value, GParamSpec * pspec);
static void gst_dspaacsink_get_property (GObject * object, guint prop_id, GValue * value, GParamSpec * pspec);

// For GstElement
static const GstQueryType *gst_dspaacsink_get_query_types (GstElement * element);
static gboolean gst_dspaacsink_query (GstElement * element, GstQuery *query);
static GstClock *gst_dspaacsink_provide_clock (GstElement * elem);
static GstStateChangeReturn gst_dspaacsink_change_state (GstElement * element, GstStateChange transition);
static gboolean gst_dspaacsink_element_event (GstElement * element, GstEvent * event);

// For GstBaseSink
static GstFlowReturn gst_dspaacsink_render (GstBaseSink *sink, GstBuffer * buffer);
static gboolean gst_dspaacsink_setcaps (GstBaseSink *sink, GstCaps *caps);
static GstFlowReturn gst_dspaacsink_preroll (GstBaseSink *sink, GstBuffer * buffer);
static gboolean gst_dspaacsink_event (GstBaseSink *sink, GstEvent *event);
static GstStateChangeReturn gst_dspaacsink_async_play (GstBaseSink *sink);
static gboolean gst_dspaacsink_unlock (GstBaseSink *sink);
static gboolean gst_dspaacsink_unlock_stop (GstBaseSink *sink);


GST_IMPLEMENT_DSPMIXER_METHODS (GstDSPAACSink, gst_dspaacsink);


/*
 * gst_dspaacsink_do_init:
 * @type:
 *
 *
 */
static void
gst_dspaacsink_do_init (GType type)
{
  static const GInterfaceInfo iface_info = {
    (GInterfaceInitFunc) gst_dspaacsink_interface_init,
    NULL,
    NULL,
  };
  static const GInterfaceInfo mixer_iface_info = {
    (GInterfaceInitFunc) gst_dspaacsink_mixer_interface_init,
    NULL,
    NULL,
  };

  gst_dspaudio_init();
  GST_DEBUG_CATEGORY_INIT (dspaac_debug, "dspaacsink", 0, "DSP AAC audio sink");
  g_type_add_interface_static (type, GST_TYPE_IMPLEMENTS_INTERFACE, &iface_info);
  g_type_add_interface_static (type, GST_TYPE_MIXER, &mixer_iface_info);
}

GST_BOILERPLATE_FULL (GstDSPAACSink, gst_dspaacsink, GstBaseSink,
                      GST_TYPE_BASE_SINK, gst_dspaacsink_do_init);


/*
 * gst_dspaacsink_dispose:
 * @object: GObject pointer to element to be deleted
 *
 * Deletes AAC sink element instance. Called automatically by
 * GLib framework when element needs to be disposed.
 */
static void
gst_dspaacsink_dispose (GObject * object)
{
  GstDSPAACSink *dsp = (GstDSPAACSink *) object;

  if(dsp->clock) {
    gst_object_unref(dsp->clock);
    dsp->clock = NULL;
  }

  if(dsp->audio) {
    gst_dspaudio_destroy(dsp->audio);
    dsp->audio = NULL;
  }
  g_object_unref(dsp->adapter);

  G_OBJECT_CLASS (parent_class)->dispose (object);
}


/*
 * gst_dspaacsink_base_init:
 * @g_class: AAC sink class
 *
 * Does the basic initialization of AAC sink element class. This is
 * called by GStreamer framework.
 */
static void
gst_dspaacsink_base_init (gpointer g_class)
{
  GstElementClass *gstelement_class = GST_ELEMENT_CLASS (g_class);
  static const GstElementDetails details = GST_ELEMENT_DETAILS (
    "DSP AAC Sink",
    "Sink/Audio",
    "AAC audio sink",
    "Makoto Sugano <makoto.sugano@nokia.com>");

  gst_element_class_add_pad_template (gstelement_class,
      gst_static_pad_template_get (&dspaacsink_sink_template));
  gst_element_class_set_details (gstelement_class, &details);

}


/*
 * gst_dspaacsink_class_init:
 * @klass: AAC Sink class object
 *
 * Initializes AAC sink element class. This is called by GStreamer
 * framework.
 */
static void
gst_dspaacsink_class_init (GstDSPAACSinkClass * klass)
{
  GObjectClass *gobject_class;
  GstElementClass *gstelement_class;
  GstBaseSinkClass *gstbase_sink_class;

  gobject_class = (GObjectClass *) klass;
  gstelement_class = (GstElementClass *) klass;
  gstbase_sink_class = (GstBaseSinkClass *) klass;

  gobject_class->set_property = GST_DEBUG_FUNCPTR (gst_dspaacsink_set_property);
  gobject_class->get_property = GST_DEBUG_FUNCPTR (gst_dspaacsink_get_property);
  gobject_class->dispose      = GST_DEBUG_FUNCPTR (gst_dspaacsink_dispose);

  gstelement_class->provide_clock   = GST_DEBUG_FUNCPTR (gst_dspaacsink_provide_clock);
  gstelement_class->change_state    = GST_DEBUG_FUNCPTR (gst_dspaacsink_change_state);
  gstelement_class->get_query_types = GST_DEBUG_FUNCPTR (gst_dspaacsink_get_query_types);
  gstelement_class->query           = GST_DEBUG_FUNCPTR (gst_dspaacsink_query);
  gstelement_class->send_event      = GST_DEBUG_FUNCPTR (gst_dspaacsink_element_event);

   // Use dspaudio utility to install default properties
  gst_dspaudio_install_sink_properties (G_OBJECT_CLASS (klass));

  gstbase_sink_class->render     = GST_DEBUG_FUNCPTR (gst_dspaacsink_render);
  gstbase_sink_class->set_caps   = GST_DEBUG_FUNCPTR (gst_dspaacsink_setcaps);
  gstbase_sink_class->event      = GST_DEBUG_FUNCPTR (gst_dspaacsink_event);
  gstbase_sink_class->preroll    = GST_DEBUG_FUNCPTR (gst_dspaacsink_preroll);
  gstbase_sink_class->async_play = GST_DEBUG_FUNCPTR (gst_dspaacsink_async_play);
  gstbase_sink_class->unlock     = GST_DEBUG_FUNCPTR (gst_dspaacsink_unlock);
  gstbase_sink_class->unlock_stop= GST_DEBUG_FUNCPTR (gst_dspaacsink_unlock_stop);
}


/*
 * gst_dspaacsink_init:
 * @dsp: DSP AAC sink object
 * @g_class: AAC sink Class object
 *
 * Initializes the given AAC sink element instance and allocates required
 * resources. This also creates an audio stream object. This is
 * called by GStreamer framework.
 */
static void
gst_dspaacsink_init (GstDSPAACSink * dspaacsink, GstDSPAACSinkClass *g_class)
{
  dspaacsink->audio = gst_dspaudio_new();
  dspaacsink->adapter = gst_adapter_new();
  dspaacsink->clock = gst_audio_clock_new ("DSPClock",
                                           (GstAudioClockGetTimeFunc) gst_dspaacsink_get_time,
                                           dspaacsink);
  // DSP sinks do not need syncing
  gst_base_sink_set_sync (GST_BASE_SINK (dspaacsink), FALSE);
}


/*
 * gst_dspaacsink_handle_sink_event:
 * @dsp: DSP AAC sink object
 *
 * Handles an incoming event by reading the incoming data
 * stream.
 *
 * Returns: TRUE if event was not EOS, otherwise FALSE
 */
static gboolean
gst_dspaacsink_send_remaining (GstDSPAACSink *dsp)
{
  if(dsp->audio->discont_sent == FALSE) {
    DSPWaitStatus status;
    const guint8 *data;

    guint avail = gst_adapter_available (dsp->adapter);
    GST_DEBUG("Remaining: %d bytes", avail);

    // FIXME: What if avail > mmap_buffer_size?

    // Send remaining bytes to DSP and wait for EOF signal
    status = gst_dspaudio_wait_buffer (dsp->audio);

    if (status == DSP_WAIT_OK) {
      GST_DEBUG ("Got EOS data write");
      if (avail) {
        data = gst_adapter_peek (dsp->adapter, avail);
        memcpy (dsp->audio->codec.mmap_buffer, data, avail);
      }
      gst_dspaudio_write_data (dsp->audio, avail/sizeof (short int), TRUE);
      GST_DEBUG ("Wrote CMD_EOF (%d bytes)", avail);
      gst_dspaudio_wait_eos (dsp->audio);
      GST_DEBUG ("Got eof reply");
      dsp->audio->mode = DSP_MODE_EOS;
    }
    gst_adapter_flush (dsp->adapter, avail);
  }
  else {
    GST_DEBUG ("Discont sent to dsp -> not sending eof");
  }

  return TRUE;
}


/*
 * gst_dspaacsink_setcaps:
 * @sink: GstBaseSink
 * @caps: GStCaps offered
 *
 * Returns: TRUE if caps are accepted
 */
static gboolean
gst_dspaacsink_setcaps(GstBaseSink *sink, GstCaps *caps)
{
  gchar *caps_str = gst_caps_to_string (caps);
  GST_DEBUG ("Offered caps: %s", caps_str);
  g_free (caps_str);

  GstDSPAACSink *dsp = GST_DSPAACSINK(sink);
  GstStructure *structure = gst_caps_get_structure (caps, 0);
  const gchar *mimetype = gst_structure_get_name (structure);
  gint mpegversion, srate = 0;

  if(strncmp(mimetype, "audio/mpeg", 10)) {
    return FALSE;
  }

  if(!gst_structure_has_field(structure, "mpegversion" )) {
    return FALSE;
  }

  gst_structure_get_int (structure, "mpegversion", &mpegversion);

  if(mpegversion != 2 && mpegversion != 4) {
    return FALSE;
  }

  // 3GP demuxer offers us this information
  if(gst_structure_has_field (structure, "rate") &&
     gst_structure_has_field (structure, "channels") &&
     gst_structure_has_field (structure, "object_type"))
  {
    gst_structure_get_int (structure, "rate", &srate);
    gst_structure_get_int (structure, "channels", &(dsp->channels));
    gst_structure_get_int (structure, "object_type", &(dsp->object_type));

    if (!gst_dspaudio_map_samplerate (srate, &(dsp->sample_rate))) {
      GST_ELEMENT_ERROR (dsp, STREAM, CODEC_NOT_FOUND, (NULL),
                         ("illegal samplerate"));
      return FALSE;
    }
    dsp->header_type = DSPAAC_HEADER_NONE;
    dsp->got_stream_info = TRUE;
  }

  // This is needed at least in case of RTP
  // Parses the codec_data information to get ObjectType,
  // number of channels and samplerate
  if (gst_structure_has_field (structure, "codec_data")) {

    const GValue *value = gst_structure_get_value (structure, "codec_data");

    if (value) {
      GstBuffer *buf = gst_value_get_buffer (value);
      const guint8 *buffer = GST_BUFFER_DATA (buf);
      dsp->object_type = (buffer[0] & 0xF8) >> 3;
      dsp->sample_rate = ((buffer[0] & 0x07) << 1) | ((buffer[1] & 0x80) >> 7);
      dsp->channels = (buffer[1] & 0x78) >> 3;
      dsp->header_type = DSPAAC_HEADER_NONE;
      dsp->got_stream_info = TRUE;
    }
    else return FALSE;
  }

  if (dsp->got_stream_info) {
    if (dsp->sample_rate < 3) {
      GST_ELEMENT_ERROR (dsp, STREAM, CODEC_NOT_FOUND, (NULL),
                         ("illegal samplerate: %d", dsp->sample_rate));
      return FALSE;
    }
    // FIXME: Temporary hack to make it work with 3GP. The codec data from
    // 3GP demuxer seems to have object_type = 2 that is not officially
    // supported.
    else if (dsp->object_type < AAC_OBJECT_TYPE_LC ||
             dsp->object_type > AAC_OBJECT_TYPE_LTP)
    {
      GST_ELEMENT_ERROR (dsp, STREAM, CODEC_NOT_FOUND, (NULL),
                         ("illegal object type: %d", dsp->object_type));
    }
  }
  GST_DEBUG ("Got stream info: %d", dsp->got_stream_info);
  return TRUE;
}


/*
 * gst_dspaacsink_get_stream_details:
 * @dsp: DSP AAC sink object
 *
 * Parses the incoming AAC data stream and extracts
 * stream information from it. This information is then
 * stored into DSP sink object itself.
 *
 * Returns: TRUE if the info was found, otherwise FALSE
 */
static gboolean
gst_dspaacsink_get_stream_details (GstDSPAACSink *dsp)
{
  gboolean retval = FALSE;
  gboolean found = FALSE;
  const guint8 *buffer;
  guint found_hdrtype;
  guint i = 0;

  guint64 avail = gst_adapter_available (dsp->adapter);
  buffer = gst_adapter_peek (dsp->adapter, avail);

  for(i=0; i < avail - ADIF_MAX_SIZE; i++) {

    if (memcmp (&(buffer[i]), "ADIF", 4) == 0) {
      int skip_size = 0;
      int bitstream;
      int sf_idx;

      found_hdrtype = DSPAAC_HEADER_ADIF;

      // Skip the "ADIF" bytes
      const guint8 *adif = &(buffer[i+4]);

      /* copyright string */
      if(adif[0] & 0x80)
        skip_size += 9; /* skip 9 bytes */

      bitstream = adif[0 + skip_size] & 0x10;
      dsp->bitrate = ((unsigned int)(adif[0 + skip_size] & 0x0F)<<19) |
                     ((unsigned int)adif[1 + skip_size]<<11) |
                     ((unsigned int)adif[2 + skip_size]<<3) |
                     ((unsigned int)adif[3 + skip_size] & 0xE0);

      if (bitstream == 0) {
        dsp->object_type = ((adif[6 + skip_size]&0x01)<<1)|((adif[7 + skip_size]&0x80)>>7);
        sf_idx = (adif[7 + skip_size]&0x78)>>3;
      }
      else {
        dsp->object_type = (adif[4 + skip_size] & 0x18)>>3;
        sf_idx = ((adif[4 + skip_size] & 0x07)<<1)|((adif[5 + skip_size] & 0x80)>>7);
      }
      
      if(!gst_dspaudio_map_samplerate (aac_sample_rates[sf_idx],&dsp->sample_rate)) {
        GST_ERROR("Not supported sample rate : %d!", aac_sample_rates[sf_idx]);
        return FALSE;
      }

      // FIXME: Can we assume this?
      dsp->channels = 2;

      GST_INFO("ADIF: br=%d, samplerate=%d, objtype=%d",
                dsp->bitrate, dsp->sample_rate, dsp->object_type);

      found = TRUE;
      break;
    }

    else if ((buffer[i] == 0xFF) && ((buffer[i+1] & 0xF6) == 0xF0)) {
      gint sf_idx;
      gint len;

      GST_INFO("ADTS ID: %d (position %d)", (buffer[i+1] & 0x08)>>3, i);
      len = ((buffer[i + 3] & 0x03) << 11) |
          (buffer[i + 4] << 3) | ((buffer[i + 5] & 0xe0) >> 5);

      GST_DEBUG("ADTS frame len: %d bytes", len);
      if(i + len + 1 > avail) {
        break;
      }

      if ((buffer[i+len] == 0xFF) && ((buffer[i+len+1] & 0xF6) == 0xF0)) {
        GST_DEBUG("Another adts frame found");
        found_hdrtype = DSPAAC_HEADER_ADTS;
        dsp->object_type = (buffer[i+2] & 0xC0)>>6;
        sf_idx = (buffer[i+2] & 0x3C)>>2;
        if(!gst_dspaudio_map_samplerate (aac_sample_rates[sf_idx],&dsp->sample_rate)) {
          GST_ERROR("Not supported sample rate : %d!", aac_sample_rates[sf_idx]);
          return FALSE;
        }
        dsp->channels = ((buffer[i+2] & 0x01)<<2) | ((buffer[i+3] & 0xC0)>>6);
        dsp->bitrate = ((buffer[i+5] & 0x1F)<<6) | ((buffer[i+6] & 0xFC)>>2);
        GST_DEBUG("ADTS: samplerate %d, channels %d, bitrate %d, objtype %d",
                   dsp->sample_rate, dsp->channels, dsp->bitrate, dsp->object_type);
        dsp->preroll_skipped = i;
        found = TRUE;
        break;
      }
    }
  }

  if (found && (dsp->object_type == AAC_OBJECT_TYPE_LC ||
                dsp->object_type == AAC_OBJECT_TYPE_LTP))
  {
      dsp->header_type = found_hdrtype;
      dsp->sync = TRUE;
      retval = TRUE;
  }
  else if (dsp->got_stream_info) {
      // It's a raw bitstream, but the stream information is available
      // (e.g. through codec_data)
      GST_INFO ("RAW AAC format. Using following parameters:");
      GST_INFO ("Samplerate: %d, Channels: %d, ObjType: %d",
                 dsp->sample_rate, dsp->channels, dsp->object_type);
      dsp->bitrate = 128000; // FIXME: Is this needed?
      retval = TRUE;
  }
  return retval;
}


/*
 * gst_dspaacsink_find_sync:
 *
 */
static gboolean
gst_dspaacsink_find_sync(GstDSPAACSink *dsp)
{
  const guint8 *data = NULL;

  if(dsp->header_type != DSPAAC_HEADER_ADTS) {
    GST_WARNING("Cannot find sync - not adts stream");
    return FALSE;
  }

  int counter = 0;
  while( TRUE ) {

    if(gst_adapter_available(dsp->adapter) < 2) {
      GST_INFO("Cannot find sync within %d bytes", counter);
      return FALSE;
    }
    data = gst_adapter_peek(dsp->adapter, 2);

    // Check for ADTS header
    if ((data[0] == 0xFF) && ((data[1] & 0xF6) == 0xF0)) {
      GST_DEBUG("ADTS sync found");
      break;
    }
    gst_adapter_flush(dsp->adapter, 1);
    counter++;
  }

  return TRUE;
}


#define AAC_CHUNK_SIZE 500
#define AAC_SYNC_SIZE 100000
#define AAC_DETECT_BLOCK_SIZE 4096

/*
 * gst_dspaacsink_render:
 * @pad:
 * @buffer:
 *
 */
static GstFlowReturn
gst_dspaacsink_render (GstBaseSink *sink, GstBuffer * buffer)
{
  GstFlowReturn ret = GST_FLOW_OK;
  DSPWaitStatus status = DSP_WAIT_ERROR;
  GstDSPAACSink *dsp = GST_DSPAACSINK (sink);
  const guint8 *data;
  guint avail;

  GST_LOG ("RENDER: %d bytes", GST_BUFFER_SIZE (buffer));

  gst_buffer_ref (buffer);
  gst_adapter_push (dsp->adapter, buffer);

  if (dsp->header_type == DSPAAC_HEADER_NOT_PARSED) {
    // Don't even try to detect the stream before enough data is available
    if (gst_adapter_available (dsp->adapter) < AAC_DETECT_BLOCK_SIZE) {
      return GST_FLOW_OK;
    }
    if(!gst_dspaacsink_get_stream_details(dsp)) {
      GST_ELEMENT_ERROR (sink, STREAM, CODEC_NOT_FOUND, (NULL),
                        ("gst_dspaacsink_get_stream_details"));
      GST_WARNING("Render returning gst_flow_error 1");
      return GST_FLOW_ERROR;
    }

    if (!gst_dspaacsink_initialize (dsp)) {
      GST_ELEMENT_ERROR (dsp, RESOURCE, SETTINGS, (NULL),
                          ("gst_dspaacsink_initialize"));
      return GST_FLOW_ERROR;
    }
  }

  if (dsp->audio->mode < DSP_MODE_PLAYING) {
    gst_adapter_flush (dsp->adapter, dsp->preroll_skipped);
    dsp->preroll_skipped = 0;

    // Settings might have changed between PAUSE->PLAY transition
    gst_dspaudio_update_dsp_settings (dsp->audio);
    gst_dspaudio_play (dsp->audio);

    GstClockTime latency = gst_base_sink_get_latency (sink);
    if (latency) {
      GST_INFO ("Latency: %"GST_TIME_FORMAT, GST_TIME_ARGS (latency));
      g_usleep (latency / GST_USECOND);
    }
  }

  avail = gst_adapter_available (dsp->adapter);

  while(avail > AAC_CHUNK_SIZE) {

    // Syncing issues -----------------------------------------------
    if(!dsp->sync && dsp->header_type == DSPAAC_HEADER_ADTS) {
      GST_DEBUG("Render syncing...");

      if(avail < AAC_SYNC_SIZE) {
        return GST_FLOW_OK;
      }

      if(!gst_dspaacsink_find_sync(dsp)) {
        GST_WARNING("Render returning gst_flow_error 2");
        GST_ELEMENT_ERROR (sink, STREAM, FAILED, (NULL),
                           ("gst_dspaacsink_find_sync"));
        return GST_FLOW_ERROR;
      }

      dsp->sync = TRUE;
      GST_DEBUG("Render syncing ok");
      return GST_FLOW_OK;
    }

    // Wait for a permission to write data --------------------------
    status = gst_dspaudio_wait_buffer (dsp->audio);

    if (status == DSP_WAIT_INTERRUPT) {
      GST_DEBUG ("DSP_WAIT_INTERRUPT");
      ret = gst_base_sink_wait_preroll (sink);
      if (ret != GST_FLOW_OK)
        break;
      continue;
    }
    else if (status == DSP_WAIT_ERROR) {
      GST_ELEMENT_ERROR (dsp, RESOURCE, WRITE, (NULL),
                         ("loop1: error cmd: %d status: %d",
                          dsp->audio->error_cmd,
                          dsp->audio->error_status));
      ret = GST_FLOW_ERROR;
      break;
    }

    // DSP_ERROR_SYNC and DSP_ERROR_STREAM errors can happen without
    // return value "DSP_WAIT_ERROR" in wait_buffer()
    if (dsp->audio->error_status) {
      if (dsp->header_type != DSPAAC_HEADER_ADTS) {
        GST_ELEMENT_ERROR (dsp, RESOURCE, WRITE, (NULL),
                           ("loop2: error cmd: %d status: %d",
                            dsp->audio->error_cmd,
                            dsp->audio->error_status));
        ret = GST_FLOW_ERROR;
      }

      GST_WARNING("Render detected error situation");
      dsp->audio->error_status = 0;
      dsp->audio->discont_sent = TRUE; // DSP has just been reset, no need for discont
      dsp->sync = FALSE;
      break; // Cannot continue, not enough data to find sync
    }

    data = gst_adapter_peek (dsp->adapter, AAC_CHUNK_SIZE);
    avail -= AAC_CHUNK_SIZE;

    memcpy (dsp->audio->codec.mmap_buffer, data, AAC_CHUNK_SIZE);
    gst_adapter_flush (dsp->adapter, AAC_CHUNK_SIZE);

    if (gst_dspaudio_write_data (dsp->audio, AAC_CHUNK_SIZE / sizeof(short int), FALSE) == FALSE) {
      GST_ELEMENT_ERROR (dsp, RESOURCE, WRITE, (NULL), ("DSP write failed"));
      ret = GST_FLOW_ERROR;
      break;
    }
    GST_LOG("Wrote %d bytes to DSP", AAC_CHUNK_SIZE);
    dsp->bytes_written += AAC_CHUNK_SIZE;
    dsp->sync = TRUE;
  }
  return ret;
}


/*
 * gst_dspaacsink_initialize:
 * @dsp: DSP AAC sink object
 *
 * Initializes the underlying audio stream object and sets up the
 * AAC audio parameters in DSP.
 *
 * Returns: TRUE if operation was successful, otherwise FALSE
 */
static gboolean
gst_dspaacsink_initialize(GstDSPAACSink *dsp)
{
  // Read the junk out...
  gst_dspaudio_flush (dsp->audio);

  if (!gst_dspaudio_set_audioparams (dsp->audio,
    0, dsp->sample_rate, dsp->channels)) {
    return FALSE;
  }

#ifdef HAVE_DSP_H
  DSP_CMD_STATUS cmd_status;
  short int stat_buf[3];
  stat_buf[0] = MP4AACDEC_SET_STATUS;
  stat_buf[1] = dsp->sample_rate;

  // 0 - Main Profile
  // 1 - LC Profile
  // 2 - SSR Profile
  // 3 - LTP profile
  stat_buf[2] = dsp->object_type;

  if(write(dsp->audio->codec.fd, stat_buf, 3 * sizeof(short int)) < 0) {
    dsp->audio->mode = DSP_MODE_ERROR;
    return FALSE;
  }

  int done = read(dsp->audio->codec.fd, &cmd_status, sizeof(DSP_CMD_STATUS));
  if(done == -1 || cmd_status.status != DSP_OK) {
    dsp->audio->mode = DSP_MODE_ERROR;
    return FALSE;
  }
#endif
  GST_INFO("DSP AAC configured");
  return TRUE;
}


/*
 * gst_dspaacsink_preroll:
 * @sink:
 * @buffer:
 *
 * This method is automatically called by GStreamer framework, when
 * a scheduler decides to give execution to this element.
 * The purpose is to check the media stream.
 */
static GstFlowReturn
gst_dspaacsink_preroll (GstBaseSink *sink, GstBuffer * buffer)
{
  GstDSPAACSink *dsp = GST_DSPAACSINK (sink);

  GST_LOG ("enter");

  // We want this to run only once (to initialize the codec)
  if (dsp->audio->mode < DSP_MODE_CONFIGURED) {
    GST_INFO("trying to init DSP");

    gboolean ret = FALSE;

    dsp->preroll_skipped = 0;
    if(dsp->got_stream_info == FALSE && buffer) {
      gst_buffer_ref(buffer);
      gst_adapter_push (dsp->adapter, buffer);
      ret = gst_dspaacsink_get_stream_details (dsp);
      gst_adapter_clear (dsp->adapter);
    }

    if (dsp->got_stream_info || ret) {
      if (!gst_dspaacsink_initialize (dsp)) {
        GST_ELEMENT_ERROR (dsp, RESOURCE, SETTINGS, (NULL),
                           ("gst_dspaacsink_initialize"));
        return GST_FLOW_ERROR;
      }
      GST_DEBUG ("Initialized in preroll");
    }
  }
  GST_LOG ("exit");
  return GST_FLOW_OK;
}


/*
 * gst_dspaacsink_set_property:
 * @object: GObject where the property should be retrieved
 * @prop_id: Unique identifier of the desired property
 * @value: Pointer to a variable where the value should be put
 * @pspec: Parameter specification
 *
 * Stores the property value into GObject
 */
static void
gst_dspaacsink_set_property(GObject * object,
                            guint prop_id,
                            const GValue * value,
                            GParamSpec * pspec)
{
  GstDSPAACSink *dsp = GST_DSPAACSINK (object);

  if(!gst_dspaudio_set_property(dsp->audio, prop_id, value)) {
    GST_WARNING("Trying to set illegal property");
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
  }
}


/*
 * gst_dspaacsink_get_property:
 * @object: GObject where the property should be retrieved
 * @prop_id: Unique identifier of the desired property
 * @value: Pointer to a variable where the value should be put
 * @pspec: Parameter specification
 *
 * Retrieves the desired parameter value from object
 */
static void
gst_dspaacsink_get_property(GObject * object,
                            guint prop_id,
                            GValue * value,
                            GParamSpec * pspec)
{
  GstDSPAACSink *dsp = GST_DSPAACSINK (object);

  if(!gst_dspaudio_get_property(dsp->audio, prop_id, value)) {
    GST_WARNING("Trying to get illegal property");
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
  }
}


/*
 * gst_dspaacsink_get_query_types:
 * @element: GstElement that received this query
 *
 * Returns: a list of supported query types
 */
static const GstQueryType *
gst_dspaacsink_get_query_types(GstElement * element)
{
  static const GstQueryType gst_dspaacsink_query_types[] = {
    GST_QUERY_POSITION,
    GST_QUERY_DURATION,
    0
  };

  return gst_dspaacsink_query_types;
}


/*
 * gst_dspaacsink_query:
 * @element: GstElement that received this query
 * @type: Type of the query, e.g. TOTAL, POSITION
 * @format: The format of the response, e.g. BYTES, TIME
 * @value: Pointer where the return value should be stored
 *
 * Returns: TRUE, iff the query was successful.
 */
static gboolean
gst_dspaacsink_query (GstElement * element, GstQuery *query)
{
  GstBaseSink *bsink = GST_BASE_SINK_CAST (element);
  GstDSPAACSink *dsp = GST_DSPAACSINK (element);
  GstQueryType type = GST_QUERY_TYPE (query);
  gboolean res = FALSE;

  if (type == GST_QUERY_LATENCY) {
    gboolean live, us_live;
    GstClockTime min_l, max_l;
    GST_DEBUG ("latency query");

    if (dsp->audio->mode < DSP_MODE_CONFIGURED) {
        GST_DEBUG ("we are not yet negotiated, can't report latency yet");
        res = FALSE;
    }
    else if ((res = gst_base_sink_query_latency (bsink, &live, &us_live,
                                                 &min_l, &max_l)))
    {
      GstClockTime min_latency, max_latency;
      /* if upstream is live, adjust the min_latency */
      if (us_live) {
        /* FIXME: Calculate the latency somehow */
        min_latency = min_l + 300 * GST_MSECOND;
        max_latency = min_latency + (max_l == -1 ? 0 : max_l);
      } else {
        GST_DEBUG ("peer is not live, don't care about latency");
        min_latency = 0;
        max_latency = -1;
      }
      GST_DEBUG ("latency set to %lld millisecs", min_latency / GST_MSECOND);
      gst_query_set_latency (query, live, min_latency, max_latency);
    }
  }
  else if (type == GST_QUERY_POSITION)
  {
    GstFormat format;

    gst_query_parse_position (query, &format, NULL);
    if (format != GST_FORMAT_TIME) {
      return FALSE;
    }

    GST_DEBUG ("position query");
    if (gst_dspaudio_get_info (dsp->audio)) {
      GstClockTime ctime = gst_dspaudio_info_to_time (dsp->audio);
      ctime -= dsp->dsp_basetime;
      ctime += dsp->seek_basetime;
      gst_query_set_position (query, GST_FORMAT_TIME, ctime);
      GST_DEBUG ("return %lld secs", ctime / GST_SECOND);
      res = TRUE;
    }
  }
  else {
    res = GST_ELEMENT_CLASS (parent_class)->query (element, query);
  }
  return res;
}


/*
 * gst_dspaacsink_element_event:
 * @element: GstElement that received this event
 * @event: The GstEvent structure itself
 *
 * Returns: TRUE, iff the event was successful
 *
 * This handles events that are directly sent to this element. Currently
 * we are not using this, it is waiting for seeking algorithm code
 */
static gboolean
gst_dspaacsink_element_event (GstElement * element, GstEvent * event)
{
  GstDSPAACSink *dsp = GST_DSPAACSINK (element);
  gboolean res = FALSE;

  GstEventType type = event
                    ? GST_EVENT_TYPE (event)
                    : GST_EVENT_UNKNOWN;

  GST_DEBUG ("dsp: event %p %d", event, type);

  // Try if upstream element can handle the event
  gst_event_ref (event);
  res = GST_ELEMENT_CLASS (parent_class)->send_event (element, event);

  if (type == GST_EVENT_SEEK && !res) {
    // As a fallback we retry in a special case when seeking to
    // the beginning. This enables looping of aac clips.
    GST_INFO("AAC SEEK");
    gdouble rate;
    GstFormat format = GST_FORMAT_UNDEFINED;
    GstSeekFlags flags;
    GstSeekType cur_type, stop_type;
    gint64 cur, stop;

    gst_event_parse_seek (event, &rate, &format, &flags,
                          &cur_type, &cur, &stop_type, &stop);
    if (cur_type == GST_SEEK_TYPE_SET && cur == 0) {
      GstEvent *new_event = gst_event_new_seek(1.0, GST_FORMAT_BYTES,
                                               GST_SEEK_FLAG_FLUSH,
                                               GST_SEEK_TYPE_SET, 0,
                                               GST_SEEK_TYPE_NONE,
                                               GST_CLOCK_TIME_NONE);
      res = gst_pad_push_event (GST_BASE_SINK (dsp)->sinkpad, new_event);
    }
  }
  gst_event_unref (event);
  return res;
}


/*
 * gst_dspaacsink_event:
 *
 * This method handles events coming from basesink
 */
static gboolean
gst_dspaacsink_event (GstBaseSink *sink, GstEvent * event)
{
  GST_DEBUG("Got event from basesink: %d", GST_EVENT_TYPE (event));
  GstDSPAACSink *dsp = GST_DSPAACSINK (sink);

  GstEventType type = event ? GST_EVENT_TYPE (event) : GST_EVENT_UNKNOWN;
  if(type == GST_EVENT_EOS) {
    GST_DEBUG ("EOS event");
    if (!dsp->sync) {
      GST_WARNING ("EOS without sync");
      GST_ELEMENT_ERROR (sink, STREAM, FAILED, (NULL),
                         ("gst_dspaacsink_find_sync"));
    }
    else {
      gst_dspaacsink_send_remaining (dsp);
    }
  }
  else if (type == GST_EVENT_FLUSH_STOP) {
    GST_DEBUG ("flush stop");
    if (dsp->audio->mode != DSP_MODE_EOS) {
      gst_dspaudio_discont (dsp->audio);
    }
    gst_adapter_clear (dsp->adapter);
  }
  else if (type == GST_EVENT_NEWSEGMENT) {
    gint64 start, position;

    gst_event_parse_new_segment (event, NULL, NULL, NULL, &start,
                                 NULL, &position);

    GST_DEBUG ("NEWSEGMENT: start = %lld, position = %lld", start, position);
    dsp->seek_basetime = position;

    if (gst_dspaudio_get_info (dsp->audio)) {
      dsp->dsp_basetime = gst_dspaudio_info_to_time (dsp->audio);
    }
    GST_DEBUG ("dsp basetime: %lld secs", dsp->dsp_basetime / GST_SECOND);
  }
  return TRUE;
}


/*
 * gst_dspaacsink_async_play:
 *
 */
static GstStateChangeReturn
gst_dspaacsink_async_play (GstBaseSink *sink)
{
  GstDSPAACSink *dspaacsink = GST_DSPAACSINK (sink);
  GstClock *clock = GST_ELEMENT_CLOCK (sink);

  GstClockTime itime, etime;
  GstClockTime rate_num, rate_denom;

  GST_DEBUG ("Async play");

  if (clock == dspaacsink->clock)
    goto done;

  /* if we are slaved to a clock, we need to set the initial
    * calibration */
  /* get external and internal time to set as calibration params */
  etime = gst_clock_get_time (clock);
  itime = gst_clock_get_internal_time (dspaacsink->clock);

  GST_DEBUG ("internal time: %" GST_TIME_FORMAT " external time: %" GST_TIME_FORMAT,
             GST_TIME_ARGS (itime), GST_TIME_ARGS (etime));

  gst_clock_get_calibration (dspaacsink->clock, NULL, NULL, &rate_num,
                             &rate_denom);
  gst_clock_set_calibration (dspaacsink->clock, itime, etime,
                             rate_num, rate_denom);


done:
  gst_dspaudio_remove_interrupt (dspaacsink->audio);
  if(dspaacsink->audio->mode == DSP_MODE_PAUSED ||
     dspaacsink->audio->mode == DSP_MODE_EOS)
  {
    gst_dspaudio_flush (dspaacsink->audio);
    gst_dspaudio_play (dspaacsink->audio);
    GST_DEBUG ("async play success");
  }
  return GST_STATE_CHANGE_SUCCESS;
}


/*
 * gst_dspaacsink_change_state:
 * @element: The GstElement whose state to be changed
 *
 * Changes the state of given element. The desired state information
 * is contained by the element itself, and this method is just
 * a callback, so the plugin knows when the state is changing.
 * This method is called by GStreamer framework
 *
 * Returns: TRUE if operation was successful, otherwise FALSE
 */
static GstStateChangeReturn
gst_dspaacsink_change_state (GstElement * element, GstStateChange transition)
{
  GstDSPAACSink *dspaacsink = GST_DSPAACSINK (element);
  GstStateChangeReturn ret;

  switch (transition) {

    case GST_STATE_CHANGE_NULL_TO_READY:
      GST_DEBUG ("NULL TO READY");
      dspaacsink->bytes_written = 0;
      dspaacsink->sync = TRUE;
      dspaacsink->header_type = DSPAAC_HEADER_NOT_PARSED;
      dspaacsink->got_stream_info = FALSE;
      dspaacsink->seek_basetime = 0;
      dspaacsink->dsp_basetime = 0;

      // Set default (guessed) stream values
      gst_dspaudio_map_samplerate (44100, &dspaacsink->sample_rate);
      dspaacsink->channels = 2;
      dspaacsink->object_type = AAC_OBJECT_TYPE_LC;

      if (gst_dspaudio_open(dspaacsink->audio, devname) == FALSE) {
        GST_ELEMENT_ERROR (element, RESOURCE,
                           OPEN_READ_WRITE, (NULL),
                           ("gst_dspaudio_open"));
        return GST_STATE_CHANGE_FAILURE;
      }
      break;

    case GST_STATE_CHANGE_READY_TO_PAUSED:
      GST_DEBUG("READY TO PAUSED");
      gst_adapter_clear(dspaacsink->adapter);
      break;

    case GST_STATE_CHANGE_PAUSED_TO_PLAYING:
      GST_DEBUG("PAUSED TO PLAYING. MODE = %d", dspaacsink->audio->mode);
      gst_dspaacsink_async_play (GST_BASE_SINK_CAST (element));
      GST_DEBUG("PAUSED TO PLAYING OK");
      break;

    default:
      break;
  }

  gst_dspaudio_clock_change_state(dspaacsink->audio, transition);
  ret = GST_ELEMENT_CLASS (parent_class)->change_state (element, transition);

  switch (transition) {

    case GST_STATE_CHANGE_PLAYING_TO_PAUSED:
      GST_DEBUG("PLAYING TO PAUSED V2");
      gst_dspaudio_pause (dspaacsink->audio);
      GST_DEBUG("PLAYING TO PAUSED V2 OK");
      break;

    case GST_STATE_CHANGE_PAUSED_TO_READY:
      GST_DEBUG("PAUSED TO READY");
      gst_dspaudio_stop (dspaacsink->audio);
      gst_adapter_clear (dspaacsink->adapter);
      break;

    case GST_STATE_CHANGE_READY_TO_NULL:
      GST_DEBUG("READY To NULL");
      gst_dspaudio_close(dspaacsink->audio);
      gst_dspaudio_reset(dspaacsink->audio);
      break;

    default:
      break;
  }

  GST_DEBUG("State change successful");
  return ret;
}


/*
 * gst_dspaacsink_provide_clock:
 *
 */
static GstClock *
gst_dspaacsink_provide_clock (GstElement * elem)
{
  GST_DEBUG ("gst_dspaacsink_provide_clock");
  GstDSPAACSink *dsp = GST_DSPAACSINK (elem);
  return GST_CLOCK_CAST (gst_object_ref (dsp->clock));
}


/*
 * gst_dspaacsink_get_time:
 *
 */
static GstClockTime
gst_dspaacsink_get_time (GstClock * clock, gpointer sink)
{
  GST_LOG ("clock query");
  GstDSPAACSink *dsp = GST_DSPAACSINK (sink);
  return gst_dspaudio_get_dsp_clock (dsp->audio, FALSE);
}


/*
 * gst_dspaacsink_unlock:
 * @sink: GstBaseSink
 *
 * Interrupts the blocking :render() function.
 *
 * Returns: TRUE
 */
static gboolean
gst_dspaacsink_unlock (GstBaseSink *sink)
{
  GST_DEBUG ("unlock");
  GstDSPAACSink *dsp = GST_DSPAACSINK (sink);
  gst_dspaudio_interrupt_render (dsp->audio);
  return TRUE;
}


/*
 * gst_dspaacsink_unlock_stop:
 * @sink: GstBaseSink
 *
 * Removes :render() unlocking.
 *
 * Returns: TRUE
 */
static gboolean
gst_dspaacsink_unlock_stop (GstBaseSink *sink)
{
  GST_DEBUG ("unlock_stop");
  GstDSPAACSink *dsp = GST_DSPAACSINK (sink);
  gst_dspaudio_remove_interrupt (dsp->audio);
  return TRUE;
}


/*
 * plugin_init:
 * @plugin: GstPlugin to be initialized
 *
 * This initializes the plugin when it is loaded by GStreamer
 *
 * Returns: Boolean value, TRUE if initialization was successful
 */
static gboolean
plugin_init (GstPlugin * plugin)
{
  return gst_element_register (plugin, "dspaacsink", GST_RANK_PRIMARY,
                               GST_TYPE_DSPAACSINK);
}

GST_PLUGIN_DEFINE (GST_VERSION_MAJOR,
                   GST_VERSION_MINOR,
                   "dspaac",
                   "DSP AAC audio sink",
                   plugin_init,
                   VERSION,
                   "LGPL",
                   "dspaudio",
                   "http://www.nokia.com/");
