/*
 * This file is part of eds-sync
 *
 * Copyright (C) 2007 Nokia Corporation. All rights reserved.
 *
 * Author: Ross Burton <ross@openedhand.com>
 * Author: Onne Gorter <onne.gorter@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>
#include <libebook/e-book.h>
#include <libtelepathy/tp-conn-gen.h>
#include <libtelepathy/tp-constants.h>
#include <libtelepathy/tp-conn-iface-aliasing-gen.h>
#include <libtelepathy/tp-chan-iface-group-gen.h>

#include "gintset.h"
#include "context.h"
#include "eds.h"
#include "util.h"

#include "eds-sync-private.h"

#define ENSURE_NORMALIZED_BOUND(attr, context) \
  if ((context)->raw_account && \
      attribute_is_bound_to ((attr), (context)->raw_account)) { \
    g_warning (G_STRLOC ": Found EVcardAttr that contains an unnormalized bound. This is a bug."); \
  } \

static void
commit_cb (EBook *book, EBookStatus status, gpointer closure)
{
  if (status != E_BOOK_ERROR_OK) {
    g_warning ("could not commit contact: status %d", status);
  }
}

static void
add_remote_address_to_local_contact (guint handle, gpointer userdata)
{
  TelepathyContext *context;
  const char *address = NULL;
  char **names = NULL;
  const char *contact_uid;
  GError *error = NULL;
  EContact *contact = NULL;
  EVCardAttribute *attr;
  EVCardAttributeParam *param;

  g_assert (userdata);
  context = userdata;

  e_debug ("adding new remote contact locally with handle: %d", handle);

  if (!inspect_handle (context->conn, handle, &address, &error)) {
    g_warning ("cannot inspect handle: %s", _ERROR_MSG (error));
    g_clear_error (&error);
    return;
  }

  v_debug ("adding %s:%s", context->vcard, address);
  
  contact_uid = eds_sync_get_uid_for_account (context->sync,
                                              context->vcard,
                                              address);
  
  /* If we got an EContact UID for this IM address, try and get the contact */
  if (contact_uid) {
    if (!e_book_get_contact (context->book, contact_uid, &contact, &error)) {
      g_warning ("cannot get contact: %s", _ERROR_MSG (error));
      g_clear_error (&error);
      /* Unset the EContact UID so that we don't try and commit a new
         contact later */
      contact_uid = NULL;
      contact = NULL;
    } else {
        v_debug("found contact address in another local contact, using that one");
    }
  }

  /* If we still don't have a contact, create a new one */
  if (contact == NULL) {
    v_debug("contact address not already found, creating new automatic contact");
    contact = e_contact_new ();

    /* Mark this contact as automatically created */
    attr = e_vcard_attribute_new (NULL, EVC_X_OSSO_CONTACT_STATE);
    e_vcard_add_attribute_with_values (E_VCARD (contact), attr, "AUTOMATIC", NULL);

    if (context->aliasing_proxy) {
      GArray *handles;

      handles = g_array_new (FALSE, FALSE, sizeof (guint));
      g_array_append_val (handles, handle);
      if (tp_conn_iface_aliasing_request_aliases (context->aliasing_proxy, handles, &names, &error)) {

        if (names && strcmp (names[0], address) == 0) {
          v_debug("ignoring alias that is same as address: %s", names[0]);
          /* Don't set name if it is the same as the address */
          g_free (names[0]);
          names[0] = NULL;
        }

      } else {

        g_warning ("cannot get alias: %s", _ERROR_MSG (error));
        g_clear_error (&error);
      }

      g_array_free (handles, TRUE);
    }

    if (names && names[0]) {
      v_debug("setting contact FN to alias: %s", names[0]);
      e_contact_set (contact, E_CONTACT_FULL_NAME, names[0]);
    }
  }

  attr = e_vcard_attribute_new (NULL, context->vcard);
  param = e_vcard_attribute_param_new (EVC_X_OSSO_BOUND);
  e_vcard_attribute_add_param_with_value (attr, param, context->account);
  e_vcard_add_attribute_with_values (E_VCARD (contact), attr, address, NULL);

  // remove this handle from unbound, to prevent the unbound address from also getting bound
  // TODO actually unbound handles need to be deleted, not bound?
  // g_intset_remove (context->bound_handles, handle);

  /* Ensure that this handle is known in EDS so that the change event from
     adding the contact won't cause a re-sync */
  v_debug("adding handle to local_handles: %d", handle);
  g_intset_add (context->local_handles, handle);

  if (contact_uid) {
    context->to_commit = g_list_prepend (context->to_commit, g_object_ref (contact));

    if (debug_flags & DEBUG_VERBOSE) {
      gchar * vs = e_vcard_to_string (E_VCARD (contact), EVC_FORMAT_VCARD_30);
      g_debug ("committed: \n%s\n", vs);
      g_free(vs);
    }

  } else {
    e_book_add_contact (context->book, contact, NULL);
    eds_sync_register_account (context->sync,
                               context->vcard, address,
                               e_contact_get_const (contact, E_CONTACT_UID));
  }

  g_object_unref (contact);
  g_strfreev (names);
}

static void
maybe_remove_local_contact (guint handle, gpointer userdata)
{
  TelepathyContext *context;
  GList *uids, *to_remove;
  GError *error = NULL;
  EContact *contact;
  GList *attrs;
  gboolean modified = FALSE;
  gboolean other_addresses = FALSE;
  gboolean other_fields = FALSE;
  const char * address;

  g_assert (userdata);
  context = userdata;

  v_debug ("maybe remove local contact with handle: %d", handle);

  /*
   * An IM account field exists locally but does not exist on the server. If
   * it's not TOADD, then it's been remotely deleted, so we delete local
   * field. If contact has no remaining fields and is AUTOMATIC, delete from
   * abook.
   */

  /* Get the UID for this handle */
  if (!inspect_handle (context->conn, handle, &address, NULL)) {
    g_warning ("cannot inspect handle %u", handle);
    return;
  }
  g_assert (address);
  
  uids = eds_sync_telepathy_lookup_uids (context, handle);
  to_remove = NULL;
  for ( ; uids; uids = uids->next) {
    char *contact_uid = NULL;
    contact_uid = uids->data;

    if (!e_book_get_contact (context->book, contact_uid, &contact, &error)) {
      g_warning ("cannot get contact with UID %s: %s", contact_uid, _ERROR_MSG (error));
      g_clear_error (&error);
      return;
    }
    g_assert (contact);

    attrs = e_vcard_get_attributes (E_VCARD (contact));
    while (attrs) {
      EVCardAttribute *attr;
      const char * s;

      attr = attrs->data;

      /* If this attribute isn't an IM field, skip */
      /* TODO how about other IM fields but local sync context? */
      if (strcmp (e_vcard_attribute_get_name (attr), context->vcard) != 0) {

        /* check if useful other field and not delete if is such */
        if (strcmp (e_vcard_attribute_get_name (attr), "FN") == 0)
          other_fields = TRUE;
        
        if (strcmp (e_vcard_attribute_get_name (attr), "N") == 0)
          other_fields = TRUE;
        
        if (strcmp (e_vcard_attribute_get_name (attr), "EMAIL") == 0)
          other_fields = TRUE;
        
        if (strcmp (e_vcard_attribute_get_name (attr), "TEL") == 0)
          other_fields = TRUE;
        
        //if (strcmp (e_vcard_attribute_get_name (attr), "SIP") == 0) other_fields = TRUE;

        goto next;
      }

      /* If this attribute isn't about to be added, skip */
      if (field_state_is (attr, FIELD_STATE_TOADD)) {
        other_addresses = TRUE;
        goto next;
      }

      ENSURE_NORMALIZED_BOUND (attr, context);

      /* If this attribute is bound to another account, skip */
      if (!attribute_is_bound_to (attr, context->account)) {
        other_addresses = TRUE;
        goto next;
      }

      /* is it our address? */
      s = e_vcard_attribute_get_value (attr);
      if (s == NULL || *s == '\0') goto next;
      if (strcmp (address, s) != 0) {
        other_addresses = TRUE;
        goto next;
      }

      v_debug ("removing contact with handle %d UID %s bound %s, address %s",
               handle,
               (char*) e_contact_get_const (contact, E_CONTACT_UID),
               context->account,
               address);

      /* Remove it */
      modified = TRUE;
      attrs = g_list_next (attrs);
      // only unbind the address, automatic contacts can't have different addresses, so this is fine for both cases
      e_vcard_attribute_remove_param_value (attr, EVC_X_OSSO_BOUND, context->account);
      // delete permenently if it was actually TODELETE
      if (!contact_state_is (contact, CONTACT_STATE_AUTOMATIC)) {
        // if not automatic contact, mark this address as another address ...
        other_addresses = TRUE;
      }
      if (field_state_is (attr, FIELD_STATE_TODELETE)) {
        e_vcard_remove_attribute (E_VCARD (contact), attr);
      }

      /* and remove handle from local handles so he is known to have been deleted */
      g_intset_remove(context->local_handles, handle);
      to_remove = g_list_append (to_remove, contact_uid);
      continue;

    next:
      attrs = g_list_next (attrs);
    }

    if (modified) {
      /* if now contact has zero addresses and is automatic or empty -> remove */
      if (!other_addresses &&
          (contact_state_is (contact, CONTACT_STATE_AUTOMATIC) || !other_fields)) {

        v_debug("marking contact as deleted");

        set_contact_flag (contact, "DELETED", TRUE);
      }

      context->to_commit = g_list_prepend (context->to_commit, g_object_ref (contact));

      if (debug_flags & DEBUG_VERBOSE) {
        gchar * vs = e_vcard_to_string (E_VCARD (contact), EVC_FORMAT_VCARD_30);
        g_debug ("committed: \n%s\n", vs);
        g_free(vs);
      }
    }

    g_object_unref (contact);
  }
  while (to_remove) {
    eds_sync_telepathy_remove_uid (context, to_remove->data);
    to_remove = g_list_delete_link (to_remove, to_remove);
  }
}

/*** blocking ***/
static void
unblock_local_contact (guint handle, gpointer userdata)
{
  TelepathyContext *context = userdata;
  GList *uids;
  EContact *contact;
  const char *address;

  GList *attrs;
  GError *error = NULL;

  v_debug("unblock_local_contact");

  /* Get the address for this handle */
  if (!inspect_handle (context->conn, handle, &address, NULL)) {
    g_warning ("cannot inspect handle %u", handle);
    return;
  }
  
  uids = eds_sync_telepathy_lookup_uids (context, handle);
  for ( ; uids; uids = uids->next) {
    char *contact_uid = NULL;

    contact_uid = uids->data;

    /* get EContact for this contact_uid */
    if (!e_book_get_contact (context->book, contact_uid, &contact, &error)) {
      g_warning ("cannot get contact with UID %s: %s", contact_uid, _ERROR_MSG (error));
      g_clear_error (&error);
      return;
    }

    contact_state_remove (contact, CONTACT_STATE_BLOCKED);
    g_intset_remove (context->local_blocked_handles, handle);

    /* TODO go through all IM vcard fields, not just the local vcard
     * field and set as TOUNBLOCK except for current address/bound,
     * set that one to normal
     */
    attrs = e_vcard_get_attributes (E_VCARD (contact));
    while (attrs) {
      EVCardAttribute *attr;
      const char * s;

      attr = attrs->data;

      /* TODO here we need any IM field, not just current sync context IM field */
      if (strcmp (e_vcard_attribute_get_name (attr), context->vcard) != 0) {
        goto next;
      }

      ENSURE_NORMALIZED_BOUND (attr, context);
      /* If this attribute is bound to our account and is us skip */
      if (attribute_is_bound_to (attr, context->account)) {
        /* is it our address? */
        s = e_vcard_attribute_get_value (attr);
        if (s == NULL || *s == '\0') goto next;
        if (strcmp (address, s) == 0) {
          // TODO must remove the marks completely
          field_state_set (attr, FIELD_STATE_TOUNBLOCK);
          goto next;
        }
      }

      v_debug ("setting field to TOUNBLOCK for %s", e_vcard_attribute_get_value (attr));
      field_state_set (attr, FIELD_STATE_TOUNBLOCK);

      next:
        attrs = g_list_next (attrs);
    }

    context->to_commit = g_list_prepend (context->to_commit, g_object_ref (contact));

    if (debug_flags & DEBUG_VERBOSE) {
      gchar * vs = e_vcard_to_string (E_VCARD (contact), EVC_FORMAT_VCARD_30);
      g_debug ("committed: \n%s\n", vs);
      g_free(vs);
    }

    g_object_unref (contact);
  }
}

static void
block_local_contact (guint handle, gpointer userdata)
{
  TelepathyContext *context = userdata;
  GList *uids;
  EContact *contact;
  const char *address;

  GList *attrs;
  GError *error = NULL;

  v_debug("block_local_contact");

  /* Get the address for this handle */
  if (!inspect_handle (context->conn, handle, &address, NULL)) {
    g_warning ("cannot inspect handle %u", handle);
    return;
  }
  
  uids = eds_sync_telepathy_lookup_uids (context, handle);
  for ( ; uids; uids = uids->next) {
    char* contact_uid = NULL;

    contact_uid = uids->data;
    
    /* get EContact for this contact_uid */
    if (!e_book_get_contact (context->book, contact_uid, &contact, &error)) {
      g_warning ("cannot get contact with UID %s: %s", contact_uid, _ERROR_MSG (error));
      g_clear_error (&error);
      return;
    }

    //contact_state_set (contact, CONTACT_STATE_BLOCKED);
    v_debug ("marking local contact as BLOCKED");
    set_contact_flag (contact, "BLOCKED", TRUE);
    g_intset_add (context->local_blocked_handles, handle);

    /* TODO go through all IM fields and set as TOUNBLOCK, except for
     * current address/bound, set that one to normal
     */
    attrs = e_vcard_get_attributes (E_VCARD (contact));
    while (attrs) {
      EVCardAttribute *attr;
      const char * s;

      attr = attrs->data;

      /* TODO how about other IM fields but local sync context? */
      if (strcmp (e_vcard_attribute_get_name (attr), context->vcard) != 0) {
        goto next;
      }

      ENSURE_NORMALIZED_BOUND (attr, context);
      /* If this attribute is bound to our account and is us skip */
      if (attribute_is_bound_to (attr, context->account)) {
        /* is it our address? */
        s = e_vcard_attribute_get_value (attr);
        if (s == NULL || *s == '\0') goto next;
        if (strcmp (address, s) == 0) {
          v_debug ("setting field to BLOCKED for %s", e_vcard_attribute_get_value (attr));
          field_state_set (attr, FIELD_STATE_BLOCKED);
          goto next;
        }
      }

      v_debug ("setting field to TOBLOCK for %s", e_vcard_attribute_get_value (attr));
      field_state_set (attr, FIELD_STATE_TOBLOCK);

      next:
        attrs = g_list_next (attrs);
    }

    context->to_commit = g_list_prepend (context->to_commit, g_object_ref (contact));

    if (debug_flags & DEBUG_VERBOSE) {
      gchar * vs = e_vcard_to_string (E_VCARD (contact), EVC_FORMAT_VCARD_30);
      g_debug ("committed: \n%s\n", vs);
      g_free(vs);
    }

    g_object_unref (contact);
  }
}


/*** local contact changed handling ***/
static void
handle_local_contact_changed (TelepathyContext *context, EContact *contact)
{
  GError *error = NULL;
  GList *attrs;
  gboolean modified = FALSE;
  gboolean other_addresses = FALSE;

  g_assert (context != NULL);

  e_debug ("local contact changed with uid %s", (char*)e_contact_get_const (contact, E_CONTACT_UID));
  if (debug_flags & DEBUG_VERBOSE) {
    char *vcard;
    vcard = e_vcard_to_string (E_VCARD (contact), EVC_FORMAT_VCARD_30);
    g_debug ("VCARD:\n%s\n", vcard);
    g_free (vcard);
  }

  attrs = e_vcard_get_attributes (E_VCARD (contact));
  for (; attrs; attrs = g_list_next (attrs)) {
    EVCardAttribute *attr;
    attr = attrs->data;

    /* TODO check for context->field */

    /* Rewrite bound with normalized bound if necessary */
    if (context->raw_account &&
        attribute_is_bound_to (attr, context->raw_account)) {
      GList *params;

      v_debug ("rewriting raw bound (%s) with normalized bound (%s) in vcard",
               context->raw_account, context->account);
      params = e_vcard_attribute_get_params (attr);
      while (params) {
        const char *name = e_vcard_attribute_param_get_name (params->data);
        if (name && (strcmp (name, EVC_X_OSSO_BOUND) == 0)) {
          e_vcard_attribute_param_remove_values (params->data);
          e_vcard_attribute_param_add_value (params->data, context->account);
          modified = TRUE;
          break;
        }
        params = params->next;
      }
    }
    ENSURE_NORMALIZED_BOUND (attr, context);

    /* check for delete */
    if (field_state_is (attr, FIELD_STATE_TODELETE)) {
      // make sure it is our address
      if (!attribute_is_bound_to (attr, context->account)) {
          // only if there actually is a bound do we want to mark this as other addresses ...
          if (!attribute_is_bound_to (attr, NULL)) other_addresses = TRUE;
          continue;
      }
      v_debug("TODELETE address found, trying to address contact from remote list");

      GArray *array;
      guint n_uids;
      guint handle;
      gboolean remove_from_roster = TRUE;
      
      if (!eds_tp_conn_request_handle (DBUS_G_PROXY (context->conn), TP_CONN_HANDLE_TYPE_CONTACT,
                                   e_vcard_attribute_get_values (attr)->data,
                                   &handle, &error)) {
        g_warning ("cannot get handle: %s", _ERROR_MSG (error));
        g_clear_error (&error);

        remove_from_roster = FALSE;

      } else {
        n_uids = g_list_length (eds_sync_telepathy_lookup_uids (context, handle));
        eds_sync_telepathy_remove_uid (context, e_contact_get_const (contact, E_CONTACT_UID));
        if (n_uids > 1) {
          /* Don't remove the contact from the roster */
          remove_from_roster = FALSE;

          v_debug ("leaving roster untouched, since there are %d more contacts that refer"
                   "to handle %d", n_uids - 1, handle);
        }
      }

      if (FALSE == remove_from_roster) {
        e_vcard_attribute_remove_param_value (attr, EVC_X_OSSO_BOUND, context->account);

        attrs = g_list_previous (attrs);
        e_vcard_remove_attribute (E_VCARD (contact), attr);
        modified = TRUE;

        continue;
      }

      array = g_array_new (FALSE, FALSE, sizeof (guint));
      g_array_append_val (array, handle);

      // remove from all groups
      tp_chan_iface_group_remove_members (context->subscribe_group_proxy, array, "", NULL);
      if (context->publish_group_proxy) {
        tp_chan_iface_group_remove_members (context->publish_group_proxy, array, "", NULL);
      }
      if (context->known_group_proxy) {
        tp_chan_iface_group_remove_members (context->known_group_proxy, array, "", NULL);
      }

      // remove from vcard
      attrs = g_list_previous(attrs);
      e_vcard_remove_attribute (E_VCARD (contact), attr);

      g_array_free (array, TRUE);
      modified = TRUE;

      // remove from all lists
      v_debug("removing contact from all lists");
      g_intset_remove (context->remote_handles, handle);
      g_intset_remove (context->known_handles, handle);
      g_intset_remove (context->blocked_handles, handle);
      g_intset_remove (context->local_handles, handle);
      g_intset_remove (context->pending_auth_handles, handle);

      /* end of TODELETE state */

    } else if (field_state_is (attr, FIELD_STATE_TOADD) && attribute_is_bound_to (attr, context->account)) {
      v_debug("TOADD address, tying to add to remote list");

       /* check for addition */
      GArray *array;
      guint handle;

      if (!eds_tp_conn_request_handle (DBUS_G_PROXY (context->conn), TP_CONN_HANDLE_TYPE_CONTACT,
                                   e_vcard_attribute_get_values (attr)->data,
                                   &handle, &error)) {
        g_warning ("cannot get handle: %s", _ERROR_MSG (error));
        g_clear_error (&error);

        continue;
      }

      v_debug("adding toadd contact to local_handles: %d", handle);
      g_intset_add (context->local_handles, handle);

      array = g_array_new (FALSE, FALSE, sizeof (gint));
      g_array_append_val (array, handle);

      // NOTICE: no message here, might want to try a default messages, but that is useless
      if (tp_chan_iface_group_add_members (context->subscribe_group_proxy, array, "", &error)) {

        if (context->aliasing_proxy) {
          const gchar *alias;
          GHashTable *aliases;

          alias = e_contact_get_const (contact, E_CONTACT_FULL_NAME);
          
          /* fall back to the nickname */
          if (!alias)
            alias = e_contact_get_const (contact, E_CONTACT_NICKNAME);

          if (!alias)
            goto no_alias;
          
          aliases = g_hash_table_new (g_direct_hash, g_direct_equal);
          g_hash_table_insert (aliases,
                               GUINT_TO_POINTER (handle),
                               (gchar *) alias);

          /* TODO - check the aliasing flags */
          tp_conn_iface_aliasing_set_aliases (context->aliasing_proxy,
                                              aliases,
                                              &error);
          if (error) {
            g_warning ("cannot set alias: %s", _ERROR_MSG (error));
            g_clear_error (&error);
          }

          g_hash_table_destroy (aliases);
        }

no_alias:
        field_state_remove (attr, FIELD_STATE_TOADD);
        g_array_free (array, TRUE);
        modified = TRUE;

        // make sure this handle is known in the group
        v_debug("adding to remote_handles: %d", handle);
        g_intset_add (context->remote_handles, handle);

      } else {
        g_warning ("cannot add members: %s", _ERROR_MSG (error));
        g_clear_error (&error);
        g_array_free (array, TRUE);

        continue;
      }

      /* endo if TOADD state */
   } else if (field_state_is (attr, FIELD_STATE_TOBLOCK) && attribute_is_bound_to (attr, context->account)) {
        v_debug("TOBLOCK address, blocking needed");

        guint handle;
        GArray *array;

        if (!eds_tp_conn_request_handle (DBUS_G_PROXY (context->conn), TP_CONN_HANDLE_TYPE_CONTACT,
                                     e_vcard_attribute_get_values (attr)->data,
                                     &handle, &error)) {
          g_warning ("cannot get handle: %s", _ERROR_MSG (error));
          g_clear_error (&error);

          continue;
        }

        v_debug("putting toblock contact in local handles, as contact is locally known: %d", handle);
        g_intset_add (context->local_handles, handle);

        if (!context->blocked_group_proxy) {
            v_debug("want to block, but not connected to block list");
            field_state_remove (attr, FIELD_STATE_TOBLOCK);
            field_state_set (attr, FIELD_STATE_BLOCKED);
            modified = TRUE;
            g_intset_add (context->local_blocked_handles, handle);
            g_intset_add (context->blocked_handles, handle);
        } else {
          array = g_array_new (FALSE, FALSE, sizeof (gint));
          g_array_append_val (array, handle);

          if (tp_chan_iface_group_add_members (context->blocked_group_proxy, array, "", &error)) {

            field_state_remove (attr, FIELD_STATE_TOBLOCK);
            field_state_set (attr, FIELD_STATE_BLOCKED);
            g_array_free (array, TRUE);
            modified = TRUE;

            // make sure this handle is known in the group
            v_debug("adding to local and remote block_handles: %d", handle);
            g_intset_add (context->local_blocked_handles, handle);
            g_intset_add (context->blocked_handles, handle);

          } else {
            g_warning ("cannot add members: %s", _ERROR_MSG (error));
            g_clear_error (&error);
            g_array_free (array, TRUE);

            continue;
          }
        }
      /* endo if TOBLOCK state */
    } else if (field_state_is (attr, FIELD_STATE_TOUNBLOCK) && attribute_is_bound_to (attr, context->account)) {
        v_debug("TOUNBLOCK address, unblocking needed");

        guint handle;
        GArray *array;

        if (!eds_tp_conn_request_handle (DBUS_G_PROXY (context->conn), TP_CONN_HANDLE_TYPE_CONTACT,
                                     e_vcard_attribute_get_values (attr)->data,
                                     &handle, &error)) {
          g_warning ("cannot get handle: %s", _ERROR_MSG (error));
          g_clear_error (&error);

          continue;
        }

        v_debug("putting tounblock contact into local handles, as contact is locally known: %d", handle);
        g_intset_add (context->local_handles, handle);

        if (!context->blocked_group_proxy) {
            v_debug("want to unblock, but not connected to block list");
            field_state_remove (attr, FIELD_STATE_TOUNBLOCK);
            field_state_remove (attr, FIELD_STATE_BLOCKED);
            modified = TRUE;
            g_intset_remove (context->local_blocked_handles, handle);
            g_intset_remove (context->blocked_handles, handle);
        } else {
          array = g_array_new (FALSE, FALSE, sizeof (gint));
          g_array_append_val (array, handle);

          if (tp_chan_iface_group_remove_members (context->blocked_group_proxy, array, "", &error)) {
            g_array_free (array, TRUE);

            field_state_remove (attr, FIELD_STATE_TOUNBLOCK);
            field_state_remove (attr, FIELD_STATE_BLOCKED);
            modified = TRUE;

            // make sure this handle is known in the group
            v_debug("removing from local and remote block_handles: %d", handle);
            g_intset_remove (context->local_blocked_handles, handle);
            g_intset_remove (context->blocked_handles, handle);

          } else {
            g_warning ("cannot unblock, probably because not blocked in first place: %s", _ERROR_MSG (error));
            g_clear_error (&error);
            g_array_free (array, TRUE);

            field_state_remove (attr, FIELD_STATE_TOUNBLOCK);
            modified = TRUE;

            continue;
          }
        }
      /* endo if TOBLOCK state */
     } else if (field_state_is (attr, FIELD_STATE_BLOCKED) && attribute_is_bound_to (attr, context->account)) {
        v_debug("BLOCKED address");

        guint handle;

        if (!eds_tp_conn_request_handle (DBUS_G_PROXY (context->conn), TP_CONN_HANDLE_TYPE_CONTACT,
                                     e_vcard_attribute_get_values (attr)->data,
                                     &handle, &error)) {
          g_warning ("cannot get handle: %s", _ERROR_MSG (error));
          g_clear_error (&error);

          continue;
        }

        // a blocked contact must be locally known and if no block list, emulate remotely known ...
        v_debug("putting blocked contact in local handles, as contact is locally known: %d", handle);
        g_intset_add (context->local_handles, handle);
        g_intset_add (context->local_blocked_handles, handle);

        if (!context->blocked_group_proxy) {
          g_intset_add (context->blocked_handles, handle);
        }

      /* end of BLOCKED state */
    }
  }

  /* If we get here, and there are no more addresses, then all of the tasks have been done */

  if (!other_addresses && contact_state_is (contact, CONTACT_STATE_DELETED)) {

    v_debug("permanently removing contact from addressbook");
    e_book_async_remove_contact (context->book, contact, NULL, NULL);

  } else if (modified) {

    /* TODO: will this contribute to a race condition? if many
     * mixed accounts/addresses
     */
    v_debug("commiting changed contact to addressbook");
    context->to_commit = g_list_prepend (context->to_commit, g_object_ref (contact));

    if (debug_flags & DEBUG_VERBOSE) {
      gchar * vs = e_vcard_to_string (E_VCARD (contact), EVC_FORMAT_VCARD_30);
      g_debug ("committed: \n%s\n", vs);
      g_free(vs);
    }

  }
}

static gboolean
foreach_todo (gpointer key, gpointer value, gpointer user_data)
{
  GError *error = NULL;
  TelepathyContext *context = user_data;
  char *uid = key;
  EContact *contact;
  
  if (!e_book_get_contact (context->book, uid, &contact, &error)) {
    g_warning ("cannot get contact: %s (probably already permanently deleted)", _ERROR_MSG (error));
    g_clear_error (&error);
  } else {
    // handle new or changed contacts by checking if they contain deleted or added addresses
    handle_local_contact_changed (context, contact);
    
    g_object_unref (contact);
  }
  
  return TRUE;
}


void
do_sync (TelepathyContext *context)
{
  GIntSet *set, *all_remote_handles = NULL;

  e_debug("starting a new do_sync for account: %s %p", context->account, context);

  /* only sync if local and remote lists have fully loaded */
  if (!context->local_contacts_done) return;
  if (!context->remote_subscribe_list_done) return;
  if (!context->known_list_done) return;
  if (!context->blocked_list_done) return;

  if (context->first_run) goto skip_remote;

  all_remote_handles = g_intset_union (context->remote_handles, context->known_handles);

  if (debug_flags & DEBUG_HANDLES) {
    g_debug ("ALL_REMOTE contacts on %s", context->account);
    g_intset_foreach (all_remote_handles, (GIntFunc)print_handle, context->conn);
    g_debug ("LOCAL contacts for %s", context->account);
    g_intset_foreach (context->local_handles, (GIntFunc)print_handle, context->conn);
    g_debug ("PENDING authorization contacts on %s", context->account);
    g_intset_foreach (context->pending_auth_handles, (GIntFunc)print_handle, context->conn);
    g_debug ("BLOCKED contacts on %s", context->account);
    g_intset_foreach (context->blocked_handles, (GIntFunc)print_handle, context->conn);
    g_debug ("LOCAL BLOCKED contacts on %s", context->account);
    g_intset_foreach (context->local_blocked_handles, (GIntFunc)print_handle, context->conn);
    g_debug ("  .  ");
  }

  /* Handles that exist locally but not on the roster */
  set = g_intset_difference (context->local_handles, all_remote_handles);
  g_intset_foreach (set, maybe_remove_local_contact, context);
  g_intset_destroy (set);

  /* The previous operation manipulates the handles list, so regenerate it */
  g_intset_destroy (all_remote_handles);
  all_remote_handles = g_intset_union (context->remote_handles, context->known_handles);

  /* Handles that exist on the roster but not locally */
  set = g_intset_difference (all_remote_handles, context->local_handles);
  g_intset_foreach (set, add_remote_address_to_local_contact, context);
  g_intset_destroy (set);

  /* Handles that are locally blocked but not blocked on the roster */
  set = g_intset_difference (context->local_blocked_handles, context->blocked_handles);
  g_intset_foreach (set, unblock_local_contact, context);
  g_intset_destroy (set);

  /* Handles that are remotely blocked but not blocked locally */
  set = g_intset_difference (context->blocked_handles, context->local_blocked_handles);
  g_intset_foreach (set, block_local_contact, context);
  g_intset_destroy (set);

  g_intset_destroy (all_remote_handles);

skip_remote:

  /* Now work through the list of contacts with stuff to do */
  g_hash_table_foreach_remove (context->local_contacts_todo, foreach_todo, context);

  if (context->first_run) {
      context->first_run = FALSE;
      do_sync(context);
  }

  /* Now commit any contacts that have been modified */
#if HAVE_E_BOOK_ASYNC_COMMIT_CONTACTS
  if (context->to_commit) {
    e_book_async_commit_contacts (context->book, context->to_commit, commit_cb, NULL);
    g_list_foreach (context->to_commit, (GFunc)g_object_unref, NULL);
  }
#else
  {
    GList *l;
    for (l = context->to_commit; l ; l = l->next) {
      EContact *contact = l->data;
      e_book_async_commit_contact (context->book, contact, commit_cb, NULL);
      g_object_unref (contact);
    }
  }
#endif
  g_list_free (context->to_commit);
  context->to_commit = NULL;

  e_debug("done with do_sync for account: %s", context->account);
}
