/*
 * This file is part of cx3110x
 *
 * SPI wrapper for Conexant Softmac driver.
 *
 * Copyright (C) 2004,2005, 2006 Nokia Corporation
 *
 * Contact: Kalle Valo <Kalle.Valo@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * version 2 as published by the Free Software Foundation. 
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */
#include <linux/init.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/irq.h>
#include <linux/dma-mapping.h>
#include <linux/clk.h>
#include <linux/spi/spi.h>

#include <asm/delay.h>
#include <asm/string.h>
#include <asm/irq.h>

#include <asm/arch/gpio.h>
#include <asm/arch/mcspi.h>
#include <asm/arch/dma.h>

#include "sm_drv.h"
#include "sm_drv_spi.h"

/* Bit 15 is read/write bit; ON = READ, OFF = WRITE */
#define ADDR_READ_BIT_15  0x8000

extern struct net_device *sm_net_device;
extern struct omap_wlan_cx3110x_config * wlan_config;

static void cx3110x_hw_reset(void)
{
	/* power down wlan chip */
	omap_set_gpio_dataout(wlan_config->power_gpio, 0);

	msleep(2);

	/* power up wlan chip */
	omap_set_gpio_dataout(wlan_config->power_gpio, 1);

	msleep(300);
}

int cx3110x_request_irq(irqreturn_t (*handler)(int, void *),
			const char * devname, void *dev_id)
{	
	int ret;
	
	ret = request_irq(OMAP_GPIO_IRQ(wlan_config->irq_gpio),
			  handler, SA_INTERRUPT , devname, dev_id);

	if (ret < 0) {
		cx3110x_error("could not install IRQ-handler: %d.", ret);
		return ret;
	}
	
	set_irq_type(OMAP_GPIO_IRQ(wlan_config->irq_gpio), IRQT_RISING);

	return 0;
}

void cx3110x_free_irq(void *dev_id)
{
	free_irq(OMAP_GPIO_IRQ(wlan_config->irq_gpio), dev_id);
}

void cx3110x_disable_irq(struct net_device *dev)
{
	DEBUG(DBG_CALLS, "omap_wlan_disable_irq\n");
	
	disable_irq(OMAP_GPIO_IRQ(wlan_config->irq_gpio));
}


int cx3110x_spi_dma_read(struct net_device *dev, unsigned long address, 
			 void * rx_buffer, unsigned int length)
{
	struct net_local * lp = dev->priv;
	struct spi_transfer t[2];
	struct spi_message m;

	spin_lock_bh(&lp->lock);

	if (length < lp->dma_threshold) {
		spin_unlock_bh(&lp->lock);
		return cx3110x_spi_read(dev, address, rx_buffer, length);
	}
	
	spin_unlock_bh(&lp->lock);

	/* We first push the address */
	address = (address << 8) | ADDR_READ_BIT_15;

	spi_message_init(&m);
	m.is_dma_mapped = 1;
	memset(t, 0, 2 * sizeof(struct spi_transfer));

	t[0].cs_change = 0;
	t[0].bits_per_word = 0;
	t[0].speed_hz = 0;
	t[0].tx_buf = &address;
	t[0].rx_buf = NULL;
	t[0].len = 2;
	spi_message_add_tail(&t[0], &m);

	t[1].cs_change = 0;
	t[1].bits_per_word = 0;
	t[1].speed_hz = 0;
	t[1].tx_buf = NULL;
	t[1].rx_buf = rx_buffer;
	t[1].tx_dma = 0;
	t[1].rx_dma = virt_to_phys(rx_buffer);
	t[1].len = length;
	spi_message_add_tail(&t[1], &m);

	spi_sync(lp->spi_device, &m);



	return 0;
}

int cx3110x_spi_dma_write(struct net_device *dev, unsigned long address,
			  void * tx_buffer, unsigned int length)
{
	struct net_local * lp = dev->priv;
	struct spi_transfer t[3];
	struct spi_message m;

	spin_lock_bh(&lp->lock);

	if (length < lp->dma_threshold) {
		spin_unlock_bh(&lp->lock);
		return cx3110x_spi_write(dev, address, tx_buffer, length);
	}

	spin_unlock_bh(&lp->lock);
	
	/* We first push the address */
	address = address << 8;
	
	spi_message_init(&m);
	m.is_dma_mapped = 1;
	memset(t, 0, 3 * sizeof(struct spi_transfer));

	t[0].cs_change = 0;
	t[0].bits_per_word = 0;
	t[0].speed_hz = 0;
	t[0].tx_buf = &address;
	t[0].rx_buf = NULL;
	t[0].tx_dma = 0;
	t[0].rx_dma = 0;
	t[0].len = 2;
	spi_message_add_tail(&t[0], &m);

	/* Now we push the actual buffer */
	t[1].cs_change = 0;
	t[1].bits_per_word = 0;
	t[1].speed_hz = 0;       
	t[1].tx_buf = tx_buffer;
	t[1].rx_buf = NULL;
	t[1].tx_dma = virt_to_phys(tx_buffer);
	t[1].rx_dma = 0;
	t[1].len = length;
	
	spi_message_add_tail(&t[1], &m);

	/* UMAC may send us odd number of bytes long frames */
	if (length % 2) {
		u16 last_word;

		last_word = *((char *)tx_buffer + length - 1);
		
		t[2].cs_change = 0;
		t[2].bits_per_word = 0;
		t[2].speed_hz = 0;
		t[2].tx_buf = &last_word;
		t[2].rx_buf = NULL;
		t[2].tx_dma = 0;
		t[2].rx_dma = 0;
		t[2].len = 2;
		spi_message_add_tail(&t[2], &m);

	}

	spi_sync(lp->spi_device, &m);

	return 0;
}

int cx3110x_spi_read(struct net_device * dev, unsigned long address,
		     unsigned char * rx_buffer, unsigned int length)
{
	struct net_local * lp = dev->priv;
	struct spi_transfer t[2];
	struct spi_message m;
	
	/* We first push the address */
	address = (address << 8) | ADDR_READ_BIT_15;

	spi_message_init(&m);
	memset(t, 0, 2 * sizeof(struct spi_transfer));

	t[0].cs_change = 0;
	t[0].bits_per_word = 0;
	t[0].speed_hz = 0;
	t[0].tx_buf = &address;
	t[0].rx_buf = NULL;
	t[0].len = 2;
	spi_message_add_tail(&t[0], &m);

	t[1].cs_change = 0;
	t[1].bits_per_word = 0;
	t[1].speed_hz = 0;
	t[1].tx_buf = NULL;
	t[1].rx_buf = rx_buffer;
	t[1].len = length;
	spi_message_add_tail(&t[1], &m);

	spi_sync(lp->spi_device, &m);

	return 0;
}


int cx3110x_spi_write(struct net_device * dev, unsigned long address,
		      unsigned char * tx_buffer, unsigned int length)
{
	struct net_local * lp = dev->priv;
	struct spi_transfer t[3];
	struct spi_message m;

	/* We first push the address */
	address = address << 8;
	
	spi_message_init(&m);
	memset(t, 0, 3 * sizeof(struct spi_transfer));

	t[0].cs_change = 0;
	t[0].bits_per_word = 0;
	t[0].speed_hz = 0;
	t[0].tx_buf = &address;
	t[0].rx_buf = NULL;
	t[0].len = 2;
	spi_message_add_tail(&t[0], &m);

	t[1].cs_change = 0;
	t[1].bits_per_word = 0;
	t[1].speed_hz = 0;
	t[1].tx_buf = tx_buffer;
	t[1].rx_buf = NULL;
	t[1].len = length;
	spi_message_add_tail(&t[1], &m);
	
	if (length % 2) {
		u16 last_word;

		last_word = tx_buffer[length - 1];
		
		t[2].cs_change = 0;
		t[2].bits_per_word = 0;
		t[2].speed_hz = 0;
		t[2].tx_buf = &last_word;
		t[2].rx_buf = NULL;
		t[2].len = 2;
		spi_message_add_tail(&t[2], &m);

	}

	spi_sync(lp->spi_device, &m);

	return 0;
}


int cx3110x_spi_start(struct net_device *dev)
{
	if (omap_request_gpio(wlan_config->power_gpio)) {
		cx3110x_error("power GPIO request failed: %d",
			      wlan_config->power_gpio);
		return -1;
	}

	if (omap_request_gpio(wlan_config->irq_gpio)) {
		cx3110x_error("irq GPIO request failed: %d",
			      wlan_config->irq_gpio);
		return -1;
	}

	omap_set_gpio_direction(wlan_config->power_gpio, 0);

	omap_set_gpio_direction(wlan_config->irq_gpio, 1);

	cx3110x_hw_reset();

	return 0;
}

void cx3110x_spi_stop(struct net_device *dev)
{
	omap_free_gpio(wlan_config->power_gpio);
	omap_free_gpio(wlan_config->irq_gpio);
}

#define WLAN_SPI_FREQ    24000000
int cx3110x_probe(struct spi_device *spi)
{
	struct net_local *lp = sm_net_device->priv;
	
	spi->mode = SPI_MODE_1;
	spi->bits_per_word = 16;
	spi->max_speed_hz = WLAN_SPI_FREQ;
	
	lp->spi_device = spi;
			
	spi_setup(lp->spi_device);

	return 0;
}

int cx3110x_remove(struct spi_device *spi)
{
	return 0;
}

struct cx3110x_spi_reg_s
{
	u16  address;
	u16  length;
	char    *name;
};


static const struct cx3110x_spi_reg_s wlan_registers_array[] =
{
        { SPI_ADRS_ARM_INTERRUPTS,     32, "ARM_INT     "  },
        { SPI_ADRS_ARM_INT_EN,         32, "ARM_INT_ENA "  },
        { SPI_ADRS_HOST_INTERRUPTS,    32, "HOST_INT    "  },
        { SPI_ADRS_HOST_INT_EN,        32, "HOST_INT_ENA"  },
        { SPI_ADRS_HOST_INT_ACK,       32, "HOST_INT_ACK"  },
        { SPI_ADRS_GEN_PURP_1,         32, "GP1_COMM    "  },
        { SPI_ADRS_GEN_PURP_2,         32, "GP2_COMM    "  },
        { SPI_ADRS_DEV_CTRL_STAT,      32, "DEV_CTRL_STA"  },
        { SPI_ADRS_DMA_DATA,           16, "DMA_DATA    "  },
        { SPI_ADRS_DMA_WRITE_CTRL,     16, "DMA_WR_CTRL "  },
        { SPI_ADRS_DMA_WRITE_LEN,      16, "DMA_WR_LEN  "  },
        { SPI_ADRS_DMA_WRITE_BASE,     32, "DMA_WR_BASE "  },
        { SPI_ADRS_DMA_READ_CTRL,      16, "DMA_RD_CTRL "  },
        { SPI_ADRS_DMA_READ_LEN,       16, "DMA_RD_LEN  "  },
        { SPI_ADRS_DMA_WRITE_BASE,     32, "DMA_RD_BASE "  }
};


void cx3110x_dump_register(struct net_device * dev)
{
	uint16_t i;
	uint32_t buffer;

	cx3110x_info("**** WLAN registers ****");
	for (i = 0; i < ARRAY_SIZE(wlan_registers_array); i++) {
		cx3110x_spi_read(dev, wlan_registers_array[i].address, 
				 (unsigned char *)&buffer, 
				 (wlan_registers_array[i].length) >> 3);
		
		if (wlan_registers_array[i].length == 32)
			cx3110x_info("%s -> 0x%08x (32 bits)",
				     wlan_registers_array[i].name, buffer);
		else
			cx3110x_info("%s -> 0x%04x (16 bits)",
				     wlan_registers_array[i].name, buffer);
	}
}
