/*
 * This file is part of cx3110x
 *
 * Wireless extensions for Conexant CX3110x chipset.
 * Based on prism54 ioctls implementation.
 *
 * Copyright (C) 2005, 2006 Nokia Corporation
 *
 * Contact: Kalle Valo <Kalle.Valo@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * version 2 as published by the Free Software Foundation. 
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <linux/types.h>
#include <linux/if_arp.h>

#include "sm_drv.h"
#include "sm_drv_wpa.h"
#include "sm_drv_ioctl.h"

extern unsigned int driver_type;

/* Wireless events helpers */
#define WPA_EID_GENERIC 0xdd
#define WPA2_EID_GENERIC 0x30

static u8 wpa_oid[4] = {0x00, 0x50, 0xf2, 0x1};
static u8 wpa2_oid[2] = {0x01, 0x00};

/* Beacon/ProbeResp payload header */
struct ieee80211_beacon_phdr {
	u8 timestamp[8];
	u16 beacon_int;
	u16 capab_info;
} __attribute__ ((packed));

/* Wireless extensions controls */

static void format_event(struct net_device *dev, char *dest, const char *str,
	     const struct obj_mlme *mlme, u16 *length, int error)
{
	struct net_local *lp = dev->priv;
	const u8 *a = mlme->address;
	int n;
	
	spin_lock_bh(&lp->lock);
	n = snprintf(dest, IW_CUSTOM_MAX,
			 "%s %s %2.2X:%2.2X:%2.2X:%2.2X:%2.2X:%2.2X %s "
			 "(%2.2X)",
			 str,
			 ((lp->sm_mode == SM_MODE_AP) ? "from" : "to"),
			 a[0], a[1], a[2], a[3], a[4], a[5],
			 (error ?
			  (mlme->code ? " : REJECTED " : " : ACCEPTED ")
			  : ""), mlme->code);
	spin_unlock_bh(&lp->lock);

	BUG_ON(n > IW_CUSTOM_MAX);
	*length = n;
}


void send_formatted_event(struct net_device *dev, const char *str,
			  const struct obj_mlme *mlme, int error)
{
	union iwreq_data wrqu;

	wrqu.data.pointer = kmalloc(IW_CUSTOM_MAX, GFP_ATOMIC);
	if (!wrqu.data.pointer)
		return;
	wrqu.data.length = 0;
	format_event(dev, wrqu.data.pointer, str, mlme, &wrqu.data.length,
		     error);
	wireless_send_event(dev, IWEVCUSTOM, &wrqu, wrqu.data.pointer);
	kfree(wrqu.data.pointer);
}

void send_wpa_ie_event(struct net_device *dev, char * bss_addr,
		       char * wpa_ie, size_t wpa_ie_len, uint32_t event)
{
	union iwreq_data wrqu;
	uint32_t we_event;

	if (wpa_ie_len > IW_GENERIC_IE_MAX)
		return;

	switch (event) {
	case DOT11_OID_ASSOCIATE:
	case DOT11_OID_REASSOCIATE:
		we_event = IWEVASSOCREQIE;
		break;
	case DOT11_STATE_ASSOC:
		we_event = IWEVASSOCRESPIE;
		break;
	default:
		cx3110x_warning("%s(): unknown event 0x%x.",
			      __FUNCTION__, event);
		return;
	}

	wrqu.data.pointer = kzalloc(ETH_ALEN + 1 + wpa_ie_len, GFP_ATOMIC);
	if (!wrqu.data.pointer)
		return;

	memcpy(wrqu.data.pointer, bss_addr, ETH_ALEN);
	*((char *)(wrqu.data.pointer + ETH_ALEN)) = ':';	
	memcpy(wrqu.data.pointer + ETH_ALEN + 1, wpa_ie, wpa_ie_len);
	
	wrqu.data.length = ETH_ALEN + 1 + wpa_ie_len;

	wireless_send_event(dev, we_event, &wrqu, wrqu.data.pointer);
	kfree(wrqu.data.pointer);
}

void send_simple_event(struct net_device *dev, const char *str)
{
	union iwreq_data wrqu;
	int n = strlen(str);

	wrqu.data.pointer = kmalloc(IW_CUSTOM_MAX, GFP_ATOMIC);
	if (!wrqu.data.pointer)
		return;
	BUG_ON(n > IW_CUSTOM_MAX);
	wrqu.data.length = n;
	strcpy(wrqu.data.pointer, str);
	wireless_send_event(dev, IWEVCUSTOM, &wrqu, wrqu.data.pointer);
	kfree(wrqu.data.pointer);
}

static size_t sm_drv_wpa_ie_get(struct net_device *dev, u8 *bssid, u8 *wpa_ie)
{
	struct net_local *lp = dev->priv;
	struct list_head *ptr;
	struct sm_drv_bss_wpa_ie *bss = NULL;
	size_t len = 0;


	if (down_trylock(&lp->wpa_sem))
		return 0;

	list_for_each(ptr, &lp->bss_wpa_list) {
		bss = list_entry(ptr, struct sm_drv_bss_wpa_ie, list);
		if (memcmp(bss->bssid, bssid, ETH_ALEN) == 0)
			break;
		bss = NULL;
	}
	if (bss) {
		len = bss->wpa_ie_len;
		memcpy(wpa_ie, bss->wpa_ie, len);
	}
	up(&lp->wpa_sem);

	return len;
}


/*
 * a little helper that will translate our data into a card independent
 * format that the Wireless Tools will understand. This was inspired by
 * the "Aironet driver for 4500 and 4800 series cards" (GPL)
 */
inline char * sm_drv_translate_bss(struct net_device *dev, char *current_ev,
				   char *end_buf, struct obj_bssex *bss,
				   char noise)
{
	struct iw_event iwe;	/* Temporary buffer */
	short cap;
	u8 wpa_ie[MAX_WPA_IE_LEN];
	size_t wpa_ie_len;

	/* The first entry must be the MAC address */
	memcpy(iwe.u.ap_addr.sa_data, bss->address, ETH_ALEN);
	iwe.u.ap_addr.sa_family = ARPHRD_ETHER;
	iwe.cmd = SIOCGIWAP;
	current_ev = iwe_stream_add_event(current_ev, end_buf, &iwe,
					  IW_EV_ADDR_LEN);

	/*
	 * The following entries will be displayed in the same order we
	 * give them.
	 */

	/*
	 * SSID
	 *
	 * First byte of obj_bssex.ssid[] is SSID length, next 32 bytes are
	 * for SSID and last byte is reserved for NULL termination.
	 */
	iwe.u.data.length = bss->ssid[0];
	iwe.u.data.flags = 1;
	iwe.cmd = SIOCGIWESSID;
	current_ev = iwe_stream_add_point(current_ev, end_buf,
					  &iwe, (bss->ssid + 1));

	/* Capabilities */
#define CAP_ESS 0x01
#define CAP_IBSS 0x02
#define CAP_CRYPT 0x10

	/* Mode */
	cap = bss->capinfo;
	iwe.u.mode = 0;
	if (cap & CAP_ESS)
		iwe.u.mode = IW_MODE_MASTER;
	else if (cap & CAP_IBSS)
		iwe.u.mode = IW_MODE_ADHOC;
	iwe.cmd = SIOCGIWMODE;
	if (iwe.u.mode)
		current_ev = iwe_stream_add_event(current_ev, end_buf, &iwe,
						  IW_EV_UINT_LEN);

	/* Encryption capability */
	if (cap & CAP_CRYPT)
		iwe.u.data.flags = IW_ENCODE_ENABLED | IW_ENCODE_NOKEY;
	else
		iwe.u.data.flags = IW_ENCODE_DISABLED;
	iwe.u.data.length = 0;
	iwe.cmd = SIOCGIWENCODE;
	current_ev = iwe_stream_add_point(current_ev, end_buf, &iwe, NULL);

	/* Add frequency. (short) bss->channel is the frequency in MHz */
	iwe.u.freq.m = channel_of_freq(bss->channel);
	iwe.u.freq.e = 0;
	iwe.cmd = SIOCGIWFREQ;
	current_ev = iwe_stream_add_event(current_ev, end_buf, &iwe,
					  IW_EV_FREQ_LEN);

	/* Add quality statistics */
	iwe.u.qual.level = bss->rssi;
	iwe.u.qual.noise = noise;
	/* do a simple SNR for quality */
	iwe.u.qual.qual = bss->rssi - noise;
	iwe.cmd = IWEVQUAL;
	current_ev =
	    iwe_stream_add_event(current_ev, end_buf, &iwe, IW_EV_QUAL_LEN);

	wpa_ie_len = sm_drv_wpa_ie_get(dev, bss->address, wpa_ie);

	if (wpa_ie_len > 0 && wpa_ie_len < MAX_WPA_IE_LEN) {
		memset(&iwe, 0, sizeof(iwe));
                iwe.cmd = IWEVGENIE;
                iwe.u.data.length = wpa_ie_len;
                current_ev = iwe_stream_add_point(
                        current_ev, end_buf, &iwe, wpa_ie);		
	}

	return current_ev;
}


int sm_drv_parse_bssid(struct net_device *dev, uint8_t * mac_filter,
		       uint32_t event, char** event_stream)
{
	struct obj_bssex * bssex, * first_bssex;	
	char * first_ev = NULL;
	uint32_t noise = 0;
	int ret;

	if (event_stream) {
		first_ev = *event_stream;
		ret = sm_drv_oid_get(dev, DOT11_OID_NOISEFLOOR,
				     (void*)&noise, sizeof(uint32_t));	
		if (ret < 0)
			goto out;
	}


	bssex = kzalloc(sizeof(struct obj_bssex) + MAX_PROBE_LENGTH,
			GFP_ATOMIC);
	if (bssex == NULL) {
		ret = -ENOMEM;
		goto out;
	}
	
	first_bssex = kzalloc(sizeof(struct obj_bssex) + MAX_PROBE_LENGTH,
			      GFP_ATOMIC);
	if (first_bssex == NULL) {
		ret = -ENOMEM;
		goto out_free_first;
	}

	/* 
	 * First we need to get a beacon frame for all APs, and
	 * build the wpa list.
	 */
	ret = sm_drv_oid_set(dev, DOT11_OID_BSSITERATE, (void*)bssex,
			     sizeof(struct obj_bssex));
	if (ret < 0)
		goto out_free;

	ret = sm_drv_oid_get(dev, DOT11_OID_BSSITERATE, (void*)first_bssex,
			     sizeof(struct obj_bssex) + MAX_PROBE_LENGTH);
	if (ret < 0)
		goto out_bssiterate;
	
	if ((mac_filter != NULL && !memcmp(first_bssex->address,
					   mac_filter, ETH_ALEN))
	    || mac_filter == NULL) {
		ret = sm_drv_process_bss_data(dev, first_bssex->address,
					      first_bssex->data,
					      first_bssex->length, event);
		/* We have found one and this is the one we were looking for */
		if (ret == 0 &&
		    (mac_filter != NULL && !memcmp(first_bssex->address,
						   mac_filter, ETH_ALEN)))
			goto out_bssiterate;
	}
	
	if (event_stream) {
		*event_stream = sm_drv_translate_bss(dev, *event_stream,
						     first_ev
						     + IW_SCAN_MAX_DATA,
						     first_bssex, noise);
	}
	
	/* BSSITERATE loops forever... */
	do {
		bssex->length = 0;
		ret = sm_drv_oid_get(dev, DOT11_OID_BSSITERATE, (void*)bssex,
				     sizeof(struct obj_bssex)
				     + MAX_PROBE_LENGTH);
		if (ret < 0)
			goto out_bssiterate;

		/*
		 * We're done whenever the length is 0,
		 * or when we're rolling back to the first bssid.
		 */
		if (!bssex->length ||
		    !memcmp(first_bssex->address, bssex->address, ETH_ALEN))
			break;
		
		if ((mac_filter != NULL && !memcmp(bssex->address, mac_filter,
						   ETH_ALEN))
		    || mac_filter == NULL) {
			ret = sm_drv_process_bss_data(dev, bssex->address,
						      bssex->data,
						      bssex->length, event);
			/*
			 * We have found one and this is the one we were
			 * looking for.
			 */
			if (ret == 0 &&
			    (mac_filter != NULL && !memcmp(bssex->address,
							   mac_filter,
							   ETH_ALEN)))
				goto out_bssiterate;
		}
		
		if (event_stream) {
			*event_stream = sm_drv_translate_bss(dev,
							     *event_stream,
							     first_ev
							     + IW_SCAN_MAX_DATA,
							     bssex, noise);
		}
	} while(memcmp(first_bssex->address, bssex->address, ETH_ALEN));


out_bssiterate:
	/*
	 * BSSITERATE needs to called with set as the last thing to unlock
	 * bss db in umac, always.
	 */
	ret = sm_drv_oid_set(dev, DOT11_OID_BSSITERATE, (void*)bssex,
			     sizeof(struct obj_bssex));

 out_free:	
	kfree(first_bssex);
 out_free_first:
	kfree(bssex);
 out:
	return ret;

}

/* In case we don't get a trap from the UMAC...*/
void send_scan_complete_timer(unsigned long handle)
{
	struct net_device *dev = (struct net_device *)handle;
	struct net_local *lp = dev->priv;
	
	spin_lock_bh(&lp->lock);
	lp->device_state = DEVSTATE_RESETTING;
	spin_unlock_bh(&lp->lock);

	queue_work(lp->workqueue, &lp->work);
}

void send_scan_complete(struct net_device *dev)
{
	union iwreq_data wrqu;	
	struct net_local *lp = dev->priv;

	wrqu.data.length = 0;
	wrqu.data.flags = 0;
	
	/* We parse the BSSIDs, but we don't send the event */
	sm_drv_parse_bssid(dev, NULL, 0, NULL);

	spin_lock_bh(&lp->lock);
	if (lp->link_state != DOT11_STATE_ASSOC) {
		uint32_t bgr_scan_disable = 0;
		spin_unlock_bh(&lp->lock);
		sm_drv_oid_set(dev, DOT11_OID_AUTOSCANDISABLE, 
			       (void*)&bgr_scan_disable, sizeof(uint32_t));
		spin_lock_bh(&lp->lock);
	}
	spin_unlock_bh(&lp->lock);
	
	wireless_send_event(dev, SIOCGIWSCAN, &wrqu, NULL);
}

void send_mic_failure(struct net_device *dev, struct obj_sta *sta)
{
	union iwreq_data wrqu;
	struct iw_michaelmicfailure micfailure;
	struct obj_stasc sta_seqcounter;
	char * p;
	
	/* Who is the guilty guy ? */
	memcpy(micfailure.src_addr.sa_data, sta->address, ETH_ALEN);
	micfailure.src_addr.sa_family = ARPHRD_ETHER;

	memcpy(sta_seqcounter.address, sta->address, ETH_ALEN);
	
	/* We need to report the sequence counter as well */
	if (sm_drv_oid_get(dev, DOT11_OID_STASC, (void*)&sta_seqcounter,
			   sizeof(struct obj_stasc)) < 0)
		memset(micfailure.tsc, 0, IW_ENCODE_SEQ_MAX_SIZE);
	else {
		memcpy(micfailure.tsc, &(sta_seqcounter.sc_low), 2);
		memcpy(micfailure.tsc + 2, &(sta_seqcounter.sc_high), 4);
	}
	
	wrqu.data.pointer = kmalloc(IW_CUSTOM_MAX, GFP_ATOMIC);
	if (!wrqu.data.pointer)
		return;
	p = wrqu.data.pointer;

	/* We will be able to send a standard event with WE-18 */
	p += sprintf(wrqu.data.pointer, "MIC_FAILURE: ");
	p += snprintf(p, sizeof(struct iw_michaelmicfailure), "%s",
		      (char*)&micfailure);

	wrqu.data.length = p - (char *)wrqu.data.pointer;
	wireless_send_event(dev, IWEVCUSTOM, &wrqu, wrqu.data.pointer);
	kfree(wrqu.data.pointer);
}

static void sm_drv_wpa_ie_add(struct net_device *dev, uint8_t *bssid,
			      uint8_t *wpa_ie, size_t wpa_ie_len,
			      uint32_t event)
{
	struct net_local *lp = dev->priv;
	struct sm_drv_bss_wpa_ie *bss = NULL, *safe;

	if (wpa_ie_len > MAX_WPA_IE_LEN)
		wpa_ie_len = MAX_WPA_IE_LEN;

	if (down_trylock(&lp->wpa_sem))
		return;
			
	/* We send the event after parsing the association frame */
	if ((lp->link_state == DOT11_STATE_ASSOCING
	     || lp->link_state == DOT11_STATE_ASSOC)  
	    && event)
		send_wpa_ie_event(dev, bssid, wpa_ie, wpa_ie_len, event);
			
	/* try to use existing entry */
	list_for_each_entry_safe(bss, safe, &lp->bss_wpa_list, list) {
		if (memcmp(bss->bssid, bssid, ETH_ALEN) == 0) {
			/*
			 * We update the existing entry, IE can change with
			 * AP settings.
			 */
			memset(bss->wpa_ie, 0, bss->wpa_ie_len);
			memcpy(bss->wpa_ie, wpa_ie, wpa_ie_len);
			memcpy(bss->bssid, bssid, ETH_ALEN);
			bss->wpa_ie_len = wpa_ie_len;
			list_move(&bss->list, &lp->bss_wpa_list);
			bss->last_update = jiffies;
			goto out;
		}
	}
	
	/* Add a new BSS entry; if max number of entries is already
	 * reached, replace the least recently updated */
	if (lp->num_bss_wpa >= MAX_BSS_WPA_IE_COUNT) {
		cx3110x_warning("reached maximum number of APs: %d.",
				lp->num_bss_wpa);
		bss = list_entry(lp->bss_wpa_list.prev,
				 struct sm_drv_bss_wpa_ie, list);
		list_del(&bss->list);
		kfree(bss);
		lp->num_bss_wpa--;
	}
	
	bss = kmalloc(sizeof (struct sm_drv_bss_wpa_ie), 
		      GFP_ATOMIC);
	if (bss != NULL) {
		lp->num_bss_wpa++;
		memset(bss, 0, sizeof (struct sm_drv_bss_wpa_ie));
		memcpy(bss->wpa_ie, wpa_ie, wpa_ie_len);
		memcpy(bss->bssid, bssid, ETH_ALEN);
		bss->wpa_ie_len = wpa_ie_len;
		bss->last_update = jiffies;
		INIT_LIST_HEAD(&bss->list);
		list_add(&bss->list, &lp->bss_wpa_list);
	} else {
		cx3110x_warning("failed to add BSS WPA entry for " MACSTR ".",
				MAC2STR(bssid));
	}

 out:
	/* expire old entries from WPA list */
	list_for_each_entry_safe(bss, safe, &lp->bss_wpa_list, list) {
		if (time_after(jiffies, bss->last_update + 60 * HZ)) {
			DEBUG(DBG_WPA, "Haven't got any beacons from "
			      MACSTR ", removing...\n", MAC2STR(bss->bssid));
			list_del(&(bss->list));
			lp->num_bss_wpa--;
			kfree(bss);
		}
	}

	up(&lp->wpa_sem);
}

static void sm_drv_wpa_ie_remove(struct net_device *dev, uint8_t *bssid)
{
	struct net_local *lp = dev->priv;
	struct list_head *ptr;
	struct sm_drv_bss_wpa_ie *bss = NULL;
	
	if (down_trylock(&lp->wpa_sem))
		return;
	
	list_for_each(ptr, &lp->bss_wpa_list) {
		bss = list_entry(ptr, struct sm_drv_bss_wpa_ie, list);
		if (memcmp(bss->bssid, bssid, ETH_ALEN) == 0) {
			/* AP is here, removing it...*/
			DEBUG(DBG_WPA, "Removing " MACSTR
			      " from wpa list...\n", MAC2STR(bssid));
			list_del(&bss->list);
			lp->num_bss_wpa--;
			kfree(bss);
			break;
		}
	}	
	
	up(&lp->wpa_sem);
}

void sm_drv_wpa_ie_clean(struct net_device *dev)
{
	struct net_local *lp = dev->priv;
	struct list_head *ptr, *n;

	if (down_trylock(&lp->wpa_sem))
		return;

	list_for_each_safe(ptr, n, &lp->bss_wpa_list) {
		struct  sm_drv_bss_wpa_ie *bss;
		bss = list_entry(ptr, struct sm_drv_bss_wpa_ie, list);
		list_del(&bss->list);
		kfree(bss);
	}
	lp->num_bss_wpa = 0;

	up(&lp->wpa_sem);
}

static void sm_drv_update_stats(struct net_device *dev)
{
	struct net_local *lp = netdev_priv(dev);
	struct obj_bss bss;
	uint8_t bssid[ETH_ALEN];
	uint32_t noise_floor, rx_failed, tx_failed;

	sm_drv_oid_get(dev, DOT11_OID_NOISEFLOOR, (void*)&noise_floor,
		       sizeof(uint32_t));

	/* copy this MAC to the bss */
	bss.ext = 1;
	if (sm_drv_oid_get(dev, DOT11_OID_BSSID, (void*)bssid,
			   ETH_ALEN * sizeof(uint8_t)) < 0)
		return;
	memcpy(bss.address, bssid, ETH_ALEN);

	/* now ask for the corresponding bss */
	if (sm_drv_oid_get(dev, DOT11_OID_BSSFIND, (void *)&bss,
			   sizeof(struct obj_bss)) < 0) {
		bss.ext = 0;
		if (sm_drv_oid_get(dev, DOT11_OID_BSSFIND, (void *)&bss,
				   sizeof(struct obj_bss)) < 0)
			return;
	}

        /* Rx : unable to decrypt the MPDU */
	sm_drv_oid_get(dev, DOT11_OID_PRIVRXFAILED, &rx_failed,
		       sizeof(uint32_t));

        /* Tx : Max MAC retries num reached */
	sm_drv_oid_get(dev, DOT11_OID_MPDUTXFAILED, &tx_failed,
		       sizeof(uint32_t));

	spin_lock_bh(&lp->lock);

	/*
	 * report the rssi and use it to calculate link quality through a
	 * signal-noise ratio
	 */
	lp->iwstatistics.qual.noise = noise_floor;
	lp->iwstatistics.qual.level = bss.rssi;
	lp->iwstatistics.qual.qual = bss.rssi - noise_floor;

	/* report that the stats are new */
	lp->iwstatistics.qual.updated = 0x7;

	lp->iwstatistics.discard.code = rx_failed;
	lp->iwstatistics.discard.retries = tx_failed;

	spin_unlock_bh(&lp->lock);

	return;
}

struct iw_statistics * sm_drv_get_wireless_stats(struct net_device *dev)
{
	struct net_local *lp = dev->priv;

	spin_lock_bh(&lp->lock);

	if(!lp->sm_initialization) {
		memset(&lp->iwstatistics, 0, sizeof(struct iw_statistics));
		goto out;
	}
	
	lp->iwstatistics.qual.updated = 0;

	/* 
	 * Update our wireless stats, but do not schedule to often 
	 * (max 1 HZ) 
	 */
	if ((lp->stats_timestamp == 0) ||
	    time_after(jiffies, lp->stats_timestamp + 1 * HZ)) {
		spin_unlock_bh(&lp->lock);
		sm_drv_update_stats(dev);
		spin_lock_bh(&lp->lock);
		lp->stats_timestamp = jiffies;
	}

out:

	spin_unlock_bh(&lp->lock);

	return &lp->iwstatistics;
}


static int sm_drv_get_name(struct net_device *dev,
			   struct iw_request_info *info,
			   char *cwrq, char *extra)
{
	struct net_local *lp = dev->priv;
	char *capabilities;
	uint32_t ret, profile;

	DEBUG(DBG_IOCTL, "GET NAME\n");

	spin_lock_bh(&lp->lock);
	if(!lp->sm_initialization) {
		capabilities = "NOT READY";
		ret = 0;
		spin_unlock_bh(&lp->lock);
		goto out;
	}
	spin_unlock_bh(&lp->lock);

	ret = sm_drv_oid_get(dev, DOT11_OID_PROFILES, (void *)&profile,
			     sizeof(uint32_t));
	
	if (ret < 0) {
		capabilities = "ERROR";
		goto out;
	}

	DEBUG(DBG_IOCTL, "sm_drv_get_name: name-> 0x%x\n", profile);
	switch (profile) {
	case DOT11_PROFILE_B_ONLY:
		capabilities = "IEEE 802.11b";
		break;
	case DOT11_PROFILE_MIXED:
		capabilities = "IEEE 802.11b/g";
		break;
	case DOT11_PROFILE_MIXED_LONG:
		capabilities = "IEEE 802.11b/g-long";
		break;
	case DOT11_PROFILE_G_ONLY:
		capabilities = "IEEE 802.11g";
		break;
	case DOT11_PROFILE_TEST:
		capabilities = "TEST profile";
		break;
	case DOT11_PROFILE_B_WIFI:
		capabilities = "IEEE 802.11b wifi";
		break;
	case DOT11_PROFILE_MIXED_SHORT:
		capabilities = "IEEE 802.1b/g-short";
		break;
	default:
		capabilities = "Unknown profile";	/* Default */
		break;
	}

 out:
	strncpy(cwrq, capabilities, IFNAMSIZ);
	return ret;
}

static int sm_drv_set_freq(struct net_device *dev,
			   struct iw_request_info *info,
			   struct iw_freq *fwrq, char *extra)
{
	int ret = 0;
	uint32_t freq;

	DEBUG(DBG_IOCTL, "SET FREQ: fwrq->m : %d fwrq->e : %d\n",
	      fwrq->m, fwrq->e);

	if (fwrq->m < 1000)
		/* we have a channel number */
		freq = freq_of_channel(fwrq->m);
	else
		freq = fwrq->m/1000;

	DEBUG(DBG_IOCTL, "SET FREQ: freq is %d\n", freq);
	
	if (!freq)
		return -EINVAL;
	else {
		
		ret = sm_drv_oid_set(dev, DOT11_OID_FREQUENCY,
				     (void *)&freq, sizeof(uint32_t));
		if (ret < 0)
			return ret;
	}
	
	return -EINPROGRESS;
}

static int sm_drv_get_freq(struct net_device *dev,
			   struct iw_request_info *info,
			   struct iw_freq *fwrq, char *extra)
{	
	uint32_t data[2];

	DEBUG(DBG_IOCTL, "GET FREQ\n");

	sm_drv_oid_get(dev, DOT11_OID_FREQUENCY, (void *)data,
		       2 * sizeof(uint32_t));
		
	fwrq->m = channel_of_freq(data[0] / 1000);
	fwrq->e = 0;

	return 0;
}


static int sm_drv_set_mode(struct net_device *dev,
			   struct iw_request_info *info,
			   __u32 * uwrq, char *extra)
{
	struct net_local * lp = dev->priv;
	uint32_t sm_mode, mode, current_mode, err, bss_type, profile,
		ret, commit = 0;
	
	DEBUG(DBG_IOCTL, "SET MODE\n");

	mode = *(uint32_t *)uwrq;

	switch (mode) {
	case IW_MODE_INFRA:
		mode = DOT11_MODE_CLIENT;
		sm_mode = SM_MODE_CLIENT;
		bss_type = DOT11_BSSTYPE_INFRA;
		profile = DOT11_PROFILE_MIXED;
		dev->type = ARPHRD_ETHER;
		break;
	case IW_MODE_ADHOC:
		mode = DOT11_MODE_CLIENT;
		sm_mode = SM_MODE_CLIENT;
		bss_type = DOT11_BSSTYPE_IBSS;
		/*
		 * When creating an Ad-Hoc network, we must be B only (wifi
		 * requirement)
		 */
		profile = DOT11_PROFILE_B_ONLY;
		dev->type = ARPHRD_ETHER;
		break;
	case IW_MODE_MASTER:
		return -1;
#if 0 /* AP mode unsupported for now */
		mode = DOT11_MODE_AP;
		sm_mode = SM_MODE_AP;
		bss_type = DOT11_BSSTYPE_INFRA;
		profile = DOT11_PROFILE_MIXED;
		break;
#endif
	case IW_MODE_MONITOR:
		mode = MODE_PROMISCUOUS;
		sm_mode = SM_MODE_PROMISCUOUS;
		bss_type = DOT11_BSSTYPE_NONE;
		dev->type = ARPHRD_IEEE80211;
		break;
		
	default:
		return -EINVAL;

	}
	
	ret = sm_drv_oid_get(dev, GEN_OID_MODE, (void *)&current_mode,
			     sizeof(uint32_t));
	if (ret < 0)
		return ret;
	
	DEBUG(DBG_IOCTL, "sm_drv_ioctl: Setting Mode to %d (current: %d)\n",
	      mode, current_mode);

	if (mode != current_mode) {		
		spin_lock_bh(&lp->lock);
		lp->sm_mode = sm_mode;
		spin_unlock_bh(&lp->lock);
		
		err = sm_drv_reset(dev, sm_mode);
		if(err < 0)
			return -EINVAL;
	}	

	ret = sm_drv_oid_set(dev, DOT11_OID_PROFILES, (void*)&profile,
			     sizeof(uint32_t));
	if (ret < 0)
		return ret;

	if (bss_type == DOT11_BSSTYPE_IBSS) {
		uint32_t preamble_settings = DOT11_PREAMBLESETTING_LONG;
		ret = sm_drv_oid_set(dev, DOT11_OID_PREAMBLESETTINGS, 
				     (void*)&preamble_settings,
				     sizeof(uint32_t));
		if (ret < 0)
			return ret;
	}

	ret = sm_drv_oid_set(dev, DOT11_OID_BSSTYPE, (void*)&bss_type,
			     sizeof(uint32_t));	
	if (ret < 0)
		return ret;

	spin_lock_bh(&lp->lock);
	lp->bss_type = bss_type;
	spin_unlock_bh(&lp->lock);

	ret = sm_drv_oid_set(dev, GEN_OID_COMMIT, (void*)&commit,
			     sizeof(uint32_t));
	if (ret < 0)
		return ret;
	
	return 0;
}


int sm_drv_get_mode(struct net_device *dev, struct iw_request_info *info,
		    __u32 * uwrq, char *extra)
{
	uint32_t mode, ret, bss_type;
	

	DEBUG(DBG_IOCTL, "GET MODE\n");

	ret = sm_drv_oid_get(dev, GEN_OID_MODE, (void *)&mode,
			     sizeof(uint32_t));
	if (ret < 0)
		return ret;

	ret = sm_drv_oid_get(dev, DOT11_OID_BSSTYPE, (void *)&bss_type,
			     sizeof(uint32_t));
	if (ret < 0)
		return ret;

	DEBUG(DBG_IOCTL, "sm_drv_get_mode: mode-> 0x%x\n", mode);
	switch (mode & 0xff) {
	case DOT11_MODE_CLIENT:
		if (bss_type == DOT11_BSSTYPE_INFRA)
			mode = IW_MODE_INFRA; /* Client Mode */
		else if (bss_type == DOT11_BSSTYPE_IBSS)
			mode = IW_MODE_ADHOC; /* Ad-Hoc mode */
		break;
	case DOT11_MODE_AP:
		mode = IW_MODE_MASTER; /* AP Mode */
		break;
	case MODE_PROMISCUOUS:
		mode = IW_MODE_MONITOR; /* Listen only */
		break;
	default:
		mode = IW_MODE_AUTO;	/* Default */
		break;
	}

	*uwrq = mode;
	return 0;
}


/* 
 * We use DOT11_OID_EDTHRESHOLD. From what I guess the card will not try to
 * emit data if (sensitivity > rssi - noise) (in dBm).
 */

static int sm_drv_set_sens(struct net_device *dev,
			   struct iw_request_info *info,
			   struct iw_param *vwrq, char *extra)
{
	uint32_t sens;

	DEBUG(DBG_IOCTL, "SET SENS\n");
	/* by default  the card sets this to 20. */
	sens = vwrq->disabled ? 20 : vwrq->value;

	/* set the ed threshold. */
	return sm_drv_oid_set(dev, DOT11_OID_EDTHRESHOLD, (void*)&sens,
			      sizeof(uint32_t));	
}

static int sm_drv_get_sens(struct net_device *dev,
			   struct iw_request_info *info,
			   struct iw_param *vwrq, char *extra)
{
	int ret;
	uint32_t sens;

	DEBUG(DBG_IOCTL, "GET SENS\n");
	ret = sm_drv_oid_get(dev, DOT11_OID_EDTHRESHOLD, (void*)&sens,
			     sizeof(uint32_t));	
	if (ret < 0)
		return ret;

	vwrq->value = sens;
	vwrq->disabled = (vwrq->value == 0);
	vwrq->fixed = 1;

	return 0;
}


static int sm_drv_get_range(struct net_device *dev,
			    struct iw_request_info *info,
			    struct iw_point *dwrq, char *extra)
{
	struct iw_range *range = (struct iw_range *) extra;
	uint8_t data[17];
	int i, ret, freq_obj_size;	
	struct obj_frequencies * freq;
	void * frequencies;

	DEBUG(DBG_IOCTL, "GET RANGE\n");

	freq_obj_size = (IW_MAX_FREQUENCIES + 1)* 2 * sizeof(uint8_t);
	frequencies = kmalloc(freq_obj_size, GFP_ATOMIC);
	if (frequencies == NULL) {
		return -ENOMEM;
	}

	memset(range, 0, sizeof (struct iw_range));
	dwrq->length = sizeof (struct iw_range);

	/* set the wireless extension version number */
	range->we_version_source = SUPPORTED_WIRELESS_EXT;
	range->we_version_compiled = WIRELESS_EXT;

	/* Now the encoding capabilities */
	range->num_encoding_sizes = 3;
	/* 64(40) bits WEP */
	range->encoding_size[0] = 5;
	/* 128(104) bits WEP */
	range->encoding_size[1] = 13;
	/* 256 bits for WPA-PSK */
	range->encoding_size[2] = 32;
	/* 4 keys are allowed */
	range->max_encoding_tokens = 4;

	/* we don't know the quality range... */
	range->max_qual.level = 0;
	range->max_qual.noise = 0;
	range->max_qual.qual = 0;
	/* these value describe an average quality. Needs more tweaking... */
	range->avg_qual.level = -80;	/* -80 dBm */
	range->avg_qual.noise = 0;	/* don't know what to put here */
	range->avg_qual.qual = 0;

	range->sensitivity = 200;

	/* retry limit capabilities */
	range->retry_capa = IW_RETRY_LIMIT | IW_RETRY_LIFETIME;
	range->retry_flags = IW_RETRY_LIMIT;
	range->r_time_flags = IW_RETRY_LIFETIME;

	/* I don't know the range. Put stupid things here */
	range->min_retry = 1;
	range->max_retry = 65535;
	range->min_r_time = 1024;
	range->max_r_time = 65535 * 1024;

	/* txpower is supported in dBm's */
	range->txpower_capa = IW_TXPOW_DBM;


	/* 
	 * Request the device for the supported frequencies
	 * not really revelant since some devices will report the 5 GHz band
	 * frequencies even if they don't support them.
	 */
	ret = sm_drv_oid_get(dev, DOT11_OID_SUPPORTEDFREQUENCIES,
			     frequencies, freq_obj_size);
	if (ret < 0) {
		kfree(frequencies);
		return ret;
	}

	freq = (struct obj_frequencies *)frequencies;

	range->num_channels = freq->nr;
	range->num_frequency = freq->nr;

	for (i = 0; i < (int)freq->nr; i++) {
		range->freq[i].m = freq->mhz[i];
		range->freq[i].e = 6;
		range->freq[i].i = i + 1;
	}
#if 0
	/* 
	 * Frequencies are not listed in the right order. The reordering is
	 * probably firmware dependant and thus should work for everyone.
	 */
	m = min(IW_MAX_FREQUENCIES, (int) freq->nr);
	for (i = 0; i < m - 12; i++) {
		range->freq[i].m = freq->mhz[12 + i];
		range->freq[i].e = 6;
		range->freq[i].i = i + 1;
	}
	for (i = m - 12; i < m; i++) {
		range->freq[i].m = freq->mhz[i - m + 12];
		range->freq[i].e = 6;
		range->freq[i].i = i + 23;
	}
#endif
	kfree(frequencies);
	ret = sm_drv_oid_get(dev, DOT11_OID_SUPPORTEDRATES,
			     (void *)data, 17 * sizeof(uint8_t));
	if (ret < 0)
		return ret;

	/* We got an array of char. It is NULL terminated. */
	i = 0;
	while ((i < IW_MAX_BITRATES) && (data[i] != 0)) {
		/* the result must be in bps. The card gives us 500Kbps */
		range->bitrate[i] = (__s32) (data[i] >> 1);
		range->bitrate[i] *= 1000000;
		i++;
	}

	range->num_bitrates = i;

	return 0;
}


static int sm_drv_set_wap(struct net_device *dev, struct iw_request_info *info,
			  struct sockaddr *awrq, char *extra)
{
	uint8_t bssid[ETH_ALEN];
	int ret;

	DEBUG(DBG_IOCTL, "SET WAP\n");
	if (awrq->sa_family != ARPHRD_ETHER)
		return -EINVAL;

	/* prepare the structure for the set object */
	memcpy(bssid, awrq->sa_data, ETH_ALEN);

	/* set the bssid -- does this make sense when in AP mode? */
	ret = sm_drv_oid_set(dev, DOT11_OID_BSSID, (void*)bssid,
			     ETH_ALEN * sizeof(uint8_t));
	if (ret < 0)
		return ret;

	return -EINPROGRESS;	/* Call commit handler */
}


int sm_drv_get_wap(struct net_device *dev, struct iw_request_info *info,
		   struct sockaddr *awrq, char *extra)
{
	uint8_t bssid[ETH_ALEN] = {0x0,0x0,0x0,0x0,0x0,0x0};

	DEBUG(DBG_IOCTL, "GET WAP\n");
	sm_drv_oid_get(dev, DOT11_OID_BSSID, (void*)bssid,
		       ETH_ALEN * sizeof(uint8_t));
	
	memcpy(awrq->sa_data, bssid, ETH_ALEN);
	awrq->sa_family = ARPHRD_ETHER;

	return 0;
}


static int sm_drv_set_scan(struct net_device *dev, struct iw_request_info *info,
			   struct iw_point *vwrq, char *extra)
{
	int ret = 0, l, msecs;
	struct obj_ssid essid;
	int16_t scan = -1;
	uint32_t scan_mode, flush_bss_list = 1;
	struct net_local *lp = dev->priv;
	uint32_t bgr_scan_disable = 1;

	memset(essid.octets, 0, sizeof(essid.octets));

	if (vwrq == NULL)
		return -EINVAL;

	/* First we flush the UMAC's AP list*/
	ret = sm_drv_oid_set(dev, DOT11_OID_BSSLISTFLUSH,
			     (void*)&flush_bss_list, sizeof(uint32_t));
	if (ret < 0)
		return ret;

	if (vwrq->flags & IW_SCAN_THIS_ESSID) {
		if (vwrq->length > 0) {
			l = vwrq->length;
			essid.length = min(l, IW_ESSID_MAX_SIZE);
			memcpy(essid.octets, vwrq->pointer, essid.length);
		} else {
			essid.length = 0;
		}
		scan_mode = SCAN_MODE_ACTIVE;
	} else {
		essid.length = 0;
		scan_mode = SCAN_MODE_PASSIVE;
	}

	ret = sm_drv_oid_set(dev, DOT11_OID_SCANMODE, (void*)&scan_mode,
			     sizeof(uint32_t));
	if (ret < 0)
		return ret;

	ret = sm_drv_oid_set(dev, DOT11_OID_SCANSSID, (void*)&essid,
			     sizeof(struct obj_ssid));
	if (ret < 0)
		return ret;

	spin_lock_bh(&lp->lock);
	
	/* We let the background scanning work a bit...*/
	if (scan_mode == SCAN_MODE_PASSIVE
	    && lp->link_state != DOT11_STATE_ASSOC)
		msecs = 2000;
	else
		/* for active scan, the delay can be smaller */
		msecs = 30;

	spin_unlock_bh(&lp->lock);

	msleep(msecs);

	spin_lock_bh(&lp->lock);
	
	/*
	 * Let's start the scan timer in case UMAC doesn't trap the scan
	 * event
	 */
	mod_timer(&lp->scan_timer, jiffies + 4 * HZ);
	
	if (lp->link_state != DOT11_STATE_ASSOC && 
	    lp->bss_type != DOT11_BSSTYPE_IBSS) {
		spin_unlock_bh(&lp->lock);
		ret = sm_drv_oid_set(dev, DOT11_OID_AUTOSCANDISABLE,
				     (void*)&bgr_scan_disable,
				     sizeof(uint32_t));
		spin_lock_bh(&lp->lock);
		if (ret < 0)
			return ret;
	}
	spin_unlock_bh(&lp->lock);

	/* And finally we send the scan request */
	ret = sm_drv_oid_set(dev, DOT11_OID_SCAN, (void*)&scan,
			     sizeof(int16_t));

	if (ret < 0) {
		/* 
		 * If we're associated, we haven't disable bgr scan,
		 * so we don't need to go there.
		 */
		spin_lock_bh(&lp->lock);
		if (lp->link_state != DOT11_STATE_ASSOC &&
		    lp->bss_type != DOT11_BSSTYPE_IBSS) {
			spin_unlock_bh(&lp->lock);
			goto scan_err_out;
		}
		spin_unlock_bh(&lp->lock);
	}

	return 0;

 scan_err_out:
	cx3110x_warning("scanning failed (%d), enabling background scan.",
			ret);
	bgr_scan_disable = 0;
	sm_drv_oid_set(dev, DOT11_OID_AUTOSCANDISABLE,
		       (void*)&bgr_scan_disable, sizeof(uint32_t));
	
	return ret;
}

int sm_drv_get_scan(struct net_device *dev, struct iw_request_info *info,
		    struct iw_point *dwrq, char *extra)
{
	struct net_local *lp = netdev_priv(dev);
	int ret;
	char *current_ev = extra;
	
	DEBUG(DBG_IOCTL, "GET SCAN\n");

	spin_lock_bh(&lp->lock);
	if (timer_pending(&lp->scan_timer)) {
		spin_unlock_bh(&lp->lock);
		return -EAGAIN;
	}
	spin_unlock_bh(&lp->lock);
	
	ret = sm_drv_parse_bssid(dev, NULL, 0, &current_ev);
	if (ret < 0)
		return ret;

	dwrq->length = (current_ev - extra);
	dwrq->flags = 0;	
	
	return 0;
}

static int set_auth_order(struct net_device *dev, uint16_t flags)
{
	uint8_t auth_order[4];
	
	DEBUG(DBG_WPA, "AUTHORDER flags: 0x%x\n", flags);

	switch (flags) {
	case IW_AUTH_NOWPA:		
		auth_order[0] = DOT11_AUTHALG_OS;
 		auth_order[1] = DOT11_AUTHALG_SK;
		auth_order[2] = DOT11_AUTHALG_NONE;
		break;
	case IW_AUTH_WPA:
		DEBUG(DBG_WPA, "WPA authorder\n");
		auth_order[0] = DOT11_AUTHALG_WPA;
		auth_order[1] = DOT11_AUTHALG_NONE;
		break;
	case IW_AUTH_WPA_PSK:
		DEBUG(DBG_WPA, "WPA_PSK authorder\n");
		auth_order[0] = DOT11_AUTHALG_WPAPSK;
		auth_order[1] = DOT11_AUTHALG_NONE;
		break;
	case IW_AUTH_WPA2:
		DEBUG(DBG_WPA, "WPA2 authorder\n");
		auth_order[0] = DOT11_AUTHALG_WPA2;
		auth_order[1] = DOT11_AUTHALG_NONE;
		break;
	case IW_AUTH_WPA2_PSK:
		DEBUG(DBG_WPA, "WPA2_PSK authorder\n");
		auth_order[0] = DOT11_AUTHALG_WPA2PSK;
		auth_order[1] = DOT11_AUTHALG_NONE;
		break;
	default:
		cx3110x_debug(DBG_WPA, "using default authentication order.");
		auth_order[0] = DOT11_AUTHALG_OS;
 		auth_order[1] = DOT11_AUTHALG_SK;
		auth_order[2] = DOT11_AUTHALG_NONE;
		break;
	}

	return sm_drv_oid_set(dev, DOT11_OID_AUTHORDER, (void*)auth_order,
			      4 * sizeof(uint8_t));
}

static int sm_drv_set_essid(struct net_device *dev,
			    struct iw_request_info *info,
			    struct iw_point *dwrq, char *extra)
{
	struct net_local *lp = netdev_priv(dev);
	uint32_t ret;
	int l;
	
	DEBUG(DBG_IOCTL, "SET ESSID\n");

	spin_lock_bh(&lp->lock);

	lp->link_state = DOT11_STATE_ASSOCING;
	
	/* Check if we were asked for `any' */
	if (dwrq->flags && dwrq->length) {
		l = sizeof(lp->ssid.octets);
		if (dwrq->length > min(l, IW_ESSID_MAX_SIZE))
			return -E2BIG;
		memset(&lp->ssid, 0, sizeof(lp->ssid));
		lp->ssid.length = dwrq->length;
		memcpy(lp->ssid.octets, extra, dwrq->length);
	} else
		lp->ssid.length = 0;

	spin_unlock_bh(&lp->lock);
	
	ret = set_auth_order(dev, dwrq->flags & IW_AUTH_MASK);
	if (ret < 0)
		goto out;

	ret = sm_drv_oid_set(dev, DOT11_OID_SSID, &lp->ssid,
			     sizeof(struct obj_ssid));
	
out:
	return ret;
}

static int sm_drv_get_essid(struct net_device *dev,
			    struct iw_request_info *info,
			    struct iw_point *dwrq, char *extra)
{
	struct net_local *lp = netdev_priv(dev);

	DEBUG(DBG_IOCTL, "GET ESSID\n");

	spin_lock_bh(&lp->lock);

	if (lp->ssid.length > 0) {
		/* set ESSID to ON for Wireless Extensions */
		dwrq->flags = 1;
		/* if it is to big, trunk it */
		dwrq->length = min((int) lp->ssid.length, IW_ESSID_MAX_SIZE);
	} else {
		dwrq->flags = 0;
		dwrq->length = 0;
	}
	
	memcpy(extra, lp->ssid.octets, dwrq->length);

	spin_unlock_bh(&lp->lock);
	
	return 0;
}

static int sm_drv_set_rate(struct net_device *dev,
			   struct iw_request_info *info,
			   struct iw_param *vwrq, char *extra)
{

	u32 rate;
	uint8_t data[17];
	int ret, i;
	
	DEBUG(DBG_IOCTL, "SET RATE\n");

	/*
	 * First we need to get the supported rates
	 */
	ret = sm_drv_oid_get(dev, DOT11_OID_SUPPORTEDRATES, (void *)data,
			     17 * sizeof(uint8_t));
	if (ret < 0)
		return ret;

	rate = (u32) (vwrq->value / 500000);
	i = 0;

	while (data[i]) {
		DEBUG(DBG_IOCTL, "rate[%d]->%d\n", i, data[i]);
		if (rate && (data[i] == rate)) {
			break;
		}
		if (vwrq->value == i) {
			break;
		}
		data[i] |= 0x80;
		i++;
	}
		
	if (!data[i])
		return -EINVAL;
	
	data[i] |= 0x80;
	data[i + 1] = 0;
	
	/* Now, check if we want a fixed or auto value */
	if (vwrq->fixed) {
		data[0] = data[i];
		data[1] = 0;
	}

	ret = sm_drv_oid_set(dev, DOT11_OID_EXTENDEDRATES, (void *)data,
			     17 * sizeof(uint8_t));
	if (ret < 0)
		return ret;
	
	ret = sm_drv_oid_set(dev, DOT11_OID_RATES, (void *)data,
			     17 * sizeof(uint8_t));
	if (ret < 0)
		return ret;
	
	return -EINPROGRESS;
}

/* Get the current bit rate */
static int sm_drv_get_rate(struct net_device *dev,
			   struct iw_request_info *info,
			   struct iw_param *vwrq, char *extra)
{
	int ret;
	uint32_t state;

	DEBUG(DBG_IOCTL, "GET RATE\n");
	
	/* Get the current bit rate */
	ret = sm_drv_oid_get(dev, GEN_OID_LINKSTATE, (void *)&state,
			     sizeof(uint32_t));
	if (ret < 0)
		return ret;

	vwrq->value = state * 500000;
	vwrq->fixed = 1;
	
 
	return 0;
}



static int sm_drv_set_rts(struct net_device *dev, struct iw_request_info *info,
			  struct iw_param *vwrq, char *extra)
{
	DEBUG(DBG_IOCTL, "SET RTS\n");
	
	return sm_drv_oid_set(dev, DOT11_OID_RTSTHRESH,
			      (void*)&vwrq->value, sizeof(uint32_t));
}

static int sm_drv_get_rts(struct net_device *dev, struct iw_request_info *info,
			  struct iw_param *vwrq, char *extra)
{
	uint32_t rts, ret;
	
	DEBUG(DBG_IOCTL, "SET RTS\n");

	/* get the rts threshold */
	ret = sm_drv_oid_get(dev, DOT11_OID_RTSTHRESH, (void *)&rts,
			     sizeof(uint32_t));
	if (ret < 0)
		return ret;
	
	vwrq->value = rts;
	
	return ret;
}

static int sm_drv_set_frag(struct net_device *dev,
			   struct iw_request_info *info,
			   struct iw_param *vwrq, char *extra)
{
	DEBUG(DBG_IOCTL, "SET RTS\n");
	
	return sm_drv_oid_set(dev, DOT11_OID_FRAGTHRESH,
			      (void*)&vwrq->value, sizeof(uint32_t));
}

static int sm_drv_get_frag(struct net_device *dev,
			   struct iw_request_info *info,
			   struct iw_param *vwrq, char *extra)
{
	uint32_t frag, ret;
	
	DEBUG(DBG_IOCTL, "SET RTS\n");

	/* get the rts threshold */
	ret = sm_drv_oid_get(dev, DOT11_OID_FRAGTHRESH, (void *)&frag,
			     sizeof(uint32_t));
	if (ret < 0)
		return ret;
	
	vwrq->value = frag;
	
	return ret;
}



static int sm_drv_set_txpower(struct net_device *dev,
			      struct iw_request_info *info,
			      struct iw_param *vwrq, char *extra)
{	
	uint32_t output_power = vwrq->value;

	DEBUG(DBG_IOCTL, "SET TXPOWER\n");
	
	/* Conexant firmware operates in 1/4 dBm */
	output_power *= 4;
	if (vwrq->disabled) {
		cx3110x_info("radio OFF not supported.");
		return 0;
	} else if (vwrq->fixed) {
		return sm_drv_oid_set(dev, DOT11_OID_OUTPUTPOWER, (void *)&output_power, sizeof(uint32_t));
	} else {
		DEBUG(DBG_IOCTL, "Radio is ON\n");
		return 0;
	}
}



static int sm_drv_get_txpower(struct net_device *dev,
			      struct iw_request_info *info,
			      struct iw_param *vwrq, char *extra)
{
	int ret;
	uint32_t output_power;
	
	DEBUG(DBG_IOCTL, "GET TXPOWER\n");

	ret = sm_drv_oid_get(dev, DOT11_OID_OUTPUTPOWER,
			     (void *)&output_power, sizeof(uint32_t));
	if (ret < 0)
		return ret;

        /* Conexant firmware operates in 0.25 dBm (1/4 dBm) */	
	vwrq->value = output_power/4;
	vwrq->fixed = 1;
	
        /* 
	 * Radio is not turned off
	 * btw: how is possible to turn off only the radio 
	 */
	vwrq->disabled = 0;

	return ret;
}


static int sm_drv_set_encode(struct net_device *dev,
			     struct iw_request_info *info,
			     struct iw_point *dwrq, char *extra)
{
	int ret = 0, force = 0;
	int invoke = 0, exunencrypt = 0;
	
	DEBUG(DBG_IOCTL, "SET ENCODE\n");
	
	/* 
	 * With the new API, it's impossible to get a NULL pointer.
	 * New version of iwconfig set the IW_ENCODE_NOKEY flag
	 * when no key is given, but older versions don't. 
	 */
	if (dwrq->length > 0) {
		/* we have a key to set */
		int index = (dwrq->flags & IW_ENCODE_INDEX) - 1;
		int current_index;
		struct obj_key key = {DOT11_PRIV_WEP, 0, ""};

		/* get the current key index */
		ret = sm_drv_oid_get(dev, DOT11_OID_DEFKEYID,
				     (void *)&current_index, sizeof(uint32_t));
		if (ret < 0)
			return ret;
		
		/* Verify that the key is not marked as invalid */
		if (!(dwrq->flags & IW_ENCODE_NOKEY)) {
			key.length = dwrq->length > sizeof (key.key) ?
			    sizeof (key.key) : dwrq->length;
			memcpy(key.key, extra, key.length);
			if ((index < 0) || (index > 3))
				/* no index provided use the current one */
				index = current_index;

			/* now send the key to the card  */
			ret = sm_drv_oid_set(dev,
					     DOT11_OID_DEFKEYID + index + 1,
					     (void *)&key,
					     sizeof(struct obj_key));
			if (ret < 0)
				return ret;
		}
		/*
		 * If a valid key is set, encryption should be enabled 
		 * (user may turn it off later).
		 * This is also how "iwconfig ethX key on" works
		 */
		if ((index == current_index) && (key.length > 0))
			force = 1;
	} else {
		int index = (dwrq->flags & IW_ENCODE_INDEX) - 1;
		struct obj_key key = {DOT11_PRIV_WEP, 0, ""};
		
		if ((index >= 0) && (index <= 3)) {
			/* we want to set the key index */
			ret = sm_drv_oid_set(dev, DOT11_OID_DEFKEYID,
					     (void *)&index, sizeof(uint32_t));
			if (ret < 0)
				return ret;
			ret = sm_drv_oid_get(dev,
					     DOT11_OID_DEFKEYID + index + 1,
					     (void *)&key,
					     sizeof(struct obj_key)); 
			if (ret < 0)
				return ret;
			/* We have a key there, let's turn encryption on */
			if (key.length > 0)
				force = 1;
		} else {
			if (!dwrq->flags & IW_ENCODE_MODE) {
				/* we cannot do anything. Complain. */
				return -EINVAL;
			}
		}
	}

	/* now read the flags     */
	if (dwrq->flags & IW_ENCODE_DISABLED) {
		/* Encoding disabled, 
		 * authen = DOT11_AUTH_OS;
		 * invoke = 0;
		 * exunencrypt = 0; */
	}
	if (dwrq->flags & IW_ENCODE_OPEN)
		/* Encode but accept non-encoded packets. No auth */
		invoke = 1;
	if ((dwrq->flags & IW_ENCODE_RESTRICTED) || force) {
		/* Refuse non-encoded packets. Auth */
		invoke = 1;
		exunencrypt = 1;
	}
	/* do the change if requested  */
	if ((dwrq->flags & IW_ENCODE_MODE) || force) {
		ret = sm_drv_oid_set(dev, DOT11_OID_PRIVACYINVOKED,
				     (void *)&invoke, sizeof(uint32_t));
		ret |= sm_drv_oid_set(dev, DOT11_OID_EXUNENCRYPTED,
				      (void *)&exunencrypt, sizeof(uint32_t));
	}
	return ret;
}

static int sm_drv_get_encode(struct net_device *dev,
			     struct iw_request_info *info,
			     struct iw_point *dwrq, char *extra)
{
	struct obj_key key;
	u32 devindex, index = (dwrq->flags & IW_ENCODE_INDEX) - 1;
	uint32_t authen = 0, invoke = 0, exunencrypt = 0;
	int ret;

	DEBUG(DBG_IOCTL, "GET ENCODE\n");
	
	/* first get the flags */
	ret = sm_drv_oid_get(dev, DOT11_OID_AUTHENABLE,
			     (void *)&authen, sizeof(uint32_t));
	ret |= sm_drv_oid_get(dev, DOT11_OID_PRIVACYINVOKED,
			      (void *)&invoke, sizeof(uint32_t));
	ret |= sm_drv_oid_get(dev, DOT11_OID_EXUNENCRYPTED,
			      (void *)&exunencrypt, sizeof(uint32_t));
	
	if (ret < 0)
		return ret;

	if (invoke 
	    && ((authen == DOT11_AUTH_OS) || (authen == DOT11_AUTH_BOTH)) 
	    && exunencrypt) {
		switch (invoke) {
		case DOT11_PRIV_INV_WEP:
			dwrq->flags = IW_ENCODE_RESTRICTED;
			break;
		case DOT11_PRIV_INV_TKIP:
			dwrq->flags |= IW_ENCODE_TKIP;
			break;
		case DOT11_PRIV_INV_AES_CCMP:
			dwrq->flags |= IW_ENCODE_AES;
			break;			
		}
	}
	else if (((authen == DOT11_AUTH_OS) || (authen == DOT11_AUTH_BOTH)) 
		 && !exunencrypt) {
		/* If we set it to DISABLED, we won't see the keys */
		dwrq->flags = IW_ENCODE_OPEN;
	} else
		/* The card should not work in this state */
		dwrq->flags = 0;

	/* get the current device key index */
	ret = sm_drv_oid_get(dev, DOT11_OID_DEFKEYID,
			     (void *)&devindex, sizeof(uint32_t)); 
	if (ret < 0)
		return ret;

	if ((index < 0) || (index > 3))
		/* no index provided, use the current one */
		index = devindex;
	
	ret = sm_drv_oid_get(dev, DOT11_OID_DEFKEYID + index + 1,
			     (void *)&key, sizeof(struct obj_key)); 
	if (ret < 0)
		return ret;
	dwrq->length = key.length;
	memcpy(extra, key.key, dwrq->length);

	/* return the used key index */
	dwrq->flags |= devindex + 1;

	return ret;
}

static int sm_drv_set_ps(struct net_device *dev, struct iw_request_info *info,
			  struct iw_param *prq, char * extra)
{
	uint32_t psm, cam_timeout;
	int ret;

	DEBUG(DBG_IOCTL, "SET PSM\n");
	
	if (prq->disabled) {
		psm = DOT11_PSM_ACTIVE;
	} else {
		if (((prq->flags & IW_POWER_MODE) == IW_POWER_ALL_R)
		    || prq->value == 0)
			psm = DOT11_PSM_POWERSAVE;
		else if (prq->value > MAX_CAM_TIMEOUT)
			psm = DOT11_PSM_ACTIVE;
		else {
			psm = DOT11_PSM_DYNAMIC;
			cam_timeout = prq->value;
		}	
	}	

	if (psm == DOT11_PSM_DYNAMIC) {
		ret = sm_drv_oid_set(dev, DOT11_OID_CAMTIMEOUT, 
				     (void *)&cam_timeout, 
				     sizeof(uint32_t));
		if (ret < 0) {
			cx3110x_warning("failed to set CAM timeout.");
			goto out;
		}
	}
	
	ret = sm_drv_oid_set(dev, DOT11_OID_PSM, (void *)&psm,
			     sizeof(uint32_t));

	if (ret < 0) {
		cx3110x_warning("failed to set PSM mode.");
		goto out;
	}

	switch (psm) {
	case DOT11_PSM_ACTIVE:
		cx3110x_info("PSM disabled.");
		break;
	case DOT11_PSM_POWERSAVE:
		cx3110x_info("PSM full.");
		break;
	case DOT11_PSM_DYNAMIC:
		cx3110x_info("PSM dynamic with %i ms CAM timeout.",
			     cam_timeout);
		break;
	}

out:
	return ret;
}

static int sm_drv_get_ps(struct net_device *dev, struct iw_request_info *info,
			 struct iw_param *prq, char * extra)
{
	int ret = 0;
	uint32_t psm, cam_timeout;

	DEBUG(DBG_IOCTL, "GET PSM\n");

	ret = sm_drv_oid_get(dev, DOT11_OID_PSM, (void *)&psm,
			     sizeof(uint32_t));
	if (ret < 0)
		return ret;

	ret = sm_drv_oid_get(dev, DOT11_OID_CAMTIMEOUT,
			     (void *)&cam_timeout, sizeof(uint32_t));
	if (ret < 0)
		return ret;

	if (psm == DOT11_PSM_ACTIVE) 
		prq->disabled = 1;
	else
		prq->disabled = 0;

	/* Note : by default, display the period */
	if ((prq->flags & IW_POWER_TYPE) == IW_POWER_TIMEOUT) {
		prq->flags = IW_POWER_TIMEOUT;
		prq->value = cam_timeout;
	} 

	return ret;
}

#define WLAN_FC_TYPE_MGMT 0
#define WLAN_FC_STYPE_ASSOC_REQ 0
#define WLAN_FC_STYPE_REASSOC_REQ 2
static int sm_drv_set_genie(struct net_device *dev,
			    struct iw_request_info *info,
			    union iwreq_data *wrqu, char *extra)
{
	int alen, ret=0, i;
	struct iw_point *dwrq;
	struct obj_attachment *attach;
	
	dwrq = &wrqu->data;

	if (dwrq == NULL)
		return -EINVAL;
	
	if (dwrq->length) {		
		DEBUG(DBG_WPA, "IE:\n");
		for (i = 0; i < dwrq->length; i++) {
			DEBUG(DBG_WPA, "0x%02x ",
			      *((char*)(dwrq->pointer) + i));
		}
		DEBUG(DBG_WPA, "\n");

		alen = sizeof(struct obj_attachment) + dwrq->length;
		attach = (struct obj_attachment *)kzalloc(alen, GFP_ATOMIC);
		if (attach == NULL) 
			return -ENOMEM;
		
		memcpy(attach->data, dwrq->pointer, dwrq->length);
				
	} else {
		/*
		 * If length is set to 0, we remove the generic element
		 * from the frame
		 */
		alen = sizeof(struct obj_attachment);
		attach = (struct obj_attachment *)kmalloc(alen, GFP_ATOMIC);
	}
	
	/*
	 * We attach the IE for association and re-association. This is
	 * neede for roaming
	 */
	attach->type = (WLAN_FC_TYPE_MGMT << 2) |
		(WLAN_FC_STYPE_ASSOC_REQ << 4);
	attach->id = -1;	
	attach->size = dwrq->length;
	
	ret = sm_drv_oid_set(dev, DOT11_OID_ATTACHMENT, (void*)attach, alen);
	if (ret < 0)
		return ret;

	attach->type = (WLAN_FC_TYPE_MGMT << 2) |
		(WLAN_FC_STYPE_REASSOC_REQ << 4);

	ret = sm_drv_oid_set(dev, DOT11_OID_ATTACHMENT, (void*)attach, alen);

	kfree(attach);

	return ret;
}

static int sm_drv_set_pmk(struct net_device *dev, struct iw_request_info *info,
			  struct iw_point *dwrq, char *extra)
{
	struct iw_pmksa * pmk;
	struct obj_stakey key;
	
	if (dwrq == NULL)
		return -EINVAL;
	
	if (!dwrq->length)
		return -EINVAL;

	pmk = (struct iw_pmksa *)dwrq->pointer;
	
	memset(&key, 0, sizeof(struct obj_stakey));

	switch (pmk->cmd) {
	case IW_PMKSA_ADD:
		key.type = DOT11_PRIV_PMKID;

		memcpy(key.address, pmk->bssid.sa_data, ETH_ALEN);
		
		key.length = IW_PMKID_LEN;
		memcpy(key.key, pmk->pmkid, IW_PMKID_LEN);		
		break;
	case IW_PMKSA_REMOVE:
		key.type = DOT11_PRIV_PMKID; 

		memcpy(key.address, pmk->bssid.sa_data, ETH_ALEN);
		
		break;
	case IW_PMKSA_FLUSH:
		break;
	default:
		DEBUG(DBG_WPA, "Invalid PMK command: %d\n", pmk->cmd);
		return -EINVAL;
	}
	
	return sm_drv_oid_set(dev, DOT11_OID_STAKEY,
			      (void *)&key, sizeof(struct obj_stakey));
}


/* Private handlers */

/* First private ioctl after the WPA ones */
#define SIOCIFWFIRST_SMPRIV                       SIOCIWFIRSTPRIV + 12

/* Preamble settings */
#define SM_DRV_SET_PREAMBLE                       SIOCIFWFIRST_SMPRIV
#define SM_DRV_GET_PREAMBLE                       SIOCIFWFIRST_SMPRIV + 1

static int sm_drv_set_preamble(struct net_device *dev,
			       struct iw_request_info *info,
			       __u32 * uwrq, char *extra)
{
	DEBUG(DBG_IOCTL, "SET PREAMBLE\n");
	
	if (uwrq == NULL)
		return -EINVAL;

	return sm_drv_oid_set(dev, DOT11_OID_PREAMBLESETTINGS,
			      uwrq, sizeof(uint32_t));
}

static int sm_drv_get_preamble(struct net_device *dev,
			       struct iw_request_info *info,
			       __u32 * uwrq, char *extra)
{
	uint32_t preamble, ret;
	
	DEBUG(DBG_IOCTL, "GET PREAMBLE\n");

	/* get the rts threshold */
	ret = sm_drv_oid_get(dev, DOT11_OID_PREAMBLESETTINGS,
			     (void *)&preamble, sizeof(uint32_t));
	if (ret < 0)
		return ret;
	memcpy(extra, &preamble, sizeof(preamble));
	
	return ret;
}

#define SM_DRV_SET_HIBERNATE                    SIOCIFWFIRST_SMPRIV + 2

static int sm_drv_set_hibernate(struct net_device *dev,
				struct iw_request_info *info,
				__u32 * uwrq, char *extra)
{
	DEBUG(DBG_IOCTL, "SET HIBERNATE\n");
	
	if (uwrq == NULL)
		return -EINVAL;

	return sm_drv_oid_set(dev, GEN_OID_HIBERNATE, uwrq, sizeof(uint32_t));
}

#define SM_DRV_SET_BGRSCAN                      SIOCIFWFIRST_SMPRIV + 4

static int sm_drv_set_bgrscan(struct net_device *dev,
			      struct iw_request_info *info,
			      __u32 * uwrq, char *extra)
{
	DEBUG(DBG_IOCTL, "SET BACKGROUND SCANNING\n");
	return sm_drv_oid_set(dev, DOT11_OID_AUTOSCANDISABLE,
			      uwrq, sizeof(uint32_t));
}

#define SM_DRV_SET_BEACONPERIOD                  SIOCIFWFIRST_SMPRIV + 6
#define SM_DRV_GET_BEACONPERIOD                  SIOCIFWFIRST_SMPRIV + 7

static int sm_drv_set_beaconperiod(struct net_device *dev,
				   struct iw_request_info *info,
				   __u32 * uwrq, char *extra)
{
	DEBUG(DBG_IOCTL, "SET BEACON PERIOD\n");

	if (uwrq == NULL)
		return -EINVAL;

	return sm_drv_oid_set(dev, DOT11_OID_BEACONPERIOD,
			      uwrq, sizeof(uint32_t));
}

static int sm_drv_get_beaconperiod(struct net_device *dev,
				   struct iw_request_info *info,
				   __u32 * uwrq, char *extra)
{
	uint32_t period, ret;
	
	DEBUG(DBG_IOCTL, "GET BEACON PERIOD\n");

	ret = sm_drv_oid_get(dev, DOT11_OID_BEACONPERIOD,
			     (void *)&period, sizeof(uint32_t));
	if (ret < 0)
		return ret;
	
	memcpy(extra, &period, sizeof(period));
	
	return ret;
}

#define SM_DRV_SET_DMA_THRESHOLD                  SIOCIFWFIRST_SMPRIV + 8
#define SM_DRV_GET_DMA_THRESHOLD                  SIOCIFWFIRST_SMPRIV + 9

static int sm_drv_set_dma_threshold(struct net_device *dev,
				    struct iw_request_info *info,
				    __u32 * uwrq, char *extra)
{
	struct net_local *lp = netdev_priv(dev);
	
	DEBUG(DBG_IOCTL, "SET DMA THRESHOLD\n");

	if (uwrq == NULL)
		return -EINVAL;

	spin_lock_bh(&lp->lock);
	lp->dma_threshold = *uwrq;
	spin_unlock_bh(&lp->lock);

	return 0;
}

static int sm_drv_get_dma_threshold(struct net_device *dev,
				    struct iw_request_info *info,
				    __u32 * uwrq, char *extra)
{
	struct net_local *lp = netdev_priv(dev);
	
	DEBUG(DBG_IOCTL, "GET DMA THRESHOLD\n");

	if (uwrq == NULL)
		return -EINVAL;

	spin_lock_bh(&lp->lock);
	*uwrq = lp->dma_threshold;
	spin_unlock_bh(&lp->lock);

	return 0;
}

static int sm_drv_set_scaninterval(struct net_device *dev,
				   struct iw_request_info *info,
				   __u32 * uwrq, char *extra)
{
	struct obj_scan params = {
		.sweep = -1,
		.min = 80,
		.max = 200,
		.interval = 20000,
		.amin = 80,
		.amax = 140,
	};
	int ret, interval;
	
	DEBUG(DBG_IOCTL, "SET BGSCAN INTERVAL\n");

	if (uwrq == NULL)
		return -EINVAL;

	interval = *uwrq;

	if (interval < 0 || interval > 65535)
		return -EINVAL;

	params.interval = interval;
	
	ret = sm_drv_oid_set(dev, DOT11_OID_SCAN, &params, sizeof(params));

	return ret;
}

static int sm_drv_get_scaninterval(struct net_device *dev,
				   struct iw_request_info *info,
				   __u32 * uwrq, char *extra)
{
	struct obj_scan params;
	int ret;
	
	DEBUG(DBG_IOCTL, "GET BGSCAN INTERVAL\n");

	ret = sm_drv_oid_get(dev, DOT11_OID_SCAN, &params, sizeof(params));

	if (ret < 0)
		return ret;

	*uwrq = params.interval;

	return 0;
}

static int sm_drv_set_scanthreshold(struct net_device *dev,
				    struct iw_request_info *info,
				    __u32 * uwrq, char *extra)
{
	int ret, threshold;
	
	DEBUG(DBG_IOCTL, "SET BGSCAN INTERVAL\n");

	if (uwrq == NULL)
		return -EINVAL;

	threshold = *uwrq;

	if (threshold < -100 || threshold > 0)
		return -EINVAL;

	ret = sm_drv_oid_set(dev, DOT11_OID_SCANTHRESHOLD, &threshold,
			     sizeof(threshold));

	return ret;
}

static int sm_drv_get_scanthreshold(struct net_device *dev,
				    struct iw_request_info *info,
				    __u32 * uwrq, char *extra)
{
	int ret, threshold;
	
	DEBUG(DBG_IOCTL, "GET BGSCAN INTERVAL\n");

	ret = sm_drv_oid_get(dev, DOT11_OID_SCANTHRESHOLD,
			     &threshold, sizeof(threshold));

	if (ret < 0)
		return ret;

	*uwrq = threshold;

	return 0;
}


/* WPA related handlers */
int sm_drv_process_bss_data(struct net_device* dev,  u8 *addr, 
			     u8 *payload, size_t len, uint32_t event)
{
	struct ieee80211_beacon_phdr *hdr;
	u8 *pos, *end, *wpa1_ie_pos = NULL;
	int i;

	DEBUG(DBG_WPA, "Dumping frame for " MACSTR "\n", MAC2STR(addr));
	for (i = 0; i < len; i++)
		DEBUG(DBG_WPA, "0x%02x ",payload[i]);
	DEBUG(DBG_WPA, "\n");

	hdr = (struct ieee80211_beacon_phdr *) payload;
	pos = (u8 *) (hdr + 1);
	end = payload + len;
	while (pos < end) {
		/*
		 * If we find an RSN IE, we're done, we can send the IE
		 * event
		 */
		if (pos[0] == WPA2_EID_GENERIC && pos[1] >= 2 &&
		    memcmp(pos + 2, wpa2_oid, 2) == 0) {
			DEBUG(DBG_WPA, "Found WPA2 OID in beacon frame for "
			      MACSTR "\n\n", MAC2STR(addr));
			sm_drv_wpa_ie_add(dev, addr, pos, pos[1] + 2, event);
			return 0;
		}
	
		/* 
		 * If we find a WPA IE, we should continue to check if
		 * there's no RSN IE. We just keep track of the WPA1 IE
		 * position.
		 */
		if (pos[0] == WPA_EID_GENERIC && pos[1] >= 4 &&
		    memcmp(pos + 2, wpa_oid, 4) == 0) {
			DEBUG(DBG_WPA, "Found WPA OID in beacon frame for "
			      MACSTR "\n\n", MAC2STR(addr));
			wpa1_ie_pos = pos;
		}
		pos++;
	}
	
	if (wpa1_ie_pos != NULL) {
		sm_drv_wpa_ie_add(dev, addr, wpa1_ie_pos,
				  wpa1_ie_pos[1] + 2, event);
		return 0;			
	}
	
	/* 
	 * We didn't find any WPA OID in the beacon,
	 * let's remove the AP from the WPA list, if it's
	 * there.
	 */
	sm_drv_wpa_ie_remove(dev, addr);
	return -1;
}

static int sm_drv_set_encryption(struct net_device *dev,
				 struct iw_request_info *info,
				 struct iw_point *dwrq, char *extra)
{
	int ret = 0, force = 0, i;
	int invoke = 0, exunencrypt = 0;
	struct wlan_crypt * crypt_info;
	struct net_local *lp = dev->priv;
	const char multicast_address[ETH_ALEN] =
		{0xff, 0xff, 0xff, 0xff, 0xff, 0xff};
	
	DEBUG(DBG_WPA, "Calling set_encryption\n");

	if (dwrq == NULL)
		return -EINVAL;
	
	if (!dwrq->length)
		return -EINVAL;

	crypt_info = (struct wlan_crypt *)dwrq->pointer;
		
	DEBUG(DBG_WPA, "Key dumping:\n");
	DEBUG(DBG_WPA, "  alg:   %s\n", crypt_info->alg);
	DEBUG(DBG_WPA, "  flags: 0x%x\n", crypt_info->flags);
	DEBUG(DBG_WPA, "  index: %d\n", crypt_info->idx);
	DEBUG(DBG_WPA, "  key length: %d\n", crypt_info->key_len);
	DEBUG(DBG_WPA, "  key:\n");
	for (i = 0; i < crypt_info->key_len; i++)
		DEBUG(DBG_WPA, "  0x%x\n", *((uint8_t *)crypt_info->key + i));

	if (crypt_info->key_len > 0) {
		int index = crypt_info->idx;
		int current_index;
		struct obj_stakey key;

		/* get the current key index */
		ret = sm_drv_oid_get(dev, DOT11_OID_DEFKEYID,
				     (void *)&current_index, sizeof(uint32_t));
		if (ret < 0)
			return ret;

		/* Verify that the key is not marked as invalid */
		if (!(crypt_info->flags & IW_ENCODE_NOKEY)) {
			key.length = dwrq->length > sizeof (key.key) ?
			    sizeof (key.key) : dwrq->length;
			
			if (crypt_info->flags & IW_ENCODE_TKIP) {
				DEBUG(DBG_WPA, "TKIP encryption\n");
				/* 
				 * UMAC expects the key in AP mode: RX mic
				 * at the end, and TX at byte 16. This is
				 * 802.11i AP compliant, but not STA.
				 * Conexant doesn't seem to care.
				 */
				memcpy(key.key, crypt_info->key,
				       key.length -16);
				memcpy(key.key+16, crypt_info->key+24, 8);
				memcpy(key.key+24, crypt_info->key+16, 8);
				key.type = DOT11_PRIV_TKIP;
				invoke =  DOT11_PRIV_INV_TKIP;
			}
			else if (crypt_info->flags & IW_ENCODE_AES) {
				DEBUG(DBG_WPA, "AES encryption\n");
				memcpy(key.key, crypt_info->key, key.length);
				key.type = DOT11_PRIV_AES_CCMP;
				invoke =  DOT11_PRIV_INV_AES_CCMP;
			}
			if ((index < 0) || (index > 3)) {
				/* no index provided use the current one */
				index = current_index;
			}

			/* 
			 * We can now send the key to the UMAC.
			 * Flags tell us which key it is.
			 */
			if (crypt_info->flags & IW_ENCODE_UNICAST) {
				spin_lock_bh(&lp->lock);
				DEBUG(DBG_WPA, "Setting key with Mac: "
				      MACSTR "\n",
				      MAC2STR(lp->ap_mac_address));
				memcpy(key.address, lp->ap_mac_address,
				       ETH_ALEN);
				spin_unlock_bh(&lp->lock);

			}
			else if (crypt_info->flags & IW_ENCODE_MULTICAST)
				memcpy(key.address, multicast_address,
				       ETH_ALEN);
			key.keyid = index;
			key.ext = 0;
			
			ret = sm_drv_oid_set(dev, DOT11_OID_STAKEY,
					     (void *)&key,
					     sizeof(struct obj_stakey));

			if (ret < 0)
				return ret;
		}
		/*
		 * If a valid key is set, encryption should be enabled 
		 * (user may turn it off later).
		 * This is also how "iwconfig ethX key on" works
		 */
		if ((index == current_index) && (key.length > 0))
			force = 1;
	} else {
		int index = (crypt_info->flags & IW_ENCODE_INDEX) - 1;
		if ((index >= 0) && (index <= 3)) {
			/* we want to set the key index */
			ret = sm_drv_oid_set(dev, DOT11_OID_DEFKEYID,
					     (void *)&index, sizeof(uint32_t));
			if (ret < 0)
				return ret;
		} else {
			if (!crypt_info->flags & IW_ENCODE_MODE) {
				/* we cannot do anything. Complain. */
				return -EINVAL;
			}
		}
	}
	
	/* now read the flags */
	if (crypt_info->flags & IW_ENCODE_OPEN) {
		/* Encode but accept non-encoded packets. No auth */
		DEBUG(DBG_WPA, "Encode but accept non-encoded packets\n");
	}
	if ((crypt_info->flags & IW_ENCODE_RESTRICTED) || force) {
		/* Refuse non-encoded packets. Auth */
		DEBUG(DBG_WPA, "Refuse non-encoded packets\n");
		exunencrypt = 1;
	}
	/* do the change if requested  */
	if ((crypt_info->flags & IW_ENCODE_MODE) || force) {
		uint32_t dot1x_enable = 1;
		DEBUG(DBG_WPA, "PRIVACY INVOKED\n");
		ret = sm_drv_oid_set(dev, DOT11_OID_PRIVACYINVOKED,
				     (void *)&invoke, sizeof(uint32_t));
		DEBUG(DBG_WPA, "EXUNENCRYPTED\n");
		ret |= sm_drv_oid_set(dev, DOT11_OID_EXUNENCRYPTED,
				      (void *)&exunencrypt, sizeof(uint32_t));
		DEBUG(DBG_WPA, "Enabling 802.1x\n");
		ret |= sm_drv_oid_set(dev, DOT11_OID_DOT1XENABLE,
				      (void *)&dot1x_enable, sizeof(uint32_t));
	}

	return ret;
}

int sm_drv_set_wpa(struct net_device *dev, struct iw_request_info *info,
		   __u32 * uwrq, char *extra)
{
	struct net_local *lp = dev->priv;
	int ret = 0, wpa;
	uint32_t filter, dot1x_enable = 1;
	
	if (uwrq == NULL)
		return -EINVAL;

	DEBUG(DBG_WPA, "Calling set_wpa\n");
	
	filter = 1; /* Filter out all unencrypted frames */
	dot1x_enable = 1;

	spin_lock_bh(&lp->lock);
	lp->wpa = wpa = *uwrq;
	spin_unlock_bh(&lp->lock);
	
	switch (wpa) {
	case DOT11_PRIV_INV_NONE: /* Clears/disables WPA and friends */
		filter = 0; /* Do not filter un-encrypted data */
		break;
	case DOT11_PRIV_INV_WEP:
		dot1x_enable = 0;
		break;
	case DOT11_PRIV_INV_TKIP:
		break;
	case DOT11_PRIV_INV_AES_CCMP:
		break;
	default:
		return -EINVAL;
	}

	ret = sm_drv_oid_set(dev, DOT11_OID_PRIVACYINVOKED, &wpa, sizeof(wpa));
	ret |= sm_drv_oid_set(dev, DOT11_OID_EXUNENCRYPTED,
			      (void *)&filter, sizeof(uint32_t));
	ret |= sm_drv_oid_set(dev, DOT11_OID_DOT1XENABLE,
			      (void *)&dot1x_enable, sizeof(uint32_t));

	return ret;
}


int sm_drv_get_wpa(struct net_device *dev, struct iw_request_info *info,
		__u32 * uwrq, char *extra)
{
	struct net_local *lp = dev->priv;

	spin_lock_bh(&lp->lock);
	*uwrq = lp->wpa;
	spin_unlock_bh(&lp->lock);

	return 0;
}

const iw_handler sm_drv_we_handler[] = {
	(iw_handler) sm_drv_commit,	        /* SIOCSIWCOMMIT */
	(iw_handler) sm_drv_get_name,	        /* SIOCGIWNAME */
	(iw_handler) NULL,	                /* SIOCSIWNWID */
	(iw_handler) NULL,	                /* SIOCGIWNWID */
	(iw_handler) sm_drv_set_freq,	        /* SIOCSIWFREQ */
	(iw_handler) sm_drv_get_freq,	        /* SIOCGIWFREQ */
	(iw_handler) sm_drv_set_mode,	        /* SIOCSIWMODE */
	(iw_handler) sm_drv_get_mode,	        /* SIOCGIWMODE */
	(iw_handler) sm_drv_set_sens,	        /* SIOCSIWSENS */
	(iw_handler) sm_drv_get_sens,	        /* SIOCGIWSENS */
	(iw_handler) NULL,	                /* SIOCSIWRANGE */
	(iw_handler) sm_drv_get_range,	        /* SIOCGIWRANGE */
	(iw_handler) NULL,	                /* SIOCSIWPRIV */
	(iw_handler) NULL,	                /* SIOCGIWPRIV */
	(iw_handler) NULL,	                /* SIOCSIWSTATS */
	(iw_handler) sm_drv_get_wireless_stats,	/* SIOCGIWSTATS */
	(iw_handler) NULL,	                /* SIOCSIWSPY */
	iw_handler_get_spy,	                /* SIOCGIWSPY */
	iw_handler_set_thrspy,	                /* SIOCSIWTHRSPY */
	iw_handler_get_thrspy,	                /* SIOCGIWTHRSPY */
	(iw_handler) sm_drv_set_wap,	        /* SIOCSIWAP */
	(iw_handler) sm_drv_get_wap,	        /* SIOCGIWAP */
	(iw_handler) NULL,	                /* -- hole -- */
	(iw_handler) NULL,	                /* SIOCGIWAPLIST deprecated */
	(iw_handler) sm_drv_set_scan,	        /* SIOCSIWSCAN */
	(iw_handler) sm_drv_get_scan,	        /* SIOCGIWSCAN */
	(iw_handler) sm_drv_set_essid,	        /* SIOCSIWESSID */
	(iw_handler) sm_drv_get_essid,	        /* SIOCGIWESSID */
	(iw_handler) NULL,	                /* SIOCSIWNICKN */
	(iw_handler) NULL,	                /* SIOCGIWNICKN */
	(iw_handler) NULL,	                /* -- hole -- */
	(iw_handler) NULL,	                /* -- hole -- */
	(iw_handler) sm_drv_set_rate,	        /* SIOCSIWRATE */
	(iw_handler) sm_drv_get_rate,	        /* SIOCGIWRATE */
	(iw_handler) sm_drv_set_rts,	        /* SIOCSIWRTS */
	(iw_handler) sm_drv_get_rts,	        /* SIOCGIWRTS */
	(iw_handler) sm_drv_set_frag,	        /* SIOCSIWFRAG */
	(iw_handler) sm_drv_get_frag,	        /* SIOCGIWFRAG */
	(iw_handler) sm_drv_set_txpower,	/* SIOCSIWTXPOW */
	(iw_handler) sm_drv_get_txpower,	/* SIOCGIWTXPOW */
	(iw_handler) NULL,	                /* SIOCSIWRETRY */
	(iw_handler) NULL,	                /* SIOCGIWRETRY */
	(iw_handler) sm_drv_set_encode,	        /* SIOCSIWENCODE */
	(iw_handler) sm_drv_get_encode,	        /* SIOCGIWENCODE */
	(iw_handler) sm_drv_set_ps,	        /* SIOCSIWPOWER */
	(iw_handler) sm_drv_get_ps,	        /* SIOCGIWPOWER */
	(iw_handler) NULL,			/* -- hole -- */
	(iw_handler) NULL,			/* -- hole -- */
	(iw_handler) sm_drv_set_genie,          /* SIOCSIWGENIE*/
	(iw_handler) NULL,	                /* SIOCGIWGENIE */
	(iw_handler) NULL,	                /* SIOCSIWAUTH */
	(iw_handler) NULL,	                /* SIOCGIWAUTH */
	(iw_handler) NULL,	                /* SIOCSIWENCODEEXT */
	(iw_handler) NULL,	                /* SIOCGIWENCODEEXT */
	(iw_handler) sm_drv_set_pmk,		/* SIOCSIWPMKSA */
};

#define SM_DRV_SET_DEBUG                       SIOCIWFIRSTPRIV
const struct iw_priv_args sm_drv_we_private_args[] = {
	{SM_DRV_SET_SCANINTERVAL, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
	 0, "set_scanint"},
	{SM_DRV_GET_SCANINTERVAL, 0, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
	 "get_scanint"},
	{SM_DRV_SET_SCANTHRESHOLD, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
	 0, "set_scanthres"},
	{SM_DRV_GET_SCANTHRESHOLD, 0, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
	 "get_scanthres"},
	{SM_DRV_SET_DEBUG, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
	 0, "set_debug"},
	{SM_DRV_SET_PREAMBLE, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
	 0, "set_preamble"},
	{SM_DRV_GET_PREAMBLE, 0, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
	 "get_preamble"},
	{SM_DRV_SET_HIBERNATE, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0,
	 "set_hibernate"},
	{SM_DRV_SET_BGRSCAN, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0,
	 "set_backscan"},
	{SM_DRV_SET_BEACONPERIOD, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
	 0, "set_bperiod"},
	{SM_DRV_GET_BEACONPERIOD, 0,
	 IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, "get_bperiod"},
	{SM_DRV_SET_DMA_THRESHOLD, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
	 0, "set_dmathres"},
	{SM_DRV_GET_DMA_THRESHOLD, 0,
	 IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, "get_dmathres"},
};

static const iw_handler sm_drv_we_private_handler[] = {
	(iw_handler) sm_drv_set_debug,
	(iw_handler) NULL,

	(iw_handler) sm_drv_set_scaninterval,
	(iw_handler) sm_drv_get_scaninterval,

	(iw_handler) sm_drv_set_scanthreshold,
	(iw_handler) sm_drv_get_scanthreshold,

	(iw_handler) sm_drv_set_wpa,
	(iw_handler) sm_drv_get_wpa,

	(iw_handler) sm_drv_set_encryption,
	(iw_handler) sm_drv_set_encryption,

	(iw_handler) sm_drv_set_pmk,
	(iw_handler) NULL,

	(iw_handler) sm_drv_set_preamble,
	(iw_handler) sm_drv_get_preamble,

	(iw_handler) sm_drv_set_hibernate,
	(iw_handler) NULL,

	(iw_handler) sm_drv_set_bgrscan,
	(iw_handler) NULL,

	(iw_handler) sm_drv_set_beaconperiod,
	(iw_handler) sm_drv_get_beaconperiod,

	(iw_handler) sm_drv_set_dma_threshold,
	(iw_handler) sm_drv_get_dma_threshold,
};

const struct iw_handler_def sm_drv_we_handler_def = {
	.num_standard = sizeof(sm_drv_we_handler) / sizeof(iw_handler),
	.num_private = sizeof(sm_drv_we_private_handler) / sizeof(iw_handler),
	.num_private_args =
	  sizeof(sm_drv_we_private_args) / sizeof(struct iw_priv_args),
	.standard = (iw_handler *) sm_drv_we_handler,
	.private = (iw_handler *) sm_drv_we_private_handler,
	.private_args = (struct iw_priv_args *) sm_drv_we_private_args,
	.get_wireless_stats = sm_drv_get_wireless_stats,
};

