/*
 * This file is part of cx3110x
 *
 * Copyright (C) 2003 Conexant Americas Inc. All Rights Reserved.
 * Copyright (C) 2004, 2005, 2006 Nokia Corporation
 *
 * Contact: Kalle Valo <Kalle.Valo@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * version 2 as published by the Free Software Foundation. 
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#define __KERNEL_SYSCALLS__

#include <linux/version.h>
#include <linux/module.h>

#include <linux/init.h>
#include <linux/types.h>
#include <linux/sched.h>
#include <linux/unistd.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/if_arp.h>
#include <linux/string.h>
#include <linux/poll.h>
#include <linux/platform_device.h>
#include <asm/uaccess.h>

#include "sm_drv.h"
#include "sm_drv_ioctl.h"
#include "sm_drv_sysfs.h"

extern char driver_name[64];

/* for linux/unistd.h */
int errno;

DECLARE_COMPLETION(softmac_init_comp);

extern struct net_device *root_sm_drv_device;
extern unsigned int driver_type;

/* Timers */
void driver_timer_expired(unsigned long handle)
{
	struct net_device *dev = (struct net_device *) handle;
	struct net_local *lp = dev->priv;
	int32_t callb_mask;

	DEBUG(DBG_TIMER, "timer expired at %lu\n", jiffies);
	
	spin_lock_bh(&lp->lock);	
	callb_mask = prism_softmac_service(lp->sm_context);
	spin_unlock_bh(&lp->lock);
	
	if(callb_mask < 0)
		cx3110x_warning("%s: prism_softmac_service returned %d.",
			      __FUNCTION__, callb_mask);
	else {
		handle_sm_callback(dev, callb_mask);
	}
	

}

void sm_drv_disassociate(struct net_device *dev)
{
	struct obj_ssid essid;
	struct net_local *lp = dev->priv;

	spin_lock_bh(&lp->lock);
	lp->link_state = DOT11_STATE_NONE;
	spin_unlock_bh(&lp->lock);

	/* 
	 * By setting a new essid, UMAC will diassociates from the current
	 * AP. If this is the broadcast SSID, it won't try to reconnect
	 * to another AP.
	 */
	memset(essid.octets, 0, 33);
	essid.length = 0;
	
	sm_drv_oid_set(dev, DOT11_OID_SSID, (void*)&essid,
		       sizeof(struct obj_ssid));
}

/* 
 * UGLY hack to workaround Nokia's AP bug:
 * When we set short slots in the capability field of the association request, 
 * A032 thinks that we can _only_ do short.
 * However, according to 802.11 standard, short slot bit set in the capability
 * bitfield implies that the device can do _both_ long and short. A032 doesn't
 * get that.
 * On the other hand, if we specifically set short slots, we will have trouble
 * with 11g devices, so we have to lamely check the AP MAC and see if this is
 * a Nokia one...Sorry.
 */
static int32_t nokia_a032_hack(struct net_device *dev, struct s_sm_conf *trap)
{
	static const char nokia_wireless_biz_oui[3] = {0x00, 0xe0, 0x03};
	struct obj_mlme *mlme = (struct obj_mlme*) trap->data;
	uint32_t profile = DOT11_PROFILE_B_WIFI;
	int32_t result = SM_ENONE;

	if (memcmp(mlme->address, nokia_wireless_biz_oui, 3) != 0)
		goto out;

	result = sm_drv_oid_set(dev, DOT11_OID_PROFILES,
				(void*)&profile, sizeof(uint32_t));
	
	if (result < 0) {
		cx3110x_warning("failed to set profile for Nokia A032: %i",
				result);
		goto out;
	}

out:
	return result;
}

static void link_associated(struct net_device *dev)
{
	struct net_local *lp = netdev_priv(dev);
	union iwreq_data uwrq;
	struct obj_bss bss;
	uint32_t mode;
	char *bssid;
	int ret;

	memset(&bss, 0, sizeof(bss));
	
	spin_lock_bh(&lp->lock);
	if (lp->sm_mode != SM_MODE_CLIENT) {
		spin_unlock_bh(&lp->lock);
		return;
	}
	spin_unlock_bh(&lp->lock);

	sm_drv_get_mode(dev, NULL, &mode, NULL);

	if (mode != IW_MODE_INFRA)
		return;
	
	sm_drv_get_wap(dev, NULL, (struct sockaddr *) &uwrq, NULL);
	bssid = ((struct sockaddr *)&uwrq)->sa_data;

	/* We can copy the AP MAC address */
	spin_lock_bh(&lp->lock);
	memcpy(lp->ap_mac_address, bssid, ETH_ALEN);
	spin_unlock_bh(&lp->lock);

	memcpy(bss.address, bssid, ETH_ALEN);
	ret = sm_drv_oid_get(dev, DOT11_OID_BSSFIND, &bss, sizeof(bss));
	if (ret < 0) {
		cx3110x_warning("failed to get beacon interval and DTIM.");
	}

	cx3110x_info("associated to " MACSTR " (bcn %i msec, DTIM %i).",
		     MAC2STR(bssid), bss.beacon_period, bss.dtim_period);
                                       
	wireless_send_event(dev, SIOCGIWAP, &uwrq, NULL);
}

static void link_disassociated(struct net_device *dev)
{
	struct net_local *lp = netdev_priv(dev);
	union iwreq_data uwrq;
	uint32_t mode, profile;
			
	/* Association has been lost */
	spin_lock_bh(&lp->lock);
	cx3110x_info("disassociated from " MACSTR ".",
		     MAC2STR(lp->ap_mac_address));
	lp->link_state = DOT11_STATE_NONE;
	spin_unlock_bh(&lp->lock);
			
	sm_drv_get_mode(dev, NULL, &mode, NULL);
			
	if (mode != IW_MODE_INFRA)
		return;
	
	sm_drv_get_wap(dev, NULL, (struct sockaddr *) &uwrq, NULL);
				
	wireless_send_event(dev, SIOCGIWAP, &uwrq, NULL);
				
	/*
	 * We get back to MIXED profile (it could have been changed for
	 * Nokia AP)
	 */
	sm_drv_oid_get(dev, DOT11_OID_PROFILES, (void*)&profile,
		       sizeof(uint32_t));
	if (profile == DOT11_PROFILE_B_WIFI) {
		profile = DOT11_PROFILE_MIXED;
		sm_drv_oid_set(dev, DOT11_OID_PROFILES,
			       (void*)&profile, sizeof(uint32_t));
	}
}

static int32_t trap_macaddress(struct net_device *dev, struct s_sm_conf *trap)
{
	struct net_local *lp = netdev_priv(dev);
	static const uint8_t null_mac[ETH_ALEN] = {0x0,0x0,0x0,0x0,0x0,0x0};
	int32_t result = SM_ENONE;
	
	cx3110x_info("MAC address %2.2x:%2.2x:%2.2x:%2.2x:%2.2x:%2.2x.",
		     trap->data[0], trap->data[1], trap->data[2],
		     trap->data[3], trap->data[4], trap->data[5]);

	if (!memcmp(dev->dev_addr, null_mac, ETH_ALEN))
		memcpy(dev->dev_addr, trap->data, ETH_ALEN);
	else
		/*
		 * If we set the MAC address before booting, let's not
		 * copy the PDA one
		 */
		result = sm_drv_oid_set(dev, GEN_OID_MACADDRESS,
					(void*)dev->dev_addr, ETH_ALEN);
		
	/*
	 * If we got the MAC address, this means the device has finished
	 * booting
	 */
	wake_up(&lp->conf_waitq);

	spin_lock_bh(&lp->lock);
	if (lp->device_state == DEVSTATE_BOOTING)
		lp->device_state = DEVSTATE_ACTIVE;
	spin_unlock_bh(&lp->lock);

	return result;
}

static int32_t trap_associate(struct net_device *dev, struct s_sm_conf *trap)
{
	struct net_local *lp = netdev_priv(dev);
	struct obj_mlme *mlme = (struct obj_mlme*) trap->data;
	int32_t result = SM_ENONE;

	spin_lock_bh(&lp->lock);

	if (lp->sm_mode != SM_MODE_CLIENT){
		spin_unlock_bh(&lp->lock);
		goto out;
	}

	if (mlme->state == lp->link_state) {
		/* already associated */
		spin_unlock_bh(&lp->lock);
		goto out;
	}
	
	DEBUG(DBG_TRACE, "Authenticate/(Re)Associate trap: state %d\n",
	      mlme->state);
			
	lp->link_state = mlme->state;

	spin_unlock_bh(&lp->lock);

	if (mlme->state == DOT11_STATE_ASSOC) {
		spin_lock_bh(&lp->lock);
		lp->ext = mlme->ext;
		spin_unlock_bh(&lp->lock);
		sm_drv_parse_bssid(dev, mlme->address, DOT11_STATE_ASSOC,
				   NULL);
	}
			
	/*
	 * This is the association request. We get the IE from the
	 * beacon.
	 */
	if ((trap->oid == DOT11_OID_REASSOCIATE
	     || trap->oid == DOT11_OID_ASSOCIATE) 
	    && (mlme->state == DOT11_STATE_ASSOCING)) {
		result = nokia_a032_hack(dev, trap);
		if (result < 0)
			goto out;
		sm_drv_process_bss_data(dev, dev->dev_addr, mlme->data,
					mlme->size, trap->oid);
	}

out:
	return result;
}

static int32_t trap_deauthenticate(struct net_device *dev,
				   struct s_sm_conf *trap)
{
	struct net_local *lp = netdev_priv(dev);
	uint32_t scan_mode = SCAN_MODE_ACTIVE, scan = -1;
	int32_t ret = SM_ENONE;

	spin_lock_bh(&lp->lock);
	
	if (lp->link_state != DOT11_STATE_ASSOC) {
		spin_unlock_bh(&lp->lock);
		goto out;
	}
	
	cx3110x_info("deauthenticated from " MACSTR ".",
		     MAC2STR(lp->ap_mac_address));
	
	send_simple_event(dev, "DEAUTHENTICATION");

	/* FIXME: check ssid.length */
	cx3110x_info("active scanning on '%s'.", lp->ssid.octets);

	spin_unlock_bh(&lp->lock);
			
	ret = sm_drv_oid_set(dev, DOT11_OID_SCANMODE, (void*)&scan_mode,
			     sizeof(uint32_t));
	if (ret < 0)
		cx3110x_warning("couldn't set scanmode: %d", ret);
			
	ret = sm_drv_oid_set(dev, DOT11_OID_SCANSSID, (void*)&lp->ssid,
			     sizeof(struct obj_ssid));
	if (ret < 0)
		cx3110x_warning("couldn't set scanssid: %d", ret);
			
	ret = sm_drv_oid_set(dev, DOT11_OID_SCAN, (void*)&scan,
			     sizeof(int16_t));
	if (ret < 0)
		cx3110x_warning("couldn't start scan: %d", ret);

out:
	return ret;
}

static int32_t trap_scan(struct net_device *dev, struct s_sm_conf *trap)
{
	struct net_local *lp = netdev_priv(dev);
	int32_t ret = SM_ENONE;

	cx3110x_info("scanned %d channels.", *((uint16_t*)trap->data));

	spin_lock_bh(&lp->lock);
	/* We got the trap, we stop the scan timer.*/
	del_timer_sync(&lp->scan_timer);
	spin_unlock_bh(&lp->lock);

	send_scan_complete(dev);

	return ret;
}

static int32_t trap_linkstate(struct net_device *dev, struct s_sm_conf *trap)
{
	uint32_t bitrate = *((uint32_t *) trap->data);

	if (driver_type != SM_DRIVER_TYPE_UMAC)
		goto out;

	if (bitrate)
		link_associated(dev);
	else
		link_disassociated(dev);

out:
	return SM_ENONE;
}

static int32_t trap_micfailure(struct net_device *dev, struct s_sm_conf *trap)
{
	struct obj_sta *sta = (struct obj_sta*) trap->data;

	cx3110x_info("MIC failure!");
	send_mic_failure(dev, sta);

	return SM_ENONE;
}

static int32_t handle_sm_trap(struct net_device *dev, struct s_sm_conf *trap)
{
	int32_t ret = SM_ENONE;

	switch(trap->oid) {
	case GEN_OID_MACADDRESS:
		cx3110x_debug(DBG_CALLS, "GEN_OID_MACADDRESS");
		ret = trap_macaddress(dev, trap);
		break;
	case GEN_OID_LINKSTATE:
		cx3110x_debug(DBG_CALLS, "GEN_OID_LINKSTATE");
		trap_linkstate(dev, trap);
		break;
	case DOT11_OID_AUTHENTICATE:
		cx3110x_debug(DBG_CALLS, "DOT11_OID_AUTHENTICATE");
		ret = trap_associate(dev, trap);
		break;
	case DOT11_OID_ASSOCIATE:
		cx3110x_debug(DBG_CALLS, "DOT11_OID_ASSOCIATE");
		ret = trap_associate(dev, trap);
		break;
	case DOT11_OID_REASSOCIATE:
		cx3110x_debug(DBG_CALLS, "DOT11_OID_REASSOCIATE");
		ret = trap_associate(dev, trap);
		break;
	case DOT11_OID_SCAN:
		cx3110x_debug(DBG_CALLS, "DOT11_OID_SCAN");
		ret = trap_scan(dev, trap);
		break;
	case DOT11_OID_MICFAILURE:
		cx3110x_debug(DBG_CALLS, "DOT11_OID_MICFAILURE");
		ret = trap_micfailure(dev, trap);
		break;
	case DOT11_OID_DEAUTHENTICATE:
		cx3110x_debug(DBG_CALLS, "DOT11_OID_DEAUTHENTICATE");
		ret = trap_deauthenticate(dev, trap);
		break;
	case DOT11_OID_DISASSOCIATE:
		cx3110x_debug(DBG_CALLS, "DOT11_OID_DISASSOCIATE");
		cx3110x_info("DISASSOCIATE trap.");
		break;
	case DOT11_OID_PREAUTH:
		cx3110x_debug(DBG_CALLS, "DOT11_OID_PREAUTH");
		/* we don't support WPA preauthentication */
		break;
	default:
		/* do nothing */
		cx3110x_warning("unhandled trap: 0x%x", trap->oid);
		break;
	}
	
	return ret;
 }

static void sm_print_callback_mask(int32_t mask)
{
	DEBUG(DBG_TRACE, "SoftMAC Mask: ");
	if (mask & SM_TRAP)
		DEBUG(DBG_TRACE, "SM_TRAP ");
	if (mask & SM_FRAMETXDONE)
		DEBUG(DBG_TRACE, "SM_FRAMETXDONE ");
	if (mask & SM_FRAMERX)
		DEBUG(DBG_TRACE, "SM_FRAMERX ");
	if (mask & SM_IC)
		DEBUG(DBG_TRACE, "SM_IC ");
	DEBUG(DBG_TRACE, "\n");
}

static inline int sm_monitor_rx(struct net_device *dev, struct sk_buff **skb)
{
	/* We get a 802.11 frame with a PPE sniffer header*/
        struct sm_promisc_header *hdr =
		(struct sm_promisc_header *) (*skb)->data;

	if (dev->type == ARPHRD_IEEE80211_PRISM) {
		struct avs_80211_1_header *avs;

		skb_pull(*skb, sizeof (struct sm_promisc_header));

		if (skb_headroom(*skb) < sizeof (struct avs_80211_1_header)) {
			struct sk_buff *newskb = skb_copy_expand(*skb,
								 sizeof (struct
									 avs_80211_1_header),
								 0, GFP_ATOMIC);
			if (newskb) {
                                dev_kfree_skb_irq(*skb);
                                *skb = newskb;
			} else
				return -1;
			/* This behavior is not very subtile... */
		}
		
		/* make room for the new header and fill it. */
		avs = (struct avs_80211_1_header *) skb_push(*skb,
							     sizeof (struct avs_80211_1_header));
		
		avs->version = cpu_to_be32(P80211CAPTURE_VERSION);
		avs->length = cpu_to_be32(sizeof (struct avs_80211_1_header));
		avs->mactime = cpu_to_be64(le64_to_cpu(hdr->clock));
		avs->hosttime = cpu_to_be64(jiffies);
		avs->phytype = cpu_to_be32(6);	/*OFDM: 6 for (g), 8 for (a) */
		avs->channel = cpu_to_be32(0);//channel_of_freq(freq));
		avs->datarate = cpu_to_be32(hdr->rate * 5);
		avs->antenna = cpu_to_be32(0);	/*unknown */
		avs->priority = cpu_to_be32(0);	/*unknown */
		avs->ssi_type = cpu_to_be32(3);	/*2: dBm, 3: raw RSSI */
		avs->ssi_signal = cpu_to_be32(hdr->rssi & 0x7f);
		avs->ssi_noise = cpu_to_be32(0);//priv->local_iwstatistics.qual.noise);	/*better than 'undefined', I assume */
		avs->preamble = cpu_to_be32(0);	/*unknown */
		avs->encoding = cpu_to_be32(0);	/*unknown */
	} else
		skb_pull(*skb, sizeof (struct sm_promisc_header));
	
	(*skb)->protocol = ETH_P_802_2;//htons(ETH_P_802_2);
	(*skb)->mac.raw = (*skb)->data;
        (*skb)->pkt_type = PACKET_OTHERHOST;

        return 0;
}

static int32_t mask_frametxdone(struct net_device *dev, int32_t mask)
{
	struct net_local *lp = netdev_priv(dev);
	struct s_sm_frame *frame;
	int32_t new_mask;
	
	spin_lock_bh(&lp->lock);
	new_mask = prism_softmac_frame_tx_done(lp->sm_context, &frame);
	spin_unlock_bh(&lp->lock);

	if( new_mask < 0 ) {
		cx3110x_warning("prism_softmac_frame_tx_done()"
				" returned %d", new_mask);
	} else {
		mask = new_mask;
	}
			
	if(frame == NULL) {
		cx3110x_warning("prism_softmac_frame_tx_done() returned an "
				"empty frame.");
		mask &= ~SM_FRAMETXDONE;
		goto out;
	}

	spin_lock_bh(&lp->lock);

	if( frame->flags & SM_FRAME_TXDROPPED ) {
		if (printk_ratelimit())
			cx3110x_warning("TX dropped.");
		lp->stats.tx_errors++;
		lp->stats.tx_dropped++;
	} else if(frame->flags & SM_FRAME_TXFAILED) {
		if (printk_ratelimit())
			cx3110x_debug(DBG_IC, "TX failed.");
		lp->stats.tx_errors++;
	} else {
		lp->stats.tx_packets++;
		lp->spi_lp->initial_packets++;
		lp->stats.tx_bytes += frame->length;
	}
	spin_unlock_bh(&lp->lock);
				
	/* Free the frame and SKB frame in done */
	frame_skb_free(dev, frame);
				
	/* We have transmitted a frame, so we can wake the queue again */
	if (netif_queue_stopped(dev))
		netif_wake_queue(dev);

out:
	return mask;
}

static int32_t mask_framerx(struct net_device *dev, int32_t mask)
{
	struct net_local *lp = netdev_priv(dev);
	struct sk_buff *skb;
	struct s_sm_frame *frame;
	int32_t new_mask;

	spin_lock_bh(&lp->lock);
	new_mask = prism_softmac_frame_rx(lp->sm_context, &frame);
	spin_unlock_bh(&lp->lock);
			
	if( new_mask < 0 ) {
		cx3110x_warning("%s: prism_softmac_frame_rx() "
				"returned error %d.",
				__FUNCTION__, new_mask);
	} else {
		mask = new_mask;
	}
			
	if( frame == NULL ) {
		mask &= ~SM_FRAMERX;
		goto out;
	}

	spin_lock_bh(&lp->lock);
	if( frame->flags & SM_FRAME_RXFAILED ) {
		cx3110x_warning("RX failed.");
		lp->stats.rx_errors++;
	} else {
		lp->stats.rx_packets++;
		lp->stats.rx_bytes += frame->length;
	}

	/* get the skb from the frame */
	skb = frame_to_skb(dev, frame);
	
	if (skb == NULL) {
		cx3110x_error("could not alloc rx skb.");
		goto out;
	}

	skb->dev = dev;
	dev->last_rx = jiffies;

	if(unlikely(lp->sm_mode == SM_MODE_PROMISCUOUS))
		sm_monitor_rx(dev, &skb);

	spin_unlock_bh(&lp->lock);
					
	/* set minimal skb parameters and deliver the network packet */
	skb->protocol = eth_type_trans(skb, dev);
	skb->ip_summed = CHECKSUM_NONE;
					
	netif_rx(skb);

out:
	return mask;
}

static int32_t mask_trap(struct net_device *dev, int32_t mask)
{
	struct net_local *lp = netdev_priv(dev);
	struct s_sm_conf t;
	int32_t result;
			
	t.length = sizeof(long);
	t.data = kmalloc(t.length, GFP_ATOMIC);

	if (t.data == NULL) {
		cx3110x_error("could not allocate first trap data.");
		goto out;
	}
	
	spin_lock_bh(&lp->lock);
	result = prism_softmac_trap(lp->sm_context, &t);
	spin_unlock_bh(&lp->lock);

	if(result == SM_EOVERFLOW) {
		kfree(t.data);
		t.data = kmalloc(t.length, GFP_ATOMIC);
		if(t.data != NULL) {
			spin_lock_bh(&lp->lock);
			result = prism_softmac_trap(lp->sm_context, &t);
			spin_unlock_bh(&lp->lock);
		} else {
			cx3110x_error("could not allocate second trap data.");
			goto out;
		}
	}
			
	if(result < SM_ENONE) {
		mask &= ~SM_TRAP;
		kfree(t.data);
		goto out;
	}
	
	mask |= result;
				
	if(t.flags & SM_CONF_OPSET) {
		/* This is a unsolicitated trap */
		result = handle_sm_trap(dev, &t);
		if(result >= SM_ENONE)
			mask |= result;
					
		kfree(t.data);
	} else {
		spin_lock_bh(&lp->lock);

		/* This is a response to a GET operation */
		memcpy(&lp->getresp, &t, sizeof(struct s_sm_conf));
					
		/* Wake up the app that requested this info.
		 * This will also free t.data. 
		 */
		wake_up(&lp->getresp_waitq);

		spin_unlock_bh(&lp->lock);

	}

out:
	return mask;
}

static int32_t mask_ic(struct net_device *dev, int32_t mask)
{
	struct net_local *lp = netdev_priv(dev);

	spin_lock_bh(&lp->lock);
	
	if (lp->device_state == DEVSTATE_ACTIVE ||
	    lp->device_state == DEVSTATE_BOOTING)
		queue_work(lp->workqueue, &lp->work);

	spin_unlock_bh(&lp->lock);
			
	mask &= ~SM_IC;

	return mask;
}


void handle_sm_callback(struct net_device *dev, int32_t mask)
{
	while(mask) {
		DEBUG(DBG_TRACE, "%s: callback mask 0x%x\n",
		      driver_name, mask);
		sm_print_callback_mask(mask);
		
		if(mask & SM_FRAMETXDONE)
			mask = mask_frametxdone(dev, mask);
		
		if( mask & SM_FRAMERX )
			mask = mask_framerx(dev, mask);
		
		if(mask & SM_TRAP)
			mask = mask_trap(dev, mask);
		
		if(mask & SM_IC)
			mask = mask_ic(dev, mask);
	}
}

/*
 * Create the net device instance that will be associated to the host
 * interface
 */
struct net_device *sm_drv_netdev_create(int ioaddr, int irq)
{
	struct net_local *lp;
	struct net_device * netdev;


	DEBUG(DBG_CALLS, "sm_drv_netdev_create\n");

	netdev = alloc_netdev(sizeof (struct net_local), "wlan%d",
			      ether_setup);
	if (!netdev) {
		cx3110x_error("could not allocate netdev.");
		return netdev;
	}

	netdev->base_addr          = ioaddr;
	netdev->irq                = irq;

	netdev->open               = &sm_drv_open;
	netdev->stop               = &sm_drv_close;
	netdev->hard_start_xmit    = &sm_drv_transmit;
	netdev->get_stats          = &sm_drv_statistics;
	netdev->wireless_handlers  = &sm_drv_we_handler_def;
	netdev->watchdog_timeo     = SM_DRV_TX_TIMEOUT;
	netdev->tx_timeout         = &sm_drv_tx_timeout;
	netdev->type               = ARPHRD_ETHER;

	lp = netdev->priv;
	memset(lp, 0, sizeof(struct net_local));

	lp->netdev = netdev;
	
	spin_lock_init(&lp->lock);

	init_waitqueue_head(&lp->conf_waitq);
	init_waitqueue_head(&lp->getresp_waitq);

	/* Init the Softmac timer */
	init_timer(&lp->softmac_timer);
	lp->softmac_timer.function = driver_timer_expired;
	lp->softmac_timer.data = (unsigned long)netdev;

	/* Init the scan timer */
	init_timer(&lp->scan_timer);
	lp->scan_timer.function = send_scan_complete_timer;
	lp->scan_timer.data = (unsigned long)netdev;

	lp->dma_threshold = CX3110X_DEFAULT_DMA_THRESHOLD;
	
	return netdev;
}

#define UMAC_VERSION_LEN 64
#define COMPONENT_UMAC 1
#define COMPONENT_LMAC 0

static void sm_drv_print_versions(struct net_device *dev)
{
	int ret;
	uint32_t component;	
	uint8_t version[UMAC_VERSION_LEN];

	component = COMPONENT_UMAC;
	ret = sm_drv_oid_set(dev, GEN_OID_COMPONENT_NR, &component,
			     sizeof(component));
	if (ret < 0) {
		cx3110x_warning("failed to set umac component number.");
		return;
	}
	
	ret = sm_drv_oid_get(dev, GEN_OID_VERSION, &version, sizeof(version));
	if (ret < 0) {
		cx3110x_warning("failed to get libumac version.");
		return;
	}

	cx3110x_info(LIBUMAC_NAME " version %s", version);

	component = COMPONENT_LMAC;
	ret = sm_drv_oid_set(dev, GEN_OID_COMPONENT_NR, &component,
			     sizeof(component));
	if (ret < 0) {
		cx3110x_warning("failed to set lmac component number.");
		return;
	}
	
	ret = sm_drv_oid_get(dev, GEN_OID_VERSION, &version, sizeof(version));
	if (ret < 0) {
		cx3110x_warning("failed to get lmac version.");
		return;
	}

	cx3110x_info(LMAC_NAME " version %s", version);
}


extern struct platform_device wlan_omap_device;
extern unsigned int driver_type;
int sm_drv_open(struct net_device *dev)
{
	struct net_local *lp = dev->priv;
	unsigned long timeout;

	DEBUG(DBG_CALLS, "sm_drv_open\n");

	if (cx3110x_spi_start(dev) < 0)
		goto err_out;

	lp->sm_pda = sm_drv_spi_get_pda(&(wlan_omap_device.dev));
	if (!lp->sm_pda)
		goto err_mcbsp_free_out;
	
	lp->device_state = DEVSTATE_BOOTING;
	lp->bss_type = DOT11_BSSTYPE_INFRA;
	
	if (sm_drv_spi_request_irq(dev) < 0)
		goto err_mcbsp_free_out;
	
	/* We bring the interface up */
	if (sm_drv_spi_up(dev) < 0)
		goto err_mcbsp_irq_free_out;
	
	/* Wait for the chip to receive its first interrupt */
	timeout = msecs_to_jiffies(CX3110X_BOOT_TIMEOUT);
	timeout = wait_for_completion_interruptible_timeout(&softmac_init_comp,
							    timeout);
	if (!timeout) {
		cx3110x_error("firmware boot failed.");
		goto err_mcbsp_irq_free_out;
	}

	if (lp->sm_mode == SM_MODE_PROMISCUOUS) {
		uint32_t commit = 0;
		
		lp->bss_type = DOT11_BSSTYPE_NONE;
		if (sm_drv_oid_set(dev, GEN_OID_COMMIT, (void*)&commit,
				   sizeof(uint32_t)) < 0)
			goto err_mcbsp_irq_free_out;
		goto start_tcp_queue;
	}
	
	if (driver_type == SM_DRIVER_TYPE_UMAC) {
		uint32_t scan_mode = SCAN_MODE_PASSIVE;
		uint32_t authen = DOT11_AUTH_BOTH;
		/* We don't need 802.11h */
		uint32_t wwr_mode = DOT11_WWR_MODE_11D; 
		uint32_t dot11d_conformance_mode = DOT11_CONFORMANCE_FAST;
		uint32_t profile = DOT11_PROFILE_MIXED;
		uint32_t bgr_scan_disable = 0;
		int32_t  scan_threshold = -75;
		struct obj_scan scan_params = {-1, 80, 200, 60000, 80, 140};
		/* U-APSD is enabled for all classes */
		uint32_t uapsd_classes = 0xff; 
		uint32_t commit = 0;
		
		if (sm_drv_oid_set(dev, DOT11_OID_AUTHENABLE,
				   (void *)&authen, sizeof(uint32_t)) < 0)
			goto err_mcbsp_irq_free_out;
		
		if (sm_drv_oid_set(dev, DOT11_OID_SCANMODE,
				   (void*)&scan_mode, sizeof(uint32_t)) < 0)
			goto err_mcbsp_irq_free_out;
		
		if (sm_drv_oid_set(dev, DOT11_OID_WWRMODE,
				   (void*)&wwr_mode, sizeof(uint32_t)) < 0)
			goto err_mcbsp_irq_free_out;
		
		if (sm_drv_oid_set(dev, DOT11_OID_CONFORMANCEMODE,
				   (void*)&dot11d_conformance_mode,
				   sizeof(uint32_t)) < 0)
			goto err_mcbsp_irq_free_out;
		
		if (sm_drv_oid_set(dev, DOT11_OID_PROFILES,
				   (void*)&profile, sizeof(uint32_t)) < 0)
			goto err_mcbsp_irq_free_out;
		
		if (sm_drv_oid_set(dev, DOT11_OID_AUTOSCANDISABLE,
				   (void*)&bgr_scan_disable,
				   sizeof(uint32_t)) < 0)
			goto err_mcbsp_irq_free_out;
		
		if (sm_drv_oid_set(dev, DOT11_OID_SCANTHRESHOLD,
				   (void*)&scan_threshold,
				   sizeof(int32_t)) < 0)
			goto err_mcbsp_irq_free_out;

		if (sm_drv_oid_set(dev, DOT11_OID_SCAN,
				   (void*)&scan_params,
				   sizeof(struct obj_scan)) < 0)
			goto err_mcbsp_irq_free_out;
		if (sm_drv_oid_set(dev, DOT11_OID_UAPSD,
				   (void*)&uapsd_classes,
				   sizeof(uint32_t)) < 0)
			goto err_mcbsp_irq_free_out;		

		/* WWR mode needs to be commited */
		if (sm_drv_oid_set(dev, GEN_OID_COMMIT,
				   (void*)&commit, sizeof(uint32_t)) < 0)
			goto err_mcbsp_irq_free_out;
	}

	sm_drv_print_versions(dev);
	
 start_tcp_queue:	
	netif_start_queue(dev);
	
	return 0;

 err_mcbsp_irq_free_out:
	sm_drv_spi_free_irq(dev);
 err_mcbsp_free_out:	
	cx3110x_spi_stop(dev);

 err_out:
	return -ENXIO;
}

int sm_drv_close(struct net_device *dev)
{
	struct net_local *lp = dev->priv;
	struct spi_local *spi_lp = lp->spi_lp;

	DEBUG(DBG_CALLS, "sm_drv_close \n");
	
	sm_drv_disassociate(dev);
	netif_stop_queue(dev);
	
        /* Disable all device interrupts */
	sm_drv_spi_cli(dev);

	spin_lock_bh(&lp->lock);

	if(lp->device_state == DEVSTATE_ACTIVE
	   || lp->device_state == DEVSTATE_BOOTING)
		lp->device_state = DEVSTATE_IDLE;

	spin_unlock_bh(&lp->lock);

	flush_workqueue(lp->workqueue);

	cx3110x_info("shut down softmac.");

	spin_lock_bh(&lp->lock);

	if (lp->sm_initialization) {
		/* shut down SoftMAC */
		lp->sm_descr.mtu = 0;
		lp->sm_initialization = 0;
		lp->spi_lp->initial_packets = 0;
		spi_lp->upload_state = UPLOAD_STATE_BOOTING;
		prism_softmac_destroy(lp->sm_context);
		kfree(lp->sm_context);
	}
	spin_unlock_bh(&lp->lock);

	sm_drv_spi_down(dev);

	sm_drv_spi_free_irq(dev);
	cx3110x_spi_stop(dev);

	return 0;
}

int sm_drv_transmit(struct sk_buff *skb, struct net_device *dev)
{
	struct s_sm_frame *frame;
	struct net_local *lp = dev->priv;
	int32_t callb_mask;

	DEBUG(DBG_CALLS, "sm_drv_transmit\n");


	spin_lock_bh(&lp->lock);

 again:

	if(lp->queued_tx_frame) {
		frame = lp->queued_tx_frame;
	} else {

		frame = skb_to_frame(dev, skb);

		if(!frame) {
			cx3110x_error("%s: could not allocate skb.",
				      __FUNCTION__);
			dev_kfree_skb(skb);
			lp->stats.tx_errors++;
			goto out;
		}
	}
	
	callb_mask = prism_softmac_frame_tx( lp->sm_context, frame );

	if(callb_mask < 0) {
		if(callb_mask == SM_EOVERFLOW) {
			cx3110x_warning("overflow, stopping TX queue.");

			/*
			 * We have to accept the packet, otherwise it gets
			 * lost. So we temporarily store it and stop the
			 * queue. We retry this packet once we have a
			 * sm_frame_tx_done
			 */
			lp->queued_tx_frame = frame;
			netif_stop_queue(dev);
		} else {
			cx3110x_warning("%s: prism_softmac_frame_tx() "
					" returned error %d.",
					__FUNCTION__, callb_mask);
			/*
			 * a normal frame failed, increment the parents
			 * device error counter
			 */
			lp->stats.tx_errors++;
			
			/* Free the frame and the sk_buff */
			frame_skb_free(dev, frame);
			if(lp->queued_tx_frame) {
				lp->queued_tx_frame = NULL;
			}
		}
		goto out;
	}

	spin_unlock_bh(&lp->lock);

	dev->trans_start = jiffies;
	handle_sm_callback(dev, callb_mask);

	spin_lock_bh(&lp->lock);

	/*
	 * If we transmitted a queued frame, now try to transmit the
	 * current frame
	 */
	if(lp->queued_tx_frame) {
		lp->queued_tx_frame = NULL;
		goto again;
	}

 out:	
	spin_unlock_bh(&lp->lock);

	return 0;
}

void sm_drv_tx_timeout(struct net_device *dev)
{
	struct net_local *lp = dev->priv;
	struct net_device_stats *statistics;
	
	DEBUG(DBG_CALLS, "sm_drv_tx_timeout \n");

	spin_lock_bh(&lp->lock);

	/* increment the transmit error counter */
	statistics = &lp->stats;
	statistics->tx_errors++;

	spin_unlock_bh(&lp->lock);

	netif_wake_queue(dev);
	
	return;
}

struct net_device_stats *sm_drv_statistics(struct net_device *dev)
{
	struct net_local *lp = dev->priv;

	DEBUG(DBG_CALLS, "sm_drv_statistics \n");

	return &lp->stats;
}

int sm_drv_reset(struct net_device *dev, uint8_t sm_mode)
{
	struct net_local *lp = dev->priv;
	int ret;

	spin_lock_bh(&lp->lock);

	if(lp->device_state == DEVSTATE_IDLE) {
		cx3110x_warning("cannot reset device in idle state.");
		spin_unlock_bh(&lp->lock);
		return -1;
	}

	spin_unlock_bh(&lp->lock);

	lp->sm_mode = sm_mode;

	ret = sm_drv_close(dev);

	if(ret == 0) {
		ret = sm_drv_open(dev);
	}

	return ret;
}
