/*
 * UPnP Browser for Maemo
 *
 * browser.c
 *
 * Copyright 2005 Nokia Corporation. All rights reserved.
 *
 * This is licensed under BSD-style license,
 * see file COPYING.
 */

#ifdef MAEMO
#include <hildon-widgets/gtk-infoprint.h>
#include <hildon-widgets/hildon-defines.h>
#endif

#include <gtk/gtk.h>

#include "browser.h"
#include "interface.h"
#include "helper.h"
#include "upnp.h"

#include <cybergarage/contentdirectory/ccontentdirectory.h>
#include <cybergarage/contentdirectory/cdidllite.h>
#include <cybergarage/avdebug.h>

extern UPnPBrowserWidgets* widgets;

/**
 * Create the device tree model & view
 */
int create_device_view()
{	
	GtkTreeViewColumn *column;
	GtkCellRenderer* renderer; 
	
	/* Create a new tree view with the tree store as its model */
	widgets->container_filter = gtk_tree_model_filter_new(
					GTK_TREE_MODEL(widgets->folder_model), 
					NULL);
	
	gtk_tree_model_filter_set_visible_func(
			GTK_TREE_MODEL_FILTER(widgets->container_filter),
			container_filter_func, widgets, NULL);
	widgets->device_view = gtk_tree_view_new_with_model(
		GTK_TREE_MODEL(widgets->container_filter));

	gtk_widget_set_size_request(widgets->device_view, 400, 200);

	/* Create expander/icon column */
	column = gtk_tree_view_column_new();
	gtk_tree_view_column_set_sizing(column, GTK_TREE_VIEW_COLUMN_AUTOSIZE);
	renderer = gtk_cell_renderer_pixbuf_new();
	gtk_cell_renderer_set_fixed_size(renderer, -1, TREE_CELL_HEIGHT);
	
	gtk_tree_view_append_column(GTK_TREE_VIEW(widgets->device_view), column);
	gtk_tree_view_column_pack_start(column, renderer, FALSE);

	
	gtk_tree_view_column_set_cell_data_func(column, renderer,
						render_mimeimage_datafunc,
						NULL, NULL);
	
	/* Create a column */
	column = gtk_tree_view_column_new();
	gtk_tree_view_column_set_sizing(column, GTK_TREE_VIEW_COLUMN_AUTOSIZE);
	renderer = gtk_cell_renderer_text_new();
	gtk_cell_renderer_set_fixed_size(renderer, -1, TREE_CELL_HEIGHT);
	
	gtk_tree_view_append_column(GTK_TREE_VIEW(widgets->device_view), column);
	gtk_tree_view_column_pack_start(column, renderer, FALSE);

	gtk_tree_view_column_add_attribute(column, renderer,
					   "text",
					   FOLDERMODEL_COLUMN_NAME);

	/* Create a scrolled window */
	widgets->device_scrolled_window = gtk_scrolled_window_new(
		gtk_tree_view_get_hadjustment(GTK_TREE_VIEW(widgets->device_view)),
		gtk_tree_view_get_vadjustment(GTK_TREE_VIEW(widgets->device_view)));

	/* Set the scrollbar showing policy */
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(widgets->device_scrolled_window),
				       GTK_POLICY_NEVER,
				       GTK_POLICY_AUTOMATIC);

	/* Add the device view to the scrolled view */
	gtk_container_add(GTK_CONTAINER(widgets->device_scrolled_window), widgets->device_view);

	/* Select browse model for the selection */
	gtk_tree_selection_set_mode(
		gtk_tree_view_get_selection(GTK_TREE_VIEW(widgets->device_view)),
		GTK_SELECTION_BROWSE);
	
	return 0;
}

/**
 * Create the folder tree model & view
 */
int create_folder_model_view()
{
	GtkCellRenderer* renderer;
	GtkTreeViewColumn *column;
	
	/* Create a new list store */
	widgets->folder_model = gtk_tree_store_new(FOLDERMODEL_COLUMNS,
						   G_TYPE_STRING, 
						   G_TYPE_BOOLEAN,
						   G_TYPE_STRING,
						   G_TYPE_STRING,
						   G_TYPE_STRING,
						   G_TYPE_STRING,
						   G_TYPE_BOOLEAN);

	widgets->file_filter = gtk_tree_model_filter_new(
					GTK_TREE_MODEL(widgets->folder_model), 
					NULL);
	
	/* Create a new tree view with the list store as its model */
	widgets->folder_view = 
		gtk_tree_view_new_with_model(
			GTK_TREE_MODEL(widgets->file_filter));
	
	gtk_widget_set_size_request(widgets->folder_view, 400, 200);
	
	/* Create expander/icon column */
	renderer = gtk_cell_renderer_pixbuf_new();
	gtk_tree_view_insert_column_with_data_func(GTK_TREE_VIEW(widgets->folder_view),
                                             -1,
                                             "",
                                             renderer,
                                             render_mimeimage_datafunc,
                                             NULL,
                                             NULL);
	
	/* Create a column */
	renderer = gtk_cell_renderer_text_new();
	column = gtk_tree_view_column_new_with_attributes("Files",
					renderer,
					"text", FOLDERMODEL_COLUMN_NAME,
					NULL); 
					
	gtk_cell_renderer_set_fixed_size(renderer, -1, TREE_CELL_HEIGHT);
	
	/* Add the column to the view. */
	gtk_tree_view_append_column (GTK_TREE_VIEW(widgets->folder_view), column);

	/* Create a scrolled window */
	widgets->folder_scrolled_window = gtk_scrolled_window_new(
		gtk_tree_view_get_hadjustment(GTK_TREE_VIEW(widgets->folder_view)),
		gtk_tree_view_get_vadjustment(GTK_TREE_VIEW(widgets->folder_view)));

	/* Set the scrollbar showing policy */
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(widgets->folder_scrolled_window),
				       GTK_POLICY_NEVER,
				       GTK_POLICY_AUTOMATIC);

	/* Add the device view to the scrolled view */
	gtk_container_add(GTK_CONTAINER(widgets->folder_scrolled_window),
			  widgets->folder_view);

	/* Select browse model for the selection */
	gtk_tree_selection_set_mode(
		gtk_tree_view_get_selection(GTK_TREE_VIEW(widgets->folder_view)),
		GTK_SELECTION_BROWSE);
	
	return 0;
}

void device_view_row_selected(GtkTreeSelection *selection,
			      gpointer user_data)
{
	GtkTreeIter iter;
	GtkTreeIter child_iter;
        GtkTreeView *tree = NULL;
	GtkTreeModel *model;
	GtkTreePath *child_path;
	gchar* udn = NULL;
	gchar* id = NULL;
	gboolean browsed = FALSE;
	UPnPBrowserWidgets* widgets = NULL;
	
	DEBUG("BEGIN: device_view_row_selected\n");
	
	widgets = (UPnPBrowserWidgets *)user_data;
	g_return_if_fail(widgets != NULL);
	
	tree = gtk_tree_selection_get_tree_view(selection);
        if (tree == NULL) return;
	
	/* Get selected row from the tree */
	if (!gtk_tree_selection_get_selected(selection, &model, &iter))
	{
		DEBUG("No selection\n");
		DEBUG("END: device_view_row_selected\n");
		return;
	}

	gtk_tree_model_get(model, &iter, FOLDERMODEL_COLUMN_DEVUDN, &udn,
					 FOLDERMODEL_COLUMN_ID, &id,
					 FOLDERMODEL_COLUMN_BROWSED, &browsed,
					 -1);

	/* Convert the iterator to be valid in the bigger model */
	gtk_tree_model_filter_convert_iter_to_child_iter(
				GTK_TREE_MODEL_FILTER(model),
				&child_iter,
				&iter);
	
	if (!browsed)
	{
		/* Get some content for the currently selected node */
		get_upnp_content_directory(widgets->folder_model, 
					   &child_iter,
					   udn, id);
	}

	g_free(udn);
	g_free(id);
	
	/* Now we need new filefilter */
	if (!gtk_tree_selection_get_selected(selection, &model,
					     &iter))
	{
		DEBUG("No selection\n");
		DEBUG("END: device_view_row_selected\n");
		return;
	}
	
	child_path = gtk_tree_model_get_path(GTK_TREE_MODEL(widgets->folder_model), &child_iter);
	
	if (!child_path)
	{
		DEBUG("No child path\n");
		DEBUG("END: device_view_row_selected\n");
		return;
	}
	
	update_folder_view_to_path(child_path);
	
	gtk_tree_path_free(child_path);

	DEBUG("END: device_view_row_selected\n");
}

gboolean expand_idlefunc(gpointer data)
{
	GtkTreePath *device_path;
	GtkTreeSelection *selection, *folder_selection;
	GtkTreePath *child_path = (GtkTreePath*)data;
	
	if (update_folder_view_to_path(child_path))
	{
		folder_selection = gtk_tree_view_get_selection(
			GTK_TREE_VIEW(widgets->folder_view));
	
		selection = gtk_tree_view_get_selection(
				GTK_TREE_VIEW(widgets->device_view));
	
		device_path = gtk_tree_model_filter_convert_child_path_to_path(
				GTK_TREE_MODEL_FILTER(widgets->container_filter),
				child_path);
	
		if (device_path != NULL)
		{
			gtk_tree_view_expand_to_path(GTK_TREE_VIEW(widgets->device_view), 
					device_path);
			gtk_tree_view_expand_row(GTK_TREE_VIEW(widgets->device_view), 
					device_path, FALSE);
			gtk_tree_selection_select_path(selection, device_path);
		}
	}
	
	gtk_tree_path_free(child_path);
	return FALSE;
}

void folder_view_row_expanded(GtkTreeView *view,
			      GtkTreeIter *iter,
			      GtkTreePath *path,
			      gpointer user_data)
{
	GtkTreePath *child_path = NULL;
	
	GtkTreeModel *model = NULL;
	
	model = gtk_tree_view_get_model(view);
	
	/* Ok, go to container */
	child_path = gtk_tree_model_filter_convert_path_to_child_path(
				GTK_TREE_MODEL_FILTER(model), path);
	
	if (child_path == NULL) return;

	g_timeout_add(100, expand_idlefunc, child_path);
	g_signal_stop_emission_by_name(G_OBJECT(view), "row-expanded");
}


void render_mimeimage_datafunc(GtkTreeViewColumn *column,
                               GtkCellRenderer *renderer,
                               GtkTreeModel *model,
                               GtkTreeIter *iter,
                               gpointer data)
{
	GdkPixbuf *pixbuf = NULL;
	GtkIconTheme *theme = NULL; 
	const gchar *image_name = NULL;
	gchar *node_class = NULL;
	
	gboolean isexpander = FALSE, isexpanded = FALSE;
	
	g_return_if_fail(renderer != NULL);
	g_return_if_fail(model != NULL);
	
	/* Get theme */
	theme = gtk_icon_theme_get_default();
	g_return_if_fail(theme != NULL);
	
	g_object_get(renderer, 
		     "is-expander", &isexpander,
		     "is-expanded", &isexpanded,
		     NULL);
	
	/* If it is expander, then it is a container */
	gtk_tree_model_get(model, iter,
		FOLDERMODEL_COLUMN_CLASS, &node_class, -1);

	if (node_class != NULL)
	{
		if (cg_strstr(node_class, 
			      DIDL_LITE_OBJECT_CONTAINER) == 0)
		{
			/* Object is container, is has no children */
			if (isexpanded)
			{
				image_name = "qgn_list_gene_fldr_opn";
			
			} 
			else 
			{
				image_name = "qgn_list_gene_fldr_cls";
			}
				
		}
		else if (cg_strstr(node_class, "object.item.audioItem") != -1)
		{
			image_name = "qgn_list_gene_music_file";
			
		} 
		else if (cg_strstr(node_class, "object.item.imageItem") != -1)
		{
			image_name = "qgn_list_gene_image_file";
			
		} 
		else if (cg_strstr(node_class, "object.item.videoItem") != -1)
		{
			image_name = "qgn_list_gene_video_file";
		}
		else if (cg_strstr(node_class, "object.item.playlist") != -1)
		{
			image_name = "qgn_list_gene_playlist";
		}
		else
		{
			image_name = "qgn_list_gene_notsupported";
		}
	}
	else
	{
		/* The node didn't have class, expect it to be device.
		 * No need to check device type because only CDS-capable
		 * devices are in our list */
		image_name = "qgn_list_filemanager";		
	}
	
	g_free(node_class);
	
	if (image_name != NULL)
	{
#ifdef MAEMO
		pixbuf = gtk_icon_theme_load_icon(theme,
					  image_name,
					  HILDON_ICON_PIXEL_SIZE_SMALL,
					  GTK_ICON_LOOKUP_NO_SVG,
					  NULL);
#endif
	}
	
	if (pixbuf != NULL)
	{
		/* Set pixbuf */
		g_object_set(G_OBJECT(renderer), 
			     "pixbuf", pixbuf, 
			     NULL);
	}
}


gboolean container_filter_func(GtkTreeModel *model,
			       GtkTreeIter *iter,
			       gpointer data)
{
	gboolean container;
	gtk_tree_model_get(model, iter, 
			   FOLDERMODEL_COLUMN_CONTAINER, &container,
			   -1);
	return container;	
}


gboolean update_folder_view_to_path(GtkTreePath *path)
{
	GtkTreePath *current_path = NULL;
	GtkTreeSelection *selection = NULL;
	
	g_return_val_if_fail(path != NULL, FALSE);
	
	if (widgets->file_filter != NULL)
	{
		g_object_get(G_OBJECT(widgets->file_filter), 
			"virtual-root", &current_path, NULL);
		
		if (current_path != NULL &&
		    gtk_tree_path_compare(path, current_path) == 0) 
		{
			gtk_tree_path_free(current_path);
			return FALSE;
		}
	
		g_object_unref(G_OBJECT(widgets->file_filter));
		widgets->file_filter = NULL;
	}
	
	widgets->file_filter = gtk_tree_model_filter_new(
			GTK_TREE_MODEL(widgets->folder_model), 
			path);
	
	selection = gtk_tree_view_get_selection(
			GTK_TREE_VIEW(widgets->folder_view));
	
	gtk_tree_view_set_model(GTK_TREE_VIEW(widgets->folder_view), 
				widgets->file_filter);

	gtk_tree_path_free(current_path);
	
	return TRUE;
}

/**
 * Helper to get the currently selected content directory item
 *
 * @param uri The URI of the currently selected resource
 * @param node_class The UPnP class of the currently selected resource
 * @param id The object ID of the currently selected resource
 * @param udn The UDN of the server offering the currently selected resource
 * @return TRUE if an item was selected; otherwise FALSE
 */
gboolean get_current_content_item(gchar** name, gchar** uri,
				  gchar** node_class, gchar** id,
				  gchar** udn)
{
	GtkTreeModel *model = NULL;
	GtkTreeSelection *selection = NULL;
	GtkTreeIter iter;
	
	/* Get the selection from the folder view */
	selection = gtk_tree_view_get_selection(
				GTK_TREE_VIEW(widgets->folder_view));

	if (!gtk_tree_selection_get_selected(selection,
					     &model,
					     &iter))
	{
		return FALSE;
	}

	/* Get the node's URI and class */
	gtk_tree_model_get(model, &iter,
		FOLDERMODEL_COLUMN_NAME, name,
		FOLDERMODEL_COLUMN_URI, uri,
		FOLDERMODEL_COLUMN_CLASS, node_class,
		FOLDERMODEL_COLUMN_ID, id,
		FOLDERMODEL_COLUMN_DEVUDN, udn,
		-1);

	return TRUE;
}

