/* 
 * CyberLinkC AV API for Connection Manager Service
 *
 * ccmscontrolpoint.h
 *
 * Copyright 2005 Nokia Corporation. All rights reserved.
 *
 * This is licensed under BSD-style license,
 * see file COPYING.
 */
 
#ifndef CG_UPNP_AV_CCMSCONTROLPOINT_H
#define CG_UPNP_AV_CCMSCONTROLPOINT_H

#include <cybergarage/connectionmanager/cconnectionmanager.h>
#include <cybergarage/connectionmanager/ccmsactionarguments.h>
#include <cybergarage/upnp/cdevice.h>

/*****************************************************************************
 * Connection Manager Service action invocations
 *****************************************************************************/

/**
 * Invoke an action on the upnp device
 *
 * @param upnpDevice The UPnP device
 * @param actionName The name of the action to invoke
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param numArgs The number of action-specific arguments
 * @param ... A list of action-specific arguments
 *
 * @return An integer containing the UPnP status code
 *
 * The list of action-specific arguments is a comma-separated list, 
 * that is organised as pairs of char* ARG_NAME and CgString* ARG_VALUE.
 * ARG_NAME contains the name of the argument, while ARG_VALUE contains
 * the value of the argument. You don't need to put anything inside OUT
 * arguments, as they are not sent along with the action; their contents
 * are set when the action post returns with OUT values that were received
 * from the peer.
 */
int cg_upnp_av_cms_control_invokeaction(CgUpnpDevice* upnpDevice,
					char* actionName,
					char** errorString,
					...);

/****************************************************************************
 * Mandatory action invocations
 ****************************************************************************/

/**
 * Get the protocol-related info that the device's connection manager
 * supports.
 *
 * @param upnpDevice The UPnP device.
 * @param source A comma-separated list of information on protocols this 
 *        ConnectionManager supports for "sourcing" (sending) data (OUT).
 * @param sink A comma-separated list of information on protocols this
 *        ConnectionManager supports for "sinking" (receiving) data (OUT).
 */
#define cg_upnp_av_cms_control_getprotocolinfo(upnpDevice, \
					       errorString, \
					       source, \
					       sink) \
	\
	cg_upnp_av_cms_control_invokeaction(upnpDevice, \
				CG_UPNP_CMS_ACTION_GETPROTOCOLINFO, \
				errorString, \
				CG_UPNP_CMS_ARG_SOURCE, \
				source, \
				CG_UPNP_CMS_ARG_SINK, \
				sink, \
				NULL)
/**
 * Get currently active connections' IDs held by the device.
 *
 * @param upnpDevice The UPnP device.
 * @param connectionIDs Comma-separated list of references to current 
 *        active Connections (OUT).
 */
#define cg_upnp_av_cms_control_getcurrentconnectionids(upnpDevice, \
						       errorString, \
						       connectionIDs) \
	\
	cg_upnp_av_cms_control_invokeaction(upnpDevice, \
				CG_UPNP_CMS_ACTION_GETCURRENTCONNECTIONIDS, \
				errorString, \
				CG_UPNP_CMS_ARG_CONNECTIONIDS, \
				connectionIDs, \
				NULL)

/**
 * Get associated information of the connection referred to by the 
 * ConnectionID parameter.
 *
 * @param upnpDevice The UPnP device.
 * @param connectionID A currently active connection ID (IN).
 * @param rcsID Identifies a logical instance of the Rendering Control
 *        service associated with a connection (OUT).
 * @param avtransportID Identifies a logical instance of the AV Transport
 *        Control service associated with a connection (OUT).
 * @param protocolInfo A string containing the supported protocols (OUT):
 *        "<protocol>:<network>:<contentFormat>:<additionalInfo>".
 * @param peerConnectionManager The UDN and Service ID of the peer's (i.e.
 *        the device's counterpart) connection manager (OUT):
 *        "<UDN>/<Service-Id>".
 * @param peerConnectionID The connection ID of the peer's (i.e.
 *        the device's counterpart) connection manager (OUT).
 * @param direction The direction of the out-of-band dataflow from the device's
 *        point of view (OUT): "Output" or "Input".
 * @param status The current status of the connection referred to by
 *        the connectionID parameter (OUT).
 */
#define cg_upnp_av_cms_control_getcurrentconnectioninfo(upnpDevice, \
							errorString, \
						        connectionID, \
							rcsID, \
							avtransportID, \
							protocolInfo, \
							peerConnectionManager, \
							peerConnectionID, \
							direction, \
							status) \
	\
	cg_upnp_av_cms_control_invokeaction(upnpDevice, \
				CG_UPNP_CMS_ACTION_GETCURRENTCONNECTIONINFO, \
				errorString, \
				CG_UPNP_CMS_ARG_CONNECTIONID, \
				connectionID, \
				CG_UPNP_CMS_ARG_RCSID, \
				rcsID, \
				CG_UPNP_CMS_ARG_AVTRANSPORTID, \
				avtransportID, \
				CG_UPNP_CMS_ARG_PROTOCOLINFO, \
				protocolInfo, \
				CG_UPNP_CMS_ARG_PEERCONNECTIONMANAGER, \
				peerConnectionManager, \
				CG_UPNP_CMS_ARG_PEERCONNECTIONID, \
				peerConnectionID, \
				CG_UPNP_CMS_ARG_DIRECTION, \
				direction, \
				CG_UPNP_CMS_ARG_STATUS, \
				status, \
				NULL)

/****************************************************************************
 * Optional action invocations
 ****************************************************************************/

/**
 * Prepare the device to connect to the network for the purposes of sending
 * or receiving media content (ex. a video stream).
 *
 * @param upnpDevice The UPnP device.
 * @param remoteProtocolInfo The protocol, network, and format that should be 
 *        used to transfer the content (IN):
 *        "<protocol>:<network>:<contentFormat>:<additionalInfo>".
 * @param peerConnectionManager The UDN and Service ID of the peer's (i.e.
 *        the device's counterpart) connection manager (IN):
 *        "<UDN>/<Service-Id>".
 * @param peerConnectionID The connection ID of the peer's (i.e. the device's
 *        counterpart) connection manager (IN).
 * @param direction The direction of the out-of-band dataflow from the device's
 *        point of view (IN): "Output" or "Input".
 * @param connectionID The connection ID that is assigned to the new 
 *        connection (OUT).
 * @param avtransportID The ID of the AV Transport service instance that is
 *        assigned to the new connection (OUT).
 * @param rcsID The ID of the Rendering Control service instance that is
 *        assigned to the new connection (OUT).
 */
#define cg_upnp_av_cms_control_prepareforconnection(upnpDevice, \
						    errorString, \
						    remoteProtocolInfo, \
						    peerConnectionManager, \
						    peerConnectionID, \
						    direction, \
						    connectionID, \
						    avtransportID, \
						    rcsID) \
	\
	cg_upnp_av_cms_control_invokeaction(upnpDevice, \
				CG_UPNP_CMS_ACTION_PREPAREFORCONNECTION, \
				errorString, \
				CG_UPNP_CMS_ARG_REMOTEPROTOCOLINFO, \
				remoteProtocolInfo, \
				CG_UPNP_CMS_ARG_PEERCONNECTIONMANAGER, \
				peerConnectionManager, \
				CG_UPNP_CMS_ARG_DIRECTION, \
				direction, \
				CG_UPNP_CMS_ARG_CONNECTIONID, \
				connectionID, \
				CG_UPNP_CMS_ARG_AVTRANSPORTID, \
				avtransportID, \
				CG_UPNP_CMS_ARG_RCSID, \
				rcsID, \
				NULL)
/**
 * Ensure that all resources associated with the connection are freed up. 
 * A control point should call the ConnectionComplete action for all
 * connections that it created via PrepareForConnection.
 *
 * @param upnpDevice The UPnP device.
 * @param connectionID The ID of the connection to terminate (IN).
 */ 
#define cg_upnp_av_cms_control_connectioncomplete(upnpDevice, \
						  errorString, \
						  connectionID) \
	\
	cg_upnp_av_cms_control_invokeaction(upnpDevice, \
				CG_UPNP_CMS_ACTION_CONNECTIONCOMPLETE, \
				errorString, \
				CG_UPNP_CMS_ARG_CONNECTIONID, \
				connectionID, \
				NULL)

#endif
